/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evidently.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains the configuration details of one Evidently launch.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Launch implements SdkPojo, Serializable, ToCopyableBuilder<Launch.Builder, Launch> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Launch::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdTime").getter(getter(Launch::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdTime").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(Launch::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<LaunchExecution> EXECUTION_FIELD = SdkField.<LaunchExecution> builder(MarshallingType.SDK_POJO)
            .memberName("execution").getter(getter(Launch::execution)).setter(setter(Builder::execution))
            .constructor(LaunchExecution::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("execution").build()).build();

    private static final SdkField<List<LaunchGroup>> GROUPS_FIELD = SdkField
            .<List<LaunchGroup>> builder(MarshallingType.LIST)
            .memberName("groups")
            .getter(getter(Launch::groups))
            .setter(setter(Builder::groups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("groups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LaunchGroup> builder(MarshallingType.SDK_POJO)
                                            .constructor(LaunchGroup::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedTime").getter(getter(Launch::lastUpdatedTime)).setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedTime").build()).build();

    private static final SdkField<List<MetricMonitor>> METRIC_MONITORS_FIELD = SdkField
            .<List<MetricMonitor>> builder(MarshallingType.LIST)
            .memberName("metricMonitors")
            .getter(getter(Launch::metricMonitors))
            .setter(setter(Builder::metricMonitors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metricMonitors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricMonitor> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricMonitor::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Launch::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> PROJECT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("project")
            .getter(getter(Launch::project)).setter(setter(Builder::project))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("project").build()).build();

    private static final SdkField<String> RANDOMIZATION_SALT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("randomizationSalt").getter(getter(Launch::randomizationSalt)).setter(setter(Builder::randomizationSalt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("randomizationSalt").build()).build();

    private static final SdkField<ScheduledSplitsLaunchDefinition> SCHEDULED_SPLITS_DEFINITION_FIELD = SdkField
            .<ScheduledSplitsLaunchDefinition> builder(MarshallingType.SDK_POJO).memberName("scheduledSplitsDefinition")
            .getter(getter(Launch::scheduledSplitsDefinition)).setter(setter(Builder::scheduledSplitsDefinition))
            .constructor(ScheduledSplitsLaunchDefinition::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scheduledSplitsDefinition").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(Launch::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusReason").getter(getter(Launch::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReason").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Launch::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Launch::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, CREATED_TIME_FIELD,
            DESCRIPTION_FIELD, EXECUTION_FIELD, GROUPS_FIELD, LAST_UPDATED_TIME_FIELD, METRIC_MONITORS_FIELD, NAME_FIELD,
            PROJECT_FIELD, RANDOMIZATION_SALT_FIELD, SCHEDULED_SPLITS_DEFINITION_FIELD, STATUS_FIELD, STATUS_REASON_FIELD,
            TAGS_FIELD, TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final Instant createdTime;

    private final String description;

    private final LaunchExecution execution;

    private final List<LaunchGroup> groups;

    private final Instant lastUpdatedTime;

    private final List<MetricMonitor> metricMonitors;

    private final String name;

    private final String project;

    private final String randomizationSalt;

    private final ScheduledSplitsLaunchDefinition scheduledSplitsDefinition;

    private final String status;

    private final String statusReason;

    private final Map<String, String> tags;

    private final String type;

    private Launch(BuilderImpl builder) {
        this.arn = builder.arn;
        this.createdTime = builder.createdTime;
        this.description = builder.description;
        this.execution = builder.execution;
        this.groups = builder.groups;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.metricMonitors = builder.metricMonitors;
        this.name = builder.name;
        this.project = builder.project;
        this.randomizationSalt = builder.randomizationSalt;
        this.scheduledSplitsDefinition = builder.scheduledSplitsDefinition;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.tags = builder.tags;
        this.type = builder.type;
    }

    /**
     * <p>
     * The ARN of the launch.
     * </p>
     * 
     * @return The ARN of the launch.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The date and time that the launch is created.
     * </p>
     * 
     * @return The date and time that the launch is created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The description of the launch.
     * </p>
     * 
     * @return The description of the launch.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A structure that contains information about the start and end times of the launch.
     * </p>
     * 
     * @return A structure that contains information about the start and end times of the launch.
     */
    public final LaunchExecution execution() {
        return execution;
    }

    /**
     * For responses, this returns true if the service returned a value for the Groups property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasGroups() {
        return groups != null && !(groups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of structures that define the feature variations that are being used in the launch.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGroups} method.
     * </p>
     * 
     * @return An array of structures that define the feature variations that are being used in the launch.
     */
    public final List<LaunchGroup> groups() {
        return groups;
    }

    /**
     * <p>
     * The date and time that the launch was most recently updated.
     * </p>
     * 
     * @return The date and time that the launch was most recently updated.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricMonitors property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricMonitors() {
        return metricMonitors != null && !(metricMonitors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of structures that define the metrics that are being used to monitor the launch performance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricMonitors} method.
     * </p>
     * 
     * @return An array of structures that define the metrics that are being used to monitor the launch performance.
     */
    public final List<MetricMonitor> metricMonitors() {
        return metricMonitors;
    }

    /**
     * <p>
     * The name of the launch.
     * </p>
     * 
     * @return The name of the launch.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The name or ARN of the project that contains the launch.
     * </p>
     * 
     * @return The name or ARN of the project that contains the launch.
     */
    public final String project() {
        return project;
    }

    /**
     * <p>
     * This value is used when Evidently assigns a particular user session to the launch, to help create a randomization
     * ID to determine which variation the user session is served. This randomization ID is a combination of the entity
     * ID and <code>randomizationSalt</code>.
     * </p>
     * 
     * @return This value is used when Evidently assigns a particular user session to the launch, to help create a
     *         randomization ID to determine which variation the user session is served. This randomization ID is a
     *         combination of the entity ID and <code>randomizationSalt</code>.
     */
    public final String randomizationSalt() {
        return randomizationSalt;
    }

    /**
     * <p>
     * An array of structures that define the traffic allocation percentages among the feature variations during each
     * step of the launch.
     * </p>
     * 
     * @return An array of structures that define the traffic allocation percentages among the feature variations during
     *         each step of the launch.
     */
    public final ScheduledSplitsLaunchDefinition scheduledSplitsDefinition() {
        return scheduledSplitsDefinition;
    }

    /**
     * <p>
     * The current state of the launch.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link LaunchStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current state of the launch.
     * @see LaunchStatus
     */
    public final LaunchStatus status() {
        return LaunchStatus.fromValue(status);
    }

    /**
     * <p>
     * The current state of the launch.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link LaunchStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current state of the launch.
     * @see LaunchStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * If the launch was stopped, this is the string that was entered by the person who stopped the launch, to explain
     * why it was stopped.
     * </p>
     * 
     * @return If the launch was stopped, this is the string that was entered by the person who stopped the launch, to
     *         explain why it was stopped.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The list of tag keys and values associated with this launch.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The list of tag keys and values associated with this launch.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The type of launch.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of launch.
     * @see LaunchType
     */
    public final LaunchType type() {
        return LaunchType.fromValue(type);
    }

    /**
     * <p>
     * The type of launch.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of launch.
     * @see LaunchType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(execution());
        hashCode = 31 * hashCode + Objects.hashCode(hasGroups() ? groups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricMonitors() ? metricMonitors() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(project());
        hashCode = 31 * hashCode + Objects.hashCode(randomizationSalt());
        hashCode = 31 * hashCode + Objects.hashCode(scheduledSplitsDefinition());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Launch)) {
            return false;
        }
        Launch other = (Launch) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(description(), other.description()) && Objects.equals(execution(), other.execution())
                && hasGroups() == other.hasGroups() && Objects.equals(groups(), other.groups())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime()) && hasMetricMonitors() == other.hasMetricMonitors()
                && Objects.equals(metricMonitors(), other.metricMonitors()) && Objects.equals(name(), other.name())
                && Objects.equals(project(), other.project()) && Objects.equals(randomizationSalt(), other.randomizationSalt())
                && Objects.equals(scheduledSplitsDefinition(), other.scheduledSplitsDefinition())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Launch").add("Arn", arn()).add("CreatedTime", createdTime()).add("Description", description())
                .add("Execution", execution()).add("Groups", hasGroups() ? groups() : null)
                .add("LastUpdatedTime", lastUpdatedTime()).add("MetricMonitors", hasMetricMonitors() ? metricMonitors() : null)
                .add("Name", name()).add("Project", project()).add("RandomizationSalt", randomizationSalt())
                .add("ScheduledSplitsDefinition", scheduledSplitsDefinition()).add("Status", statusAsString())
                .add("StatusReason", statusReason()).add("Tags", hasTags() ? tags() : null).add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "execution":
            return Optional.ofNullable(clazz.cast(execution()));
        case "groups":
            return Optional.ofNullable(clazz.cast(groups()));
        case "lastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "metricMonitors":
            return Optional.ofNullable(clazz.cast(metricMonitors()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "project":
            return Optional.ofNullable(clazz.cast(project()));
        case "randomizationSalt":
            return Optional.ofNullable(clazz.cast(randomizationSalt()));
        case "scheduledSplitsDefinition":
            return Optional.ofNullable(clazz.cast(scheduledSplitsDefinition()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("arn", ARN_FIELD);
        map.put("createdTime", CREATED_TIME_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("execution", EXECUTION_FIELD);
        map.put("groups", GROUPS_FIELD);
        map.put("lastUpdatedTime", LAST_UPDATED_TIME_FIELD);
        map.put("metricMonitors", METRIC_MONITORS_FIELD);
        map.put("name", NAME_FIELD);
        map.put("project", PROJECT_FIELD);
        map.put("randomizationSalt", RANDOMIZATION_SALT_FIELD);
        map.put("scheduledSplitsDefinition", SCHEDULED_SPLITS_DEFINITION_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("statusReason", STATUS_REASON_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("type", TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Launch, T> g) {
        return obj -> g.apply((Launch) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Launch> {
        /**
         * <p>
         * The ARN of the launch.
         * </p>
         * 
         * @param arn
         *        The ARN of the launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The date and time that the launch is created.
         * </p>
         * 
         * @param createdTime
         *        The date and time that the launch is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The description of the launch.
         * </p>
         * 
         * @param description
         *        The description of the launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A structure that contains information about the start and end times of the launch.
         * </p>
         * 
         * @param execution
         *        A structure that contains information about the start and end times of the launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder execution(LaunchExecution execution);

        /**
         * <p>
         * A structure that contains information about the start and end times of the launch.
         * </p>
         * This is a convenience method that creates an instance of the {@link LaunchExecution.Builder} avoiding the
         * need to create one manually via {@link LaunchExecution#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LaunchExecution.Builder#build()} is called immediately and its
         * result is passed to {@link #execution(LaunchExecution)}.
         * 
         * @param execution
         *        a consumer that will call methods on {@link LaunchExecution.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #execution(LaunchExecution)
         */
        default Builder execution(Consumer<LaunchExecution.Builder> execution) {
            return execution(LaunchExecution.builder().applyMutation(execution).build());
        }

        /**
         * <p>
         * An array of structures that define the feature variations that are being used in the launch.
         * </p>
         * 
         * @param groups
         *        An array of structures that define the feature variations that are being used in the launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(Collection<LaunchGroup> groups);

        /**
         * <p>
         * An array of structures that define the feature variations that are being used in the launch.
         * </p>
         * 
         * @param groups
         *        An array of structures that define the feature variations that are being used in the launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(LaunchGroup... groups);

        /**
         * <p>
         * An array of structures that define the feature variations that are being used in the launch.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.evidently.model.LaunchGroup.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.evidently.model.LaunchGroup#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.evidently.model.LaunchGroup.Builder#build()} is called immediately and
         * its result is passed to {@link #groups(List<LaunchGroup>)}.
         * 
         * @param groups
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.evidently.model.LaunchGroup.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #groups(java.util.Collection<LaunchGroup>)
         */
        Builder groups(Consumer<LaunchGroup.Builder>... groups);

        /**
         * <p>
         * The date and time that the launch was most recently updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The date and time that the launch was most recently updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * An array of structures that define the metrics that are being used to monitor the launch performance.
         * </p>
         * 
         * @param metricMonitors
         *        An array of structures that define the metrics that are being used to monitor the launch performance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricMonitors(Collection<MetricMonitor> metricMonitors);

        /**
         * <p>
         * An array of structures that define the metrics that are being used to monitor the launch performance.
         * </p>
         * 
         * @param metricMonitors
         *        An array of structures that define the metrics that are being used to monitor the launch performance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricMonitors(MetricMonitor... metricMonitors);

        /**
         * <p>
         * An array of structures that define the metrics that are being used to monitor the launch performance.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.evidently.model.MetricMonitor.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.evidently.model.MetricMonitor#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.evidently.model.MetricMonitor.Builder#build()} is called immediately
         * and its result is passed to {@link #metricMonitors(List<MetricMonitor>)}.
         * 
         * @param metricMonitors
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.evidently.model.MetricMonitor.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricMonitors(java.util.Collection<MetricMonitor>)
         */
        Builder metricMonitors(Consumer<MetricMonitor.Builder>... metricMonitors);

        /**
         * <p>
         * The name of the launch.
         * </p>
         * 
         * @param name
         *        The name of the launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The name or ARN of the project that contains the launch.
         * </p>
         * 
         * @param project
         *        The name or ARN of the project that contains the launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder project(String project);

        /**
         * <p>
         * This value is used when Evidently assigns a particular user session to the launch, to help create a
         * randomization ID to determine which variation the user session is served. This randomization ID is a
         * combination of the entity ID and <code>randomizationSalt</code>.
         * </p>
         * 
         * @param randomizationSalt
         *        This value is used when Evidently assigns a particular user session to the launch, to help create a
         *        randomization ID to determine which variation the user session is served. This randomization ID is a
         *        combination of the entity ID and <code>randomizationSalt</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder randomizationSalt(String randomizationSalt);

        /**
         * <p>
         * An array of structures that define the traffic allocation percentages among the feature variations during
         * each step of the launch.
         * </p>
         * 
         * @param scheduledSplitsDefinition
         *        An array of structures that define the traffic allocation percentages among the feature variations
         *        during each step of the launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduledSplitsDefinition(ScheduledSplitsLaunchDefinition scheduledSplitsDefinition);

        /**
         * <p>
         * An array of structures that define the traffic allocation percentages among the feature variations during
         * each step of the launch.
         * </p>
         * This is a convenience method that creates an instance of the {@link ScheduledSplitsLaunchDefinition.Builder}
         * avoiding the need to create one manually via {@link ScheduledSplitsLaunchDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ScheduledSplitsLaunchDefinition.Builder#build()} is called
         * immediately and its result is passed to {@link #scheduledSplitsDefinition(ScheduledSplitsLaunchDefinition)}.
         * 
         * @param scheduledSplitsDefinition
         *        a consumer that will call methods on {@link ScheduledSplitsLaunchDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scheduledSplitsDefinition(ScheduledSplitsLaunchDefinition)
         */
        default Builder scheduledSplitsDefinition(Consumer<ScheduledSplitsLaunchDefinition.Builder> scheduledSplitsDefinition) {
            return scheduledSplitsDefinition(ScheduledSplitsLaunchDefinition.builder().applyMutation(scheduledSplitsDefinition)
                    .build());
        }

        /**
         * <p>
         * The current state of the launch.
         * </p>
         * 
         * @param status
         *        The current state of the launch.
         * @see LaunchStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current state of the launch.
         * </p>
         * 
         * @param status
         *        The current state of the launch.
         * @see LaunchStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchStatus
         */
        Builder status(LaunchStatus status);

        /**
         * <p>
         * If the launch was stopped, this is the string that was entered by the person who stopped the launch, to
         * explain why it was stopped.
         * </p>
         * 
         * @param statusReason
         *        If the launch was stopped, this is the string that was entered by the person who stopped the launch,
         *        to explain why it was stopped.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * The list of tag keys and values associated with this launch.
         * </p>
         * 
         * @param tags
         *        The list of tag keys and values associated with this launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The type of launch.
         * </p>
         * 
         * @param type
         *        The type of launch.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of launch.
         * </p>
         * 
         * @param type
         *        The type of launch.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder type(LaunchType type);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private Instant createdTime;

        private String description;

        private LaunchExecution execution;

        private List<LaunchGroup> groups = DefaultSdkAutoConstructList.getInstance();

        private Instant lastUpdatedTime;

        private List<MetricMonitor> metricMonitors = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private String project;

        private String randomizationSalt;

        private ScheduledSplitsLaunchDefinition scheduledSplitsDefinition;

        private String status;

        private String statusReason;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(Launch model) {
            arn(model.arn);
            createdTime(model.createdTime);
            description(model.description);
            execution(model.execution);
            groups(model.groups);
            lastUpdatedTime(model.lastUpdatedTime);
            metricMonitors(model.metricMonitors);
            name(model.name);
            project(model.project);
            randomizationSalt(model.randomizationSalt);
            scheduledSplitsDefinition(model.scheduledSplitsDefinition);
            status(model.status);
            statusReason(model.statusReason);
            tags(model.tags);
            type(model.type);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final LaunchExecution.Builder getExecution() {
            return execution != null ? execution.toBuilder() : null;
        }

        public final void setExecution(LaunchExecution.BuilderImpl execution) {
            this.execution = execution != null ? execution.build() : null;
        }

        @Override
        public final Builder execution(LaunchExecution execution) {
            this.execution = execution;
            return this;
        }

        public final List<LaunchGroup.Builder> getGroups() {
            List<LaunchGroup.Builder> result = LaunchGroupListCopier.copyToBuilder(this.groups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setGroups(Collection<LaunchGroup.BuilderImpl> groups) {
            this.groups = LaunchGroupListCopier.copyFromBuilder(groups);
        }

        @Override
        public final Builder groups(Collection<LaunchGroup> groups) {
            this.groups = LaunchGroupListCopier.copy(groups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(LaunchGroup... groups) {
            groups(Arrays.asList(groups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(Consumer<LaunchGroup.Builder>... groups) {
            groups(Stream.of(groups).map(c -> LaunchGroup.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final List<MetricMonitor.Builder> getMetricMonitors() {
            List<MetricMonitor.Builder> result = MetricMonitorListCopier.copyToBuilder(this.metricMonitors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMetricMonitors(Collection<MetricMonitor.BuilderImpl> metricMonitors) {
            this.metricMonitors = MetricMonitorListCopier.copyFromBuilder(metricMonitors);
        }

        @Override
        public final Builder metricMonitors(Collection<MetricMonitor> metricMonitors) {
            this.metricMonitors = MetricMonitorListCopier.copy(metricMonitors);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricMonitors(MetricMonitor... metricMonitors) {
            metricMonitors(Arrays.asList(metricMonitors));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricMonitors(Consumer<MetricMonitor.Builder>... metricMonitors) {
            metricMonitors(Stream.of(metricMonitors).map(c -> MetricMonitor.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getProject() {
            return project;
        }

        public final void setProject(String project) {
            this.project = project;
        }

        @Override
        public final Builder project(String project) {
            this.project = project;
            return this;
        }

        public final String getRandomizationSalt() {
            return randomizationSalt;
        }

        public final void setRandomizationSalt(String randomizationSalt) {
            this.randomizationSalt = randomizationSalt;
        }

        @Override
        public final Builder randomizationSalt(String randomizationSalt) {
            this.randomizationSalt = randomizationSalt;
            return this;
        }

        public final ScheduledSplitsLaunchDefinition.Builder getScheduledSplitsDefinition() {
            return scheduledSplitsDefinition != null ? scheduledSplitsDefinition.toBuilder() : null;
        }

        public final void setScheduledSplitsDefinition(ScheduledSplitsLaunchDefinition.BuilderImpl scheduledSplitsDefinition) {
            this.scheduledSplitsDefinition = scheduledSplitsDefinition != null ? scheduledSplitsDefinition.build() : null;
        }

        @Override
        public final Builder scheduledSplitsDefinition(ScheduledSplitsLaunchDefinition scheduledSplitsDefinition) {
            this.scheduledSplitsDefinition = scheduledSplitsDefinition;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(LaunchStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(LaunchType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public Launch build() {
            return new Launch(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
