/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evidently.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure specifies a segment that you have already created, and defines the traffic split for that segment to
 * be used in a launch.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SegmentOverride implements SdkPojo, Serializable, ToCopyableBuilder<SegmentOverride.Builder, SegmentOverride> {
    private static final SdkField<Long> EVALUATION_ORDER_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("evaluationOrder").getter(getter(SegmentOverride::evaluationOrder))
            .setter(setter(Builder::evaluationOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("evaluationOrder").build()).build();

    private static final SdkField<String> SEGMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("segment")
            .getter(getter(SegmentOverride::segment)).setter(setter(Builder::segment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("segment").build()).build();

    private static final SdkField<Map<String, Long>> WEIGHTS_FIELD = SdkField
            .<Map<String, Long>> builder(MarshallingType.MAP)
            .memberName("weights")
            .getter(getter(SegmentOverride::weights))
            .setter(setter(Builder::weights))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("weights").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Long> builder(MarshallingType.LONG)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVALUATION_ORDER_FIELD,
            SEGMENT_FIELD, WEIGHTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Long evaluationOrder;

    private final String segment;

    private final Map<String, Long> weights;

    private SegmentOverride(BuilderImpl builder) {
        this.evaluationOrder = builder.evaluationOrder;
        this.segment = builder.segment;
        this.weights = builder.weights;
    }

    /**
     * <p>
     * A number indicating the order to use to evaluate segment overrides, if there are more than one. Segment overrides
     * with lower numbers are evaluated first.
     * </p>
     * 
     * @return A number indicating the order to use to evaluate segment overrides, if there are more than one. Segment
     *         overrides with lower numbers are evaluated first.
     */
    public final Long evaluationOrder() {
        return evaluationOrder;
    }

    /**
     * <p>
     * The ARN of the segment to use.
     * </p>
     * 
     * @return The ARN of the segment to use.
     */
    public final String segment() {
        return segment;
    }

    /**
     * For responses, this returns true if the service returned a value for the Weights property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasWeights() {
        return weights != null && !(weights instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The traffic allocation percentages among the feature variations to assign to this segment. This is a set of
     * key-value pairs. The keys are variation names. The values represent the amount of traffic to allocate to that
     * variation for this segment. This is expressed in thousandths of a percent, so a weight of 50000 represents 50% of
     * traffic.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasWeights} method.
     * </p>
     * 
     * @return The traffic allocation percentages among the feature variations to assign to this segment. This is a set
     *         of key-value pairs. The keys are variation names. The values represent the amount of traffic to allocate
     *         to that variation for this segment. This is expressed in thousandths of a percent, so a weight of 50000
     *         represents 50% of traffic.
     */
    public final Map<String, Long> weights() {
        return weights;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(evaluationOrder());
        hashCode = 31 * hashCode + Objects.hashCode(segment());
        hashCode = 31 * hashCode + Objects.hashCode(hasWeights() ? weights() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SegmentOverride)) {
            return false;
        }
        SegmentOverride other = (SegmentOverride) obj;
        return Objects.equals(evaluationOrder(), other.evaluationOrder()) && Objects.equals(segment(), other.segment())
                && hasWeights() == other.hasWeights() && Objects.equals(weights(), other.weights());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SegmentOverride").add("EvaluationOrder", evaluationOrder()).add("Segment", segment())
                .add("Weights", hasWeights() ? weights() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "evaluationOrder":
            return Optional.ofNullable(clazz.cast(evaluationOrder()));
        case "segment":
            return Optional.ofNullable(clazz.cast(segment()));
        case "weights":
            return Optional.ofNullable(clazz.cast(weights()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("evaluationOrder", EVALUATION_ORDER_FIELD);
        map.put("segment", SEGMENT_FIELD);
        map.put("weights", WEIGHTS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SegmentOverride, T> g) {
        return obj -> g.apply((SegmentOverride) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SegmentOverride> {
        /**
         * <p>
         * A number indicating the order to use to evaluate segment overrides, if there are more than one. Segment
         * overrides with lower numbers are evaluated first.
         * </p>
         * 
         * @param evaluationOrder
         *        A number indicating the order to use to evaluate segment overrides, if there are more than one.
         *        Segment overrides with lower numbers are evaluated first.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationOrder(Long evaluationOrder);

        /**
         * <p>
         * The ARN of the segment to use.
         * </p>
         * 
         * @param segment
         *        The ARN of the segment to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder segment(String segment);

        /**
         * <p>
         * The traffic allocation percentages among the feature variations to assign to this segment. This is a set of
         * key-value pairs. The keys are variation names. The values represent the amount of traffic to allocate to that
         * variation for this segment. This is expressed in thousandths of a percent, so a weight of 50000 represents
         * 50% of traffic.
         * </p>
         * 
         * @param weights
         *        The traffic allocation percentages among the feature variations to assign to this segment. This is a
         *        set of key-value pairs. The keys are variation names. The values represent the amount of traffic to
         *        allocate to that variation for this segment. This is expressed in thousandths of a percent, so a
         *        weight of 50000 represents 50% of traffic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weights(Map<String, Long> weights);
    }

    static final class BuilderImpl implements Builder {
        private Long evaluationOrder;

        private String segment;

        private Map<String, Long> weights = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SegmentOverride model) {
            evaluationOrder(model.evaluationOrder);
            segment(model.segment);
            weights(model.weights);
        }

        public final Long getEvaluationOrder() {
            return evaluationOrder;
        }

        public final void setEvaluationOrder(Long evaluationOrder) {
            this.evaluationOrder = evaluationOrder;
        }

        @Override
        public final Builder evaluationOrder(Long evaluationOrder) {
            this.evaluationOrder = evaluationOrder;
            return this;
        }

        public final String getSegment() {
            return segment;
        }

        public final void setSegment(String segment) {
            this.segment = segment;
        }

        @Override
        public final Builder segment(String segment) {
            this.segment = segment;
            return this;
        }

        public final Map<String, Long> getWeights() {
            if (weights instanceof SdkAutoConstructMap) {
                return null;
            }
            return weights;
        }

        public final void setWeights(Map<String, Long> weights) {
            this.weights = GroupToWeightMapCopier.copy(weights);
        }

        @Override
        public final Builder weights(Map<String, Long> weights) {
            this.weights = GroupToWeightMapCopier.copy(weights);
            return this;
        }

        @Override
        public SegmentOverride build() {
            return new SegmentOverride(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
