/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evs;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.evs.model.CreateEnvironmentHostRequest;
import software.amazon.awssdk.services.evs.model.CreateEnvironmentHostResponse;
import software.amazon.awssdk.services.evs.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.evs.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.evs.model.DeleteEnvironmentHostRequest;
import software.amazon.awssdk.services.evs.model.DeleteEnvironmentHostResponse;
import software.amazon.awssdk.services.evs.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.evs.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.evs.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.evs.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest;
import software.amazon.awssdk.services.evs.model.ListEnvironmentHostsResponse;
import software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest;
import software.amazon.awssdk.services.evs.model.ListEnvironmentVlansResponse;
import software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.evs.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.evs.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.evs.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.evs.model.TagResourceRequest;
import software.amazon.awssdk.services.evs.model.TagResourceResponse;
import software.amazon.awssdk.services.evs.model.UntagResourceRequest;
import software.amazon.awssdk.services.evs.model.UntagResourceResponse;
import software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsPublisher;
import software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansPublisher;
import software.amazon.awssdk.services.evs.paginators.ListEnvironmentsPublisher;

/**
 * Service client for accessing EVS asynchronously. This can be created using the static {@link #builder()} method.The
 * asynchronous client performs non-blocking I/O when configured with any {@code SdkAsyncHttpClient} supported in the
 * SDK. However, full non-blocking is not guaranteed as the async client may perform blocking calls in some cases such
 * as credentials retrieval and endpoint discovery as part of the async API call.
 *
 * <p>
 * Amazon Elastic VMware Service (Amazon EVS) is a service that you can use to deploy a VMware Cloud Foundation (VCF)
 * software environment directly on EC2 bare metal instances within an Amazon Virtual Private Cloud (VPC).
 * </p>
 * <p>
 * Workloads running on Amazon EVS are fully compatible with workloads running on any standard VMware vSphere
 * environment. This means that you can migrate any VMware-based workload to Amazon EVS without workload modification.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface EvsAsyncClient extends AwsClient {
    String SERVICE_NAME = "evs";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "evs";

    /**
     * <p>
     * Creates an Amazon EVS environment that runs VCF software, such as SDDC Manager, NSX Manager, and vCenter Server.
     * </p>
     * <p>
     * During environment creation, Amazon EVS performs validations on DNS settings, provisions VLAN subnets and hosts,
     * and deploys the supplied version of VCF.
     * </p>
     * <p>
     * It can take several hours to create an environment. After the deployment completes, you can configure VCF in the
     * vSphere user interface according to your needs.
     * </p>
     * <note>
     * <p>
     * You cannot use the <code>dedicatedHostId</code> and <code>placementGroupId</code> parameters together in the same
     * <code>CreateEnvironment</code> action. This results in a <code>ValidationException</code> response.
     * </p>
     * </note>
     *
     * @param createEnvironmentRequest
     * @return A Java Future containing the result of the CreateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/CreateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateEnvironmentResponse> createEnvironment(CreateEnvironmentRequest createEnvironmentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an Amazon EVS environment that runs VCF software, such as SDDC Manager, NSX Manager, and vCenter Server.
     * </p>
     * <p>
     * During environment creation, Amazon EVS performs validations on DNS settings, provisions VLAN subnets and hosts,
     * and deploys the supplied version of VCF.
     * </p>
     * <p>
     * It can take several hours to create an environment. After the deployment completes, you can configure VCF in the
     * vSphere user interface according to your needs.
     * </p>
     * <note>
     * <p>
     * You cannot use the <code>dedicatedHostId</code> and <code>placementGroupId</code> parameters together in the same
     * <code>CreateEnvironment</code> action. This results in a <code>ValidationException</code> response.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateEnvironmentRequest.Builder} avoiding the need
     * to create one manually via {@link CreateEnvironmentRequest#builder()}
     * </p>
     *
     * @param createEnvironmentRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.CreateEnvironmentRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/CreateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateEnvironmentResponse> createEnvironment(
            Consumer<CreateEnvironmentRequest.Builder> createEnvironmentRequest) {
        return createEnvironment(CreateEnvironmentRequest.builder().applyMutation(createEnvironmentRequest).build());
    }

    /**
     * <p>
     * Creates an ESXi host and adds it to an Amazon EVS environment. Amazon EVS supports 4-16 hosts per environment.
     * </p>
     * <p>
     * This action can only be used after the Amazon EVS environment is deployed.
     * </p>
     * <p>
     * You can use the <code>dedicatedHostId</code> parameter to specify an Amazon EC2 Dedicated Host for ESXi host
     * creation.
     * </p>
     * <p>
     * You can use the <code>placementGroupId</code> parameter to specify a cluster or partition placement group to
     * launch EC2 instances into.
     * </p>
     * <note>
     * <p>
     * You cannot use the <code>dedicatedHostId</code> and <code>placementGroupId</code> parameters together in the same
     * <code>CreateEnvironmentHost</code> action. This results in a <code>ValidationException</code> response.
     * </p>
     * </note>
     *
     * @param createEnvironmentHostRequest
     * @return A Java Future containing the result of the CreateEnvironmentHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The <code>CreateEnvironmentHost</code> operation couldn't be performed because
     *         the service is throttling requests. This exception is thrown when the <code>CreateEnvironmentHost</code>
     *         request exceeds concurrency of 1 transaction per second (TPS).</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.CreateEnvironmentHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/CreateEnvironmentHost" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateEnvironmentHostResponse> createEnvironmentHost(
            CreateEnvironmentHostRequest createEnvironmentHostRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an ESXi host and adds it to an Amazon EVS environment. Amazon EVS supports 4-16 hosts per environment.
     * </p>
     * <p>
     * This action can only be used after the Amazon EVS environment is deployed.
     * </p>
     * <p>
     * You can use the <code>dedicatedHostId</code> parameter to specify an Amazon EC2 Dedicated Host for ESXi host
     * creation.
     * </p>
     * <p>
     * You can use the <code>placementGroupId</code> parameter to specify a cluster or partition placement group to
     * launch EC2 instances into.
     * </p>
     * <note>
     * <p>
     * You cannot use the <code>dedicatedHostId</code> and <code>placementGroupId</code> parameters together in the same
     * <code>CreateEnvironmentHost</code> action. This results in a <code>ValidationException</code> response.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateEnvironmentHostRequest.Builder} avoiding the
     * need to create one manually via {@link CreateEnvironmentHostRequest#builder()}
     * </p>
     *
     * @param createEnvironmentHostRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.CreateEnvironmentHostRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the CreateEnvironmentHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The <code>CreateEnvironmentHost</code> operation couldn't be performed because
     *         the service is throttling requests. This exception is thrown when the <code>CreateEnvironmentHost</code>
     *         request exceeds concurrency of 1 transaction per second (TPS).</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.CreateEnvironmentHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/CreateEnvironmentHost" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateEnvironmentHostResponse> createEnvironmentHost(
            Consumer<CreateEnvironmentHostRequest.Builder> createEnvironmentHostRequest) {
        return createEnvironmentHost(CreateEnvironmentHostRequest.builder().applyMutation(createEnvironmentHostRequest).build());
    }

    /**
     * <p>
     * Deletes an Amazon EVS environment.
     * </p>
     * <p>
     * Amazon EVS environments will only be enabled for deletion once the hosts are deleted. You can delete hosts using
     * the <code>DeleteEnvironmentHost</code> action.
     * </p>
     * <p>
     * Environment deletion also deletes the associated Amazon EVS VLAN subnets and Amazon Web Services Secrets Manager
     * secrets that Amazon EVS created. Amazon Web Services resources that you create are not deleted. These resources
     * may continue to incur costs.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return A Java Future containing the result of the DeleteEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/DeleteEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteEnvironmentResponse> deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes an Amazon EVS environment.
     * </p>
     * <p>
     * Amazon EVS environments will only be enabled for deletion once the hosts are deleted. You can delete hosts using
     * the <code>DeleteEnvironmentHost</code> action.
     * </p>
     * <p>
     * Environment deletion also deletes the associated Amazon EVS VLAN subnets and Amazon Web Services Secrets Manager
     * secrets that Amazon EVS created. Amazon Web Services resources that you create are not deleted. These resources
     * may continue to incur costs.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteEnvironmentRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteEnvironmentRequest#builder()}
     * </p>
     *
     * @param deleteEnvironmentRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.DeleteEnvironmentRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/DeleteEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteEnvironmentResponse> deleteEnvironment(
            Consumer<DeleteEnvironmentRequest.Builder> deleteEnvironmentRequest) {
        return deleteEnvironment(DeleteEnvironmentRequest.builder().applyMutation(deleteEnvironmentRequest).build());
    }

    /**
     * <p>
     * Deletes a host from an Amazon EVS environment.
     * </p>
     * <note>
     * <p>
     * Before deleting a host, you must unassign and decommission the host from within the SDDC Manager user interface.
     * Not doing so could impact the availability of your virtual machines or result in data loss.
     * </p>
     * </note>
     *
     * @param deleteEnvironmentHostRequest
     * @return A Java Future containing the result of the DeleteEnvironmentHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.DeleteEnvironmentHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/DeleteEnvironmentHost" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteEnvironmentHostResponse> deleteEnvironmentHost(
            DeleteEnvironmentHostRequest deleteEnvironmentHostRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a host from an Amazon EVS environment.
     * </p>
     * <note>
     * <p>
     * Before deleting a host, you must unassign and decommission the host from within the SDDC Manager user interface.
     * Not doing so could impact the availability of your virtual machines or result in data loss.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteEnvironmentHostRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteEnvironmentHostRequest#builder()}
     * </p>
     *
     * @param deleteEnvironmentHostRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.DeleteEnvironmentHostRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteEnvironmentHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.DeleteEnvironmentHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/DeleteEnvironmentHost" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteEnvironmentHostResponse> deleteEnvironmentHost(
            Consumer<DeleteEnvironmentHostRequest.Builder> deleteEnvironmentHostRequest) {
        return deleteEnvironmentHost(DeleteEnvironmentHostRequest.builder().applyMutation(deleteEnvironmentHostRequest).build());
    }

    /**
     * <p>
     * Returns a description of the specified environment.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return A Java Future containing the result of the GetEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetEnvironmentResponse> getEnvironment(GetEnvironmentRequest getEnvironmentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a description of the specified environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetEnvironmentRequest.Builder} avoiding the need to
     * create one manually via {@link GetEnvironmentRequest#builder()}
     * </p>
     *
     * @param getEnvironmentRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.GetEnvironmentRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetEnvironmentResponse> getEnvironment(Consumer<GetEnvironmentRequest.Builder> getEnvironmentRequest) {
        return getEnvironment(GetEnvironmentRequest.builder().applyMutation(getEnvironmentRequest).build());
    }

    /**
     * <p>
     * List the hosts within an environment.
     * </p>
     *
     * @param listEnvironmentHostsRequest
     * @return A Java Future containing the result of the ListEnvironmentHosts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironmentHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentHosts" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListEnvironmentHostsResponse> listEnvironmentHosts(
            ListEnvironmentHostsRequest listEnvironmentHostsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List the hosts within an environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentHostsRequest.Builder} avoiding the
     * need to create one manually via {@link ListEnvironmentHostsRequest#builder()}
     * </p>
     *
     * @param listEnvironmentHostsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListEnvironmentHosts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironmentHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentHosts" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListEnvironmentHostsResponse> listEnvironmentHosts(
            Consumer<ListEnvironmentHostsRequest.Builder> listEnvironmentHostsRequest) {
        return listEnvironmentHosts(ListEnvironmentHostsRequest.builder().applyMutation(listEnvironmentHostsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEnvironmentHosts(software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsPublisher publisher = client.listEnvironmentHostsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsPublisher publisher = client.listEnvironmentHostsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.evs.model.ListEnvironmentHostsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.evs.model.ListEnvironmentHostsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironmentHosts(software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEnvironmentHostsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironmentHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentHosts" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentHostsPublisher listEnvironmentHostsPaginator(ListEnvironmentHostsRequest listEnvironmentHostsRequest) {
        return new ListEnvironmentHostsPublisher(this, listEnvironmentHostsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEnvironmentHosts(software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsPublisher publisher = client.listEnvironmentHostsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentHostsPublisher publisher = client.listEnvironmentHostsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.evs.model.ListEnvironmentHostsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.evs.model.ListEnvironmentHostsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironmentHosts(software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentHostsRequest.Builder} avoiding the
     * need to create one manually via {@link ListEnvironmentHostsRequest#builder()}
     * </p>
     *
     * @param listEnvironmentHostsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentHostsRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironmentHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentHosts" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentHostsPublisher listEnvironmentHostsPaginator(
            Consumer<ListEnvironmentHostsRequest.Builder> listEnvironmentHostsRequest) {
        return listEnvironmentHostsPaginator(ListEnvironmentHostsRequest.builder().applyMutation(listEnvironmentHostsRequest)
                .build());
    }

    /**
     * <p>
     * Lists environment VLANs that are associated with the specified environment.
     * </p>
     *
     * @param listEnvironmentVlansRequest
     * @return A Java Future containing the result of the ListEnvironmentVlans operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironmentVlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentVlans" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListEnvironmentVlansResponse> listEnvironmentVlans(
            ListEnvironmentVlansRequest listEnvironmentVlansRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists environment VLANs that are associated with the specified environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentVlansRequest.Builder} avoiding the
     * need to create one manually via {@link ListEnvironmentVlansRequest#builder()}
     * </p>
     *
     * @param listEnvironmentVlansRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListEnvironmentVlans operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironmentVlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentVlans" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListEnvironmentVlansResponse> listEnvironmentVlans(
            Consumer<ListEnvironmentVlansRequest.Builder> listEnvironmentVlansRequest) {
        return listEnvironmentVlans(ListEnvironmentVlansRequest.builder().applyMutation(listEnvironmentVlansRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEnvironmentVlans(software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansPublisher publisher = client.listEnvironmentVlansPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansPublisher publisher = client.listEnvironmentVlansPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.evs.model.ListEnvironmentVlansResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.evs.model.ListEnvironmentVlansResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironmentVlans(software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEnvironmentVlansRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironmentVlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentVlans" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentVlansPublisher listEnvironmentVlansPaginator(ListEnvironmentVlansRequest listEnvironmentVlansRequest) {
        return new ListEnvironmentVlansPublisher(this, listEnvironmentVlansRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEnvironmentVlans(software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansPublisher publisher = client.listEnvironmentVlansPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentVlansPublisher publisher = client.listEnvironmentVlansPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.evs.model.ListEnvironmentVlansResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.evs.model.ListEnvironmentVlansResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironmentVlans(software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentVlansRequest.Builder} avoiding the
     * need to create one manually via {@link ListEnvironmentVlansRequest#builder()}
     * </p>
     *
     * @param listEnvironmentVlansRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentVlansRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironmentVlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironmentVlans" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentVlansPublisher listEnvironmentVlansPaginator(
            Consumer<ListEnvironmentVlansRequest.Builder> listEnvironmentVlansRequest) {
        return listEnvironmentVlansPaginator(ListEnvironmentVlansRequest.builder().applyMutation(listEnvironmentVlansRequest)
                .build());
    }

    /**
     * <p>
     * Lists the Amazon EVS environments in your Amazon Web Services account in the specified Amazon Web Services
     * Region.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A Java Future containing the result of the ListEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListEnvironmentsResponse> listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the Amazon EVS environments in your Amazon Web Services account in the specified Amazon Web Services
     * Region.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentsRequest.Builder} avoiding the need
     * to create one manually via {@link ListEnvironmentsRequest#builder()}
     * </p>
     *
     * @param listEnvironmentsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListEnvironmentsResponse> listEnvironments(
            Consumer<ListEnvironmentsRequest.Builder> listEnvironmentsRequest) {
        return listEnvironments(ListEnvironmentsRequest.builder().applyMutation(listEnvironmentsRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listEnvironments(software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentsPublisher publisher = client.listEnvironmentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentsPublisher publisher = client.listEnvironmentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.evs.model.ListEnvironmentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.evs.model.ListEnvironmentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironments(software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest)} operation.</b>
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentsPublisher listEnvironmentsPaginator(ListEnvironmentsRequest listEnvironmentsRequest) {
        return new ListEnvironmentsPublisher(this, listEnvironmentsRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listEnvironments(software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentsPublisher publisher = client.listEnvironmentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evs.paginators.ListEnvironmentsPublisher publisher = client.listEnvironmentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.evs.model.ListEnvironmentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.evs.model.ListEnvironmentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironments(software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnvironmentsRequest.Builder} avoiding the need
     * to create one manually via {@link ListEnvironmentsRequest#builder()}
     * </p>
     *
     * @param listEnvironmentsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListEnvironmentsRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the specified constraints. You will see this exception
     *         if invalid inputs are provided for any of the Amazon EVS environment operations, or if a list operation
     *         is performed on an environment resource that is still initializing.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEnvironmentsPublisher listEnvironmentsPaginator(Consumer<ListEnvironmentsRequest.Builder> listEnvironmentsRequest) {
        return listEnvironmentsPaginator(ListEnvironmentsRequest.builder().applyMutation(listEnvironmentsRequest).build());
    }

    /**
     * <p>
     * Lists the tags for an Amazon EVS resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the tags for an Amazon EVS resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.ListTagsForResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Associates the specified tags to an Amazon EVS resource with the specified <code>resourceArn</code>. If existing
     * tags on a resource are not specified in the request parameters, they aren't changed. When a resource is deleted,
     * the tags associated with that resource are also deleted. Tags that you create for Amazon EVS resources don't
     * propagate to any other resources associated with the environment. For example, if you tag an environment with
     * this operation, that tag doesn't automatically propagate to the VLAN subnets and hosts associated with the
     * environment.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The number of one or more Amazon EVS resources exceeds the maximum
     *         allowed. For a list of Amazon EVS quotas, see <a
     *         href="https://docs.aws.amazon.com/evs/latest/userguide/service-quotas-evs.html">Amazon EVS endpoints and
     *         quotas</a> in the <i>Amazon EVS User Guide</i>. Delete some resources or request an increase in your
     *         service quota. To request an increase, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html">Amazon Web Services Service
     *         Quotas</a> in the <i>Amazon Web Services General Reference Guide</i>.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>TooManyTagsException
     *         <p>
     *         <code>TooManyTagsException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ServiceQuotaExceededException.html">
     *         <code>ServiceQuotaExceededException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         A service resource associated with the request has more than 200 tags.</li>
     *         <li>TagPolicyException
     *         <p>
     *         <code>TagPolicyException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ValidationException.html">
     *         <code>ValidationException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         The request doesn't comply with IAM tag policy. Correct your request and then retry it.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Associates the specified tags to an Amazon EVS resource with the specified <code>resourceArn</code>. If existing
     * tags on a resource are not specified in the request parameters, they aren't changed. When a resource is deleted,
     * the tags associated with that resource are also deleted. Tags that you create for Amazon EVS resources don't
     * propagate to any other resources associated with the environment. For example, if you tag an environment with
     * this operation, that tag doesn't automatically propagate to the VLAN subnets and hosts associated with the
     * environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.TagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The number of one or more Amazon EVS resources exceeds the maximum
     *         allowed. For a list of Amazon EVS quotas, see <a
     *         href="https://docs.aws.amazon.com/evs/latest/userguide/service-quotas-evs.html">Amazon EVS endpoints and
     *         quotas</a> in the <i>Amazon EVS User Guide</i>. Delete some resources or request an increase in your
     *         service quota. To request an increase, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html">Amazon Web Services Service
     *         Quotas</a> in the <i>Amazon Web Services General Reference Guide</i>.</li>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>TooManyTagsException
     *         <p>
     *         <code>TooManyTagsException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ServiceQuotaExceededException.html">
     *         <code>ServiceQuotaExceededException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         A service resource associated with the request has more than 200 tags.</li>
     *         <li>TagPolicyException
     *         <p>
     *         <code>TagPolicyException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ValidationException.html">
     *         <code>ValidationException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         The request doesn't comply with IAM tag policy. Correct your request and then retry it.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Deletes specified tags from an Amazon EVS resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>TagPolicyException
     *         <p>
     *         <code>TagPolicyException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ValidationException.html">
     *         <code>ValidationException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         The request doesn't comply with IAM tag policy. Correct your request and then retry it.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes specified tags from an Amazon EVS resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.evs.model.UntagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException A service resource associated with the request could not be found. The
     *         resource might not be specified correctly, or it may have a <code>state</code> of <code>DELETED</code>.</li>
     *         <li>TagPolicyException
     *         <p>
     *         <code>TagPolicyException</code> is deprecated. See <a
     *         href="https://docs.aws.amazon.com/evs/latest/APIReference/API_ValidationException.html">
     *         <code>ValidationException</code> </a> instead.
     *         </p>
     *         </note>
     *         <p>
     *         The request doesn't comply with IAM tag policy. Correct your request and then retry it.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EvsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evs-2023-07-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    @Override
    default EvsServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link EvsAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static EvsAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link EvsAsyncClient}.
     */
    static EvsAsyncClientBuilder builder() {
        return new DefaultEvsAsyncClientBuilder();
    }
}
