/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.finspace.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The structure of database cache configuration that is used for mapping database paths to cache types in clusters.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KxDatabaseCacheConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<KxDatabaseCacheConfiguration.Builder, KxDatabaseCacheConfiguration> {
    private static final SdkField<String> CACHE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("cacheType").getter(getter(KxDatabaseCacheConfiguration::cacheType)).setter(setter(Builder::cacheType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cacheType").build()).build();

    private static final SdkField<List<String>> DB_PATHS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("dbPaths")
            .getter(getter(KxDatabaseCacheConfiguration::dbPaths))
            .setter(setter(Builder::dbPaths))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dbPaths").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DATAVIEW_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dataviewName").getter(getter(KxDatabaseCacheConfiguration::dataviewName))
            .setter(setter(Builder::dataviewName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataviewName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CACHE_TYPE_FIELD,
            DB_PATHS_FIELD, DATAVIEW_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String cacheType;

    private final List<String> dbPaths;

    private final String dataviewName;

    private KxDatabaseCacheConfiguration(BuilderImpl builder) {
        this.cacheType = builder.cacheType;
        this.dbPaths = builder.dbPaths;
        this.dataviewName = builder.dataviewName;
    }

    /**
     * <p>
     * The type of disk cache. This parameter is used to map the database path to cache storage. The valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * CACHE_1000 – This type provides at least 1000 MB/s disk access throughput.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The type of disk cache. This parameter is used to map the database path to cache storage. The valid
     *         values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         CACHE_1000 – This type provides at least 1000 MB/s disk access throughput.
     *         </p>
     *         </li>
     */
    public final String cacheType() {
        return cacheType;
    }

    /**
     * For responses, this returns true if the service returned a value for the DbPaths property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDbPaths() {
        return dbPaths != null && !(dbPaths instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the portions of database that will be loaded into the cache for access.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDbPaths} method.
     * </p>
     * 
     * @return Specifies the portions of database that will be loaded into the cache for access.
     */
    public final List<String> dbPaths() {
        return dbPaths;
    }

    /**
     * <p>
     * The name of the dataview to be used for caching historical data on disk.
     * </p>
     * 
     * @return The name of the dataview to be used for caching historical data on disk.
     */
    public final String dataviewName() {
        return dataviewName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cacheType());
        hashCode = 31 * hashCode + Objects.hashCode(hasDbPaths() ? dbPaths() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dataviewName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KxDatabaseCacheConfiguration)) {
            return false;
        }
        KxDatabaseCacheConfiguration other = (KxDatabaseCacheConfiguration) obj;
        return Objects.equals(cacheType(), other.cacheType()) && hasDbPaths() == other.hasDbPaths()
                && Objects.equals(dbPaths(), other.dbPaths()) && Objects.equals(dataviewName(), other.dataviewName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KxDatabaseCacheConfiguration").add("CacheType", cacheType())
                .add("DbPaths", hasDbPaths() ? dbPaths() : null).add("DataviewName", dataviewName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "cacheType":
            return Optional.ofNullable(clazz.cast(cacheType()));
        case "dbPaths":
            return Optional.ofNullable(clazz.cast(dbPaths()));
        case "dataviewName":
            return Optional.ofNullable(clazz.cast(dataviewName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<KxDatabaseCacheConfiguration, T> g) {
        return obj -> g.apply((KxDatabaseCacheConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KxDatabaseCacheConfiguration> {
        /**
         * <p>
         * The type of disk cache. This parameter is used to map the database path to cache storage. The valid values
         * are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * CACHE_1000 – This type provides at least 1000 MB/s disk access throughput.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cacheType
         *        The type of disk cache. This parameter is used to map the database path to cache storage. The valid
         *        values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        CACHE_1000 – This type provides at least 1000 MB/s disk access throughput.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheType(String cacheType);

        /**
         * <p>
         * Specifies the portions of database that will be loaded into the cache for access.
         * </p>
         * 
         * @param dbPaths
         *        Specifies the portions of database that will be loaded into the cache for access.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbPaths(Collection<String> dbPaths);

        /**
         * <p>
         * Specifies the portions of database that will be loaded into the cache for access.
         * </p>
         * 
         * @param dbPaths
         *        Specifies the portions of database that will be loaded into the cache for access.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbPaths(String... dbPaths);

        /**
         * <p>
         * The name of the dataview to be used for caching historical data on disk.
         * </p>
         * 
         * @param dataviewName
         *        The name of the dataview to be used for caching historical data on disk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataviewName(String dataviewName);
    }

    static final class BuilderImpl implements Builder {
        private String cacheType;

        private List<String> dbPaths = DefaultSdkAutoConstructList.getInstance();

        private String dataviewName;

        private BuilderImpl() {
        }

        private BuilderImpl(KxDatabaseCacheConfiguration model) {
            cacheType(model.cacheType);
            dbPaths(model.dbPaths);
            dataviewName(model.dataviewName);
        }

        public final String getCacheType() {
            return cacheType;
        }

        public final void setCacheType(String cacheType) {
            this.cacheType = cacheType;
        }

        @Override
        public final Builder cacheType(String cacheType) {
            this.cacheType = cacheType;
            return this;
        }

        public final Collection<String> getDbPaths() {
            if (dbPaths instanceof SdkAutoConstructList) {
                return null;
            }
            return dbPaths;
        }

        public final void setDbPaths(Collection<String> dbPaths) {
            this.dbPaths = DbPathsCopier.copy(dbPaths);
        }

        @Override
        public final Builder dbPaths(Collection<String> dbPaths) {
            this.dbPaths = DbPathsCopier.copy(dbPaths);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dbPaths(String... dbPaths) {
            dbPaths(Arrays.asList(dbPaths));
            return this;
        }

        public final String getDataviewName() {
            return dataviewName;
        }

        public final void setDataviewName(String dataviewName) {
            this.dataviewName = dataviewName;
        }

        @Override
        public final Builder dataviewName(String dataviewName) {
            this.dataviewName = dataviewName;
            return this;
        }

        @Override
        public KxDatabaseCacheConfiguration build() {
            return new KxDatabaseCacheConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
