/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.finspace.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration that contains the database path of the data that you want to place on each selected volume. Each
 * segment must have a unique database path for each volume. If you do not explicitly specify any database path for a
 * volume, they are accessible from the cluster through the default S3/object store segment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KxDataviewSegmentConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<KxDataviewSegmentConfiguration.Builder, KxDataviewSegmentConfiguration> {
    private static final SdkField<List<String>> DB_PATHS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("dbPaths")
            .getter(getter(KxDataviewSegmentConfiguration::dbPaths))
            .setter(setter(Builder::dbPaths))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dbPaths").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> VOLUME_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("volumeName").getter(getter(KxDataviewSegmentConfiguration::volumeName))
            .setter(setter(Builder::volumeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumeName").build()).build();

    private static final SdkField<Boolean> ON_DEMAND_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("onDemand").getter(getter(KxDataviewSegmentConfiguration::onDemand)).setter(setter(Builder::onDemand))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("onDemand").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DB_PATHS_FIELD,
            VOLUME_NAME_FIELD, ON_DEMAND_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> dbPaths;

    private final String volumeName;

    private final Boolean onDemand;

    private KxDataviewSegmentConfiguration(BuilderImpl builder) {
        this.dbPaths = builder.dbPaths;
        this.volumeName = builder.volumeName;
        this.onDemand = builder.onDemand;
    }

    /**
     * For responses, this returns true if the service returned a value for the DbPaths property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDbPaths() {
        return dbPaths != null && !(dbPaths instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The database path of the data that you want to place on each selected volume for the segment. Each segment must
     * have a unique database path for each volume.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDbPaths} method.
     * </p>
     * 
     * @return The database path of the data that you want to place on each selected volume for the segment. Each
     *         segment must have a unique database path for each volume.
     */
    public final List<String> dbPaths() {
        return dbPaths;
    }

    /**
     * <p>
     * The name of the volume where you want to add data.
     * </p>
     * 
     * @return The name of the volume where you want to add data.
     */
    public final String volumeName() {
        return volumeName;
    }

    /**
     * <p>
     * Enables on-demand caching on the selected database path when a particular file or a column of the database is
     * accessed. When on demand caching is <b>True</b>, dataviews perform minimal loading of files on the filesystem as
     * needed. When it is set to <b>False</b>, everything is cached. The default value is <b>False</b>.
     * </p>
     * 
     * @return Enables on-demand caching on the selected database path when a particular file or a column of the
     *         database is accessed. When on demand caching is <b>True</b>, dataviews perform minimal loading of files
     *         on the filesystem as needed. When it is set to <b>False</b>, everything is cached. The default value is
     *         <b>False</b>.
     */
    public final Boolean onDemand() {
        return onDemand;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasDbPaths() ? dbPaths() : null);
        hashCode = 31 * hashCode + Objects.hashCode(volumeName());
        hashCode = 31 * hashCode + Objects.hashCode(onDemand());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KxDataviewSegmentConfiguration)) {
            return false;
        }
        KxDataviewSegmentConfiguration other = (KxDataviewSegmentConfiguration) obj;
        return hasDbPaths() == other.hasDbPaths() && Objects.equals(dbPaths(), other.dbPaths())
                && Objects.equals(volumeName(), other.volumeName()) && Objects.equals(onDemand(), other.onDemand());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KxDataviewSegmentConfiguration").add("DbPaths", hasDbPaths() ? dbPaths() : null)
                .add("VolumeName", volumeName()).add("OnDemand", onDemand()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "dbPaths":
            return Optional.ofNullable(clazz.cast(dbPaths()));
        case "volumeName":
            return Optional.ofNullable(clazz.cast(volumeName()));
        case "onDemand":
            return Optional.ofNullable(clazz.cast(onDemand()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("dbPaths", DB_PATHS_FIELD);
        map.put("volumeName", VOLUME_NAME_FIELD);
        map.put("onDemand", ON_DEMAND_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<KxDataviewSegmentConfiguration, T> g) {
        return obj -> g.apply((KxDataviewSegmentConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KxDataviewSegmentConfiguration> {
        /**
         * <p>
         * The database path of the data that you want to place on each selected volume for the segment. Each segment
         * must have a unique database path for each volume.
         * </p>
         * 
         * @param dbPaths
         *        The database path of the data that you want to place on each selected volume for the segment. Each
         *        segment must have a unique database path for each volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbPaths(Collection<String> dbPaths);

        /**
         * <p>
         * The database path of the data that you want to place on each selected volume for the segment. Each segment
         * must have a unique database path for each volume.
         * </p>
         * 
         * @param dbPaths
         *        The database path of the data that you want to place on each selected volume for the segment. Each
         *        segment must have a unique database path for each volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbPaths(String... dbPaths);

        /**
         * <p>
         * The name of the volume where you want to add data.
         * </p>
         * 
         * @param volumeName
         *        The name of the volume where you want to add data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeName(String volumeName);

        /**
         * <p>
         * Enables on-demand caching on the selected database path when a particular file or a column of the database is
         * accessed. When on demand caching is <b>True</b>, dataviews perform minimal loading of files on the filesystem
         * as needed. When it is set to <b>False</b>, everything is cached. The default value is <b>False</b>.
         * </p>
         * 
         * @param onDemand
         *        Enables on-demand caching on the selected database path when a particular file or a column of the
         *        database is accessed. When on demand caching is <b>True</b>, dataviews perform minimal loading of
         *        files on the filesystem as needed. When it is set to <b>False</b>, everything is cached. The default
         *        value is <b>False</b>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onDemand(Boolean onDemand);
    }

    static final class BuilderImpl implements Builder {
        private List<String> dbPaths = DefaultSdkAutoConstructList.getInstance();

        private String volumeName;

        private Boolean onDemand;

        private BuilderImpl() {
        }

        private BuilderImpl(KxDataviewSegmentConfiguration model) {
            dbPaths(model.dbPaths);
            volumeName(model.volumeName);
            onDemand(model.onDemand);
        }

        public final Collection<String> getDbPaths() {
            if (dbPaths instanceof SdkAutoConstructList) {
                return null;
            }
            return dbPaths;
        }

        public final void setDbPaths(Collection<String> dbPaths) {
            this.dbPaths = SegmentConfigurationDbPathListCopier.copy(dbPaths);
        }

        @Override
        public final Builder dbPaths(Collection<String> dbPaths) {
            this.dbPaths = SegmentConfigurationDbPathListCopier.copy(dbPaths);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dbPaths(String... dbPaths) {
            dbPaths(Arrays.asList(dbPaths));
            return this;
        }

        public final String getVolumeName() {
            return volumeName;
        }

        public final void setVolumeName(String volumeName) {
            this.volumeName = volumeName;
        }

        @Override
        public final Builder volumeName(String volumeName) {
            this.volumeName = volumeName;
            return this;
        }

        public final Boolean getOnDemand() {
            return onDemand;
        }

        public final void setOnDemand(Boolean onDemand) {
            this.onDemand = onDemand;
        }

        @Override
        public final Builder onDemand(Boolean onDemand) {
            this.onDemand = onDemand;
            return this;
        }

        @Override
        public KxDataviewSegmentConfiguration build() {
            return new KxDataviewSegmentConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
