/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.finspacedata;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.finspacedata.model.AccessDeniedException;
import software.amazon.awssdk.services.finspacedata.model.CreateChangesetRequest;
import software.amazon.awssdk.services.finspacedata.model.CreateChangesetResponse;
import software.amazon.awssdk.services.finspacedata.model.FinspaceDataException;
import software.amazon.awssdk.services.finspacedata.model.GetProgrammaticAccessCredentialsRequest;
import software.amazon.awssdk.services.finspacedata.model.GetProgrammaticAccessCredentialsResponse;
import software.amazon.awssdk.services.finspacedata.model.GetWorkingLocationRequest;
import software.amazon.awssdk.services.finspacedata.model.GetWorkingLocationResponse;
import software.amazon.awssdk.services.finspacedata.model.InternalServerException;
import software.amazon.awssdk.services.finspacedata.model.ResourceNotFoundException;
import software.amazon.awssdk.services.finspacedata.model.ThrottlingException;
import software.amazon.awssdk.services.finspacedata.model.ValidationException;
import software.amazon.awssdk.services.finspacedata.transform.CreateChangesetRequestMarshaller;
import software.amazon.awssdk.services.finspacedata.transform.GetProgrammaticAccessCredentialsRequestMarshaller;
import software.amazon.awssdk.services.finspacedata.transform.GetWorkingLocationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link FinspaceDataAsyncClient}.
 *
 * @see FinspaceDataAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFinspaceDataAsyncClient implements FinspaceDataAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultFinspaceDataAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultFinspaceDataAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new changeset in a FinSpace dataset.
     * </p>
     *
     * @param createChangesetRequest
     * @return A Java Future containing the result of the CreateChangeset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceDataAsyncClient.CreateChangeset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2020-07-13/CreateChangeset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChangesetResponse> createChangeset(CreateChangesetRequest createChangesetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChangesetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChangeset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChangesetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateChangesetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChangesetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChangesetRequest, CreateChangesetResponse>()
                            .withOperationName("CreateChangeset")
                            .withMarshaller(new CreateChangesetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createChangesetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createChangesetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateChangesetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Request programmatic credentials to use with Habanero SDK.
     * </p>
     *
     * @param getProgrammaticAccessCredentialsRequest
     * @return A Java Future containing the result of the GetProgrammaticAccessCredentials operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceDataAsyncClient.GetProgrammaticAccessCredentials
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2020-07-13/GetProgrammaticAccessCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetProgrammaticAccessCredentialsResponse> getProgrammaticAccessCredentials(
            GetProgrammaticAccessCredentialsRequest getProgrammaticAccessCredentialsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getProgrammaticAccessCredentialsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProgrammaticAccessCredentials");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProgrammaticAccessCredentialsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetProgrammaticAccessCredentialsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProgrammaticAccessCredentialsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProgrammaticAccessCredentialsRequest, GetProgrammaticAccessCredentialsResponse>()
                            .withOperationName("GetProgrammaticAccessCredentials")
                            .withMarshaller(new GetProgrammaticAccessCredentialsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getProgrammaticAccessCredentialsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getProgrammaticAccessCredentialsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetProgrammaticAccessCredentialsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A temporary Amazon S3 location to copy your files from a source location to stage or use as a scratch space in
     * Habanero notebook.
     * </p>
     *
     * @param getWorkingLocationRequest
     * @return A Java Future containing the result of the GetWorkingLocation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceDataException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceDataAsyncClient.GetWorkingLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2020-07-13/GetWorkingLocation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWorkingLocationResponse> getWorkingLocation(GetWorkingLocationRequest getWorkingLocationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkingLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkingLocation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkingLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetWorkingLocationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetWorkingLocationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkingLocationRequest, GetWorkingLocationResponse>()
                            .withOperationName("GetWorkingLocation")
                            .withMarshaller(new GetWorkingLocationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getWorkingLocationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getWorkingLocationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetWorkingLocationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FinspaceDataException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
