/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.finspacedata.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * The request for a CreateChangeset operation.
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateChangesetRequest extends FinspaceDataRequest implements
        ToCopyableBuilder<CreateChangesetRequest.Builder, CreateChangesetRequest> {
    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("clientToken")
            .getter(getter(CreateChangesetRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<String> DATASET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("datasetId").getter(getter(CreateChangesetRequest::datasetId)).setter(setter(Builder::datasetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("datasetId").build()).build();

    private static final SdkField<String> CHANGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("changeType").getter(getter(CreateChangesetRequest::changeTypeAsString))
            .setter(setter(Builder::changeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("changeType").build()).build();

    private static final SdkField<Map<String, String>> SOURCE_PARAMS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("sourceParams")
            .getter(getter(CreateChangesetRequest::sourceParams))
            .setter(setter(Builder::sourceParams))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceParams").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> FORMAT_PARAMS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("formatParams")
            .getter(getter(CreateChangesetRequest::formatParams))
            .setter(setter(Builder::formatParams))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("formatParams").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_TOKEN_FIELD,
            DATASET_ID_FIELD, CHANGE_TYPE_FIELD, SOURCE_PARAMS_FIELD, FORMAT_PARAMS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("clientToken", CLIENT_TOKEN_FIELD);
                    put("datasetId", DATASET_ID_FIELD);
                    put("changeType", CHANGE_TYPE_FIELD);
                    put("sourceParams", SOURCE_PARAMS_FIELD);
                    put("formatParams", FORMAT_PARAMS_FIELD);
                }
            });

    private final String clientToken;

    private final String datasetId;

    private final String changeType;

    private final Map<String, String> sourceParams;

    private final Map<String, String> formatParams;

    private CreateChangesetRequest(BuilderImpl builder) {
        super(builder);
        this.clientToken = builder.clientToken;
        this.datasetId = builder.datasetId;
        this.changeType = builder.changeType;
        this.sourceParams = builder.sourceParams;
        this.formatParams = builder.formatParams;
    }

    /**
     * <p>
     * A token that ensures idempotency. This token expires in 10 minutes.
     * </p>
     * 
     * @return A token that ensures idempotency. This token expires in 10 minutes.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * The unique identifier for the FinSpace Dataset where the Changeset will be created.
     * </p>
     * 
     * @return The unique identifier for the FinSpace Dataset where the Changeset will be created.
     */
    public final String datasetId() {
        return datasetId;
    }

    /**
     * <p>
     * The option to indicate how a Changeset will be applied to a Dataset.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>REPLACE</code> – Changeset will be considered as a replacement to all prior loaded Changesets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>APPEND</code> – Changeset will be considered as an addition to the end of all prior loaded Changesets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MODIFY</code> – Changeset is considered as a replacement to a specific prior ingested Changeset.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #changeType} will
     * return {@link ChangeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #changeTypeAsString}.
     * </p>
     * 
     * @return The option to indicate how a Changeset will be applied to a Dataset.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>REPLACE</code> – Changeset will be considered as a replacement to all prior loaded Changesets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>APPEND</code> – Changeset will be considered as an addition to the end of all prior loaded
     *         Changesets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MODIFY</code> – Changeset is considered as a replacement to a specific prior ingested Changeset.
     *         </p>
     *         </li>
     * @see ChangeType
     */
    public final ChangeType changeType() {
        return ChangeType.fromValue(changeType);
    }

    /**
     * <p>
     * The option to indicate how a Changeset will be applied to a Dataset.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>REPLACE</code> – Changeset will be considered as a replacement to all prior loaded Changesets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>APPEND</code> – Changeset will be considered as an addition to the end of all prior loaded Changesets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MODIFY</code> – Changeset is considered as a replacement to a specific prior ingested Changeset.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #changeType} will
     * return {@link ChangeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #changeTypeAsString}.
     * </p>
     * 
     * @return The option to indicate how a Changeset will be applied to a Dataset.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>REPLACE</code> – Changeset will be considered as a replacement to all prior loaded Changesets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>APPEND</code> – Changeset will be considered as an addition to the end of all prior loaded
     *         Changesets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MODIFY</code> – Changeset is considered as a replacement to a specific prior ingested Changeset.
     *         </p>
     *         </li>
     * @see ChangeType
     */
    public final String changeTypeAsString() {
        return changeType;
    }

    /**
     * For responses, this returns true if the service returned a value for the SourceParams property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSourceParams() {
        return sourceParams != null && !(sourceParams instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Options that define the location of the data being ingested (<code>s3SourcePath</code>) and the source of the
     * changeset (<code>sourceType</code>).
     * </p>
     * <p>
     * Both <code>s3SourcePath</code> and <code>sourceType</code> are required attributes.
     * </p>
     * <p>
     * Here is an example of how you could specify the <code>sourceParams</code>:
     * </p>
     * <p>
     * <code> "sourceParams": { "s3SourcePath": "s3://finspace-landing-us-east-2-bk7gcfvitndqa6ebnvys4d/scratch/wr5hh8pwkpqqkxa4sxrmcw/ingestion/equity.csv", "sourceType": "S3" } </code>
     * </p>
     * <p>
     * The S3 path that you specify must allow the FinSpace role access. To do that, you first need to configure the IAM
     * policy on S3 bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/finspace/latest/data-api/fs-using-the-finspace-api.html#access-s3-buckets"
     * >Loading data from an Amazon S3 Bucket using the FinSpace API</a> section.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSourceParams} method.
     * </p>
     * 
     * @return Options that define the location of the data being ingested (<code>s3SourcePath</code>) and the source of
     *         the changeset (<code>sourceType</code>).</p>
     *         <p>
     *         Both <code>s3SourcePath</code> and <code>sourceType</code> are required attributes.
     *         </p>
     *         <p>
     *         Here is an example of how you could specify the <code>sourceParams</code>:
     *         </p>
     *         <p>
     *         <code> "sourceParams": { "s3SourcePath": "s3://finspace-landing-us-east-2-bk7gcfvitndqa6ebnvys4d/scratch/wr5hh8pwkpqqkxa4sxrmcw/ingestion/equity.csv", "sourceType": "S3" } </code>
     *         </p>
     *         <p>
     *         The S3 path that you specify must allow the FinSpace role access. To do that, you first need to configure
     *         the IAM policy on S3 bucket. For more information, see <a href=
     *         "https://docs.aws.amazon.com/finspace/latest/data-api/fs-using-the-finspace-api.html#access-s3-buckets"
     *         >Loading data from an Amazon S3 Bucket using the FinSpace API</a> section.
     */
    public final Map<String, String> sourceParams() {
        return sourceParams;
    }

    /**
     * For responses, this returns true if the service returned a value for the FormatParams property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFormatParams() {
        return formatParams != null && !(formatParams instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Options that define the structure of the source file(s) including the format type (<code>formatType</code>),
     * header row (<code>withHeader</code>), data separation character (<code>separator</code>) and the type of
     * compression (<code>compression</code>).
     * </p>
     * <p>
     * <code>formatType</code> is a required attribute and can have the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PARQUET</code> – Parquet source file format.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CSV</code> – CSV source file format.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>JSON</code> – JSON source file format.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>XML</code> – XML source file format.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Here is an example of how you could specify the <code>formatParams</code>:
     * </p>
     * <p>
     * <code> "formatParams": { "formatType": "CSV", "withHeader": "true", "separator": ",", "compression":"None" } </code>
     * </p>
     * <p>
     * Note that if you only provide <code>formatType</code> as <code>CSV</code>, the rest of the attributes will
     * automatically default to CSV values as following:
     * </p>
     * <p>
     * <code> { "withHeader": "true", "separator": "," } </code>
     * </p>
     * <p>
     * For more information about supported file formats, see <a
     * href="https://docs.aws.amazon.com/finspace/latest/userguide/supported-data-types.html">Supported Data Types and
     * File Formats</a> in the FinSpace User Guide.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFormatParams} method.
     * </p>
     * 
     * @return Options that define the structure of the source file(s) including the format type (
     *         <code>formatType</code>), header row (<code>withHeader</code>), data separation character (
     *         <code>separator</code>) and the type of compression (<code>compression</code>). </p>
     *         <p>
     *         <code>formatType</code> is a required attribute and can have the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PARQUET</code> – Parquet source file format.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CSV</code> – CSV source file format.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>JSON</code> – JSON source file format.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>XML</code> – XML source file format.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Here is an example of how you could specify the <code>formatParams</code>:
     *         </p>
     *         <p>
     *         <code> "formatParams": { "formatType": "CSV", "withHeader": "true", "separator": ",", "compression":"None" } </code>
     *         </p>
     *         <p>
     *         Note that if you only provide <code>formatType</code> as <code>CSV</code>, the rest of the attributes
     *         will automatically default to CSV values as following:
     *         </p>
     *         <p>
     *         <code> { "withHeader": "true", "separator": "," } </code>
     *         </p>
     *         <p>
     *         For more information about supported file formats, see <a
     *         href="https://docs.aws.amazon.com/finspace/latest/userguide/supported-data-types.html">Supported Data
     *         Types and File Formats</a> in the FinSpace User Guide.
     */
    public final Map<String, String> formatParams() {
        return formatParams;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(datasetId());
        hashCode = 31 * hashCode + Objects.hashCode(changeTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceParams() ? sourceParams() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFormatParams() ? formatParams() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateChangesetRequest)) {
            return false;
        }
        CreateChangesetRequest other = (CreateChangesetRequest) obj;
        return Objects.equals(clientToken(), other.clientToken()) && Objects.equals(datasetId(), other.datasetId())
                && Objects.equals(changeTypeAsString(), other.changeTypeAsString())
                && hasSourceParams() == other.hasSourceParams() && Objects.equals(sourceParams(), other.sourceParams())
                && hasFormatParams() == other.hasFormatParams() && Objects.equals(formatParams(), other.formatParams());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateChangesetRequest").add("ClientToken", clientToken()).add("DatasetId", datasetId())
                .add("ChangeType", changeTypeAsString()).add("SourceParams", hasSourceParams() ? sourceParams() : null)
                .add("FormatParams", hasFormatParams() ? formatParams() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "clientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "datasetId":
            return Optional.ofNullable(clazz.cast(datasetId()));
        case "changeType":
            return Optional.ofNullable(clazz.cast(changeTypeAsString()));
        case "sourceParams":
            return Optional.ofNullable(clazz.cast(sourceParams()));
        case "formatParams":
            return Optional.ofNullable(clazz.cast(formatParams()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<CreateChangesetRequest, T> g) {
        return obj -> g.apply((CreateChangesetRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends FinspaceDataRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateChangesetRequest> {
        /**
         * <p>
         * A token that ensures idempotency. This token expires in 10 minutes.
         * </p>
         * 
         * @param clientToken
         *        A token that ensures idempotency. This token expires in 10 minutes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * The unique identifier for the FinSpace Dataset where the Changeset will be created.
         * </p>
         * 
         * @param datasetId
         *        The unique identifier for the FinSpace Dataset where the Changeset will be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datasetId(String datasetId);

        /**
         * <p>
         * The option to indicate how a Changeset will be applied to a Dataset.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>REPLACE</code> – Changeset will be considered as a replacement to all prior loaded Changesets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>APPEND</code> – Changeset will be considered as an addition to the end of all prior loaded Changesets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MODIFY</code> – Changeset is considered as a replacement to a specific prior ingested Changeset.
         * </p>
         * </li>
         * </ul>
         * 
         * @param changeType
         *        The option to indicate how a Changeset will be applied to a Dataset.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>REPLACE</code> – Changeset will be considered as a replacement to all prior loaded Changesets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>APPEND</code> – Changeset will be considered as an addition to the end of all prior loaded
         *        Changesets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MODIFY</code> – Changeset is considered as a replacement to a specific prior ingested Changeset.
         *        </p>
         *        </li>
         * @see ChangeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChangeType
         */
        Builder changeType(String changeType);

        /**
         * <p>
         * The option to indicate how a Changeset will be applied to a Dataset.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>REPLACE</code> – Changeset will be considered as a replacement to all prior loaded Changesets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>APPEND</code> – Changeset will be considered as an addition to the end of all prior loaded Changesets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MODIFY</code> – Changeset is considered as a replacement to a specific prior ingested Changeset.
         * </p>
         * </li>
         * </ul>
         * 
         * @param changeType
         *        The option to indicate how a Changeset will be applied to a Dataset.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>REPLACE</code> – Changeset will be considered as a replacement to all prior loaded Changesets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>APPEND</code> – Changeset will be considered as an addition to the end of all prior loaded
         *        Changesets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MODIFY</code> – Changeset is considered as a replacement to a specific prior ingested Changeset.
         *        </p>
         *        </li>
         * @see ChangeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChangeType
         */
        Builder changeType(ChangeType changeType);

        /**
         * <p>
         * Options that define the location of the data being ingested (<code>s3SourcePath</code>) and the source of the
         * changeset (<code>sourceType</code>).
         * </p>
         * <p>
         * Both <code>s3SourcePath</code> and <code>sourceType</code> are required attributes.
         * </p>
         * <p>
         * Here is an example of how you could specify the <code>sourceParams</code>:
         * </p>
         * <p>
         * <code> "sourceParams": { "s3SourcePath": "s3://finspace-landing-us-east-2-bk7gcfvitndqa6ebnvys4d/scratch/wr5hh8pwkpqqkxa4sxrmcw/ingestion/equity.csv", "sourceType": "S3" } </code>
         * </p>
         * <p>
         * The S3 path that you specify must allow the FinSpace role access. To do that, you first need to configure the
         * IAM policy on S3 bucket. For more information, see <a href=
         * "https://docs.aws.amazon.com/finspace/latest/data-api/fs-using-the-finspace-api.html#access-s3-buckets"
         * >Loading data from an Amazon S3 Bucket using the FinSpace API</a> section.
         * </p>
         * 
         * @param sourceParams
         *        Options that define the location of the data being ingested (<code>s3SourcePath</code>) and the source
         *        of the changeset (<code>sourceType</code>).</p>
         *        <p>
         *        Both <code>s3SourcePath</code> and <code>sourceType</code> are required attributes.
         *        </p>
         *        <p>
         *        Here is an example of how you could specify the <code>sourceParams</code>:
         *        </p>
         *        <p>
         *        <code> "sourceParams": { "s3SourcePath": "s3://finspace-landing-us-east-2-bk7gcfvitndqa6ebnvys4d/scratch/wr5hh8pwkpqqkxa4sxrmcw/ingestion/equity.csv", "sourceType": "S3" } </code>
         *        </p>
         *        <p>
         *        The S3 path that you specify must allow the FinSpace role access. To do that, you first need to
         *        configure the IAM policy on S3 bucket. For more information, see <a href=
         *        "https://docs.aws.amazon.com/finspace/latest/data-api/fs-using-the-finspace-api.html#access-s3-buckets"
         *        >Loading data from an Amazon S3 Bucket using the FinSpace API</a> section.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceParams(Map<String, String> sourceParams);

        /**
         * <p>
         * Options that define the structure of the source file(s) including the format type (<code>formatType</code>),
         * header row (<code>withHeader</code>), data separation character (<code>separator</code>) and the type of
         * compression (<code>compression</code>).
         * </p>
         * <p>
         * <code>formatType</code> is a required attribute and can have the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PARQUET</code> – Parquet source file format.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CSV</code> – CSV source file format.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>JSON</code> – JSON source file format.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>XML</code> – XML source file format.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Here is an example of how you could specify the <code>formatParams</code>:
         * </p>
         * <p>
         * <code> "formatParams": { "formatType": "CSV", "withHeader": "true", "separator": ",", "compression":"None" } </code>
         * </p>
         * <p>
         * Note that if you only provide <code>formatType</code> as <code>CSV</code>, the rest of the attributes will
         * automatically default to CSV values as following:
         * </p>
         * <p>
         * <code> { "withHeader": "true", "separator": "," } </code>
         * </p>
         * <p>
         * For more information about supported file formats, see <a
         * href="https://docs.aws.amazon.com/finspace/latest/userguide/supported-data-types.html">Supported Data Types
         * and File Formats</a> in the FinSpace User Guide.
         * </p>
         * 
         * @param formatParams
         *        Options that define the structure of the source file(s) including the format type (
         *        <code>formatType</code>), header row (<code>withHeader</code>), data separation character (
         *        <code>separator</code>) and the type of compression (<code>compression</code>). </p>
         *        <p>
         *        <code>formatType</code> is a required attribute and can have the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PARQUET</code> – Parquet source file format.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CSV</code> – CSV source file format.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>JSON</code> – JSON source file format.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>XML</code> – XML source file format.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Here is an example of how you could specify the <code>formatParams</code>:
         *        </p>
         *        <p>
         *        <code> "formatParams": { "formatType": "CSV", "withHeader": "true", "separator": ",", "compression":"None" } </code>
         *        </p>
         *        <p>
         *        Note that if you only provide <code>formatType</code> as <code>CSV</code>, the rest of the attributes
         *        will automatically default to CSV values as following:
         *        </p>
         *        <p>
         *        <code> { "withHeader": "true", "separator": "," } </code>
         *        </p>
         *        <p>
         *        For more information about supported file formats, see <a
         *        href="https://docs.aws.amazon.com/finspace/latest/userguide/supported-data-types.html">Supported Data
         *        Types and File Formats</a> in the FinSpace User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder formatParams(Map<String, String> formatParams);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends FinspaceDataRequest.BuilderImpl implements Builder {
        private String clientToken;

        private String datasetId;

        private String changeType;

        private Map<String, String> sourceParams = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> formatParams = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateChangesetRequest model) {
            super(model);
            clientToken(model.clientToken);
            datasetId(model.datasetId);
            changeType(model.changeType);
            sourceParams(model.sourceParams);
            formatParams(model.formatParams);
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final String getDatasetId() {
            return datasetId;
        }

        public final void setDatasetId(String datasetId) {
            this.datasetId = datasetId;
        }

        @Override
        public final Builder datasetId(String datasetId) {
            this.datasetId = datasetId;
            return this;
        }

        public final String getChangeType() {
            return changeType;
        }

        public final void setChangeType(String changeType) {
            this.changeType = changeType;
        }

        @Override
        public final Builder changeType(String changeType) {
            this.changeType = changeType;
            return this;
        }

        @Override
        public final Builder changeType(ChangeType changeType) {
            this.changeType(changeType == null ? null : changeType.toString());
            return this;
        }

        public final Map<String, String> getSourceParams() {
            if (sourceParams instanceof SdkAutoConstructMap) {
                return null;
            }
            return sourceParams;
        }

        public final void setSourceParams(Map<String, String> sourceParams) {
            this.sourceParams = SourceParamsCopier.copy(sourceParams);
        }

        @Override
        public final Builder sourceParams(Map<String, String> sourceParams) {
            this.sourceParams = SourceParamsCopier.copy(sourceParams);
            return this;
        }

        public final Map<String, String> getFormatParams() {
            if (formatParams instanceof SdkAutoConstructMap) {
                return null;
            }
            return formatParams;
        }

        public final void setFormatParams(Map<String, String> formatParams) {
            this.formatParams = FormatParamsCopier.copy(formatParams);
        }

        @Override
        public final Builder formatParams(Map<String, String> formatParams) {
            this.formatParams = FormatParamsCopier.copy(formatParams);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateChangesetRequest build() {
            return new CreateChangesetRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
