/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.finspacedata.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The structure for a permission group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PermissionGroup implements SdkPojo, Serializable, ToCopyableBuilder<PermissionGroup.Builder, PermissionGroup> {
    private static final SdkField<String> PERMISSION_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("permissionGroupId").getter(getter(PermissionGroup::permissionGroupId))
            .setter(setter(Builder::permissionGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("permissionGroupId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(PermissionGroup::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(PermissionGroup::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> APPLICATION_PERMISSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("applicationPermissions")
            .getter(getter(PermissionGroup::applicationPermissionsAsStrings))
            .setter(setter(Builder::applicationPermissionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("applicationPermissions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> CREATE_TIME_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("createTime").getter(getter(PermissionGroup::createTime)).setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createTime").build()).build();

    private static final SdkField<Long> LAST_MODIFIED_TIME_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("lastModifiedTime").getter(getter(PermissionGroup::lastModifiedTime))
            .setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastModifiedTime").build()).build();

    private static final SdkField<String> MEMBERSHIP_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("membershipStatus").getter(getter(PermissionGroup::membershipStatusAsString))
            .setter(setter(Builder::membershipStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("membershipStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PERMISSION_GROUP_ID_FIELD,
            NAME_FIELD, DESCRIPTION_FIELD, APPLICATION_PERMISSIONS_FIELD, CREATE_TIME_FIELD, LAST_MODIFIED_TIME_FIELD,
            MEMBERSHIP_STATUS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("permissionGroupId", PERMISSION_GROUP_ID_FIELD);
                    put("name", NAME_FIELD);
                    put("description", DESCRIPTION_FIELD);
                    put("applicationPermissions", APPLICATION_PERMISSIONS_FIELD);
                    put("createTime", CREATE_TIME_FIELD);
                    put("lastModifiedTime", LAST_MODIFIED_TIME_FIELD);
                    put("membershipStatus", MEMBERSHIP_STATUS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String permissionGroupId;

    private final String name;

    private final String description;

    private final List<String> applicationPermissions;

    private final Long createTime;

    private final Long lastModifiedTime;

    private final String membershipStatus;

    private PermissionGroup(BuilderImpl builder) {
        this.permissionGroupId = builder.permissionGroupId;
        this.name = builder.name;
        this.description = builder.description;
        this.applicationPermissions = builder.applicationPermissions;
        this.createTime = builder.createTime;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.membershipStatus = builder.membershipStatus;
    }

    /**
     * <p>
     * The unique identifier for the permission group.
     * </p>
     * 
     * @return The unique identifier for the permission group.
     */
    public final String permissionGroupId() {
        return permissionGroupId;
    }

    /**
     * <p>
     * The name of the permission group.
     * </p>
     * 
     * @return The name of the permission group.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A brief description for the permission group.
     * </p>
     * 
     * @return A brief description for the permission group.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Indicates the permissions that are granted to a specific group for accessing the FinSpace application.
     * </p>
     * <important>
     * <p>
     * When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code> allows
     * users to grant themselves or others access to any functionality in their FinSpace environment's application. It
     * should only be granted to trusted users.
     * </p>
     * </important>
     * <ul>
     * <li>
     * <p>
     * <code>CreateDataset</code> – Group members can create new datasets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a privileged
     * permission that allows users to grant themselves or others access to any functionality in the application. It
     * should only be granted to trusted users.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ManageAttributeSets</code> – Group members can manage attribute sets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ViewAuditData</code> – Group members can view audit data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasApplicationPermissions} method.
     * </p>
     * 
     * @return Indicates the permissions that are granted to a specific group for accessing the FinSpace
     *         application.</p> <important>
     *         <p>
     *         When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code>
     *         allows users to grant themselves or others access to any functionality in their FinSpace environment's
     *         application. It should only be granted to trusted users.
     *         </p>
     *         </important>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CreateDataset</code> – Group members can create new datasets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a
     *         privileged permission that allows users to grant themselves or others access to any functionality in the
     *         application. It should only be granted to trusted users.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ManageAttributeSets</code> – Group members can manage attribute sets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ViewAuditData</code> – Group members can view audit data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
     *         </p>
     *         </li>
     */
    public final List<ApplicationPermission> applicationPermissions() {
        return ApplicationPermissionListCopier.copyStringToEnum(applicationPermissions);
    }

    /**
     * For responses, this returns true if the service returned a value for the ApplicationPermissions property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasApplicationPermissions() {
        return applicationPermissions != null && !(applicationPermissions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates the permissions that are granted to a specific group for accessing the FinSpace application.
     * </p>
     * <important>
     * <p>
     * When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code> allows
     * users to grant themselves or others access to any functionality in their FinSpace environment's application. It
     * should only be granted to trusted users.
     * </p>
     * </important>
     * <ul>
     * <li>
     * <p>
     * <code>CreateDataset</code> – Group members can create new datasets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a privileged
     * permission that allows users to grant themselves or others access to any functionality in the application. It
     * should only be granted to trusted users.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ManageAttributeSets</code> – Group members can manage attribute sets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ViewAuditData</code> – Group members can view audit data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasApplicationPermissions} method.
     * </p>
     * 
     * @return Indicates the permissions that are granted to a specific group for accessing the FinSpace
     *         application.</p> <important>
     *         <p>
     *         When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code>
     *         allows users to grant themselves or others access to any functionality in their FinSpace environment's
     *         application. It should only be granted to trusted users.
     *         </p>
     *         </important>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CreateDataset</code> – Group members can create new datasets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a
     *         privileged permission that allows users to grant themselves or others access to any functionality in the
     *         application. It should only be granted to trusted users.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ManageAttributeSets</code> – Group members can manage attribute sets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ViewAuditData</code> – Group members can view audit data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
     *         </p>
     *         </li>
     */
    public final List<String> applicationPermissionsAsStrings() {
        return applicationPermissions;
    }

    /**
     * <p>
     * The timestamp at which the group was created in FinSpace. The value is determined as epoch time in milliseconds.
     * </p>
     * 
     * @return The timestamp at which the group was created in FinSpace. The value is determined as epoch time in
     *         milliseconds.
     */
    public final Long createTime() {
        return createTime;
    }

    /**
     * <p>
     * Describes the last time the permission group was updated. The value is determined as epoch time in milliseconds.
     * </p>
     * 
     * @return Describes the last time the permission group was updated. The value is determined as epoch time in
     *         milliseconds.
     */
    public final Long lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * Indicates the status of the user within a permission group.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ADDITION_IN_PROGRESS</code> – The user is currently being added to the permission group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ADDITION_SUCCESS</code> – The user is successfully added to the permission group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REMOVAL_IN_PROGRESS</code> – The user is currently being removed from the permission group.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #membershipStatus}
     * will return {@link PermissionGroupMembershipStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #membershipStatusAsString}.
     * </p>
     * 
     * @return Indicates the status of the user within a permission group.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ADDITION_IN_PROGRESS</code> – The user is currently being added to the permission group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ADDITION_SUCCESS</code> – The user is successfully added to the permission group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REMOVAL_IN_PROGRESS</code> – The user is currently being removed from the permission group.
     *         </p>
     *         </li>
     * @see PermissionGroupMembershipStatus
     */
    public final PermissionGroupMembershipStatus membershipStatus() {
        return PermissionGroupMembershipStatus.fromValue(membershipStatus);
    }

    /**
     * <p>
     * Indicates the status of the user within a permission group.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ADDITION_IN_PROGRESS</code> – The user is currently being added to the permission group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ADDITION_SUCCESS</code> – The user is successfully added to the permission group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REMOVAL_IN_PROGRESS</code> – The user is currently being removed from the permission group.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #membershipStatus}
     * will return {@link PermissionGroupMembershipStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #membershipStatusAsString}.
     * </p>
     * 
     * @return Indicates the status of the user within a permission group.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ADDITION_IN_PROGRESS</code> – The user is currently being added to the permission group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ADDITION_SUCCESS</code> – The user is successfully added to the permission group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REMOVAL_IN_PROGRESS</code> – The user is currently being removed from the permission group.
     *         </p>
     *         </li>
     * @see PermissionGroupMembershipStatus
     */
    public final String membershipStatusAsString() {
        return membershipStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(permissionGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasApplicationPermissions() ? applicationPermissionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(membershipStatusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PermissionGroup)) {
            return false;
        }
        PermissionGroup other = (PermissionGroup) obj;
        return Objects.equals(permissionGroupId(), other.permissionGroupId()) && Objects.equals(name(), other.name())
                && Objects.equals(description(), other.description())
                && hasApplicationPermissions() == other.hasApplicationPermissions()
                && Objects.equals(applicationPermissionsAsStrings(), other.applicationPermissionsAsStrings())
                && Objects.equals(createTime(), other.createTime())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(membershipStatusAsString(), other.membershipStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PermissionGroup").add("PermissionGroupId", permissionGroupId())
                .add("Name", name() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Description", description() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ApplicationPermissions", hasApplicationPermissions() ? applicationPermissionsAsStrings() : null)
                .add("CreateTime", createTime()).add("LastModifiedTime", lastModifiedTime())
                .add("MembershipStatus", membershipStatusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "permissionGroupId":
            return Optional.ofNullable(clazz.cast(permissionGroupId()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "applicationPermissions":
            return Optional.ofNullable(clazz.cast(applicationPermissionsAsStrings()));
        case "createTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "lastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "membershipStatus":
            return Optional.ofNullable(clazz.cast(membershipStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<PermissionGroup, T> g) {
        return obj -> g.apply((PermissionGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PermissionGroup> {
        /**
         * <p>
         * The unique identifier for the permission group.
         * </p>
         * 
         * @param permissionGroupId
         *        The unique identifier for the permission group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissionGroupId(String permissionGroupId);

        /**
         * <p>
         * The name of the permission group.
         * </p>
         * 
         * @param name
         *        The name of the permission group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A brief description for the permission group.
         * </p>
         * 
         * @param description
         *        A brief description for the permission group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Indicates the permissions that are granted to a specific group for accessing the FinSpace application.
         * </p>
         * <important>
         * <p>
         * When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code> allows
         * users to grant themselves or others access to any functionality in their FinSpace environment's application.
         * It should only be granted to trusted users.
         * </p>
         * </important>
         * <ul>
         * <li>
         * <p>
         * <code>CreateDataset</code> – Group members can create new datasets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a
         * privileged permission that allows users to grant themselves or others access to any functionality in the
         * application. It should only be granted to trusted users.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageAttributeSets</code> – Group members can manage attribute sets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ViewAuditData</code> – Group members can view audit data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
         * </p>
         * </li>
         * </ul>
         * 
         * @param applicationPermissions
         *        Indicates the permissions that are granted to a specific group for accessing the FinSpace
         *        application.</p> <important>
         *        <p>
         *        When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code>
         *        allows users to grant themselves or others access to any functionality in their FinSpace environment's
         *        application. It should only be granted to trusted users.
         *        </p>
         *        </important>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CreateDataset</code> – Group members can create new datasets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a
         *        privileged permission that allows users to grant themselves or others access to any functionality in
         *        the application. It should only be granted to trusted users.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageAttributeSets</code> – Group members can manage attribute sets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ViewAuditData</code> – Group members can view audit data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationPermissionsWithStrings(Collection<String> applicationPermissions);

        /**
         * <p>
         * Indicates the permissions that are granted to a specific group for accessing the FinSpace application.
         * </p>
         * <important>
         * <p>
         * When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code> allows
         * users to grant themselves or others access to any functionality in their FinSpace environment's application.
         * It should only be granted to trusted users.
         * </p>
         * </important>
         * <ul>
         * <li>
         * <p>
         * <code>CreateDataset</code> – Group members can create new datasets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a
         * privileged permission that allows users to grant themselves or others access to any functionality in the
         * application. It should only be granted to trusted users.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageAttributeSets</code> – Group members can manage attribute sets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ViewAuditData</code> – Group members can view audit data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
         * </p>
         * </li>
         * </ul>
         * 
         * @param applicationPermissions
         *        Indicates the permissions that are granted to a specific group for accessing the FinSpace
         *        application.</p> <important>
         *        <p>
         *        When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code>
         *        allows users to grant themselves or others access to any functionality in their FinSpace environment's
         *        application. It should only be granted to trusted users.
         *        </p>
         *        </important>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CreateDataset</code> – Group members can create new datasets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a
         *        privileged permission that allows users to grant themselves or others access to any functionality in
         *        the application. It should only be granted to trusted users.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageAttributeSets</code> – Group members can manage attribute sets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ViewAuditData</code> – Group members can view audit data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationPermissionsWithStrings(String... applicationPermissions);

        /**
         * <p>
         * Indicates the permissions that are granted to a specific group for accessing the FinSpace application.
         * </p>
         * <important>
         * <p>
         * When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code> allows
         * users to grant themselves or others access to any functionality in their FinSpace environment's application.
         * It should only be granted to trusted users.
         * </p>
         * </important>
         * <ul>
         * <li>
         * <p>
         * <code>CreateDataset</code> – Group members can create new datasets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a
         * privileged permission that allows users to grant themselves or others access to any functionality in the
         * application. It should only be granted to trusted users.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageAttributeSets</code> – Group members can manage attribute sets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ViewAuditData</code> – Group members can view audit data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
         * </p>
         * </li>
         * </ul>
         * 
         * @param applicationPermissions
         *        Indicates the permissions that are granted to a specific group for accessing the FinSpace
         *        application.</p> <important>
         *        <p>
         *        When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code>
         *        allows users to grant themselves or others access to any functionality in their FinSpace environment's
         *        application. It should only be granted to trusted users.
         *        </p>
         *        </important>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CreateDataset</code> – Group members can create new datasets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a
         *        privileged permission that allows users to grant themselves or others access to any functionality in
         *        the application. It should only be granted to trusted users.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageAttributeSets</code> – Group members can manage attribute sets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ViewAuditData</code> – Group members can view audit data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationPermissions(Collection<ApplicationPermission> applicationPermissions);

        /**
         * <p>
         * Indicates the permissions that are granted to a specific group for accessing the FinSpace application.
         * </p>
         * <important>
         * <p>
         * When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code> allows
         * users to grant themselves or others access to any functionality in their FinSpace environment's application.
         * It should only be granted to trusted users.
         * </p>
         * </important>
         * <ul>
         * <li>
         * <p>
         * <code>CreateDataset</code> – Group members can create new datasets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a
         * privileged permission that allows users to grant themselves or others access to any functionality in the
         * application. It should only be granted to trusted users.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ManageAttributeSets</code> – Group members can manage attribute sets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ViewAuditData</code> – Group members can view audit data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
         * </p>
         * </li>
         * </ul>
         * 
         * @param applicationPermissions
         *        Indicates the permissions that are granted to a specific group for accessing the FinSpace
         *        application.</p> <important>
         *        <p>
         *        When assigning application permissions, be aware that the permission <code>ManageUsersAndGroups</code>
         *        allows users to grant themselves or others access to any functionality in their FinSpace environment's
         *        application. It should only be granted to trusted users.
         *        </p>
         *        </important>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CreateDataset</code> – Group members can create new datasets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageClusters</code> – Group members can manage Apache Spark clusters from FinSpace notebooks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageUsersAndGroups</code> – Group members can manage users and permission groups. This is a
         *        privileged permission that allows users to grant themselves or others access to any functionality in
         *        the application. It should only be granted to trusted users.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ManageAttributeSets</code> – Group members can manage attribute sets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ViewAuditData</code> – Group members can view audit data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AccessNotebooks</code> – Group members will have access to FinSpace notebooks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GetTemporaryCredentials</code> – Group members can get temporary API credentials.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationPermissions(ApplicationPermission... applicationPermissions);

        /**
         * <p>
         * The timestamp at which the group was created in FinSpace. The value is determined as epoch time in
         * milliseconds.
         * </p>
         * 
         * @param createTime
         *        The timestamp at which the group was created in FinSpace. The value is determined as epoch time in
         *        milliseconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Long createTime);

        /**
         * <p>
         * Describes the last time the permission group was updated. The value is determined as epoch time in
         * milliseconds.
         * </p>
         * 
         * @param lastModifiedTime
         *        Describes the last time the permission group was updated. The value is determined as epoch time in
         *        milliseconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Long lastModifiedTime);

        /**
         * <p>
         * Indicates the status of the user within a permission group.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ADDITION_IN_PROGRESS</code> – The user is currently being added to the permission group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ADDITION_SUCCESS</code> – The user is successfully added to the permission group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REMOVAL_IN_PROGRESS</code> – The user is currently being removed from the permission group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param membershipStatus
         *        Indicates the status of the user within a permission group.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ADDITION_IN_PROGRESS</code> – The user is currently being added to the permission group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ADDITION_SUCCESS</code> – The user is successfully added to the permission group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REMOVAL_IN_PROGRESS</code> – The user is currently being removed from the permission group.
         *        </p>
         *        </li>
         * @see PermissionGroupMembershipStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionGroupMembershipStatus
         */
        Builder membershipStatus(String membershipStatus);

        /**
         * <p>
         * Indicates the status of the user within a permission group.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ADDITION_IN_PROGRESS</code> – The user is currently being added to the permission group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ADDITION_SUCCESS</code> – The user is successfully added to the permission group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REMOVAL_IN_PROGRESS</code> – The user is currently being removed from the permission group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param membershipStatus
         *        Indicates the status of the user within a permission group.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ADDITION_IN_PROGRESS</code> – The user is currently being added to the permission group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ADDITION_SUCCESS</code> – The user is successfully added to the permission group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REMOVAL_IN_PROGRESS</code> – The user is currently being removed from the permission group.
         *        </p>
         *        </li>
         * @see PermissionGroupMembershipStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionGroupMembershipStatus
         */
        Builder membershipStatus(PermissionGroupMembershipStatus membershipStatus);
    }

    static final class BuilderImpl implements Builder {
        private String permissionGroupId;

        private String name;

        private String description;

        private List<String> applicationPermissions = DefaultSdkAutoConstructList.getInstance();

        private Long createTime;

        private Long lastModifiedTime;

        private String membershipStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(PermissionGroup model) {
            permissionGroupId(model.permissionGroupId);
            name(model.name);
            description(model.description);
            applicationPermissionsWithStrings(model.applicationPermissions);
            createTime(model.createTime);
            lastModifiedTime(model.lastModifiedTime);
            membershipStatus(model.membershipStatus);
        }

        public final String getPermissionGroupId() {
            return permissionGroupId;
        }

        public final void setPermissionGroupId(String permissionGroupId) {
            this.permissionGroupId = permissionGroupId;
        }

        @Override
        public final Builder permissionGroupId(String permissionGroupId) {
            this.permissionGroupId = permissionGroupId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getApplicationPermissions() {
            if (applicationPermissions instanceof SdkAutoConstructList) {
                return null;
            }
            return applicationPermissions;
        }

        public final void setApplicationPermissions(Collection<String> applicationPermissions) {
            this.applicationPermissions = ApplicationPermissionListCopier.copy(applicationPermissions);
        }

        @Override
        public final Builder applicationPermissionsWithStrings(Collection<String> applicationPermissions) {
            this.applicationPermissions = ApplicationPermissionListCopier.copy(applicationPermissions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder applicationPermissionsWithStrings(String... applicationPermissions) {
            applicationPermissionsWithStrings(Arrays.asList(applicationPermissions));
            return this;
        }

        @Override
        public final Builder applicationPermissions(Collection<ApplicationPermission> applicationPermissions) {
            this.applicationPermissions = ApplicationPermissionListCopier.copyEnumToString(applicationPermissions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder applicationPermissions(ApplicationPermission... applicationPermissions) {
            applicationPermissions(Arrays.asList(applicationPermissions));
            return this;
        }

        public final Long getCreateTime() {
            return createTime;
        }

        public final void setCreateTime(Long createTime) {
            this.createTime = createTime;
        }

        @Override
        public final Builder createTime(Long createTime) {
            this.createTime = createTime;
            return this;
        }

        public final Long getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Long lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Long lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final String getMembershipStatus() {
            return membershipStatus;
        }

        public final void setMembershipStatus(String membershipStatus) {
            this.membershipStatus = membershipStatus;
        }

        @Override
        public final Builder membershipStatus(String membershipStatus) {
            this.membershipStatus = membershipStatus;
            return this;
        }

        @Override
        public final Builder membershipStatus(PermissionGroupMembershipStatus membershipStatus) {
            this.membershipStatus(membershipStatus == null ? null : membershipStatus.toString());
            return this;
        }

        @Override
        public PermissionGroup build() {
            return new PermissionGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
