/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.firehose.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies that you want Kinesis Data Firehose to convert data from the JSON format to the Parquet or ORC format
 * before writing it to Amazon S3. Kinesis Data Firehose uses the serializer and deserializer that you specify, in
 * addition to the column information from the AWS Glue table, to deserialize your input data from JSON and then
 * serialize it to the Parquet or ORC format. For more information, see <a
 * href="https://docs.aws.amazon.com/firehose/latest/dev/record-format-conversion.html">Kinesis Data Firehose Record
 * Format Conversion</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataFormatConversionConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<DataFormatConversionConfiguration.Builder, DataFormatConversionConfiguration> {
    private static final SdkField<SchemaConfiguration> SCHEMA_CONFIGURATION_FIELD = SdkField
            .<SchemaConfiguration> builder(MarshallingType.SDK_POJO).memberName("SchemaConfiguration")
            .getter(getter(DataFormatConversionConfiguration::schemaConfiguration)).setter(setter(Builder::schemaConfiguration))
            .constructor(SchemaConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaConfiguration").build())
            .build();

    private static final SdkField<InputFormatConfiguration> INPUT_FORMAT_CONFIGURATION_FIELD = SdkField
            .<InputFormatConfiguration> builder(MarshallingType.SDK_POJO).memberName("InputFormatConfiguration")
            .getter(getter(DataFormatConversionConfiguration::inputFormatConfiguration))
            .setter(setter(Builder::inputFormatConfiguration)).constructor(InputFormatConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputFormatConfiguration").build())
            .build();

    private static final SdkField<OutputFormatConfiguration> OUTPUT_FORMAT_CONFIGURATION_FIELD = SdkField
            .<OutputFormatConfiguration> builder(MarshallingType.SDK_POJO).memberName("OutputFormatConfiguration")
            .getter(getter(DataFormatConversionConfiguration::outputFormatConfiguration))
            .setter(setter(Builder::outputFormatConfiguration)).constructor(OutputFormatConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputFormatConfiguration").build())
            .build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled").getter(getter(DataFormatConversionConfiguration::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SCHEMA_CONFIGURATION_FIELD,
            INPUT_FORMAT_CONFIGURATION_FIELD, OUTPUT_FORMAT_CONFIGURATION_FIELD, ENABLED_FIELD));

    private static final long serialVersionUID = 1L;

    private final SchemaConfiguration schemaConfiguration;

    private final InputFormatConfiguration inputFormatConfiguration;

    private final OutputFormatConfiguration outputFormatConfiguration;

    private final Boolean enabled;

    private DataFormatConversionConfiguration(BuilderImpl builder) {
        this.schemaConfiguration = builder.schemaConfiguration;
        this.inputFormatConfiguration = builder.inputFormatConfiguration;
        this.outputFormatConfiguration = builder.outputFormatConfiguration;
        this.enabled = builder.enabled;
    }

    /**
     * <p>
     * Specifies the AWS Glue Data Catalog table that contains the column information. This parameter is required if
     * <code>Enabled</code> is set to true.
     * </p>
     * 
     * @return Specifies the AWS Glue Data Catalog table that contains the column information. This parameter is
     *         required if <code>Enabled</code> is set to true.
     */
    public final SchemaConfiguration schemaConfiguration() {
        return schemaConfiguration;
    }

    /**
     * <p>
     * Specifies the deserializer that you want Kinesis Data Firehose to use to convert the format of your data from
     * JSON. This parameter is required if <code>Enabled</code> is set to true.
     * </p>
     * 
     * @return Specifies the deserializer that you want Kinesis Data Firehose to use to convert the format of your data
     *         from JSON. This parameter is required if <code>Enabled</code> is set to true.
     */
    public final InputFormatConfiguration inputFormatConfiguration() {
        return inputFormatConfiguration;
    }

    /**
     * <p>
     * Specifies the serializer that you want Kinesis Data Firehose to use to convert the format of your data to the
     * Parquet or ORC format. This parameter is required if <code>Enabled</code> is set to true.
     * </p>
     * 
     * @return Specifies the serializer that you want Kinesis Data Firehose to use to convert the format of your data to
     *         the Parquet or ORC format. This parameter is required if <code>Enabled</code> is set to true.
     */
    public final OutputFormatConfiguration outputFormatConfiguration() {
        return outputFormatConfiguration;
    }

    /**
     * <p>
     * Defaults to <code>true</code>. Set it to <code>false</code> if you want to disable format conversion while
     * preserving the configuration details.
     * </p>
     * 
     * @return Defaults to <code>true</code>. Set it to <code>false</code> if you want to disable format conversion
     *         while preserving the configuration details.
     */
    public final Boolean enabled() {
        return enabled;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(schemaConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(inputFormatConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(outputFormatConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataFormatConversionConfiguration)) {
            return false;
        }
        DataFormatConversionConfiguration other = (DataFormatConversionConfiguration) obj;
        return Objects.equals(schemaConfiguration(), other.schemaConfiguration())
                && Objects.equals(inputFormatConfiguration(), other.inputFormatConfiguration())
                && Objects.equals(outputFormatConfiguration(), other.outputFormatConfiguration())
                && Objects.equals(enabled(), other.enabled());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataFormatConversionConfiguration").add("SchemaConfiguration", schemaConfiguration())
                .add("InputFormatConfiguration", inputFormatConfiguration())
                .add("OutputFormatConfiguration", outputFormatConfiguration()).add("Enabled", enabled()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SchemaConfiguration":
            return Optional.ofNullable(clazz.cast(schemaConfiguration()));
        case "InputFormatConfiguration":
            return Optional.ofNullable(clazz.cast(inputFormatConfiguration()));
        case "OutputFormatConfiguration":
            return Optional.ofNullable(clazz.cast(outputFormatConfiguration()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DataFormatConversionConfiguration, T> g) {
        return obj -> g.apply((DataFormatConversionConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataFormatConversionConfiguration> {
        /**
         * <p>
         * Specifies the AWS Glue Data Catalog table that contains the column information. This parameter is required if
         * <code>Enabled</code> is set to true.
         * </p>
         * 
         * @param schemaConfiguration
         *        Specifies the AWS Glue Data Catalog table that contains the column information. This parameter is
         *        required if <code>Enabled</code> is set to true.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaConfiguration(SchemaConfiguration schemaConfiguration);

        /**
         * <p>
         * Specifies the AWS Glue Data Catalog table that contains the column information. This parameter is required if
         * <code>Enabled</code> is set to true.
         * </p>
         * This is a convenience method that creates an instance of the {@link SchemaConfiguration.Builder} avoiding the
         * need to create one manually via {@link SchemaConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link SchemaConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #schemaConfiguration(SchemaConfiguration)}.
         * 
         * @param schemaConfiguration
         *        a consumer that will call methods on {@link SchemaConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schemaConfiguration(SchemaConfiguration)
         */
        default Builder schemaConfiguration(Consumer<SchemaConfiguration.Builder> schemaConfiguration) {
            return schemaConfiguration(SchemaConfiguration.builder().applyMutation(schemaConfiguration).build());
        }

        /**
         * <p>
         * Specifies the deserializer that you want Kinesis Data Firehose to use to convert the format of your data from
         * JSON. This parameter is required if <code>Enabled</code> is set to true.
         * </p>
         * 
         * @param inputFormatConfiguration
         *        Specifies the deserializer that you want Kinesis Data Firehose to use to convert the format of your
         *        data from JSON. This parameter is required if <code>Enabled</code> is set to true.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputFormatConfiguration(InputFormatConfiguration inputFormatConfiguration);

        /**
         * <p>
         * Specifies the deserializer that you want Kinesis Data Firehose to use to convert the format of your data from
         * JSON. This parameter is required if <code>Enabled</code> is set to true.
         * </p>
         * This is a convenience method that creates an instance of the {@link InputFormatConfiguration.Builder}
         * avoiding the need to create one manually via {@link InputFormatConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link InputFormatConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #inputFormatConfiguration(InputFormatConfiguration)}.
         * 
         * @param inputFormatConfiguration
         *        a consumer that will call methods on {@link InputFormatConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputFormatConfiguration(InputFormatConfiguration)
         */
        default Builder inputFormatConfiguration(Consumer<InputFormatConfiguration.Builder> inputFormatConfiguration) {
            return inputFormatConfiguration(InputFormatConfiguration.builder().applyMutation(inputFormatConfiguration).build());
        }

        /**
         * <p>
         * Specifies the serializer that you want Kinesis Data Firehose to use to convert the format of your data to the
         * Parquet or ORC format. This parameter is required if <code>Enabled</code> is set to true.
         * </p>
         * 
         * @param outputFormatConfiguration
         *        Specifies the serializer that you want Kinesis Data Firehose to use to convert the format of your data
         *        to the Parquet or ORC format. This parameter is required if <code>Enabled</code> is set to true.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputFormatConfiguration(OutputFormatConfiguration outputFormatConfiguration);

        /**
         * <p>
         * Specifies the serializer that you want Kinesis Data Firehose to use to convert the format of your data to the
         * Parquet or ORC format. This parameter is required if <code>Enabled</code> is set to true.
         * </p>
         * This is a convenience method that creates an instance of the {@link OutputFormatConfiguration.Builder}
         * avoiding the need to create one manually via {@link OutputFormatConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link OutputFormatConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #outputFormatConfiguration(OutputFormatConfiguration)}.
         * 
         * @param outputFormatConfiguration
         *        a consumer that will call methods on {@link OutputFormatConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputFormatConfiguration(OutputFormatConfiguration)
         */
        default Builder outputFormatConfiguration(Consumer<OutputFormatConfiguration.Builder> outputFormatConfiguration) {
            return outputFormatConfiguration(OutputFormatConfiguration.builder().applyMutation(outputFormatConfiguration).build());
        }

        /**
         * <p>
         * Defaults to <code>true</code>. Set it to <code>false</code> if you want to disable format conversion while
         * preserving the configuration details.
         * </p>
         * 
         * @param enabled
         *        Defaults to <code>true</code>. Set it to <code>false</code> if you want to disable format conversion
         *        while preserving the configuration details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);
    }

    static final class BuilderImpl implements Builder {
        private SchemaConfiguration schemaConfiguration;

        private InputFormatConfiguration inputFormatConfiguration;

        private OutputFormatConfiguration outputFormatConfiguration;

        private Boolean enabled;

        private BuilderImpl() {
        }

        private BuilderImpl(DataFormatConversionConfiguration model) {
            schemaConfiguration(model.schemaConfiguration);
            inputFormatConfiguration(model.inputFormatConfiguration);
            outputFormatConfiguration(model.outputFormatConfiguration);
            enabled(model.enabled);
        }

        public final SchemaConfiguration.Builder getSchemaConfiguration() {
            return schemaConfiguration != null ? schemaConfiguration.toBuilder() : null;
        }

        public final void setSchemaConfiguration(SchemaConfiguration.BuilderImpl schemaConfiguration) {
            this.schemaConfiguration = schemaConfiguration != null ? schemaConfiguration.build() : null;
        }

        @Override
        public final Builder schemaConfiguration(SchemaConfiguration schemaConfiguration) {
            this.schemaConfiguration = schemaConfiguration;
            return this;
        }

        public final InputFormatConfiguration.Builder getInputFormatConfiguration() {
            return inputFormatConfiguration != null ? inputFormatConfiguration.toBuilder() : null;
        }

        public final void setInputFormatConfiguration(InputFormatConfiguration.BuilderImpl inputFormatConfiguration) {
            this.inputFormatConfiguration = inputFormatConfiguration != null ? inputFormatConfiguration.build() : null;
        }

        @Override
        public final Builder inputFormatConfiguration(InputFormatConfiguration inputFormatConfiguration) {
            this.inputFormatConfiguration = inputFormatConfiguration;
            return this;
        }

        public final OutputFormatConfiguration.Builder getOutputFormatConfiguration() {
            return outputFormatConfiguration != null ? outputFormatConfiguration.toBuilder() : null;
        }

        public final void setOutputFormatConfiguration(OutputFormatConfiguration.BuilderImpl outputFormatConfiguration) {
            this.outputFormatConfiguration = outputFormatConfiguration != null ? outputFormatConfiguration.build() : null;
        }

        @Override
        public final Builder outputFormatConfiguration(OutputFormatConfiguration outputFormatConfiguration) {
            this.outputFormatConfiguration = outputFormatConfiguration;
            return this;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        @Override
        public DataFormatConversionConfiguration build() {
            return new DataFormatConversionConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
