/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.firehose.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a delivery stream.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeliveryStreamDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<DeliveryStreamDescription.Builder, DeliveryStreamDescription> {
    private static final SdkField<String> DELIVERY_STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeliveryStreamName").getter(getter(DeliveryStreamDescription::deliveryStreamName))
            .setter(setter(Builder::deliveryStreamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliveryStreamName").build())
            .build();

    private static final SdkField<String> DELIVERY_STREAM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeliveryStreamARN").getter(getter(DeliveryStreamDescription::deliveryStreamARN))
            .setter(setter(Builder::deliveryStreamARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliveryStreamARN").build()).build();

    private static final SdkField<String> DELIVERY_STREAM_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeliveryStreamStatus").getter(getter(DeliveryStreamDescription::deliveryStreamStatusAsString))
            .setter(setter(Builder::deliveryStreamStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliveryStreamStatus").build())
            .build();

    private static final SdkField<FailureDescription> FAILURE_DESCRIPTION_FIELD = SdkField
            .<FailureDescription> builder(MarshallingType.SDK_POJO).memberName("FailureDescription")
            .getter(getter(DeliveryStreamDescription::failureDescription)).setter(setter(Builder::failureDescription))
            .constructor(FailureDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureDescription").build())
            .build();

    private static final SdkField<DeliveryStreamEncryptionConfiguration> DELIVERY_STREAM_ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<DeliveryStreamEncryptionConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("DeliveryStreamEncryptionConfiguration")
            .getter(getter(DeliveryStreamDescription::deliveryStreamEncryptionConfiguration))
            .setter(setter(Builder::deliveryStreamEncryptionConfiguration))
            .constructor(DeliveryStreamEncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("DeliveryStreamEncryptionConfiguration").build()).build();

    private static final SdkField<String> DELIVERY_STREAM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeliveryStreamType").getter(getter(DeliveryStreamDescription::deliveryStreamTypeAsString))
            .setter(setter(Builder::deliveryStreamType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliveryStreamType").build())
            .build();

    private static final SdkField<String> VERSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionId").getter(getter(DeliveryStreamDescription::versionId)).setter(setter(Builder::versionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionId").build()).build();

    private static final SdkField<Instant> CREATE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTimestamp").getter(getter(DeliveryStreamDescription::createTimestamp))
            .setter(setter(Builder::createTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTimestamp").build()).build();

    private static final SdkField<Instant> LAST_UPDATE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdateTimestamp").getter(getter(DeliveryStreamDescription::lastUpdateTimestamp))
            .setter(setter(Builder::lastUpdateTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdateTimestamp").build())
            .build();

    private static final SdkField<SourceDescription> SOURCE_FIELD = SdkField
            .<SourceDescription> builder(MarshallingType.SDK_POJO).memberName("Source")
            .getter(getter(DeliveryStreamDescription::source)).setter(setter(Builder::source))
            .constructor(SourceDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<List<DestinationDescription>> DESTINATIONS_FIELD = SdkField
            .<List<DestinationDescription>> builder(MarshallingType.LIST)
            .memberName("Destinations")
            .getter(getter(DeliveryStreamDescription::destinations))
            .setter(setter(Builder::destinations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DestinationDescription> builder(MarshallingType.SDK_POJO)
                                            .constructor(DestinationDescription::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> HAS_MORE_DESTINATIONS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("HasMoreDestinations").getter(getter(DeliveryStreamDescription::hasMoreDestinations))
            .setter(setter(Builder::hasMoreDestinations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HasMoreDestinations").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DELIVERY_STREAM_NAME_FIELD,
            DELIVERY_STREAM_ARN_FIELD, DELIVERY_STREAM_STATUS_FIELD, FAILURE_DESCRIPTION_FIELD,
            DELIVERY_STREAM_ENCRYPTION_CONFIGURATION_FIELD, DELIVERY_STREAM_TYPE_FIELD, VERSION_ID_FIELD, CREATE_TIMESTAMP_FIELD,
            LAST_UPDATE_TIMESTAMP_FIELD, SOURCE_FIELD, DESTINATIONS_FIELD, HAS_MORE_DESTINATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String deliveryStreamName;

    private final String deliveryStreamARN;

    private final String deliveryStreamStatus;

    private final FailureDescription failureDescription;

    private final DeliveryStreamEncryptionConfiguration deliveryStreamEncryptionConfiguration;

    private final String deliveryStreamType;

    private final String versionId;

    private final Instant createTimestamp;

    private final Instant lastUpdateTimestamp;

    private final SourceDescription source;

    private final List<DestinationDescription> destinations;

    private final Boolean hasMoreDestinations;

    private DeliveryStreamDescription(BuilderImpl builder) {
        this.deliveryStreamName = builder.deliveryStreamName;
        this.deliveryStreamARN = builder.deliveryStreamARN;
        this.deliveryStreamStatus = builder.deliveryStreamStatus;
        this.failureDescription = builder.failureDescription;
        this.deliveryStreamEncryptionConfiguration = builder.deliveryStreamEncryptionConfiguration;
        this.deliveryStreamType = builder.deliveryStreamType;
        this.versionId = builder.versionId;
        this.createTimestamp = builder.createTimestamp;
        this.lastUpdateTimestamp = builder.lastUpdateTimestamp;
        this.source = builder.source;
        this.destinations = builder.destinations;
        this.hasMoreDestinations = builder.hasMoreDestinations;
    }

    /**
     * <p>
     * The name of the delivery stream.
     * </p>
     * 
     * @return The name of the delivery stream.
     */
    public final String deliveryStreamName() {
        return deliveryStreamName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the delivery stream. For more information, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names (ARNs)
     * and AWS Service Namespaces</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the delivery stream. For more information, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
     *         (ARNs) and AWS Service Namespaces</a>.
     */
    public final String deliveryStreamARN() {
        return deliveryStreamARN;
    }

    /**
     * <p>
     * The status of the delivery stream. If the status of a delivery stream is <code>CREATING_FAILED</code>, this
     * status doesn't change, and you can't invoke <code>CreateDeliveryStream</code> again on it. However, you can
     * invoke the <a>DeleteDeliveryStream</a> operation to delete it.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryStreamStatus} will return {@link DeliveryStreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #deliveryStreamStatusAsString}.
     * </p>
     * 
     * @return The status of the delivery stream. If the status of a delivery stream is <code>CREATING_FAILED</code>,
     *         this status doesn't change, and you can't invoke <code>CreateDeliveryStream</code> again on it. However,
     *         you can invoke the <a>DeleteDeliveryStream</a> operation to delete it.
     * @see DeliveryStreamStatus
     */
    public final DeliveryStreamStatus deliveryStreamStatus() {
        return DeliveryStreamStatus.fromValue(deliveryStreamStatus);
    }

    /**
     * <p>
     * The status of the delivery stream. If the status of a delivery stream is <code>CREATING_FAILED</code>, this
     * status doesn't change, and you can't invoke <code>CreateDeliveryStream</code> again on it. However, you can
     * invoke the <a>DeleteDeliveryStream</a> operation to delete it.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryStreamStatus} will return {@link DeliveryStreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #deliveryStreamStatusAsString}.
     * </p>
     * 
     * @return The status of the delivery stream. If the status of a delivery stream is <code>CREATING_FAILED</code>,
     *         this status doesn't change, and you can't invoke <code>CreateDeliveryStream</code> again on it. However,
     *         you can invoke the <a>DeleteDeliveryStream</a> operation to delete it.
     * @see DeliveryStreamStatus
     */
    public final String deliveryStreamStatusAsString() {
        return deliveryStreamStatus;
    }

    /**
     * <p>
     * Provides details in case one of the following operations fails due to an error related to KMS:
     * <a>CreateDeliveryStream</a>, <a>DeleteDeliveryStream</a>, <a>StartDeliveryStreamEncryption</a>,
     * <a>StopDeliveryStreamEncryption</a>.
     * </p>
     * 
     * @return Provides details in case one of the following operations fails due to an error related to KMS:
     *         <a>CreateDeliveryStream</a>, <a>DeleteDeliveryStream</a>, <a>StartDeliveryStreamEncryption</a>,
     *         <a>StopDeliveryStreamEncryption</a>.
     */
    public final FailureDescription failureDescription() {
        return failureDescription;
    }

    /**
     * <p>
     * Indicates the server-side encryption (SSE) status for the delivery stream.
     * </p>
     * 
     * @return Indicates the server-side encryption (SSE) status for the delivery stream.
     */
    public final DeliveryStreamEncryptionConfiguration deliveryStreamEncryptionConfiguration() {
        return deliveryStreamEncryptionConfiguration;
    }

    /**
     * <p>
     * The delivery stream type. This can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DirectPut</code>: Provider applications access the delivery stream directly.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis data stream as a source.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryStreamType} will return {@link DeliveryStreamType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #deliveryStreamTypeAsString}.
     * </p>
     * 
     * @return The delivery stream type. This can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DirectPut</code>: Provider applications access the delivery stream directly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis data stream as a source.
     *         </p>
     *         </li>
     * @see DeliveryStreamType
     */
    public final DeliveryStreamType deliveryStreamType() {
        return DeliveryStreamType.fromValue(deliveryStreamType);
    }

    /**
     * <p>
     * The delivery stream type. This can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DirectPut</code>: Provider applications access the delivery stream directly.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis data stream as a source.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryStreamType} will return {@link DeliveryStreamType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #deliveryStreamTypeAsString}.
     * </p>
     * 
     * @return The delivery stream type. This can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DirectPut</code>: Provider applications access the delivery stream directly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis data stream as a source.
     *         </p>
     *         </li>
     * @see DeliveryStreamType
     */
    public final String deliveryStreamTypeAsString() {
        return deliveryStreamType;
    }

    /**
     * <p>
     * Each time the destination is updated for a delivery stream, the version ID is changed, and the current version ID
     * is required when updating the destination. This is so that the service knows it is applying the changes to the
     * correct version of the delivery stream.
     * </p>
     * 
     * @return Each time the destination is updated for a delivery stream, the version ID is changed, and the current
     *         version ID is required when updating the destination. This is so that the service knows it is applying
     *         the changes to the correct version of the delivery stream.
     */
    public final String versionId() {
        return versionId;
    }

    /**
     * <p>
     * The date and time that the delivery stream was created.
     * </p>
     * 
     * @return The date and time that the delivery stream was created.
     */
    public final Instant createTimestamp() {
        return createTimestamp;
    }

    /**
     * <p>
     * The date and time that the delivery stream was last updated.
     * </p>
     * 
     * @return The date and time that the delivery stream was last updated.
     */
    public final Instant lastUpdateTimestamp() {
        return lastUpdateTimestamp;
    }

    /**
     * <p>
     * If the <code>DeliveryStreamType</code> parameter is <code>KinesisStreamAsSource</code>, a
     * <a>SourceDescription</a> object describing the source Kinesis data stream.
     * </p>
     * 
     * @return If the <code>DeliveryStreamType</code> parameter is <code>KinesisStreamAsSource</code>, a
     *         <a>SourceDescription</a> object describing the source Kinesis data stream.
     */
    public final SourceDescription source() {
        return source;
    }

    /**
     * For responses, this returns true if the service returned a value for the Destinations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDestinations() {
        return destinations != null && !(destinations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The destinations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDestinations} method.
     * </p>
     * 
     * @return The destinations.
     */
    public final List<DestinationDescription> destinations() {
        return destinations;
    }

    /**
     * <p>
     * Indicates whether there are more destinations available to list.
     * </p>
     * 
     * @return Indicates whether there are more destinations available to list.
     */
    public final Boolean hasMoreDestinations() {
        return hasMoreDestinations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deliveryStreamName());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryStreamARN());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryStreamStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureDescription());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryStreamEncryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryStreamTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(versionId());
        hashCode = 31 * hashCode + Objects.hashCode(createTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(hasDestinations() ? destinations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMoreDestinations());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeliveryStreamDescription)) {
            return false;
        }
        DeliveryStreamDescription other = (DeliveryStreamDescription) obj;
        return Objects.equals(deliveryStreamName(), other.deliveryStreamName())
                && Objects.equals(deliveryStreamARN(), other.deliveryStreamARN())
                && Objects.equals(deliveryStreamStatusAsString(), other.deliveryStreamStatusAsString())
                && Objects.equals(failureDescription(), other.failureDescription())
                && Objects.equals(deliveryStreamEncryptionConfiguration(), other.deliveryStreamEncryptionConfiguration())
                && Objects.equals(deliveryStreamTypeAsString(), other.deliveryStreamTypeAsString())
                && Objects.equals(versionId(), other.versionId()) && Objects.equals(createTimestamp(), other.createTimestamp())
                && Objects.equals(lastUpdateTimestamp(), other.lastUpdateTimestamp()) && Objects.equals(source(), other.source())
                && hasDestinations() == other.hasDestinations() && Objects.equals(destinations(), other.destinations())
                && Objects.equals(hasMoreDestinations(), other.hasMoreDestinations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeliveryStreamDescription").add("DeliveryStreamName", deliveryStreamName())
                .add("DeliveryStreamARN", deliveryStreamARN()).add("DeliveryStreamStatus", deliveryStreamStatusAsString())
                .add("FailureDescription", failureDescription())
                .add("DeliveryStreamEncryptionConfiguration", deliveryStreamEncryptionConfiguration())
                .add("DeliveryStreamType", deliveryStreamTypeAsString()).add("VersionId", versionId())
                .add("CreateTimestamp", createTimestamp()).add("LastUpdateTimestamp", lastUpdateTimestamp())
                .add("Source", source()).add("Destinations", hasDestinations() ? destinations() : null)
                .add("HasMoreDestinations", hasMoreDestinations()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DeliveryStreamName":
            return Optional.ofNullable(clazz.cast(deliveryStreamName()));
        case "DeliveryStreamARN":
            return Optional.ofNullable(clazz.cast(deliveryStreamARN()));
        case "DeliveryStreamStatus":
            return Optional.ofNullable(clazz.cast(deliveryStreamStatusAsString()));
        case "FailureDescription":
            return Optional.ofNullable(clazz.cast(failureDescription()));
        case "DeliveryStreamEncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(deliveryStreamEncryptionConfiguration()));
        case "DeliveryStreamType":
            return Optional.ofNullable(clazz.cast(deliveryStreamTypeAsString()));
        case "VersionId":
            return Optional.ofNullable(clazz.cast(versionId()));
        case "CreateTimestamp":
            return Optional.ofNullable(clazz.cast(createTimestamp()));
        case "LastUpdateTimestamp":
            return Optional.ofNullable(clazz.cast(lastUpdateTimestamp()));
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "Destinations":
            return Optional.ofNullable(clazz.cast(destinations()));
        case "HasMoreDestinations":
            return Optional.ofNullable(clazz.cast(hasMoreDestinations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeliveryStreamDescription, T> g) {
        return obj -> g.apply((DeliveryStreamDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeliveryStreamDescription> {
        /**
         * <p>
         * The name of the delivery stream.
         * </p>
         * 
         * @param deliveryStreamName
         *        The name of the delivery stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliveryStreamName(String deliveryStreamName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the delivery stream. For more information, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
         * (ARNs) and AWS Service Namespaces</a>.
         * </p>
         * 
         * @param deliveryStreamARN
         *        The Amazon Resource Name (ARN) of the delivery stream. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
         *        Names (ARNs) and AWS Service Namespaces</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliveryStreamARN(String deliveryStreamARN);

        /**
         * <p>
         * The status of the delivery stream. If the status of a delivery stream is <code>CREATING_FAILED</code>, this
         * status doesn't change, and you can't invoke <code>CreateDeliveryStream</code> again on it. However, you can
         * invoke the <a>DeleteDeliveryStream</a> operation to delete it.
         * </p>
         * 
         * @param deliveryStreamStatus
         *        The status of the delivery stream. If the status of a delivery stream is <code>CREATING_FAILED</code>,
         *        this status doesn't change, and you can't invoke <code>CreateDeliveryStream</code> again on it.
         *        However, you can invoke the <a>DeleteDeliveryStream</a> operation to delete it.
         * @see DeliveryStreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryStreamStatus
         */
        Builder deliveryStreamStatus(String deliveryStreamStatus);

        /**
         * <p>
         * The status of the delivery stream. If the status of a delivery stream is <code>CREATING_FAILED</code>, this
         * status doesn't change, and you can't invoke <code>CreateDeliveryStream</code> again on it. However, you can
         * invoke the <a>DeleteDeliveryStream</a> operation to delete it.
         * </p>
         * 
         * @param deliveryStreamStatus
         *        The status of the delivery stream. If the status of a delivery stream is <code>CREATING_FAILED</code>,
         *        this status doesn't change, and you can't invoke <code>CreateDeliveryStream</code> again on it.
         *        However, you can invoke the <a>DeleteDeliveryStream</a> operation to delete it.
         * @see DeliveryStreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryStreamStatus
         */
        Builder deliveryStreamStatus(DeliveryStreamStatus deliveryStreamStatus);

        /**
         * <p>
         * Provides details in case one of the following operations fails due to an error related to KMS:
         * <a>CreateDeliveryStream</a>, <a>DeleteDeliveryStream</a>, <a>StartDeliveryStreamEncryption</a>,
         * <a>StopDeliveryStreamEncryption</a>.
         * </p>
         * 
         * @param failureDescription
         *        Provides details in case one of the following operations fails due to an error related to KMS:
         *        <a>CreateDeliveryStream</a>, <a>DeleteDeliveryStream</a>, <a>StartDeliveryStreamEncryption</a>,
         *        <a>StopDeliveryStreamEncryption</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureDescription(FailureDescription failureDescription);

        /**
         * <p>
         * Provides details in case one of the following operations fails due to an error related to KMS:
         * <a>CreateDeliveryStream</a>, <a>DeleteDeliveryStream</a>, <a>StartDeliveryStreamEncryption</a>,
         * <a>StopDeliveryStreamEncryption</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link FailureDescription.Builder} avoiding the need to
         * create one manually via {@link FailureDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link FailureDescription.Builder#build()} is called immediately and its
         * result is passed to {@link #failureDescription(FailureDescription)}.
         * 
         * @param failureDescription
         *        a consumer that will call methods on {@link FailureDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureDescription(FailureDescription)
         */
        default Builder failureDescription(Consumer<FailureDescription.Builder> failureDescription) {
            return failureDescription(FailureDescription.builder().applyMutation(failureDescription).build());
        }

        /**
         * <p>
         * Indicates the server-side encryption (SSE) status for the delivery stream.
         * </p>
         * 
         * @param deliveryStreamEncryptionConfiguration
         *        Indicates the server-side encryption (SSE) status for the delivery stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliveryStreamEncryptionConfiguration(DeliveryStreamEncryptionConfiguration deliveryStreamEncryptionConfiguration);

        /**
         * <p>
         * Indicates the server-side encryption (SSE) status for the delivery stream.
         * </p>
         * This is a convenience that creates an instance of the {@link DeliveryStreamEncryptionConfiguration.Builder}
         * avoiding the need to create one manually via {@link DeliveryStreamEncryptionConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link DeliveryStreamEncryptionConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #deliveryStreamEncryptionConfiguration(DeliveryStreamEncryptionConfiguration)}.
         * 
         * @param deliveryStreamEncryptionConfiguration
         *        a consumer that will call methods on {@link DeliveryStreamEncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deliveryStreamEncryptionConfiguration(DeliveryStreamEncryptionConfiguration)
         */
        default Builder deliveryStreamEncryptionConfiguration(
                Consumer<DeliveryStreamEncryptionConfiguration.Builder> deliveryStreamEncryptionConfiguration) {
            return deliveryStreamEncryptionConfiguration(DeliveryStreamEncryptionConfiguration.builder()
                    .applyMutation(deliveryStreamEncryptionConfiguration).build());
        }

        /**
         * <p>
         * The delivery stream type. This can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DirectPut</code>: Provider applications access the delivery stream directly.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis data stream as a source.
         * </p>
         * </li>
         * </ul>
         * 
         * @param deliveryStreamType
         *        The delivery stream type. This can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DirectPut</code>: Provider applications access the delivery stream directly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis data stream as a source.
         *        </p>
         *        </li>
         * @see DeliveryStreamType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryStreamType
         */
        Builder deliveryStreamType(String deliveryStreamType);

        /**
         * <p>
         * The delivery stream type. This can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DirectPut</code>: Provider applications access the delivery stream directly.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis data stream as a source.
         * </p>
         * </li>
         * </ul>
         * 
         * @param deliveryStreamType
         *        The delivery stream type. This can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DirectPut</code>: Provider applications access the delivery stream directly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KinesisStreamAsSource</code>: The delivery stream uses a Kinesis data stream as a source.
         *        </p>
         *        </li>
         * @see DeliveryStreamType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryStreamType
         */
        Builder deliveryStreamType(DeliveryStreamType deliveryStreamType);

        /**
         * <p>
         * Each time the destination is updated for a delivery stream, the version ID is changed, and the current
         * version ID is required when updating the destination. This is so that the service knows it is applying the
         * changes to the correct version of the delivery stream.
         * </p>
         * 
         * @param versionId
         *        Each time the destination is updated for a delivery stream, the version ID is changed, and the current
         *        version ID is required when updating the destination. This is so that the service knows it is applying
         *        the changes to the correct version of the delivery stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionId(String versionId);

        /**
         * <p>
         * The date and time that the delivery stream was created.
         * </p>
         * 
         * @param createTimestamp
         *        The date and time that the delivery stream was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTimestamp(Instant createTimestamp);

        /**
         * <p>
         * The date and time that the delivery stream was last updated.
         * </p>
         * 
         * @param lastUpdateTimestamp
         *        The date and time that the delivery stream was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTimestamp(Instant lastUpdateTimestamp);

        /**
         * <p>
         * If the <code>DeliveryStreamType</code> parameter is <code>KinesisStreamAsSource</code>, a
         * <a>SourceDescription</a> object describing the source Kinesis data stream.
         * </p>
         * 
         * @param source
         *        If the <code>DeliveryStreamType</code> parameter is <code>KinesisStreamAsSource</code>, a
         *        <a>SourceDescription</a> object describing the source Kinesis data stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(SourceDescription source);

        /**
         * <p>
         * If the <code>DeliveryStreamType</code> parameter is <code>KinesisStreamAsSource</code>, a
         * <a>SourceDescription</a> object describing the source Kinesis data stream.
         * </p>
         * This is a convenience that creates an instance of the {@link SourceDescription.Builder} avoiding the need to
         * create one manually via {@link SourceDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link SourceDescription.Builder#build()} is called immediately and its
         * result is passed to {@link #source(SourceDescription)}.
         * 
         * @param source
         *        a consumer that will call methods on {@link SourceDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #source(SourceDescription)
         */
        default Builder source(Consumer<SourceDescription.Builder> source) {
            return source(SourceDescription.builder().applyMutation(source).build());
        }

        /**
         * <p>
         * The destinations.
         * </p>
         * 
         * @param destinations
         *        The destinations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<DestinationDescription> destinations);

        /**
         * <p>
         * The destinations.
         * </p>
         * 
         * @param destinations
         *        The destinations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(DestinationDescription... destinations);

        /**
         * <p>
         * The destinations.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DestinationDescription>.Builder} avoiding
         * the need to create one manually via {@link List<DestinationDescription>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DestinationDescription>.Builder#build()} is called
         * immediately and its result is passed to {@link #destinations(List<DestinationDescription>)}.
         * 
         * @param destinations
         *        a consumer that will call methods on {@link List<DestinationDescription>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinations(List<DestinationDescription>)
         */
        Builder destinations(Consumer<DestinationDescription.Builder>... destinations);

        /**
         * <p>
         * Indicates whether there are more destinations available to list.
         * </p>
         * 
         * @param hasMoreDestinations
         *        Indicates whether there are more destinations available to list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hasMoreDestinations(Boolean hasMoreDestinations);
    }

    static final class BuilderImpl implements Builder {
        private String deliveryStreamName;

        private String deliveryStreamARN;

        private String deliveryStreamStatus;

        private FailureDescription failureDescription;

        private DeliveryStreamEncryptionConfiguration deliveryStreamEncryptionConfiguration;

        private String deliveryStreamType;

        private String versionId;

        private Instant createTimestamp;

        private Instant lastUpdateTimestamp;

        private SourceDescription source;

        private List<DestinationDescription> destinations = DefaultSdkAutoConstructList.getInstance();

        private Boolean hasMoreDestinations;

        private BuilderImpl() {
        }

        private BuilderImpl(DeliveryStreamDescription model) {
            deliveryStreamName(model.deliveryStreamName);
            deliveryStreamARN(model.deliveryStreamARN);
            deliveryStreamStatus(model.deliveryStreamStatus);
            failureDescription(model.failureDescription);
            deliveryStreamEncryptionConfiguration(model.deliveryStreamEncryptionConfiguration);
            deliveryStreamType(model.deliveryStreamType);
            versionId(model.versionId);
            createTimestamp(model.createTimestamp);
            lastUpdateTimestamp(model.lastUpdateTimestamp);
            source(model.source);
            destinations(model.destinations);
            hasMoreDestinations(model.hasMoreDestinations);
        }

        public final String getDeliveryStreamName() {
            return deliveryStreamName;
        }

        public final void setDeliveryStreamName(String deliveryStreamName) {
            this.deliveryStreamName = deliveryStreamName;
        }

        @Override
        @Transient
        public final Builder deliveryStreamName(String deliveryStreamName) {
            this.deliveryStreamName = deliveryStreamName;
            return this;
        }

        public final String getDeliveryStreamARN() {
            return deliveryStreamARN;
        }

        public final void setDeliveryStreamARN(String deliveryStreamARN) {
            this.deliveryStreamARN = deliveryStreamARN;
        }

        @Override
        @Transient
        public final Builder deliveryStreamARN(String deliveryStreamARN) {
            this.deliveryStreamARN = deliveryStreamARN;
            return this;
        }

        public final String getDeliveryStreamStatus() {
            return deliveryStreamStatus;
        }

        public final void setDeliveryStreamStatus(String deliveryStreamStatus) {
            this.deliveryStreamStatus = deliveryStreamStatus;
        }

        @Override
        @Transient
        public final Builder deliveryStreamStatus(String deliveryStreamStatus) {
            this.deliveryStreamStatus = deliveryStreamStatus;
            return this;
        }

        @Override
        @Transient
        public final Builder deliveryStreamStatus(DeliveryStreamStatus deliveryStreamStatus) {
            this.deliveryStreamStatus(deliveryStreamStatus == null ? null : deliveryStreamStatus.toString());
            return this;
        }

        public final FailureDescription.Builder getFailureDescription() {
            return failureDescription != null ? failureDescription.toBuilder() : null;
        }

        public final void setFailureDescription(FailureDescription.BuilderImpl failureDescription) {
            this.failureDescription = failureDescription != null ? failureDescription.build() : null;
        }

        @Override
        @Transient
        public final Builder failureDescription(FailureDescription failureDescription) {
            this.failureDescription = failureDescription;
            return this;
        }

        public final DeliveryStreamEncryptionConfiguration.Builder getDeliveryStreamEncryptionConfiguration() {
            return deliveryStreamEncryptionConfiguration != null ? deliveryStreamEncryptionConfiguration.toBuilder() : null;
        }

        public final void setDeliveryStreamEncryptionConfiguration(
                DeliveryStreamEncryptionConfiguration.BuilderImpl deliveryStreamEncryptionConfiguration) {
            this.deliveryStreamEncryptionConfiguration = deliveryStreamEncryptionConfiguration != null ? deliveryStreamEncryptionConfiguration
                    .build() : null;
        }

        @Override
        @Transient
        public final Builder deliveryStreamEncryptionConfiguration(
                DeliveryStreamEncryptionConfiguration deliveryStreamEncryptionConfiguration) {
            this.deliveryStreamEncryptionConfiguration = deliveryStreamEncryptionConfiguration;
            return this;
        }

        public final String getDeliveryStreamType() {
            return deliveryStreamType;
        }

        public final void setDeliveryStreamType(String deliveryStreamType) {
            this.deliveryStreamType = deliveryStreamType;
        }

        @Override
        @Transient
        public final Builder deliveryStreamType(String deliveryStreamType) {
            this.deliveryStreamType = deliveryStreamType;
            return this;
        }

        @Override
        @Transient
        public final Builder deliveryStreamType(DeliveryStreamType deliveryStreamType) {
            this.deliveryStreamType(deliveryStreamType == null ? null : deliveryStreamType.toString());
            return this;
        }

        public final String getVersionId() {
            return versionId;
        }

        public final void setVersionId(String versionId) {
            this.versionId = versionId;
        }

        @Override
        @Transient
        public final Builder versionId(String versionId) {
            this.versionId = versionId;
            return this;
        }

        public final Instant getCreateTimestamp() {
            return createTimestamp;
        }

        public final void setCreateTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
        }

        @Override
        @Transient
        public final Builder createTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
            return this;
        }

        public final Instant getLastUpdateTimestamp() {
            return lastUpdateTimestamp;
        }

        public final void setLastUpdateTimestamp(Instant lastUpdateTimestamp) {
            this.lastUpdateTimestamp = lastUpdateTimestamp;
        }

        @Override
        @Transient
        public final Builder lastUpdateTimestamp(Instant lastUpdateTimestamp) {
            this.lastUpdateTimestamp = lastUpdateTimestamp;
            return this;
        }

        public final SourceDescription.Builder getSource() {
            return source != null ? source.toBuilder() : null;
        }

        public final void setSource(SourceDescription.BuilderImpl source) {
            this.source = source != null ? source.build() : null;
        }

        @Override
        @Transient
        public final Builder source(SourceDescription source) {
            this.source = source;
            return this;
        }

        public final List<DestinationDescription.Builder> getDestinations() {
            List<DestinationDescription.Builder> result = DestinationDescriptionListCopier.copyToBuilder(this.destinations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDestinations(Collection<DestinationDescription.BuilderImpl> destinations) {
            this.destinations = DestinationDescriptionListCopier.copyFromBuilder(destinations);
        }

        @Override
        @Transient
        public final Builder destinations(Collection<DestinationDescription> destinations) {
            this.destinations = DestinationDescriptionListCopier.copy(destinations);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder destinations(DestinationDescription... destinations) {
            destinations(Arrays.asList(destinations));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder destinations(Consumer<DestinationDescription.Builder>... destinations) {
            destinations(Stream.of(destinations).map(c -> DestinationDescription.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getHasMoreDestinations() {
            return hasMoreDestinations;
        }

        public final void setHasMoreDestinations(Boolean hasMoreDestinations) {
            this.hasMoreDestinations = hasMoreDestinations;
        }

        @Override
        @Transient
        public final Builder hasMoreDestinations(Boolean hasMoreDestinations) {
            this.hasMoreDestinations = hasMoreDestinations;
            return this;
        }

        @Override
        public DeliveryStreamDescription build() {
            return new DeliveryStreamDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
