/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fms;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.fms.model.AssociateAdminAccountRequest;
import software.amazon.awssdk.services.fms.model.AssociateAdminAccountResponse;
import software.amazon.awssdk.services.fms.model.DeleteAppsListRequest;
import software.amazon.awssdk.services.fms.model.DeleteAppsListResponse;
import software.amazon.awssdk.services.fms.model.DeleteNotificationChannelRequest;
import software.amazon.awssdk.services.fms.model.DeleteNotificationChannelResponse;
import software.amazon.awssdk.services.fms.model.DeletePolicyRequest;
import software.amazon.awssdk.services.fms.model.DeletePolicyResponse;
import software.amazon.awssdk.services.fms.model.DeleteProtocolsListRequest;
import software.amazon.awssdk.services.fms.model.DeleteProtocolsListResponse;
import software.amazon.awssdk.services.fms.model.DisassociateAdminAccountRequest;
import software.amazon.awssdk.services.fms.model.DisassociateAdminAccountResponse;
import software.amazon.awssdk.services.fms.model.FmsException;
import software.amazon.awssdk.services.fms.model.FmsRequest;
import software.amazon.awssdk.services.fms.model.GetAdminAccountRequest;
import software.amazon.awssdk.services.fms.model.GetAdminAccountResponse;
import software.amazon.awssdk.services.fms.model.GetAppsListRequest;
import software.amazon.awssdk.services.fms.model.GetAppsListResponse;
import software.amazon.awssdk.services.fms.model.GetComplianceDetailRequest;
import software.amazon.awssdk.services.fms.model.GetComplianceDetailResponse;
import software.amazon.awssdk.services.fms.model.GetNotificationChannelRequest;
import software.amazon.awssdk.services.fms.model.GetNotificationChannelResponse;
import software.amazon.awssdk.services.fms.model.GetPolicyRequest;
import software.amazon.awssdk.services.fms.model.GetPolicyResponse;
import software.amazon.awssdk.services.fms.model.GetProtectionStatusRequest;
import software.amazon.awssdk.services.fms.model.GetProtectionStatusResponse;
import software.amazon.awssdk.services.fms.model.GetProtocolsListRequest;
import software.amazon.awssdk.services.fms.model.GetProtocolsListResponse;
import software.amazon.awssdk.services.fms.model.GetViolationDetailsRequest;
import software.amazon.awssdk.services.fms.model.GetViolationDetailsResponse;
import software.amazon.awssdk.services.fms.model.InternalErrorException;
import software.amazon.awssdk.services.fms.model.InvalidInputException;
import software.amazon.awssdk.services.fms.model.InvalidOperationException;
import software.amazon.awssdk.services.fms.model.InvalidTypeException;
import software.amazon.awssdk.services.fms.model.LimitExceededException;
import software.amazon.awssdk.services.fms.model.ListAppsListsRequest;
import software.amazon.awssdk.services.fms.model.ListAppsListsResponse;
import software.amazon.awssdk.services.fms.model.ListComplianceStatusRequest;
import software.amazon.awssdk.services.fms.model.ListComplianceStatusResponse;
import software.amazon.awssdk.services.fms.model.ListMemberAccountsRequest;
import software.amazon.awssdk.services.fms.model.ListMemberAccountsResponse;
import software.amazon.awssdk.services.fms.model.ListPoliciesRequest;
import software.amazon.awssdk.services.fms.model.ListPoliciesResponse;
import software.amazon.awssdk.services.fms.model.ListProtocolsListsRequest;
import software.amazon.awssdk.services.fms.model.ListProtocolsListsResponse;
import software.amazon.awssdk.services.fms.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.fms.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.fms.model.PutAppsListRequest;
import software.amazon.awssdk.services.fms.model.PutAppsListResponse;
import software.amazon.awssdk.services.fms.model.PutNotificationChannelRequest;
import software.amazon.awssdk.services.fms.model.PutNotificationChannelResponse;
import software.amazon.awssdk.services.fms.model.PutPolicyRequest;
import software.amazon.awssdk.services.fms.model.PutPolicyResponse;
import software.amazon.awssdk.services.fms.model.PutProtocolsListRequest;
import software.amazon.awssdk.services.fms.model.PutProtocolsListResponse;
import software.amazon.awssdk.services.fms.model.ResourceNotFoundException;
import software.amazon.awssdk.services.fms.model.TagResourceRequest;
import software.amazon.awssdk.services.fms.model.TagResourceResponse;
import software.amazon.awssdk.services.fms.model.UntagResourceRequest;
import software.amazon.awssdk.services.fms.model.UntagResourceResponse;
import software.amazon.awssdk.services.fms.paginators.ListComplianceStatusPublisher;
import software.amazon.awssdk.services.fms.paginators.ListMemberAccountsPublisher;
import software.amazon.awssdk.services.fms.paginators.ListPoliciesPublisher;
import software.amazon.awssdk.services.fms.transform.AssociateAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeleteAppsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeleteNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeletePolicyRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeleteProtocolsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DisassociateAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetAppsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetComplianceDetailRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetPolicyRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetProtectionStatusRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetProtocolsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetViolationDetailsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListAppsListsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListComplianceStatusRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListMemberAccountsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListPoliciesRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListProtocolsListsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutAppsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutPolicyRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutProtocolsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link FmsAsyncClient}.
 *
 * @see FmsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFmsAsyncClient implements FmsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultFmsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultFmsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Sets the AWS Firewall Manager administrator account. AWS Firewall Manager must be associated with the master
     * account of your AWS organization or associated with a member account that has the appropriate permissions. If the
     * account ID that you submit is not an AWS Organizations master account, AWS Firewall Manager will set the
     * appropriate permissions for the given member account.
     * </p>
     * <p>
     * The account that you associate with AWS Firewall Manager is called the AWS Firewall Manager administrator
     * account.
     * </p>
     *
     * @param associateAdminAccountRequest
     * @return A Java Future containing the result of the AssociateAdminAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.AssociateAdminAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/AssociateAdminAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateAdminAccountResponse> associateAdminAccount(
            AssociateAdminAccountRequest associateAdminAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateAdminAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateAdminAccountRequest, AssociateAdminAccountResponse>()
                            .withOperationName("AssociateAdminAccount")
                            .withMarshaller(new AssociateAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateAdminAccountRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = associateAdminAccountRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<AssociateAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes an AWS Firewall Manager applications list.
     * </p>
     *
     * @param deleteAppsListRequest
     * @return A Java Future containing the result of the DeleteAppsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DeleteAppsList
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeleteAppsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppsListResponse> deleteAppsList(DeleteAppsListRequest deleteAppsListRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppsListResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppsListRequest, DeleteAppsListResponse>()
                            .withOperationName("DeleteAppsList")
                            .withMarshaller(new DeleteAppsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAppsListRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteAppsListRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteAppsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an AWS Firewall Manager association with the IAM role and the Amazon Simple Notification Service (SNS)
     * topic that is used to record AWS Firewall Manager SNS logs.
     * </p>
     *
     * @param deleteNotificationChannelRequest
     * @return A Java Future containing the result of the DeleteNotificationChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DeleteNotificationChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeleteNotificationChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNotificationChannelResponse> deleteNotificationChannel(
            DeleteNotificationChannelRequest deleteNotificationChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNotificationChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNotificationChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNotificationChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNotificationChannelRequest, DeleteNotificationChannelResponse>()
                            .withOperationName("DeleteNotificationChannel")
                            .withMarshaller(new DeleteNotificationChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteNotificationChannelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteNotificationChannelRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteNotificationChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes an AWS Firewall Manager policy.
     * </p>
     *
     * @param deletePolicyRequest
     * @return A Java Future containing the result of the DeletePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DeletePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeletePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePolicyResponse> deletePolicy(DeletePolicyRequest deletePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeletePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePolicyRequest, DeletePolicyResponse>()
                            .withOperationName("DeletePolicy").withMarshaller(new DeletePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deletePolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeletePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes an AWS Firewall Manager protocols list.
     * </p>
     *
     * @param deleteProtocolsListRequest
     * @return A Java Future containing the result of the DeleteProtocolsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DeleteProtocolsList
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeleteProtocolsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProtocolsListResponse> deleteProtocolsList(
            DeleteProtocolsListRequest deleteProtocolsListRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProtocolsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProtocolsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProtocolsListResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteProtocolsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProtocolsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProtocolsListRequest, DeleteProtocolsListResponse>()
                            .withOperationName("DeleteProtocolsList")
                            .withMarshaller(new DeleteProtocolsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProtocolsListRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteProtocolsListRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteProtocolsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the account that has been set as the AWS Firewall Manager administrator account. To set a different
     * account as the administrator account, you must submit an <code>AssociateAdminAccount</code> request.
     * </p>
     *
     * @param disassociateAdminAccountRequest
     * @return A Java Future containing the result of the DisassociateAdminAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DisassociateAdminAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DisassociateAdminAccount" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateAdminAccountResponse> disassociateAdminAccount(
            DisassociateAdminAccountRequest disassociateAdminAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateAdminAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateAdminAccountRequest, DisassociateAdminAccountResponse>()
                            .withOperationName("DisassociateAdminAccount")
                            .withMarshaller(new DisassociateAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateAdminAccountRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disassociateAdminAccountRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DisassociateAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the AWS Organizations master account that is associated with AWS Firewall Manager as the AWS Firewall
     * Manager administrator.
     * </p>
     *
     * @param getAdminAccountRequest
     * @return A Java Future containing the result of the GetAdminAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetAdminAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetAdminAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAdminAccountResponse> getAdminAccount(GetAdminAccountRequest getAdminAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAdminAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAdminAccountRequest, GetAdminAccountResponse>()
                            .withOperationName("GetAdminAccount")
                            .withMarshaller(new GetAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAdminAccountRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAdminAccountRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified AWS Firewall Manager applications list.
     * </p>
     *
     * @param getAppsListRequest
     * @return A Java Future containing the result of the GetAppsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetAppsList
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetAppsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppsListResponse> getAppsList(GetAppsListRequest getAppsListRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppsListResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAppsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppsListRequest, GetAppsListResponse>()
                            .withOperationName("GetAppsList").withMarshaller(new GetAppsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAppsListRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAppsListRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetAppsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed compliance information about the specified member account. Details include resources that are in
     * and out of compliance with the specified policy. Resources are considered noncompliant for AWS WAF and Shield
     * Advanced policies if the specified policy has not been applied to them. Resources are considered noncompliant for
     * security group policies if they are in scope of the policy, they violate one or more of the policy rules, and
     * remediation is disabled or not possible.
     * </p>
     *
     * @param getComplianceDetailRequest
     * @return A Java Future containing the result of the GetComplianceDetail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetComplianceDetail
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetComplianceDetail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetComplianceDetailResponse> getComplianceDetail(
            GetComplianceDetailRequest getComplianceDetailRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComplianceDetailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComplianceDetail");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetComplianceDetailResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetComplianceDetailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetComplianceDetailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetComplianceDetailRequest, GetComplianceDetailResponse>()
                            .withOperationName("GetComplianceDetail")
                            .withMarshaller(new GetComplianceDetailRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getComplianceDetailRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getComplianceDetailRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetComplianceDetailResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Information about the Amazon Simple Notification Service (SNS) topic that is used to record AWS Firewall Manager
     * SNS logs.
     * </p>
     *
     * @param getNotificationChannelRequest
     * @return A Java Future containing the result of the GetNotificationChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetNotificationChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetNotificationChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetNotificationChannelResponse> getNotificationChannel(
            GetNotificationChannelRequest getNotificationChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNotificationChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetNotificationChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNotificationChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNotificationChannelRequest, GetNotificationChannelResponse>()
                            .withOperationName("GetNotificationChannel")
                            .withMarshaller(new GetNotificationChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getNotificationChannelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getNotificationChannelRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetNotificationChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified AWS Firewall Manager policy.
     * </p>
     *
     * @param getPolicyRequest
     * @return A Java Future containing the result of the GetPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidTypeException The value of the <code>Type</code> parameter is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPolicyResponse> getPolicy(GetPolicyRequest getPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPolicyRequest, GetPolicyResponse>().withOperationName("GetPolicy")
                            .withMarshaller(new GetPolicyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getPolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * If you created a Shield Advanced policy, returns policy-level attack summary information in the event of a
     * potential DDoS attack. Other policy types are currently unsupported.
     * </p>
     *
     * @param getProtectionStatusRequest
     * @return A Java Future containing the result of the GetProtectionStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetProtectionStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetProtectionStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetProtectionStatusResponse> getProtectionStatus(
            GetProtectionStatusRequest getProtectionStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProtectionStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProtectionStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProtectionStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetProtectionStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProtectionStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProtectionStatusRequest, GetProtectionStatusResponse>()
                            .withOperationName("GetProtectionStatus")
                            .withMarshaller(new GetProtectionStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getProtectionStatusRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getProtectionStatusRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetProtectionStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified AWS Firewall Manager protocols list.
     * </p>
     *
     * @param getProtocolsListRequest
     * @return A Java Future containing the result of the GetProtocolsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetProtocolsList
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetProtocolsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetProtocolsListResponse> getProtocolsList(GetProtocolsListRequest getProtocolsListRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProtocolsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProtocolsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProtocolsListResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetProtocolsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProtocolsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProtocolsListRequest, GetProtocolsListResponse>()
                            .withOperationName("GetProtocolsList")
                            .withMarshaller(new GetProtocolsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getProtocolsListRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getProtocolsListRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetProtocolsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves violations for a resource based on the specified AWS Firewall Manager policy and AWS account.
     * </p>
     *
     * @param getViolationDetailsRequest
     * @return A Java Future containing the result of the GetViolationDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetViolationDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetViolationDetails" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetViolationDetailsResponse> getViolationDetails(
            GetViolationDetailsRequest getViolationDetailsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getViolationDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetViolationDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetViolationDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetViolationDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetViolationDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetViolationDetailsRequest, GetViolationDetailsResponse>()
                            .withOperationName("GetViolationDetails")
                            .withMarshaller(new GetViolationDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getViolationDetailsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getViolationDetailsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetViolationDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>AppsListDataSummary</code> objects.
     * </p>
     *
     * @param listAppsListsRequest
     * @return A Java Future containing the result of the ListAppsLists operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an AWS account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>AWS WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListAppsLists
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListAppsLists" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppsListsResponse> listAppsLists(ListAppsListsRequest listAppsListsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppsListsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppsLists");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppsListsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAppsListsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppsListsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppsListsRequest, ListAppsListsResponse>()
                            .withOperationName("ListAppsLists")
                            .withMarshaller(new ListAppsListsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAppsListsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAppsListsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListAppsListsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>PolicyComplianceStatus</code> objects. Use <code>PolicyComplianceStatus</code> to get a
     * summary of which member accounts are protected by the specified policy.
     * </p>
     *
     * @param listComplianceStatusRequest
     * @return A Java Future containing the result of the ListComplianceStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListComplianceStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListComplianceStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListComplianceStatusResponse> listComplianceStatus(
            ListComplianceStatusRequest listComplianceStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComplianceStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComplianceStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListComplianceStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListComplianceStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListComplianceStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListComplianceStatusRequest, ListComplianceStatusResponse>()
                            .withOperationName("ListComplianceStatus")
                            .withMarshaller(new ListComplianceStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listComplianceStatusRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listComplianceStatusRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListComplianceStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>PolicyComplianceStatus</code> objects. Use <code>PolicyComplianceStatus</code> to get a
     * summary of which member accounts are protected by the specified policy.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listComplianceStatus(software.amazon.awssdk.services.fms.model.ListComplianceStatusRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fms.paginators.ListComplianceStatusPublisher publisher = client.listComplianceStatusPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fms.paginators.ListComplianceStatusPublisher publisher = client.listComplianceStatusPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.fms.model.ListComplianceStatusResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.fms.model.ListComplianceStatusResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listComplianceStatus(software.amazon.awssdk.services.fms.model.ListComplianceStatusRequest)}
     * operation.</b>
     * </p>
     *
     * @param listComplianceStatusRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListComplianceStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListComplianceStatus" target="_top">AWS API
     *      Documentation</a>
     */
    public ListComplianceStatusPublisher listComplianceStatusPaginator(ListComplianceStatusRequest listComplianceStatusRequest) {
        return new ListComplianceStatusPublisher(this, applyPaginatorUserAgent(listComplianceStatusRequest));
    }

    /**
     * <p>
     * Returns a <code>MemberAccounts</code> object that lists the member accounts in the administrator's AWS
     * organization.
     * </p>
     * <p>
     * The <code>ListMemberAccounts</code> must be submitted by the account that is set as the AWS Firewall Manager
     * administrator.
     * </p>
     *
     * @param listMemberAccountsRequest
     * @return A Java Future containing the result of the ListMemberAccounts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListMemberAccounts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMemberAccountsResponse> listMemberAccounts(ListMemberAccountsRequest listMemberAccountsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMemberAccountsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMemberAccounts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMemberAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMemberAccountsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMemberAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMemberAccountsRequest, ListMemberAccountsResponse>()
                            .withOperationName("ListMemberAccounts")
                            .withMarshaller(new ListMemberAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMemberAccountsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listMemberAccountsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListMemberAccountsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a <code>MemberAccounts</code> object that lists the member accounts in the administrator's AWS
     * organization.
     * </p>
     * <p>
     * The <code>ListMemberAccounts</code> must be submitted by the account that is set as the AWS Firewall Manager
     * administrator.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMemberAccounts(software.amazon.awssdk.services.fms.model.ListMemberAccountsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fms.paginators.ListMemberAccountsPublisher publisher = client.listMemberAccountsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fms.paginators.ListMemberAccountsPublisher publisher = client.listMemberAccountsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.fms.model.ListMemberAccountsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.fms.model.ListMemberAccountsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMemberAccounts(software.amazon.awssdk.services.fms.model.ListMemberAccountsRequest)} operation.</b>
     * </p>
     *
     * @param listMemberAccountsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListMemberAccounts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    public ListMemberAccountsPublisher listMemberAccountsPaginator(ListMemberAccountsRequest listMemberAccountsRequest) {
        return new ListMemberAccountsPublisher(this, applyPaginatorUserAgent(listMemberAccountsRequest));
    }

    /**
     * <p>
     * Returns an array of <code>PolicySummary</code> objects.
     * </p>
     *
     * @param listPoliciesRequest
     * @return A Java Future containing the result of the ListPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an AWS account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>AWS WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListPolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListPolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPoliciesResponse> listPolicies(ListPoliciesRequest listPoliciesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPoliciesRequest, ListPoliciesResponse>()
                            .withOperationName("ListPolicies").withMarshaller(new ListPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPoliciesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPoliciesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListPoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>PolicySummary</code> objects.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listPolicies(software.amazon.awssdk.services.fms.model.ListPoliciesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fms.paginators.ListPoliciesPublisher publisher = client.listPoliciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fms.paginators.ListPoliciesPublisher publisher = client.listPoliciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.fms.model.ListPoliciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.fms.model.ListPoliciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPolicies(software.amazon.awssdk.services.fms.model.ListPoliciesRequest)} operation.</b>
     * </p>
     *
     * @param listPoliciesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an AWS account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>AWS WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListPolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListPolicies" target="_top">AWS API
     *      Documentation</a>
     */
    public ListPoliciesPublisher listPoliciesPaginator(ListPoliciesRequest listPoliciesRequest) {
        return new ListPoliciesPublisher(this, applyPaginatorUserAgent(listPoliciesRequest));
    }

    /**
     * <p>
     * Returns an array of <code>ProtocolsListDataSummary</code> objects.
     * </p>
     *
     * @param listProtocolsListsRequest
     * @return A Java Future containing the result of the ListProtocolsLists operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListProtocolsLists
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListProtocolsLists" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProtocolsListsResponse> listProtocolsLists(ListProtocolsListsRequest listProtocolsListsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProtocolsListsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProtocolsLists");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProtocolsListsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProtocolsListsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProtocolsListsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProtocolsListsRequest, ListProtocolsListsResponse>()
                            .withOperationName("ListProtocolsLists")
                            .withMarshaller(new ListProtocolsListsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProtocolsListsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listProtocolsListsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListProtocolsListsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of tags for the specified AWS resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AWS Firewall Manager applications list.
     * </p>
     *
     * @param putAppsListRequest
     * @return A Java Future containing the result of the PutAppsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an AWS account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>AWS WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.PutAppsList
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutAppsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutAppsListResponse> putAppsList(PutAppsListRequest putAppsListRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAppsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAppsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAppsListResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutAppsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAppsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAppsListRequest, PutAppsListResponse>()
                            .withOperationName("PutAppsList").withMarshaller(new PutAppsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putAppsListRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putAppsListRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutAppsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Designates the IAM role and Amazon Simple Notification Service (SNS) topic that AWS Firewall Manager uses to
     * record SNS logs.
     * </p>
     *
     * @param putNotificationChannelRequest
     * @return A Java Future containing the result of the PutNotificationChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.PutNotificationChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutNotificationChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutNotificationChannelResponse> putNotificationChannel(
            PutNotificationChannelRequest putNotificationChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutNotificationChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutNotificationChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutNotificationChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutNotificationChannelRequest, PutNotificationChannelResponse>()
                            .withOperationName("PutNotificationChannel")
                            .withMarshaller(new PutNotificationChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putNotificationChannelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putNotificationChannelRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<PutNotificationChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AWS Firewall Manager policy.
     * </p>
     * <p>
     * Firewall Manager provides the following types of policies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A Shield Advanced policy, which applies Shield Advanced protection to specified accounts and resources
     * </p>
     * </li>
     * <li>
     * <p>
     * An AWS WAF policy (type WAFV2), which defines rule groups to run first in the corresponding AWS WAF web ACL and
     * rule groups to run last in the web ACL.
     * </p>
     * </li>
     * <li>
     * <p>
     * An AWS WAF Classic policy (type WAF), which defines a rule group.
     * </p>
     * </li>
     * <li>
     * <p>
     * A security group policy, which manages VPC security groups across your AWS organization.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Each policy is specific to one of the types. If you want to enforce more than one policy type across accounts,
     * create multiple policies. You can create multiple policies for each type.
     * </p>
     * <p>
     * You must be subscribed to Shield Advanced to create a Shield Advanced policy. For more information about
     * subscribing to Shield Advanced, see <a
     * href="https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_CreateSubscription.html"
     * >CreateSubscription</a>.
     * </p>
     *
     * @param putPolicyRequest
     * @return A Java Future containing the result of the PutPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an AWS account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>AWS WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidTypeException The value of the <code>Type</code> parameter is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.PutPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutPolicyResponse> putPolicy(PutPolicyRequest putPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPolicyRequest, PutPolicyResponse>().withOperationName("PutPolicy")
                            .withMarshaller(new PutPolicyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(putPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putPolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AWS Firewall Manager protocols list.
     * </p>
     *
     * @param putProtocolsListRequest
     * @return A Java Future containing the result of the PutProtocolsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an AWS account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>AWS WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.PutProtocolsList
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutProtocolsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutProtocolsListResponse> putProtocolsList(PutProtocolsListRequest putProtocolsListRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putProtocolsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutProtocolsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutProtocolsListResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutProtocolsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutProtocolsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutProtocolsListRequest, PutProtocolsListResponse>()
                            .withOperationName("PutProtocolsList")
                            .withMarshaller(new PutProtocolsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putProtocolsListRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putProtocolsListRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutProtocolsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to an AWS resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an AWS account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>AWS WAF Developer Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from an AWS resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the AWS Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for AWS Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FmsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOperationException")
                                .exceptionBuilderSupplier(InvalidOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTypeException")
                                .exceptionBuilderSupplier(InvalidTypeException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends FmsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
