/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fms.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An Firewall Manager policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Policy implements SdkPojo, Serializable, ToCopyableBuilder<Policy.Builder, Policy> {
    private static final SdkField<String> POLICY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyId").getter(getter(Policy::policyId)).setter(setter(Builder::policyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyId").build()).build();

    private static final SdkField<String> POLICY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyName").getter(getter(Policy::policyName)).setter(setter(Builder::policyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyName").build()).build();

    private static final SdkField<String> POLICY_UPDATE_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyUpdateToken").getter(getter(Policy::policyUpdateToken)).setter(setter(Builder::policyUpdateToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyUpdateToken").build()).build();

    private static final SdkField<SecurityServicePolicyData> SECURITY_SERVICE_POLICY_DATA_FIELD = SdkField
            .<SecurityServicePolicyData> builder(MarshallingType.SDK_POJO).memberName("SecurityServicePolicyData")
            .getter(getter(Policy::securityServicePolicyData)).setter(setter(Builder::securityServicePolicyData))
            .constructor(SecurityServicePolicyData::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityServicePolicyData").build())
            .build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(Policy::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<List<String>> RESOURCE_TYPE_LIST_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceTypeList")
            .getter(getter(Policy::resourceTypeList))
            .setter(setter(Builder::resourceTypeList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTypeList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ResourceTag>> RESOURCE_TAGS_FIELD = SdkField
            .<List<ResourceTag>> builder(MarshallingType.LIST)
            .memberName("ResourceTags")
            .getter(getter(Policy::resourceTags))
            .setter(setter(Builder::resourceTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> EXCLUDE_RESOURCE_TAGS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ExcludeResourceTags").getter(getter(Policy::excludeResourceTags))
            .setter(setter(Builder::excludeResourceTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludeResourceTags").build())
            .build();

    private static final SdkField<Boolean> REMEDIATION_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RemediationEnabled").getter(getter(Policy::remediationEnabled))
            .setter(setter(Builder::remediationEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemediationEnabled").build())
            .build();

    private static final SdkField<Map<String, List<String>>> INCLUDE_MAP_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("IncludeMap")
            .getter(getter(Policy::includeMapAsStrings))
            .setter(setter(Builder::includeMapWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeMap").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<Map<String, List<String>>> EXCLUDE_MAP_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("ExcludeMap")
            .getter(getter(Policy::excludeMapAsStrings))
            .setter(setter(Builder::excludeMapWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludeMap").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POLICY_ID_FIELD,
            POLICY_NAME_FIELD, POLICY_UPDATE_TOKEN_FIELD, SECURITY_SERVICE_POLICY_DATA_FIELD, RESOURCE_TYPE_FIELD,
            RESOURCE_TYPE_LIST_FIELD, RESOURCE_TAGS_FIELD, EXCLUDE_RESOURCE_TAGS_FIELD, REMEDIATION_ENABLED_FIELD,
            INCLUDE_MAP_FIELD, EXCLUDE_MAP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String policyId;

    private final String policyName;

    private final String policyUpdateToken;

    private final SecurityServicePolicyData securityServicePolicyData;

    private final String resourceType;

    private final List<String> resourceTypeList;

    private final List<ResourceTag> resourceTags;

    private final Boolean excludeResourceTags;

    private final Boolean remediationEnabled;

    private final Map<String, List<String>> includeMap;

    private final Map<String, List<String>> excludeMap;

    private Policy(BuilderImpl builder) {
        this.policyId = builder.policyId;
        this.policyName = builder.policyName;
        this.policyUpdateToken = builder.policyUpdateToken;
        this.securityServicePolicyData = builder.securityServicePolicyData;
        this.resourceType = builder.resourceType;
        this.resourceTypeList = builder.resourceTypeList;
        this.resourceTags = builder.resourceTags;
        this.excludeResourceTags = builder.excludeResourceTags;
        this.remediationEnabled = builder.remediationEnabled;
        this.includeMap = builder.includeMap;
        this.excludeMap = builder.excludeMap;
    }

    /**
     * <p>
     * The ID of the Firewall Manager policy.
     * </p>
     * 
     * @return The ID of the Firewall Manager policy.
     */
    public final String policyId() {
        return policyId;
    }

    /**
     * <p>
     * The name of the Firewall Manager policy.
     * </p>
     * 
     * @return The name of the Firewall Manager policy.
     */
    public final String policyName() {
        return policyName;
    }

    /**
     * <p>
     * A unique identifier for each update to the policy. When issuing a <code>PutPolicy</code> request, the
     * <code>PolicyUpdateToken</code> in the request must match the <code>PolicyUpdateToken</code> of the current policy
     * version. To get the <code>PolicyUpdateToken</code> of the current policy version, use a <code>GetPolicy</code>
     * request.
     * </p>
     * 
     * @return A unique identifier for each update to the policy. When issuing a <code>PutPolicy</code> request, the
     *         <code>PolicyUpdateToken</code> in the request must match the <code>PolicyUpdateToken</code> of the
     *         current policy version. To get the <code>PolicyUpdateToken</code> of the current policy version, use a
     *         <code>GetPolicy</code> request.
     */
    public final String policyUpdateToken() {
        return policyUpdateToken;
    }

    /**
     * <p>
     * Details about the security service that is being used to protect the resources.
     * </p>
     * 
     * @return Details about the security service that is being used to protect the resources.
     */
    public final SecurityServicePolicyData securityServicePolicyData() {
        return securityServicePolicyData;
    }

    /**
     * <p>
     * The type of resource protected by or in scope of the policy. This is in the format shown in the <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">Amazon
     * Web Services Resource Types Reference</a>. To apply this policy to multiple resource types, specify a resource
     * type of <code>ResourceTypeList</code> and then specify the resource types in a <code>ResourceTypeList</code>.
     * </p>
     * <p>
     * For WAF and Shield Advanced, example resource types include
     * <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code> and <code>AWS::CloudFront::Distribution</code>. For a
     * security group common policy, valid values are <code>AWS::EC2::NetworkInterface</code> and
     * <code>AWS::EC2::Instance</code>. For a security group content audit policy, valid values are
     * <code>AWS::EC2::SecurityGroup</code>, <code>AWS::EC2::NetworkInterface</code>, and
     * <code>AWS::EC2::Instance</code>. For a security group usage audit policy, the value is
     * <code>AWS::EC2::SecurityGroup</code>. For an Network Firewall policy or DNS Firewall policy, the value is
     * <code>AWS::EC2::VPC</code>.
     * </p>
     * 
     * @return The type of resource protected by or in scope of the policy. This is in the format shown in the <a
     *         href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
     *         >Amazon Web Services Resource Types Reference</a>. To apply this policy to multiple resource types,
     *         specify a resource type of <code>ResourceTypeList</code> and then specify the resource types in a
     *         <code>ResourceTypeList</code>.</p>
     *         <p>
     *         For WAF and Shield Advanced, example resource types include
     *         <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code> and <code>AWS::CloudFront::Distribution</code>.
     *         For a security group common policy, valid values are <code>AWS::EC2::NetworkInterface</code> and
     *         <code>AWS::EC2::Instance</code>. For a security group content audit policy, valid values are
     *         <code>AWS::EC2::SecurityGroup</code>, <code>AWS::EC2::NetworkInterface</code>, and
     *         <code>AWS::EC2::Instance</code>. For a security group usage audit policy, the value is
     *         <code>AWS::EC2::SecurityGroup</code>. For an Network Firewall policy or DNS Firewall policy, the value is
     *         <code>AWS::EC2::VPC</code>.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * Returns true if the ResourceTypeList property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasResourceTypeList() {
        return resourceTypeList != null && !(resourceTypeList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To specify a
     * single resource type, use <code>ResourceType</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResourceTypeList()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To
     *         specify a single resource type, use <code>ResourceType</code>.
     */
    public final List<String> resourceTypeList() {
        return resourceTypeList;
    }

    /**
     * Returns true if the ResourceTags property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasResourceTags() {
        return resourceTags != null && !(resourceTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ResourceTag</code> objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResourceTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of <code>ResourceTag</code> objects.
     */
    public final List<ResourceTag> resourceTags() {
        return resourceTags;
    }

    /**
     * <p>
     * If set to <code>True</code>, resources with the tags that are specified in the <code>ResourceTag</code> array are
     * not in scope of the policy. If set to <code>False</code>, and the <code>ResourceTag</code> array is not null,
     * only resources with the specified tags are in scope of the policy.
     * </p>
     * 
     * @return If set to <code>True</code>, resources with the tags that are specified in the <code>ResourceTag</code>
     *         array are not in scope of the policy. If set to <code>False</code>, and the <code>ResourceTag</code>
     *         array is not null, only resources with the specified tags are in scope of the policy.
     */
    public final Boolean excludeResourceTags() {
        return excludeResourceTags;
    }

    /**
     * <p>
     * Indicates if the policy should be automatically applied to new resources.
     * </p>
     * 
     * @return Indicates if the policy should be automatically applied to new resources.
     */
    public final Boolean remediationEnabled() {
        return remediationEnabled;
    }

    /**
     * <p>
     * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in the
     * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
     * including any child OUs and accounts that are added at a later time.
     * </p>
     * <p>
     * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
     * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate any
     * <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall Manager
     * applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
     * </p>
     * <p>
     * You can specify account IDs, OUs, or a combination:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
     * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a valid
     * map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasIncludeMap()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in
     *         the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its
     *         child OUs, including any child OUs and accounts that are added at a later time.</p>
     *         <p>
     *         You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
     *         Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
     *         not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>
     *         , then Firewall Manager applies the policy to all accounts except for those specified by the
     *         <code>ExcludeMap</code>.
     *         </p>
     *         <p>
     *         You can specify account IDs, OUs, or a combination:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
     *         map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     *         <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify accounts and OUs together in a single map, separated with a comma. For example, the following is
     *         a valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>
     *         .
     *         </p>
     *         </li>
     */
    public final Map<CustomerPolicyScopeIdType, List<String>> includeMap() {
        return CustomerPolicyScopeMapCopier.copyStringToEnum(includeMap);
    }

    /**
     * Returns true if the IncludeMap property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasIncludeMap() {
        return includeMap != null && !(includeMap instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in the
     * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
     * including any child OUs and accounts that are added at a later time.
     * </p>
     * <p>
     * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
     * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate any
     * <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall Manager
     * applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
     * </p>
     * <p>
     * You can specify account IDs, OUs, or a combination:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
     * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a valid
     * map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasIncludeMap()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in
     *         the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its
     *         child OUs, including any child OUs and accounts that are added at a later time.</p>
     *         <p>
     *         You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
     *         Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
     *         not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>
     *         , then Firewall Manager applies the policy to all accounts except for those specified by the
     *         <code>ExcludeMap</code>.
     *         </p>
     *         <p>
     *         You can specify account IDs, OUs, or a combination:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
     *         map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     *         <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify accounts and OUs together in a single map, separated with a comma. For example, the following is
     *         a valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>
     *         .
     *         </p>
     *         </li>
     */
    public final Map<String, List<String>> includeMapAsStrings() {
        return includeMap;
    }

    /**
     * <p>
     * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude from the
     * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
     * including any child OUs and accounts that are added at a later time.
     * </p>
     * <p>
     * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
     * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate any
     * <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall Manager
     * applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
     * </p>
     * <p>
     * You can specify account IDs, OUs, or a combination:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
     * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a valid
     * map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasExcludeMap()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude
     *         from the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
     *         its child OUs, including any child OUs and accounts that are added at a later time.</p>
     *         <p>
     *         You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
     *         Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
     *         not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>
     *         , then Firewall Manager applies the policy to all accounts except for those specified by the
     *         <code>ExcludeMap</code>.
     *         </p>
     *         <p>
     *         You can specify account IDs, OUs, or a combination:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
     *         map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     *         <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify accounts and OUs together in a single map, separated with a comma. For example, the following is
     *         a valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>
     *         .
     *         </p>
     *         </li>
     */
    public final Map<CustomerPolicyScopeIdType, List<String>> excludeMap() {
        return CustomerPolicyScopeMapCopier.copyStringToEnum(excludeMap);
    }

    /**
     * Returns true if the ExcludeMap property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasExcludeMap() {
        return excludeMap != null && !(excludeMap instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude from the
     * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
     * including any child OUs and accounts that are added at a later time.
     * </p>
     * <p>
     * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
     * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate any
     * <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall Manager
     * applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
     * </p>
     * <p>
     * You can specify account IDs, OUs, or a combination:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
     * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a valid
     * map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasExcludeMap()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude
     *         from the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
     *         its child OUs, including any child OUs and accounts that are added at a later time.</p>
     *         <p>
     *         You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
     *         Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
     *         not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>
     *         , then Firewall Manager applies the policy to all accounts except for those specified by the
     *         <code>ExcludeMap</code>.
     *         </p>
     *         <p>
     *         You can specify account IDs, OUs, or a combination:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
     *         map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     *         <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify accounts and OUs together in a single map, separated with a comma. For example, the following is
     *         a valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>
     *         .
     *         </p>
     *         </li>
     */
    public final Map<String, List<String>> excludeMapAsStrings() {
        return excludeMap;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(policyId());
        hashCode = 31 * hashCode + Objects.hashCode(policyName());
        hashCode = 31 * hashCode + Objects.hashCode(policyUpdateToken());
        hashCode = 31 * hashCode + Objects.hashCode(securityServicePolicyData());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypeList() ? resourceTypeList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTags() ? resourceTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(excludeResourceTags());
        hashCode = 31 * hashCode + Objects.hashCode(remediationEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasIncludeMap() ? includeMapAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludeMap() ? excludeMapAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Policy)) {
            return false;
        }
        Policy other = (Policy) obj;
        return Objects.equals(policyId(), other.policyId()) && Objects.equals(policyName(), other.policyName())
                && Objects.equals(policyUpdateToken(), other.policyUpdateToken())
                && Objects.equals(securityServicePolicyData(), other.securityServicePolicyData())
                && Objects.equals(resourceType(), other.resourceType()) && hasResourceTypeList() == other.hasResourceTypeList()
                && Objects.equals(resourceTypeList(), other.resourceTypeList()) && hasResourceTags() == other.hasResourceTags()
                && Objects.equals(resourceTags(), other.resourceTags())
                && Objects.equals(excludeResourceTags(), other.excludeResourceTags())
                && Objects.equals(remediationEnabled(), other.remediationEnabled()) && hasIncludeMap() == other.hasIncludeMap()
                && Objects.equals(includeMapAsStrings(), other.includeMapAsStrings()) && hasExcludeMap() == other.hasExcludeMap()
                && Objects.equals(excludeMapAsStrings(), other.excludeMapAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Policy").add("PolicyId", policyId()).add("PolicyName", policyName())
                .add("PolicyUpdateToken", policyUpdateToken()).add("SecurityServicePolicyData", securityServicePolicyData())
                .add("ResourceType", resourceType()).add("ResourceTypeList", hasResourceTypeList() ? resourceTypeList() : null)
                .add("ResourceTags", hasResourceTags() ? resourceTags() : null).add("ExcludeResourceTags", excludeResourceTags())
                .add("RemediationEnabled", remediationEnabled())
                .add("IncludeMap", hasIncludeMap() ? includeMapAsStrings() : null)
                .add("ExcludeMap", hasExcludeMap() ? excludeMapAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PolicyId":
            return Optional.ofNullable(clazz.cast(policyId()));
        case "PolicyName":
            return Optional.ofNullable(clazz.cast(policyName()));
        case "PolicyUpdateToken":
            return Optional.ofNullable(clazz.cast(policyUpdateToken()));
        case "SecurityServicePolicyData":
            return Optional.ofNullable(clazz.cast(securityServicePolicyData()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "ResourceTypeList":
            return Optional.ofNullable(clazz.cast(resourceTypeList()));
        case "ResourceTags":
            return Optional.ofNullable(clazz.cast(resourceTags()));
        case "ExcludeResourceTags":
            return Optional.ofNullable(clazz.cast(excludeResourceTags()));
        case "RemediationEnabled":
            return Optional.ofNullable(clazz.cast(remediationEnabled()));
        case "IncludeMap":
            return Optional.ofNullable(clazz.cast(includeMapAsStrings()));
        case "ExcludeMap":
            return Optional.ofNullable(clazz.cast(excludeMapAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Policy, T> g) {
        return obj -> g.apply((Policy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Policy> {
        /**
         * <p>
         * The ID of the Firewall Manager policy.
         * </p>
         * 
         * @param policyId
         *        The ID of the Firewall Manager policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyId(String policyId);

        /**
         * <p>
         * The name of the Firewall Manager policy.
         * </p>
         * 
         * @param policyName
         *        The name of the Firewall Manager policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyName(String policyName);

        /**
         * <p>
         * A unique identifier for each update to the policy. When issuing a <code>PutPolicy</code> request, the
         * <code>PolicyUpdateToken</code> in the request must match the <code>PolicyUpdateToken</code> of the current
         * policy version. To get the <code>PolicyUpdateToken</code> of the current policy version, use a
         * <code>GetPolicy</code> request.
         * </p>
         * 
         * @param policyUpdateToken
         *        A unique identifier for each update to the policy. When issuing a <code>PutPolicy</code> request, the
         *        <code>PolicyUpdateToken</code> in the request must match the <code>PolicyUpdateToken</code> of the
         *        current policy version. To get the <code>PolicyUpdateToken</code> of the current policy version, use a
         *        <code>GetPolicy</code> request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyUpdateToken(String policyUpdateToken);

        /**
         * <p>
         * Details about the security service that is being used to protect the resources.
         * </p>
         * 
         * @param securityServicePolicyData
         *        Details about the security service that is being used to protect the resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityServicePolicyData(SecurityServicePolicyData securityServicePolicyData);

        /**
         * <p>
         * Details about the security service that is being used to protect the resources.
         * </p>
         * This is a convenience that creates an instance of the {@link SecurityServicePolicyData.Builder} avoiding the
         * need to create one manually via {@link SecurityServicePolicyData#builder()}.
         *
         * When the {@link Consumer} completes, {@link SecurityServicePolicyData.Builder#build()} is called immediately
         * and its result is passed to {@link #securityServicePolicyData(SecurityServicePolicyData)}.
         * 
         * @param securityServicePolicyData
         *        a consumer that will call methods on {@link SecurityServicePolicyData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #securityServicePolicyData(SecurityServicePolicyData)
         */
        default Builder securityServicePolicyData(Consumer<SecurityServicePolicyData.Builder> securityServicePolicyData) {
            return securityServicePolicyData(SecurityServicePolicyData.builder().applyMutation(securityServicePolicyData).build());
        }

        /**
         * <p>
         * The type of resource protected by or in scope of the policy. This is in the format shown in the <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         * >Amazon Web Services Resource Types Reference</a>. To apply this policy to multiple resource types, specify a
         * resource type of <code>ResourceTypeList</code> and then specify the resource types in a
         * <code>ResourceTypeList</code>.
         * </p>
         * <p>
         * For WAF and Shield Advanced, example resource types include
         * <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code> and <code>AWS::CloudFront::Distribution</code>. For a
         * security group common policy, valid values are <code>AWS::EC2::NetworkInterface</code> and
         * <code>AWS::EC2::Instance</code>. For a security group content audit policy, valid values are
         * <code>AWS::EC2::SecurityGroup</code>, <code>AWS::EC2::NetworkInterface</code>, and
         * <code>AWS::EC2::Instance</code>. For a security group usage audit policy, the value is
         * <code>AWS::EC2::SecurityGroup</code>. For an Network Firewall policy or DNS Firewall policy, the value is
         * <code>AWS::EC2::VPC</code>.
         * </p>
         * 
         * @param resourceType
         *        The type of resource protected by or in scope of the policy. This is in the format shown in the <a
         *        href
         *        ="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         *        >Amazon Web Services Resource Types Reference</a>. To apply this policy to multiple resource types,
         *        specify a resource type of <code>ResourceTypeList</code> and then specify the resource types in a
         *        <code>ResourceTypeList</code>.</p>
         *        <p>
         *        For WAF and Shield Advanced, example resource types include
         *        <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code> and <code>AWS::CloudFront::Distribution</code>.
         *        For a security group common policy, valid values are <code>AWS::EC2::NetworkInterface</code> and
         *        <code>AWS::EC2::Instance</code>. For a security group content audit policy, valid values are
         *        <code>AWS::EC2::SecurityGroup</code>, <code>AWS::EC2::NetworkInterface</code>, and
         *        <code>AWS::EC2::Instance</code>. For a security group usage audit policy, the value is
         *        <code>AWS::EC2::SecurityGroup</code>. For an Network Firewall policy or DNS Firewall policy, the value
         *        is <code>AWS::EC2::VPC</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To specify a
         * single resource type, use <code>ResourceType</code>.
         * </p>
         * 
         * @param resourceTypeList
         *        An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To
         *        specify a single resource type, use <code>ResourceType</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypeList(Collection<String> resourceTypeList);

        /**
         * <p>
         * An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To specify a
         * single resource type, use <code>ResourceType</code>.
         * </p>
         * 
         * @param resourceTypeList
         *        An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To
         *        specify a single resource type, use <code>ResourceType</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypeList(String... resourceTypeList);

        /**
         * <p>
         * An array of <code>ResourceTag</code> objects.
         * </p>
         * 
         * @param resourceTags
         *        An array of <code>ResourceTag</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(Collection<ResourceTag> resourceTags);

        /**
         * <p>
         * An array of <code>ResourceTag</code> objects.
         * </p>
         * 
         * @param resourceTags
         *        An array of <code>ResourceTag</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(ResourceTag... resourceTags);

        /**
         * <p>
         * An array of <code>ResourceTag</code> objects.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ResourceTag>.Builder} avoiding the need to
         * create one manually via {@link List<ResourceTag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResourceTag>.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceTags(List<ResourceTag>)}.
         * 
         * @param resourceTags
         *        a consumer that will call methods on {@link List<ResourceTag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceTags(List<ResourceTag>)
         */
        Builder resourceTags(Consumer<ResourceTag.Builder>... resourceTags);

        /**
         * <p>
         * If set to <code>True</code>, resources with the tags that are specified in the <code>ResourceTag</code> array
         * are not in scope of the policy. If set to <code>False</code>, and the <code>ResourceTag</code> array is not
         * null, only resources with the specified tags are in scope of the policy.
         * </p>
         * 
         * @param excludeResourceTags
         *        If set to <code>True</code>, resources with the tags that are specified in the
         *        <code>ResourceTag</code> array are not in scope of the policy. If set to <code>False</code>, and the
         *        <code>ResourceTag</code> array is not null, only resources with the specified tags are in scope of the
         *        policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeResourceTags(Boolean excludeResourceTags);

        /**
         * <p>
         * Indicates if the policy should be automatically applied to new resources.
         * </p>
         * 
         * @param remediationEnabled
         *        Indicates if the policy should be automatically applied to new resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remediationEnabled(Boolean remediationEnabled);

        /**
         * <p>
         * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in the
         * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
         * including any child OUs and accounts that are added at a later time.
         * </p>
         * <p>
         * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
         * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate
         * any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall
         * Manager applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
         * </p>
         * <p>
         * You can specify account IDs, OUs, or a combination:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
         * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a
         * valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param includeMap
         *        Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include
         *        in the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
         *        its child OUs, including any child OUs and accounts that are added at a later time.</p>
         *        <p>
         *        You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
         *        Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
         *        not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an
         *        <code>IncludeMap</code>, then Firewall Manager applies the policy to all accounts except for those
         *        specified by the <code>ExcludeMap</code>.
         *        </p>
         *        <p>
         *        You can specify account IDs, OUs, or a combination:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
         *        map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         *        <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify accounts and OUs together in a single map, separated with a comma. For example, the following
         *        is a valid map:
         *        <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeMapWithStrings(Map<String, ? extends Collection<String>> includeMap);

        /**
         * <p>
         * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in the
         * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
         * including any child OUs and accounts that are added at a later time.
         * </p>
         * <p>
         * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
         * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate
         * any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall
         * Manager applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
         * </p>
         * <p>
         * You can specify account IDs, OUs, or a combination:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
         * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a
         * valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param includeMap
         *        Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include
         *        in the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
         *        its child OUs, including any child OUs and accounts that are added at a later time.</p>
         *        <p>
         *        You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
         *        Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
         *        not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an
         *        <code>IncludeMap</code>, then Firewall Manager applies the policy to all accounts except for those
         *        specified by the <code>ExcludeMap</code>.
         *        </p>
         *        <p>
         *        You can specify account IDs, OUs, or a combination:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
         *        map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         *        <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify accounts and OUs together in a single map, separated with a comma. For example, the following
         *        is a valid map:
         *        <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeMap(Map<CustomerPolicyScopeIdType, ? extends Collection<String>> includeMap);

        /**
         * <p>
         * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude from
         * the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child
         * OUs, including any child OUs and accounts that are added at a later time.
         * </p>
         * <p>
         * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
         * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate
         * any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall
         * Manager applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
         * </p>
         * <p>
         * You can specify account IDs, OUs, or a combination:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
         * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a
         * valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param excludeMap
         *        Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude
         *        from the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
         *        its child OUs, including any child OUs and accounts that are added at a later time.</p>
         *        <p>
         *        You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
         *        Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
         *        not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an
         *        <code>IncludeMap</code>, then Firewall Manager applies the policy to all accounts except for those
         *        specified by the <code>ExcludeMap</code>.
         *        </p>
         *        <p>
         *        You can specify account IDs, OUs, or a combination:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
         *        map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         *        <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify accounts and OUs together in a single map, separated with a comma. For example, the following
         *        is a valid map:
         *        <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeMapWithStrings(Map<String, ? extends Collection<String>> excludeMap);

        /**
         * <p>
         * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude from
         * the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child
         * OUs, including any child OUs and accounts that are added at a later time.
         * </p>
         * <p>
         * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
         * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate
         * any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall
         * Manager applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
         * </p>
         * <p>
         * You can specify account IDs, OUs, or a combination:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
         * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a
         * valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param excludeMap
         *        Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude
         *        from the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
         *        its child OUs, including any child OUs and accounts that are added at a later time.</p>
         *        <p>
         *        You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
         *        Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
         *        not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an
         *        <code>IncludeMap</code>, then Firewall Manager applies the policy to all accounts except for those
         *        specified by the <code>ExcludeMap</code>.
         *        </p>
         *        <p>
         *        You can specify account IDs, OUs, or a combination:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
         *        map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         *        <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify accounts and OUs together in a single map, separated with a comma. For example, the following
         *        is a valid map:
         *        <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeMap(Map<CustomerPolicyScopeIdType, ? extends Collection<String>> excludeMap);
    }

    static final class BuilderImpl implements Builder {
        private String policyId;

        private String policyName;

        private String policyUpdateToken;

        private SecurityServicePolicyData securityServicePolicyData;

        private String resourceType;

        private List<String> resourceTypeList = DefaultSdkAutoConstructList.getInstance();

        private List<ResourceTag> resourceTags = DefaultSdkAutoConstructList.getInstance();

        private Boolean excludeResourceTags;

        private Boolean remediationEnabled;

        private Map<String, List<String>> includeMap = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, List<String>> excludeMap = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Policy model) {
            policyId(model.policyId);
            policyName(model.policyName);
            policyUpdateToken(model.policyUpdateToken);
            securityServicePolicyData(model.securityServicePolicyData);
            resourceType(model.resourceType);
            resourceTypeList(model.resourceTypeList);
            resourceTags(model.resourceTags);
            excludeResourceTags(model.excludeResourceTags);
            remediationEnabled(model.remediationEnabled);
            includeMapWithStrings(model.includeMap);
            excludeMapWithStrings(model.excludeMap);
        }

        public final String getPolicyId() {
            return policyId;
        }

        @Override
        public final Builder policyId(String policyId) {
            this.policyId = policyId;
            return this;
        }

        public final void setPolicyId(String policyId) {
            this.policyId = policyId;
        }

        public final String getPolicyName() {
            return policyName;
        }

        @Override
        public final Builder policyName(String policyName) {
            this.policyName = policyName;
            return this;
        }

        public final void setPolicyName(String policyName) {
            this.policyName = policyName;
        }

        public final String getPolicyUpdateToken() {
            return policyUpdateToken;
        }

        @Override
        public final Builder policyUpdateToken(String policyUpdateToken) {
            this.policyUpdateToken = policyUpdateToken;
            return this;
        }

        public final void setPolicyUpdateToken(String policyUpdateToken) {
            this.policyUpdateToken = policyUpdateToken;
        }

        public final SecurityServicePolicyData.Builder getSecurityServicePolicyData() {
            return securityServicePolicyData != null ? securityServicePolicyData.toBuilder() : null;
        }

        @Override
        public final Builder securityServicePolicyData(SecurityServicePolicyData securityServicePolicyData) {
            this.securityServicePolicyData = securityServicePolicyData;
            return this;
        }

        public final void setSecurityServicePolicyData(SecurityServicePolicyData.BuilderImpl securityServicePolicyData) {
            this.securityServicePolicyData = securityServicePolicyData != null ? securityServicePolicyData.build() : null;
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final Collection<String> getResourceTypeList() {
            if (resourceTypeList instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceTypeList;
        }

        @Override
        public final Builder resourceTypeList(Collection<String> resourceTypeList) {
            this.resourceTypeList = ResourceTypeListCopier.copy(resourceTypeList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypeList(String... resourceTypeList) {
            resourceTypeList(Arrays.asList(resourceTypeList));
            return this;
        }

        public final void setResourceTypeList(Collection<String> resourceTypeList) {
            this.resourceTypeList = ResourceTypeListCopier.copy(resourceTypeList);
        }

        public final List<ResourceTag.Builder> getResourceTags() {
            List<ResourceTag.Builder> result = ResourceTagsCopier.copyToBuilder(this.resourceTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder resourceTags(Collection<ResourceTag> resourceTags) {
            this.resourceTags = ResourceTagsCopier.copy(resourceTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(ResourceTag... resourceTags) {
            resourceTags(Arrays.asList(resourceTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(Consumer<ResourceTag.Builder>... resourceTags) {
            resourceTags(Stream.of(resourceTags).map(c -> ResourceTag.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setResourceTags(Collection<ResourceTag.BuilderImpl> resourceTags) {
            this.resourceTags = ResourceTagsCopier.copyFromBuilder(resourceTags);
        }

        public final Boolean getExcludeResourceTags() {
            return excludeResourceTags;
        }

        @Override
        public final Builder excludeResourceTags(Boolean excludeResourceTags) {
            this.excludeResourceTags = excludeResourceTags;
            return this;
        }

        public final void setExcludeResourceTags(Boolean excludeResourceTags) {
            this.excludeResourceTags = excludeResourceTags;
        }

        public final Boolean getRemediationEnabled() {
            return remediationEnabled;
        }

        @Override
        public final Builder remediationEnabled(Boolean remediationEnabled) {
            this.remediationEnabled = remediationEnabled;
            return this;
        }

        public final void setRemediationEnabled(Boolean remediationEnabled) {
            this.remediationEnabled = remediationEnabled;
        }

        public final Map<String, ? extends Collection<String>> getIncludeMap() {
            if (includeMap instanceof SdkAutoConstructMap) {
                return null;
            }
            return includeMap;
        }

        @Override
        public final Builder includeMapWithStrings(Map<String, ? extends Collection<String>> includeMap) {
            this.includeMap = CustomerPolicyScopeMapCopier.copy(includeMap);
            return this;
        }

        @Override
        public final Builder includeMap(Map<CustomerPolicyScopeIdType, ? extends Collection<String>> includeMap) {
            this.includeMap = CustomerPolicyScopeMapCopier.copyEnumToString(includeMap);
            return this;
        }

        public final void setIncludeMap(Map<String, ? extends Collection<String>> includeMap) {
            this.includeMap = CustomerPolicyScopeMapCopier.copy(includeMap);
        }

        public final Map<String, ? extends Collection<String>> getExcludeMap() {
            if (excludeMap instanceof SdkAutoConstructMap) {
                return null;
            }
            return excludeMap;
        }

        @Override
        public final Builder excludeMapWithStrings(Map<String, ? extends Collection<String>> excludeMap) {
            this.excludeMap = CustomerPolicyScopeMapCopier.copy(excludeMap);
            return this;
        }

        @Override
        public final Builder excludeMap(Map<CustomerPolicyScopeIdType, ? extends Collection<String>> excludeMap) {
            this.excludeMap = CustomerPolicyScopeMapCopier.copyEnumToString(excludeMap);
            return this;
        }

        public final void setExcludeMap(Map<String, ? extends Collection<String>> excludeMap) {
            this.excludeMap = CustomerPolicyScopeMapCopier.copy(excludeMap);
        }

        @Override
        public Policy build() {
            return new Policy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
