/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fms.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about the security service that is being used to protect the resources.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SecurityServicePolicyData implements SdkPojo, Serializable,
        ToCopyableBuilder<SecurityServicePolicyData.Builder, SecurityServicePolicyData> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(SecurityServicePolicyData::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> MANAGED_SERVICE_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ManagedServiceData").getter(getter(SecurityServicePolicyData::managedServiceData))
            .setter(setter(Builder::managedServiceData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManagedServiceData").build())
            .build();

    private static final SdkField<PolicyOption> POLICY_OPTION_FIELD = SdkField.<PolicyOption> builder(MarshallingType.SDK_POJO)
            .memberName("PolicyOption").getter(getter(SecurityServicePolicyData::policyOption))
            .setter(setter(Builder::policyOption)).constructor(PolicyOption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyOption").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD,
            MANAGED_SERVICE_DATA_FIELD, POLICY_OPTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String managedServiceData;

    private final PolicyOption policyOption;

    private SecurityServicePolicyData(BuilderImpl builder) {
        this.type = builder.type;
        this.managedServiceData = builder.managedServiceData;
        this.policyOption = builder.policyOption;
    }

    /**
     * <p>
     * The service that the policy is using to protect the resources. This specifies the type of policy that is created,
     * either an WAF policy, a Shield Advanced policy, or a security group policy. For security group policies, Firewall
     * Manager supports one security group for each common policy and for each content audit policy. This is an
     * adjustable limit that you can increase by contacting Amazon Web Services Support.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SecurityServiceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The service that the policy is using to protect the resources. This specifies the type of policy that is
     *         created, either an WAF policy, a Shield Advanced policy, or a security group policy. For security group
     *         policies, Firewall Manager supports one security group for each common policy and for each content audit
     *         policy. This is an adjustable limit that you can increase by contacting Amazon Web Services Support.
     * @see SecurityServiceType
     */
    public final SecurityServiceType type() {
        return SecurityServiceType.fromValue(type);
    }

    /**
     * <p>
     * The service that the policy is using to protect the resources. This specifies the type of policy that is created,
     * either an WAF policy, a Shield Advanced policy, or a security group policy. For security group policies, Firewall
     * Manager supports one security group for each common policy and for each content audit policy. This is an
     * adjustable limit that you can increase by contacting Amazon Web Services Support.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SecurityServiceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The service that the policy is using to protect the resources. This specifies the type of policy that is
     *         created, either an WAF policy, a Shield Advanced policy, or a security group policy. For security group
     *         policies, Firewall Manager supports one security group for each common policy and for each content audit
     *         policy. This is an adjustable limit that you can increase by contacting Amazon Web Services Support.
     * @see SecurityServiceType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Details about the service that are specific to the service type, in JSON format.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Example: <code>DNS_FIREWALL</code>
     * </p>
     * <p>
     * <code>"{\"type\":\"DNS_FIREWALL\",\"preProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-1\",\"priority\":10}],\"postProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-2\",\"priority\":9911}]}"</code>
     * </p>
     * <note>
     * <p>
     * Valid values for <code>preProcessRuleGroups</code> are between 1 and 99. Valid values for
     * <code>postProcessRuleGroups</code> are between 9901 and 10000.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * Example: <code>DNS_FIREWALL</code>
     * </p>
     * <p>
     * <code>"{\"type\":\"DNS_FIREWALL\",\"preProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-1\",\"priority\":10}],\"postProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-2\",\"priority\":9911}]}"</code>
     * </p>
     * <note>
     * <p>
     * Valid values for <code>preProcessRuleGroups</code> are between 1 and 99. Valid values for
     * <code>postProcessRuleGroups</code> are between 9901 and 10000.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with automatic Availability Zone
     * configuration. With automatic Availbility Zone configuration, Firewall Manager chooses which Availability Zones
     * to create the endpoints in.
     * </p>
     * <p>
     * <code>"{ \"type\": \"NETWORK_FIREWALL\", \"networkFirewallStatelessRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\", \"priority\": 1 } ], \"networkFirewallStatelessDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessCustomActions\": [ { \"actionName\": \"customActionName\", \"actionDefinition\": { \"publishMetricAction\": { \"dimensions\": [ { \"value\": \"metricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\": { \"singleFirewallEndpointPerVPC\": false, \"allowedIPV4CidrList\": [ \"10.0.0.0/28\", \"192.168.0.0/28\" ], \"routeManagementAction\": \"OFF\" }, \"networkFirewallLoggingConfiguration\": { \"logDestinationConfigs\": [ { \"logDestinationType\": \"S3\", \"logType\": \"ALERT\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } }, { \"logDestinationType\": \"S3\", \"logType\": \"FLOW\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } } ], \"overrideExistingConfig\": true } }"</code>
     * </p>
     * <p>
     * To use the distributed deployment model, you must set <a
     * href="https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PolicyOption.html">PolicyOption</a> to
     * <code>NULL</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with automatic Availability Zone
     * configuration, and route management.
     * </p>
     * <p>
     * <code>"{ \"type\": \"NETWORK_FIREWALL\", \"networkFirewallStatelessRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\", \"priority\": 1 } ], \"networkFirewallStatelessDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessCustomActions\": [ { \"actionName\": \"customActionName\", \"actionDefinition\": { \"publishMetricAction\": { \"dimensions\": [ { \"value\": \"metricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\": { \"singleFirewallEndpointPerVPC\": false, \"allowedIPV4CidrList\": [ \"10.0.0.0/28\", \"192.168.0.0/28\" ], \"routeManagementAction\": \"MONITOR\", \"routeManagementTargetTypes\": [ \"InternetGateway\" ] }, \"networkFirewallLoggingConfiguration\": { \"logDestinationConfigs\": [ { \"logDestinationType\": \"S3\", \"logType\": \"ALERT\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } }, { \"logDestinationType\": \"S3\", \"logType\": \"FLOW\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } } ], \"overrideExistingConfig\": true } }"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with custom Availability Zone
     * configuration. With custom Availability Zone configuration, you define which specific Availability Zones to
     * create endpoints in by configuring <code>firewallCreationConfig</code>.
     * </p>
     * <p>
     * <code>"{ \"type\":\"NETWORK_FIREWALL\",\"networkFirewallStatelessRuleGroupReferences\":[{\"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\",\"priority\":1}], \"networkFirewallStatelessDefaultActions\":[ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\":[ \"aws:forward_to_sfe\", \"fragmentcustomactionname\" ], \"networkFirewallStatelessCustomActions\":[ { \"actionName\":\"customActionName\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"metricdimensionvalue\" } ] } } }, { \"actionName\":\"fragmentcustomactionname\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"fragmentmetricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\":[ { \"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\":{ \"firewallCreationConfig\":{ \"endpointLocation\":{ \"availabilityZoneConfigList\":[ { \"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1a\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] }, { ¯\"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1b\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] } ] } }, \"singleFirewallEndpointPerVPC\":false, \"allowedIPV4CidrList\":null, \"routeManagementAction\":\"OFF\", \"networkFirewallLoggingConfiguration\":{ \"logDestinationConfigs\":[ { \"logDestinationType\":\"S3\", \"logType\":\"ALERT\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } }, { \"logDestinationType\":\"S3\", \"logType\":\"FLOW\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } } ], \"overrideExistingConfig\":boolean } }"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with custom Availability Zone
     * configuration, and route management.
     * </p>
     * <p>
     * <code>"{ \"type\":\"NETWORK_FIREWALL\",\"networkFirewallStatelessRuleGroupReferences\":[{\"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\",\"priority\":1}], \"networkFirewallStatelessDefaultActions\":[ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\":[ \"aws:forward_to_sfe\", \"fragmentcustomactionname\" ], \"networkFirewallStatelessCustomActions\":[ { \"actionName\":\"customActionName\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"metricdimensionvalue\" } ] } } }, { \"actionName\":\"fragmentcustomactionname\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"fragmentmetricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\":[ { \"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\":{ \"firewallCreationConfig\":{ \"endpointLocation\":{ \"availabilityZoneConfigList\":[ { \"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1a\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] }, { ¯\"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1b\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] } ] } }, \"singleFirewallEndpointPerVPC\":false, \"allowedIPV4CidrList\":null, \"routeManagementAction\":\"MONITOR\", \"routeManagementTargetTypes\":[ \"InternetGateway\" ], \"routeManagementConfig\":{ \"allowCrossAZTrafficIfNoEndpoint\":true } }, \"networkFirewallLoggingConfiguration\":{ \"logDestinationConfigs\":[ { \"logDestinationType\":\"S3\", \"logType\":\"ALERT\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } }, { \"logDestinationType\":\"S3\", \"logType\":\"FLOW\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } } ], \"overrideExistingConfig\":boolean } }"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Example: <code>PARTNER_FIREWALL</code> for Firewall Manager
     * </p>
     * <p>
     * <code>"{\"type\":\"THIRD_PARTY_FIREWALL\",\"thirdPartyrFirewall\":\"PALO_ALTO_NETWORKS_CLOUD_NGFW\",\"thirdPartyFirewallConfig\":{\"thirdPartyFirewallPolicyList\":[\"global-123456789012-1\"],\"networkFirewallLoggingConfiguration\":null},\"firewallDeploymentModel\":{\"distributedFirewallDeploymentModel\":{\"distributedFirewallOrchestrationConfig\":{\"firewallCreationConfig\":{\"endpointLocation\":{\"availabilityZoneConfigList\":[{\"availabilityZoneId\":null,\"availabilityZoneName\":\"us-east-1a\",\"allowedIPV4CidrList\":[\"10.0.1.0/28\"]}]}},\"allowedIPV4CidrList\":null},\"distributedRouteManagementConfig\":null},\"centralizedFirewallDeploymentModel\":null}}""</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Specification for <code>SHIELD_ADVANCED</code> for Amazon CloudFront distributions
     * </p>
     * <p>
     * <code>"{\"type\":\"SHIELD_ADVANCED\",\"automaticResponseConfiguration\": {\"automaticResponseStatus\":\"ENABLED|IGNORED|DISABLED\", \"automaticResponseAction\":\"BLOCK|COUNT\"}, \"overrideCustomerWebaclClassic\":true|false}"</code>
     * </p>
     * <p>
     * For example:
     * <code>"{\"type\":\"SHIELD_ADVANCED\",\"automaticResponseConfiguration\": {\"automaticResponseStatus\":\"ENABLED\", \"automaticResponseAction\":\"COUNT\"}}"</code>
     * </p>
     * <p>
     * The default value for <code>automaticResponseStatus</code> is <code>IGNORED</code>. The value for
     * <code>automaticResponseAction</code> is only required when <code>automaticResponseStatus</code> is set to
     * <code>ENABLED</code>. The default value for <code>overrideCustomerWebaclClassic</code> is <code>false</code>.
     * </p>
     * <p>
     * For other resource types that you can protect with a Shield Advanced policy, this <code>ManagedServiceData</code>
     * configuration is an empty string.
     * </p>
     * </li>
     * <li>
     * <p>
     * Example: <code>WAFV2</code>
     * </p>
     * <p>
     * <code>"{\"type\":\"WAFV2\",\"preProcessRuleGroups\":[{\"ruleGroupArn\":null,\"overrideAction\":{\"type\":\"NONE\"},\"managedRuleGroupIdentifier\":{\"version\":null,\"vendorName\":\"AWS\",\"managedRuleGroupName\":\"AWSManagedRulesAmazonIpReputationList\"},\"ruleGroupType\":\"ManagedRuleGroup\",\"excludeRules\":[{\"name\":\"NoUserAgent_HEADER\"}]}],\"postProcessRuleGroups\":[],\"defaultAction\":{\"type\":\"ALLOW\"},\"overrideCustomerWebACLAssociation\":false,\"loggingConfiguration\":{\"logDestinationConfigs\":[\"arn:aws:firehose:us-west-2:12345678912:deliverystream/aws-waf-logs-fms-admin-destination\"],\"redactedFields\":[{\"redactedFieldType\":\"SingleHeader\",\"redactedFieldValue\":\"Cookies\"},{\"redactedFieldType\":\"Method\"}]}}"</code>
     * </p>
     * <p>
     * In the <code>loggingConfiguration</code>, you can specify one <code>logDestinationConfigs</code>, you can
     * optionally provide up to 20 <code>redactedFields</code>, and the <code>RedactedFieldType</code> must be one of
     * <code>URI</code>, <code>QUERY_STRING</code>, <code>HEADER</code>, or <code>METHOD</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Example: <code>WAF Classic</code>
     * </p>
     * <p>
     * <code>"{\"type\": \"WAF\", \"ruleGroups\": [{\"id\":\"12345678-1bcd-9012-efga-0987654321ab\", \"overrideAction\" : {\"type\": \"COUNT\"}}], \"defaultAction\": {\"type\": \"BLOCK\"}}"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Example: <code>WAFV2</code> - Firewall Manager support for WAF managed rule group versioning
     * </p>
     * <p>
     * <code>"{\"type\":\"WAFV2\",\"preProcessRuleGroups\":[{\"ruleGroupArn\":null,\"overrideAction\":{\"type\":\"NONE\"},\"managedRuleGroupIdentifier\":{\"versionEnabled\":true,\"version\":\"Version_2.0\",\"vendorName\":\"AWS\",\"managedRuleGroupName\":\"AWSManagedRulesCommonRuleSet\"},\"ruleGroupType\":\"ManagedRuleGroup\",\"excludeRules\":[{\"name\":\"NoUserAgent_HEADER\"}]}],\"postProcessRuleGroups\":[],\"defaultAction\":{\"type\":\"ALLOW\"},\"overrideCustomerWebACLAssociation\":false,\"loggingConfiguration\":{\"logDestinationConfigs\":[\"arn:aws:firehose:us-west-2:12345678912:deliverystream/aws-waf-logs-fms-admin-destination\"],\"redactedFields\":[{\"redactedFieldType\":\"SingleHeader\",\"redactedFieldValue\":\"Cookies\"},{\"redactedFieldType\":\"Method\"}]}}"</code>
     * </p>
     * <p>
     * To use a specific version of a WAF managed rule group in your Firewall Manager policy, you must set
     * <code>versionEnabled</code> to <code>true</code>, and set <code>version</code> to the version you'd like to use.
     * If you don't set <code>versionEnabled</code> to <code>true</code>, or if you omit <code>versionEnabled</code>,
     * then Firewall Manager uses the default version of the WAF managed rule group.
     * </p>
     * </li>
     * <li>
     * <p>
     * Example: <code>SECURITY_GROUPS_COMMON</code>
     * </p>
     * <p>
     * <code>"{\"type\":\"SECURITY_GROUPS_COMMON\",\"revertManualSecurityGroupChanges\":false,\"exclusiveResourceSecurityGroupManagement\":false, \"applyToAllEC2InstanceENIs\":false,\"securityGroups\":[{\"id\":\" sg-000e55995d61a06bd\"}]}"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Example: Shared VPCs. Apply the preceding policy to resources in shared VPCs as well as to those in VPCs that the
     * account owns
     * </p>
     * <p>
     * <code>"{\"type\":\"SECURITY_GROUPS_COMMON\",\"revertManualSecurityGroupChanges\":false,\"exclusiveResourceSecurityGroupManagement\":false, \"applyToAllEC2InstanceENIs\":false,\"includeSharedVPC\":true,\"securityGroups\":[{\"id\":\" sg-000e55995d61a06bd\"}]}"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Example: <code>SECURITY_GROUPS_CONTENT_AUDIT</code>
     * </p>
     * <p>
     * <code>"{\"type\":\"SECURITY_GROUPS_CONTENT_AUDIT\",\"securityGroups\":[{\"id\":\"sg-000e55995d61a06bd\"}],\"securityGroupAction\":{\"type\":\"ALLOW\"}}"</code>
     * </p>
     * <p>
     * The security group action for content audit can be <code>ALLOW</code> or <code>DENY</code>. For
     * <code>ALLOW</code>, all in-scope security group rules must be within the allowed range of the policy's security
     * group rules. For <code>DENY</code>, all in-scope security group rules must not contain a value or a range that
     * matches a rule value or range in the policy security group.
     * </p>
     * </li>
     * <li>
     * <p>
     * Example: <code>SECURITY_GROUPS_USAGE_AUDIT</code>
     * </p>
     * <p>
     * <code>"{\"type\":\"SECURITY_GROUPS_USAGE_AUDIT\",\"deleteUnusedSecurityGroups\":true,\"coalesceRedundantSecurityGroups\":true}"</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return Details about the service that are specific to the service type, in JSON format. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Example: <code>DNS_FIREWALL</code>
     *         </p>
     *         <p>
     *         <code>"{\"type\":\"DNS_FIREWALL\",\"preProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-1\",\"priority\":10}],\"postProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-2\",\"priority\":9911}]}"</code>
     *         </p>
     *         <note>
     *         <p>
     *         Valid values for <code>preProcessRuleGroups</code> are between 1 and 99. Valid values for
     *         <code>postProcessRuleGroups</code> are between 9901 and 10000.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         Example: <code>DNS_FIREWALL</code>
     *         </p>
     *         <p>
     *         <code>"{\"type\":\"DNS_FIREWALL\",\"preProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-1\",\"priority\":10}],\"postProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-2\",\"priority\":9911}]}"</code>
     *         </p>
     *         <note>
     *         <p>
     *         Valid values for <code>preProcessRuleGroups</code> are between 1 and 99. Valid values for
     *         <code>postProcessRuleGroups</code> are between 9901 and 10000.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with automatic Availability Zone
     *         configuration. With automatic Availbility Zone configuration, Firewall Manager chooses which Availability
     *         Zones to create the endpoints in.
     *         </p>
     *         <p>
     *         <code>"{ \"type\": \"NETWORK_FIREWALL\", \"networkFirewallStatelessRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\", \"priority\": 1 } ], \"networkFirewallStatelessDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessCustomActions\": [ { \"actionName\": \"customActionName\", \"actionDefinition\": { \"publishMetricAction\": { \"dimensions\": [ { \"value\": \"metricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\": { \"singleFirewallEndpointPerVPC\": false, \"allowedIPV4CidrList\": [ \"10.0.0.0/28\", \"192.168.0.0/28\" ], \"routeManagementAction\": \"OFF\" }, \"networkFirewallLoggingConfiguration\": { \"logDestinationConfigs\": [ { \"logDestinationType\": \"S3\", \"logType\": \"ALERT\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } }, { \"logDestinationType\": \"S3\", \"logType\": \"FLOW\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } } ], \"overrideExistingConfig\": true } }"</code>
     *         </p>
     *         <p>
     *         To use the distributed deployment model, you must set <a
     *         href="https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PolicyOption.html">PolicyOption</a> to
     *         <code>NULL</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with automatic Availability Zone
     *         configuration, and route management.
     *         </p>
     *         <p>
     *         <code>"{ \"type\": \"NETWORK_FIREWALL\", \"networkFirewallStatelessRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\", \"priority\": 1 } ], \"networkFirewallStatelessDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessCustomActions\": [ { \"actionName\": \"customActionName\", \"actionDefinition\": { \"publishMetricAction\": { \"dimensions\": [ { \"value\": \"metricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\": { \"singleFirewallEndpointPerVPC\": false, \"allowedIPV4CidrList\": [ \"10.0.0.0/28\", \"192.168.0.0/28\" ], \"routeManagementAction\": \"MONITOR\", \"routeManagementTargetTypes\": [ \"InternetGateway\" ] }, \"networkFirewallLoggingConfiguration\": { \"logDestinationConfigs\": [ { \"logDestinationType\": \"S3\", \"logType\": \"ALERT\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } }, { \"logDestinationType\": \"S3\", \"logType\": \"FLOW\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } } ], \"overrideExistingConfig\": true } }"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with custom Availability Zone
     *         configuration. With custom Availability Zone configuration, you define which specific Availability Zones
     *         to create endpoints in by configuring <code>firewallCreationConfig</code>.
     *         </p>
     *         <p>
     *         <code>"{ \"type\":\"NETWORK_FIREWALL\",\"networkFirewallStatelessRuleGroupReferences\":[{\"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\",\"priority\":1}], \"networkFirewallStatelessDefaultActions\":[ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\":[ \"aws:forward_to_sfe\", \"fragmentcustomactionname\" ], \"networkFirewallStatelessCustomActions\":[ { \"actionName\":\"customActionName\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"metricdimensionvalue\" } ] } } }, { \"actionName\":\"fragmentcustomactionname\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"fragmentmetricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\":[ { \"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\":{ \"firewallCreationConfig\":{ \"endpointLocation\":{ \"availabilityZoneConfigList\":[ { \"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1a\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] }, { ¯\"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1b\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] } ] } }, \"singleFirewallEndpointPerVPC\":false, \"allowedIPV4CidrList\":null, \"routeManagementAction\":\"OFF\", \"networkFirewallLoggingConfiguration\":{ \"logDestinationConfigs\":[ { \"logDestinationType\":\"S3\", \"logType\":\"ALERT\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } }, { \"logDestinationType\":\"S3\", \"logType\":\"FLOW\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } } ], \"overrideExistingConfig\":boolean } }"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with custom Availability Zone
     *         configuration, and route management.
     *         </p>
     *         <p>
     *         <code>"{ \"type\":\"NETWORK_FIREWALL\",\"networkFirewallStatelessRuleGroupReferences\":[{\"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\",\"priority\":1}], \"networkFirewallStatelessDefaultActions\":[ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\":[ \"aws:forward_to_sfe\", \"fragmentcustomactionname\" ], \"networkFirewallStatelessCustomActions\":[ { \"actionName\":\"customActionName\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"metricdimensionvalue\" } ] } } }, { \"actionName\":\"fragmentcustomactionname\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"fragmentmetricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\":[ { \"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\":{ \"firewallCreationConfig\":{ \"endpointLocation\":{ \"availabilityZoneConfigList\":[ { \"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1a\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] }, { ¯\"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1b\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] } ] } }, \"singleFirewallEndpointPerVPC\":false, \"allowedIPV4CidrList\":null, \"routeManagementAction\":\"MONITOR\", \"routeManagementTargetTypes\":[ \"InternetGateway\" ], \"routeManagementConfig\":{ \"allowCrossAZTrafficIfNoEndpoint\":true } }, \"networkFirewallLoggingConfiguration\":{ \"logDestinationConfigs\":[ { \"logDestinationType\":\"S3\", \"logType\":\"ALERT\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } }, { \"logDestinationType\":\"S3\", \"logType\":\"FLOW\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } } ], \"overrideExistingConfig\":boolean } }"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example: <code>PARTNER_FIREWALL</code> for Firewall Manager
     *         </p>
     *         <p>
     *         <code>"{\"type\":\"THIRD_PARTY_FIREWALL\",\"thirdPartyrFirewall\":\"PALO_ALTO_NETWORKS_CLOUD_NGFW\",\"thirdPartyFirewallConfig\":{\"thirdPartyFirewallPolicyList\":[\"global-123456789012-1\"],\"networkFirewallLoggingConfiguration\":null},\"firewallDeploymentModel\":{\"distributedFirewallDeploymentModel\":{\"distributedFirewallOrchestrationConfig\":{\"firewallCreationConfig\":{\"endpointLocation\":{\"availabilityZoneConfigList\":[{\"availabilityZoneId\":null,\"availabilityZoneName\":\"us-east-1a\",\"allowedIPV4CidrList\":[\"10.0.1.0/28\"]}]}},\"allowedIPV4CidrList\":null},\"distributedRouteManagementConfig\":null},\"centralizedFirewallDeploymentModel\":null}}""</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specification for <code>SHIELD_ADVANCED</code> for Amazon CloudFront distributions
     *         </p>
     *         <p>
     *         <code>"{\"type\":\"SHIELD_ADVANCED\",\"automaticResponseConfiguration\": {\"automaticResponseStatus\":\"ENABLED|IGNORED|DISABLED\", \"automaticResponseAction\":\"BLOCK|COUNT\"}, \"overrideCustomerWebaclClassic\":true|false}"</code>
     *         </p>
     *         <p>
     *         For example:
     *         <code>"{\"type\":\"SHIELD_ADVANCED\",\"automaticResponseConfiguration\": {\"automaticResponseStatus\":\"ENABLED\", \"automaticResponseAction\":\"COUNT\"}}"</code>
     *         </p>
     *         <p>
     *         The default value for <code>automaticResponseStatus</code> is <code>IGNORED</code>. The value for
     *         <code>automaticResponseAction</code> is only required when <code>automaticResponseStatus</code> is set to
     *         <code>ENABLED</code>. The default value for <code>overrideCustomerWebaclClassic</code> is
     *         <code>false</code>.
     *         </p>
     *         <p>
     *         For other resource types that you can protect with a Shield Advanced policy, this
     *         <code>ManagedServiceData</code> configuration is an empty string.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example: <code>WAFV2</code>
     *         </p>
     *         <p>
     *         <code>"{\"type\":\"WAFV2\",\"preProcessRuleGroups\":[{\"ruleGroupArn\":null,\"overrideAction\":{\"type\":\"NONE\"},\"managedRuleGroupIdentifier\":{\"version\":null,\"vendorName\":\"AWS\",\"managedRuleGroupName\":\"AWSManagedRulesAmazonIpReputationList\"},\"ruleGroupType\":\"ManagedRuleGroup\",\"excludeRules\":[{\"name\":\"NoUserAgent_HEADER\"}]}],\"postProcessRuleGroups\":[],\"defaultAction\":{\"type\":\"ALLOW\"},\"overrideCustomerWebACLAssociation\":false,\"loggingConfiguration\":{\"logDestinationConfigs\":[\"arn:aws:firehose:us-west-2:12345678912:deliverystream/aws-waf-logs-fms-admin-destination\"],\"redactedFields\":[{\"redactedFieldType\":\"SingleHeader\",\"redactedFieldValue\":\"Cookies\"},{\"redactedFieldType\":\"Method\"}]}}"</code>
     *         </p>
     *         <p>
     *         In the <code>loggingConfiguration</code>, you can specify one <code>logDestinationConfigs</code>, you can
     *         optionally provide up to 20 <code>redactedFields</code>, and the <code>RedactedFieldType</code> must be
     *         one of <code>URI</code>, <code>QUERY_STRING</code>, <code>HEADER</code>, or <code>METHOD</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example: <code>WAF Classic</code>
     *         </p>
     *         <p>
     *         <code>"{\"type\": \"WAF\", \"ruleGroups\": [{\"id\":\"12345678-1bcd-9012-efga-0987654321ab\", \"overrideAction\" : {\"type\": \"COUNT\"}}], \"defaultAction\": {\"type\": \"BLOCK\"}}"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example: <code>WAFV2</code> - Firewall Manager support for WAF managed rule group versioning
     *         </p>
     *         <p>
     *         <code>"{\"type\":\"WAFV2\",\"preProcessRuleGroups\":[{\"ruleGroupArn\":null,\"overrideAction\":{\"type\":\"NONE\"},\"managedRuleGroupIdentifier\":{\"versionEnabled\":true,\"version\":\"Version_2.0\",\"vendorName\":\"AWS\",\"managedRuleGroupName\":\"AWSManagedRulesCommonRuleSet\"},\"ruleGroupType\":\"ManagedRuleGroup\",\"excludeRules\":[{\"name\":\"NoUserAgent_HEADER\"}]}],\"postProcessRuleGroups\":[],\"defaultAction\":{\"type\":\"ALLOW\"},\"overrideCustomerWebACLAssociation\":false,\"loggingConfiguration\":{\"logDestinationConfigs\":[\"arn:aws:firehose:us-west-2:12345678912:deliverystream/aws-waf-logs-fms-admin-destination\"],\"redactedFields\":[{\"redactedFieldType\":\"SingleHeader\",\"redactedFieldValue\":\"Cookies\"},{\"redactedFieldType\":\"Method\"}]}}"</code>
     *         </p>
     *         <p>
     *         To use a specific version of a WAF managed rule group in your Firewall Manager policy, you must set
     *         <code>versionEnabled</code> to <code>true</code>, and set <code>version</code> to the version you'd like
     *         to use. If you don't set <code>versionEnabled</code> to <code>true</code>, or if you omit
     *         <code>versionEnabled</code>, then Firewall Manager uses the default version of the WAF managed rule
     *         group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example: <code>SECURITY_GROUPS_COMMON</code>
     *         </p>
     *         <p>
     *         <code>"{\"type\":\"SECURITY_GROUPS_COMMON\",\"revertManualSecurityGroupChanges\":false,\"exclusiveResourceSecurityGroupManagement\":false, \"applyToAllEC2InstanceENIs\":false,\"securityGroups\":[{\"id\":\" sg-000e55995d61a06bd\"}]}"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example: Shared VPCs. Apply the preceding policy to resources in shared VPCs as well as to those in VPCs
     *         that the account owns
     *         </p>
     *         <p>
     *         <code>"{\"type\":\"SECURITY_GROUPS_COMMON\",\"revertManualSecurityGroupChanges\":false,\"exclusiveResourceSecurityGroupManagement\":false, \"applyToAllEC2InstanceENIs\":false,\"includeSharedVPC\":true,\"securityGroups\":[{\"id\":\" sg-000e55995d61a06bd\"}]}"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example: <code>SECURITY_GROUPS_CONTENT_AUDIT</code>
     *         </p>
     *         <p>
     *         <code>"{\"type\":\"SECURITY_GROUPS_CONTENT_AUDIT\",\"securityGroups\":[{\"id\":\"sg-000e55995d61a06bd\"}],\"securityGroupAction\":{\"type\":\"ALLOW\"}}"</code>
     *         </p>
     *         <p>
     *         The security group action for content audit can be <code>ALLOW</code> or <code>DENY</code>. For
     *         <code>ALLOW</code>, all in-scope security group rules must be within the allowed range of the policy's
     *         security group rules. For <code>DENY</code>, all in-scope security group rules must not contain a value
     *         or a range that matches a rule value or range in the policy security group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example: <code>SECURITY_GROUPS_USAGE_AUDIT</code>
     *         </p>
     *         <p>
     *         <code>"{\"type\":\"SECURITY_GROUPS_USAGE_AUDIT\",\"deleteUnusedSecurityGroups\":true,\"coalesceRedundantSecurityGroups\":true}"</code>
     *         </p>
     *         </li>
     */
    public final String managedServiceData() {
        return managedServiceData;
    }

    /**
     * <p>
     * Contains the Network Firewall firewall policy options to configure a centralized deployment model.
     * </p>
     * 
     * @return Contains the Network Firewall firewall policy options to configure a centralized deployment model.
     */
    public final PolicyOption policyOption() {
        return policyOption;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(managedServiceData());
        hashCode = 31 * hashCode + Objects.hashCode(policyOption());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SecurityServicePolicyData)) {
            return false;
        }
        SecurityServicePolicyData other = (SecurityServicePolicyData) obj;
        return Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(managedServiceData(), other.managedServiceData())
                && Objects.equals(policyOption(), other.policyOption());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SecurityServicePolicyData").add("Type", typeAsString())
                .add("ManagedServiceData", managedServiceData()).add("PolicyOption", policyOption()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "ManagedServiceData":
            return Optional.ofNullable(clazz.cast(managedServiceData()));
        case "PolicyOption":
            return Optional.ofNullable(clazz.cast(policyOption()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SecurityServicePolicyData, T> g) {
        return obj -> g.apply((SecurityServicePolicyData) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SecurityServicePolicyData> {
        /**
         * <p>
         * The service that the policy is using to protect the resources. This specifies the type of policy that is
         * created, either an WAF policy, a Shield Advanced policy, or a security group policy. For security group
         * policies, Firewall Manager supports one security group for each common policy and for each content audit
         * policy. This is an adjustable limit that you can increase by contacting Amazon Web Services Support.
         * </p>
         * 
         * @param type
         *        The service that the policy is using to protect the resources. This specifies the type of policy that
         *        is created, either an WAF policy, a Shield Advanced policy, or a security group policy. For security
         *        group policies, Firewall Manager supports one security group for each common policy and for each
         *        content audit policy. This is an adjustable limit that you can increase by contacting Amazon Web
         *        Services Support.
         * @see SecurityServiceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SecurityServiceType
         */
        Builder type(String type);

        /**
         * <p>
         * The service that the policy is using to protect the resources. This specifies the type of policy that is
         * created, either an WAF policy, a Shield Advanced policy, or a security group policy. For security group
         * policies, Firewall Manager supports one security group for each common policy and for each content audit
         * policy. This is an adjustable limit that you can increase by contacting Amazon Web Services Support.
         * </p>
         * 
         * @param type
         *        The service that the policy is using to protect the resources. This specifies the type of policy that
         *        is created, either an WAF policy, a Shield Advanced policy, or a security group policy. For security
         *        group policies, Firewall Manager supports one security group for each common policy and for each
         *        content audit policy. This is an adjustable limit that you can increase by contacting Amazon Web
         *        Services Support.
         * @see SecurityServiceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SecurityServiceType
         */
        Builder type(SecurityServiceType type);

        /**
         * <p>
         * Details about the service that are specific to the service type, in JSON format.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Example: <code>DNS_FIREWALL</code>
         * </p>
         * <p>
         * <code>"{\"type\":\"DNS_FIREWALL\",\"preProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-1\",\"priority\":10}],\"postProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-2\",\"priority\":9911}]}"</code>
         * </p>
         * <note>
         * <p>
         * Valid values for <code>preProcessRuleGroups</code> are between 1 and 99. Valid values for
         * <code>postProcessRuleGroups</code> are between 9901 and 10000.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * Example: <code>DNS_FIREWALL</code>
         * </p>
         * <p>
         * <code>"{\"type\":\"DNS_FIREWALL\",\"preProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-1\",\"priority\":10}],\"postProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-2\",\"priority\":9911}]}"</code>
         * </p>
         * <note>
         * <p>
         * Valid values for <code>preProcessRuleGroups</code> are between 1 and 99. Valid values for
         * <code>postProcessRuleGroups</code> are between 9901 and 10000.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with automatic Availability Zone
         * configuration. With automatic Availbility Zone configuration, Firewall Manager chooses which Availability
         * Zones to create the endpoints in.
         * </p>
         * <p>
         * <code>"{ \"type\": \"NETWORK_FIREWALL\", \"networkFirewallStatelessRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\", \"priority\": 1 } ], \"networkFirewallStatelessDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessCustomActions\": [ { \"actionName\": \"customActionName\", \"actionDefinition\": { \"publishMetricAction\": { \"dimensions\": [ { \"value\": \"metricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\": { \"singleFirewallEndpointPerVPC\": false, \"allowedIPV4CidrList\": [ \"10.0.0.0/28\", \"192.168.0.0/28\" ], \"routeManagementAction\": \"OFF\" }, \"networkFirewallLoggingConfiguration\": { \"logDestinationConfigs\": [ { \"logDestinationType\": \"S3\", \"logType\": \"ALERT\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } }, { \"logDestinationType\": \"S3\", \"logType\": \"FLOW\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } } ], \"overrideExistingConfig\": true } }"</code>
         * </p>
         * <p>
         * To use the distributed deployment model, you must set <a
         * href="https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PolicyOption.html">PolicyOption</a> to
         * <code>NULL</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with automatic Availability Zone
         * configuration, and route management.
         * </p>
         * <p>
         * <code>"{ \"type\": \"NETWORK_FIREWALL\", \"networkFirewallStatelessRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\", \"priority\": 1 } ], \"networkFirewallStatelessDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessCustomActions\": [ { \"actionName\": \"customActionName\", \"actionDefinition\": { \"publishMetricAction\": { \"dimensions\": [ { \"value\": \"metricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\": { \"singleFirewallEndpointPerVPC\": false, \"allowedIPV4CidrList\": [ \"10.0.0.0/28\", \"192.168.0.0/28\" ], \"routeManagementAction\": \"MONITOR\", \"routeManagementTargetTypes\": [ \"InternetGateway\" ] }, \"networkFirewallLoggingConfiguration\": { \"logDestinationConfigs\": [ { \"logDestinationType\": \"S3\", \"logType\": \"ALERT\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } }, { \"logDestinationType\": \"S3\", \"logType\": \"FLOW\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } } ], \"overrideExistingConfig\": true } }"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with custom Availability Zone
         * configuration. With custom Availability Zone configuration, you define which specific Availability Zones to
         * create endpoints in by configuring <code>firewallCreationConfig</code>.
         * </p>
         * <p>
         * <code>"{ \"type\":\"NETWORK_FIREWALL\",\"networkFirewallStatelessRuleGroupReferences\":[{\"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\",\"priority\":1}], \"networkFirewallStatelessDefaultActions\":[ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\":[ \"aws:forward_to_sfe\", \"fragmentcustomactionname\" ], \"networkFirewallStatelessCustomActions\":[ { \"actionName\":\"customActionName\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"metricdimensionvalue\" } ] } } }, { \"actionName\":\"fragmentcustomactionname\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"fragmentmetricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\":[ { \"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\":{ \"firewallCreationConfig\":{ \"endpointLocation\":{ \"availabilityZoneConfigList\":[ { \"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1a\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] }, { ¯\"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1b\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] } ] } }, \"singleFirewallEndpointPerVPC\":false, \"allowedIPV4CidrList\":null, \"routeManagementAction\":\"OFF\", \"networkFirewallLoggingConfiguration\":{ \"logDestinationConfigs\":[ { \"logDestinationType\":\"S3\", \"logType\":\"ALERT\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } }, { \"logDestinationType\":\"S3\", \"logType\":\"FLOW\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } } ], \"overrideExistingConfig\":boolean } }"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with custom Availability Zone
         * configuration, and route management.
         * </p>
         * <p>
         * <code>"{ \"type\":\"NETWORK_FIREWALL\",\"networkFirewallStatelessRuleGroupReferences\":[{\"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\",\"priority\":1}], \"networkFirewallStatelessDefaultActions\":[ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\":[ \"aws:forward_to_sfe\", \"fragmentcustomactionname\" ], \"networkFirewallStatelessCustomActions\":[ { \"actionName\":\"customActionName\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"metricdimensionvalue\" } ] } } }, { \"actionName\":\"fragmentcustomactionname\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"fragmentmetricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\":[ { \"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\":{ \"firewallCreationConfig\":{ \"endpointLocation\":{ \"availabilityZoneConfigList\":[ { \"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1a\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] }, { ¯\"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1b\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] } ] } }, \"singleFirewallEndpointPerVPC\":false, \"allowedIPV4CidrList\":null, \"routeManagementAction\":\"MONITOR\", \"routeManagementTargetTypes\":[ \"InternetGateway\" ], \"routeManagementConfig\":{ \"allowCrossAZTrafficIfNoEndpoint\":true } }, \"networkFirewallLoggingConfiguration\":{ \"logDestinationConfigs\":[ { \"logDestinationType\":\"S3\", \"logType\":\"ALERT\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } }, { \"logDestinationType\":\"S3\", \"logType\":\"FLOW\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } } ], \"overrideExistingConfig\":boolean } }"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Example: <code>PARTNER_FIREWALL</code> for Firewall Manager
         * </p>
         * <p>
         * <code>"{\"type\":\"THIRD_PARTY_FIREWALL\",\"thirdPartyrFirewall\":\"PALO_ALTO_NETWORKS_CLOUD_NGFW\",\"thirdPartyFirewallConfig\":{\"thirdPartyFirewallPolicyList\":[\"global-123456789012-1\"],\"networkFirewallLoggingConfiguration\":null},\"firewallDeploymentModel\":{\"distributedFirewallDeploymentModel\":{\"distributedFirewallOrchestrationConfig\":{\"firewallCreationConfig\":{\"endpointLocation\":{\"availabilityZoneConfigList\":[{\"availabilityZoneId\":null,\"availabilityZoneName\":\"us-east-1a\",\"allowedIPV4CidrList\":[\"10.0.1.0/28\"]}]}},\"allowedIPV4CidrList\":null},\"distributedRouteManagementConfig\":null},\"centralizedFirewallDeploymentModel\":null}}""</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Specification for <code>SHIELD_ADVANCED</code> for Amazon CloudFront distributions
         * </p>
         * <p>
         * <code>"{\"type\":\"SHIELD_ADVANCED\",\"automaticResponseConfiguration\": {\"automaticResponseStatus\":\"ENABLED|IGNORED|DISABLED\", \"automaticResponseAction\":\"BLOCK|COUNT\"}, \"overrideCustomerWebaclClassic\":true|false}"</code>
         * </p>
         * <p>
         * For example:
         * <code>"{\"type\":\"SHIELD_ADVANCED\",\"automaticResponseConfiguration\": {\"automaticResponseStatus\":\"ENABLED\", \"automaticResponseAction\":\"COUNT\"}}"</code>
         * </p>
         * <p>
         * The default value for <code>automaticResponseStatus</code> is <code>IGNORED</code>. The value for
         * <code>automaticResponseAction</code> is only required when <code>automaticResponseStatus</code> is set to
         * <code>ENABLED</code>. The default value for <code>overrideCustomerWebaclClassic</code> is <code>false</code>.
         * </p>
         * <p>
         * For other resource types that you can protect with a Shield Advanced policy, this
         * <code>ManagedServiceData</code> configuration is an empty string.
         * </p>
         * </li>
         * <li>
         * <p>
         * Example: <code>WAFV2</code>
         * </p>
         * <p>
         * <code>"{\"type\":\"WAFV2\",\"preProcessRuleGroups\":[{\"ruleGroupArn\":null,\"overrideAction\":{\"type\":\"NONE\"},\"managedRuleGroupIdentifier\":{\"version\":null,\"vendorName\":\"AWS\",\"managedRuleGroupName\":\"AWSManagedRulesAmazonIpReputationList\"},\"ruleGroupType\":\"ManagedRuleGroup\",\"excludeRules\":[{\"name\":\"NoUserAgent_HEADER\"}]}],\"postProcessRuleGroups\":[],\"defaultAction\":{\"type\":\"ALLOW\"},\"overrideCustomerWebACLAssociation\":false,\"loggingConfiguration\":{\"logDestinationConfigs\":[\"arn:aws:firehose:us-west-2:12345678912:deliverystream/aws-waf-logs-fms-admin-destination\"],\"redactedFields\":[{\"redactedFieldType\":\"SingleHeader\",\"redactedFieldValue\":\"Cookies\"},{\"redactedFieldType\":\"Method\"}]}}"</code>
         * </p>
         * <p>
         * In the <code>loggingConfiguration</code>, you can specify one <code>logDestinationConfigs</code>, you can
         * optionally provide up to 20 <code>redactedFields</code>, and the <code>RedactedFieldType</code> must be one
         * of <code>URI</code>, <code>QUERY_STRING</code>, <code>HEADER</code>, or <code>METHOD</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Example: <code>WAF Classic</code>
         * </p>
         * <p>
         * <code>"{\"type\": \"WAF\", \"ruleGroups\": [{\"id\":\"12345678-1bcd-9012-efga-0987654321ab\", \"overrideAction\" : {\"type\": \"COUNT\"}}], \"defaultAction\": {\"type\": \"BLOCK\"}}"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Example: <code>WAFV2</code> - Firewall Manager support for WAF managed rule group versioning
         * </p>
         * <p>
         * <code>"{\"type\":\"WAFV2\",\"preProcessRuleGroups\":[{\"ruleGroupArn\":null,\"overrideAction\":{\"type\":\"NONE\"},\"managedRuleGroupIdentifier\":{\"versionEnabled\":true,\"version\":\"Version_2.0\",\"vendorName\":\"AWS\",\"managedRuleGroupName\":\"AWSManagedRulesCommonRuleSet\"},\"ruleGroupType\":\"ManagedRuleGroup\",\"excludeRules\":[{\"name\":\"NoUserAgent_HEADER\"}]}],\"postProcessRuleGroups\":[],\"defaultAction\":{\"type\":\"ALLOW\"},\"overrideCustomerWebACLAssociation\":false,\"loggingConfiguration\":{\"logDestinationConfigs\":[\"arn:aws:firehose:us-west-2:12345678912:deliverystream/aws-waf-logs-fms-admin-destination\"],\"redactedFields\":[{\"redactedFieldType\":\"SingleHeader\",\"redactedFieldValue\":\"Cookies\"},{\"redactedFieldType\":\"Method\"}]}}"</code>
         * </p>
         * <p>
         * To use a specific version of a WAF managed rule group in your Firewall Manager policy, you must set
         * <code>versionEnabled</code> to <code>true</code>, and set <code>version</code> to the version you'd like to
         * use. If you don't set <code>versionEnabled</code> to <code>true</code>, or if you omit
         * <code>versionEnabled</code>, then Firewall Manager uses the default version of the WAF managed rule group.
         * </p>
         * </li>
         * <li>
         * <p>
         * Example: <code>SECURITY_GROUPS_COMMON</code>
         * </p>
         * <p>
         * <code>"{\"type\":\"SECURITY_GROUPS_COMMON\",\"revertManualSecurityGroupChanges\":false,\"exclusiveResourceSecurityGroupManagement\":false, \"applyToAllEC2InstanceENIs\":false,\"securityGroups\":[{\"id\":\" sg-000e55995d61a06bd\"}]}"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Example: Shared VPCs. Apply the preceding policy to resources in shared VPCs as well as to those in VPCs that
         * the account owns
         * </p>
         * <p>
         * <code>"{\"type\":\"SECURITY_GROUPS_COMMON\",\"revertManualSecurityGroupChanges\":false,\"exclusiveResourceSecurityGroupManagement\":false, \"applyToAllEC2InstanceENIs\":false,\"includeSharedVPC\":true,\"securityGroups\":[{\"id\":\" sg-000e55995d61a06bd\"}]}"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Example: <code>SECURITY_GROUPS_CONTENT_AUDIT</code>
         * </p>
         * <p>
         * <code>"{\"type\":\"SECURITY_GROUPS_CONTENT_AUDIT\",\"securityGroups\":[{\"id\":\"sg-000e55995d61a06bd\"}],\"securityGroupAction\":{\"type\":\"ALLOW\"}}"</code>
         * </p>
         * <p>
         * The security group action for content audit can be <code>ALLOW</code> or <code>DENY</code>. For
         * <code>ALLOW</code>, all in-scope security group rules must be within the allowed range of the policy's
         * security group rules. For <code>DENY</code>, all in-scope security group rules must not contain a value or a
         * range that matches a rule value or range in the policy security group.
         * </p>
         * </li>
         * <li>
         * <p>
         * Example: <code>SECURITY_GROUPS_USAGE_AUDIT</code>
         * </p>
         * <p>
         * <code>"{\"type\":\"SECURITY_GROUPS_USAGE_AUDIT\",\"deleteUnusedSecurityGroups\":true,\"coalesceRedundantSecurityGroups\":true}"</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param managedServiceData
         *        Details about the service that are specific to the service type, in JSON format. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Example: <code>DNS_FIREWALL</code>
         *        </p>
         *        <p>
         *        <code>"{\"type\":\"DNS_FIREWALL\",\"preProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-1\",\"priority\":10}],\"postProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-2\",\"priority\":9911}]}"</code>
         *        </p>
         *        <note>
         *        <p>
         *        Valid values for <code>preProcessRuleGroups</code> are between 1 and 99. Valid values for
         *        <code>postProcessRuleGroups</code> are between 9901 and 10000.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        Example: <code>DNS_FIREWALL</code>
         *        </p>
         *        <p>
         *        <code>"{\"type\":\"DNS_FIREWALL\",\"preProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-1\",\"priority\":10}],\"postProcessRuleGroups\":[{\"ruleGroupId\":\"rslvr-frg-2\",\"priority\":9911}]}"</code>
         *        </p>
         *        <note>
         *        <p>
         *        Valid values for <code>preProcessRuleGroups</code> are between 1 and 99. Valid values for
         *        <code>postProcessRuleGroups</code> are between 9901 and 10000.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with automatic Availability Zone
         *        configuration. With automatic Availbility Zone configuration, Firewall Manager chooses which
         *        Availability Zones to create the endpoints in.
         *        </p>
         *        <p>
         *        <code>"{ \"type\": \"NETWORK_FIREWALL\", \"networkFirewallStatelessRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\", \"priority\": 1 } ], \"networkFirewallStatelessDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessCustomActions\": [ { \"actionName\": \"customActionName\", \"actionDefinition\": { \"publishMetricAction\": { \"dimensions\": [ { \"value\": \"metricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\": { \"singleFirewallEndpointPerVPC\": false, \"allowedIPV4CidrList\": [ \"10.0.0.0/28\", \"192.168.0.0/28\" ], \"routeManagementAction\": \"OFF\" }, \"networkFirewallLoggingConfiguration\": { \"logDestinationConfigs\": [ { \"logDestinationType\": \"S3\", \"logType\": \"ALERT\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } }, { \"logDestinationType\": \"S3\", \"logType\": \"FLOW\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } } ], \"overrideExistingConfig\": true } }"</code>
         *        </p>
         *        <p>
         *        To use the distributed deployment model, you must set <a
         *        href="https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PolicyOption.html">PolicyOption</a>
         *        to <code>NULL</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with automatic Availability Zone
         *        configuration, and route management.
         *        </p>
         *        <p>
         *        <code>"{ \"type\": \"NETWORK_FIREWALL\", \"networkFirewallStatelessRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\", \"priority\": 1 } ], \"networkFirewallStatelessDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\": [ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessCustomActions\": [ { \"actionName\": \"customActionName\", \"actionDefinition\": { \"publishMetricAction\": { \"dimensions\": [ { \"value\": \"metricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\": [ { \"resourceARN\": \"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\": { \"singleFirewallEndpointPerVPC\": false, \"allowedIPV4CidrList\": [ \"10.0.0.0/28\", \"192.168.0.0/28\" ], \"routeManagementAction\": \"MONITOR\", \"routeManagementTargetTypes\": [ \"InternetGateway\" ] }, \"networkFirewallLoggingConfiguration\": { \"logDestinationConfigs\": [ { \"logDestinationType\": \"S3\", \"logType\": \"ALERT\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } }, { \"logDestinationType\": \"S3\", \"logType\": \"FLOW\", \"logDestination\": { \"bucketName\": \"s3-bucket-name\" } } ], \"overrideExistingConfig\": true } }"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with custom Availability Zone
         *        configuration. With custom Availability Zone configuration, you define which specific Availability
         *        Zones to create endpoints in by configuring <code>firewallCreationConfig</code>.
         *        </p>
         *        <p>
         *        <code>"{ \"type\":\"NETWORK_FIREWALL\",\"networkFirewallStatelessRuleGroupReferences\":[{\"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\",\"priority\":1}], \"networkFirewallStatelessDefaultActions\":[ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\":[ \"aws:forward_to_sfe\", \"fragmentcustomactionname\" ], \"networkFirewallStatelessCustomActions\":[ { \"actionName\":\"customActionName\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"metricdimensionvalue\" } ] } } }, { \"actionName\":\"fragmentcustomactionname\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"fragmentmetricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\":[ { \"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\":{ \"firewallCreationConfig\":{ \"endpointLocation\":{ \"availabilityZoneConfigList\":[ { \"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1a\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] }, { ¯\"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1b\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] } ] } }, \"singleFirewallEndpointPerVPC\":false, \"allowedIPV4CidrList\":null, \"routeManagementAction\":\"OFF\", \"networkFirewallLoggingConfiguration\":{ \"logDestinationConfigs\":[ { \"logDestinationType\":\"S3\", \"logType\":\"ALERT\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } }, { \"logDestinationType\":\"S3\", \"logType\":\"FLOW\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } } ], \"overrideExistingConfig\":boolean } }"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example: <code>NETWORK_FIREWALL</code> - Distributed deployment model with custom Availability Zone
         *        configuration, and route management.
         *        </p>
         *        <p>
         *        <code>"{ \"type\":\"NETWORK_FIREWALL\",\"networkFirewallStatelessRuleGroupReferences\":[{\"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateless-rulegroup/test\",\"priority\":1}], \"networkFirewallStatelessDefaultActions\":[ \"aws:forward_to_sfe\", \"customActionName\" ], \"networkFirewallStatelessFragmentDefaultActions\":[ \"aws:forward_to_sfe\", \"fragmentcustomactionname\" ], \"networkFirewallStatelessCustomActions\":[ { \"actionName\":\"customActionName\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"metricdimensionvalue\" } ] } } }, { \"actionName\":\"fragmentcustomactionname\", \"actionDefinition\":{ \"publishMetricAction\":{ \"dimensions\":[ { \"value\":\"fragmentmetricdimensionvalue\" } ] } } } ], \"networkFirewallStatefulRuleGroupReferences\":[ { \"resourceARN\":\"arn:aws:network-firewall:us-east-1:123456789011:stateful-rulegroup/test\" } ], \"networkFirewallOrchestrationConfig\":{ \"firewallCreationConfig\":{ \"endpointLocation\":{ \"availabilityZoneConfigList\":[ { \"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1a\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] }, { ¯\"availabilityZoneId\":null, \"availabilityZoneName\":\"us-east-1b\", \"allowedIPV4CidrList\":[ \"10.0.0.0/28\" ] } ] } }, \"singleFirewallEndpointPerVPC\":false, \"allowedIPV4CidrList\":null, \"routeManagementAction\":\"MONITOR\", \"routeManagementTargetTypes\":[ \"InternetGateway\" ], \"routeManagementConfig\":{ \"allowCrossAZTrafficIfNoEndpoint\":true } }, \"networkFirewallLoggingConfiguration\":{ \"logDestinationConfigs\":[ { \"logDestinationType\":\"S3\", \"logType\":\"ALERT\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } }, { \"logDestinationType\":\"S3\", \"logType\":\"FLOW\", \"logDestination\":{ \"bucketName\":\"s3-bucket-name\" } } ], \"overrideExistingConfig\":boolean } }"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example: <code>PARTNER_FIREWALL</code> for Firewall Manager
         *        </p>
         *        <p>
         *        <code>"{\"type\":\"THIRD_PARTY_FIREWALL\",\"thirdPartyrFirewall\":\"PALO_ALTO_NETWORKS_CLOUD_NGFW\",\"thirdPartyFirewallConfig\":{\"thirdPartyFirewallPolicyList\":[\"global-123456789012-1\"],\"networkFirewallLoggingConfiguration\":null},\"firewallDeploymentModel\":{\"distributedFirewallDeploymentModel\":{\"distributedFirewallOrchestrationConfig\":{\"firewallCreationConfig\":{\"endpointLocation\":{\"availabilityZoneConfigList\":[{\"availabilityZoneId\":null,\"availabilityZoneName\":\"us-east-1a\",\"allowedIPV4CidrList\":[\"10.0.1.0/28\"]}]}},\"allowedIPV4CidrList\":null},\"distributedRouteManagementConfig\":null},\"centralizedFirewallDeploymentModel\":null}}""</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specification for <code>SHIELD_ADVANCED</code> for Amazon CloudFront distributions
         *        </p>
         *        <p>
         *        <code>"{\"type\":\"SHIELD_ADVANCED\",\"automaticResponseConfiguration\": {\"automaticResponseStatus\":\"ENABLED|IGNORED|DISABLED\", \"automaticResponseAction\":\"BLOCK|COUNT\"}, \"overrideCustomerWebaclClassic\":true|false}"</code>
         *        </p>
         *        <p>
         *        For example:
         *        <code>"{\"type\":\"SHIELD_ADVANCED\",\"automaticResponseConfiguration\": {\"automaticResponseStatus\":\"ENABLED\", \"automaticResponseAction\":\"COUNT\"}}"</code>
         *        </p>
         *        <p>
         *        The default value for <code>automaticResponseStatus</code> is <code>IGNORED</code>. The value for
         *        <code>automaticResponseAction</code> is only required when <code>automaticResponseStatus</code> is set
         *        to <code>ENABLED</code>. The default value for <code>overrideCustomerWebaclClassic</code> is
         *        <code>false</code>.
         *        </p>
         *        <p>
         *        For other resource types that you can protect with a Shield Advanced policy, this
         *        <code>ManagedServiceData</code> configuration is an empty string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example: <code>WAFV2</code>
         *        </p>
         *        <p>
         *        <code>"{\"type\":\"WAFV2\",\"preProcessRuleGroups\":[{\"ruleGroupArn\":null,\"overrideAction\":{\"type\":\"NONE\"},\"managedRuleGroupIdentifier\":{\"version\":null,\"vendorName\":\"AWS\",\"managedRuleGroupName\":\"AWSManagedRulesAmazonIpReputationList\"},\"ruleGroupType\":\"ManagedRuleGroup\",\"excludeRules\":[{\"name\":\"NoUserAgent_HEADER\"}]}],\"postProcessRuleGroups\":[],\"defaultAction\":{\"type\":\"ALLOW\"},\"overrideCustomerWebACLAssociation\":false,\"loggingConfiguration\":{\"logDestinationConfigs\":[\"arn:aws:firehose:us-west-2:12345678912:deliverystream/aws-waf-logs-fms-admin-destination\"],\"redactedFields\":[{\"redactedFieldType\":\"SingleHeader\",\"redactedFieldValue\":\"Cookies\"},{\"redactedFieldType\":\"Method\"}]}}"</code>
         *        </p>
         *        <p>
         *        In the <code>loggingConfiguration</code>, you can specify one <code>logDestinationConfigs</code>, you
         *        can optionally provide up to 20 <code>redactedFields</code>, and the <code>RedactedFieldType</code>
         *        must be one of <code>URI</code>, <code>QUERY_STRING</code>, <code>HEADER</code>, or
         *        <code>METHOD</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example: <code>WAF Classic</code>
         *        </p>
         *        <p>
         *        <code>"{\"type\": \"WAF\", \"ruleGroups\": [{\"id\":\"12345678-1bcd-9012-efga-0987654321ab\", \"overrideAction\" : {\"type\": \"COUNT\"}}], \"defaultAction\": {\"type\": \"BLOCK\"}}"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example: <code>WAFV2</code> - Firewall Manager support for WAF managed rule group versioning
         *        </p>
         *        <p>
         *        <code>"{\"type\":\"WAFV2\",\"preProcessRuleGroups\":[{\"ruleGroupArn\":null,\"overrideAction\":{\"type\":\"NONE\"},\"managedRuleGroupIdentifier\":{\"versionEnabled\":true,\"version\":\"Version_2.0\",\"vendorName\":\"AWS\",\"managedRuleGroupName\":\"AWSManagedRulesCommonRuleSet\"},\"ruleGroupType\":\"ManagedRuleGroup\",\"excludeRules\":[{\"name\":\"NoUserAgent_HEADER\"}]}],\"postProcessRuleGroups\":[],\"defaultAction\":{\"type\":\"ALLOW\"},\"overrideCustomerWebACLAssociation\":false,\"loggingConfiguration\":{\"logDestinationConfigs\":[\"arn:aws:firehose:us-west-2:12345678912:deliverystream/aws-waf-logs-fms-admin-destination\"],\"redactedFields\":[{\"redactedFieldType\":\"SingleHeader\",\"redactedFieldValue\":\"Cookies\"},{\"redactedFieldType\":\"Method\"}]}}"</code>
         *        </p>
         *        <p>
         *        To use a specific version of a WAF managed rule group in your Firewall Manager policy, you must set
         *        <code>versionEnabled</code> to <code>true</code>, and set <code>version</code> to the version you'd
         *        like to use. If you don't set <code>versionEnabled</code> to <code>true</code>, or if you omit
         *        <code>versionEnabled</code>, then Firewall Manager uses the default version of the WAF managed rule
         *        group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example: <code>SECURITY_GROUPS_COMMON</code>
         *        </p>
         *        <p>
         *        <code>"{\"type\":\"SECURITY_GROUPS_COMMON\",\"revertManualSecurityGroupChanges\":false,\"exclusiveResourceSecurityGroupManagement\":false, \"applyToAllEC2InstanceENIs\":false,\"securityGroups\":[{\"id\":\" sg-000e55995d61a06bd\"}]}"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example: Shared VPCs. Apply the preceding policy to resources in shared VPCs as well as to those in
         *        VPCs that the account owns
         *        </p>
         *        <p>
         *        <code>"{\"type\":\"SECURITY_GROUPS_COMMON\",\"revertManualSecurityGroupChanges\":false,\"exclusiveResourceSecurityGroupManagement\":false, \"applyToAllEC2InstanceENIs\":false,\"includeSharedVPC\":true,\"securityGroups\":[{\"id\":\" sg-000e55995d61a06bd\"}]}"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example: <code>SECURITY_GROUPS_CONTENT_AUDIT</code>
         *        </p>
         *        <p>
         *        <code>"{\"type\":\"SECURITY_GROUPS_CONTENT_AUDIT\",\"securityGroups\":[{\"id\":\"sg-000e55995d61a06bd\"}],\"securityGroupAction\":{\"type\":\"ALLOW\"}}"</code>
         *        </p>
         *        <p>
         *        The security group action for content audit can be <code>ALLOW</code> or <code>DENY</code>. For
         *        <code>ALLOW</code>, all in-scope security group rules must be within the allowed range of the policy's
         *        security group rules. For <code>DENY</code>, all in-scope security group rules must not contain a
         *        value or a range that matches a rule value or range in the policy security group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example: <code>SECURITY_GROUPS_USAGE_AUDIT</code>
         *        </p>
         *        <p>
         *        <code>"{\"type\":\"SECURITY_GROUPS_USAGE_AUDIT\",\"deleteUnusedSecurityGroups\":true,\"coalesceRedundantSecurityGroups\":true}"</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedServiceData(String managedServiceData);

        /**
         * <p>
         * Contains the Network Firewall firewall policy options to configure a centralized deployment model.
         * </p>
         * 
         * @param policyOption
         *        Contains the Network Firewall firewall policy options to configure a centralized deployment model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyOption(PolicyOption policyOption);

        /**
         * <p>
         * Contains the Network Firewall firewall policy options to configure a centralized deployment model.
         * </p>
         * This is a convenience method that creates an instance of the {@link PolicyOption.Builder} avoiding the need
         * to create one manually via {@link PolicyOption#builder()}.
         *
         * When the {@link Consumer} completes, {@link PolicyOption.Builder#build()} is called immediately and its
         * result is passed to {@link #policyOption(PolicyOption)}.
         * 
         * @param policyOption
         *        a consumer that will call methods on {@link PolicyOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #policyOption(PolicyOption)
         */
        default Builder policyOption(Consumer<PolicyOption.Builder> policyOption) {
            return policyOption(PolicyOption.builder().applyMutation(policyOption).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String managedServiceData;

        private PolicyOption policyOption;

        private BuilderImpl() {
        }

        private BuilderImpl(SecurityServicePolicyData model) {
            type(model.type);
            managedServiceData(model.managedServiceData);
            policyOption(model.policyOption);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(SecurityServiceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getManagedServiceData() {
            return managedServiceData;
        }

        public final void setManagedServiceData(String managedServiceData) {
            this.managedServiceData = managedServiceData;
        }

        @Override
        public final Builder managedServiceData(String managedServiceData) {
            this.managedServiceData = managedServiceData;
            return this;
        }

        public final PolicyOption.Builder getPolicyOption() {
            return policyOption != null ? policyOption.toBuilder() : null;
        }

        public final void setPolicyOption(PolicyOption.BuilderImpl policyOption) {
            this.policyOption = policyOption != null ? policyOption.build() : null;
        }

        @Override
        public final Builder policyOption(PolicyOption policyOption) {
            this.policyOption = policyOption;
            return this;
        }

        @Override
        public SecurityServicePolicyData build() {
            return new SecurityServicePolicyData(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
