/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fms;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.fms.internal.FmsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.fms.model.AssociateAdminAccountRequest;
import software.amazon.awssdk.services.fms.model.AssociateAdminAccountResponse;
import software.amazon.awssdk.services.fms.model.AssociateThirdPartyFirewallRequest;
import software.amazon.awssdk.services.fms.model.AssociateThirdPartyFirewallResponse;
import software.amazon.awssdk.services.fms.model.BatchAssociateResourceRequest;
import software.amazon.awssdk.services.fms.model.BatchAssociateResourceResponse;
import software.amazon.awssdk.services.fms.model.BatchDisassociateResourceRequest;
import software.amazon.awssdk.services.fms.model.BatchDisassociateResourceResponse;
import software.amazon.awssdk.services.fms.model.DeleteAppsListRequest;
import software.amazon.awssdk.services.fms.model.DeleteAppsListResponse;
import software.amazon.awssdk.services.fms.model.DeleteNotificationChannelRequest;
import software.amazon.awssdk.services.fms.model.DeleteNotificationChannelResponse;
import software.amazon.awssdk.services.fms.model.DeletePolicyRequest;
import software.amazon.awssdk.services.fms.model.DeletePolicyResponse;
import software.amazon.awssdk.services.fms.model.DeleteProtocolsListRequest;
import software.amazon.awssdk.services.fms.model.DeleteProtocolsListResponse;
import software.amazon.awssdk.services.fms.model.DeleteResourceSetRequest;
import software.amazon.awssdk.services.fms.model.DeleteResourceSetResponse;
import software.amazon.awssdk.services.fms.model.DisassociateAdminAccountRequest;
import software.amazon.awssdk.services.fms.model.DisassociateAdminAccountResponse;
import software.amazon.awssdk.services.fms.model.DisassociateThirdPartyFirewallRequest;
import software.amazon.awssdk.services.fms.model.DisassociateThirdPartyFirewallResponse;
import software.amazon.awssdk.services.fms.model.FmsException;
import software.amazon.awssdk.services.fms.model.GetAdminAccountRequest;
import software.amazon.awssdk.services.fms.model.GetAdminAccountResponse;
import software.amazon.awssdk.services.fms.model.GetAdminScopeRequest;
import software.amazon.awssdk.services.fms.model.GetAdminScopeResponse;
import software.amazon.awssdk.services.fms.model.GetAppsListRequest;
import software.amazon.awssdk.services.fms.model.GetAppsListResponse;
import software.amazon.awssdk.services.fms.model.GetComplianceDetailRequest;
import software.amazon.awssdk.services.fms.model.GetComplianceDetailResponse;
import software.amazon.awssdk.services.fms.model.GetNotificationChannelRequest;
import software.amazon.awssdk.services.fms.model.GetNotificationChannelResponse;
import software.amazon.awssdk.services.fms.model.GetPolicyRequest;
import software.amazon.awssdk.services.fms.model.GetPolicyResponse;
import software.amazon.awssdk.services.fms.model.GetProtectionStatusRequest;
import software.amazon.awssdk.services.fms.model.GetProtectionStatusResponse;
import software.amazon.awssdk.services.fms.model.GetProtocolsListRequest;
import software.amazon.awssdk.services.fms.model.GetProtocolsListResponse;
import software.amazon.awssdk.services.fms.model.GetResourceSetRequest;
import software.amazon.awssdk.services.fms.model.GetResourceSetResponse;
import software.amazon.awssdk.services.fms.model.GetThirdPartyFirewallAssociationStatusRequest;
import software.amazon.awssdk.services.fms.model.GetThirdPartyFirewallAssociationStatusResponse;
import software.amazon.awssdk.services.fms.model.GetViolationDetailsRequest;
import software.amazon.awssdk.services.fms.model.GetViolationDetailsResponse;
import software.amazon.awssdk.services.fms.model.InternalErrorException;
import software.amazon.awssdk.services.fms.model.InvalidInputException;
import software.amazon.awssdk.services.fms.model.InvalidOperationException;
import software.amazon.awssdk.services.fms.model.InvalidTypeException;
import software.amazon.awssdk.services.fms.model.LimitExceededException;
import software.amazon.awssdk.services.fms.model.ListAdminAccountsForOrganizationRequest;
import software.amazon.awssdk.services.fms.model.ListAdminAccountsForOrganizationResponse;
import software.amazon.awssdk.services.fms.model.ListAdminsManagingAccountRequest;
import software.amazon.awssdk.services.fms.model.ListAdminsManagingAccountResponse;
import software.amazon.awssdk.services.fms.model.ListAppsListsRequest;
import software.amazon.awssdk.services.fms.model.ListAppsListsResponse;
import software.amazon.awssdk.services.fms.model.ListComplianceStatusRequest;
import software.amazon.awssdk.services.fms.model.ListComplianceStatusResponse;
import software.amazon.awssdk.services.fms.model.ListDiscoveredResourcesRequest;
import software.amazon.awssdk.services.fms.model.ListDiscoveredResourcesResponse;
import software.amazon.awssdk.services.fms.model.ListMemberAccountsRequest;
import software.amazon.awssdk.services.fms.model.ListMemberAccountsResponse;
import software.amazon.awssdk.services.fms.model.ListPoliciesRequest;
import software.amazon.awssdk.services.fms.model.ListPoliciesResponse;
import software.amazon.awssdk.services.fms.model.ListProtocolsListsRequest;
import software.amazon.awssdk.services.fms.model.ListProtocolsListsResponse;
import software.amazon.awssdk.services.fms.model.ListResourceSetResourcesRequest;
import software.amazon.awssdk.services.fms.model.ListResourceSetResourcesResponse;
import software.amazon.awssdk.services.fms.model.ListResourceSetsRequest;
import software.amazon.awssdk.services.fms.model.ListResourceSetsResponse;
import software.amazon.awssdk.services.fms.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.fms.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.fms.model.ListThirdPartyFirewallFirewallPoliciesRequest;
import software.amazon.awssdk.services.fms.model.ListThirdPartyFirewallFirewallPoliciesResponse;
import software.amazon.awssdk.services.fms.model.PutAdminAccountRequest;
import software.amazon.awssdk.services.fms.model.PutAdminAccountResponse;
import software.amazon.awssdk.services.fms.model.PutAppsListRequest;
import software.amazon.awssdk.services.fms.model.PutAppsListResponse;
import software.amazon.awssdk.services.fms.model.PutNotificationChannelRequest;
import software.amazon.awssdk.services.fms.model.PutNotificationChannelResponse;
import software.amazon.awssdk.services.fms.model.PutPolicyRequest;
import software.amazon.awssdk.services.fms.model.PutPolicyResponse;
import software.amazon.awssdk.services.fms.model.PutProtocolsListRequest;
import software.amazon.awssdk.services.fms.model.PutProtocolsListResponse;
import software.amazon.awssdk.services.fms.model.PutResourceSetRequest;
import software.amazon.awssdk.services.fms.model.PutResourceSetResponse;
import software.amazon.awssdk.services.fms.model.ResourceNotFoundException;
import software.amazon.awssdk.services.fms.model.TagResourceRequest;
import software.amazon.awssdk.services.fms.model.TagResourceResponse;
import software.amazon.awssdk.services.fms.model.UntagResourceRequest;
import software.amazon.awssdk.services.fms.model.UntagResourceResponse;
import software.amazon.awssdk.services.fms.transform.AssociateAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.AssociateThirdPartyFirewallRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.BatchAssociateResourceRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.BatchDisassociateResourceRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeleteAppsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeleteNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeletePolicyRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeleteProtocolsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeleteResourceSetRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DisassociateAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DisassociateThirdPartyFirewallRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetAdminScopeRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetAppsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetComplianceDetailRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetPolicyRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetProtectionStatusRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetProtocolsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetResourceSetRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetThirdPartyFirewallAssociationStatusRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetViolationDetailsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListAdminAccountsForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListAdminsManagingAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListAppsListsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListComplianceStatusRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListDiscoveredResourcesRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListMemberAccountsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListPoliciesRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListProtocolsListsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListResourceSetResourcesRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListResourceSetsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListThirdPartyFirewallFirewallPoliciesRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutAppsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutPolicyRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutProtocolsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutResourceSetRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link FmsAsyncClient}.
 *
 * @see FmsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFmsAsyncClient implements FmsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultFmsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final FmsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultFmsAsyncClient(FmsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Sets a Firewall Manager default administrator account. The Firewall Manager default administrator account can
     * manage third-party firewalls and has full administrative scope that allows administration of all policy types,
     * accounts, organizational units, and Regions. This account must be a member account of the organization in
     * Organizations whose resources you want to protect.
     * </p>
     * <p>
     * For information about working with Firewall Manager administrator accounts, see <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/fms-administrators.html">Managing Firewall
     * Manager administrators</a> in the <i>Firewall Manager Developer Guide</i>.
     * </p>
     *
     * @param associateAdminAccountRequest
     * @return A Java Future containing the result of the AssociateAdminAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.AssociateAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/AssociateAdminAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateAdminAccountResponse> associateAdminAccount(
            AssociateAdminAccountRequest associateAdminAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateAdminAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateAdminAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateAdminAccountRequest, AssociateAdminAccountResponse>()
                            .withOperationName("AssociateAdminAccount")
                            .withMarshaller(new AssociateAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateAdminAccountRequest));
            CompletableFuture<AssociateAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the Firewall Manager policy administrator as a tenant administrator of a third-party firewall service. A
     * tenant is an instance of the third-party firewall service that's associated with your Amazon Web Services
     * customer account.
     * </p>
     *
     * @param associateThirdPartyFirewallRequest
     * @return A Java Future containing the result of the AssociateThirdPartyFirewall operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.AssociateThirdPartyFirewall
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/AssociateThirdPartyFirewall"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateThirdPartyFirewallResponse> associateThirdPartyFirewall(
            AssociateThirdPartyFirewallRequest associateThirdPartyFirewallRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateThirdPartyFirewallRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateThirdPartyFirewallRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateThirdPartyFirewall");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateThirdPartyFirewallResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateThirdPartyFirewallResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateThirdPartyFirewallResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateThirdPartyFirewallRequest, AssociateThirdPartyFirewallResponse>()
                            .withOperationName("AssociateThirdPartyFirewall")
                            .withMarshaller(new AssociateThirdPartyFirewallRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateThirdPartyFirewallRequest));
            CompletableFuture<AssociateThirdPartyFirewallResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associate resources to a Firewall Manager resource set.
     * </p>
     *
     * @param batchAssociateResourceRequest
     * @return A Java Future containing the result of the BatchAssociateResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.BatchAssociateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/BatchAssociateResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchAssociateResourceResponse> batchAssociateResource(
            BatchAssociateResourceRequest batchAssociateResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchAssociateResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchAssociateResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchAssociateResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchAssociateResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchAssociateResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchAssociateResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchAssociateResourceRequest, BatchAssociateResourceResponse>()
                            .withOperationName("BatchAssociateResource")
                            .withMarshaller(new BatchAssociateResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchAssociateResourceRequest));
            CompletableFuture<BatchAssociateResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates resources from a Firewall Manager resource set.
     * </p>
     *
     * @param batchDisassociateResourceRequest
     * @return A Java Future containing the result of the BatchDisassociateResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.BatchDisassociateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/BatchDisassociateResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDisassociateResourceResponse> batchDisassociateResource(
            BatchDisassociateResourceRequest batchDisassociateResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDisassociateResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDisassociateResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDisassociateResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDisassociateResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDisassociateResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDisassociateResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDisassociateResourceRequest, BatchDisassociateResourceResponse>()
                            .withOperationName("BatchDisassociateResource")
                            .withMarshaller(new BatchDisassociateResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDisassociateResourceRequest));
            CompletableFuture<BatchDisassociateResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes an Firewall Manager applications list.
     * </p>
     *
     * @param deleteAppsListRequest
     * @return A Java Future containing the result of the DeleteAppsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DeleteAppsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeleteAppsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppsListResponse> deleteAppsList(DeleteAppsListRequest deleteAppsListRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppsListRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppsListResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppsListRequest, DeleteAppsListResponse>()
                            .withOperationName("DeleteAppsList")
                            .withMarshaller(new DeleteAppsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAppsListRequest));
            CompletableFuture<DeleteAppsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Firewall Manager association with the IAM role and the Amazon Simple Notification Service (SNS) topic
     * that is used to record Firewall Manager SNS logs.
     * </p>
     *
     * @param deleteNotificationChannelRequest
     * @return A Java Future containing the result of the DeleteNotificationChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DeleteNotificationChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeleteNotificationChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNotificationChannelResponse> deleteNotificationChannel(
            DeleteNotificationChannelRequest deleteNotificationChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteNotificationChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNotificationChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNotificationChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNotificationChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNotificationChannelRequest, DeleteNotificationChannelResponse>()
                            .withOperationName("DeleteNotificationChannel")
                            .withMarshaller(new DeleteNotificationChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteNotificationChannelRequest));
            CompletableFuture<DeleteNotificationChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes an Firewall Manager policy.
     * </p>
     *
     * @param deletePolicyRequest
     * @return A Java Future containing the result of the DeletePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DeletePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeletePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePolicyResponse> deletePolicy(DeletePolicyRequest deletePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeletePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePolicyRequest, DeletePolicyResponse>()
                            .withOperationName("DeletePolicy").withMarshaller(new DeletePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePolicyRequest));
            CompletableFuture<DeletePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes an Firewall Manager protocols list.
     * </p>
     *
     * @param deleteProtocolsListRequest
     * @return A Java Future containing the result of the DeleteProtocolsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DeleteProtocolsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeleteProtocolsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProtocolsListResponse> deleteProtocolsList(
            DeleteProtocolsListRequest deleteProtocolsListRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProtocolsListRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProtocolsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProtocolsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProtocolsListResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteProtocolsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProtocolsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProtocolsListRequest, DeleteProtocolsListResponse>()
                            .withOperationName("DeleteProtocolsList")
                            .withMarshaller(new DeleteProtocolsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteProtocolsListRequest));
            CompletableFuture<DeleteProtocolsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified <a>ResourceSet</a>.
     * </p>
     *
     * @param deleteResourceSetRequest
     * @return A Java Future containing the result of the DeleteResourceSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DeleteResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeleteResourceSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourceSetResponse> deleteResourceSet(DeleteResourceSetRequest deleteResourceSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourceSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourceSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourceSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourceSetRequest, DeleteResourceSetResponse>()
                            .withOperationName("DeleteResourceSet")
                            .withMarshaller(new DeleteResourceSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteResourceSetRequest));
            CompletableFuture<DeleteResourceSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates an Firewall Manager administrator account. To set a different account as an Firewall Manager
     * administrator, submit a <a>PutAdminAccount</a> request. To set an account as a default administrator account, you
     * must submit an <a>AssociateAdminAccount</a> request.
     * </p>
     * <p>
     * Disassociation of the default administrator account follows the first in, last out principle. If you are the
     * default administrator, all Firewall Manager administrators within the organization must first disassociate their
     * accounts before you can disassociate your account.
     * </p>
     *
     * @param disassociateAdminAccountRequest
     * @return A Java Future containing the result of the DisassociateAdminAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DisassociateAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DisassociateAdminAccount" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateAdminAccountResponse> disassociateAdminAccount(
            DisassociateAdminAccountRequest disassociateAdminAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateAdminAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateAdminAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateAdminAccountRequest, DisassociateAdminAccountResponse>()
                            .withOperationName("DisassociateAdminAccount")
                            .withMarshaller(new DisassociateAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateAdminAccountRequest));
            CompletableFuture<DisassociateAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a Firewall Manager policy administrator from a third-party firewall tenant. When you call
     * <code>DisassociateThirdPartyFirewall</code>, the third-party firewall vendor deletes all of the firewalls that
     * are associated with the account.
     * </p>
     *
     * @param disassociateThirdPartyFirewallRequest
     * @return A Java Future containing the result of the DisassociateThirdPartyFirewall operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.DisassociateThirdPartyFirewall
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DisassociateThirdPartyFirewall"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateThirdPartyFirewallResponse> disassociateThirdPartyFirewall(
            DisassociateThirdPartyFirewallRequest disassociateThirdPartyFirewallRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateThirdPartyFirewallRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateThirdPartyFirewallRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateThirdPartyFirewall");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateThirdPartyFirewallResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateThirdPartyFirewallResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateThirdPartyFirewallResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateThirdPartyFirewallRequest, DisassociateThirdPartyFirewallResponse>()
                            .withOperationName("DisassociateThirdPartyFirewall")
                            .withMarshaller(new DisassociateThirdPartyFirewallRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateThirdPartyFirewallRequest));
            CompletableFuture<DisassociateThirdPartyFirewallResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the Organizations account that is associated with Firewall Manager as the Firewall Manager default
     * administrator.
     * </p>
     *
     * @param getAdminAccountRequest
     * @return A Java Future containing the result of the GetAdminAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetAdminAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAdminAccountResponse> getAdminAccount(GetAdminAccountRequest getAdminAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAdminAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAdminAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAdminAccountRequest, GetAdminAccountResponse>()
                            .withOperationName("GetAdminAccount")
                            .withMarshaller(new GetAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAdminAccountRequest));
            CompletableFuture<GetAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified account's administrative scope. The admistrative scope defines the
     * resources that an Firewall Manager administrator can manage.
     * </p>
     *
     * @param getAdminScopeRequest
     * @return A Java Future containing the result of the GetAdminScope operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetAdminScope
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetAdminScope" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAdminScopeResponse> getAdminScope(GetAdminScopeRequest getAdminScopeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAdminScopeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAdminScopeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAdminScope");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAdminScopeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAdminScopeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAdminScopeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAdminScopeRequest, GetAdminScopeResponse>()
                            .withOperationName("GetAdminScope")
                            .withMarshaller(new GetAdminScopeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAdminScopeRequest));
            CompletableFuture<GetAdminScopeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified Firewall Manager applications list.
     * </p>
     *
     * @param getAppsListRequest
     * @return A Java Future containing the result of the GetAppsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetAppsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetAppsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppsListResponse> getAppsList(GetAppsListRequest getAppsListRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAppsListRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppsListResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAppsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppsListRequest, GetAppsListResponse>()
                            .withOperationName("GetAppsList").withMarshaller(new GetAppsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAppsListRequest));
            CompletableFuture<GetAppsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed compliance information about the specified member account. Details include resources that are in
     * and out of compliance with the specified policy.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Resources are considered noncompliant for WAF and Shield Advanced policies if the specified policy has not been
     * applied to them.
     * </p>
     * </li>
     * <li>
     * <p>
     * Resources are considered noncompliant for security group policies if they are in scope of the policy, they
     * violate one or more of the policy rules, and remediation is disabled or not possible.
     * </p>
     * </li>
     * <li>
     * <p>
     * Resources are considered noncompliant for Network Firewall policies if a firewall is missing in the VPC, if the
     * firewall endpoint isn't set up in an expected Availability Zone and subnet, if a subnet created by the Firewall
     * Manager doesn't have the expected route table, and for modifications to a firewall policy that violate the
     * Firewall Manager policy's rules.
     * </p>
     * </li>
     * <li>
     * <p>
     * Resources are considered noncompliant for DNS Firewall policies if a DNS Firewall rule group is missing from the
     * rule group associations for the VPC.
     * </p>
     * </li>
     * </ul>
     *
     * @param getComplianceDetailRequest
     * @return A Java Future containing the result of the GetComplianceDetail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetComplianceDetail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetComplianceDetail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetComplianceDetailResponse> getComplianceDetail(
            GetComplianceDetailRequest getComplianceDetailRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getComplianceDetailRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComplianceDetailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComplianceDetail");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetComplianceDetailResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetComplianceDetailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetComplianceDetailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetComplianceDetailRequest, GetComplianceDetailResponse>()
                            .withOperationName("GetComplianceDetail")
                            .withMarshaller(new GetComplianceDetailRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getComplianceDetailRequest));
            CompletableFuture<GetComplianceDetailResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Information about the Amazon Simple Notification Service (SNS) topic that is used to record Firewall Manager SNS
     * logs.
     * </p>
     *
     * @param getNotificationChannelRequest
     * @return A Java Future containing the result of the GetNotificationChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetNotificationChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetNotificationChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetNotificationChannelResponse> getNotificationChannel(
            GetNotificationChannelRequest getNotificationChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getNotificationChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNotificationChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetNotificationChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNotificationChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNotificationChannelRequest, GetNotificationChannelResponse>()
                            .withOperationName("GetNotificationChannel")
                            .withMarshaller(new GetNotificationChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getNotificationChannelRequest));
            CompletableFuture<GetNotificationChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified Firewall Manager policy.
     * </p>
     *
     * @param getPolicyRequest
     * @return A Java Future containing the result of the GetPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidTypeException The value of the <code>Type</code> parameter is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPolicyResponse> getPolicy(GetPolicyRequest getPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPolicyRequest, GetPolicyResponse>().withOperationName("GetPolicy")
                            .withMarshaller(new GetPolicyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPolicyRequest));
            CompletableFuture<GetPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * If you created a Shield Advanced policy, returns policy-level attack summary information in the event of a
     * potential DDoS attack. Other policy types are currently unsupported.
     * </p>
     *
     * @param getProtectionStatusRequest
     * @return A Java Future containing the result of the GetProtectionStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetProtectionStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetProtectionStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetProtectionStatusResponse> getProtectionStatus(
            GetProtectionStatusRequest getProtectionStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProtectionStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProtectionStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProtectionStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProtectionStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetProtectionStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProtectionStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProtectionStatusRequest, GetProtectionStatusResponse>()
                            .withOperationName("GetProtectionStatus")
                            .withMarshaller(new GetProtectionStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getProtectionStatusRequest));
            CompletableFuture<GetProtectionStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified Firewall Manager protocols list.
     * </p>
     *
     * @param getProtocolsListRequest
     * @return A Java Future containing the result of the GetProtocolsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetProtocolsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetProtocolsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetProtocolsListResponse> getProtocolsList(GetProtocolsListRequest getProtocolsListRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProtocolsListRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProtocolsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProtocolsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProtocolsListResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetProtocolsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProtocolsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProtocolsListRequest, GetProtocolsListResponse>()
                            .withOperationName("GetProtocolsList")
                            .withMarshaller(new GetProtocolsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getProtocolsListRequest));
            CompletableFuture<GetProtocolsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a specific resource set.
     * </p>
     *
     * @param getResourceSetRequest
     * @return A Java Future containing the result of the GetResourceSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetResourceSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceSetResponse> getResourceSet(GetResourceSetRequest getResourceSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceSetRequest, GetResourceSetResponse>()
                            .withOperationName("GetResourceSet")
                            .withMarshaller(new GetResourceSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceSetRequest));
            CompletableFuture<GetResourceSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The onboarding status of a Firewall Manager admin account to third-party firewall vendor tenant.
     * </p>
     *
     * @param getThirdPartyFirewallAssociationStatusRequest
     * @return A Java Future containing the result of the GetThirdPartyFirewallAssociationStatus operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetThirdPartyFirewallAssociationStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetThirdPartyFirewallAssociationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetThirdPartyFirewallAssociationStatusResponse> getThirdPartyFirewallAssociationStatus(
            GetThirdPartyFirewallAssociationStatusRequest getThirdPartyFirewallAssociationStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getThirdPartyFirewallAssociationStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getThirdPartyFirewallAssociationStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetThirdPartyFirewallAssociationStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetThirdPartyFirewallAssociationStatusResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetThirdPartyFirewallAssociationStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetThirdPartyFirewallAssociationStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetThirdPartyFirewallAssociationStatusRequest, GetThirdPartyFirewallAssociationStatusResponse>()
                            .withOperationName("GetThirdPartyFirewallAssociationStatus")
                            .withMarshaller(new GetThirdPartyFirewallAssociationStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getThirdPartyFirewallAssociationStatusRequest));
            CompletableFuture<GetThirdPartyFirewallAssociationStatusResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves violations for a resource based on the specified Firewall Manager policy and Amazon Web Services
     * account.
     * </p>
     *
     * @param getViolationDetailsRequest
     * @return A Java Future containing the result of the GetViolationDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.GetViolationDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetViolationDetails" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetViolationDetailsResponse> getViolationDetails(
            GetViolationDetailsRequest getViolationDetailsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getViolationDetailsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getViolationDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetViolationDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetViolationDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetViolationDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetViolationDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetViolationDetailsRequest, GetViolationDetailsResponse>()
                            .withOperationName("GetViolationDetails")
                            .withMarshaller(new GetViolationDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getViolationDetailsRequest));
            CompletableFuture<GetViolationDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a <code>AdminAccounts</code> object that lists the Firewall Manager administrators within the
     * organization that are onboarded to Firewall Manager by <a>AssociateAdminAccount</a>.
     * </p>
     * <p>
     * This operation can be called only from the organization's management account.
     * </p>
     *
     * @param listAdminAccountsForOrganizationRequest
     * @return A Java Future containing the result of the ListAdminAccountsForOrganization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListAdminAccountsForOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListAdminAccountsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAdminAccountsForOrganizationResponse> listAdminAccountsForOrganization(
            ListAdminAccountsForOrganizationRequest listAdminAccountsForOrganizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAdminAccountsForOrganizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAdminAccountsForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAdminAccountsForOrganization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAdminAccountsForOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListAdminAccountsForOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAdminAccountsForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAdminAccountsForOrganizationRequest, ListAdminAccountsForOrganizationResponse>()
                            .withOperationName("ListAdminAccountsForOrganization")
                            .withMarshaller(new ListAdminAccountsForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAdminAccountsForOrganizationRequest));
            CompletableFuture<ListAdminAccountsForOrganizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the accounts that are managing the specified Organizations member account. This is useful for any member
     * account so that they can view the accounts who are managing their account. This operation only returns the
     * managing administrators that have the requested account within their <a>AdminScope</a>.
     * </p>
     *
     * @param listAdminsManagingAccountRequest
     * @return A Java Future containing the result of the ListAdminsManagingAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListAdminsManagingAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListAdminsManagingAccount" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAdminsManagingAccountResponse> listAdminsManagingAccount(
            ListAdminsManagingAccountRequest listAdminsManagingAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAdminsManagingAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAdminsManagingAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAdminsManagingAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAdminsManagingAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAdminsManagingAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAdminsManagingAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAdminsManagingAccountRequest, ListAdminsManagingAccountResponse>()
                            .withOperationName("ListAdminsManagingAccount")
                            .withMarshaller(new ListAdminsManagingAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAdminsManagingAccountRequest));
            CompletableFuture<ListAdminsManagingAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>AppsListDataSummary</code> objects.
     * </p>
     *
     * @param listAppsListsRequest
     * @return A Java Future containing the result of the ListAppsLists operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListAppsLists
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListAppsLists" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppsListsResponse> listAppsLists(ListAppsListsRequest listAppsListsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppsListsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppsListsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppsLists");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppsListsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAppsListsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppsListsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppsListsRequest, ListAppsListsResponse>()
                            .withOperationName("ListAppsLists")
                            .withMarshaller(new ListAppsListsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAppsListsRequest));
            CompletableFuture<ListAppsListsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>PolicyComplianceStatus</code> objects. Use <code>PolicyComplianceStatus</code> to get a
     * summary of which member accounts are protected by the specified policy.
     * </p>
     *
     * @param listComplianceStatusRequest
     * @return A Java Future containing the result of the ListComplianceStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListComplianceStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListComplianceStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListComplianceStatusResponse> listComplianceStatus(
            ListComplianceStatusRequest listComplianceStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listComplianceStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComplianceStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComplianceStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListComplianceStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListComplianceStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListComplianceStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListComplianceStatusRequest, ListComplianceStatusResponse>()
                            .withOperationName("ListComplianceStatus")
                            .withMarshaller(new ListComplianceStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listComplianceStatusRequest));
            CompletableFuture<ListComplianceStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of resources in the organization's accounts that are available to be associated with a resource
     * set.
     * </p>
     *
     * @param listDiscoveredResourcesRequest
     * @return A Java Future containing the result of the ListDiscoveredResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListDiscoveredResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListDiscoveredResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDiscoveredResourcesResponse> listDiscoveredResources(
            ListDiscoveredResourcesRequest listDiscoveredResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDiscoveredResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDiscoveredResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDiscoveredResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDiscoveredResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDiscoveredResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDiscoveredResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDiscoveredResourcesRequest, ListDiscoveredResourcesResponse>()
                            .withOperationName("ListDiscoveredResources")
                            .withMarshaller(new ListDiscoveredResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDiscoveredResourcesRequest));
            CompletableFuture<ListDiscoveredResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a <code>MemberAccounts</code> object that lists the member accounts in the administrator's Amazon Web
     * Services organization.
     * </p>
     * <p>
     * Either an Firewall Manager administrator or the organization's management account can make this request.
     * </p>
     *
     * @param listMemberAccountsRequest
     * @return A Java Future containing the result of the ListMemberAccounts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListMemberAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMemberAccountsResponse> listMemberAccounts(ListMemberAccountsRequest listMemberAccountsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMemberAccountsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMemberAccountsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMemberAccounts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMemberAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMemberAccountsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMemberAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMemberAccountsRequest, ListMemberAccountsResponse>()
                            .withOperationName("ListMemberAccounts")
                            .withMarshaller(new ListMemberAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMemberAccountsRequest));
            CompletableFuture<ListMemberAccountsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>PolicySummary</code> objects.
     * </p>
     *
     * @param listPoliciesRequest
     * @return A Java Future containing the result of the ListPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListPolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPoliciesResponse> listPolicies(ListPoliciesRequest listPoliciesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPoliciesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPoliciesRequest, ListPoliciesResponse>()
                            .withOperationName("ListPolicies").withMarshaller(new ListPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPoliciesRequest));
            CompletableFuture<ListPoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>ProtocolsListDataSummary</code> objects.
     * </p>
     *
     * @param listProtocolsListsRequest
     * @return A Java Future containing the result of the ListProtocolsLists operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListProtocolsLists
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListProtocolsLists" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProtocolsListsResponse> listProtocolsLists(ListProtocolsListsRequest listProtocolsListsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProtocolsListsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProtocolsListsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProtocolsLists");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProtocolsListsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProtocolsListsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProtocolsListsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProtocolsListsRequest, ListProtocolsListsResponse>()
                            .withOperationName("ListProtocolsLists")
                            .withMarshaller(new ListProtocolsListsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listProtocolsListsRequest));
            CompletableFuture<ListProtocolsListsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of resources that are currently associated to a resource set.
     * </p>
     *
     * @param listResourceSetResourcesRequest
     * @return A Java Future containing the result of the ListResourceSetResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListResourceSetResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListResourceSetResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourceSetResourcesResponse> listResourceSetResources(
            ListResourceSetResourcesRequest listResourceSetResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceSetResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceSetResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceSetResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourceSetResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourceSetResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourceSetResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourceSetResourcesRequest, ListResourceSetResourcesResponse>()
                            .withOperationName("ListResourceSetResources")
                            .withMarshaller(new ListResourceSetResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listResourceSetResourcesRequest));
            CompletableFuture<ListResourceSetResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of <code>ResourceSetSummary</code> objects.
     * </p>
     *
     * @param listResourceSetsRequest
     * @return A Java Future containing the result of the ListResourceSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListResourceSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListResourceSets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourceSetsResponse> listResourceSets(ListResourceSetsRequest listResourceSetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceSetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourceSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourceSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourceSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourceSetsRequest, ListResourceSetsResponse>()
                            .withOperationName("ListResourceSets")
                            .withMarshaller(new ListResourceSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listResourceSetsRequest));
            CompletableFuture<ListResourceSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of tags for the specified Amazon Web Services resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all of the third-party firewall policies that are associated with the third-party firewall
     * administrator's account.
     * </p>
     *
     * @param listThirdPartyFirewallFirewallPoliciesRequest
     * @return A Java Future containing the result of the ListThirdPartyFirewallFirewallPolicies operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.ListThirdPartyFirewallFirewallPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListThirdPartyFirewallFirewallPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListThirdPartyFirewallFirewallPoliciesResponse> listThirdPartyFirewallFirewallPolicies(
            ListThirdPartyFirewallFirewallPoliciesRequest listThirdPartyFirewallFirewallPoliciesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listThirdPartyFirewallFirewallPoliciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listThirdPartyFirewallFirewallPoliciesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListThirdPartyFirewallFirewallPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListThirdPartyFirewallFirewallPoliciesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListThirdPartyFirewallFirewallPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListThirdPartyFirewallFirewallPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListThirdPartyFirewallFirewallPoliciesRequest, ListThirdPartyFirewallFirewallPoliciesResponse>()
                            .withOperationName("ListThirdPartyFirewallFirewallPolicies")
                            .withMarshaller(new ListThirdPartyFirewallFirewallPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listThirdPartyFirewallFirewallPoliciesRequest));
            CompletableFuture<ListThirdPartyFirewallFirewallPoliciesResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates an Firewall Manager administrator account. The account must be a member of the organization
     * that was onboarded to Firewall Manager by <a>AssociateAdminAccount</a>. Only the organization's management
     * account can create an Firewall Manager administrator account. When you create an Firewall Manager administrator
     * account, the service checks to see if the account is already a delegated administrator within Organizations. If
     * the account isn't a delegated administrator, Firewall Manager calls Organizations to delegate the account within
     * Organizations. For more information about administrator accounts within Organizations, see <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts.html">Managing the Amazon
     * Web Services Accounts in Your Organization</a>.
     * </p>
     *
     * @param putAdminAccountRequest
     * @return A Java Future containing the result of the PutAdminAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.PutAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutAdminAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutAdminAccountResponse> putAdminAccount(PutAdminAccountRequest putAdminAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAdminAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAdminAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAdminAccountRequest, PutAdminAccountResponse>()
                            .withOperationName("PutAdminAccount")
                            .withMarshaller(new PutAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putAdminAccountRequest));
            CompletableFuture<PutAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Firewall Manager applications list.
     * </p>
     *
     * @param putAppsListRequest
     * @return A Java Future containing the result of the PutAppsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.PutAppsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutAppsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutAppsListResponse> putAppsList(PutAppsListRequest putAppsListRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAppsListRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAppsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAppsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAppsListResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutAppsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAppsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAppsListRequest, PutAppsListResponse>()
                            .withOperationName("PutAppsList").withMarshaller(new PutAppsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putAppsListRequest));
            CompletableFuture<PutAppsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Designates the IAM role and Amazon Simple Notification Service (SNS) topic that Firewall Manager uses to record
     * SNS logs.
     * </p>
     * <p>
     * To perform this action outside of the console, you must first configure the SNS topic's access policy to allow
     * the <code>SnsRoleName</code> to publish SNS logs. If the <code>SnsRoleName</code> provided is a role other than
     * the <code>AWSServiceRoleForFMS</code> service-linked role, this role must have a trust relationship configured to
     * allow the Firewall Manager service principal <code>fms.amazonaws.com</code> to assume this role. For information
     * about configuring an SNS access policy, see <a href=
     * "https://docs.aws.amazon.com/waf/latest/developerguide/fms-security_iam_service-with-iam.html#fms-security_iam_service-with-iam-roles-service"
     * >Service roles for Firewall Manager</a> in the <i>Firewall Manager Developer Guide</i>.
     * </p>
     *
     * @param putNotificationChannelRequest
     * @return A Java Future containing the result of the PutNotificationChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.PutNotificationChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutNotificationChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutNotificationChannelResponse> putNotificationChannel(
            PutNotificationChannelRequest putNotificationChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putNotificationChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutNotificationChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutNotificationChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutNotificationChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutNotificationChannelRequest, PutNotificationChannelResponse>()
                            .withOperationName("PutNotificationChannel")
                            .withMarshaller(new PutNotificationChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putNotificationChannelRequest));
            CompletableFuture<PutNotificationChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Firewall Manager policy.
     * </p>
     * <p>
     * Firewall Manager provides the following types of policies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An WAF policy (type WAFV2), which defines rule groups to run first in the corresponding WAF web ACL and rule
     * groups to run last in the web ACL.
     * </p>
     * </li>
     * <li>
     * <p>
     * An WAF Classic policy (type WAF), which defines a rule group.
     * </p>
     * </li>
     * <li>
     * <p>
     * A Shield Advanced policy, which applies Shield Advanced protection to specified accounts and resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * A security group policy, which manages VPC security groups across your Amazon Web Services organization.
     * </p>
     * </li>
     * <li>
     * <p>
     * An Network Firewall policy, which provides firewall rules to filter network traffic in specified Amazon VPCs.
     * </p>
     * </li>
     * <li>
     * <p>
     * A DNS Firewall policy, which provides Route 53 Resolver DNS Firewall rules to filter DNS queries for specified
     * VPCs.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Each policy is specific to one of the types. If you want to enforce more than one policy type across accounts,
     * create multiple policies. You can create multiple policies for each type.
     * </p>
     * <p>
     * You must be subscribed to Shield Advanced to create a Shield Advanced policy. For more information about
     * subscribing to Shield Advanced, see <a
     * href="https://docs.aws.amazon.com/waf/latest/DDOSAPIReference/API_CreateSubscription.html"
     * >CreateSubscription</a>.
     * </p>
     *
     * @param putPolicyRequest
     * @return A Java Future containing the result of the PutPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidTypeException The value of the <code>Type</code> parameter is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.PutPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutPolicyResponse> putPolicy(PutPolicyRequest putPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putPolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPolicyRequest, PutPolicyResponse>().withOperationName("PutPolicy")
                            .withMarshaller(new PutPolicyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(putPolicyRequest));
            CompletableFuture<PutPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Firewall Manager protocols list.
     * </p>
     *
     * @param putProtocolsListRequest
     * @return A Java Future containing the result of the PutProtocolsList operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.PutProtocolsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutProtocolsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutProtocolsListResponse> putProtocolsList(PutProtocolsListRequest putProtocolsListRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putProtocolsListRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putProtocolsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutProtocolsList");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutProtocolsListResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutProtocolsListResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutProtocolsListResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutProtocolsListRequest, PutProtocolsListResponse>()
                            .withOperationName("PutProtocolsList")
                            .withMarshaller(new PutProtocolsListRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putProtocolsListRequest));
            CompletableFuture<PutProtocolsListResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates the resource set.
     * </p>
     * <p>
     * An Firewall Manager resource set defines the resources to import into an Firewall Manager policy from another
     * Amazon Web Services service.
     * </p>
     *
     * @param putResourceSetRequest
     * @return A Java Future containing the result of the PutResourceSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.PutResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutResourceSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutResourceSetResponse> putResourceSet(PutResourceSetRequest putResourceSetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourceSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourceSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResourceSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResourceSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResourceSetRequest, PutResourceSetResponse>()
                            .withOperationName("PutResourceSet")
                            .withMarshaller(new PutResourceSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putResourceSetRequest));
            CompletableFuture<PutResourceSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to an Amazon Web Services resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>LimitExceededException The operation exceeds a resource limit, for example, the maximum number of
     *         <code>policy</code> objects that you can create for an Amazon Web Services account. For more information,
     *         see <a href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager
     *         Limits</a> in the <i>WAF Developer Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from an Amazon Web Services resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidOperationException The operation failed because there was nothing to do or the operation
     *         wasn't possible. For example, you might have submitted an <code>AssociateAdminAccount</code> request for
     *         an account ID that was already set as the Firewall Manager administrator. Or you might have tried to
     *         access a Region that's disabled by default, and that you need to enable for the Firewall Manager
     *         administrator account and for Organizations before you can access it.</li>
     *         <li>InternalErrorException The operation failed because of a system problem, even though the request was
     *         valid. Retry your request.</li>
     *         <li>InvalidInputException The parameters of the request were invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FmsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FmsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final FmsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FmsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOperationException")
                                .exceptionBuilderSupplier(InvalidOperationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTypeException")
                                .exceptionBuilderSupplier(InvalidTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        FmsServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = FmsServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
