/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fms;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.fms.internal.FmsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.fms.model.AssociateAdminAccountRequest;
import software.amazon.awssdk.services.fms.model.AssociateAdminAccountResponse;
import software.amazon.awssdk.services.fms.model.AssociateThirdPartyFirewallRequest;
import software.amazon.awssdk.services.fms.model.AssociateThirdPartyFirewallResponse;
import software.amazon.awssdk.services.fms.model.BatchAssociateResourceRequest;
import software.amazon.awssdk.services.fms.model.BatchAssociateResourceResponse;
import software.amazon.awssdk.services.fms.model.BatchDisassociateResourceRequest;
import software.amazon.awssdk.services.fms.model.BatchDisassociateResourceResponse;
import software.amazon.awssdk.services.fms.model.DeleteAppsListRequest;
import software.amazon.awssdk.services.fms.model.DeleteAppsListResponse;
import software.amazon.awssdk.services.fms.model.DeleteNotificationChannelRequest;
import software.amazon.awssdk.services.fms.model.DeleteNotificationChannelResponse;
import software.amazon.awssdk.services.fms.model.DeletePolicyRequest;
import software.amazon.awssdk.services.fms.model.DeletePolicyResponse;
import software.amazon.awssdk.services.fms.model.DeleteProtocolsListRequest;
import software.amazon.awssdk.services.fms.model.DeleteProtocolsListResponse;
import software.amazon.awssdk.services.fms.model.DeleteResourceSetRequest;
import software.amazon.awssdk.services.fms.model.DeleteResourceSetResponse;
import software.amazon.awssdk.services.fms.model.DisassociateAdminAccountRequest;
import software.amazon.awssdk.services.fms.model.DisassociateAdminAccountResponse;
import software.amazon.awssdk.services.fms.model.DisassociateThirdPartyFirewallRequest;
import software.amazon.awssdk.services.fms.model.DisassociateThirdPartyFirewallResponse;
import software.amazon.awssdk.services.fms.model.FmsException;
import software.amazon.awssdk.services.fms.model.GetAdminAccountRequest;
import software.amazon.awssdk.services.fms.model.GetAdminAccountResponse;
import software.amazon.awssdk.services.fms.model.GetAdminScopeRequest;
import software.amazon.awssdk.services.fms.model.GetAdminScopeResponse;
import software.amazon.awssdk.services.fms.model.GetAppsListRequest;
import software.amazon.awssdk.services.fms.model.GetAppsListResponse;
import software.amazon.awssdk.services.fms.model.GetComplianceDetailRequest;
import software.amazon.awssdk.services.fms.model.GetComplianceDetailResponse;
import software.amazon.awssdk.services.fms.model.GetNotificationChannelRequest;
import software.amazon.awssdk.services.fms.model.GetNotificationChannelResponse;
import software.amazon.awssdk.services.fms.model.GetPolicyRequest;
import software.amazon.awssdk.services.fms.model.GetPolicyResponse;
import software.amazon.awssdk.services.fms.model.GetProtectionStatusRequest;
import software.amazon.awssdk.services.fms.model.GetProtectionStatusResponse;
import software.amazon.awssdk.services.fms.model.GetProtocolsListRequest;
import software.amazon.awssdk.services.fms.model.GetProtocolsListResponse;
import software.amazon.awssdk.services.fms.model.GetResourceSetRequest;
import software.amazon.awssdk.services.fms.model.GetResourceSetResponse;
import software.amazon.awssdk.services.fms.model.GetThirdPartyFirewallAssociationStatusRequest;
import software.amazon.awssdk.services.fms.model.GetThirdPartyFirewallAssociationStatusResponse;
import software.amazon.awssdk.services.fms.model.GetViolationDetailsRequest;
import software.amazon.awssdk.services.fms.model.GetViolationDetailsResponse;
import software.amazon.awssdk.services.fms.model.InternalErrorException;
import software.amazon.awssdk.services.fms.model.InvalidInputException;
import software.amazon.awssdk.services.fms.model.InvalidOperationException;
import software.amazon.awssdk.services.fms.model.InvalidTypeException;
import software.amazon.awssdk.services.fms.model.LimitExceededException;
import software.amazon.awssdk.services.fms.model.ListAdminAccountsForOrganizationRequest;
import software.amazon.awssdk.services.fms.model.ListAdminAccountsForOrganizationResponse;
import software.amazon.awssdk.services.fms.model.ListAdminsManagingAccountRequest;
import software.amazon.awssdk.services.fms.model.ListAdminsManagingAccountResponse;
import software.amazon.awssdk.services.fms.model.ListAppsListsRequest;
import software.amazon.awssdk.services.fms.model.ListAppsListsResponse;
import software.amazon.awssdk.services.fms.model.ListComplianceStatusRequest;
import software.amazon.awssdk.services.fms.model.ListComplianceStatusResponse;
import software.amazon.awssdk.services.fms.model.ListDiscoveredResourcesRequest;
import software.amazon.awssdk.services.fms.model.ListDiscoveredResourcesResponse;
import software.amazon.awssdk.services.fms.model.ListMemberAccountsRequest;
import software.amazon.awssdk.services.fms.model.ListMemberAccountsResponse;
import software.amazon.awssdk.services.fms.model.ListPoliciesRequest;
import software.amazon.awssdk.services.fms.model.ListPoliciesResponse;
import software.amazon.awssdk.services.fms.model.ListProtocolsListsRequest;
import software.amazon.awssdk.services.fms.model.ListProtocolsListsResponse;
import software.amazon.awssdk.services.fms.model.ListResourceSetResourcesRequest;
import software.amazon.awssdk.services.fms.model.ListResourceSetResourcesResponse;
import software.amazon.awssdk.services.fms.model.ListResourceSetsRequest;
import software.amazon.awssdk.services.fms.model.ListResourceSetsResponse;
import software.amazon.awssdk.services.fms.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.fms.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.fms.model.ListThirdPartyFirewallFirewallPoliciesRequest;
import software.amazon.awssdk.services.fms.model.ListThirdPartyFirewallFirewallPoliciesResponse;
import software.amazon.awssdk.services.fms.model.PutAdminAccountRequest;
import software.amazon.awssdk.services.fms.model.PutAdminAccountResponse;
import software.amazon.awssdk.services.fms.model.PutAppsListRequest;
import software.amazon.awssdk.services.fms.model.PutAppsListResponse;
import software.amazon.awssdk.services.fms.model.PutNotificationChannelRequest;
import software.amazon.awssdk.services.fms.model.PutNotificationChannelResponse;
import software.amazon.awssdk.services.fms.model.PutPolicyRequest;
import software.amazon.awssdk.services.fms.model.PutPolicyResponse;
import software.amazon.awssdk.services.fms.model.PutProtocolsListRequest;
import software.amazon.awssdk.services.fms.model.PutProtocolsListResponse;
import software.amazon.awssdk.services.fms.model.PutResourceSetRequest;
import software.amazon.awssdk.services.fms.model.PutResourceSetResponse;
import software.amazon.awssdk.services.fms.model.ResourceNotFoundException;
import software.amazon.awssdk.services.fms.model.TagResourceRequest;
import software.amazon.awssdk.services.fms.model.TagResourceResponse;
import software.amazon.awssdk.services.fms.model.UntagResourceRequest;
import software.amazon.awssdk.services.fms.model.UntagResourceResponse;
import software.amazon.awssdk.services.fms.transform.AssociateAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.AssociateThirdPartyFirewallRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.BatchAssociateResourceRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.BatchDisassociateResourceRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeleteAppsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeleteNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeletePolicyRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeleteProtocolsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DeleteResourceSetRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DisassociateAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.DisassociateThirdPartyFirewallRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetAdminScopeRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetAppsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetComplianceDetailRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetPolicyRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetProtectionStatusRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetProtocolsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetResourceSetRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetThirdPartyFirewallAssociationStatusRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.GetViolationDetailsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListAdminAccountsForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListAdminsManagingAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListAppsListsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListComplianceStatusRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListDiscoveredResourcesRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListMemberAccountsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListPoliciesRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListProtocolsListsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListResourceSetResourcesRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListResourceSetsRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.ListThirdPartyFirewallFirewallPoliciesRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutAppsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutPolicyRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutProtocolsListRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.PutResourceSetRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.fms.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link FmsClient}.
 *
 * @see FmsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFmsClient implements FmsClient {
    private static final Logger log = Logger.loggerFor(DefaultFmsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultFmsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Sets a Firewall Manager default administrator account. The Firewall Manager default administrator account can
     * manage third-party firewalls and has full administrative scope that allows administration of all policy types,
     * accounts, organizational units, and Regions. This account must be a member account of the organization in
     * Organizations whose resources you want to protect.
     * </p>
     * <p>
     * For information about working with Firewall Manager administrator accounts, see <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/fms-administrators.html">Managing Firewall
     * Manager administrators</a> in the <i>Firewall Manager Developer Guide</i>.
     * </p>
     *
     * @param associateAdminAccountRequest
     * @return Result of the AssociateAdminAccount operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.AssociateAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/AssociateAdminAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AssociateAdminAccountResponse associateAdminAccount(AssociateAdminAccountRequest associateAdminAccountRequest)
            throws InvalidOperationException, InvalidInputException, ResourceNotFoundException, InternalErrorException,
            LimitExceededException, AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateAdminAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateAdminAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateAdminAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateAdminAccount");

            return clientHandler.execute(new ClientExecutionParams<AssociateAdminAccountRequest, AssociateAdminAccountResponse>()
                    .withOperationName("AssociateAdminAccount").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(associateAdminAccountRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateAdminAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sets the Firewall Manager policy administrator as a tenant administrator of a third-party firewall service. A
     * tenant is an instance of the third-party firewall service that's associated with your Amazon Web Services
     * customer account.
     * </p>
     *
     * @param associateThirdPartyFirewallRequest
     * @return Result of the AssociateThirdPartyFirewall operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.AssociateThirdPartyFirewall
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/AssociateThirdPartyFirewall"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateThirdPartyFirewallResponse associateThirdPartyFirewall(
            AssociateThirdPartyFirewallRequest associateThirdPartyFirewallRequest) throws InvalidOperationException,
            InvalidInputException, ResourceNotFoundException, InternalErrorException, AwsServiceException, SdkClientException,
            FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateThirdPartyFirewallResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateThirdPartyFirewallResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateThirdPartyFirewallRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateThirdPartyFirewallRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateThirdPartyFirewall");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateThirdPartyFirewallRequest, AssociateThirdPartyFirewallResponse>()
                            .withOperationName("AssociateThirdPartyFirewall").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(associateThirdPartyFirewallRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateThirdPartyFirewallRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associate resources to a Firewall Manager resource set.
     * </p>
     *
     * @param batchAssociateResourceRequest
     * @return Result of the BatchAssociateResource operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.BatchAssociateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/BatchAssociateResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchAssociateResourceResponse batchAssociateResource(BatchAssociateResourceRequest batchAssociateResourceRequest)
            throws InvalidOperationException, InternalErrorException, InvalidInputException, LimitExceededException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchAssociateResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchAssociateResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchAssociateResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchAssociateResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchAssociateResource");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchAssociateResourceRequest, BatchAssociateResourceResponse>()
                            .withOperationName("BatchAssociateResource").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(batchAssociateResourceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchAssociateResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates resources from a Firewall Manager resource set.
     * </p>
     *
     * @param batchDisassociateResourceRequest
     * @return Result of the BatchDisassociateResource operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.BatchDisassociateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/BatchDisassociateResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchDisassociateResourceResponse batchDisassociateResource(
            BatchDisassociateResourceRequest batchDisassociateResourceRequest) throws InvalidOperationException,
            InternalErrorException, InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchDisassociateResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchDisassociateResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDisassociateResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDisassociateResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDisassociateResource");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchDisassociateResourceRequest, BatchDisassociateResourceResponse>()
                            .withOperationName("BatchDisassociateResource").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(batchDisassociateResourceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchDisassociateResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Permanently deletes an Firewall Manager applications list.
     * </p>
     *
     * @param deleteAppsListRequest
     * @return Result of the DeleteAppsList operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.DeleteAppsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeleteAppsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAppsListResponse deleteAppsList(DeleteAppsListRequest deleteAppsListRequest) throws ResourceNotFoundException,
            InvalidOperationException, InternalErrorException, AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAppsListResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAppsListResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppsListRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppsList");

            return clientHandler.execute(new ClientExecutionParams<DeleteAppsListRequest, DeleteAppsListResponse>()
                    .withOperationName("DeleteAppsList").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteAppsListRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAppsListRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Firewall Manager association with the IAM role and the Amazon Simple Notification Service (SNS) topic
     * that is used to record Firewall Manager SNS logs.
     * </p>
     *
     * @param deleteNotificationChannelRequest
     * @return Result of the DeleteNotificationChannel operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.DeleteNotificationChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeleteNotificationChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteNotificationChannelResponse deleteNotificationChannel(
            DeleteNotificationChannelRequest deleteNotificationChannelRequest) throws ResourceNotFoundException,
            InvalidOperationException, InternalErrorException, AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteNotificationChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteNotificationChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNotificationChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteNotificationChannelRequest, DeleteNotificationChannelResponse>()
                            .withOperationName("DeleteNotificationChannel").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteNotificationChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteNotificationChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Permanently deletes an Firewall Manager policy.
     * </p>
     *
     * @param deletePolicyRequest
     * @return Result of the DeletePolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.DeletePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeletePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePolicyResponse deletePolicy(DeletePolicyRequest deletePolicyRequest) throws ResourceNotFoundException,
            InvalidOperationException, InternalErrorException, InvalidInputException, LimitExceededException,
            AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePolicy");

            return clientHandler.execute(new ClientExecutionParams<DeletePolicyRequest, DeletePolicyResponse>()
                    .withOperationName("DeletePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deletePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Permanently deletes an Firewall Manager protocols list.
     * </p>
     *
     * @param deleteProtocolsListRequest
     * @return Result of the DeleteProtocolsList operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.DeleteProtocolsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeleteProtocolsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteProtocolsListResponse deleteProtocolsList(DeleteProtocolsListRequest deleteProtocolsListRequest)
            throws ResourceNotFoundException, InvalidOperationException, InternalErrorException, AwsServiceException,
            SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProtocolsListResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteProtocolsListResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProtocolsListRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProtocolsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProtocolsList");

            return clientHandler.execute(new ClientExecutionParams<DeleteProtocolsListRequest, DeleteProtocolsListResponse>()
                    .withOperationName("DeleteProtocolsList").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteProtocolsListRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteProtocolsListRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified <a>ResourceSet</a>.
     * </p>
     *
     * @param deleteResourceSetRequest
     * @return Result of the DeleteResourceSet operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.DeleteResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DeleteResourceSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteResourceSetResponse deleteResourceSet(DeleteResourceSetRequest deleteResourceSetRequest)
            throws ResourceNotFoundException, InvalidOperationException, InternalErrorException, InvalidInputException,
            AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteResourceSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourceSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceSet");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourceSetRequest, DeleteResourceSetResponse>()
                    .withOperationName("DeleteResourceSet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteResourceSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourceSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates an Firewall Manager administrator account. To set a different account as an Firewall Manager
     * administrator, submit a <a>PutAdminAccount</a> request. To set an account as a default administrator account, you
     * must submit an <a>AssociateAdminAccount</a> request.
     * </p>
     * <p>
     * Disassociation of the default administrator account follows the first in, last out principle. If you are the
     * default administrator, all Firewall Manager administrators within the organization must first disassociate their
     * accounts before you can disassociate your account.
     * </p>
     *
     * @param disassociateAdminAccountRequest
     * @return Result of the DisassociateAdminAccount operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.DisassociateAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DisassociateAdminAccount" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisassociateAdminAccountResponse disassociateAdminAccount(
            DisassociateAdminAccountRequest disassociateAdminAccountRequest) throws InvalidOperationException,
            ResourceNotFoundException, InternalErrorException, AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateAdminAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateAdminAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateAdminAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateAdminAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateAdminAccountRequest, DisassociateAdminAccountResponse>()
                            .withOperationName("DisassociateAdminAccount").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(disassociateAdminAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateAdminAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates a Firewall Manager policy administrator from a third-party firewall tenant. When you call
     * <code>DisassociateThirdPartyFirewall</code>, the third-party firewall vendor deletes all of the firewalls that
     * are associated with the account.
     * </p>
     *
     * @param disassociateThirdPartyFirewallRequest
     * @return Result of the DisassociateThirdPartyFirewall operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.DisassociateThirdPartyFirewall
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/DisassociateThirdPartyFirewall"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateThirdPartyFirewallResponse disassociateThirdPartyFirewall(
            DisassociateThirdPartyFirewallRequest disassociateThirdPartyFirewallRequest) throws InvalidOperationException,
            InvalidInputException, ResourceNotFoundException, InternalErrorException, AwsServiceException, SdkClientException,
            FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateThirdPartyFirewallResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateThirdPartyFirewallResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateThirdPartyFirewallRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateThirdPartyFirewallRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateThirdPartyFirewall");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateThirdPartyFirewallRequest, DisassociateThirdPartyFirewallResponse>()
                            .withOperationName("DisassociateThirdPartyFirewall").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(disassociateThirdPartyFirewallRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateThirdPartyFirewallRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the Organizations account that is associated with Firewall Manager as the Firewall Manager default
     * administrator.
     * </p>
     *
     * @param getAdminAccountRequest
     * @return Result of the GetAdminAccount operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.GetAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetAdminAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAdminAccountResponse getAdminAccount(GetAdminAccountRequest getAdminAccountRequest)
            throws InvalidOperationException, ResourceNotFoundException, InternalErrorException, AwsServiceException,
            SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAdminAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAdminAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAdminAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAdminAccount");

            return clientHandler.execute(new ClientExecutionParams<GetAdminAccountRequest, GetAdminAccountResponse>()
                    .withOperationName("GetAdminAccount").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAdminAccountRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAdminAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the specified account's administrative scope. The admistrative scope defines the
     * resources that an Firewall Manager administrator can manage.
     * </p>
     *
     * @param getAdminScopeRequest
     * @return Result of the GetAdminScope operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.GetAdminScope
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetAdminScope" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAdminScopeResponse getAdminScope(GetAdminScopeRequest getAdminScopeRequest) throws InvalidOperationException,
            InvalidInputException, ResourceNotFoundException, InternalErrorException, LimitExceededException,
            AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAdminScopeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAdminScopeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAdminScopeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAdminScopeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAdminScope");

            return clientHandler.execute(new ClientExecutionParams<GetAdminScopeRequest, GetAdminScopeResponse>()
                    .withOperationName("GetAdminScope").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAdminScopeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAdminScopeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the specified Firewall Manager applications list.
     * </p>
     *
     * @param getAppsListRequest
     * @return Result of the GetAppsList operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.GetAppsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetAppsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAppsListResponse getAppsList(GetAppsListRequest getAppsListRequest) throws ResourceNotFoundException,
            InvalidOperationException, InternalErrorException, AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAppsListResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAppsListResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAppsListRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppsList");

            return clientHandler.execute(new ClientExecutionParams<GetAppsListRequest, GetAppsListResponse>()
                    .withOperationName("GetAppsList").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getAppsListRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAppsListRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns detailed compliance information about the specified member account. Details include resources that are in
     * and out of compliance with the specified policy.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Resources are considered noncompliant for WAF and Shield Advanced policies if the specified policy has not been
     * applied to them.
     * </p>
     * </li>
     * <li>
     * <p>
     * Resources are considered noncompliant for security group policies if they are in scope of the policy, they
     * violate one or more of the policy rules, and remediation is disabled or not possible.
     * </p>
     * </li>
     * <li>
     * <p>
     * Resources are considered noncompliant for Network Firewall policies if a firewall is missing in the VPC, if the
     * firewall endpoint isn't set up in an expected Availability Zone and subnet, if a subnet created by the Firewall
     * Manager doesn't have the expected route table, and for modifications to a firewall policy that violate the
     * Firewall Manager policy's rules.
     * </p>
     * </li>
     * <li>
     * <p>
     * Resources are considered noncompliant for DNS Firewall policies if a DNS Firewall rule group is missing from the
     * rule group associations for the VPC.
     * </p>
     * </li>
     * </ul>
     *
     * @param getComplianceDetailRequest
     * @return Result of the GetComplianceDetail operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.GetComplianceDetail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetComplianceDetail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetComplianceDetailResponse getComplianceDetail(GetComplianceDetailRequest getComplianceDetailRequest)
            throws ResourceNotFoundException, InternalErrorException, InvalidInputException, InvalidOperationException,
            AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetComplianceDetailResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetComplianceDetailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getComplianceDetailRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComplianceDetailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComplianceDetail");

            return clientHandler.execute(new ClientExecutionParams<GetComplianceDetailRequest, GetComplianceDetailResponse>()
                    .withOperationName("GetComplianceDetail").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getComplianceDetailRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetComplianceDetailRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Information about the Amazon Simple Notification Service (SNS) topic that is used to record Firewall Manager SNS
     * logs.
     * </p>
     *
     * @param getNotificationChannelRequest
     * @return Result of the GetNotificationChannel operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.GetNotificationChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetNotificationChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetNotificationChannelResponse getNotificationChannel(GetNotificationChannelRequest getNotificationChannelRequest)
            throws ResourceNotFoundException, InvalidOperationException, InternalErrorException, AwsServiceException,
            SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetNotificationChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getNotificationChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNotificationChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<GetNotificationChannelRequest, GetNotificationChannelResponse>()
                            .withOperationName("GetNotificationChannel").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getNotificationChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetNotificationChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the specified Firewall Manager policy.
     * </p>
     *
     * @param getPolicyRequest
     * @return Result of the GetPolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws InvalidTypeException
     *         The value of the <code>Type</code> parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.GetPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPolicyResponse getPolicy(GetPolicyRequest getPolicyRequest) throws ResourceNotFoundException,
            InvalidOperationException, InternalErrorException, InvalidTypeException, AwsServiceException, SdkClientException,
            FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetPolicyRequest, GetPolicyResponse>()
                    .withOperationName("GetPolicy").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getPolicyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * If you created a Shield Advanced policy, returns policy-level attack summary information in the event of a
     * potential DDoS attack. Other policy types are currently unsupported.
     * </p>
     *
     * @param getProtectionStatusRequest
     * @return Result of the GetProtectionStatus operation returned by the service.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.GetProtectionStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetProtectionStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetProtectionStatusResponse getProtectionStatus(GetProtectionStatusRequest getProtectionStatusRequest)
            throws InvalidInputException, ResourceNotFoundException, InternalErrorException, AwsServiceException,
            SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetProtectionStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetProtectionStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProtectionStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProtectionStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProtectionStatus");

            return clientHandler.execute(new ClientExecutionParams<GetProtectionStatusRequest, GetProtectionStatusResponse>()
                    .withOperationName("GetProtectionStatus").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getProtectionStatusRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetProtectionStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the specified Firewall Manager protocols list.
     * </p>
     *
     * @param getProtocolsListRequest
     * @return Result of the GetProtocolsList operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.GetProtocolsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetProtocolsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetProtocolsListResponse getProtocolsList(GetProtocolsListRequest getProtocolsListRequest)
            throws ResourceNotFoundException, InvalidOperationException, InternalErrorException, AwsServiceException,
            SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetProtocolsListResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetProtocolsListResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProtocolsListRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProtocolsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProtocolsList");

            return clientHandler.execute(new ClientExecutionParams<GetProtocolsListRequest, GetProtocolsListResponse>()
                    .withOperationName("GetProtocolsList").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getProtocolsListRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetProtocolsListRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a specific resource set.
     * </p>
     *
     * @param getResourceSetRequest
     * @return Result of the GetResourceSet operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.GetResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetResourceSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourceSetResponse getResourceSet(GetResourceSetRequest getResourceSetRequest) throws InvalidOperationException,
            InvalidInputException, ResourceNotFoundException, InternalErrorException, AwsServiceException, SdkClientException,
            FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourceSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceSet");

            return clientHandler.execute(new ClientExecutionParams<GetResourceSetRequest, GetResourceSetResponse>()
                    .withOperationName("GetResourceSet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourceSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The onboarding status of a Firewall Manager admin account to third-party firewall vendor tenant.
     * </p>
     *
     * @param getThirdPartyFirewallAssociationStatusRequest
     * @return Result of the GetThirdPartyFirewallAssociationStatus operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.GetThirdPartyFirewallAssociationStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetThirdPartyFirewallAssociationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetThirdPartyFirewallAssociationStatusResponse getThirdPartyFirewallAssociationStatus(
            GetThirdPartyFirewallAssociationStatusRequest getThirdPartyFirewallAssociationStatusRequest)
            throws InvalidOperationException, InvalidInputException, ResourceNotFoundException, InternalErrorException,
            AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetThirdPartyFirewallAssociationStatusResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetThirdPartyFirewallAssociationStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getThirdPartyFirewallAssociationStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getThirdPartyFirewallAssociationStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetThirdPartyFirewallAssociationStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetThirdPartyFirewallAssociationStatusRequest, GetThirdPartyFirewallAssociationStatusResponse>()
                            .withOperationName("GetThirdPartyFirewallAssociationStatus").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(getThirdPartyFirewallAssociationStatusRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetThirdPartyFirewallAssociationStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves violations for a resource based on the specified Firewall Manager policy and Amazon Web Services
     * account.
     * </p>
     *
     * @param getViolationDetailsRequest
     * @return Result of the GetViolationDetails operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.GetViolationDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/GetViolationDetails" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetViolationDetailsResponse getViolationDetails(GetViolationDetailsRequest getViolationDetailsRequest)
            throws ResourceNotFoundException, InvalidInputException, InternalErrorException, AwsServiceException,
            SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetViolationDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetViolationDetailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getViolationDetailsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getViolationDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetViolationDetails");

            return clientHandler.execute(new ClientExecutionParams<GetViolationDetailsRequest, GetViolationDetailsResponse>()
                    .withOperationName("GetViolationDetails").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getViolationDetailsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetViolationDetailsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a <code>AdminAccounts</code> object that lists the Firewall Manager administrators within the
     * organization that are onboarded to Firewall Manager by <a>AssociateAdminAccount</a>.
     * </p>
     * <p>
     * This operation can be called only from the organization's management account.
     * </p>
     *
     * @param listAdminAccountsForOrganizationRequest
     * @return Result of the ListAdminAccountsForOrganization operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListAdminAccountsForOrganization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListAdminAccountsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAdminAccountsForOrganizationResponse listAdminAccountsForOrganization(
            ListAdminAccountsForOrganizationRequest listAdminAccountsForOrganizationRequest) throws InvalidOperationException,
            ResourceNotFoundException, InternalErrorException, LimitExceededException, AwsServiceException, SdkClientException,
            FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAdminAccountsForOrganizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAdminAccountsForOrganizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAdminAccountsForOrganizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAdminAccountsForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAdminAccountsForOrganization");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAdminAccountsForOrganizationRequest, ListAdminAccountsForOrganizationResponse>()
                            .withOperationName("ListAdminAccountsForOrganization").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listAdminAccountsForOrganizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAdminAccountsForOrganizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the accounts that are managing the specified Organizations member account. This is useful for any member
     * account so that they can view the accounts who are managing their account. This operation only returns the
     * managing administrators that have the requested account within their <a>AdminScope</a>.
     * </p>
     *
     * @param listAdminsManagingAccountRequest
     * @return Result of the ListAdminsManagingAccount operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListAdminsManagingAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListAdminsManagingAccount" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListAdminsManagingAccountResponse listAdminsManagingAccount(
            ListAdminsManagingAccountRequest listAdminsManagingAccountRequest) throws ResourceNotFoundException,
            InvalidInputException, InternalErrorException, AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAdminsManagingAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAdminsManagingAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAdminsManagingAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAdminsManagingAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAdminsManagingAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAdminsManagingAccountRequest, ListAdminsManagingAccountResponse>()
                            .withOperationName("ListAdminsManagingAccount").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listAdminsManagingAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAdminsManagingAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>AppsListDataSummary</code> objects.
     * </p>
     *
     * @param listAppsListsRequest
     * @return Result of the ListAppsLists operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListAppsLists
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListAppsLists" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAppsListsResponse listAppsLists(ListAppsListsRequest listAppsListsRequest) throws ResourceNotFoundException,
            InvalidOperationException, LimitExceededException, InternalErrorException, AwsServiceException, SdkClientException,
            FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAppsListsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAppsListsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppsListsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppsListsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppsLists");

            return clientHandler.execute(new ClientExecutionParams<ListAppsListsRequest, ListAppsListsResponse>()
                    .withOperationName("ListAppsLists").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAppsListsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAppsListsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>PolicyComplianceStatus</code> objects. Use <code>PolicyComplianceStatus</code> to get a
     * summary of which member accounts are protected by the specified policy.
     * </p>
     *
     * @param listComplianceStatusRequest
     * @return Result of the ListComplianceStatus operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListComplianceStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListComplianceStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListComplianceStatusResponse listComplianceStatus(ListComplianceStatusRequest listComplianceStatusRequest)
            throws ResourceNotFoundException, InternalErrorException, AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListComplianceStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListComplianceStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listComplianceStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComplianceStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComplianceStatus");

            return clientHandler.execute(new ClientExecutionParams<ListComplianceStatusRequest, ListComplianceStatusResponse>()
                    .withOperationName("ListComplianceStatus").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listComplianceStatusRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListComplianceStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of resources in the organization's accounts that are available to be associated with a resource
     * set.
     * </p>
     *
     * @param listDiscoveredResourcesRequest
     * @return Result of the ListDiscoveredResources operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListDiscoveredResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListDiscoveredResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDiscoveredResourcesResponse listDiscoveredResources(ListDiscoveredResourcesRequest listDiscoveredResourcesRequest)
            throws InvalidOperationException, InvalidInputException, InternalErrorException, AwsServiceException,
            SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDiscoveredResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDiscoveredResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDiscoveredResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDiscoveredResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDiscoveredResources");

            return clientHandler
                    .execute(new ClientExecutionParams<ListDiscoveredResourcesRequest, ListDiscoveredResourcesResponse>()
                            .withOperationName("ListDiscoveredResources").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listDiscoveredResourcesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListDiscoveredResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a <code>MemberAccounts</code> object that lists the member accounts in the administrator's Amazon Web
     * Services organization.
     * </p>
     * <p>
     * Either an Firewall Manager administrator or the organization's management account can make this request.
     * </p>
     *
     * @param listMemberAccountsRequest
     * @return Result of the ListMemberAccounts operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListMemberAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMemberAccountsResponse listMemberAccounts(ListMemberAccountsRequest listMemberAccountsRequest)
            throws ResourceNotFoundException, InternalErrorException, AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMemberAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMemberAccountsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMemberAccountsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMemberAccountsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMemberAccounts");

            return clientHandler.execute(new ClientExecutionParams<ListMemberAccountsRequest, ListMemberAccountsResponse>()
                    .withOperationName("ListMemberAccounts").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listMemberAccountsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMemberAccountsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>PolicySummary</code> objects.
     * </p>
     *
     * @param listPoliciesRequest
     * @return Result of the ListPolicies operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListPolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPoliciesResponse listPolicies(ListPoliciesRequest listPoliciesRequest) throws ResourceNotFoundException,
            InvalidOperationException, LimitExceededException, InternalErrorException, AwsServiceException, SdkClientException,
            FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPoliciesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPolicies");

            return clientHandler.execute(new ClientExecutionParams<ListPoliciesRequest, ListPoliciesResponse>()
                    .withOperationName("ListPolicies").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPoliciesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>ProtocolsListDataSummary</code> objects.
     * </p>
     *
     * @param listProtocolsListsRequest
     * @return Result of the ListProtocolsLists operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListProtocolsLists
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListProtocolsLists" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListProtocolsListsResponse listProtocolsLists(ListProtocolsListsRequest listProtocolsListsRequest)
            throws ResourceNotFoundException, InvalidOperationException, InternalErrorException, AwsServiceException,
            SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProtocolsListsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProtocolsListsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProtocolsListsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProtocolsListsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProtocolsLists");

            return clientHandler.execute(new ClientExecutionParams<ListProtocolsListsRequest, ListProtocolsListsResponse>()
                    .withOperationName("ListProtocolsLists").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listProtocolsListsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProtocolsListsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of resources that are currently associated to a resource set.
     * </p>
     *
     * @param listResourceSetResourcesRequest
     * @return Result of the ListResourceSetResources operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListResourceSetResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListResourceSetResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListResourceSetResourcesResponse listResourceSetResources(
            ListResourceSetResourcesRequest listResourceSetResourcesRequest) throws InvalidOperationException,
            InternalErrorException, InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourceSetResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResourceSetResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceSetResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceSetResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceSetResources");

            return clientHandler
                    .execute(new ClientExecutionParams<ListResourceSetResourcesRequest, ListResourceSetResourcesResponse>()
                            .withOperationName("ListResourceSetResources").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listResourceSetResourcesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListResourceSetResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of <code>ResourceSetSummary</code> objects.
     * </p>
     *
     * @param listResourceSetsRequest
     * @return Result of the ListResourceSets operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListResourceSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListResourceSets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListResourceSetsResponse listResourceSets(ListResourceSetsRequest listResourceSetsRequest)
            throws InvalidOperationException, InvalidInputException, InternalErrorException, AwsServiceException,
            SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourceSetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResourceSetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceSetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceSets");

            return clientHandler.execute(new ClientExecutionParams<ListResourceSetsRequest, ListResourceSetsResponse>()
                    .withOperationName("ListResourceSets").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listResourceSetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourceSetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the list of tags for the specified Amazon Web Services resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InvalidOperationException, InternalErrorException, InvalidInputException,
            AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all of the third-party firewall policies that are associated with the third-party firewall
     * administrator's account.
     * </p>
     *
     * @param listThirdPartyFirewallFirewallPoliciesRequest
     * @return Result of the ListThirdPartyFirewallFirewallPolicies operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.ListThirdPartyFirewallFirewallPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/ListThirdPartyFirewallFirewallPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListThirdPartyFirewallFirewallPoliciesResponse listThirdPartyFirewallFirewallPolicies(
            ListThirdPartyFirewallFirewallPoliciesRequest listThirdPartyFirewallFirewallPoliciesRequest)
            throws InvalidOperationException, InvalidInputException, ResourceNotFoundException, InternalErrorException,
            AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListThirdPartyFirewallFirewallPoliciesResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListThirdPartyFirewallFirewallPoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listThirdPartyFirewallFirewallPoliciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listThirdPartyFirewallFirewallPoliciesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListThirdPartyFirewallFirewallPolicies");

            return clientHandler
                    .execute(new ClientExecutionParams<ListThirdPartyFirewallFirewallPoliciesRequest, ListThirdPartyFirewallFirewallPoliciesResponse>()
                            .withOperationName("ListThirdPartyFirewallFirewallPolicies").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(listThirdPartyFirewallFirewallPoliciesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListThirdPartyFirewallFirewallPoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates an Firewall Manager administrator account. The account must be a member of the organization
     * that was onboarded to Firewall Manager by <a>AssociateAdminAccount</a>. Only the organization's management
     * account can create an Firewall Manager administrator account. When you create an Firewall Manager administrator
     * account, the service checks to see if the account is already a delegated administrator within Organizations. If
     * the account isn't a delegated administrator, Firewall Manager calls Organizations to delegate the account within
     * Organizations. For more information about administrator accounts within Organizations, see <a
     * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_accounts.html">Managing the Amazon
     * Web Services Accounts in Your Organization</a>.
     * </p>
     *
     * @param putAdminAccountRequest
     * @return Result of the PutAdminAccount operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.PutAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutAdminAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutAdminAccountResponse putAdminAccount(PutAdminAccountRequest putAdminAccountRequest)
            throws InvalidOperationException, InvalidInputException, InternalErrorException, LimitExceededException,
            AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutAdminAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAdminAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAdminAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAdminAccount");

            return clientHandler.execute(new ClientExecutionParams<PutAdminAccountRequest, PutAdminAccountResponse>()
                    .withOperationName("PutAdminAccount").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putAdminAccountRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAdminAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Firewall Manager applications list.
     * </p>
     *
     * @param putAppsListRequest
     * @return Result of the PutAppsList operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.PutAppsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutAppsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutAppsListResponse putAppsList(PutAppsListRequest putAppsListRequest) throws ResourceNotFoundException,
            InvalidOperationException, InvalidInputException, LimitExceededException, InternalErrorException,
            AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAppsListResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutAppsListResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAppsListRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAppsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAppsList");

            return clientHandler.execute(new ClientExecutionParams<PutAppsListRequest, PutAppsListResponse>()
                    .withOperationName("PutAppsList").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putAppsListRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAppsListRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Designates the IAM role and Amazon Simple Notification Service (SNS) topic that Firewall Manager uses to record
     * SNS logs.
     * </p>
     * <p>
     * To perform this action outside of the console, you must first configure the SNS topic's access policy to allow
     * the <code>SnsRoleName</code> to publish SNS logs. If the <code>SnsRoleName</code> provided is a role other than
     * the <code>AWSServiceRoleForFMS</code> service-linked role, this role must have a trust relationship configured to
     * allow the Firewall Manager service principal <code>fms.amazonaws.com</code> to assume this role. For information
     * about configuring an SNS access policy, see <a href=
     * "https://docs.aws.amazon.com/waf/latest/developerguide/fms-security_iam_service-with-iam.html#fms-security_iam_service-with-iam-roles-service"
     * >Service roles for Firewall Manager</a> in the <i>Firewall Manager Developer Guide</i>.
     * </p>
     *
     * @param putNotificationChannelRequest
     * @return Result of the PutNotificationChannel operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.PutNotificationChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutNotificationChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutNotificationChannelResponse putNotificationChannel(PutNotificationChannelRequest putNotificationChannelRequest)
            throws ResourceNotFoundException, InvalidOperationException, InternalErrorException, AwsServiceException,
            SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutNotificationChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putNotificationChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutNotificationChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<PutNotificationChannelRequest, PutNotificationChannelResponse>()
                            .withOperationName("PutNotificationChannel").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putNotificationChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutNotificationChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Firewall Manager policy.
     * </p>
     * <p>
     * A Firewall Manager policy is specific to the individual policy type. If you want to enforce multiple policy types
     * across accounts, you can create multiple policies. You can create more than one policy for each type.
     * </p>
     * <p>
     * If you add a new account to an organization that you created with Organizations, Firewall Manager automatically
     * applies the policy to the resources in that account that are within scope of the policy.
     * </p>
     * <p>
     * Firewall Manager provides the following types of policies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Shield Advanced policy</b> - This policy applies Shield Advanced protection to specified accounts and
     * resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Security Groups policy</b> - This type of policy gives you control over security groups that are in use
     * throughout your organization in Organizations and lets you enforce a baseline set of rules across your
     * organization.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Network Firewall policy</b> - This policy applies Network Firewall protection to your organization's VPCs.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DNS Firewall policy</b> - This policy applies Amazon Route 53 Resolver DNS Firewall protections to your
     * organization's VPCs.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Third-party firewall policy</b> - This policy applies third-party firewall protections. Third-party firewalls
     * are available by subscription through the Amazon Web Services Marketplace console at <a
     * href="https://aws.amazon.com/marketplace">Amazon Web Services Marketplace</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Palo Alto Networks Cloud NGFW policy</b> - This policy applies Palo Alto Networks Cloud Next Generation
     * Firewall (NGFW) protections and Palo Alto Networks Cloud NGFW rulestacks to your organization's VPCs.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Fortigate CNF policy</b> - This policy applies Fortigate Cloud Native Firewall (CNF) protections. Fortigate
     * CNF is a cloud-centered solution that blocks Zero-Day threats and secures cloud infrastructures with
     * industry-leading advanced threat prevention, smart web application firewalls (WAF), and API protection.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param putPolicyRequest
     * @return Result of the PutPolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws InvalidTypeException
     *         The value of the <code>Type</code> parameter is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.PutPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutPolicyResponse putPolicy(PutPolicyRequest putPolicyRequest) throws ResourceNotFoundException,
            InvalidOperationException, InvalidInputException, LimitExceededException, InternalErrorException,
            InvalidTypeException, AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putPolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutPolicyRequest, PutPolicyResponse>()
                    .withOperationName("PutPolicy").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putPolicyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Firewall Manager protocols list.
     * </p>
     *
     * @param putProtocolsListRequest
     * @return Result of the PutProtocolsList operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.PutProtocolsList
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutProtocolsList" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutProtocolsListResponse putProtocolsList(PutProtocolsListRequest putProtocolsListRequest)
            throws ResourceNotFoundException, InvalidOperationException, InvalidInputException, LimitExceededException,
            InternalErrorException, AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutProtocolsListResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutProtocolsListResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putProtocolsListRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putProtocolsListRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutProtocolsList");

            return clientHandler.execute(new ClientExecutionParams<PutProtocolsListRequest, PutProtocolsListResponse>()
                    .withOperationName("PutProtocolsList").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putProtocolsListRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutProtocolsListRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates the resource set.
     * </p>
     * <p>
     * An Firewall Manager resource set defines the resources to import into an Firewall Manager policy from another
     * Amazon Web Services service.
     * </p>
     *
     * @param putResourceSetRequest
     * @return Result of the PutResourceSet operation returned by the service.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.PutResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/PutResourceSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutResourceSetResponse putResourceSet(PutResourceSetRequest putResourceSetRequest) throws InvalidOperationException,
            InvalidInputException, LimitExceededException, InternalErrorException, AwsServiceException, SdkClientException,
            FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutResourceSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourceSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourceSet");

            return clientHandler.execute(new ClientExecutionParams<PutResourceSetRequest, PutResourceSetResponse>()
                    .withOperationName("PutResourceSet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putResourceSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutResourceSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more tags to an Amazon Web Services resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws LimitExceededException
     *         The operation exceeds a resource limit, for example, the maximum number of <code>policy</code> objects
     *         that you can create for an Amazon Web Services account. For more information, see <a
     *         href="https://docs.aws.amazon.com/waf/latest/developerguide/fms-limits.html">Firewall Manager Limits</a>
     *         in the <i>WAF Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InvalidOperationException, InternalErrorException, InvalidInputException, LimitExceededException,
            AwsServiceException, SdkClientException, FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from an Amazon Web Services resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidOperationException
     *         The operation failed because there was nothing to do or the operation wasn't possible. For example, you
     *         might have submitted an <code>AssociateAdminAccount</code> request for an account ID that was already set
     *         as the Firewall Manager administrator. Or you might have tried to access a Region that's disabled by
     *         default, and that you need to enable for the Firewall Manager administrator account and for Organizations
     *         before you can access it.
     * @throws InternalErrorException
     *         The operation failed because of a system problem, even though the request was valid. Retry your request.
     * @throws InvalidInputException
     *         The parameters of the request were invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FmsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FmsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fms-2018-01-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InvalidOperationException, InternalErrorException, InvalidInputException, AwsServiceException, SdkClientException,
            FmsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FMS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        FmsServiceClientConfigurationBuilder serviceConfigBuilder = new FmsServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FmsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOperationException")
                                .exceptionBuilderSupplier(InvalidOperationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTypeException")
                                .exceptionBuilderSupplier(InvalidTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build());
    }

    @Override
    public final FmsServiceClientConfiguration serviceClientConfiguration() {
        return new FmsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
