/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fms.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A rule group that Firewall Manager tried to associate with a VPC has the same priority as a rule group that's already
 * associated.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DnsRuleGroupPriorityConflictViolation implements SdkPojo, Serializable,
        ToCopyableBuilder<DnsRuleGroupPriorityConflictViolation.Builder, DnsRuleGroupPriorityConflictViolation> {
    private static final SdkField<String> VIOLATION_TARGET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ViolationTarget").getter(getter(DnsRuleGroupPriorityConflictViolation::violationTarget))
            .setter(setter(Builder::violationTarget))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ViolationTarget").build()).build();

    private static final SdkField<String> VIOLATION_TARGET_DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ViolationTargetDescription")
            .getter(getter(DnsRuleGroupPriorityConflictViolation::violationTargetDescription))
            .setter(setter(Builder::violationTargetDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ViolationTargetDescription").build())
            .build();

    private static final SdkField<Integer> CONFLICTING_PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ConflictingPriority").getter(getter(DnsRuleGroupPriorityConflictViolation::conflictingPriority))
            .setter(setter(Builder::conflictingPriority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConflictingPriority").build())
            .build();

    private static final SdkField<String> CONFLICTING_POLICY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConflictingPolicyId").getter(getter(DnsRuleGroupPriorityConflictViolation::conflictingPolicyId))
            .setter(setter(Builder::conflictingPolicyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConflictingPolicyId").build())
            .build();

    private static final SdkField<List<Integer>> UNAVAILABLE_PRIORITIES_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("UnavailablePriorities")
            .getter(getter(DnsRuleGroupPriorityConflictViolation::unavailablePriorities))
            .setter(setter(Builder::unavailablePriorities))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UnavailablePriorities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VIOLATION_TARGET_FIELD,
            VIOLATION_TARGET_DESCRIPTION_FIELD, CONFLICTING_PRIORITY_FIELD, CONFLICTING_POLICY_ID_FIELD,
            UNAVAILABLE_PRIORITIES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String violationTarget;

    private final String violationTargetDescription;

    private final Integer conflictingPriority;

    private final String conflictingPolicyId;

    private final List<Integer> unavailablePriorities;

    private DnsRuleGroupPriorityConflictViolation(BuilderImpl builder) {
        this.violationTarget = builder.violationTarget;
        this.violationTargetDescription = builder.violationTargetDescription;
        this.conflictingPriority = builder.conflictingPriority;
        this.conflictingPolicyId = builder.conflictingPolicyId;
        this.unavailablePriorities = builder.unavailablePriorities;
    }

    /**
     * <p>
     * Information about the VPC ID.
     * </p>
     * 
     * @return Information about the VPC ID.
     */
    public final String violationTarget() {
        return violationTarget;
    }

    /**
     * <p>
     * A description of the violation that specifies the VPC and the rule group that's already associated with it.
     * </p>
     * 
     * @return A description of the violation that specifies the VPC and the rule group that's already associated with
     *         it.
     */
    public final String violationTargetDescription() {
        return violationTargetDescription;
    }

    /**
     * <p>
     * The priority setting of the two conflicting rule groups.
     * </p>
     * 
     * @return The priority setting of the two conflicting rule groups.
     */
    public final Integer conflictingPriority() {
        return conflictingPriority;
    }

    /**
     * <p>
     * The ID of the Firewall Manager DNS Firewall policy that was already applied to the VPC. This policy contains the
     * rule group that's already associated with the VPC.
     * </p>
     * 
     * @return The ID of the Firewall Manager DNS Firewall policy that was already applied to the VPC. This policy
     *         contains the rule group that's already associated with the VPC.
     */
    public final String conflictingPolicyId() {
        return conflictingPolicyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the UnavailablePriorities property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasUnavailablePriorities() {
        return unavailablePriorities != null && !(unavailablePriorities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The priorities of rule groups that are already associated with the VPC. To retry your operation, choose priority
     * settings that aren't in this list for the rule groups in your new DNS Firewall policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUnavailablePriorities} method.
     * </p>
     * 
     * @return The priorities of rule groups that are already associated with the VPC. To retry your operation, choose
     *         priority settings that aren't in this list for the rule groups in your new DNS Firewall policy.
     */
    public final List<Integer> unavailablePriorities() {
        return unavailablePriorities;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(violationTarget());
        hashCode = 31 * hashCode + Objects.hashCode(violationTargetDescription());
        hashCode = 31 * hashCode + Objects.hashCode(conflictingPriority());
        hashCode = 31 * hashCode + Objects.hashCode(conflictingPolicyId());
        hashCode = 31 * hashCode + Objects.hashCode(hasUnavailablePriorities() ? unavailablePriorities() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DnsRuleGroupPriorityConflictViolation)) {
            return false;
        }
        DnsRuleGroupPriorityConflictViolation other = (DnsRuleGroupPriorityConflictViolation) obj;
        return Objects.equals(violationTarget(), other.violationTarget())
                && Objects.equals(violationTargetDescription(), other.violationTargetDescription())
                && Objects.equals(conflictingPriority(), other.conflictingPriority())
                && Objects.equals(conflictingPolicyId(), other.conflictingPolicyId())
                && hasUnavailablePriorities() == other.hasUnavailablePriorities()
                && Objects.equals(unavailablePriorities(), other.unavailablePriorities());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DnsRuleGroupPriorityConflictViolation").add("ViolationTarget", violationTarget())
                .add("ViolationTargetDescription", violationTargetDescription())
                .add("ConflictingPriority", conflictingPriority()).add("ConflictingPolicyId", conflictingPolicyId())
                .add("UnavailablePriorities", hasUnavailablePriorities() ? unavailablePriorities() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ViolationTarget":
            return Optional.ofNullable(clazz.cast(violationTarget()));
        case "ViolationTargetDescription":
            return Optional.ofNullable(clazz.cast(violationTargetDescription()));
        case "ConflictingPriority":
            return Optional.ofNullable(clazz.cast(conflictingPriority()));
        case "ConflictingPolicyId":
            return Optional.ofNullable(clazz.cast(conflictingPolicyId()));
        case "UnavailablePriorities":
            return Optional.ofNullable(clazz.cast(unavailablePriorities()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ViolationTarget", VIOLATION_TARGET_FIELD);
        map.put("ViolationTargetDescription", VIOLATION_TARGET_DESCRIPTION_FIELD);
        map.put("ConflictingPriority", CONFLICTING_PRIORITY_FIELD);
        map.put("ConflictingPolicyId", CONFLICTING_POLICY_ID_FIELD);
        map.put("UnavailablePriorities", UNAVAILABLE_PRIORITIES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DnsRuleGroupPriorityConflictViolation, T> g) {
        return obj -> g.apply((DnsRuleGroupPriorityConflictViolation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DnsRuleGroupPriorityConflictViolation> {
        /**
         * <p>
         * Information about the VPC ID.
         * </p>
         * 
         * @param violationTarget
         *        Information about the VPC ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder violationTarget(String violationTarget);

        /**
         * <p>
         * A description of the violation that specifies the VPC and the rule group that's already associated with it.
         * </p>
         * 
         * @param violationTargetDescription
         *        A description of the violation that specifies the VPC and the rule group that's already associated
         *        with it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder violationTargetDescription(String violationTargetDescription);

        /**
         * <p>
         * The priority setting of the two conflicting rule groups.
         * </p>
         * 
         * @param conflictingPriority
         *        The priority setting of the two conflicting rule groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conflictingPriority(Integer conflictingPriority);

        /**
         * <p>
         * The ID of the Firewall Manager DNS Firewall policy that was already applied to the VPC. This policy contains
         * the rule group that's already associated with the VPC.
         * </p>
         * 
         * @param conflictingPolicyId
         *        The ID of the Firewall Manager DNS Firewall policy that was already applied to the VPC. This policy
         *        contains the rule group that's already associated with the VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conflictingPolicyId(String conflictingPolicyId);

        /**
         * <p>
         * The priorities of rule groups that are already associated with the VPC. To retry your operation, choose
         * priority settings that aren't in this list for the rule groups in your new DNS Firewall policy.
         * </p>
         * 
         * @param unavailablePriorities
         *        The priorities of rule groups that are already associated with the VPC. To retry your operation,
         *        choose priority settings that aren't in this list for the rule groups in your new DNS Firewall policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unavailablePriorities(Collection<Integer> unavailablePriorities);

        /**
         * <p>
         * The priorities of rule groups that are already associated with the VPC. To retry your operation, choose
         * priority settings that aren't in this list for the rule groups in your new DNS Firewall policy.
         * </p>
         * 
         * @param unavailablePriorities
         *        The priorities of rule groups that are already associated with the VPC. To retry your operation,
         *        choose priority settings that aren't in this list for the rule groups in your new DNS Firewall policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unavailablePriorities(Integer... unavailablePriorities);
    }

    static final class BuilderImpl implements Builder {
        private String violationTarget;

        private String violationTargetDescription;

        private Integer conflictingPriority;

        private String conflictingPolicyId;

        private List<Integer> unavailablePriorities = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DnsRuleGroupPriorityConflictViolation model) {
            violationTarget(model.violationTarget);
            violationTargetDescription(model.violationTargetDescription);
            conflictingPriority(model.conflictingPriority);
            conflictingPolicyId(model.conflictingPolicyId);
            unavailablePriorities(model.unavailablePriorities);
        }

        public final String getViolationTarget() {
            return violationTarget;
        }

        public final void setViolationTarget(String violationTarget) {
            this.violationTarget = violationTarget;
        }

        @Override
        public final Builder violationTarget(String violationTarget) {
            this.violationTarget = violationTarget;
            return this;
        }

        public final String getViolationTargetDescription() {
            return violationTargetDescription;
        }

        public final void setViolationTargetDescription(String violationTargetDescription) {
            this.violationTargetDescription = violationTargetDescription;
        }

        @Override
        public final Builder violationTargetDescription(String violationTargetDescription) {
            this.violationTargetDescription = violationTargetDescription;
            return this;
        }

        public final Integer getConflictingPriority() {
            return conflictingPriority;
        }

        public final void setConflictingPriority(Integer conflictingPriority) {
            this.conflictingPriority = conflictingPriority;
        }

        @Override
        public final Builder conflictingPriority(Integer conflictingPriority) {
            this.conflictingPriority = conflictingPriority;
            return this;
        }

        public final String getConflictingPolicyId() {
            return conflictingPolicyId;
        }

        public final void setConflictingPolicyId(String conflictingPolicyId) {
            this.conflictingPolicyId = conflictingPolicyId;
        }

        @Override
        public final Builder conflictingPolicyId(String conflictingPolicyId) {
            this.conflictingPolicyId = conflictingPolicyId;
            return this;
        }

        public final Collection<Integer> getUnavailablePriorities() {
            if (unavailablePriorities instanceof SdkAutoConstructList) {
                return null;
            }
            return unavailablePriorities;
        }

        public final void setUnavailablePriorities(Collection<Integer> unavailablePriorities) {
            this.unavailablePriorities = DnsRuleGroupPrioritiesCopier.copy(unavailablePriorities);
        }

        @Override
        public final Builder unavailablePriorities(Collection<Integer> unavailablePriorities) {
            this.unavailablePriorities = DnsRuleGroupPrioritiesCopier.copy(unavailablePriorities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder unavailablePriorities(Integer... unavailablePriorities) {
            unavailablePriorities(Arrays.asList(unavailablePriorities));
            return this;
        }

        @Override
        public DnsRuleGroupPriorityConflictViolation build() {
            return new DnsRuleGroupPriorityConflictViolation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
