/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fms.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An Firewall Manager policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Policy implements SdkPojo, Serializable, ToCopyableBuilder<Policy.Builder, Policy> {
    private static final SdkField<String> POLICY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyId").getter(getter(Policy::policyId)).setter(setter(Builder::policyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyId").build()).build();

    private static final SdkField<String> POLICY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyName").getter(getter(Policy::policyName)).setter(setter(Builder::policyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyName").build()).build();

    private static final SdkField<String> POLICY_UPDATE_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyUpdateToken").getter(getter(Policy::policyUpdateToken)).setter(setter(Builder::policyUpdateToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyUpdateToken").build()).build();

    private static final SdkField<SecurityServicePolicyData> SECURITY_SERVICE_POLICY_DATA_FIELD = SdkField
            .<SecurityServicePolicyData> builder(MarshallingType.SDK_POJO).memberName("SecurityServicePolicyData")
            .getter(getter(Policy::securityServicePolicyData)).setter(setter(Builder::securityServicePolicyData))
            .constructor(SecurityServicePolicyData::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityServicePolicyData").build())
            .build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(Policy::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<List<String>> RESOURCE_TYPE_LIST_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceTypeList")
            .getter(getter(Policy::resourceTypeList))
            .setter(setter(Builder::resourceTypeList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTypeList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ResourceTag>> RESOURCE_TAGS_FIELD = SdkField
            .<List<ResourceTag>> builder(MarshallingType.LIST)
            .memberName("ResourceTags")
            .getter(getter(Policy::resourceTags))
            .setter(setter(Builder::resourceTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> EXCLUDE_RESOURCE_TAGS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ExcludeResourceTags").getter(getter(Policy::excludeResourceTags))
            .setter(setter(Builder::excludeResourceTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludeResourceTags").build())
            .build();

    private static final SdkField<Boolean> REMEDIATION_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RemediationEnabled").getter(getter(Policy::remediationEnabled))
            .setter(setter(Builder::remediationEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemediationEnabled").build())
            .build();

    private static final SdkField<Boolean> DELETE_UNUSED_FM_MANAGED_RESOURCES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeleteUnusedFMManagedResources")
            .getter(getter(Policy::deleteUnusedFMManagedResources))
            .setter(setter(Builder::deleteUnusedFMManagedResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteUnusedFMManagedResources")
                    .build()).build();

    private static final SdkField<Map<String, List<String>>> INCLUDE_MAP_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("IncludeMap")
            .getter(getter(Policy::includeMapAsStrings))
            .setter(setter(Builder::includeMapWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeMap").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<Map<String, List<String>>> EXCLUDE_MAP_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("ExcludeMap")
            .getter(getter(Policy::excludeMapAsStrings))
            .setter(setter(Builder::excludeMapWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludeMap").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<List<String>> RESOURCE_SET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceSetIds")
            .getter(getter(Policy::resourceSetIds))
            .setter(setter(Builder::resourceSetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceSetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> POLICY_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyDescription").getter(getter(Policy::policyDescription)).setter(setter(Builder::policyDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyDescription").build()).build();

    private static final SdkField<String> POLICY_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyStatus").getter(getter(Policy::policyStatusAsString)).setter(setter(Builder::policyStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyStatus").build()).build();

    private static final SdkField<String> RESOURCE_TAG_LOGICAL_OPERATOR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceTagLogicalOperator")
            .getter(getter(Policy::resourceTagLogicalOperatorAsString))
            .setter(setter(Builder::resourceTagLogicalOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTagLogicalOperator").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POLICY_ID_FIELD,
            POLICY_NAME_FIELD, POLICY_UPDATE_TOKEN_FIELD, SECURITY_SERVICE_POLICY_DATA_FIELD, RESOURCE_TYPE_FIELD,
            RESOURCE_TYPE_LIST_FIELD, RESOURCE_TAGS_FIELD, EXCLUDE_RESOURCE_TAGS_FIELD, REMEDIATION_ENABLED_FIELD,
            DELETE_UNUSED_FM_MANAGED_RESOURCES_FIELD, INCLUDE_MAP_FIELD, EXCLUDE_MAP_FIELD, RESOURCE_SET_IDS_FIELD,
            POLICY_DESCRIPTION_FIELD, POLICY_STATUS_FIELD, RESOURCE_TAG_LOGICAL_OPERATOR_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String policyId;

    private final String policyName;

    private final String policyUpdateToken;

    private final SecurityServicePolicyData securityServicePolicyData;

    private final String resourceType;

    private final List<String> resourceTypeList;

    private final List<ResourceTag> resourceTags;

    private final Boolean excludeResourceTags;

    private final Boolean remediationEnabled;

    private final Boolean deleteUnusedFMManagedResources;

    private final Map<String, List<String>> includeMap;

    private final Map<String, List<String>> excludeMap;

    private final List<String> resourceSetIds;

    private final String policyDescription;

    private final String policyStatus;

    private final String resourceTagLogicalOperator;

    private Policy(BuilderImpl builder) {
        this.policyId = builder.policyId;
        this.policyName = builder.policyName;
        this.policyUpdateToken = builder.policyUpdateToken;
        this.securityServicePolicyData = builder.securityServicePolicyData;
        this.resourceType = builder.resourceType;
        this.resourceTypeList = builder.resourceTypeList;
        this.resourceTags = builder.resourceTags;
        this.excludeResourceTags = builder.excludeResourceTags;
        this.remediationEnabled = builder.remediationEnabled;
        this.deleteUnusedFMManagedResources = builder.deleteUnusedFMManagedResources;
        this.includeMap = builder.includeMap;
        this.excludeMap = builder.excludeMap;
        this.resourceSetIds = builder.resourceSetIds;
        this.policyDescription = builder.policyDescription;
        this.policyStatus = builder.policyStatus;
        this.resourceTagLogicalOperator = builder.resourceTagLogicalOperator;
    }

    /**
     * <p>
     * The ID of the Firewall Manager policy.
     * </p>
     * 
     * @return The ID of the Firewall Manager policy.
     */
    public final String policyId() {
        return policyId;
    }

    /**
     * <p>
     * The name of the Firewall Manager policy.
     * </p>
     * 
     * @return The name of the Firewall Manager policy.
     */
    public final String policyName() {
        return policyName;
    }

    /**
     * <p>
     * A unique identifier for each update to the policy. When issuing a <code>PutPolicy</code> request, the
     * <code>PolicyUpdateToken</code> in the request must match the <code>PolicyUpdateToken</code> of the current policy
     * version. To get the <code>PolicyUpdateToken</code> of the current policy version, use a <code>GetPolicy</code>
     * request.
     * </p>
     * 
     * @return A unique identifier for each update to the policy. When issuing a <code>PutPolicy</code> request, the
     *         <code>PolicyUpdateToken</code> in the request must match the <code>PolicyUpdateToken</code> of the
     *         current policy version. To get the <code>PolicyUpdateToken</code> of the current policy version, use a
     *         <code>GetPolicy</code> request.
     */
    public final String policyUpdateToken() {
        return policyUpdateToken;
    }

    /**
     * <p>
     * Details about the security service that is being used to protect the resources.
     * </p>
     * 
     * @return Details about the security service that is being used to protect the resources.
     */
    public final SecurityServicePolicyData securityServicePolicyData() {
        return securityServicePolicyData;
    }

    /**
     * <p>
     * The type of resource protected by or in scope of the policy. This is in the format shown in the <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">Amazon
     * Web Services Resource Types Reference</a>. To apply this policy to multiple resource types, specify a resource
     * type of <code>ResourceTypeList</code> and then specify the resource types in a <code>ResourceTypeList</code>.
     * </p>
     * <p>
     * The following are valid resource types for each Firewall Manager policy type:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Web Services WAF Classic - <code>AWS::ApiGateway::Stage</code>, <code>AWS::CloudFront::Distribution</code>
     * , and <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * WAF - <code>AWS::ApiGateway::Stage</code>, <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>, and
     * <code>AWS::CloudFront::Distribution</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Shield Advanced - <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>,
     * <code>AWS::ElasticLoadBalancing::LoadBalancer</code>, <code>AWS::EC2::EIP</code>, and
     * <code>AWS::CloudFront::Distribution</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Network ACL - <code>AWS::EC2::Subnet</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Security group usage audit - <code>AWS::EC2::SecurityGroup</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Security group content audit - <code>AWS::EC2::SecurityGroup</code>, <code>AWS::EC2::NetworkInterface</code>, and
     * <code>AWS::EC2::Instance</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * DNS Firewall, Network Firewall, and third-party firewall - <code>AWS::EC2::VPC</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The type of resource protected by or in scope of the policy. This is in the format shown in the <a
     *         href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
     *         >Amazon Web Services Resource Types Reference</a>. To apply this policy to multiple resource types,
     *         specify a resource type of <code>ResourceTypeList</code> and then specify the resource types in a
     *         <code>ResourceTypeList</code>.</p>
     *         <p>
     *         The following are valid resource types for each Firewall Manager policy type:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Amazon Web Services WAF Classic - <code>AWS::ApiGateway::Stage</code>,
     *         <code>AWS::CloudFront::Distribution</code>, and <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         WAF - <code>AWS::ApiGateway::Stage</code>, <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>, and
     *         <code>AWS::CloudFront::Distribution</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Shield Advanced - <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>,
     *         <code>AWS::ElasticLoadBalancing::LoadBalancer</code>, <code>AWS::EC2::EIP</code>, and
     *         <code>AWS::CloudFront::Distribution</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Network ACL - <code>AWS::EC2::Subnet</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Security group usage audit - <code>AWS::EC2::SecurityGroup</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Security group content audit - <code>AWS::EC2::SecurityGroup</code>,
     *         <code>AWS::EC2::NetworkInterface</code>, and <code>AWS::EC2::Instance</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DNS Firewall, Network Firewall, and third-party firewall - <code>AWS::EC2::VPC</code>.
     *         </p>
     *         </li>
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTypeList property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTypeList() {
        return resourceTypeList != null && !(resourceTypeList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To specify a
     * single resource type, use <code>ResourceType</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypeList} method.
     * </p>
     * 
     * @return An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To
     *         specify a single resource type, use <code>ResourceType</code>.
     */
    public final List<String> resourceTypeList() {
        return resourceTypeList;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTags() {
        return resourceTags != null && !(resourceTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ResourceTag</code> objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTags} method.
     * </p>
     * 
     * @return An array of <code>ResourceTag</code> objects.
     */
    public final List<ResourceTag> resourceTags() {
        return resourceTags;
    }

    /**
     * <p>
     * If set to <code>True</code>, resources with the tags that are specified in the <code>ResourceTag</code> array are
     * not in scope of the policy. If set to <code>False</code>, and the <code>ResourceTag</code> array is not null,
     * only resources with the specified tags are in scope of the policy.
     * </p>
     * 
     * @return If set to <code>True</code>, resources with the tags that are specified in the <code>ResourceTag</code>
     *         array are not in scope of the policy. If set to <code>False</code>, and the <code>ResourceTag</code>
     *         array is not null, only resources with the specified tags are in scope of the policy.
     */
    public final Boolean excludeResourceTags() {
        return excludeResourceTags;
    }

    /**
     * <p>
     * Indicates if the policy should be automatically applied to new resources.
     * </p>
     * 
     * @return Indicates if the policy should be automatically applied to new resources.
     */
    public final Boolean remediationEnabled() {
        return remediationEnabled;
    }

    /**
     * <p>
     * Indicates whether Firewall Manager should automatically remove protections from resources that leave the policy
     * scope and clean up resources that Firewall Manager is managing for accounts when those accounts leave policy
     * scope. For example, Firewall Manager will disassociate a Firewall Manager managed web ACL from a protected
     * customer resource when the customer resource leaves policy scope.
     * </p>
     * <p>
     * By default, Firewall Manager doesn't remove protections or delete Firewall Manager managed resources.
     * </p>
     * <p>
     * This option is not available for Shield Advanced or WAF Classic policies.
     * </p>
     * 
     * @return Indicates whether Firewall Manager should automatically remove protections from resources that leave the
     *         policy scope and clean up resources that Firewall Manager is managing for accounts when those accounts
     *         leave policy scope. For example, Firewall Manager will disassociate a Firewall Manager managed web ACL
     *         from a protected customer resource when the customer resource leaves policy scope. </p>
     *         <p>
     *         By default, Firewall Manager doesn't remove protections or delete Firewall Manager managed resources.
     *         </p>
     *         <p>
     *         This option is not available for Shield Advanced or WAF Classic policies.
     */
    public final Boolean deleteUnusedFMManagedResources() {
        return deleteUnusedFMManagedResources;
    }

    /**
     * <p>
     * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in the
     * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
     * including any child OUs and accounts that are added at a later time.
     * </p>
     * <p>
     * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
     * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate any
     * <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall Manager
     * applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
     * </p>
     * <p>
     * You can specify account IDs, OUs, or a combination:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
     * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a valid
     * map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludeMap} method.
     * </p>
     * 
     * @return Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in
     *         the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its
     *         child OUs, including any child OUs and accounts that are added at a later time.</p>
     *         <p>
     *         You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
     *         Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
     *         not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>
     *         , then Firewall Manager applies the policy to all accounts except for those specified by the
     *         <code>ExcludeMap</code>.
     *         </p>
     *         <p>
     *         You can specify account IDs, OUs, or a combination:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
     *         map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     *         <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify accounts and OUs together in a single map, separated with a comma. For example, the following is
     *         a valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>
     *         .
     *         </p>
     *         </li>
     */
    public final Map<CustomerPolicyScopeIdType, List<String>> includeMap() {
        return CustomerPolicyScopeMapCopier.copyStringToEnum(includeMap);
    }

    /**
     * For responses, this returns true if the service returned a value for the IncludeMap property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasIncludeMap() {
        return includeMap != null && !(includeMap instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in the
     * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
     * including any child OUs and accounts that are added at a later time.
     * </p>
     * <p>
     * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
     * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate any
     * <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall Manager
     * applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
     * </p>
     * <p>
     * You can specify account IDs, OUs, or a combination:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
     * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a valid
     * map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludeMap} method.
     * </p>
     * 
     * @return Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in
     *         the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its
     *         child OUs, including any child OUs and accounts that are added at a later time.</p>
     *         <p>
     *         You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
     *         Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
     *         not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>
     *         , then Firewall Manager applies the policy to all accounts except for those specified by the
     *         <code>ExcludeMap</code>.
     *         </p>
     *         <p>
     *         You can specify account IDs, OUs, or a combination:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
     *         map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     *         <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify accounts and OUs together in a single map, separated with a comma. For example, the following is
     *         a valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>
     *         .
     *         </p>
     *         </li>
     */
    public final Map<String, List<String>> includeMapAsStrings() {
        return includeMap;
    }

    /**
     * <p>
     * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude from the
     * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
     * including any child OUs and accounts that are added at a later time.
     * </p>
     * <p>
     * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
     * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate any
     * <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall Manager
     * applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
     * </p>
     * <p>
     * You can specify account IDs, OUs, or a combination:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
     * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a valid
     * map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludeMap} method.
     * </p>
     * 
     * @return Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude
     *         from the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
     *         its child OUs, including any child OUs and accounts that are added at a later time.</p>
     *         <p>
     *         You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
     *         Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
     *         not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>
     *         , then Firewall Manager applies the policy to all accounts except for those specified by the
     *         <code>ExcludeMap</code>.
     *         </p>
     *         <p>
     *         You can specify account IDs, OUs, or a combination:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
     *         map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     *         <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify accounts and OUs together in a single map, separated with a comma. For example, the following is
     *         a valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>
     *         .
     *         </p>
     *         </li>
     */
    public final Map<CustomerPolicyScopeIdType, List<String>> excludeMap() {
        return CustomerPolicyScopeMapCopier.copyStringToEnum(excludeMap);
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludeMap property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasExcludeMap() {
        return excludeMap != null && !(excludeMap instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude from the
     * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
     * including any child OUs and accounts that are added at a later time.
     * </p>
     * <p>
     * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
     * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate any
     * <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall Manager
     * applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
     * </p>
     * <p>
     * You can specify account IDs, OUs, or a combination:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
     * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a valid
     * map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludeMap} method.
     * </p>
     * 
     * @return Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude
     *         from the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
     *         its child OUs, including any child OUs and accounts that are added at a later time.</p>
     *         <p>
     *         You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
     *         Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
     *         not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>
     *         , then Firewall Manager applies the policy to all accounts except for those specified by the
     *         <code>ExcludeMap</code>.
     *         </p>
     *         <p>
     *         You can specify account IDs, OUs, or a combination:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
     *         map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
     *         <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify accounts and OUs together in a single map, separated with a comma. For example, the following is
     *         a valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>
     *         .
     *         </p>
     *         </li>
     */
    public final Map<String, List<String>> excludeMapAsStrings() {
        return excludeMap;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceSetIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceSetIds() {
        return resourceSetIds != null && !(resourceSetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The unique identifiers of the resource sets used by the policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceSetIds} method.
     * </p>
     * 
     * @return The unique identifiers of the resource sets used by the policy.
     */
    public final List<String> resourceSetIds() {
        return resourceSetIds;
    }

    /**
     * <p>
     * Your description of the Firewall Manager policy.
     * </p>
     * 
     * @return Your description of the Firewall Manager policy.
     */
    public final String policyDescription() {
        return policyDescription;
    }

    /**
     * <p>
     * Indicates whether the policy is in or out of an admin's policy or Region scope.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The administrator can manage and delete the policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OUT_OF_ADMIN_SCOPE</code> - The administrator can view the policy, but they can't edit or delete the
     * policy. Existing policy protections stay in place. Any new resources that come into scope of the policy won't be
     * protected.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyStatus} will
     * return {@link CustomerPolicyStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #policyStatusAsString}.
     * </p>
     * 
     * @return Indicates whether the policy is in or out of an admin's policy or Region scope.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The administrator can manage and delete the policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OUT_OF_ADMIN_SCOPE</code> - The administrator can view the policy, but they can't edit or delete
     *         the policy. Existing policy protections stay in place. Any new resources that come into scope of the
     *         policy won't be protected.
     *         </p>
     *         </li>
     * @see CustomerPolicyStatus
     */
    public final CustomerPolicyStatus policyStatus() {
        return CustomerPolicyStatus.fromValue(policyStatus);
    }

    /**
     * <p>
     * Indicates whether the policy is in or out of an admin's policy or Region scope.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The administrator can manage and delete the policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OUT_OF_ADMIN_SCOPE</code> - The administrator can view the policy, but they can't edit or delete the
     * policy. Existing policy protections stay in place. Any new resources that come into scope of the policy won't be
     * protected.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyStatus} will
     * return {@link CustomerPolicyStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #policyStatusAsString}.
     * </p>
     * 
     * @return Indicates whether the policy is in or out of an admin's policy or Region scope.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The administrator can manage and delete the policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OUT_OF_ADMIN_SCOPE</code> - The administrator can view the policy, but they can't edit or delete
     *         the policy. Existing policy protections stay in place. Any new resources that come into scope of the
     *         policy won't be protected.
     *         </p>
     *         </li>
     * @see CustomerPolicyStatus
     */
    public final String policyStatusAsString() {
        return policyStatus;
    }

    /**
     * <p>
     * Specifies whether to combine multiple resource tags with AND, so that a resource must have all tags to be
     * included or excluded, or OR, so that a resource must have at least one tag.
     * </p>
     * <p>
     * Default: <code>AND</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #resourceTagLogicalOperator} will return {@link ResourceTagLogicalOperator#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #resourceTagLogicalOperatorAsString}.
     * </p>
     * 
     * @return Specifies whether to combine multiple resource tags with AND, so that a resource must have all tags to be
     *         included or excluded, or OR, so that a resource must have at least one tag.</p>
     *         <p>
     *         Default: <code>AND</code>
     * @see ResourceTagLogicalOperator
     */
    public final ResourceTagLogicalOperator resourceTagLogicalOperator() {
        return ResourceTagLogicalOperator.fromValue(resourceTagLogicalOperator);
    }

    /**
     * <p>
     * Specifies whether to combine multiple resource tags with AND, so that a resource must have all tags to be
     * included or excluded, or OR, so that a resource must have at least one tag.
     * </p>
     * <p>
     * Default: <code>AND</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #resourceTagLogicalOperator} will return {@link ResourceTagLogicalOperator#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #resourceTagLogicalOperatorAsString}.
     * </p>
     * 
     * @return Specifies whether to combine multiple resource tags with AND, so that a resource must have all tags to be
     *         included or excluded, or OR, so that a resource must have at least one tag.</p>
     *         <p>
     *         Default: <code>AND</code>
     * @see ResourceTagLogicalOperator
     */
    public final String resourceTagLogicalOperatorAsString() {
        return resourceTagLogicalOperator;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(policyId());
        hashCode = 31 * hashCode + Objects.hashCode(policyName());
        hashCode = 31 * hashCode + Objects.hashCode(policyUpdateToken());
        hashCode = 31 * hashCode + Objects.hashCode(securityServicePolicyData());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypeList() ? resourceTypeList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTags() ? resourceTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(excludeResourceTags());
        hashCode = 31 * hashCode + Objects.hashCode(remediationEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(deleteUnusedFMManagedResources());
        hashCode = 31 * hashCode + Objects.hashCode(hasIncludeMap() ? includeMapAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludeMap() ? excludeMapAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceSetIds() ? resourceSetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(policyDescription());
        hashCode = 31 * hashCode + Objects.hashCode(policyStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTagLogicalOperatorAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Policy)) {
            return false;
        }
        Policy other = (Policy) obj;
        return Objects.equals(policyId(), other.policyId()) && Objects.equals(policyName(), other.policyName())
                && Objects.equals(policyUpdateToken(), other.policyUpdateToken())
                && Objects.equals(securityServicePolicyData(), other.securityServicePolicyData())
                && Objects.equals(resourceType(), other.resourceType()) && hasResourceTypeList() == other.hasResourceTypeList()
                && Objects.equals(resourceTypeList(), other.resourceTypeList()) && hasResourceTags() == other.hasResourceTags()
                && Objects.equals(resourceTags(), other.resourceTags())
                && Objects.equals(excludeResourceTags(), other.excludeResourceTags())
                && Objects.equals(remediationEnabled(), other.remediationEnabled())
                && Objects.equals(deleteUnusedFMManagedResources(), other.deleteUnusedFMManagedResources())
                && hasIncludeMap() == other.hasIncludeMap() && Objects.equals(includeMapAsStrings(), other.includeMapAsStrings())
                && hasExcludeMap() == other.hasExcludeMap() && Objects.equals(excludeMapAsStrings(), other.excludeMapAsStrings())
                && hasResourceSetIds() == other.hasResourceSetIds() && Objects.equals(resourceSetIds(), other.resourceSetIds())
                && Objects.equals(policyDescription(), other.policyDescription())
                && Objects.equals(policyStatusAsString(), other.policyStatusAsString())
                && Objects.equals(resourceTagLogicalOperatorAsString(), other.resourceTagLogicalOperatorAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Policy").add("PolicyId", policyId()).add("PolicyName", policyName())
                .add("PolicyUpdateToken", policyUpdateToken()).add("SecurityServicePolicyData", securityServicePolicyData())
                .add("ResourceType", resourceType()).add("ResourceTypeList", hasResourceTypeList() ? resourceTypeList() : null)
                .add("ResourceTags", hasResourceTags() ? resourceTags() : null).add("ExcludeResourceTags", excludeResourceTags())
                .add("RemediationEnabled", remediationEnabled())
                .add("DeleteUnusedFMManagedResources", deleteUnusedFMManagedResources())
                .add("IncludeMap", hasIncludeMap() ? includeMapAsStrings() : null)
                .add("ExcludeMap", hasExcludeMap() ? excludeMapAsStrings() : null)
                .add("ResourceSetIds", hasResourceSetIds() ? resourceSetIds() : null)
                .add("PolicyDescription", policyDescription()).add("PolicyStatus", policyStatusAsString())
                .add("ResourceTagLogicalOperator", resourceTagLogicalOperatorAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PolicyId":
            return Optional.ofNullable(clazz.cast(policyId()));
        case "PolicyName":
            return Optional.ofNullable(clazz.cast(policyName()));
        case "PolicyUpdateToken":
            return Optional.ofNullable(clazz.cast(policyUpdateToken()));
        case "SecurityServicePolicyData":
            return Optional.ofNullable(clazz.cast(securityServicePolicyData()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "ResourceTypeList":
            return Optional.ofNullable(clazz.cast(resourceTypeList()));
        case "ResourceTags":
            return Optional.ofNullable(clazz.cast(resourceTags()));
        case "ExcludeResourceTags":
            return Optional.ofNullable(clazz.cast(excludeResourceTags()));
        case "RemediationEnabled":
            return Optional.ofNullable(clazz.cast(remediationEnabled()));
        case "DeleteUnusedFMManagedResources":
            return Optional.ofNullable(clazz.cast(deleteUnusedFMManagedResources()));
        case "IncludeMap":
            return Optional.ofNullable(clazz.cast(includeMapAsStrings()));
        case "ExcludeMap":
            return Optional.ofNullable(clazz.cast(excludeMapAsStrings()));
        case "ResourceSetIds":
            return Optional.ofNullable(clazz.cast(resourceSetIds()));
        case "PolicyDescription":
            return Optional.ofNullable(clazz.cast(policyDescription()));
        case "PolicyStatus":
            return Optional.ofNullable(clazz.cast(policyStatusAsString()));
        case "ResourceTagLogicalOperator":
            return Optional.ofNullable(clazz.cast(resourceTagLogicalOperatorAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("PolicyId", POLICY_ID_FIELD);
        map.put("PolicyName", POLICY_NAME_FIELD);
        map.put("PolicyUpdateToken", POLICY_UPDATE_TOKEN_FIELD);
        map.put("SecurityServicePolicyData", SECURITY_SERVICE_POLICY_DATA_FIELD);
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("ResourceTypeList", RESOURCE_TYPE_LIST_FIELD);
        map.put("ResourceTags", RESOURCE_TAGS_FIELD);
        map.put("ExcludeResourceTags", EXCLUDE_RESOURCE_TAGS_FIELD);
        map.put("RemediationEnabled", REMEDIATION_ENABLED_FIELD);
        map.put("DeleteUnusedFMManagedResources", DELETE_UNUSED_FM_MANAGED_RESOURCES_FIELD);
        map.put("IncludeMap", INCLUDE_MAP_FIELD);
        map.put("ExcludeMap", EXCLUDE_MAP_FIELD);
        map.put("ResourceSetIds", RESOURCE_SET_IDS_FIELD);
        map.put("PolicyDescription", POLICY_DESCRIPTION_FIELD);
        map.put("PolicyStatus", POLICY_STATUS_FIELD);
        map.put("ResourceTagLogicalOperator", RESOURCE_TAG_LOGICAL_OPERATOR_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Policy, T> g) {
        return obj -> g.apply((Policy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Policy> {
        /**
         * <p>
         * The ID of the Firewall Manager policy.
         * </p>
         * 
         * @param policyId
         *        The ID of the Firewall Manager policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyId(String policyId);

        /**
         * <p>
         * The name of the Firewall Manager policy.
         * </p>
         * 
         * @param policyName
         *        The name of the Firewall Manager policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyName(String policyName);

        /**
         * <p>
         * A unique identifier for each update to the policy. When issuing a <code>PutPolicy</code> request, the
         * <code>PolicyUpdateToken</code> in the request must match the <code>PolicyUpdateToken</code> of the current
         * policy version. To get the <code>PolicyUpdateToken</code> of the current policy version, use a
         * <code>GetPolicy</code> request.
         * </p>
         * 
         * @param policyUpdateToken
         *        A unique identifier for each update to the policy. When issuing a <code>PutPolicy</code> request, the
         *        <code>PolicyUpdateToken</code> in the request must match the <code>PolicyUpdateToken</code> of the
         *        current policy version. To get the <code>PolicyUpdateToken</code> of the current policy version, use a
         *        <code>GetPolicy</code> request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyUpdateToken(String policyUpdateToken);

        /**
         * <p>
         * Details about the security service that is being used to protect the resources.
         * </p>
         * 
         * @param securityServicePolicyData
         *        Details about the security service that is being used to protect the resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityServicePolicyData(SecurityServicePolicyData securityServicePolicyData);

        /**
         * <p>
         * Details about the security service that is being used to protect the resources.
         * </p>
         * This is a convenience method that creates an instance of the {@link SecurityServicePolicyData.Builder}
         * avoiding the need to create one manually via {@link SecurityServicePolicyData#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SecurityServicePolicyData.Builder#build()} is called immediately
         * and its result is passed to {@link #securityServicePolicyData(SecurityServicePolicyData)}.
         * 
         * @param securityServicePolicyData
         *        a consumer that will call methods on {@link SecurityServicePolicyData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #securityServicePolicyData(SecurityServicePolicyData)
         */
        default Builder securityServicePolicyData(Consumer<SecurityServicePolicyData.Builder> securityServicePolicyData) {
            return securityServicePolicyData(SecurityServicePolicyData.builder().applyMutation(securityServicePolicyData).build());
        }

        /**
         * <p>
         * The type of resource protected by or in scope of the policy. This is in the format shown in the <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         * >Amazon Web Services Resource Types Reference</a>. To apply this policy to multiple resource types, specify a
         * resource type of <code>ResourceTypeList</code> and then specify the resource types in a
         * <code>ResourceTypeList</code>.
         * </p>
         * <p>
         * The following are valid resource types for each Firewall Manager policy type:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Amazon Web Services WAF Classic - <code>AWS::ApiGateway::Stage</code>,
         * <code>AWS::CloudFront::Distribution</code>, and <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * WAF - <code>AWS::ApiGateway::Stage</code>, <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>, and
         * <code>AWS::CloudFront::Distribution</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Shield Advanced - <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>,
         * <code>AWS::ElasticLoadBalancing::LoadBalancer</code>, <code>AWS::EC2::EIP</code>, and
         * <code>AWS::CloudFront::Distribution</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Network ACL - <code>AWS::EC2::Subnet</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Security group usage audit - <code>AWS::EC2::SecurityGroup</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Security group content audit - <code>AWS::EC2::SecurityGroup</code>, <code>AWS::EC2::NetworkInterface</code>,
         * and <code>AWS::EC2::Instance</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * DNS Firewall, Network Firewall, and third-party firewall - <code>AWS::EC2::VPC</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceType
         *        The type of resource protected by or in scope of the policy. This is in the format shown in the <a
         *        href
         *        ="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html"
         *        >Amazon Web Services Resource Types Reference</a>. To apply this policy to multiple resource types,
         *        specify a resource type of <code>ResourceTypeList</code> and then specify the resource types in a
         *        <code>ResourceTypeList</code>.</p>
         *        <p>
         *        The following are valid resource types for each Firewall Manager policy type:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Amazon Web Services WAF Classic - <code>AWS::ApiGateway::Stage</code>,
         *        <code>AWS::CloudFront::Distribution</code>, and <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>
         *        .
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        WAF - <code>AWS::ApiGateway::Stage</code>, <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>, and
         *        <code>AWS::CloudFront::Distribution</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Shield Advanced - <code>AWS::ElasticLoadBalancingV2::LoadBalancer</code>,
         *        <code>AWS::ElasticLoadBalancing::LoadBalancer</code>, <code>AWS::EC2::EIP</code>, and
         *        <code>AWS::CloudFront::Distribution</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Network ACL - <code>AWS::EC2::Subnet</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Security group usage audit - <code>AWS::EC2::SecurityGroup</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Security group content audit - <code>AWS::EC2::SecurityGroup</code>,
         *        <code>AWS::EC2::NetworkInterface</code>, and <code>AWS::EC2::Instance</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DNS Firewall, Network Firewall, and third-party firewall - <code>AWS::EC2::VPC</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To specify a
         * single resource type, use <code>ResourceType</code>.
         * </p>
         * 
         * @param resourceTypeList
         *        An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To
         *        specify a single resource type, use <code>ResourceType</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypeList(Collection<String> resourceTypeList);

        /**
         * <p>
         * An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To specify a
         * single resource type, use <code>ResourceType</code>.
         * </p>
         * 
         * @param resourceTypeList
         *        An array of <code>ResourceType</code> objects. Use this only to specify multiple resource types. To
         *        specify a single resource type, use <code>ResourceType</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypeList(String... resourceTypeList);

        /**
         * <p>
         * An array of <code>ResourceTag</code> objects.
         * </p>
         * 
         * @param resourceTags
         *        An array of <code>ResourceTag</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(Collection<ResourceTag> resourceTags);

        /**
         * <p>
         * An array of <code>ResourceTag</code> objects.
         * </p>
         * 
         * @param resourceTags
         *        An array of <code>ResourceTag</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(ResourceTag... resourceTags);

        /**
         * <p>
         * An array of <code>ResourceTag</code> objects.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fms.model.ResourceTag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.fms.model.ResourceTag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.fms.model.ResourceTag.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceTags(List<ResourceTag>)}.
         * 
         * @param resourceTags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.fms.model.ResourceTag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceTags(java.util.Collection<ResourceTag>)
         */
        Builder resourceTags(Consumer<ResourceTag.Builder>... resourceTags);

        /**
         * <p>
         * If set to <code>True</code>, resources with the tags that are specified in the <code>ResourceTag</code> array
         * are not in scope of the policy. If set to <code>False</code>, and the <code>ResourceTag</code> array is not
         * null, only resources with the specified tags are in scope of the policy.
         * </p>
         * 
         * @param excludeResourceTags
         *        If set to <code>True</code>, resources with the tags that are specified in the
         *        <code>ResourceTag</code> array are not in scope of the policy. If set to <code>False</code>, and the
         *        <code>ResourceTag</code> array is not null, only resources with the specified tags are in scope of the
         *        policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeResourceTags(Boolean excludeResourceTags);

        /**
         * <p>
         * Indicates if the policy should be automatically applied to new resources.
         * </p>
         * 
         * @param remediationEnabled
         *        Indicates if the policy should be automatically applied to new resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remediationEnabled(Boolean remediationEnabled);

        /**
         * <p>
         * Indicates whether Firewall Manager should automatically remove protections from resources that leave the
         * policy scope and clean up resources that Firewall Manager is managing for accounts when those accounts leave
         * policy scope. For example, Firewall Manager will disassociate a Firewall Manager managed web ACL from a
         * protected customer resource when the customer resource leaves policy scope.
         * </p>
         * <p>
         * By default, Firewall Manager doesn't remove protections or delete Firewall Manager managed resources.
         * </p>
         * <p>
         * This option is not available for Shield Advanced or WAF Classic policies.
         * </p>
         * 
         * @param deleteUnusedFMManagedResources
         *        Indicates whether Firewall Manager should automatically remove protections from resources that leave
         *        the policy scope and clean up resources that Firewall Manager is managing for accounts when those
         *        accounts leave policy scope. For example, Firewall Manager will disassociate a Firewall Manager
         *        managed web ACL from a protected customer resource when the customer resource leaves policy scope.
         *        </p>
         *        <p>
         *        By default, Firewall Manager doesn't remove protections or delete Firewall Manager managed resources.
         *        </p>
         *        <p>
         *        This option is not available for Shield Advanced or WAF Classic policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteUnusedFMManagedResources(Boolean deleteUnusedFMManagedResources);

        /**
         * <p>
         * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in the
         * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
         * including any child OUs and accounts that are added at a later time.
         * </p>
         * <p>
         * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
         * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate
         * any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall
         * Manager applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
         * </p>
         * <p>
         * You can specify account IDs, OUs, or a combination:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
         * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a
         * valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param includeMap
         *        Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include
         *        in the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
         *        its child OUs, including any child OUs and accounts that are added at a later time.</p>
         *        <p>
         *        You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
         *        Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
         *        not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an
         *        <code>IncludeMap</code>, then Firewall Manager applies the policy to all accounts except for those
         *        specified by the <code>ExcludeMap</code>.
         *        </p>
         *        <p>
         *        You can specify account IDs, OUs, or a combination:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
         *        map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         *        <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify accounts and OUs together in a single map, separated with a comma. For example, the following
         *        is a valid map:
         *        <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeMapWithStrings(Map<String, ? extends Collection<String>> includeMap);

        /**
         * <p>
         * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include in the
         * policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child OUs,
         * including any child OUs and accounts that are added at a later time.
         * </p>
         * <p>
         * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
         * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate
         * any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall
         * Manager applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
         * </p>
         * <p>
         * You can specify account IDs, OUs, or a combination:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
         * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a
         * valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param includeMap
         *        Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to include
         *        in the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
         *        its child OUs, including any child OUs and accounts that are added at a later time.</p>
         *        <p>
         *        You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
         *        Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
         *        not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an
         *        <code>IncludeMap</code>, then Firewall Manager applies the policy to all accounts except for those
         *        specified by the <code>ExcludeMap</code>.
         *        </p>
         *        <p>
         *        You can specify account IDs, OUs, or a combination:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
         *        map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         *        <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify accounts and OUs together in a single map, separated with a comma. For example, the following
         *        is a valid map:
         *        <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeMap(Map<CustomerPolicyScopeIdType, ? extends Collection<String>> includeMap);

        /**
         * <p>
         * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude from
         * the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child
         * OUs, including any child OUs and accounts that are added at a later time.
         * </p>
         * <p>
         * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
         * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate
         * any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall
         * Manager applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
         * </p>
         * <p>
         * You can specify account IDs, OUs, or a combination:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
         * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a
         * valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param excludeMap
         *        Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude
         *        from the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
         *        its child OUs, including any child OUs and accounts that are added at a later time.</p>
         *        <p>
         *        You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
         *        Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
         *        not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an
         *        <code>IncludeMap</code>, then Firewall Manager applies the policy to all accounts except for those
         *        specified by the <code>ExcludeMap</code>.
         *        </p>
         *        <p>
         *        You can specify account IDs, OUs, or a combination:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
         *        map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         *        <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify accounts and OUs together in a single map, separated with a comma. For example, the following
         *        is a valid map:
         *        <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeMapWithStrings(Map<String, ? extends Collection<String>> excludeMap);

        /**
         * <p>
         * Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude from
         * the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of its child
         * OUs, including any child OUs and accounts that are added at a later time.
         * </p>
         * <p>
         * You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>, Firewall
         * Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does not evaluate
         * any <code>ExcludeMap</code> specifications. If you do not specify an <code>IncludeMap</code>, then Firewall
         * Manager applies the policy to all accounts except for those specified by the <code>ExcludeMap</code>.
         * </p>
         * <p>
         * You can specify account IDs, OUs, or a combination:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid map:
         * <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         * <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Specify accounts and OUs together in a single map, separated with a comma. For example, the following is a
         * valid map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param excludeMap
         *        Specifies the Amazon Web Services account IDs and Organizations organizational units (OUs) to exclude
         *        from the policy. Specifying an OU is the equivalent of specifying all accounts in the OU and in any of
         *        its child OUs, including any child OUs and accounts that are added at a later time.</p>
         *        <p>
         *        You can specify inclusions or exclusions, but not both. If you specify an <code>IncludeMap</code>,
         *        Firewall Manager applies the policy to all accounts specified by the <code>IncludeMap</code>, and does
         *        not evaluate any <code>ExcludeMap</code> specifications. If you do not specify an
         *        <code>IncludeMap</code>, then Firewall Manager applies the policy to all accounts except for those
         *        specified by the <code>ExcludeMap</code>.
         *        </p>
         *        <p>
         *        You can specify account IDs, OUs, or a combination:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Specify account IDs by setting the key to <code>ACCOUNT</code>. For example, the following is a valid
         *        map: <code>{“ACCOUNT” : [“accountID1”, “accountID2”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify OUs by setting the key to <code>ORG_UNIT</code>. For example, the following is a valid map:
         *        <code>{“ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Specify accounts and OUs together in a single map, separated with a comma. For example, the following
         *        is a valid map:
         *        <code>{“ACCOUNT” : [“accountID1”, “accountID2”], “ORG_UNIT” : [“ouid111”, “ouid112”]}</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeMap(Map<CustomerPolicyScopeIdType, ? extends Collection<String>> excludeMap);

        /**
         * <p>
         * The unique identifiers of the resource sets used by the policy.
         * </p>
         * 
         * @param resourceSetIds
         *        The unique identifiers of the resource sets used by the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceSetIds(Collection<String> resourceSetIds);

        /**
         * <p>
         * The unique identifiers of the resource sets used by the policy.
         * </p>
         * 
         * @param resourceSetIds
         *        The unique identifiers of the resource sets used by the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceSetIds(String... resourceSetIds);

        /**
         * <p>
         * Your description of the Firewall Manager policy.
         * </p>
         * 
         * @param policyDescription
         *        Your description of the Firewall Manager policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyDescription(String policyDescription);

        /**
         * <p>
         * Indicates whether the policy is in or out of an admin's policy or Region scope.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The administrator can manage and delete the policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OUT_OF_ADMIN_SCOPE</code> - The administrator can view the policy, but they can't edit or delete the
         * policy. Existing policy protections stay in place. Any new resources that come into scope of the policy won't
         * be protected.
         * </p>
         * </li>
         * </ul>
         * 
         * @param policyStatus
         *        Indicates whether the policy is in or out of an admin's policy or Region scope.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The administrator can manage and delete the policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OUT_OF_ADMIN_SCOPE</code> - The administrator can view the policy, but they can't edit or delete
         *        the policy. Existing policy protections stay in place. Any new resources that come into scope of the
         *        policy won't be protected.
         *        </p>
         *        </li>
         * @see CustomerPolicyStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomerPolicyStatus
         */
        Builder policyStatus(String policyStatus);

        /**
         * <p>
         * Indicates whether the policy is in or out of an admin's policy or Region scope.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The administrator can manage and delete the policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OUT_OF_ADMIN_SCOPE</code> - The administrator can view the policy, but they can't edit or delete the
         * policy. Existing policy protections stay in place. Any new resources that come into scope of the policy won't
         * be protected.
         * </p>
         * </li>
         * </ul>
         * 
         * @param policyStatus
         *        Indicates whether the policy is in or out of an admin's policy or Region scope.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The administrator can manage and delete the policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OUT_OF_ADMIN_SCOPE</code> - The administrator can view the policy, but they can't edit or delete
         *        the policy. Existing policy protections stay in place. Any new resources that come into scope of the
         *        policy won't be protected.
         *        </p>
         *        </li>
         * @see CustomerPolicyStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomerPolicyStatus
         */
        Builder policyStatus(CustomerPolicyStatus policyStatus);

        /**
         * <p>
         * Specifies whether to combine multiple resource tags with AND, so that a resource must have all tags to be
         * included or excluded, or OR, so that a resource must have at least one tag.
         * </p>
         * <p>
         * Default: <code>AND</code>
         * </p>
         * 
         * @param resourceTagLogicalOperator
         *        Specifies whether to combine multiple resource tags with AND, so that a resource must have all tags to
         *        be included or excluded, or OR, so that a resource must have at least one tag.</p>
         *        <p>
         *        Default: <code>AND</code>
         * @see ResourceTagLogicalOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceTagLogicalOperator
         */
        Builder resourceTagLogicalOperator(String resourceTagLogicalOperator);

        /**
         * <p>
         * Specifies whether to combine multiple resource tags with AND, so that a resource must have all tags to be
         * included or excluded, or OR, so that a resource must have at least one tag.
         * </p>
         * <p>
         * Default: <code>AND</code>
         * </p>
         * 
         * @param resourceTagLogicalOperator
         *        Specifies whether to combine multiple resource tags with AND, so that a resource must have all tags to
         *        be included or excluded, or OR, so that a resource must have at least one tag.</p>
         *        <p>
         *        Default: <code>AND</code>
         * @see ResourceTagLogicalOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceTagLogicalOperator
         */
        Builder resourceTagLogicalOperator(ResourceTagLogicalOperator resourceTagLogicalOperator);
    }

    static final class BuilderImpl implements Builder {
        private String policyId;

        private String policyName;

        private String policyUpdateToken;

        private SecurityServicePolicyData securityServicePolicyData;

        private String resourceType;

        private List<String> resourceTypeList = DefaultSdkAutoConstructList.getInstance();

        private List<ResourceTag> resourceTags = DefaultSdkAutoConstructList.getInstance();

        private Boolean excludeResourceTags;

        private Boolean remediationEnabled;

        private Boolean deleteUnusedFMManagedResources;

        private Map<String, List<String>> includeMap = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, List<String>> excludeMap = DefaultSdkAutoConstructMap.getInstance();

        private List<String> resourceSetIds = DefaultSdkAutoConstructList.getInstance();

        private String policyDescription;

        private String policyStatus;

        private String resourceTagLogicalOperator;

        private BuilderImpl() {
        }

        private BuilderImpl(Policy model) {
            policyId(model.policyId);
            policyName(model.policyName);
            policyUpdateToken(model.policyUpdateToken);
            securityServicePolicyData(model.securityServicePolicyData);
            resourceType(model.resourceType);
            resourceTypeList(model.resourceTypeList);
            resourceTags(model.resourceTags);
            excludeResourceTags(model.excludeResourceTags);
            remediationEnabled(model.remediationEnabled);
            deleteUnusedFMManagedResources(model.deleteUnusedFMManagedResources);
            includeMapWithStrings(model.includeMap);
            excludeMapWithStrings(model.excludeMap);
            resourceSetIds(model.resourceSetIds);
            policyDescription(model.policyDescription);
            policyStatus(model.policyStatus);
            resourceTagLogicalOperator(model.resourceTagLogicalOperator);
        }

        public final String getPolicyId() {
            return policyId;
        }

        public final void setPolicyId(String policyId) {
            this.policyId = policyId;
        }

        @Override
        public final Builder policyId(String policyId) {
            this.policyId = policyId;
            return this;
        }

        public final String getPolicyName() {
            return policyName;
        }

        public final void setPolicyName(String policyName) {
            this.policyName = policyName;
        }

        @Override
        public final Builder policyName(String policyName) {
            this.policyName = policyName;
            return this;
        }

        public final String getPolicyUpdateToken() {
            return policyUpdateToken;
        }

        public final void setPolicyUpdateToken(String policyUpdateToken) {
            this.policyUpdateToken = policyUpdateToken;
        }

        @Override
        public final Builder policyUpdateToken(String policyUpdateToken) {
            this.policyUpdateToken = policyUpdateToken;
            return this;
        }

        public final SecurityServicePolicyData.Builder getSecurityServicePolicyData() {
            return securityServicePolicyData != null ? securityServicePolicyData.toBuilder() : null;
        }

        public final void setSecurityServicePolicyData(SecurityServicePolicyData.BuilderImpl securityServicePolicyData) {
            this.securityServicePolicyData = securityServicePolicyData != null ? securityServicePolicyData.build() : null;
        }

        @Override
        public final Builder securityServicePolicyData(SecurityServicePolicyData securityServicePolicyData) {
            this.securityServicePolicyData = securityServicePolicyData;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final Collection<String> getResourceTypeList() {
            if (resourceTypeList instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceTypeList;
        }

        public final void setResourceTypeList(Collection<String> resourceTypeList) {
            this.resourceTypeList = ResourceTypeListCopier.copy(resourceTypeList);
        }

        @Override
        public final Builder resourceTypeList(Collection<String> resourceTypeList) {
            this.resourceTypeList = ResourceTypeListCopier.copy(resourceTypeList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypeList(String... resourceTypeList) {
            resourceTypeList(Arrays.asList(resourceTypeList));
            return this;
        }

        public final List<ResourceTag.Builder> getResourceTags() {
            List<ResourceTag.Builder> result = ResourceTagsCopier.copyToBuilder(this.resourceTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourceTags(Collection<ResourceTag.BuilderImpl> resourceTags) {
            this.resourceTags = ResourceTagsCopier.copyFromBuilder(resourceTags);
        }

        @Override
        public final Builder resourceTags(Collection<ResourceTag> resourceTags) {
            this.resourceTags = ResourceTagsCopier.copy(resourceTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(ResourceTag... resourceTags) {
            resourceTags(Arrays.asList(resourceTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(Consumer<ResourceTag.Builder>... resourceTags) {
            resourceTags(Stream.of(resourceTags).map(c -> ResourceTag.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getExcludeResourceTags() {
            return excludeResourceTags;
        }

        public final void setExcludeResourceTags(Boolean excludeResourceTags) {
            this.excludeResourceTags = excludeResourceTags;
        }

        @Override
        public final Builder excludeResourceTags(Boolean excludeResourceTags) {
            this.excludeResourceTags = excludeResourceTags;
            return this;
        }

        public final Boolean getRemediationEnabled() {
            return remediationEnabled;
        }

        public final void setRemediationEnabled(Boolean remediationEnabled) {
            this.remediationEnabled = remediationEnabled;
        }

        @Override
        public final Builder remediationEnabled(Boolean remediationEnabled) {
            this.remediationEnabled = remediationEnabled;
            return this;
        }

        public final Boolean getDeleteUnusedFMManagedResources() {
            return deleteUnusedFMManagedResources;
        }

        public final void setDeleteUnusedFMManagedResources(Boolean deleteUnusedFMManagedResources) {
            this.deleteUnusedFMManagedResources = deleteUnusedFMManagedResources;
        }

        @Override
        public final Builder deleteUnusedFMManagedResources(Boolean deleteUnusedFMManagedResources) {
            this.deleteUnusedFMManagedResources = deleteUnusedFMManagedResources;
            return this;
        }

        public final Map<String, ? extends Collection<String>> getIncludeMap() {
            if (includeMap instanceof SdkAutoConstructMap) {
                return null;
            }
            return includeMap;
        }

        public final void setIncludeMap(Map<String, ? extends Collection<String>> includeMap) {
            this.includeMap = CustomerPolicyScopeMapCopier.copy(includeMap);
        }

        @Override
        public final Builder includeMapWithStrings(Map<String, ? extends Collection<String>> includeMap) {
            this.includeMap = CustomerPolicyScopeMapCopier.copy(includeMap);
            return this;
        }

        @Override
        public final Builder includeMap(Map<CustomerPolicyScopeIdType, ? extends Collection<String>> includeMap) {
            this.includeMap = CustomerPolicyScopeMapCopier.copyEnumToString(includeMap);
            return this;
        }

        public final Map<String, ? extends Collection<String>> getExcludeMap() {
            if (excludeMap instanceof SdkAutoConstructMap) {
                return null;
            }
            return excludeMap;
        }

        public final void setExcludeMap(Map<String, ? extends Collection<String>> excludeMap) {
            this.excludeMap = CustomerPolicyScopeMapCopier.copy(excludeMap);
        }

        @Override
        public final Builder excludeMapWithStrings(Map<String, ? extends Collection<String>> excludeMap) {
            this.excludeMap = CustomerPolicyScopeMapCopier.copy(excludeMap);
            return this;
        }

        @Override
        public final Builder excludeMap(Map<CustomerPolicyScopeIdType, ? extends Collection<String>> excludeMap) {
            this.excludeMap = CustomerPolicyScopeMapCopier.copyEnumToString(excludeMap);
            return this;
        }

        public final Collection<String> getResourceSetIds() {
            if (resourceSetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceSetIds;
        }

        public final void setResourceSetIds(Collection<String> resourceSetIds) {
            this.resourceSetIds = ResourceSetIdsCopier.copy(resourceSetIds);
        }

        @Override
        public final Builder resourceSetIds(Collection<String> resourceSetIds) {
            this.resourceSetIds = ResourceSetIdsCopier.copy(resourceSetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceSetIds(String... resourceSetIds) {
            resourceSetIds(Arrays.asList(resourceSetIds));
            return this;
        }

        public final String getPolicyDescription() {
            return policyDescription;
        }

        public final void setPolicyDescription(String policyDescription) {
            this.policyDescription = policyDescription;
        }

        @Override
        public final Builder policyDescription(String policyDescription) {
            this.policyDescription = policyDescription;
            return this;
        }

        public final String getPolicyStatus() {
            return policyStatus;
        }

        public final void setPolicyStatus(String policyStatus) {
            this.policyStatus = policyStatus;
        }

        @Override
        public final Builder policyStatus(String policyStatus) {
            this.policyStatus = policyStatus;
            return this;
        }

        @Override
        public final Builder policyStatus(CustomerPolicyStatus policyStatus) {
            this.policyStatus(policyStatus == null ? null : policyStatus.toString());
            return this;
        }

        public final String getResourceTagLogicalOperator() {
            return resourceTagLogicalOperator;
        }

        public final void setResourceTagLogicalOperator(String resourceTagLogicalOperator) {
            this.resourceTagLogicalOperator = resourceTagLogicalOperator;
        }

        @Override
        public final Builder resourceTagLogicalOperator(String resourceTagLogicalOperator) {
            this.resourceTagLogicalOperator = resourceTagLogicalOperator;
            return this;
        }

        @Override
        public final Builder resourceTagLogicalOperator(ResourceTagLogicalOperator resourceTagLogicalOperator) {
            this.resourceTagLogicalOperator(resourceTagLogicalOperator == null ? null : resourceTagLogicalOperator.toString());
            return this;
        }

        @Override
        public Policy build() {
            return new Policy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
