/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.forecast.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an additional dataset. This object is part of the <a>DataConfig</a> object. Forecast supports the Weather
 * Index and Holidays additional datasets.
 * </p>
 * <p>
 * <b>Weather Index</b>
 * </p>
 * <p>
 * The Amazon Forecast Weather Index is a built-in dataset that incorporates historical and projected weather
 * information into your model. The Weather Index supplements your datasets with over two years of historical weather
 * data and up to 14 days of projected weather data. For more information, see <a
 * href="https://docs.aws.amazon.com/forecast/latest/dg/weather.html">Amazon Forecast Weather Index</a>.
 * </p>
 * <p>
 * <b>Holidays</b>
 * </p>
 * <p>
 * Holidays is a built-in dataset that incorporates national holiday information into your model. It provides native
 * support for the holiday calendars of 66 countries. To view the holiday calendars, refer to the <a
 * href="http://jollyday.sourceforge.net/data.html">Jollyday</a> library. For more information, see <a
 * href="https://docs.aws.amazon.com/forecast/latest/dg/holidays.html">Holidays Featurization</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdditionalDataset implements SdkPojo, Serializable,
        ToCopyableBuilder<AdditionalDataset.Builder, AdditionalDataset> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(AdditionalDataset::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Map<String, List<String>>> CONFIGURATION_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("Configuration")
            .getter(getter(AdditionalDataset::configuration))
            .setter(setter(Builder::configuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configuration").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Map<String, List<String>> configuration;

    private AdditionalDataset(BuilderImpl builder) {
        this.name = builder.name;
        this.configuration = builder.configuration;
    }

    /**
     * <p>
     * The name of the additional dataset. Valid names: <code>"holiday"</code> and <code>"weather"</code>.
     * </p>
     * 
     * @return The name of the additional dataset. Valid names: <code>"holiday"</code> and <code>"weather"</code>.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Configuration property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasConfiguration() {
        return configuration != null && !(configuration instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * <b>Weather Index</b>
     * </p>
     * <p>
     * To enable the Weather Index, do not specify a value for <code>Configuration</code>.
     * </p>
     * <p>
     * <b>Holidays</b>
     * </p>
     * <p>
     * <b>Holidays</b>
     * </p>
     * <p>
     * To enable Holidays, set <code>CountryCode</code> to one of the following two-letter country codes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * "AL" - ALBANIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "AR" - ARGENTINA
     * </p>
     * </li>
     * <li>
     * <p>
     * "AT" - AUSTRIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "AU" - AUSTRALIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "BA" - BOSNIA HERZEGOVINA
     * </p>
     * </li>
     * <li>
     * <p>
     * "BE" - BELGIUM
     * </p>
     * </li>
     * <li>
     * <p>
     * "BG" - BULGARIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "BO" - BOLIVIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "BR" - BRAZIL
     * </p>
     * </li>
     * <li>
     * <p>
     * "BY" - BELARUS
     * </p>
     * </li>
     * <li>
     * <p>
     * "CA" - CANADA
     * </p>
     * </li>
     * <li>
     * <p>
     * "CL" - CHILE
     * </p>
     * </li>
     * <li>
     * <p>
     * "CO" - COLOMBIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "CR" - COSTA RICA
     * </p>
     * </li>
     * <li>
     * <p>
     * "HR" - CROATIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "CZ" - CZECH REPUBLIC
     * </p>
     * </li>
     * <li>
     * <p>
     * "DK" - DENMARK
     * </p>
     * </li>
     * <li>
     * <p>
     * "EC" - ECUADOR
     * </p>
     * </li>
     * <li>
     * <p>
     * "EE" - ESTONIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "ET" - ETHIOPIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "FI" - FINLAND
     * </p>
     * </li>
     * <li>
     * <p>
     * "FR" - FRANCE
     * </p>
     * </li>
     * <li>
     * <p>
     * "DE" - GERMANY
     * </p>
     * </li>
     * <li>
     * <p>
     * "GR" - GREECE
     * </p>
     * </li>
     * <li>
     * <p>
     * "HU" - HUNGARY
     * </p>
     * </li>
     * <li>
     * <p>
     * "IS" - ICELAND
     * </p>
     * </li>
     * <li>
     * <p>
     * "IN" - INDIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "IE" - IRELAND
     * </p>
     * </li>
     * <li>
     * <p>
     * "IT" - ITALY
     * </p>
     * </li>
     * <li>
     * <p>
     * "JP" - JAPAN
     * </p>
     * </li>
     * <li>
     * <p>
     * "KZ" - KAZAKHSTAN
     * </p>
     * </li>
     * <li>
     * <p>
     * "KR" - KOREA
     * </p>
     * </li>
     * <li>
     * <p>
     * "LV" - LATVIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "LI" - LIECHTENSTEIN
     * </p>
     * </li>
     * <li>
     * <p>
     * "LT" - LITHUANIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "LU" - LUXEMBOURG
     * </p>
     * </li>
     * <li>
     * <p>
     * "MK" - MACEDONIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "MT" - MALTA
     * </p>
     * </li>
     * <li>
     * <p>
     * "MX" - MEXICO
     * </p>
     * </li>
     * <li>
     * <p>
     * "MD" - MOLDOVA
     * </p>
     * </li>
     * <li>
     * <p>
     * "ME" - MONTENEGRO
     * </p>
     * </li>
     * <li>
     * <p>
     * "NL" - NETHERLANDS
     * </p>
     * </li>
     * <li>
     * <p>
     * "NZ" - NEW ZEALAND
     * </p>
     * </li>
     * <li>
     * <p>
     * "NI" - NICARAGUA
     * </p>
     * </li>
     * <li>
     * <p>
     * "NG" - NIGERIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "NO" - NORWAY
     * </p>
     * </li>
     * <li>
     * <p>
     * "PA" - PANAMA
     * </p>
     * </li>
     * <li>
     * <p>
     * "PY" - PARAGUAY
     * </p>
     * </li>
     * <li>
     * <p>
     * "PE" - PERU
     * </p>
     * </li>
     * <li>
     * <p>
     * "PL" - POLAND
     * </p>
     * </li>
     * <li>
     * <p>
     * "PT" - PORTUGAL
     * </p>
     * </li>
     * <li>
     * <p>
     * "RO" - ROMANIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "RU" - RUSSIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "RS" - SERBIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "SK" - SLOVAKIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "SI" - SLOVENIA
     * </p>
     * </li>
     * <li>
     * <p>
     * "ZA" - SOUTH AFRICA
     * </p>
     * </li>
     * <li>
     * <p>
     * "ES" - SPAIN
     * </p>
     * </li>
     * <li>
     * <p>
     * "SE" - SWEDEN
     * </p>
     * </li>
     * <li>
     * <p>
     * "CH" - SWITZERLAND
     * </p>
     * </li>
     * <li>
     * <p>
     * "UA" - UKRAINE
     * </p>
     * </li>
     * <li>
     * <p>
     * "AE" - UNITED ARAB EMIRATES
     * </p>
     * </li>
     * <li>
     * <p>
     * "US" - UNITED STATES
     * </p>
     * </li>
     * <li>
     * <p>
     * "UK" - UNITED KINGDOM
     * </p>
     * </li>
     * <li>
     * <p>
     * "UY" - URUGUAY
     * </p>
     * </li>
     * <li>
     * <p>
     * "VE" - VENEZUELA
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConfiguration} method.
     * </p>
     * 
     * @return <b>Weather Index</b> </p>
     *         <p>
     *         To enable the Weather Index, do not specify a value for <code>Configuration</code>.
     *         </p>
     *         <p>
     *         <b>Holidays</b>
     *         </p>
     *         <p>
     *         <b>Holidays</b>
     *         </p>
     *         <p>
     *         To enable Holidays, set <code>CountryCode</code> to one of the following two-letter country codes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         "AL" - ALBANIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "AR" - ARGENTINA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "AT" - AUSTRIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "AU" - AUSTRALIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "BA" - BOSNIA HERZEGOVINA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "BE" - BELGIUM
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "BG" - BULGARIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "BO" - BOLIVIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "BR" - BRAZIL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "BY" - BELARUS
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "CA" - CANADA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "CL" - CHILE
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "CO" - COLOMBIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "CR" - COSTA RICA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "HR" - CROATIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "CZ" - CZECH REPUBLIC
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "DK" - DENMARK
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "EC" - ECUADOR
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "EE" - ESTONIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "ET" - ETHIOPIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "FI" - FINLAND
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "FR" - FRANCE
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "DE" - GERMANY
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "GR" - GREECE
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "HU" - HUNGARY
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "IS" - ICELAND
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "IN" - INDIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "IE" - IRELAND
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "IT" - ITALY
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "JP" - JAPAN
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "KZ" - KAZAKHSTAN
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "KR" - KOREA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "LV" - LATVIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "LI" - LIECHTENSTEIN
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "LT" - LITHUANIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "LU" - LUXEMBOURG
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "MK" - MACEDONIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "MT" - MALTA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "MX" - MEXICO
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "MD" - MOLDOVA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "ME" - MONTENEGRO
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "NL" - NETHERLANDS
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "NZ" - NEW ZEALAND
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "NI" - NICARAGUA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "NG" - NIGERIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "NO" - NORWAY
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "PA" - PANAMA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "PY" - PARAGUAY
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "PE" - PERU
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "PL" - POLAND
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "PT" - PORTUGAL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "RO" - ROMANIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "RU" - RUSSIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "RS" - SERBIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "SK" - SLOVAKIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "SI" - SLOVENIA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "ZA" - SOUTH AFRICA
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "ES" - SPAIN
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "SE" - SWEDEN
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "CH" - SWITZERLAND
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "UA" - UKRAINE
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "AE" - UNITED ARAB EMIRATES
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "US" - UNITED STATES
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "UK" - UNITED KINGDOM
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "UY" - URUGUAY
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         "VE" - VENEZUELA
     *         </p>
     *         </li>
     */
    public final Map<String, List<String>> configuration() {
        return configuration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasConfiguration() ? configuration() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdditionalDataset)) {
            return false;
        }
        AdditionalDataset other = (AdditionalDataset) obj;
        return Objects.equals(name(), other.name()) && hasConfiguration() == other.hasConfiguration()
                && Objects.equals(configuration(), other.configuration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AdditionalDataset").add("Name", name())
                .add("Configuration", hasConfiguration() ? configuration() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Configuration", CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AdditionalDataset, T> g) {
        return obj -> g.apply((AdditionalDataset) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AdditionalDataset> {
        /**
         * <p>
         * The name of the additional dataset. Valid names: <code>"holiday"</code> and <code>"weather"</code>.
         * </p>
         * 
         * @param name
         *        The name of the additional dataset. Valid names: <code>"holiday"</code> and <code>"weather"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * <b>Weather Index</b>
         * </p>
         * <p>
         * To enable the Weather Index, do not specify a value for <code>Configuration</code>.
         * </p>
         * <p>
         * <b>Holidays</b>
         * </p>
         * <p>
         * <b>Holidays</b>
         * </p>
         * <p>
         * To enable Holidays, set <code>CountryCode</code> to one of the following two-letter country codes:
         * </p>
         * <ul>
         * <li>
         * <p>
         * "AL" - ALBANIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "AR" - ARGENTINA
         * </p>
         * </li>
         * <li>
         * <p>
         * "AT" - AUSTRIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "AU" - AUSTRALIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "BA" - BOSNIA HERZEGOVINA
         * </p>
         * </li>
         * <li>
         * <p>
         * "BE" - BELGIUM
         * </p>
         * </li>
         * <li>
         * <p>
         * "BG" - BULGARIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "BO" - BOLIVIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "BR" - BRAZIL
         * </p>
         * </li>
         * <li>
         * <p>
         * "BY" - BELARUS
         * </p>
         * </li>
         * <li>
         * <p>
         * "CA" - CANADA
         * </p>
         * </li>
         * <li>
         * <p>
         * "CL" - CHILE
         * </p>
         * </li>
         * <li>
         * <p>
         * "CO" - COLOMBIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "CR" - COSTA RICA
         * </p>
         * </li>
         * <li>
         * <p>
         * "HR" - CROATIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "CZ" - CZECH REPUBLIC
         * </p>
         * </li>
         * <li>
         * <p>
         * "DK" - DENMARK
         * </p>
         * </li>
         * <li>
         * <p>
         * "EC" - ECUADOR
         * </p>
         * </li>
         * <li>
         * <p>
         * "EE" - ESTONIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "ET" - ETHIOPIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "FI" - FINLAND
         * </p>
         * </li>
         * <li>
         * <p>
         * "FR" - FRANCE
         * </p>
         * </li>
         * <li>
         * <p>
         * "DE" - GERMANY
         * </p>
         * </li>
         * <li>
         * <p>
         * "GR" - GREECE
         * </p>
         * </li>
         * <li>
         * <p>
         * "HU" - HUNGARY
         * </p>
         * </li>
         * <li>
         * <p>
         * "IS" - ICELAND
         * </p>
         * </li>
         * <li>
         * <p>
         * "IN" - INDIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "IE" - IRELAND
         * </p>
         * </li>
         * <li>
         * <p>
         * "IT" - ITALY
         * </p>
         * </li>
         * <li>
         * <p>
         * "JP" - JAPAN
         * </p>
         * </li>
         * <li>
         * <p>
         * "KZ" - KAZAKHSTAN
         * </p>
         * </li>
         * <li>
         * <p>
         * "KR" - KOREA
         * </p>
         * </li>
         * <li>
         * <p>
         * "LV" - LATVIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "LI" - LIECHTENSTEIN
         * </p>
         * </li>
         * <li>
         * <p>
         * "LT" - LITHUANIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "LU" - LUXEMBOURG
         * </p>
         * </li>
         * <li>
         * <p>
         * "MK" - MACEDONIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "MT" - MALTA
         * </p>
         * </li>
         * <li>
         * <p>
         * "MX" - MEXICO
         * </p>
         * </li>
         * <li>
         * <p>
         * "MD" - MOLDOVA
         * </p>
         * </li>
         * <li>
         * <p>
         * "ME" - MONTENEGRO
         * </p>
         * </li>
         * <li>
         * <p>
         * "NL" - NETHERLANDS
         * </p>
         * </li>
         * <li>
         * <p>
         * "NZ" - NEW ZEALAND
         * </p>
         * </li>
         * <li>
         * <p>
         * "NI" - NICARAGUA
         * </p>
         * </li>
         * <li>
         * <p>
         * "NG" - NIGERIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "NO" - NORWAY
         * </p>
         * </li>
         * <li>
         * <p>
         * "PA" - PANAMA
         * </p>
         * </li>
         * <li>
         * <p>
         * "PY" - PARAGUAY
         * </p>
         * </li>
         * <li>
         * <p>
         * "PE" - PERU
         * </p>
         * </li>
         * <li>
         * <p>
         * "PL" - POLAND
         * </p>
         * </li>
         * <li>
         * <p>
         * "PT" - PORTUGAL
         * </p>
         * </li>
         * <li>
         * <p>
         * "RO" - ROMANIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "RU" - RUSSIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "RS" - SERBIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "SK" - SLOVAKIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "SI" - SLOVENIA
         * </p>
         * </li>
         * <li>
         * <p>
         * "ZA" - SOUTH AFRICA
         * </p>
         * </li>
         * <li>
         * <p>
         * "ES" - SPAIN
         * </p>
         * </li>
         * <li>
         * <p>
         * "SE" - SWEDEN
         * </p>
         * </li>
         * <li>
         * <p>
         * "CH" - SWITZERLAND
         * </p>
         * </li>
         * <li>
         * <p>
         * "UA" - UKRAINE
         * </p>
         * </li>
         * <li>
         * <p>
         * "AE" - UNITED ARAB EMIRATES
         * </p>
         * </li>
         * <li>
         * <p>
         * "US" - UNITED STATES
         * </p>
         * </li>
         * <li>
         * <p>
         * "UK" - UNITED KINGDOM
         * </p>
         * </li>
         * <li>
         * <p>
         * "UY" - URUGUAY
         * </p>
         * </li>
         * <li>
         * <p>
         * "VE" - VENEZUELA
         * </p>
         * </li>
         * </ul>
         * 
         * @param configuration
         *        <b>Weather Index</b> </p>
         *        <p>
         *        To enable the Weather Index, do not specify a value for <code>Configuration</code>.
         *        </p>
         *        <p>
         *        <b>Holidays</b>
         *        </p>
         *        <p>
         *        <b>Holidays</b>
         *        </p>
         *        <p>
         *        To enable Holidays, set <code>CountryCode</code> to one of the following two-letter country codes:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        "AL" - ALBANIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "AR" - ARGENTINA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "AT" - AUSTRIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "AU" - AUSTRALIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "BA" - BOSNIA HERZEGOVINA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "BE" - BELGIUM
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "BG" - BULGARIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "BO" - BOLIVIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "BR" - BRAZIL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "BY" - BELARUS
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "CA" - CANADA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "CL" - CHILE
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "CO" - COLOMBIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "CR" - COSTA RICA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "HR" - CROATIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "CZ" - CZECH REPUBLIC
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "DK" - DENMARK
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "EC" - ECUADOR
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "EE" - ESTONIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "ET" - ETHIOPIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "FI" - FINLAND
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "FR" - FRANCE
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "DE" - GERMANY
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "GR" - GREECE
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "HU" - HUNGARY
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "IS" - ICELAND
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "IN" - INDIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "IE" - IRELAND
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "IT" - ITALY
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "JP" - JAPAN
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "KZ" - KAZAKHSTAN
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "KR" - KOREA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "LV" - LATVIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "LI" - LIECHTENSTEIN
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "LT" - LITHUANIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "LU" - LUXEMBOURG
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "MK" - MACEDONIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "MT" - MALTA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "MX" - MEXICO
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "MD" - MOLDOVA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "ME" - MONTENEGRO
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "NL" - NETHERLANDS
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "NZ" - NEW ZEALAND
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "NI" - NICARAGUA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "NG" - NIGERIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "NO" - NORWAY
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "PA" - PANAMA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "PY" - PARAGUAY
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "PE" - PERU
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "PL" - POLAND
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "PT" - PORTUGAL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "RO" - ROMANIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "RU" - RUSSIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "RS" - SERBIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "SK" - SLOVAKIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "SI" - SLOVENIA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "ZA" - SOUTH AFRICA
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "ES" - SPAIN
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "SE" - SWEDEN
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "CH" - SWITZERLAND
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "UA" - UKRAINE
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "AE" - UNITED ARAB EMIRATES
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "US" - UNITED STATES
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "UK" - UNITED KINGDOM
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "UY" - URUGUAY
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        "VE" - VENEZUELA
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(Map<String, ? extends Collection<String>> configuration);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Map<String, List<String>> configuration = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AdditionalDataset model) {
            name(model.name);
            configuration(model.configuration);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Map<String, ? extends Collection<String>> getConfiguration() {
            if (configuration instanceof SdkAutoConstructMap) {
                return null;
            }
            return configuration;
        }

        public final void setConfiguration(Map<String, ? extends Collection<String>> configuration) {
            this.configuration = ConfigurationCopier.copy(configuration);
        }

        @Override
        public final Builder configuration(Map<String, ? extends Collection<String>> configuration) {
            this.configuration = ConfigurationCopier.copy(configuration);
            return this;
        }

        @Override
        public AdditionalDataset build() {
            return new AdditionalDataset(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
