/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.forecast;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.forecast.model.CreateDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.CreateDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.model.CreateDatasetImportJobRequest;
import software.amazon.awssdk.services.forecast.model.CreateDatasetImportJobResponse;
import software.amazon.awssdk.services.forecast.model.CreateDatasetRequest;
import software.amazon.awssdk.services.forecast.model.CreateDatasetResponse;
import software.amazon.awssdk.services.forecast.model.CreateForecastExportJobRequest;
import software.amazon.awssdk.services.forecast.model.CreateForecastExportJobResponse;
import software.amazon.awssdk.services.forecast.model.CreateForecastRequest;
import software.amazon.awssdk.services.forecast.model.CreateForecastResponse;
import software.amazon.awssdk.services.forecast.model.CreatePredictorRequest;
import software.amazon.awssdk.services.forecast.model.CreatePredictorResponse;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetImportJobRequest;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetImportJobResponse;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetRequest;
import software.amazon.awssdk.services.forecast.model.DeleteDatasetResponse;
import software.amazon.awssdk.services.forecast.model.DeleteForecastExportJobRequest;
import software.amazon.awssdk.services.forecast.model.DeleteForecastExportJobResponse;
import software.amazon.awssdk.services.forecast.model.DeleteForecastRequest;
import software.amazon.awssdk.services.forecast.model.DeleteForecastResponse;
import software.amazon.awssdk.services.forecast.model.DeletePredictorRequest;
import software.amazon.awssdk.services.forecast.model.DeletePredictorResponse;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetImportJobRequest;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetImportJobResponse;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetRequest;
import software.amazon.awssdk.services.forecast.model.DescribeDatasetResponse;
import software.amazon.awssdk.services.forecast.model.DescribeForecastExportJobRequest;
import software.amazon.awssdk.services.forecast.model.DescribeForecastExportJobResponse;
import software.amazon.awssdk.services.forecast.model.DescribeForecastRequest;
import software.amazon.awssdk.services.forecast.model.DescribeForecastResponse;
import software.amazon.awssdk.services.forecast.model.DescribePredictorRequest;
import software.amazon.awssdk.services.forecast.model.DescribePredictorResponse;
import software.amazon.awssdk.services.forecast.model.ForecastException;
import software.amazon.awssdk.services.forecast.model.ForecastRequest;
import software.amazon.awssdk.services.forecast.model.GetAccuracyMetricsRequest;
import software.amazon.awssdk.services.forecast.model.GetAccuracyMetricsResponse;
import software.amazon.awssdk.services.forecast.model.InvalidInputException;
import software.amazon.awssdk.services.forecast.model.InvalidNextTokenException;
import software.amazon.awssdk.services.forecast.model.LimitExceededException;
import software.amazon.awssdk.services.forecast.model.ListDatasetGroupsRequest;
import software.amazon.awssdk.services.forecast.model.ListDatasetGroupsResponse;
import software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsRequest;
import software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsResponse;
import software.amazon.awssdk.services.forecast.model.ListDatasetsRequest;
import software.amazon.awssdk.services.forecast.model.ListDatasetsResponse;
import software.amazon.awssdk.services.forecast.model.ListForecastExportJobsRequest;
import software.amazon.awssdk.services.forecast.model.ListForecastExportJobsResponse;
import software.amazon.awssdk.services.forecast.model.ListForecastsRequest;
import software.amazon.awssdk.services.forecast.model.ListForecastsResponse;
import software.amazon.awssdk.services.forecast.model.ListPredictorsRequest;
import software.amazon.awssdk.services.forecast.model.ListPredictorsResponse;
import software.amazon.awssdk.services.forecast.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.forecast.model.ResourceInUseException;
import software.amazon.awssdk.services.forecast.model.ResourceNotFoundException;
import software.amazon.awssdk.services.forecast.model.UpdateDatasetGroupRequest;
import software.amazon.awssdk.services.forecast.model.UpdateDatasetGroupResponse;
import software.amazon.awssdk.services.forecast.paginators.ListDatasetGroupsPublisher;
import software.amazon.awssdk.services.forecast.paginators.ListDatasetImportJobsPublisher;
import software.amazon.awssdk.services.forecast.paginators.ListDatasetsPublisher;
import software.amazon.awssdk.services.forecast.paginators.ListForecastExportJobsPublisher;
import software.amazon.awssdk.services.forecast.paginators.ListForecastsPublisher;
import software.amazon.awssdk.services.forecast.paginators.ListPredictorsPublisher;
import software.amazon.awssdk.services.forecast.transform.CreateDatasetGroupRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateDatasetImportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateDatasetRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateForecastExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreateForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.CreatePredictorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteDatasetGroupRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteDatasetImportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteDatasetRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteForecastExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeleteForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DeletePredictorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeDatasetGroupRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeDatasetImportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeDatasetRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeForecastExportJobRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribeForecastRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.DescribePredictorRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.GetAccuracyMetricsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListDatasetGroupsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListDatasetImportJobsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListDatasetsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListForecastExportJobsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListForecastsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.ListPredictorsRequestMarshaller;
import software.amazon.awssdk.services.forecast.transform.UpdateDatasetGroupRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ForecastAsyncClient}.
 *
 * @see ForecastAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultForecastAsyncClient implements ForecastAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultForecastAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultForecastAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an Amazon Forecast dataset. The information about the dataset that you provide helps Forecast understand
     * how to consume the data for model training. This includes the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i> <code>DataFrequency</code> </i> - How frequently your historical time-series data is collected. Amazon
     * Forecast uses this information when training the model and generating a forecast.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i> <code>Domain</code> </i> and <i> <code>DatasetType</code> </i> - Each dataset has an associated dataset
     * domain and a type within the domain. Amazon Forecast provides a list of predefined domains and types within each
     * domain. For each unique dataset domain and type within the domain, Amazon Forecast requires your data to include
     * a minimum set of predefined fields.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i> <code>Schema</code> </i> - A schema specifies the fields of the dataset, including the field name and data
     * type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * After creating a dataset, you import your training data into the dataset and add the dataset to a dataset group.
     * You then use the dataset group to create a predictor. For more information, see
     * <a>howitworks-datasets-groups</a>.
     * </p>
     * <p>
     * To get a list of all your datasets, use the <a>ListDatasets</a> operation.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of a dataset must be <code>ACTIVE</code> before you can import training data. Use the
     * <a>DescribeDataset</a> operation to get the status.
     * </p>
     * </note>
     *
     * @param createDatasetRequest
     * @return A Java Future containing the result of the CreateDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with that Amazon Resource Name (ARN). Try
     *         again with a different ARN.</li>
     *         <li>LimitExceededException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateDataset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatasetResponse> createDataset(CreateDatasetRequest createDatasetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetRequest, CreateDatasetResponse>()
                            .withOperationName("CreateDataset")
                            .withMarshaller(new CreateDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDatasetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon Forecast dataset group, which holds a collection of related datasets. You can add datasets to
     * the dataset group when you create the dataset group, or you can add datasets later with the
     * <a>UpdateDatasetGroup</a> operation.
     * </p>
     * <p>
     * After creating a dataset group and adding datasets, you use the dataset group when you create a predictor. For
     * more information, see <a>howitworks-datasets-groups</a>.
     * </p>
     * <p>
     * To get a list of all your datasets groups, use the <a>ListDatasetGroups</a> operation.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of a dataset group must be <code>ACTIVE</code> before you can create a predictor using
     * the dataset group. Use the <a>DescribeDatasetGroup</a> operation to get the status.
     * </p>
     * </note>
     *
     * @param createDatasetGroupRequest
     * @return A Java Future containing the result of the CreateDatasetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with that Amazon Resource Name (ARN). Try
     *         again with a different ARN.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateDatasetGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatasetGroupResponse> createDatasetGroup(CreateDatasetGroupRequest createDatasetGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDatasetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatasetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetGroupRequest, CreateDatasetGroupResponse>()
                            .withOperationName("CreateDatasetGroup")
                            .withMarshaller(new CreateDatasetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDatasetGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports your training data to an Amazon Forecast dataset. You provide the location of your training data in an
     * Amazon Simple Storage Service (Amazon S3) bucket and the Amazon Resource Name (ARN) of the dataset that you want
     * to import the data to.
     * </p>
     * <p>
     * You must specify a <a>DataSource</a> object that includes an AWS Identity and Access Management (IAM) role that
     * Amazon Forecast can assume to access the data. For more information, see <a>aws-forecast-iam-roles</a>.
     * </p>
     * <p>
     * Two properties of the training data are optionally specified:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The delimiter that separates the data fields.
     * </p>
     * <p>
     * The default delimiter is a comma (,), which is the only supported delimiter in this release.
     * </p>
     * </li>
     * <li>
     * <p>
     * The format of timestamps.
     * </p>
     * <p>
     * If the format is not specified, Amazon Forecast expects the format to be "yyyy-MM-dd HH:mm:ss".
     * </p>
     * </li>
     * </ul>
     * <p>
     * When Amazon Forecast uploads your training data, it verifies that the data was collected at the
     * <code>DataFrequency</code> specified when the target dataset was created. For more information, see
     * <a>CreateDataset</a> and <a>howitworks-datasets-groups</a>. Amazon Forecast also verifies the delimiter and
     * timestamp format.
     * </p>
     * <p>
     * You can use the <a>ListDatasetImportJobs</a> operation to get a list of all your dataset import jobs, filtered by
     * specified criteria.
     * </p>
     * <p>
     * To get a list of all your dataset import jobs, filtered by the specified criteria, use the
     * <a>ListDatasetGroups</a> operation.
     * </p>
     *
     * @param createDatasetImportJobRequest
     * @return A Java Future containing the result of the CreateDatasetImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with that Amazon Resource Name (ARN). Try
     *         again with a different ARN.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateDatasetImportJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateDatasetImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatasetImportJobResponse> createDatasetImportJob(
            CreateDatasetImportJobRequest createDatasetImportJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatasetImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDatasetImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatasetImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetImportJobRequest, CreateDatasetImportJobResponse>()
                            .withOperationName("CreateDatasetImportJob")
                            .withMarshaller(new CreateDatasetImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDatasetImportJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a forecast for each item in the <code>TARGET_TIME_SERIES</code> dataset that was used to train the
     * predictor. This is known as inference. To retrieve the forecast for a single item at low latency, use the
     * operation. To export the complete forecast into your Amazon Simple Storage Service (Amazon S3), use the
     * <a>CreateForecastExportJob</a> operation.
     * </p>
     * <p>
     * The range of the forecast is determined by the <code>ForecastHorizon</code>, specified in the
     * <a>CreatePredictor</a> request, multiplied by the <code>DataFrequency</code>, specified in the
     * <a>CreateDataset</a> request. When you query a forecast, you can request a specific date range within the
     * complete forecast.
     * </p>
     * <p>
     * To get a list of all your forecasts, use the <a>ListForecasts</a> operation.
     * </p>
     * <note>
     * <p>
     * The forecasts generated by Amazon Forecast are in the same timezone as the dataset that was used to create the
     * predictor.
     * </p>
     * </note>
     * <p>
     * For more information, see <a>howitworks-forecast</a>.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the forecast must be <code>ACTIVE</code> before you can query or export the forecast.
     * Use the <a>DescribeForecast</a> operation to get the status.
     * </p>
     * </note>
     *
     * @param createForecastRequest
     * @return A Java Future containing the result of the CreateForecast operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with that Amazon Resource Name (ARN). Try
     *         again with a different ARN.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateForecast
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateForecastResponse> createForecast(CreateForecastRequest createForecastRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateForecastResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateForecastRequest, CreateForecastResponse>()
                            .withOperationName("CreateForecast")
                            .withMarshaller(new CreateForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createForecastRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports a forecast created by the <a>CreateForecast</a> operation to your Amazon Simple Storage Service (Amazon
     * S3) bucket.
     * </p>
     * <p>
     * You must specify a <a>DataDestination</a> object that includes an AWS Identity and Access Management (IAM) role
     * that Amazon Forecast can assume to access the Amazon S3 bucket. For more information, see
     * <a>aws-forecast-iam-roles</a>.
     * </p>
     * <p>
     * For more information, see <a>howitworks-forecast</a>.
     * </p>
     * <p>
     * To get a list of all your forecast export jobs, use the <a>ListForecastExportJobs</a> operation.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the forecast export job must be <code>ACTIVE</code> before you can access the forecast
     * in your Amazon S3 bucket. Use the <a>DescribeForecastExportJob</a> operation to get the status.
     * </p>
     * </note>
     *
     * @param createForecastExportJobRequest
     * @return A Java Future containing the result of the CreateForecastExportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with that Amazon Resource Name (ARN). Try
     *         again with a different ARN.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreateForecastExportJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreateForecastExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateForecastExportJobResponse> createForecastExportJob(
            CreateForecastExportJobRequest createForecastExportJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateForecastExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateForecastExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateForecastExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateForecastExportJobRequest, CreateForecastExportJobResponse>()
                            .withOperationName("CreateForecastExportJob")
                            .withMarshaller(new CreateForecastExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createForecastExportJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon Forecast predictor.
     * </p>
     * <p>
     * In the request, you provide a dataset group and either specify an algorithm or let Amazon Forecast choose the
     * algorithm for you using AutoML. If you specify an algorithm, you also can override algorithm-specific
     * hyperparameters.
     * </p>
     * <p>
     * Amazon Forecast uses the chosen algorithm to train a model using the latest version of the datasets in the
     * specified dataset group. The result is called a predictor. You then generate a forecast using the
     * <a>CreateForecast</a> operation.
     * </p>
     * <p>
     * After training a model, the <code>CreatePredictor</code> operation also evaluates it. To see the evaluation
     * metrics, use the <a>GetAccuracyMetrics</a> operation. Always review the evaluation metrics before deciding to use
     * the predictor to generate a forecast.
     * </p>
     * <p>
     * Optionally, you can specify a featurization configuration to fill and aggragate the data fields in the
     * <code>TARGET_TIME_SERIES</code> dataset to improve model training. For more information, see
     * <a>FeaturizationConfig</a>.
     * </p>
     * <p>
     * <b>AutoML</b>
     * </p>
     * <p>
     * If you set <code>PerformAutoML</code> to <code>true</code>, Amazon Forecast evaluates each algorithm and chooses
     * the one that minimizes the <code>objective function</code>. The <code>objective function</code> is defined as the
     * mean of the weighted p10, p50, and p90 quantile losses. For more information, see <a>EvaluationResult</a>.
     * </p>
     * <p>
     * When AutoML is enabled, the following properties are disallowed:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AlgorithmArn</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HPOConfig</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PerformHPO</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TrainingParameters</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To get a list of all your predictors, use the <a>ListPredictors</a> operation.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the predictor must be <code>ACTIVE</code>, signifying that training has completed,
     * before you can use the predictor to create a forecast. Use the <a>DescribePredictor</a> operation to get the
     * status.
     * </p>
     * </note>
     *
     * @param createPredictorRequest
     * @return A Java Future containing the result of the CreatePredictor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceAlreadyExistsException There is already a resource with that Amazon Resource Name (ARN). Try
     *         again with a different ARN.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>LimitExceededException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.CreatePredictor
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/CreatePredictor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePredictorResponse> createPredictor(CreatePredictorRequest createPredictorRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePredictorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePredictorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePredictorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePredictorRequest, CreatePredictorResponse>()
                            .withOperationName("CreatePredictor")
                            .withMarshaller(new CreatePredictorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createPredictorRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon Forecast dataset created using the <a>CreateDataset</a> operation. To be deleted, the dataset
     * must have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. Use the <a>DescribeDataset</a> operation
     * to get the status.
     * </p>
     *
     * @param deleteDatasetRequest
     * @return A Java Future containing the result of the DeleteDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteDataset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetResponse> deleteDataset(DeleteDatasetRequest deleteDatasetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetRequest, DeleteDatasetResponse>()
                            .withOperationName("DeleteDataset")
                            .withMarshaller(new DeleteDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDatasetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a dataset group created using the <a>CreateDatasetGroup</a> operation. To be deleted, the dataset group
     * must have a status of <code>ACTIVE</code>, <code>CREATE_FAILED</code>, or <code>UPDATE_FAILED</code>. Use the
     * <a>DescribeDatasetGroup</a> operation to get the status.
     * </p>
     * <p>
     * The operation deletes only the dataset group, not the datasets in the group.
     * </p>
     *
     * @param deleteDatasetGroupRequest
     * @return A Java Future containing the result of the DeleteDatasetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteDatasetGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetGroupResponse> deleteDatasetGroup(DeleteDatasetGroupRequest deleteDatasetGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDatasetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetGroupRequest, DeleteDatasetGroupResponse>()
                            .withOperationName("DeleteDatasetGroup")
                            .withMarshaller(new DeleteDatasetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDatasetGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a dataset import job created using the <a>CreateDatasetImportJob</a> operation. To be deleted, the import
     * job must have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. Use the
     * <a>DescribeDatasetImportJob</a> operation to get the status.
     * </p>
     *
     * @param deleteDatasetImportJobRequest
     * @return A Java Future containing the result of the DeleteDatasetImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteDatasetImportJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteDatasetImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetImportJobResponse> deleteDatasetImportJob(
            DeleteDatasetImportJobRequest deleteDatasetImportJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDatasetImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetImportJobRequest, DeleteDatasetImportJobResponse>()
                            .withOperationName("DeleteDatasetImportJob")
                            .withMarshaller(new DeleteDatasetImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDatasetImportJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a forecast created using the <a>CreateForecast</a> operation. To be deleted, the forecast must have a
     * status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. Use the <a>DescribeForecast</a> operation to get the
     * status.
     * </p>
     * <p>
     * You can't delete a forecast while it is being exported.
     * </p>
     *
     * @param deleteForecastRequest
     * @return A Java Future containing the result of the DeleteForecast operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteForecast
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteForecastResponse> deleteForecast(DeleteForecastRequest deleteForecastRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteForecastResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteForecastRequest, DeleteForecastResponse>()
                            .withOperationName("DeleteForecast")
                            .withMarshaller(new DeleteForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteForecastRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a forecast export job created using the <a>CreateForecastExportJob</a> operation. To be deleted, the
     * export job must have a status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. Use the
     * <a>DescribeForecastExportJob</a> operation to get the status.
     * </p>
     *
     * @param deleteForecastExportJobRequest
     * @return A Java Future containing the result of the DeleteForecastExportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeleteForecastExportJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeleteForecastExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteForecastExportJobResponse> deleteForecastExportJob(
            DeleteForecastExportJobRequest deleteForecastExportJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteForecastExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteForecastExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteForecastExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteForecastExportJobRequest, DeleteForecastExportJobResponse>()
                            .withOperationName("DeleteForecastExportJob")
                            .withMarshaller(new DeleteForecastExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteForecastExportJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a predictor created using the <a>CreatePredictor</a> operation. To be deleted, the predictor must have a
     * status of <code>ACTIVE</code> or <code>CREATE_FAILED</code>. Use the <a>DescribePredictor</a> operation to get
     * the status.
     * </p>
     * <p>
     * Any forecasts generated by the predictor will no longer be available.
     * </p>
     *
     * @param deletePredictorRequest
     * @return A Java Future containing the result of the DeletePredictor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DeletePredictor
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DeletePredictor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePredictorResponse> deletePredictor(DeletePredictorRequest deletePredictorRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePredictorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePredictorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePredictorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePredictorRequest, DeletePredictorResponse>()
                            .withOperationName("DeletePredictor")
                            .withMarshaller(new DeletePredictorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deletePredictorRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an Amazon Forecast dataset created using the <a>CreateDataset</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided by the user in the <code>CreateDataset</code> request, this
     * operation includes the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeDatasetRequest
     * @return A Java Future containing the result of the DescribeDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeDataset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatasetResponse> describeDataset(DescribeDatasetRequest describeDatasetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetRequest, DescribeDatasetResponse>()
                            .withOperationName("DescribeDataset")
                            .withMarshaller(new DescribeDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDatasetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a dataset group created using the <a>CreateDatasetGroup</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided by the user in the <code>CreateDatasetGroup</code> request, this
     * operation includes the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DatasetArns</code> - The datasets belonging to the group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeDatasetGroupRequest
     * @return A Java Future containing the result of the DescribeDatasetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeDatasetGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatasetGroupResponse> describeDatasetGroup(
            DescribeDatasetGroupRequest describeDatasetGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatasetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatasetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetGroupRequest, DescribeDatasetGroupResponse>()
                            .withOperationName("DescribeDatasetGroup")
                            .withMarshaller(new DescribeDatasetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDatasetGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a dataset import job created using the <a>CreateDatasetImportJob</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided by the user in the <code>CreateDatasetImportJob</code> request,
     * this operation includes the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DataSize</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FieldStatistics</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeDatasetImportJobRequest
     * @return A Java Future containing the result of the DescribeDatasetImportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeDatasetImportJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeDatasetImportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatasetImportJobResponse> describeDatasetImportJob(
            DescribeDatasetImportJobRequest describeDatasetImportJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatasetImportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatasetImportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatasetImportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetImportJobRequest, DescribeDatasetImportJobResponse>()
                            .withOperationName("DescribeDatasetImportJob")
                            .withMarshaller(new DescribeDatasetImportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDatasetImportJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a forecast created using the <a>CreateForecast</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided by the user in the <code>CreateForecast</code> request, this
     * operation includes the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DatasetGroupArn</code> - The dataset group that provided the training data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeForecastRequest
     * @return A Java Future containing the result of the DescribeForecast operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeForecast
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeForecastResponse> describeForecast(DescribeForecastRequest describeForecastRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeForecastResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeForecastRequest, DescribeForecastResponse>()
                            .withOperationName("DescribeForecast")
                            .withMarshaller(new DescribeForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeForecastRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a forecast export job created using the <a>CreateForecastExportJob</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided by the user in the <code>CreateForecastExportJob</code> request,
     * this operation includes the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeForecastExportJobRequest
     * @return A Java Future containing the result of the DescribeForecastExportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribeForecastExportJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribeForecastExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeForecastExportJobResponse> describeForecastExportJob(
            DescribeForecastExportJobRequest describeForecastExportJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeForecastExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeForecastExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeForecastExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeForecastExportJobRequest, DescribeForecastExportJobResponse>()
                            .withOperationName("DescribeForecastExportJob")
                            .withMarshaller(new DescribeForecastExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeForecastExportJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a predictor created using the <a>CreatePredictor</a> operation.
     * </p>
     * <p>
     * In addition to listing the properties provided by the user in the <code>CreatePredictor</code> request, this
     * operation includes the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DatasetImportJobArns</code> - The dataset import jobs used to import training data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutoMLAlgorithmArns</code> - If AutoML is performed, the algorithms evaluated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CreationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LastModificationTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Status</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Message</code> - If an error occurred, information about the error.
     * </p>
     * </li>
     * </ul>
     *
     * @param describePredictorRequest
     * @return A Java Future containing the result of the DescribePredictor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.DescribePredictor
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/DescribePredictor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePredictorResponse> describePredictor(DescribePredictorRequest describePredictorRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePredictorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePredictorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePredictorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePredictorRequest, DescribePredictorResponse>()
                            .withOperationName("DescribePredictor")
                            .withMarshaller(new DescribePredictorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describePredictorRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides metrics on the accuracy of the models that were trained by the <a>CreatePredictor</a> operation. Use
     * metrics to see how well the model performed and to decide whether to use the predictor to generate a forecast.
     * </p>
     * <p>
     * Metrics are generated for each backtest window evaluated. For more information, see <a>EvaluationParameters</a>.
     * </p>
     * <p>
     * The parameters of the <code>filling</code> method determine which items contribute to the metrics. If
     * <code>zero</code> is specified, all items contribute. If <code>nan</code> is specified, only those items that
     * have complete data in the range being evaluated contribute. For more information, see <a>FeaturizationMethod</a>.
     * </p>
     * <p>
     * For an example of how to train a model and review metrics, see <a>getting-started</a>.
     * </p>
     *
     * @param getAccuracyMetricsRequest
     * @return A Java Future containing the result of the GetAccuracyMetrics operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.GetAccuracyMetrics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/GetAccuracyMetrics" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccuracyMetricsResponse> getAccuracyMetrics(GetAccuracyMetricsRequest getAccuracyMetricsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccuracyMetricsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccuracyMetricsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccuracyMetricsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccuracyMetricsRequest, GetAccuracyMetricsResponse>()
                            .withOperationName("GetAccuracyMetrics")
                            .withMarshaller(new GetAccuracyMetricsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getAccuracyMetricsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of dataset groups created using the <a>CreateDatasetGroup</a> operation. For each dataset group, a
     * summary of its properties, including its Amazon Resource Name (ARN), is returned. You can retrieve the complete
     * set of properties by using the ARN with the <a>DescribeDatasetGroup</a> operation.
     * </p>
     *
     * @param listDatasetGroupsRequest
     * @return A Java Future containing the result of the ListDatasetGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListDatasetGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasetGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetGroupsResponse> listDatasetGroups(ListDatasetGroupsRequest listDatasetGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDatasetGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetGroupsRequest, ListDatasetGroupsResponse>()
                            .withOperationName("ListDatasetGroups")
                            .withMarshaller(new ListDatasetGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDatasetGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of dataset groups created using the <a>CreateDatasetGroup</a> operation. For each dataset group, a
     * summary of its properties, including its Amazon Resource Name (ARN), is returned. You can retrieve the complete
     * set of properties by using the ARN with the <a>DescribeDatasetGroup</a> operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatasetGroups(software.amazon.awssdk.services.forecast.model.ListDatasetGroupsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetGroupsPublisher publisher = client.listDatasetGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetGroupsPublisher publisher = client.listDatasetGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.forecast.model.ListDatasetGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.forecast.model.ListDatasetGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatasetGroups(software.amazon.awssdk.services.forecast.model.ListDatasetGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDatasetGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListDatasetGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasetGroups" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDatasetGroupsPublisher listDatasetGroupsPaginator(ListDatasetGroupsRequest listDatasetGroupsRequest) {
        return new ListDatasetGroupsPublisher(this, applyPaginatorUserAgent(listDatasetGroupsRequest));
    }

    /**
     * <p>
     * Returns a list of dataset import jobs created using the <a>CreateDatasetImportJob</a> operation. For each import
     * job, a summary of its properties, including its Amazon Resource Name (ARN), is returned. You can retrieve the
     * complete set of properties by using the ARN with the <a>DescribeDatasetImportJob</a> operation. You can filter
     * the list by providing an array of <a>Filter</a> objects.
     * </p>
     *
     * @param listDatasetImportJobsRequest
     * @return A Java Future containing the result of the ListDatasetImportJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListDatasetImportJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasetImportJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetImportJobsResponse> listDatasetImportJobs(
            ListDatasetImportJobsRequest listDatasetImportJobsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetImportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDatasetImportJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetImportJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetImportJobsRequest, ListDatasetImportJobsResponse>()
                            .withOperationName("ListDatasetImportJobs")
                            .withMarshaller(new ListDatasetImportJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDatasetImportJobsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of dataset import jobs created using the <a>CreateDatasetImportJob</a> operation. For each import
     * job, a summary of its properties, including its Amazon Resource Name (ARN), is returned. You can retrieve the
     * complete set of properties by using the ARN with the <a>DescribeDatasetImportJob</a> operation. You can filter
     * the list by providing an array of <a>Filter</a> objects.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatasetImportJobs(software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetImportJobsPublisher publisher = client.listDatasetImportJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetImportJobsPublisher publisher = client.listDatasetImportJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatasetImportJobs(software.amazon.awssdk.services.forecast.model.ListDatasetImportJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDatasetImportJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListDatasetImportJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasetImportJobs" target="_top">AWS
     *      API Documentation</a>
     */
    public ListDatasetImportJobsPublisher listDatasetImportJobsPaginator(ListDatasetImportJobsRequest listDatasetImportJobsRequest) {
        return new ListDatasetImportJobsPublisher(this, applyPaginatorUserAgent(listDatasetImportJobsRequest));
    }

    /**
     * <p>
     * Returns a list of datasets created using the <a>CreateDataset</a> operation. For each dataset, a summary of its
     * properties, including its Amazon Resource Name (ARN), is returned. You can retrieve the complete set of
     * properties by using the ARN with the <a>DescribeDataset</a> operation.
     * </p>
     *
     * @param listDatasetsRequest
     * @return A Java Future containing the result of the ListDatasets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListDatasets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetsResponse> listDatasets(ListDatasetsRequest listDatasetsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDatasetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetsRequest, ListDatasetsResponse>()
                            .withOperationName("ListDatasets").withMarshaller(new ListDatasetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDatasetsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of datasets created using the <a>CreateDataset</a> operation. For each dataset, a summary of its
     * properties, including its Amazon Resource Name (ARN), is returned. You can retrieve the complete set of
     * properties by using the ARN with the <a>DescribeDataset</a> operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listDatasets(software.amazon.awssdk.services.forecast.model.ListDatasetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetsPublisher publisher = client.listDatasetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListDatasetsPublisher publisher = client.listDatasetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.forecast.model.ListDatasetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.forecast.model.ListDatasetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatasets(software.amazon.awssdk.services.forecast.model.ListDatasetsRequest)} operation.</b>
     * </p>
     *
     * @param listDatasetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListDatasets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListDatasets" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDatasetsPublisher listDatasetsPaginator(ListDatasetsRequest listDatasetsRequest) {
        return new ListDatasetsPublisher(this, applyPaginatorUserAgent(listDatasetsRequest));
    }

    /**
     * <p>
     * Returns a list of forecast export jobs created using the <a>CreateForecastExportJob</a> operation. For each
     * forecast export job, a summary of its properties, including its Amazon Resource Name (ARN), is returned. You can
     * retrieve the complete set of properties by using the ARN with the <a>DescribeForecastExportJob</a> operation. The
     * list can be filtered using an array of <a>Filter</a> objects.
     * </p>
     *
     * @param listForecastExportJobsRequest
     * @return A Java Future containing the result of the ListForecastExportJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListForecastExportJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListForecastExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListForecastExportJobsResponse> listForecastExportJobs(
            ListForecastExportJobsRequest listForecastExportJobsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListForecastExportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListForecastExportJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListForecastExportJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListForecastExportJobsRequest, ListForecastExportJobsResponse>()
                            .withOperationName("ListForecastExportJobs")
                            .withMarshaller(new ListForecastExportJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listForecastExportJobsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of forecast export jobs created using the <a>CreateForecastExportJob</a> operation. For each
     * forecast export job, a summary of its properties, including its Amazon Resource Name (ARN), is returned. You can
     * retrieve the complete set of properties by using the ARN with the <a>DescribeForecastExportJob</a> operation. The
     * list can be filtered using an array of <a>Filter</a> objects.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listForecastExportJobs(software.amazon.awssdk.services.forecast.model.ListForecastExportJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListForecastExportJobsPublisher publisher = client.listForecastExportJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListForecastExportJobsPublisher publisher = client.listForecastExportJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.forecast.model.ListForecastExportJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.forecast.model.ListForecastExportJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listForecastExportJobs(software.amazon.awssdk.services.forecast.model.ListForecastExportJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listForecastExportJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListForecastExportJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListForecastExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListForecastExportJobsPublisher listForecastExportJobsPaginator(
            ListForecastExportJobsRequest listForecastExportJobsRequest) {
        return new ListForecastExportJobsPublisher(this, applyPaginatorUserAgent(listForecastExportJobsRequest));
    }

    /**
     * <p>
     * Returns a list of forecasts created using the <a>CreateForecast</a> operation. For each forecast, a summary of
     * its properties, including its Amazon Resource Name (ARN), is returned. You can retrieve the complete set of
     * properties by using the ARN with the <a>DescribeForecast</a> operation. The list can be filtered using an array
     * of <a>Filter</a> objects.
     * </p>
     *
     * @param listForecastsRequest
     * @return A Java Future containing the result of the ListForecasts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListForecasts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListForecasts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListForecastsResponse> listForecasts(ListForecastsRequest listForecastsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListForecastsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListForecastsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListForecastsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListForecastsRequest, ListForecastsResponse>()
                            .withOperationName("ListForecasts")
                            .withMarshaller(new ListForecastsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listForecastsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of forecasts created using the <a>CreateForecast</a> operation. For each forecast, a summary of
     * its properties, including its Amazon Resource Name (ARN), is returned. You can retrieve the complete set of
     * properties by using the ARN with the <a>DescribeForecast</a> operation. The list can be filtered using an array
     * of <a>Filter</a> objects.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listForecasts(software.amazon.awssdk.services.forecast.model.ListForecastsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListForecastsPublisher publisher = client.listForecastsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListForecastsPublisher publisher = client.listForecastsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.forecast.model.ListForecastsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.forecast.model.ListForecastsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listForecasts(software.amazon.awssdk.services.forecast.model.ListForecastsRequest)} operation.</b>
     * </p>
     *
     * @param listForecastsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListForecasts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListForecasts" target="_top">AWS API
     *      Documentation</a>
     */
    public ListForecastsPublisher listForecastsPaginator(ListForecastsRequest listForecastsRequest) {
        return new ListForecastsPublisher(this, applyPaginatorUserAgent(listForecastsRequest));
    }

    /**
     * <p>
     * Returns a list of predictors created using the <a>CreatePredictor</a> operation. For each predictor, a summary of
     * its properties, including its Amazon Resource Name (ARN), is returned. You can retrieve the complete set of
     * properties by using the ARN with the <a>DescribePredictor</a> operation. The list can be filtered using an array
     * of <a>Filter</a> objects.
     * </p>
     *
     * @param listPredictorsRequest
     * @return A Java Future containing the result of the ListPredictors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListPredictors
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListPredictors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPredictorsResponse> listPredictors(ListPredictorsRequest listPredictorsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPredictorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPredictorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPredictorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPredictorsRequest, ListPredictorsResponse>()
                            .withOperationName("ListPredictors")
                            .withMarshaller(new ListPredictorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listPredictorsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of predictors created using the <a>CreatePredictor</a> operation. For each predictor, a summary of
     * its properties, including its Amazon Resource Name (ARN), is returned. You can retrieve the complete set of
     * properties by using the ARN with the <a>DescribePredictor</a> operation. The list can be filtered using an array
     * of <a>Filter</a> objects.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPredictors(software.amazon.awssdk.services.forecast.model.ListPredictorsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListPredictorsPublisher publisher = client.listPredictorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.forecast.paginators.ListPredictorsPublisher publisher = client.listPredictorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.forecast.model.ListPredictorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.forecast.model.ListPredictorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPredictors(software.amazon.awssdk.services.forecast.model.ListPredictorsRequest)} operation.</b>
     * </p>
     *
     * @param listPredictorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token is not valid. Tokens expire after 24 hours.</li>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.ListPredictors
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/ListPredictors" target="_top">AWS API
     *      Documentation</a>
     */
    public ListPredictorsPublisher listPredictorsPaginator(ListPredictorsRequest listPredictorsRequest) {
        return new ListPredictorsPublisher(this, applyPaginatorUserAgent(listPredictorsRequest));
    }

    /**
     * <p>
     * Replaces any existing datasets in the dataset group with the specified datasets.
     * </p>
     * <note>
     * <p>
     * The <code>Status</code> of the dataset group must be <code>ACTIVE</code> before creating a predictor using the
     * dataset group. Use the <a>DescribeDatasetGroup</a> operation to get the status.
     * </p>
     * </note>
     *
     * @param updateDatasetGroupRequest
     * @return A Java Future containing the result of the UpdateDatasetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException We can't process the request because it includes an invalid value or a value
     *         that exceeds the valid range.</li>
     *         <li>ResourceNotFoundException We can't find a resource with that Amazon Resource Name (ARN). Check the
     *         ARN and try again.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ForecastException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ForecastAsyncClient.UpdateDatasetGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/forecast-2018-06-26/UpdateDatasetGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDatasetGroupResponse> updateDatasetGroup(UpdateDatasetGroupRequest updateDatasetGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDatasetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDatasetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDatasetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDatasetGroupRequest, UpdateDatasetGroupResponse>()
                            .withOperationName("UpdateDatasetGroup")
                            .withMarshaller(new UpdateDatasetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDatasetGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ForecastException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private <T extends ForecastRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
