/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.forecast.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Parameters that define how to split a dataset into training data and testing data, and the number of iterations to
 * perform. These parameters are specified in the predefined algorithms and can be overridden in the
 * <a>CreatePredictor</a> request.
 * </p>
 * <p>
 * For example, suppose that you have a dataset with data collection frequency set to every day and you have 200 days
 * worth of data (that is, 200 data points). Now suppose that you set the <code>NumberOfBacktestWindows</code> to 2 and
 * the <code>BackTestWindowOffset</code> parameter to 20. The algorithm splits the data twice. The first time, the
 * algorithm trains the model using the first 180 data points and uses the last 20 data points for evaluation. The
 * second time, the algorithm trains the model using the first 160 data points and uses the last 40 data points for
 * evaluation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EvaluationParameters implements SdkPojo, Serializable,
        ToCopyableBuilder<EvaluationParameters.Builder, EvaluationParameters> {
    private static final SdkField<Integer> NUMBER_OF_BACKTEST_WINDOWS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(EvaluationParameters::numberOfBacktestWindows)).setter(setter(Builder::numberOfBacktestWindows))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfBacktestWindows").build())
            .build();

    private static final SdkField<Integer> BACK_TEST_WINDOW_OFFSET_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(EvaluationParameters::backTestWindowOffset)).setter(setter(Builder::backTestWindowOffset))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackTestWindowOffset").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            NUMBER_OF_BACKTEST_WINDOWS_FIELD, BACK_TEST_WINDOW_OFFSET_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer numberOfBacktestWindows;

    private final Integer backTestWindowOffset;

    private EvaluationParameters(BuilderImpl builder) {
        this.numberOfBacktestWindows = builder.numberOfBacktestWindows;
        this.backTestWindowOffset = builder.backTestWindowOffset;
    }

    /**
     * <p>
     * The number of times to split the input data. The default is 1. The range is 1 through 5.
     * </p>
     * 
     * @return The number of times to split the input data. The default is 1. The range is 1 through 5.
     */
    public Integer numberOfBacktestWindows() {
        return numberOfBacktestWindows;
    }

    /**
     * <p>
     * The point from the end of the dataset where you want to split the data for model training and evaluation. The
     * value is specified as the number of data points.
     * </p>
     * 
     * @return The point from the end of the dataset where you want to split the data for model training and evaluation.
     *         The value is specified as the number of data points.
     */
    public Integer backTestWindowOffset() {
        return backTestWindowOffset;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(numberOfBacktestWindows());
        hashCode = 31 * hashCode + Objects.hashCode(backTestWindowOffset());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EvaluationParameters)) {
            return false;
        }
        EvaluationParameters other = (EvaluationParameters) obj;
        return Objects.equals(numberOfBacktestWindows(), other.numberOfBacktestWindows())
                && Objects.equals(backTestWindowOffset(), other.backTestWindowOffset());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("EvaluationParameters").add("NumberOfBacktestWindows", numberOfBacktestWindows())
                .add("BackTestWindowOffset", backTestWindowOffset()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NumberOfBacktestWindows":
            return Optional.ofNullable(clazz.cast(numberOfBacktestWindows()));
        case "BackTestWindowOffset":
            return Optional.ofNullable(clazz.cast(backTestWindowOffset()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EvaluationParameters, T> g) {
        return obj -> g.apply((EvaluationParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EvaluationParameters> {
        /**
         * <p>
         * The number of times to split the input data. The default is 1. The range is 1 through 5.
         * </p>
         * 
         * @param numberOfBacktestWindows
         *        The number of times to split the input data. The default is 1. The range is 1 through 5.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfBacktestWindows(Integer numberOfBacktestWindows);

        /**
         * <p>
         * The point from the end of the dataset where you want to split the data for model training and evaluation. The
         * value is specified as the number of data points.
         * </p>
         * 
         * @param backTestWindowOffset
         *        The point from the end of the dataset where you want to split the data for model training and
         *        evaluation. The value is specified as the number of data points.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backTestWindowOffset(Integer backTestWindowOffset);
    }

    static final class BuilderImpl implements Builder {
        private Integer numberOfBacktestWindows;

        private Integer backTestWindowOffset;

        private BuilderImpl() {
        }

        private BuilderImpl(EvaluationParameters model) {
            numberOfBacktestWindows(model.numberOfBacktestWindows);
            backTestWindowOffset(model.backTestWindowOffset);
        }

        public final Integer getNumberOfBacktestWindows() {
            return numberOfBacktestWindows;
        }

        @Override
        public final Builder numberOfBacktestWindows(Integer numberOfBacktestWindows) {
            this.numberOfBacktestWindows = numberOfBacktestWindows;
            return this;
        }

        public final void setNumberOfBacktestWindows(Integer numberOfBacktestWindows) {
            this.numberOfBacktestWindows = numberOfBacktestWindows;
        }

        public final Integer getBackTestWindowOffset() {
            return backTestWindowOffset;
        }

        @Override
        public final Builder backTestWindowOffset(Integer backTestWindowOffset) {
            this.backTestWindowOffset = backTestWindowOffset;
            return this;
        }

        public final void setBackTestWindowOffset(Integer backTestWindowOffset) {
            this.backTestWindowOffset = backTestWindowOffset;
        }

        @Override
        public EvaluationParameters build() {
            return new EvaluationParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
