/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.freetier;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.freetier.internal.FreeTierServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.freetier.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.freetier.model.AccessDeniedException;
import software.amazon.awssdk.services.freetier.model.FreeTierException;
import software.amazon.awssdk.services.freetier.model.GetAccountActivityRequest;
import software.amazon.awssdk.services.freetier.model.GetAccountActivityResponse;
import software.amazon.awssdk.services.freetier.model.GetAccountPlanStateRequest;
import software.amazon.awssdk.services.freetier.model.GetAccountPlanStateResponse;
import software.amazon.awssdk.services.freetier.model.GetFreeTierUsageRequest;
import software.amazon.awssdk.services.freetier.model.GetFreeTierUsageResponse;
import software.amazon.awssdk.services.freetier.model.InternalServerException;
import software.amazon.awssdk.services.freetier.model.ListAccountActivitiesRequest;
import software.amazon.awssdk.services.freetier.model.ListAccountActivitiesResponse;
import software.amazon.awssdk.services.freetier.model.ResourceNotFoundException;
import software.amazon.awssdk.services.freetier.model.ThrottlingException;
import software.amazon.awssdk.services.freetier.model.UpgradeAccountPlanRequest;
import software.amazon.awssdk.services.freetier.model.UpgradeAccountPlanResponse;
import software.amazon.awssdk.services.freetier.model.ValidationException;
import software.amazon.awssdk.services.freetier.transform.GetAccountActivityRequestMarshaller;
import software.amazon.awssdk.services.freetier.transform.GetAccountPlanStateRequestMarshaller;
import software.amazon.awssdk.services.freetier.transform.GetFreeTierUsageRequestMarshaller;
import software.amazon.awssdk.services.freetier.transform.ListAccountActivitiesRequestMarshaller;
import software.amazon.awssdk.services.freetier.transform.UpgradeAccountPlanRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link FreeTierClient}.
 *
 * @see FreeTierClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFreeTierClient implements FreeTierClient {
    private static final Logger log = Logger.loggerFor(DefaultFreeTierClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultFreeTierClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "FreeTier" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns a specific activity record that is available to the customer.
     * </p>
     *
     * @param getAccountActivityRequest
     * @return Result of the GetAccountActivity operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the requested resource cannot be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.GetAccountActivity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/GetAccountActivity" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetAccountActivityResponse getAccountActivity(GetAccountActivityRequest getAccountActivityRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, FreeTierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAccountActivityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAccountActivityResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountActivityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountActivityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FreeTier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountActivity");

            return clientHandler.execute(new ClientExecutionParams<GetAccountActivityRequest, GetAccountActivityResponse>()
                    .withOperationName("GetAccountActivity").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAccountActivityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAccountActivityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This returns all of the information related to the state of the account plan related to Free Tier.
     * </p>
     *
     * @param getAccountPlanStateRequest
     * @return Result of the GetAccountPlanState operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the requested resource cannot be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.GetAccountPlanState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/GetAccountPlanState" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetAccountPlanStateResponse getAccountPlanState(GetAccountPlanStateRequest getAccountPlanStateRequest)
            throws InternalServerException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, FreeTierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAccountPlanStateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAccountPlanStateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountPlanStateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountPlanStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FreeTier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountPlanState");

            return clientHandler.execute(new ClientExecutionParams<GetAccountPlanStateRequest, GetAccountPlanStateResponse>()
                    .withOperationName("GetAccountPlanState").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAccountPlanStateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAccountPlanStateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all Free Tier usage objects that match your filters.
     * </p>
     *
     * @param getFreeTierUsageRequest
     * @return Result of the GetFreeTierUsage operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.GetFreeTierUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/GetFreeTierUsage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetFreeTierUsageResponse getFreeTierUsage(GetFreeTierUsageRequest getFreeTierUsageRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            FreeTierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFreeTierUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetFreeTierUsageResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getFreeTierUsageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFreeTierUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FreeTier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFreeTierUsage");

            return clientHandler.execute(new ClientExecutionParams<GetFreeTierUsageRequest, GetFreeTierUsageResponse>()
                    .withOperationName("GetFreeTierUsage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getFreeTierUsageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetFreeTierUsageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of activities that are available. This operation supports pagination and filtering by status.
     * </p>
     *
     * @param listAccountActivitiesRequest
     * @return Result of the ListAccountActivities operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.ListAccountActivities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/ListAccountActivities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAccountActivitiesResponse listAccountActivities(ListAccountActivitiesRequest listAccountActivitiesRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            FreeTierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAccountActivitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAccountActivitiesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAccountActivitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccountActivitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FreeTier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccountActivities");

            return clientHandler.execute(new ClientExecutionParams<ListAccountActivitiesRequest, ListAccountActivitiesResponse>()
                    .withOperationName("ListAccountActivities").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAccountActivitiesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAccountActivitiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The account plan type for the Amazon Web Services account.
     * </p>
     *
     * @param upgradeAccountPlanRequest
     * @return Result of the UpgradeAccountPlan operation returned by the service.
     * @throws InternalServerException
     *         An unexpected error occurred during the processing of your request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the requested resource cannot be found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FreeTierException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FreeTierClient.UpgradeAccountPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/freetier-2023-09-07/UpgradeAccountPlan" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpgradeAccountPlanResponse upgradeAccountPlan(UpgradeAccountPlanRequest upgradeAccountPlanRequest)
            throws InternalServerException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, FreeTierException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpgradeAccountPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpgradeAccountPlanResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(upgradeAccountPlanRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, upgradeAccountPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FreeTier");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpgradeAccountPlan");

            return clientHandler.execute(new ClientExecutionParams<UpgradeAccountPlanRequest, UpgradeAccountPlanResponse>()
                    .withOperationName("UpgradeAccountPlan").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(upgradeAccountPlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpgradeAccountPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        FreeTierServiceClientConfigurationBuilder serviceConfigBuilder = new FreeTierServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(FreeTierException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.0");
    }

    @Override
    public final FreeTierServiceClientConfiguration serviceClientConfiguration() {
        return new FreeTierServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
