/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A description of a specific Amazon FSx file system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FileSystem implements SdkPojo, Serializable, ToCopyableBuilder<FileSystem.Builder, FileSystem> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerId").getter(getter(FileSystem::ownerId)).setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(FileSystem::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> FILE_SYSTEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemId").getter(getter(FileSystem::fileSystemId)).setter(setter(Builder::fileSystemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemId").build()).build();

    private static final SdkField<String> FILE_SYSTEM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemType").getter(getter(FileSystem::fileSystemTypeAsString))
            .setter(setter(Builder::fileSystemType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemType").build()).build();

    private static final SdkField<String> LIFECYCLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Lifecycle").getter(getter(FileSystem::lifecycleAsString)).setter(setter(Builder::lifecycle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<FileSystemFailureDetails> FAILURE_DETAILS_FIELD = SdkField
            .<FileSystemFailureDetails> builder(MarshallingType.SDK_POJO).memberName("FailureDetails")
            .getter(getter(FileSystem::failureDetails)).setter(setter(Builder::failureDetails))
            .constructor(FileSystemFailureDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureDetails").build()).build();

    private static final SdkField<Integer> STORAGE_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("StorageCapacity").getter(getter(FileSystem::storageCapacity)).setter(setter(Builder::storageCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageCapacity").build()).build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StorageType").getter(getter(FileSystem::storageTypeAsString)).setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(FileSystem::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SubnetIds")
            .getter(getter(FileSystem::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> NETWORK_INTERFACE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NetworkInterfaceIds")
            .getter(getter(FileSystem::networkInterfaceIds))
            .setter(setter(Builder::networkInterfaceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DNSName").getter(getter(FileSystem::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DNSName").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(FileSystem::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceARN").getter(getter(FileSystem::resourceARN)).setter(setter(Builder::resourceARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceARN").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(FileSystem::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<WindowsFileSystemConfiguration> WINDOWS_CONFIGURATION_FIELD = SdkField
            .<WindowsFileSystemConfiguration> builder(MarshallingType.SDK_POJO).memberName("WindowsConfiguration")
            .getter(getter(FileSystem::windowsConfiguration)).setter(setter(Builder::windowsConfiguration))
            .constructor(WindowsFileSystemConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowsConfiguration").build())
            .build();

    private static final SdkField<LustreFileSystemConfiguration> LUSTRE_CONFIGURATION_FIELD = SdkField
            .<LustreFileSystemConfiguration> builder(MarshallingType.SDK_POJO).memberName("LustreConfiguration")
            .getter(getter(FileSystem::lustreConfiguration)).setter(setter(Builder::lustreConfiguration))
            .constructor(LustreFileSystemConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LustreConfiguration").build())
            .build();

    private static final SdkField<List<AdministrativeAction>> ADMINISTRATIVE_ACTIONS_FIELD = SdkField
            .<List<AdministrativeAction>> builder(MarshallingType.LIST)
            .memberName("AdministrativeActions")
            .getter(getter(FileSystem::administrativeActions))
            .setter(setter(Builder::administrativeActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdministrativeActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AdministrativeAction> builder(MarshallingType.SDK_POJO)
                                            .constructor(AdministrativeAction::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD,
            CREATION_TIME_FIELD, FILE_SYSTEM_ID_FIELD, FILE_SYSTEM_TYPE_FIELD, LIFECYCLE_FIELD, FAILURE_DETAILS_FIELD,
            STORAGE_CAPACITY_FIELD, STORAGE_TYPE_FIELD, VPC_ID_FIELD, SUBNET_IDS_FIELD, NETWORK_INTERFACE_IDS_FIELD,
            DNS_NAME_FIELD, KMS_KEY_ID_FIELD, RESOURCE_ARN_FIELD, TAGS_FIELD, WINDOWS_CONFIGURATION_FIELD,
            LUSTRE_CONFIGURATION_FIELD, ADMINISTRATIVE_ACTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final Instant creationTime;

    private final String fileSystemId;

    private final String fileSystemType;

    private final String lifecycle;

    private final FileSystemFailureDetails failureDetails;

    private final Integer storageCapacity;

    private final String storageType;

    private final String vpcId;

    private final List<String> subnetIds;

    private final List<String> networkInterfaceIds;

    private final String dnsName;

    private final String kmsKeyId;

    private final String resourceARN;

    private final List<Tag> tags;

    private final WindowsFileSystemConfiguration windowsConfiguration;

    private final LustreFileSystemConfiguration lustreConfiguration;

    private final List<AdministrativeAction> administrativeActions;

    private FileSystem(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.creationTime = builder.creationTime;
        this.fileSystemId = builder.fileSystemId;
        this.fileSystemType = builder.fileSystemType;
        this.lifecycle = builder.lifecycle;
        this.failureDetails = builder.failureDetails;
        this.storageCapacity = builder.storageCapacity;
        this.storageType = builder.storageType;
        this.vpcId = builder.vpcId;
        this.subnetIds = builder.subnetIds;
        this.networkInterfaceIds = builder.networkInterfaceIds;
        this.dnsName = builder.dnsName;
        this.kmsKeyId = builder.kmsKeyId;
        this.resourceARN = builder.resourceARN;
        this.tags = builder.tags;
        this.windowsConfiguration = builder.windowsConfiguration;
        this.lustreConfiguration = builder.lustreConfiguration;
        this.administrativeActions = builder.administrativeActions;
    }

    /**
     * <p>
     * The AWS account that created the file system. If the file system was created by an AWS Identity and Access
     * Management (IAM) user, the AWS account to which the IAM user belongs is the owner.
     * </p>
     * 
     * @return The AWS account that created the file system. If the file system was created by an AWS Identity and
     *         Access Management (IAM) user, the AWS account to which the IAM user belongs is the owner.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The time that the file system was created, in seconds (since 1970-01-01T00:00:00Z), also known as Unix time.
     * </p>
     * 
     * @return The time that the file system was created, in seconds (since 1970-01-01T00:00:00Z), also known as Unix
     *         time.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The system-generated, unique 17-digit ID of the file system.
     * </p>
     * 
     * @return The system-generated, unique 17-digit ID of the file system.
     */
    public final String fileSystemId() {
        return fileSystemId;
    }

    /**
     * <p>
     * The type of Amazon FSx file system, either <code>LUSTRE</code> or <code>WINDOWS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileSystemType}
     * will return {@link FileSystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fileSystemTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon FSx file system, either <code>LUSTRE</code> or <code>WINDOWS</code>.
     * @see FileSystemType
     */
    public final FileSystemType fileSystemType() {
        return FileSystemType.fromValue(fileSystemType);
    }

    /**
     * <p>
     * The type of Amazon FSx file system, either <code>LUSTRE</code> or <code>WINDOWS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileSystemType}
     * will return {@link FileSystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fileSystemTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon FSx file system, either <code>LUSTRE</code> or <code>WINDOWS</code>.
     * @see FileSystemType
     */
    public final String fileSystemTypeAsString() {
        return fileSystemType;
    }

    /**
     * <p>
     * The lifecycle status of the file system, following are the possible values and what they mean:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> - Amazon FSx is creating the new file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - Amazon FSx is deleting an existing file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a new file
     * system, Amazon FSx was unable to create the file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> indicates that the file system is in a failed but recoverable state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> indicates that the file system is undergoing a customer initiated update.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link FileSystemLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return The lifecycle status of the file system, following are the possible values and what they mean:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - Amazon FSx is creating the new file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - Amazon FSx is deleting an existing file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a
     *         new file system, Amazon FSx was unable to create the file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> indicates that the file system is in a failed but recoverable state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> indicates that the file system is undergoing a customer initiated update.
     *         </p>
     *         </li>
     * @see FileSystemLifecycle
     */
    public final FileSystemLifecycle lifecycle() {
        return FileSystemLifecycle.fromValue(lifecycle);
    }

    /**
     * <p>
     * The lifecycle status of the file system, following are the possible values and what they mean:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> - Amazon FSx is creating the new file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - Amazon FSx is deleting an existing file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a new file
     * system, Amazon FSx was unable to create the file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> indicates that the file system is in a failed but recoverable state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> indicates that the file system is undergoing a customer initiated update.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link FileSystemLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return The lifecycle status of the file system, following are the possible values and what they mean:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - Amazon FSx is creating the new file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - Amazon FSx is deleting an existing file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a
     *         new file system, Amazon FSx was unable to create the file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> indicates that the file system is in a failed but recoverable state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> indicates that the file system is undergoing a customer initiated update.
     *         </p>
     *         </li>
     * @see FileSystemLifecycle
     */
    public final String lifecycleAsString() {
        return lifecycle;
    }

    /**
     * Returns the value of the FailureDetails property for this object.
     * 
     * @return The value of the FailureDetails property for this object.
     */
    public final FileSystemFailureDetails failureDetails() {
        return failureDetails;
    }

    /**
     * <p>
     * The storage capacity of the file system in gibibytes (GiB).
     * </p>
     * 
     * @return The storage capacity of the file system in gibibytes (GiB).
     */
    public final Integer storageCapacity() {
        return storageCapacity;
    }

    /**
     * <p>
     * The storage type of the file system. Valid values are <code>SSD</code> and <code>HDD</code>. If set to
     * <code>SSD</code>, the file system uses solid state drive storage. If set to <code>HDD</code>, the file system
     * uses hard disk drive storage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageType} will
     * return {@link StorageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTypeAsString}.
     * </p>
     * 
     * @return The storage type of the file system. Valid values are <code>SSD</code> and <code>HDD</code>. If set to
     *         <code>SSD</code>, the file system uses solid state drive storage. If set to <code>HDD</code>, the file
     *         system uses hard disk drive storage.
     * @see StorageType
     */
    public final StorageType storageType() {
        return StorageType.fromValue(storageType);
    }

    /**
     * <p>
     * The storage type of the file system. Valid values are <code>SSD</code> and <code>HDD</code>. If set to
     * <code>SSD</code>, the file system uses solid state drive storage. If set to <code>HDD</code>, the file system
     * uses hard disk drive storage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageType} will
     * return {@link StorageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTypeAsString}.
     * </p>
     * 
     * @return The storage type of the file system. Valid values are <code>SSD</code> and <code>HDD</code>. If set to
     *         <code>SSD</code>, the file system uses solid state drive storage. If set to <code>HDD</code>, the file
     *         system uses hard disk drive storage.
     * @see StorageType
     */
    public final String storageTypeAsString() {
        return storageType;
    }

    /**
     * <p>
     * The ID of the primary VPC for the file system.
     * </p>
     * 
     * @return The ID of the primary VPC for the file system.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * Returns true if the SubnetIds property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the IDs of the subnets that the file system is accessible from. For Windows <code>MULTI_AZ_1</code>
     * file system deployment type, there are two subnet IDs, one for the preferred file server and one for the standby
     * file server. The preferred file server subnet identified in the <code>PreferredSubnetID</code> property. All
     * other file systems have only one subnet ID.
     * </p>
     * <p>
     * For Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that contains the
     * endpoint for the file system. For <code>MULTI_AZ_1</code> Windows file systems, the endpoint for the file system
     * is available in the <code>PreferredSubnetID</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubnetIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the IDs of the subnets that the file system is accessible from. For Windows
     *         <code>MULTI_AZ_1</code> file system deployment type, there are two subnet IDs, one for the preferred file
     *         server and one for the standby file server. The preferred file server subnet identified in the
     *         <code>PreferredSubnetID</code> property. All other file systems have only one subnet ID.</p>
     *         <p>
     *         For Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that contains
     *         the endpoint for the file system. For <code>MULTI_AZ_1</code> Windows file systems, the endpoint for the
     *         file system is available in the <code>PreferredSubnetID</code>.
     */
    public final List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * Returns true if the NetworkInterfaceIds property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasNetworkInterfaceIds() {
        return networkInterfaceIds != null && !(networkInterfaceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the elastic network interface from which a specific file system is accessible. The elastic network
     * interface is automatically created in the same VPC that the Amazon FSx file system was created in. For more
     * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network
     * Interfaces</a> in the <i>Amazon EC2 User Guide.</i>
     * </p>
     * <p>
     * For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an Amazon FSx
     * for Lustre file system, you can have more than one.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNetworkInterfaceIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The IDs of the elastic network interface from which a specific file system is accessible. The elastic
     *         network interface is automatically created in the same VPC that the Amazon FSx file system was created
     *         in. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network Interfaces</a>
     *         in the <i>Amazon EC2 User Guide.</i> </p>
     *         <p>
     *         For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an
     *         Amazon FSx for Lustre file system, you can have more than one.
     */
    public final List<String> networkInterfaceIds() {
        return networkInterfaceIds;
    }

    /**
     * <p>
     * The DNS name for the file system.
     * </p>
     * 
     * @return The DNS name for the file system.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * The ID of the AWS Key Management Service (AWS KMS) key used to encrypt the file system's data for Amazon FSx for
     * Windows File Server file systems and persistent Amazon FSx for Lustre file systems at rest. In either case, if
     * not specified, the Amazon FSx managed key is used. The scratch Amazon FSx for Lustre file systems are always
     * encrypted at rest using Amazon FSx managed keys. For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html">Encrypt</a> in the <i>AWS Key
     * Management Service API Reference</i>.
     * </p>
     * 
     * @return The ID of the AWS Key Management Service (AWS KMS) key used to encrypt the file system's data for Amazon
     *         FSx for Windows File Server file systems and persistent Amazon FSx for Lustre file systems at rest. In
     *         either case, if not specified, the Amazon FSx managed key is used. The scratch Amazon FSx for Lustre file
     *         systems are always encrypted at rest using Amazon FSx managed keys. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html">Encrypt</a> in the <i>AWS Key
     *         Management Service API Reference</i>.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the file system resource.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the file system resource.
     */
    public final String resourceARN() {
        return resourceARN;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to associate with the file system. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html">Tagging Your Amazon EC2 Resources</a>
     * in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tags to associate with the file system. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html">Tagging Your Amazon EC2
     *         Resources</a> in the <i>Amazon EC2 User Guide</i>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The configuration for this Microsoft Windows file system.
     * </p>
     * 
     * @return The configuration for this Microsoft Windows file system.
     */
    public final WindowsFileSystemConfiguration windowsConfiguration() {
        return windowsConfiguration;
    }

    /**
     * Returns the value of the LustreConfiguration property for this object.
     * 
     * @return The value of the LustreConfiguration property for this object.
     */
    public final LustreFileSystemConfiguration lustreConfiguration() {
        return lustreConfiguration;
    }

    /**
     * Returns true if the AdministrativeActions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasAdministrativeActions() {
        return administrativeActions != null && !(administrativeActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of administrative actions for the file system that are in process or waiting to be processed.
     * Administrative actions describe changes to the Windows file system that you have initiated using the
     * <code>UpdateFileSystem</code> action.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAdministrativeActions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of administrative actions for the file system that are in process or waiting to be processed.
     *         Administrative actions describe changes to the Windows file system that you have initiated using the
     *         <code>UpdateFileSystem</code> action.
     */
    public final List<AdministrativeAction> administrativeActions() {
        return administrativeActions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureDetails());
        hashCode = 31 * hashCode + Objects.hashCode(storageCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(storageTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetIds() ? subnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaceIds() ? networkInterfaceIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceARN());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(windowsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(lustreConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasAdministrativeActions() ? administrativeActions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FileSystem)) {
            return false;
        }
        FileSystem other = (FileSystem) obj;
        return Objects.equals(ownerId(), other.ownerId()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(fileSystemId(), other.fileSystemId())
                && Objects.equals(fileSystemTypeAsString(), other.fileSystemTypeAsString())
                && Objects.equals(lifecycleAsString(), other.lifecycleAsString())
                && Objects.equals(failureDetails(), other.failureDetails())
                && Objects.equals(storageCapacity(), other.storageCapacity())
                && Objects.equals(storageTypeAsString(), other.storageTypeAsString()) && Objects.equals(vpcId(), other.vpcId())
                && hasSubnetIds() == other.hasSubnetIds() && Objects.equals(subnetIds(), other.subnetIds())
                && hasNetworkInterfaceIds() == other.hasNetworkInterfaceIds()
                && Objects.equals(networkInterfaceIds(), other.networkInterfaceIds())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(resourceARN(), other.resourceARN()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(windowsConfiguration(), other.windowsConfiguration())
                && Objects.equals(lustreConfiguration(), other.lustreConfiguration())
                && hasAdministrativeActions() == other.hasAdministrativeActions()
                && Objects.equals(administrativeActions(), other.administrativeActions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FileSystem").add("OwnerId", ownerId()).add("CreationTime", creationTime())
                .add("FileSystemId", fileSystemId()).add("FileSystemType", fileSystemTypeAsString())
                .add("Lifecycle", lifecycleAsString()).add("FailureDetails", failureDetails())
                .add("StorageCapacity", storageCapacity()).add("StorageType", storageTypeAsString()).add("VpcId", vpcId())
                .add("SubnetIds", hasSubnetIds() ? subnetIds() : null)
                .add("NetworkInterfaceIds", hasNetworkInterfaceIds() ? networkInterfaceIds() : null).add("DNSName", dnsName())
                .add("KmsKeyId", kmsKeyId()).add("ResourceARN", resourceARN()).add("Tags", hasTags() ? tags() : null)
                .add("WindowsConfiguration", windowsConfiguration()).add("LustreConfiguration", lustreConfiguration())
                .add("AdministrativeActions", hasAdministrativeActions() ? administrativeActions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "FileSystemId":
            return Optional.ofNullable(clazz.cast(fileSystemId()));
        case "FileSystemType":
            return Optional.ofNullable(clazz.cast(fileSystemTypeAsString()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycleAsString()));
        case "FailureDetails":
            return Optional.ofNullable(clazz.cast(failureDetails()));
        case "StorageCapacity":
            return Optional.ofNullable(clazz.cast(storageCapacity()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageTypeAsString()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SubnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "NetworkInterfaceIds":
            return Optional.ofNullable(clazz.cast(networkInterfaceIds()));
        case "DNSName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "ResourceARN":
            return Optional.ofNullable(clazz.cast(resourceARN()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "WindowsConfiguration":
            return Optional.ofNullable(clazz.cast(windowsConfiguration()));
        case "LustreConfiguration":
            return Optional.ofNullable(clazz.cast(lustreConfiguration()));
        case "AdministrativeActions":
            return Optional.ofNullable(clazz.cast(administrativeActions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FileSystem, T> g) {
        return obj -> g.apply((FileSystem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FileSystem> {
        /**
         * <p>
         * The AWS account that created the file system. If the file system was created by an AWS Identity and Access
         * Management (IAM) user, the AWS account to which the IAM user belongs is the owner.
         * </p>
         * 
         * @param ownerId
         *        The AWS account that created the file system. If the file system was created by an AWS Identity and
         *        Access Management (IAM) user, the AWS account to which the IAM user belongs is the owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The time that the file system was created, in seconds (since 1970-01-01T00:00:00Z), also known as Unix time.
         * </p>
         * 
         * @param creationTime
         *        The time that the file system was created, in seconds (since 1970-01-01T00:00:00Z), also known as Unix
         *        time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The system-generated, unique 17-digit ID of the file system.
         * </p>
         * 
         * @param fileSystemId
         *        The system-generated, unique 17-digit ID of the file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemId(String fileSystemId);

        /**
         * <p>
         * The type of Amazon FSx file system, either <code>LUSTRE</code> or <code>WINDOWS</code>.
         * </p>
         * 
         * @param fileSystemType
         *        The type of Amazon FSx file system, either <code>LUSTRE</code> or <code>WINDOWS</code>.
         * @see FileSystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemType
         */
        Builder fileSystemType(String fileSystemType);

        /**
         * <p>
         * The type of Amazon FSx file system, either <code>LUSTRE</code> or <code>WINDOWS</code>.
         * </p>
         * 
         * @param fileSystemType
         *        The type of Amazon FSx file system, either <code>LUSTRE</code> or <code>WINDOWS</code>.
         * @see FileSystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemType
         */
        Builder fileSystemType(FileSystemType fileSystemType);

        /**
         * <p>
         * The lifecycle status of the file system, following are the possible values and what they mean:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> - Amazon FSx is creating the new file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - Amazon FSx is deleting an existing file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a new
         * file system, Amazon FSx was unable to create the file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> indicates that the file system is in a failed but recoverable state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> indicates that the file system is undergoing a customer initiated update.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        The lifecycle status of the file system, following are the possible values and what they mean:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for
         *        use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - Amazon FSx is creating the new file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - Amazon FSx is deleting an existing file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating
         *        a new file system, Amazon FSx was unable to create the file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> indicates that the file system is in a failed but recoverable state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> indicates that the file system is undergoing a customer initiated update.
         *        </p>
         *        </li>
         * @see FileSystemLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemLifecycle
         */
        Builder lifecycle(String lifecycle);

        /**
         * <p>
         * The lifecycle status of the file system, following are the possible values and what they mean:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> - Amazon FSx is creating the new file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - Amazon FSx is deleting an existing file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating a new
         * file system, Amazon FSx was unable to create the file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> indicates that the file system is in a failed but recoverable state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> indicates that the file system is undergoing a customer initiated update.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        The lifecycle status of the file system, following are the possible values and what they mean:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The file system is in a healthy state, and is reachable and available for
         *        use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - Amazon FSx is creating the new file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - Amazon FSx is deleting an existing file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - An existing file system has experienced an unrecoverable failure. When creating
         *        a new file system, Amazon FSx was unable to create the file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> indicates that the file system is in a failed but recoverable state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> indicates that the file system is undergoing a customer initiated update.
         *        </p>
         *        </li>
         * @see FileSystemLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemLifecycle
         */
        Builder lifecycle(FileSystemLifecycle lifecycle);

        /**
         * Sets the value of the FailureDetails property for this object.
         *
         * @param failureDetails
         *        The new value for the FailureDetails property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureDetails(FileSystemFailureDetails failureDetails);

        /**
         * Sets the value of the FailureDetails property for this object.
         *
         * This is a convenience that creates an instance of the {@link FileSystemFailureDetails.Builder} avoiding the
         * need to create one manually via {@link FileSystemFailureDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link FileSystemFailureDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #failureDetails(FileSystemFailureDetails)}.
         * 
         * @param failureDetails
         *        a consumer that will call methods on {@link FileSystemFailureDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureDetails(FileSystemFailureDetails)
         */
        default Builder failureDetails(Consumer<FileSystemFailureDetails.Builder> failureDetails) {
            return failureDetails(FileSystemFailureDetails.builder().applyMutation(failureDetails).build());
        }

        /**
         * <p>
         * The storage capacity of the file system in gibibytes (GiB).
         * </p>
         * 
         * @param storageCapacity
         *        The storage capacity of the file system in gibibytes (GiB).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageCapacity(Integer storageCapacity);

        /**
         * <p>
         * The storage type of the file system. Valid values are <code>SSD</code> and <code>HDD</code>. If set to
         * <code>SSD</code>, the file system uses solid state drive storage. If set to <code>HDD</code>, the file system
         * uses hard disk drive storage.
         * </p>
         * 
         * @param storageType
         *        The storage type of the file system. Valid values are <code>SSD</code> and <code>HDD</code>. If set to
         *        <code>SSD</code>, the file system uses solid state drive storage. If set to <code>HDD</code>, the file
         *        system uses hard disk drive storage.
         * @see StorageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageType
         */
        Builder storageType(String storageType);

        /**
         * <p>
         * The storage type of the file system. Valid values are <code>SSD</code> and <code>HDD</code>. If set to
         * <code>SSD</code>, the file system uses solid state drive storage. If set to <code>HDD</code>, the file system
         * uses hard disk drive storage.
         * </p>
         * 
         * @param storageType
         *        The storage type of the file system. Valid values are <code>SSD</code> and <code>HDD</code>. If set to
         *        <code>SSD</code>, the file system uses solid state drive storage. If set to <code>HDD</code>, the file
         *        system uses hard disk drive storage.
         * @see StorageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageType
         */
        Builder storageType(StorageType storageType);

        /**
         * <p>
         * The ID of the primary VPC for the file system.
         * </p>
         * 
         * @param vpcId
         *        The ID of the primary VPC for the file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * Specifies the IDs of the subnets that the file system is accessible from. For Windows <code>MULTI_AZ_1</code>
         * file system deployment type, there are two subnet IDs, one for the preferred file server and one for the
         * standby file server. The preferred file server subnet identified in the <code>PreferredSubnetID</code>
         * property. All other file systems have only one subnet ID.
         * </p>
         * <p>
         * For Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that contains the
         * endpoint for the file system. For <code>MULTI_AZ_1</code> Windows file systems, the endpoint for the file
         * system is available in the <code>PreferredSubnetID</code>.
         * </p>
         * 
         * @param subnetIds
         *        Specifies the IDs of the subnets that the file system is accessible from. For Windows
         *        <code>MULTI_AZ_1</code> file system deployment type, there are two subnet IDs, one for the preferred
         *        file server and one for the standby file server. The preferred file server subnet identified in the
         *        <code>PreferredSubnetID</code> property. All other file systems have only one subnet ID.</p>
         *        <p>
         *        For Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that
         *        contains the endpoint for the file system. For <code>MULTI_AZ_1</code> Windows file systems, the
         *        endpoint for the file system is available in the <code>PreferredSubnetID</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * Specifies the IDs of the subnets that the file system is accessible from. For Windows <code>MULTI_AZ_1</code>
         * file system deployment type, there are two subnet IDs, one for the preferred file server and one for the
         * standby file server. The preferred file server subnet identified in the <code>PreferredSubnetID</code>
         * property. All other file systems have only one subnet ID.
         * </p>
         * <p>
         * For Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that contains the
         * endpoint for the file system. For <code>MULTI_AZ_1</code> Windows file systems, the endpoint for the file
         * system is available in the <code>PreferredSubnetID</code>.
         * </p>
         * 
         * @param subnetIds
         *        Specifies the IDs of the subnets that the file system is accessible from. For Windows
         *        <code>MULTI_AZ_1</code> file system deployment type, there are two subnet IDs, one for the preferred
         *        file server and one for the standby file server. The preferred file server subnet identified in the
         *        <code>PreferredSubnetID</code> property. All other file systems have only one subnet ID.</p>
         *        <p>
         *        For Lustre file systems, and Single-AZ Windows file systems, this is the ID of the subnet that
         *        contains the endpoint for the file system. For <code>MULTI_AZ_1</code> Windows file systems, the
         *        endpoint for the file system is available in the <code>PreferredSubnetID</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * The IDs of the elastic network interface from which a specific file system is accessible. The elastic network
         * interface is automatically created in the same VPC that the Amazon FSx file system was created in. For more
         * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network
         * Interfaces</a> in the <i>Amazon EC2 User Guide.</i>
         * </p>
         * <p>
         * For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an Amazon
         * FSx for Lustre file system, you can have more than one.
         * </p>
         * 
         * @param networkInterfaceIds
         *        The IDs of the elastic network interface from which a specific file system is accessible. The elastic
         *        network interface is automatically created in the same VPC that the Amazon FSx file system was created
         *        in. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network
         *        Interfaces</a> in the <i>Amazon EC2 User Guide.</i> </p>
         *        <p>
         *        For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an
         *        Amazon FSx for Lustre file system, you can have more than one.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceIds(Collection<String> networkInterfaceIds);

        /**
         * <p>
         * The IDs of the elastic network interface from which a specific file system is accessible. The elastic network
         * interface is automatically created in the same VPC that the Amazon FSx file system was created in. For more
         * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network
         * Interfaces</a> in the <i>Amazon EC2 User Guide.</i>
         * </p>
         * <p>
         * For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an Amazon
         * FSx for Lustre file system, you can have more than one.
         * </p>
         * 
         * @param networkInterfaceIds
         *        The IDs of the elastic network interface from which a specific file system is accessible. The elastic
         *        network interface is automatically created in the same VPC that the Amazon FSx file system was created
         *        in. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html">Elastic Network
         *        Interfaces</a> in the <i>Amazon EC2 User Guide.</i> </p>
         *        <p>
         *        For an Amazon FSx for Windows File Server file system, you can have one network interface ID. For an
         *        Amazon FSx for Lustre file system, you can have more than one.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceIds(String... networkInterfaceIds);

        /**
         * <p>
         * The DNS name for the file system.
         * </p>
         * 
         * @param dnsName
         *        The DNS name for the file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * The ID of the AWS Key Management Service (AWS KMS) key used to encrypt the file system's data for Amazon FSx
         * for Windows File Server file systems and persistent Amazon FSx for Lustre file systems at rest. In either
         * case, if not specified, the Amazon FSx managed key is used. The scratch Amazon FSx for Lustre file systems
         * are always encrypted at rest using Amazon FSx managed keys. For more information, see <a
         * href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html">Encrypt</a> in the <i>AWS Key
         * Management Service API Reference</i>.
         * </p>
         * 
         * @param kmsKeyId
         *        The ID of the AWS Key Management Service (AWS KMS) key used to encrypt the file system's data for
         *        Amazon FSx for Windows File Server file systems and persistent Amazon FSx for Lustre file systems at
         *        rest. In either case, if not specified, the Amazon FSx managed key is used. The scratch Amazon FSx for
         *        Lustre file systems are always encrypted at rest using Amazon FSx managed keys. For more information,
         *        see <a href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html">Encrypt</a> in the
         *        <i>AWS Key Management Service API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the file system resource.
         * </p>
         * 
         * @param resourceARN
         *        The Amazon Resource Name (ARN) for the file system resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceARN(String resourceARN);

        /**
         * <p>
         * The tags to associate with the file system. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html">Tagging Your Amazon EC2
         * Resources</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param tags
         *        The tags to associate with the file system. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html">Tagging Your Amazon EC2
         *        Resources</a> in the <i>Amazon EC2 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags to associate with the file system. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html">Tagging Your Amazon EC2
         * Resources</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param tags
         *        The tags to associate with the file system. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html">Tagging Your Amazon EC2
         *        Resources</a> in the <i>Amazon EC2 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags to associate with the file system. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html">Tagging Your Amazon EC2
         * Resources</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The configuration for this Microsoft Windows file system.
         * </p>
         * 
         * @param windowsConfiguration
         *        The configuration for this Microsoft Windows file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowsConfiguration(WindowsFileSystemConfiguration windowsConfiguration);

        /**
         * <p>
         * The configuration for this Microsoft Windows file system.
         * </p>
         * This is a convenience that creates an instance of the {@link WindowsFileSystemConfiguration.Builder} avoiding
         * the need to create one manually via {@link WindowsFileSystemConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link WindowsFileSystemConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #windowsConfiguration(WindowsFileSystemConfiguration)}.
         * 
         * @param windowsConfiguration
         *        a consumer that will call methods on {@link WindowsFileSystemConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #windowsConfiguration(WindowsFileSystemConfiguration)
         */
        default Builder windowsConfiguration(Consumer<WindowsFileSystemConfiguration.Builder> windowsConfiguration) {
            return windowsConfiguration(WindowsFileSystemConfiguration.builder().applyMutation(windowsConfiguration).build());
        }

        /**
         * Sets the value of the LustreConfiguration property for this object.
         *
         * @param lustreConfiguration
         *        The new value for the LustreConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lustreConfiguration(LustreFileSystemConfiguration lustreConfiguration);

        /**
         * Sets the value of the LustreConfiguration property for this object.
         *
         * This is a convenience that creates an instance of the {@link LustreFileSystemConfiguration.Builder} avoiding
         * the need to create one manually via {@link LustreFileSystemConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link LustreFileSystemConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #lustreConfiguration(LustreFileSystemConfiguration)}.
         * 
         * @param lustreConfiguration
         *        a consumer that will call methods on {@link LustreFileSystemConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lustreConfiguration(LustreFileSystemConfiguration)
         */
        default Builder lustreConfiguration(Consumer<LustreFileSystemConfiguration.Builder> lustreConfiguration) {
            return lustreConfiguration(LustreFileSystemConfiguration.builder().applyMutation(lustreConfiguration).build());
        }

        /**
         * <p>
         * A list of administrative actions for the file system that are in process or waiting to be processed.
         * Administrative actions describe changes to the Windows file system that you have initiated using the
         * <code>UpdateFileSystem</code> action.
         * </p>
         * 
         * @param administrativeActions
         *        A list of administrative actions for the file system that are in process or waiting to be processed.
         *        Administrative actions describe changes to the Windows file system that you have initiated using the
         *        <code>UpdateFileSystem</code> action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder administrativeActions(Collection<AdministrativeAction> administrativeActions);

        /**
         * <p>
         * A list of administrative actions for the file system that are in process or waiting to be processed.
         * Administrative actions describe changes to the Windows file system that you have initiated using the
         * <code>UpdateFileSystem</code> action.
         * </p>
         * 
         * @param administrativeActions
         *        A list of administrative actions for the file system that are in process or waiting to be processed.
         *        Administrative actions describe changes to the Windows file system that you have initiated using the
         *        <code>UpdateFileSystem</code> action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder administrativeActions(AdministrativeAction... administrativeActions);

        /**
         * <p>
         * A list of administrative actions for the file system that are in process or waiting to be processed.
         * Administrative actions describe changes to the Windows file system that you have initiated using the
         * <code>UpdateFileSystem</code> action.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AdministrativeAction>.Builder} avoiding the
         * need to create one manually via {@link List<AdministrativeAction>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AdministrativeAction>.Builder#build()} is called immediately
         * and its result is passed to {@link #administrativeActions(List<AdministrativeAction>)}.
         * 
         * @param administrativeActions
         *        a consumer that will call methods on {@link List<AdministrativeAction>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #administrativeActions(List<AdministrativeAction>)
         */
        Builder administrativeActions(Consumer<AdministrativeAction.Builder>... administrativeActions);
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private Instant creationTime;

        private String fileSystemId;

        private String fileSystemType;

        private String lifecycle;

        private FileSystemFailureDetails failureDetails;

        private Integer storageCapacity;

        private String storageType;

        private String vpcId;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> networkInterfaceIds = DefaultSdkAutoConstructList.getInstance();

        private String dnsName;

        private String kmsKeyId;

        private String resourceARN;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private WindowsFileSystemConfiguration windowsConfiguration;

        private LustreFileSystemConfiguration lustreConfiguration;

        private List<AdministrativeAction> administrativeActions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FileSystem model) {
            ownerId(model.ownerId);
            creationTime(model.creationTime);
            fileSystemId(model.fileSystemId);
            fileSystemType(model.fileSystemType);
            lifecycle(model.lifecycle);
            failureDetails(model.failureDetails);
            storageCapacity(model.storageCapacity);
            storageType(model.storageType);
            vpcId(model.vpcId);
            subnetIds(model.subnetIds);
            networkInterfaceIds(model.networkInterfaceIds);
            dnsName(model.dnsName);
            kmsKeyId(model.kmsKeyId);
            resourceARN(model.resourceARN);
            tags(model.tags);
            windowsConfiguration(model.windowsConfiguration);
            lustreConfiguration(model.lustreConfiguration);
            administrativeActions(model.administrativeActions);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final String getFileSystemId() {
            return fileSystemId;
        }

        @Override
        public final Builder fileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
            return this;
        }

        public final void setFileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
        }

        public final String getFileSystemType() {
            return fileSystemType;
        }

        @Override
        public final Builder fileSystemType(String fileSystemType) {
            this.fileSystemType = fileSystemType;
            return this;
        }

        @Override
        public final Builder fileSystemType(FileSystemType fileSystemType) {
            this.fileSystemType(fileSystemType == null ? null : fileSystemType.toString());
            return this;
        }

        public final void setFileSystemType(String fileSystemType) {
            this.fileSystemType = fileSystemType;
        }

        public final String getLifecycle() {
            return lifecycle;
        }

        @Override
        public final Builder lifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        @Override
        public final Builder lifecycle(FileSystemLifecycle lifecycle) {
            this.lifecycle(lifecycle == null ? null : lifecycle.toString());
            return this;
        }

        public final void setLifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
        }

        public final FileSystemFailureDetails.Builder getFailureDetails() {
            return failureDetails != null ? failureDetails.toBuilder() : null;
        }

        @Override
        public final Builder failureDetails(FileSystemFailureDetails failureDetails) {
            this.failureDetails = failureDetails;
            return this;
        }

        public final void setFailureDetails(FileSystemFailureDetails.BuilderImpl failureDetails) {
            this.failureDetails = failureDetails != null ? failureDetails.build() : null;
        }

        public final Integer getStorageCapacity() {
            return storageCapacity;
        }

        @Override
        public final Builder storageCapacity(Integer storageCapacity) {
            this.storageCapacity = storageCapacity;
            return this;
        }

        public final void setStorageCapacity(Integer storageCapacity) {
            this.storageCapacity = storageCapacity;
        }

        public final String getStorageType() {
            return storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        @Override
        public final Builder storageType(StorageType storageType) {
            this.storageType(storageType == null ? null : storageType.toString());
            return this;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        public final String getVpcId() {
            return vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        public final Collection<String> getSubnetIds() {
            if (subnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetIds;
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
        }

        public final Collection<String> getNetworkInterfaceIds() {
            if (networkInterfaceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return networkInterfaceIds;
        }

        @Override
        public final Builder networkInterfaceIds(Collection<String> networkInterfaceIds) {
            this.networkInterfaceIds = NetworkInterfaceIdsCopier.copy(networkInterfaceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaceIds(String... networkInterfaceIds) {
            networkInterfaceIds(Arrays.asList(networkInterfaceIds));
            return this;
        }

        public final void setNetworkInterfaceIds(Collection<String> networkInterfaceIds) {
            this.networkInterfaceIds = NetworkInterfaceIdsCopier.copy(networkInterfaceIds);
        }

        public final String getDnsName() {
            return dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final String getResourceARN() {
            return resourceARN;
        }

        @Override
        public final Builder resourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
            return this;
        }

        public final void setResourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final WindowsFileSystemConfiguration.Builder getWindowsConfiguration() {
            return windowsConfiguration != null ? windowsConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder windowsConfiguration(WindowsFileSystemConfiguration windowsConfiguration) {
            this.windowsConfiguration = windowsConfiguration;
            return this;
        }

        public final void setWindowsConfiguration(WindowsFileSystemConfiguration.BuilderImpl windowsConfiguration) {
            this.windowsConfiguration = windowsConfiguration != null ? windowsConfiguration.build() : null;
        }

        public final LustreFileSystemConfiguration.Builder getLustreConfiguration() {
            return lustreConfiguration != null ? lustreConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder lustreConfiguration(LustreFileSystemConfiguration lustreConfiguration) {
            this.lustreConfiguration = lustreConfiguration;
            return this;
        }

        public final void setLustreConfiguration(LustreFileSystemConfiguration.BuilderImpl lustreConfiguration) {
            this.lustreConfiguration = lustreConfiguration != null ? lustreConfiguration.build() : null;
        }

        public final List<AdministrativeAction.Builder> getAdministrativeActions() {
            List<AdministrativeAction.Builder> result = AdministrativeActionsCopier.copyToBuilder(this.administrativeActions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder administrativeActions(Collection<AdministrativeAction> administrativeActions) {
            this.administrativeActions = AdministrativeActionsCopier.copy(administrativeActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder administrativeActions(AdministrativeAction... administrativeActions) {
            administrativeActions(Arrays.asList(administrativeActions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder administrativeActions(Consumer<AdministrativeAction.Builder>... administrativeActions) {
            administrativeActions(Stream.of(administrativeActions)
                    .map(c -> AdministrativeAction.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setAdministrativeActions(Collection<AdministrativeAction.BuilderImpl> administrativeActions) {
            this.administrativeActions = AdministrativeActionsCopier.copyFromBuilder(administrativeActions);
        }

        @Override
        public FileSystem build() {
            return new FileSystem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
