/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.fsx.model.ActiveDirectoryErrorException;
import software.amazon.awssdk.services.fsx.model.AssociateFileSystemAliasesRequest;
import software.amazon.awssdk.services.fsx.model.AssociateFileSystemAliasesResponse;
import software.amazon.awssdk.services.fsx.model.BackupBeingCopiedException;
import software.amazon.awssdk.services.fsx.model.BackupInProgressException;
import software.amazon.awssdk.services.fsx.model.BackupNotFoundException;
import software.amazon.awssdk.services.fsx.model.BackupRestoringException;
import software.amazon.awssdk.services.fsx.model.BadRequestException;
import software.amazon.awssdk.services.fsx.model.CancelDataRepositoryTaskRequest;
import software.amazon.awssdk.services.fsx.model.CancelDataRepositoryTaskResponse;
import software.amazon.awssdk.services.fsx.model.CopyBackupRequest;
import software.amazon.awssdk.services.fsx.model.CopyBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateBackupRequest;
import software.amazon.awssdk.services.fsx.model.CreateBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryAssociationRequest;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryAssociationResponse;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryTaskRequest;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryTaskResponse;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemFromBackupRequest;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemFromBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemResponse;
import software.amazon.awssdk.services.fsx.model.CreateSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.CreateSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.CreateStorageVirtualMachineRequest;
import software.amazon.awssdk.services.fsx.model.CreateStorageVirtualMachineResponse;
import software.amazon.awssdk.services.fsx.model.CreateVolumeFromBackupRequest;
import software.amazon.awssdk.services.fsx.model.CreateVolumeFromBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateVolumeRequest;
import software.amazon.awssdk.services.fsx.model.CreateVolumeResponse;
import software.amazon.awssdk.services.fsx.model.DataRepositoryAssociationNotFoundException;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskEndedException;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskExecutingException;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskNotFoundException;
import software.amazon.awssdk.services.fsx.model.DeleteBackupRequest;
import software.amazon.awssdk.services.fsx.model.DeleteBackupResponse;
import software.amazon.awssdk.services.fsx.model.DeleteDataRepositoryAssociationRequest;
import software.amazon.awssdk.services.fsx.model.DeleteDataRepositoryAssociationResponse;
import software.amazon.awssdk.services.fsx.model.DeleteFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.DeleteFileSystemResponse;
import software.amazon.awssdk.services.fsx.model.DeleteSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.DeleteSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.DeleteStorageVirtualMachineRequest;
import software.amazon.awssdk.services.fsx.model.DeleteStorageVirtualMachineResponse;
import software.amazon.awssdk.services.fsx.model.DeleteVolumeRequest;
import software.amazon.awssdk.services.fsx.model.DeleteVolumeResponse;
import software.amazon.awssdk.services.fsx.model.DescribeBackupsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeBackupsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryAssociationsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryAssociationsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksRequest;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksResponse;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemAliasesRequest;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemAliasesResponse;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeSnapshotsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeSnapshotsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeStorageVirtualMachinesRequest;
import software.amazon.awssdk.services.fsx.model.DescribeStorageVirtualMachinesResponse;
import software.amazon.awssdk.services.fsx.model.DescribeVolumesRequest;
import software.amazon.awssdk.services.fsx.model.DescribeVolumesResponse;
import software.amazon.awssdk.services.fsx.model.DisassociateFileSystemAliasesRequest;
import software.amazon.awssdk.services.fsx.model.DisassociateFileSystemAliasesResponse;
import software.amazon.awssdk.services.fsx.model.FSxException;
import software.amazon.awssdk.services.fsx.model.FSxRequest;
import software.amazon.awssdk.services.fsx.model.FileSystemNotFoundException;
import software.amazon.awssdk.services.fsx.model.IncompatibleParameterErrorException;
import software.amazon.awssdk.services.fsx.model.IncompatibleRegionForMultiAzException;
import software.amazon.awssdk.services.fsx.model.InternalServerErrorException;
import software.amazon.awssdk.services.fsx.model.InvalidDataRepositoryTypeException;
import software.amazon.awssdk.services.fsx.model.InvalidDestinationKmsKeyException;
import software.amazon.awssdk.services.fsx.model.InvalidExportPathException;
import software.amazon.awssdk.services.fsx.model.InvalidImportPathException;
import software.amazon.awssdk.services.fsx.model.InvalidNetworkSettingsException;
import software.amazon.awssdk.services.fsx.model.InvalidPerUnitStorageThroughputException;
import software.amazon.awssdk.services.fsx.model.InvalidRegionException;
import software.amazon.awssdk.services.fsx.model.InvalidSourceKmsKeyException;
import software.amazon.awssdk.services.fsx.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.fsx.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.fsx.model.MissingFileSystemConfigurationException;
import software.amazon.awssdk.services.fsx.model.MissingVolumeConfigurationException;
import software.amazon.awssdk.services.fsx.model.NotServiceResourceErrorException;
import software.amazon.awssdk.services.fsx.model.ReleaseFileSystemNfsV3LocksRequest;
import software.amazon.awssdk.services.fsx.model.ReleaseFileSystemNfsV3LocksResponse;
import software.amazon.awssdk.services.fsx.model.ResourceDoesNotSupportTaggingException;
import software.amazon.awssdk.services.fsx.model.ResourceNotFoundException;
import software.amazon.awssdk.services.fsx.model.RestoreVolumeFromSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.RestoreVolumeFromSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.ServiceLimitExceededException;
import software.amazon.awssdk.services.fsx.model.SnapshotNotFoundException;
import software.amazon.awssdk.services.fsx.model.SourceBackupUnavailableException;
import software.amazon.awssdk.services.fsx.model.StorageVirtualMachineNotFoundException;
import software.amazon.awssdk.services.fsx.model.TagResourceRequest;
import software.amazon.awssdk.services.fsx.model.TagResourceResponse;
import software.amazon.awssdk.services.fsx.model.UnsupportedOperationException;
import software.amazon.awssdk.services.fsx.model.UntagResourceRequest;
import software.amazon.awssdk.services.fsx.model.UntagResourceResponse;
import software.amazon.awssdk.services.fsx.model.UpdateDataRepositoryAssociationRequest;
import software.amazon.awssdk.services.fsx.model.UpdateDataRepositoryAssociationResponse;
import software.amazon.awssdk.services.fsx.model.UpdateFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.UpdateFileSystemResponse;
import software.amazon.awssdk.services.fsx.model.UpdateSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.UpdateSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.UpdateStorageVirtualMachineRequest;
import software.amazon.awssdk.services.fsx.model.UpdateStorageVirtualMachineResponse;
import software.amazon.awssdk.services.fsx.model.UpdateVolumeRequest;
import software.amazon.awssdk.services.fsx.model.UpdateVolumeResponse;
import software.amazon.awssdk.services.fsx.model.VolumeNotFoundException;
import software.amazon.awssdk.services.fsx.paginators.DescribeBackupsIterable;
import software.amazon.awssdk.services.fsx.paginators.DescribeDataRepositoryAssociationsIterable;
import software.amazon.awssdk.services.fsx.paginators.DescribeDataRepositoryTasksIterable;
import software.amazon.awssdk.services.fsx.paginators.DescribeFileSystemAliasesIterable;
import software.amazon.awssdk.services.fsx.paginators.DescribeFileSystemsIterable;
import software.amazon.awssdk.services.fsx.paginators.DescribeSnapshotsIterable;
import software.amazon.awssdk.services.fsx.paginators.DescribeStorageVirtualMachinesIterable;
import software.amazon.awssdk.services.fsx.paginators.DescribeVolumesIterable;
import software.amazon.awssdk.services.fsx.paginators.ListTagsForResourceIterable;
import software.amazon.awssdk.services.fsx.transform.AssociateFileSystemAliasesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CancelDataRepositoryTaskRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CopyBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateDataRepositoryAssociationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateDataRepositoryTaskRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateFileSystemFromBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateFileSystemRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateStorageVirtualMachineRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateVolumeFromBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateVolumeRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteDataRepositoryAssociationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteFileSystemRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteStorageVirtualMachineRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteVolumeRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeBackupsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeDataRepositoryAssociationsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeDataRepositoryTasksRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeFileSystemAliasesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeFileSystemsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeSnapshotsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeStorageVirtualMachinesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeVolumesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DisassociateFileSystemAliasesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.ReleaseFileSystemNfsV3LocksRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.RestoreVolumeFromSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateDataRepositoryAssociationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateFileSystemRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateStorageVirtualMachineRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateVolumeRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link FSxClient}.
 *
 * @see FSxClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFSxClient implements FSxClient {
    private static final Logger log = Logger.loggerFor(DefaultFSxClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultFSxClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Use this action to associate one or more Domain Name Server (DNS) aliases with an existing Amazon FSx for Windows
     * File Server file system. A file system can have a maximum of 50 DNS aliases associated with it at any one time.
     * If you try to associate a DNS alias that is already associated with the file system, FSx takes no action on that
     * alias in the request. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS Aliases</a>
     * and <a href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/walkthrough05-file-system-custom-CNAME.html">
     * Walkthrough 5: Using DNS aliases to access your file system</a>, including additional steps you must take to be
     * able to access your file system using a DNS alias.
     * </p>
     * <p>
     * The system response shows the DNS aliases that Amazon FSx is attempting to associate with the file system. Use
     * the API operation to monitor the status of the aliases Amazon FSx is associating with the file system.
     * </p>
     *
     * @param associateFileSystemAliasesRequest
     *        The request object specifying one or more DNS alias names to associate with an Amazon FSx for Windows File
     *        Server file system.
     * @return Result of the AssociateFileSystemAliases operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.AssociateFileSystemAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/AssociateFileSystemAliases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateFileSystemAliasesResponse associateFileSystemAliases(
            AssociateFileSystemAliasesRequest associateFileSystemAliasesRequest) throws BadRequestException,
            FileSystemNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateFileSystemAliasesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateFileSystemAliasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateFileSystemAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateFileSystemAliases");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateFileSystemAliasesRequest, AssociateFileSystemAliasesResponse>()
                            .withOperationName("AssociateFileSystemAliases").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateFileSystemAliasesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateFileSystemAliasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels an existing Amazon FSx for Lustre data repository task if that task is in either the <code>PENDING</code>
     * or <code>EXECUTING</code> state. When you cancel a task, Amazon FSx does the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Any files that FSx has already exported are not reverted.
     * </p>
     * </li>
     * <li>
     * <p>
     * FSx continues to export any files that are "in-flight" when the cancel operation is received.
     * </p>
     * </li>
     * <li>
     * <p>
     * FSx does not export any files that have not yet been exported.
     * </p>
     * </li>
     * </ul>
     *
     * @param cancelDataRepositoryTaskRequest
     *        Cancels a data repository task.
     * @return Result of the CancelDataRepositoryTask operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws DataRepositoryTaskNotFoundException
     *         The data repository task or tasks you specified could not be found.
     * @throws DataRepositoryTaskEndedException
     *         The data repository task could not be canceled because the task has already ended.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CancelDataRepositoryTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CancelDataRepositoryTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelDataRepositoryTaskResponse cancelDataRepositoryTask(
            CancelDataRepositoryTaskRequest cancelDataRepositoryTaskRequest) throws BadRequestException,
            UnsupportedOperationException, DataRepositoryTaskNotFoundException, DataRepositoryTaskEndedException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelDataRepositoryTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CancelDataRepositoryTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelDataRepositoryTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelDataRepositoryTask");

            return clientHandler
                    .execute(new ClientExecutionParams<CancelDataRepositoryTaskRequest, CancelDataRepositoryTaskResponse>()
                            .withOperationName("CancelDataRepositoryTask").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(cancelDataRepositoryTaskRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CancelDataRepositoryTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Copies an existing backup within the same Amazon Web Services account to another Amazon Web Services Region
     * (cross-Region copy) or within the same Amazon Web Services Region (in-Region copy). You can have up to five
     * backup copy requests in progress to a single destination Region per account.
     * </p>
     * <p>
     * You can use cross-Region backup copies for cross-Region disaster recovery. You can periodically take backups and
     * copy them to another Region so that in the event of a disaster in the primary Region, you can restore from backup
     * and recover availability quickly in the other Region. You can make cross-Region copies only within your Amazon
     * Web Services partition. A partition is a grouping of Regions. Amazon Web Services currently has three partitions:
     * <code>aws</code> (Standard Regions), <code>aws-cn</code> (China Regions), and <code>aws-us-gov</code> (Amazon Web
     * Services GovCloud [US] Regions).
     * </p>
     * <p>
     * You can also use backup copies to clone your file dataset to another Region or within the same Region.
     * </p>
     * <p>
     * You can use the <code>SourceRegion</code> parameter to specify the Amazon Web Services Region from which the
     * backup will be copied. For example, if you make the call from the <code>us-west-1</code> Region and want to copy
     * a backup from the <code>us-east-2</code> Region, you specify <code>us-east-2</code> in the
     * <code>SourceRegion</code> parameter to make a cross-Region copy. If you don't specify a Region, the backup copy
     * is created in the same Region where the request is sent from (in-Region copy).
     * </p>
     * <p>
     * For more information about creating backup copies, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/using-backups.html#copy-backups"> Copying backups</a>
     * in the <i>Amazon FSx for Windows User Guide</i>, <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/using-backups-fsx.html#copy-backups">Copying backups</a>
     * in the <i>Amazon FSx for Lustre User Guide</i>, and <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/using-backups.html#copy-backups">Copying backups</a> in
     * the <i>Amazon FSx for OpenZFS User Guide</i>.
     * </p>
     *
     * @param copyBackupRequest
     * @return Result of the CopyBackup operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws BackupNotFoundException
     *         No Amazon FSx backups were found based upon the supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws InvalidSourceKmsKeyException
     *         The Key Management Service (KMS) key of the source backup is not valid.
     * @throws InvalidDestinationKmsKeyException
     *         The Key Management Service (KMS) key of the destination backup is not valid.
     * @throws InvalidRegionException
     *         The Region provided for <code>SourceRegion</code> is not valid or is in a different Amazon Web Services
     *         partition.
     * @throws SourceBackupUnavailableException
     *         The request was rejected because the lifecycle status of the source backup isn't <code>AVAILABLE</code>.
     * @throws IncompatibleRegionForMultiAzException
     *         Amazon FSx doesn't support Multi-AZ Windows File Server copy backup in the destination Region, so the
     *         copied backup can't be restored.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CopyBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CopyBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CopyBackupResponse copyBackup(CopyBackupRequest copyBackupRequest) throws BadRequestException,
            BackupNotFoundException, ServiceLimitExceededException, UnsupportedOperationException,
            IncompatibleParameterErrorException, InternalServerErrorException, InvalidSourceKmsKeyException,
            InvalidDestinationKmsKeyException, InvalidRegionException, SourceBackupUnavailableException,
            IncompatibleRegionForMultiAzException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CopyBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CopyBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copyBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopyBackup");

            return clientHandler
                    .execute(new ClientExecutionParams<CopyBackupRequest, CopyBackupResponse>().withOperationName("CopyBackup")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(copyBackupRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CopyBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a backup of an existing Amazon FSx for Windows File Server file system, Amazon FSx for Lustre file
     * system, Amazon FSx for NetApp ONTAP volume, or Amazon FSx for OpenZFS file system. We recommend creating regular
     * backups so that you can restore a file system or volume from a backup if an issue arises with the original file
     * system or volume.
     * </p>
     * <p>
     * For Amazon FSx for Lustre file systems, you can create a backup only for file systems that have the following
     * configuration:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A Persistent deployment type
     * </p>
     * </li>
     * <li>
     * <p>
     * Are <i>not</i> linked to a data repository
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about backups, see the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For Amazon FSx for Lustre, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/using-backups-fsx.html">Working with FSx for Lustre
     * backups</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon FSx for Windows, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/using-backups.html">Working with FSx for Windows
     * backups</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon FSx for NetApp ONTAP, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/using-backups.html">Working with FSx for NetApp ONTAP
     * backups</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon FSx for OpenZFS, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/using-backups.html">Working with FSx for OpenZFS
     * backups</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If a backup with the specified client request token exists and the parameters match, this operation returns the
     * description of the existing backup. If a backup with the specified client request token exists and the parameters
     * don't match, this operation returns <code>IncompatibleParameterError</code>. If a backup with the specified
     * client request token doesn't exist, <code>CreateBackup</code> does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new Amazon FSx backup with an assigned ID, and an initial lifecycle state of <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the backup.
     * </p>
     * </li>
     * </ul>
     * <p>
     * By using the idempotent operation, you can retry a <code>CreateBackup</code> operation without the risk of
     * creating an extra backup. This approach can be useful when an initial call fails in a way that makes it unclear
     * whether a backup was created. If you use the same client request token and the initial call created a backup, the
     * operation returns a successful result because all the parameters are the same.
     * </p>
     * <p>
     * The <code>CreateBackup</code> operation returns while the backup's lifecycle state is still <code>CREATING</code>
     * . You can check the backup creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeBackups.html">DescribeBackups</a>
     * operation, which returns the backup state along with other information.
     * </p>
     *
     * @param createBackupRequest
     *        The request object for the <code>CreateBackup</code> operation.
     * @return Result of the CreateBackup operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws VolumeNotFoundException
     *         No Amazon FSx for NetApp ONTAP volumes were found based upon the supplied parameters.
     * @throws BackupInProgressException
     *         Another backup is already under way. Wait for completion before initiating additional backups of this
     *         file system.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateBackupResponse createBackup(CreateBackupRequest createBackupRequest) throws BadRequestException,
            UnsupportedOperationException, FileSystemNotFoundException, VolumeNotFoundException, BackupInProgressException,
            IncompatibleParameterErrorException, ServiceLimitExceededException, InternalServerErrorException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackup");

            return clientHandler.execute(new ClientExecutionParams<CreateBackupRequest, CreateBackupResponse>()
                    .withOperationName("CreateBackup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createBackupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon FSx for Lustre data repository association (DRA). A data repository association is a link
     * between a directory on the file system and an Amazon S3 bucket or prefix. You can have a maximum of 8 data
     * repository associations on a file system. Data repository associations are supported only for file systems with
     * the <code>Persistent_2</code> deployment type.
     * </p>
     * <p>
     * Each data repository association must have a unique Amazon FSx file system directory and a unique S3 bucket or
     * prefix associated with it. You can configure a data repository association for automatic import only, for
     * automatic export only, or for both. To learn more about linking a data repository to your file system, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/create-dra-linked-data-repo.html">Linking your file
     * system to an S3 bucket</a>.
     * </p>
     *
     * @param createDataRepositoryAssociationRequest
     * @return Result of the CreateDataRepositoryAssociation operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateDataRepositoryAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateDataRepositoryAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDataRepositoryAssociationResponse createDataRepositoryAssociation(
            CreateDataRepositoryAssociationRequest createDataRepositoryAssociationRequest) throws BadRequestException,
            UnsupportedOperationException, FileSystemNotFoundException, IncompatibleParameterErrorException,
            ServiceLimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataRepositoryAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataRepositoryAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createDataRepositoryAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataRepositoryAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataRepositoryAssociationRequest, CreateDataRepositoryAssociationResponse>()
                            .withOperationName("CreateDataRepositoryAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createDataRepositoryAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataRepositoryAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon FSx for Lustre data repository task. You use data repository tasks to perform bulk operations
     * between your Amazon FSx file system and its linked data repositories. An example of a data repository task is
     * exporting any data and metadata changes, including POSIX metadata, to files, directories, and symbolic links
     * (symlinks) from your FSx file system to a linked data repository. A <code>CreateDataRepositoryTask</code>
     * operation will fail if a data repository is not linked to the FSx file system. To learn more about data
     * repository tasks, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-repository-tasks.html">Data Repository Tasks</a>.
     * To learn more about linking a data repository to your file system, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/create-dra-linked-data-repo.html">Linking your file
     * system to an S3 bucket</a>.
     * </p>
     *
     * @param createDataRepositoryTaskRequest
     * @return Result of the CreateDataRepositoryTask operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws DataRepositoryTaskExecutingException
     *         An existing data repository task is currently executing on the file system. Wait until the existing task
     *         has completed, then create the new task.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateDataRepositoryTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateDataRepositoryTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateDataRepositoryTaskResponse createDataRepositoryTask(
            CreateDataRepositoryTaskRequest createDataRepositoryTaskRequest) throws BadRequestException,
            UnsupportedOperationException, FileSystemNotFoundException, IncompatibleParameterErrorException,
            ServiceLimitExceededException, InternalServerErrorException, DataRepositoryTaskExecutingException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataRepositoryTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataRepositoryTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataRepositoryTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataRepositoryTask");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataRepositoryTaskRequest, CreateDataRepositoryTaskResponse>()
                            .withOperationName("CreateDataRepositoryTask").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createDataRepositoryTaskRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataRepositoryTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new, empty Amazon FSx file system. You can create the following supported Amazon FSx file systems using
     * the <code>CreateFileSystem</code> API operation:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon FSx for Lustre
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx for NetApp ONTAP
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx for Windows File Server
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation requires a client request token in the request that Amazon FSx uses to ensure idempotent creation.
     * This means that calling the operation multiple times with the same client request token has no effect. By using
     * the idempotent operation, you can retry a <code>CreateFileSystem</code> operation without the risk of creating an
     * extra file system. This approach can be useful when an initial call fails in a way that makes it unclear whether
     * a file system was created. Examples are if a transport level timeout occurred, or your connection was reset. If
     * you use the same client request token and the initial call created a file system, the client receives success as
     * long as the parameters are the same.
     * </p>
     * <p>
     * If a file system with the specified client request token exists and the parameters match,
     * <code>CreateFileSystem</code> returns the description of the existing file system. If a file system with the
     * specified client request token exists and the parameters don't match, this call returns
     * <code>IncompatibleParameterError</code>. If a file system with the specified client request token doesn't exist,
     * <code>CreateFileSystem</code> does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new, empty Amazon FSx file system with an assigned ID, and an initial lifecycle state of
     * <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation requires a client request token in the request that Amazon FSx uses to ensure idempotent creation.
     * This means that calling the operation multiple times with the same client request token has no effect. By using
     * the idempotent operation, you can retry a <code>CreateFileSystem</code> operation without the risk of creating an
     * extra file system. This approach can be useful when an initial call fails in a way that makes it unclear whether
     * a file system was created. Examples are if a transport-level timeout occurred, or your connection was reset. If
     * you use the same client request token and the initial call created a file system, the client receives a success
     * message as long as the parameters are the same.
     * </p>
     * <note>
     * <p>
     * The <code>CreateFileSystem</code> call returns while the file system's lifecycle state is still
     * <code>CREATING</code>. You can check the file-system creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystems.html">DescribeFileSystems</a>
     * operation, which returns the file system state along with other information.
     * </p>
     * </note>
     *
     * @param createFileSystemRequest
     *        The request object used to create a new Amazon FSx file system.
     * @return Result of the CreateFileSystem operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws ActiveDirectoryErrorException
     *         An Active Directory error.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InvalidImportPathException
     *         The path provided for data repository import isn't valid.
     * @throws InvalidExportPathException
     *         The path provided for data repository export isn't valid.
     * @throws InvalidNetworkSettingsException
     *         One or more network settings specified in the request are invalid.
     * @throws InvalidPerUnitStorageThroughputException
     *         An invalid value for <code>PerUnitStorageThroughput</code> was provided. Please create your file system
     *         again, using a valid value.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws MissingFileSystemConfigurationException
     *         A file system configuration is required for this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFileSystemResponse createFileSystem(CreateFileSystemRequest createFileSystemRequest) throws BadRequestException,
            ActiveDirectoryErrorException, IncompatibleParameterErrorException, InvalidImportPathException,
            InvalidExportPathException, InvalidNetworkSettingsException, InvalidPerUnitStorageThroughputException,
            ServiceLimitExceededException, InternalServerErrorException, MissingFileSystemConfigurationException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFileSystemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFileSystem");

            return clientHandler.execute(new ClientExecutionParams<CreateFileSystemRequest, CreateFileSystemResponse>()
                    .withOperationName("CreateFileSystem").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createFileSystemRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateFileSystemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Amazon FSx for Lustre, Amazon FSx for Windows File Server, or Amazon FSx for OpenZFS file system
     * from an existing Amazon FSx backup.
     * </p>
     * <p>
     * If a file system with the specified client request token exists and the parameters match, this operation returns
     * the description of the file system. If a client request token with the specified by the file system exists and
     * the parameters don't match, this call returns <code>IncompatibleParameterError</code>. If a file system with the
     * specified client request token doesn't exist, this operation does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new Amazon FSx file system from backup with an assigned ID, and an initial lifecycle state of
     * <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Parameters like the Active Directory, default share name, automatic backup, and backup settings default to the
     * parameters of the file system that was backed up, unless overridden. You can explicitly supply other settings.
     * </p>
     * <p>
     * By using the idempotent operation, you can retry a <code>CreateFileSystemFromBackup</code> call without the risk
     * of creating an extra file system. This approach can be useful when an initial call fails in a way that makes it
     * unclear whether a file system was created. Examples are if a transport level timeout occurred, or your connection
     * was reset. If you use the same client request token and the initial call created a file system, the client
     * receives a success message as long as the parameters are the same.
     * </p>
     * <note>
     * <p>
     * The <code>CreateFileSystemFromBackup</code> call returns while the file system's lifecycle state is still
     * <code>CREATING</code>. You can check the file-system creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystems.html"> DescribeFileSystems</a>
     * operation, which returns the file system state along with other information.
     * </p>
     * </note>
     *
     * @param createFileSystemFromBackupRequest
     *        The request object for the <code>CreateFileSystemFromBackup</code> operation.
     * @return Result of the CreateFileSystemFromBackup operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws ActiveDirectoryErrorException
     *         An Active Directory error.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InvalidNetworkSettingsException
     *         One or more network settings specified in the request are invalid.
     * @throws InvalidPerUnitStorageThroughputException
     *         An invalid value for <code>PerUnitStorageThroughput</code> was provided. Please create your file system
     *         again, using a valid value.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws BackupNotFoundException
     *         No Amazon FSx backups were found based upon the supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws MissingFileSystemConfigurationException
     *         A file system configuration is required for this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateFileSystemFromBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateFileSystemFromBackup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateFileSystemFromBackupResponse createFileSystemFromBackup(
            CreateFileSystemFromBackupRequest createFileSystemFromBackupRequest) throws BadRequestException,
            ActiveDirectoryErrorException, IncompatibleParameterErrorException, InvalidNetworkSettingsException,
            InvalidPerUnitStorageThroughputException, ServiceLimitExceededException, BackupNotFoundException,
            InternalServerErrorException, MissingFileSystemConfigurationException, AwsServiceException, SdkClientException,
            FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFileSystemFromBackupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateFileSystemFromBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFileSystemFromBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFileSystemFromBackup");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateFileSystemFromBackupRequest, CreateFileSystemFromBackupResponse>()
                            .withOperationName("CreateFileSystemFromBackup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createFileSystemFromBackupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateFileSystemFromBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a snapshot of an existing Amazon FSx for OpenZFS file system. With snapshots, you can easily undo file
     * changes and compare file versions by restoring the volume to a previous version.
     * </p>
     * <p>
     * If a snapshot with the specified client request token exists, and the parameters match, this operation returns
     * the description of the existing snapshot. If a snapshot with the specified client request token exists, and the
     * parameters don't match, this operation returns <code>IncompatibleParameterError</code>. If a snapshot with the
     * specified client request token doesn't exist, <code>CreateSnapshot</code> does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new OpenZFS snapshot with an assigned ID, and an initial lifecycle state of <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the snapshot.
     * </p>
     * </li>
     * </ul>
     * <p>
     * By using the idempotent operation, you can retry a <code>CreateSnapshot</code> operation without the risk of
     * creating an extra snapshot. This approach can be useful when an initial call fails in a way that makes it unclear
     * whether a snapshot was created. If you use the same client request token and the initial call created a snapshot,
     * the operation returns a successful result because all the parameters are the same.
     * </p>
     * <p>
     * The <code>CreateSnapshot</code> operation returns while the snapshot's lifecycle state is still
     * <code>CREATING</code>. You can check the snapshot creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeSnapshots.html">DescribeSnapshots</a>
     * operation, which returns the snapshot state along with other information.
     * </p>
     *
     * @param createSnapshotRequest
     * @return Result of the CreateSnapshot operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws VolumeNotFoundException
     *         No Amazon FSx for NetApp ONTAP volumes were found based upon the supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSnapshotResponse createSnapshot(CreateSnapshotRequest createSnapshotRequest) throws BadRequestException,
            VolumeNotFoundException, ServiceLimitExceededException, InternalServerErrorException, AwsServiceException,
            SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSnapshot");

            return clientHandler.execute(new ClientExecutionParams<CreateSnapshotRequest, CreateSnapshotResponse>()
                    .withOperationName("CreateSnapshot").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a storage virtual machine (SVM) for an Amazon FSx for ONTAP file system.
     * </p>
     *
     * @param createStorageVirtualMachineRequest
     * @return Result of the CreateStorageVirtualMachine operation returned by the service.
     * @throws ActiveDirectoryErrorException
     *         An Active Directory error.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateStorageVirtualMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateStorageVirtualMachine"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateStorageVirtualMachineResponse createStorageVirtualMachine(
            CreateStorageVirtualMachineRequest createStorageVirtualMachineRequest) throws ActiveDirectoryErrorException,
            BadRequestException, FileSystemNotFoundException, IncompatibleParameterErrorException, InternalServerErrorException,
            ServiceLimitExceededException, UnsupportedOperationException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStorageVirtualMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateStorageVirtualMachineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStorageVirtualMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStorageVirtualMachine");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateStorageVirtualMachineRequest, CreateStorageVirtualMachineResponse>()
                            .withOperationName("CreateStorageVirtualMachine").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createStorageVirtualMachineRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateStorageVirtualMachineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS storage volume.
     * </p>
     *
     * @param createVolumeRequest
     * @return Result of the CreateVolume operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws MissingVolumeConfigurationException
     *         A volume configuration is required for this operation.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws StorageVirtualMachineNotFoundException
     *         No Amazon FSx for NetApp ONTAP SVMs were found based upon the supplied parameters.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateVolume" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateVolumeResponse createVolume(CreateVolumeRequest createVolumeRequest) throws BadRequestException,
            FileSystemNotFoundException, IncompatibleParameterErrorException, InternalServerErrorException,
            MissingVolumeConfigurationException, ServiceLimitExceededException, StorageVirtualMachineNotFoundException,
            UnsupportedOperationException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateVolumeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVolume");

            return clientHandler.execute(new ClientExecutionParams<CreateVolumeRequest, CreateVolumeResponse>()
                    .withOperationName("CreateVolume").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createVolumeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVolumeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Amazon FSx for NetApp ONTAP volume from an existing Amazon FSx volume backup.
     * </p>
     *
     * @param createVolumeFromBackupRequest
     * @return Result of the CreateVolumeFromBackup operation returned by the service.
     * @throws BackupNotFoundException
     *         No Amazon FSx backups were found based upon the supplied parameters.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws MissingVolumeConfigurationException
     *         A volume configuration is required for this operation.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws StorageVirtualMachineNotFoundException
     *         No Amazon FSx for NetApp ONTAP SVMs were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateVolumeFromBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateVolumeFromBackup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateVolumeFromBackupResponse createVolumeFromBackup(CreateVolumeFromBackupRequest createVolumeFromBackupRequest)
            throws BackupNotFoundException, BadRequestException, FileSystemNotFoundException,
            IncompatibleParameterErrorException, InternalServerErrorException, MissingVolumeConfigurationException,
            ServiceLimitExceededException, StorageVirtualMachineNotFoundException, AwsServiceException, SdkClientException,
            FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVolumeFromBackupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateVolumeFromBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVolumeFromBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVolumeFromBackup");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateVolumeFromBackupRequest, CreateVolumeFromBackupResponse>()
                            .withOperationName("CreateVolumeFromBackup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createVolumeFromBackupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateVolumeFromBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon FSx backup. After deletion, the backup no longer exists, and its data is gone.
     * </p>
     * <p>
     * The <code>DeleteBackup</code> call returns instantly. The backup won't show up in later
     * <code>DescribeBackups</code> calls.
     * </p>
     * <important>
     * <p>
     * The data in a deleted backup is also deleted and can't be recovered by any means.
     * </p>
     * </important>
     *
     * @param deleteBackupRequest
     *        The request object for the <code>DeleteBackup</code> operation.
     * @return Result of the DeleteBackup operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws BackupInProgressException
     *         Another backup is already under way. Wait for completion before initiating additional backups of this
     *         file system.
     * @throws BackupNotFoundException
     *         No Amazon FSx backups were found based upon the supplied parameters.
     * @throws BackupRestoringException
     *         You can't delete a backup while it's being used to restore a file system.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws BackupBeingCopiedException
     *         You can't delete a backup while it's being copied.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBackupResponse deleteBackup(DeleteBackupRequest deleteBackupRequest) throws BadRequestException,
            BackupInProgressException, BackupNotFoundException, BackupRestoringException, IncompatibleParameterErrorException,
            InternalServerErrorException, BackupBeingCopiedException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackup");

            return clientHandler.execute(new ClientExecutionParams<DeleteBackupRequest, DeleteBackupResponse>()
                    .withOperationName("DeleteBackup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteBackupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a data repository association on an Amazon FSx for Lustre file system. Deleting the data repository
     * association unlinks the file system from the Amazon S3 bucket. When deleting a data repository association, you
     * have the option of deleting the data in the file system that corresponds to the data repository association. Data
     * repository associations are supported only for file systems with the <code>Persistent_2</code> deployment type.
     * </p>
     *
     * @param deleteDataRepositoryAssociationRequest
     * @return Result of the DeleteDataRepositoryAssociation operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws DataRepositoryAssociationNotFoundException
     *         No data repository associations were found based upon the supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteDataRepositoryAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteDataRepositoryAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDataRepositoryAssociationResponse deleteDataRepositoryAssociation(
            DeleteDataRepositoryAssociationRequest deleteDataRepositoryAssociationRequest) throws BadRequestException,
            IncompatibleParameterErrorException, DataRepositoryAssociationNotFoundException, ServiceLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataRepositoryAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDataRepositoryAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteDataRepositoryAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataRepositoryAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteDataRepositoryAssociationRequest, DeleteDataRepositoryAssociationResponse>()
                            .withOperationName("DeleteDataRepositoryAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteDataRepositoryAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteDataRepositoryAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a file system. After deletion, the file system no longer exists, and its data is gone. Any existing
     * automatic backups and snapshots are also deleted.
     * </p>
     * <p>
     * To delete an Amazon FSx for NetApp ONTAP file system, first delete all the volumes and storage virtual machines
     * (SVMs) on the file system. Then provide a <code>FileSystemId</code> value to the <code>DeleFileSystem</code>
     * operation.
     * </p>
     * <p>
     * By default, when you delete an Amazon FSx for Windows File Server file system, a final backup is created upon
     * deletion. This final backup isn't subject to the file system's retention policy, and must be manually deleted.
     * </p>
     * <p>
     * The <code>DeleteFileSystem</code> operation returns while the file system has the <code>DELETING</code> status.
     * You can check the file system deletion status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystems.html">DescribeFileSystems</a>
     * operation, which returns a list of file systems in your account. If you pass the file system ID for a deleted
     * file system, the <code>DescribeFileSystems</code> operation returns a <code>FileSystemNotFound</code> error.
     * </p>
     * <note>
     * <p>
     * If a data repository task is in a <code>PENDING</code> or <code>EXECUTING</code> state, deleting an Amazon FSx
     * for Lustre file system will fail with an HTTP status code 400 (Bad Request).
     * </p>
     * </note> <important>
     * <p>
     * The data in a deleted file system is also deleted and can't be recovered by any means.
     * </p>
     * </important>
     *
     * @param deleteFileSystemRequest
     *        The request object for <code>DeleteFileSystem</code> operation.
     * @return Result of the DeleteFileSystem operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFileSystemResponse deleteFileSystem(DeleteFileSystemRequest deleteFileSystemRequest) throws BadRequestException,
            IncompatibleParameterErrorException, FileSystemNotFoundException, ServiceLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFileSystemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFileSystem");

            return clientHandler.execute(new ClientExecutionParams<DeleteFileSystemRequest, DeleteFileSystemResponse>()
                    .withOperationName("DeleteFileSystem").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteFileSystemRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteFileSystemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the Amazon FSx snapshot. After deletion, the snapshot no longer exists, and its data is gone. Deleting a
     * snapshot doesn't affect snapshots stored in a file system backup.
     * </p>
     * <p>
     * The <code>DeleteSnapshot</code> operation returns instantly. The snapshot appears with the lifecycle status of
     * <code>DELETING</code> until the deletion is complete.
     * </p>
     *
     * @param deleteSnapshotRequest
     * @return Result of the DeleteSnapshot operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SnapshotNotFoundException
     *         No Amazon FSx snapshots were found based on the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSnapshotResponse deleteSnapshot(DeleteSnapshotRequest deleteSnapshotRequest) throws BadRequestException,
            InternalServerErrorException, SnapshotNotFoundException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSnapshot");

            return clientHandler.execute(new ClientExecutionParams<DeleteSnapshotRequest, DeleteSnapshotResponse>()
                    .withOperationName("DeleteSnapshot").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing Amazon FSx for ONTAP storage virtual machine (SVM). Prior to deleting an SVM, you must delete
     * all non-root volumes in the SVM, otherwise the operation will fail.
     * </p>
     *
     * @param deleteStorageVirtualMachineRequest
     * @return Result of the DeleteStorageVirtualMachine operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws StorageVirtualMachineNotFoundException
     *         No Amazon FSx for NetApp ONTAP SVMs were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteStorageVirtualMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteStorageVirtualMachine"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteStorageVirtualMachineResponse deleteStorageVirtualMachine(
            DeleteStorageVirtualMachineRequest deleteStorageVirtualMachineRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, StorageVirtualMachineNotFoundException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStorageVirtualMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteStorageVirtualMachineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStorageVirtualMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStorageVirtualMachine");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteStorageVirtualMachineRequest, DeleteStorageVirtualMachineResponse>()
                            .withOperationName("DeleteStorageVirtualMachine").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteStorageVirtualMachineRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteStorageVirtualMachineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volume.
     * </p>
     *
     * @param deleteVolumeRequest
     * @return Result of the DeleteVolume operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws VolumeNotFoundException
     *         No Amazon FSx for NetApp ONTAP volumes were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteVolume" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteVolumeResponse deleteVolume(DeleteVolumeRequest deleteVolumeRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, VolumeNotFoundException, AwsServiceException,
            SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteVolumeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVolume");

            return clientHandler.execute(new ClientExecutionParams<DeleteVolumeRequest, DeleteVolumeResponse>()
                    .withOperationName("DeleteVolume").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteVolumeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVolumeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of a specific Amazon FSx backup, if a <code>BackupIds</code> value is provided for that
     * backup. Otherwise, it returns all backups owned by your Amazon Web Services account in the Amazon Web Services
     * Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all backups, you can optionally specify the <code>MaxResults</code> parameter to limit the number
     * of backups in a response. If more backups remain, Amazon FSx returns a <code>NextToken</code> value in the
     * response. In this case, send a later request with the <code>NextToken</code> request parameter set to the value
     * of the <code>NextToken</code> value from the last response.
     * </p>
     * <p>
     * This operation is used in an iterative process to retrieve a list of your backups. <code>DescribeBackups</code>
     * is called first without a <code>NextToken</code> value. Then the operation continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code> value.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The operation might return fewer than the <code>MaxResults</code> value of backup descriptions while still
     * including a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of the backups returned in the response of one <code>DescribeBackups</code> call and the order of the
     * backups returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeBackupsRequest
     *        The request object for the <code>DescribeBackups</code> operation.
     * @return Result of the DescribeBackups operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws VolumeNotFoundException
     *         No Amazon FSx for NetApp ONTAP volumes were found based upon the supplied parameters.
     * @throws BackupNotFoundException
     *         No Amazon FSx backups were found based upon the supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeBackups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeBackupsResponse describeBackups(DescribeBackupsRequest describeBackupsRequest) throws BadRequestException,
            FileSystemNotFoundException, VolumeNotFoundException, BackupNotFoundException, InternalServerErrorException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBackupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeBackupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackups");

            return clientHandler.execute(new ClientExecutionParams<DescribeBackupsRequest, DescribeBackupsResponse>()
                    .withOperationName("DescribeBackups").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeBackupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBackupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of a specific Amazon FSx backup, if a <code>BackupIds</code> value is provided for that
     * backup. Otherwise, it returns all backups owned by your Amazon Web Services account in the Amazon Web Services
     * Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all backups, you can optionally specify the <code>MaxResults</code> parameter to limit the number
     * of backups in a response. If more backups remain, Amazon FSx returns a <code>NextToken</code> value in the
     * response. In this case, send a later request with the <code>NextToken</code> request parameter set to the value
     * of the <code>NextToken</code> value from the last response.
     * </p>
     * <p>
     * This operation is used in an iterative process to retrieve a list of your backups. <code>DescribeBackups</code>
     * is called first without a <code>NextToken</code> value. Then the operation continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code> value.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The operation might return fewer than the <code>MaxResults</code> value of backup descriptions while still
     * including a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of the backups returned in the response of one <code>DescribeBackups</code> call and the order of the
     * backups returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #describeBackups(software.amazon.awssdk.services.fsx.model.DescribeBackupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeBackupsIterable responses = client.describeBackupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.fsx.paginators.DescribeBackupsIterable responses = client.describeBackupsPaginator(request);
     *     for (software.amazon.awssdk.services.fsx.model.DescribeBackupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeBackupsIterable responses = client.describeBackupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBackups(software.amazon.awssdk.services.fsx.model.DescribeBackupsRequest)} operation.</b>
     * </p>
     *
     * @param describeBackupsRequest
     *        The request object for the <code>DescribeBackups</code> operation.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws VolumeNotFoundException
     *         No Amazon FSx for NetApp ONTAP volumes were found based upon the supplied parameters.
     * @throws BackupNotFoundException
     *         No Amazon FSx backups were found based upon the supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeBackups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeBackupsIterable describeBackupsPaginator(DescribeBackupsRequest describeBackupsRequest)
            throws BadRequestException, FileSystemNotFoundException, VolumeNotFoundException, BackupNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        return new DescribeBackupsIterable(this, applyPaginatorUserAgent(describeBackupsRequest));
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Lustre data repository associations, if one or more
     * <code>AssociationIds</code> values are provided in the request, or if filters are used in the request. Data
     * repository associations are supported only for file systems with the <code>Persistent_2</code> deployment type.
     * </p>
     * <p>
     * You can use filters to narrow the response to include just data repository associations for specific file systems
     * (use the <code>file-system-id</code> filter with the ID of the file system) or data repository associations for a
     * specific repository type (use the <code>data-repository-type</code> filter with a value of <code>S3</code>). If
     * you don't use filters, the response returns all data repository associations owned by your Amazon Web Services
     * account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all data repository associations, you can paginate the response by using the optional
     * <code>MaxResults</code> parameter to limit the number of data repository associations returned in a response. If
     * more data repository associations remain, Amazon FSx returns a <code>NextToken</code> value in the response. In
     * this case, send a later request with the <code>NextToken</code> request parameter set to the value of
     * <code>NextToken</code> from the last response.
     * </p>
     *
     * @param describeDataRepositoryAssociationsRequest
     * @return Result of the DescribeDataRepositoryAssociations operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws DataRepositoryAssociationNotFoundException
     *         No data repository associations were found based upon the supplied parameters.
     * @throws InvalidDataRepositoryTypeException
     *         You have filtered the response to a data repository type that is not supported.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeDataRepositoryAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeDataRepositoryAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDataRepositoryAssociationsResponse describeDataRepositoryAssociations(
            DescribeDataRepositoryAssociationsRequest describeDataRepositoryAssociationsRequest) throws BadRequestException,
            FileSystemNotFoundException, DataRepositoryAssociationNotFoundException, InvalidDataRepositoryTypeException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDataRepositoryAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDataRepositoryAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeDataRepositoryAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataRepositoryAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeDataRepositoryAssociationsRequest, DescribeDataRepositoryAssociationsResponse>()
                            .withOperationName("DescribeDataRepositoryAssociations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeDataRepositoryAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeDataRepositoryAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Lustre data repository associations, if one or more
     * <code>AssociationIds</code> values are provided in the request, or if filters are used in the request. Data
     * repository associations are supported only for file systems with the <code>Persistent_2</code> deployment type.
     * </p>
     * <p>
     * You can use filters to narrow the response to include just data repository associations for specific file systems
     * (use the <code>file-system-id</code> filter with the ID of the file system) or data repository associations for a
     * specific repository type (use the <code>data-repository-type</code> filter with a value of <code>S3</code>). If
     * you don't use filters, the response returns all data repository associations owned by your Amazon Web Services
     * account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all data repository associations, you can paginate the response by using the optional
     * <code>MaxResults</code> parameter to limit the number of data repository associations returned in a response. If
     * more data repository associations remain, Amazon FSx returns a <code>NextToken</code> value in the response. In
     * this case, send a later request with the <code>NextToken</code> request parameter set to the value of
     * <code>NextToken</code> from the last response.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeDataRepositoryAssociations(software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryAssociationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeDataRepositoryAssociationsIterable responses = client.describeDataRepositoryAssociationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.fsx.paginators.DescribeDataRepositoryAssociationsIterable responses = client
     *             .describeDataRepositoryAssociationsPaginator(request);
     *     for (software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryAssociationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeDataRepositoryAssociationsIterable responses = client.describeDataRepositoryAssociationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeDataRepositoryAssociations(software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeDataRepositoryAssociationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws DataRepositoryAssociationNotFoundException
     *         No data repository associations were found based upon the supplied parameters.
     * @throws InvalidDataRepositoryTypeException
     *         You have filtered the response to a data repository type that is not supported.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeDataRepositoryAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeDataRepositoryAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDataRepositoryAssociationsIterable describeDataRepositoryAssociationsPaginator(
            DescribeDataRepositoryAssociationsRequest describeDataRepositoryAssociationsRequest) throws BadRequestException,
            FileSystemNotFoundException, DataRepositoryAssociationNotFoundException, InvalidDataRepositoryTypeException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        return new DescribeDataRepositoryAssociationsIterable(this,
                applyPaginatorUserAgent(describeDataRepositoryAssociationsRequest));
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Lustre data repository tasks, if one or more
     * <code>TaskIds</code> values are provided in the request, or if filters are used in the request. You can use
     * filters to narrow the response to include just tasks for specific file systems, or tasks in a specific lifecycle
     * state. Otherwise, it returns all data repository tasks owned by your Amazon Web Services account in the Amazon
     * Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all tasks, you can paginate the response by using the optional <code>MaxResults</code> parameter
     * to limit the number of tasks returned in a response. If more tasks remain, Amazon FSx returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     *
     * @param describeDataRepositoryTasksRequest
     * @return Result of the DescribeDataRepositoryTasks operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws DataRepositoryTaskNotFoundException
     *         The data repository task or tasks you specified could not be found.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeDataRepositoryTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeDataRepositoryTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDataRepositoryTasksResponse describeDataRepositoryTasks(
            DescribeDataRepositoryTasksRequest describeDataRepositoryTasksRequest) throws BadRequestException,
            FileSystemNotFoundException, DataRepositoryTaskNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDataRepositoryTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDataRepositoryTasksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDataRepositoryTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataRepositoryTasks");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeDataRepositoryTasksRequest, DescribeDataRepositoryTasksResponse>()
                            .withOperationName("DescribeDataRepositoryTasks").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeDataRepositoryTasksRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeDataRepositoryTasksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Lustre data repository tasks, if one or more
     * <code>TaskIds</code> values are provided in the request, or if filters are used in the request. You can use
     * filters to narrow the response to include just tasks for specific file systems, or tasks in a specific lifecycle
     * state. Otherwise, it returns all data repository tasks owned by your Amazon Web Services account in the Amazon
     * Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all tasks, you can paginate the response by using the optional <code>MaxResults</code> parameter
     * to limit the number of tasks returned in a response. If more tasks remain, Amazon FSx returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeDataRepositoryTasks(software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeDataRepositoryTasksIterable responses = client.describeDataRepositoryTasksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.fsx.paginators.DescribeDataRepositoryTasksIterable responses = client
     *             .describeDataRepositoryTasksPaginator(request);
     *     for (software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeDataRepositoryTasksIterable responses = client.describeDataRepositoryTasksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeDataRepositoryTasks(software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeDataRepositoryTasksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws DataRepositoryTaskNotFoundException
     *         The data repository task or tasks you specified could not be found.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeDataRepositoryTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeDataRepositoryTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDataRepositoryTasksIterable describeDataRepositoryTasksPaginator(
            DescribeDataRepositoryTasksRequest describeDataRepositoryTasksRequest) throws BadRequestException,
            FileSystemNotFoundException, DataRepositoryTaskNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, FSxException {
        return new DescribeDataRepositoryTasksIterable(this, applyPaginatorUserAgent(describeDataRepositoryTasksRequest));
    }

    /**
     * <p>
     * Returns the DNS aliases that are associated with the specified Amazon FSx for Windows File Server file system. A
     * history of all DNS aliases that have been associated with and disassociated from the file system is available in
     * the list of <a>AdministrativeAction</a> provided in the <a>DescribeFileSystems</a> operation response.
     * </p>
     *
     * @param describeFileSystemAliasesRequest
     *        The request object for <code>DescribeFileSystemAliases</code> operation.
     * @return Result of the DescribeFileSystemAliases operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeFileSystemAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileSystemAliases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeFileSystemAliasesResponse describeFileSystemAliases(
            DescribeFileSystemAliasesRequest describeFileSystemAliasesRequest) throws BadRequestException,
            FileSystemNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFileSystemAliasesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFileSystemAliasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFileSystemAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFileSystemAliases");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeFileSystemAliasesRequest, DescribeFileSystemAliasesResponse>()
                            .withOperationName("DescribeFileSystemAliases").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeFileSystemAliasesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeFileSystemAliasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the DNS aliases that are associated with the specified Amazon FSx for Windows File Server file system. A
     * history of all DNS aliases that have been associated with and disassociated from the file system is available in
     * the list of <a>AdministrativeAction</a> provided in the <a>DescribeFileSystems</a> operation response.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeFileSystemAliases(software.amazon.awssdk.services.fsx.model.DescribeFileSystemAliasesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeFileSystemAliasesIterable responses = client.describeFileSystemAliasesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.fsx.paginators.DescribeFileSystemAliasesIterable responses = client
     *             .describeFileSystemAliasesPaginator(request);
     *     for (software.amazon.awssdk.services.fsx.model.DescribeFileSystemAliasesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeFileSystemAliasesIterable responses = client.describeFileSystemAliasesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFileSystemAliases(software.amazon.awssdk.services.fsx.model.DescribeFileSystemAliasesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeFileSystemAliasesRequest
     *        The request object for <code>DescribeFileSystemAliases</code> operation.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeFileSystemAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileSystemAliases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeFileSystemAliasesIterable describeFileSystemAliasesPaginator(
            DescribeFileSystemAliasesRequest describeFileSystemAliasesRequest) throws BadRequestException,
            FileSystemNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        return new DescribeFileSystemAliasesIterable(this, applyPaginatorUserAgent(describeFileSystemAliasesRequest));
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx file systems, if a <code>FileSystemIds</code> value is provided
     * for that file system. Otherwise, it returns descriptions of all file systems owned by your Amazon Web Services
     * account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxResults</code> parameter to
     * limit the number of descriptions in a response. If more file system descriptions remain, Amazon FSx returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This operation is used in an iterative process to retrieve a list of your file system descriptions.
     * <code>DescribeFileSystems</code> is called first without a <code>NextToken</code>value. Then the operation
     * continues to be called with the <code>NextToken</code> parameter set to the value of the last
     * <code>NextToken</code> value until a response has no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multicall iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeFileSystemsRequest
     *        The request object for <code>DescribeFileSystems</code> operation.
     * @return Result of the DescribeFileSystems operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileSystems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeFileSystemsResponse describeFileSystems(DescribeFileSystemsRequest describeFileSystemsRequest)
            throws BadRequestException, FileSystemNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFileSystemsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFileSystemsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFileSystemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFileSystems");

            return clientHandler.execute(new ClientExecutionParams<DescribeFileSystemsRequest, DescribeFileSystemsResponse>()
                    .withOperationName("DescribeFileSystems").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeFileSystemsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFileSystemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx file systems, if a <code>FileSystemIds</code> value is provided
     * for that file system. Otherwise, it returns descriptions of all file systems owned by your Amazon Web Services
     * account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxResults</code> parameter to
     * limit the number of descriptions in a response. If more file system descriptions remain, Amazon FSx returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This operation is used in an iterative process to retrieve a list of your file system descriptions.
     * <code>DescribeFileSystems</code> is called first without a <code>NextToken</code>value. Then the operation
     * continues to be called with the <code>NextToken</code> parameter set to the value of the last
     * <code>NextToken</code> value until a response has no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multicall iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeFileSystems(software.amazon.awssdk.services.fsx.model.DescribeFileSystemsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeFileSystemsIterable responses = client.describeFileSystemsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.fsx.paginators.DescribeFileSystemsIterable responses = client
     *             .describeFileSystemsPaginator(request);
     *     for (software.amazon.awssdk.services.fsx.model.DescribeFileSystemsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeFileSystemsIterable responses = client.describeFileSystemsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFileSystems(software.amazon.awssdk.services.fsx.model.DescribeFileSystemsRequest)} operation.</b>
     * </p>
     *
     * @param describeFileSystemsRequest
     *        The request object for <code>DescribeFileSystems</code> operation.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileSystems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeFileSystemsIterable describeFileSystemsPaginator(DescribeFileSystemsRequest describeFileSystemsRequest)
            throws BadRequestException, FileSystemNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, FSxException {
        return new DescribeFileSystemsIterable(this, applyPaginatorUserAgent(describeFileSystemsRequest));
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx snapshots, if a <code>SnapshotIds</code> value is provided.
     * Otherwise, this operation returns all snapshots owned by your Amazon Web Services account in the Amazon Web
     * Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all snapshots, you can optionally specify the <code>MaxResults</code> parameter to limit the
     * number of snapshots in a response. If more backups remain, Amazon FSx returns a <code>NextToken</code> value in
     * the response. In this case, send a later request with the <code>NextToken</code> request parameter set to the
     * value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * Use this operation in an iterative process to retrieve a list of your snapshots. <code>DescribeSnapshots</code>
     * is called first without a <code>NextToken</code> value. Then the operation continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code> value.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The operation might return fewer than the <code>MaxResults</code> value of snapshot descriptions while still
     * including a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of snapshots returned in the response of one <code>DescribeSnapshots</code> call and the order of
     * backups returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeSnapshotsRequest
     * @return Result of the DescribeSnapshots operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SnapshotNotFoundException
     *         No Amazon FSx snapshots were found based on the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeSnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeSnapshots" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeSnapshotsResponse describeSnapshots(DescribeSnapshotsRequest describeSnapshotsRequest)
            throws BadRequestException, InternalServerErrorException, SnapshotNotFoundException, AwsServiceException,
            SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSnapshotsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeSnapshotsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSnapshotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSnapshots");

            return clientHandler.execute(new ClientExecutionParams<DescribeSnapshotsRequest, DescribeSnapshotsResponse>()
                    .withOperationName("DescribeSnapshots").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeSnapshotsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeSnapshotsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx snapshots, if a <code>SnapshotIds</code> value is provided.
     * Otherwise, this operation returns all snapshots owned by your Amazon Web Services account in the Amazon Web
     * Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all snapshots, you can optionally specify the <code>MaxResults</code> parameter to limit the
     * number of snapshots in a response. If more backups remain, Amazon FSx returns a <code>NextToken</code> value in
     * the response. In this case, send a later request with the <code>NextToken</code> request parameter set to the
     * value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * Use this operation in an iterative process to retrieve a list of your snapshots. <code>DescribeSnapshots</code>
     * is called first without a <code>NextToken</code> value. Then the operation continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code> value.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The operation might return fewer than the <code>MaxResults</code> value of snapshot descriptions while still
     * including a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of snapshots returned in the response of one <code>DescribeSnapshots</code> call and the order of
     * backups returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeSnapshots(software.amazon.awssdk.services.fsx.model.DescribeSnapshotsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeSnapshotsIterable responses = client.describeSnapshotsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.fsx.paginators.DescribeSnapshotsIterable responses = client
     *             .describeSnapshotsPaginator(request);
     *     for (software.amazon.awssdk.services.fsx.model.DescribeSnapshotsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeSnapshotsIterable responses = client.describeSnapshotsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeSnapshots(software.amazon.awssdk.services.fsx.model.DescribeSnapshotsRequest)} operation.</b>
     * </p>
     *
     * @param describeSnapshotsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SnapshotNotFoundException
     *         No Amazon FSx snapshots were found based on the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeSnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeSnapshots" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeSnapshotsIterable describeSnapshotsPaginator(DescribeSnapshotsRequest describeSnapshotsRequest)
            throws BadRequestException, InternalServerErrorException, SnapshotNotFoundException, AwsServiceException,
            SdkClientException, FSxException {
        return new DescribeSnapshotsIterable(this, applyPaginatorUserAgent(describeSnapshotsRequest));
    }

    /**
     * <p>
     * Describes one or more Amazon FSx for NetApp ONTAP storage virtual machines (SVMs).
     * </p>
     *
     * @param describeStorageVirtualMachinesRequest
     * @return Result of the DescribeStorageVirtualMachines operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws StorageVirtualMachineNotFoundException
     *         No Amazon FSx for NetApp ONTAP SVMs were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeStorageVirtualMachines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeStorageVirtualMachines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeStorageVirtualMachinesResponse describeStorageVirtualMachines(
            DescribeStorageVirtualMachinesRequest describeStorageVirtualMachinesRequest) throws BadRequestException,
            InternalServerErrorException, StorageVirtualMachineNotFoundException, AwsServiceException, SdkClientException,
            FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStorageVirtualMachinesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeStorageVirtualMachinesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeStorageVirtualMachinesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStorageVirtualMachines");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeStorageVirtualMachinesRequest, DescribeStorageVirtualMachinesResponse>()
                            .withOperationName("DescribeStorageVirtualMachines").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeStorageVirtualMachinesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeStorageVirtualMachinesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more Amazon FSx for NetApp ONTAP storage virtual machines (SVMs).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeStorageVirtualMachines(software.amazon.awssdk.services.fsx.model.DescribeStorageVirtualMachinesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeStorageVirtualMachinesIterable responses = client.describeStorageVirtualMachinesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.fsx.paginators.DescribeStorageVirtualMachinesIterable responses = client
     *             .describeStorageVirtualMachinesPaginator(request);
     *     for (software.amazon.awssdk.services.fsx.model.DescribeStorageVirtualMachinesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeStorageVirtualMachinesIterable responses = client.describeStorageVirtualMachinesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeStorageVirtualMachines(software.amazon.awssdk.services.fsx.model.DescribeStorageVirtualMachinesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeStorageVirtualMachinesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws StorageVirtualMachineNotFoundException
     *         No Amazon FSx for NetApp ONTAP SVMs were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeStorageVirtualMachines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeStorageVirtualMachines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeStorageVirtualMachinesIterable describeStorageVirtualMachinesPaginator(
            DescribeStorageVirtualMachinesRequest describeStorageVirtualMachinesRequest) throws BadRequestException,
            InternalServerErrorException, StorageVirtualMachineNotFoundException, AwsServiceException, SdkClientException,
            FSxException {
        return new DescribeStorageVirtualMachinesIterable(this, applyPaginatorUserAgent(describeStorageVirtualMachinesRequest));
    }

    /**
     * <p>
     * Describes one or more Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volumes.
     * </p>
     *
     * @param describeVolumesRequest
     * @return Result of the DescribeVolumes operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws VolumeNotFoundException
     *         No Amazon FSx for NetApp ONTAP volumes were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeVolumes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeVolumes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeVolumesResponse describeVolumes(DescribeVolumesRequest describeVolumesRequest) throws BadRequestException,
            InternalServerErrorException, VolumeNotFoundException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeVolumesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeVolumesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVolumesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVolumes");

            return clientHandler.execute(new ClientExecutionParams<DescribeVolumesRequest, DescribeVolumesResponse>()
                    .withOperationName("DescribeVolumes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeVolumesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeVolumesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volumes.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #describeVolumes(software.amazon.awssdk.services.fsx.model.DescribeVolumesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeVolumesIterable responses = client.describeVolumesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.fsx.paginators.DescribeVolumesIterable responses = client.describeVolumesPaginator(request);
     *     for (software.amazon.awssdk.services.fsx.model.DescribeVolumesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.DescribeVolumesIterable responses = client.describeVolumesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeVolumes(software.amazon.awssdk.services.fsx.model.DescribeVolumesRequest)} operation.</b>
     * </p>
     *
     * @param describeVolumesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws VolumeNotFoundException
     *         No Amazon FSx for NetApp ONTAP volumes were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeVolumes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeVolumes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeVolumesIterable describeVolumesPaginator(DescribeVolumesRequest describeVolumesRequest)
            throws BadRequestException, InternalServerErrorException, VolumeNotFoundException, AwsServiceException,
            SdkClientException, FSxException {
        return new DescribeVolumesIterable(this, applyPaginatorUserAgent(describeVolumesRequest));
    }

    /**
     * <p>
     * Use this action to disassociate, or remove, one or more Domain Name Service (DNS) aliases from an Amazon FSx for
     * Windows File Server file system. If you attempt to disassociate a DNS alias that is not associated with the file
     * system, Amazon FSx responds with a 400 Bad Request. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS
     * Aliases</a>.
     * </p>
     * <p>
     * The system generated response showing the DNS aliases that Amazon FSx is attempting to disassociate from the file
     * system. Use the API operation to monitor the status of the aliases Amazon FSx is disassociating with the file
     * system.
     * </p>
     *
     * @param disassociateFileSystemAliasesRequest
     *        The request object of DNS aliases to disassociate from an Amazon FSx for Windows File Server file system.
     * @return Result of the DisassociateFileSystemAliases operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DisassociateFileSystemAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DisassociateFileSystemAliases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateFileSystemAliasesResponse disassociateFileSystemAliases(
            DisassociateFileSystemAliasesRequest disassociateFileSystemAliasesRequest) throws BadRequestException,
            FileSystemNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateFileSystemAliasesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateFileSystemAliasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateFileSystemAliasesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateFileSystemAliases");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateFileSystemAliasesRequest, DisassociateFileSystemAliasesResponse>()
                            .withOperationName("DisassociateFileSystemAliases").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateFileSystemAliasesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateFileSystemAliasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tags for an Amazon FSx file systems and backups in the case of Amazon FSx for Windows File Server.
     * </p>
     * <p>
     * When retrieving all tags, you can optionally specify the <code>MaxResults</code> parameter to limit the number of
     * tags in a response. If more tags remain, Amazon FSx returns a <code>NextToken</code> value in the response. In
     * this case, send a later request with the <code>NextToken</code> request parameter set to the value of
     * <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This action is used in an iterative process to retrieve a list of your tags. <code>ListTagsForResource</code> is
     * called first without a <code>NextToken</code>value. Then the action continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this action, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of tags returned in the response of one <code>ListTagsForResource</code> call and the order of tags
     * returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param listTagsForResourceRequest
     *        The request object for <code>ListTagsForResource</code> operation.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws ResourceNotFoundException
     *         The resource specified by the Amazon Resource Name (ARN) can't be found.
     * @throws NotServiceResourceErrorException
     *         The resource specified for the tagging operation is not a resource type owned by Amazon FSx. Use the API
     *         of the relevant service to perform the operation.
     * @throws ResourceDoesNotSupportTaggingException
     *         The resource specified does not support tagging.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, InternalServerErrorException, ResourceNotFoundException,
            NotServiceResourceErrorException, ResourceDoesNotSupportTaggingException, AwsServiceException, SdkClientException,
            FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tags for an Amazon FSx file systems and backups in the case of Amazon FSx for Windows File Server.
     * </p>
     * <p>
     * When retrieving all tags, you can optionally specify the <code>MaxResults</code> parameter to limit the number of
     * tags in a response. If more tags remain, Amazon FSx returns a <code>NextToken</code> value in the response. In
     * this case, send a later request with the <code>NextToken</code> request parameter set to the value of
     * <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This action is used in an iterative process to retrieve a list of your tags. <code>ListTagsForResource</code> is
     * called first without a <code>NextToken</code>value. Then the action continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this action, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of tags returned in the response of one <code>ListTagsForResource</code> call and the order of tags
     * returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.fsx.model.ListTagsForResourceRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.fsx.paginators.ListTagsForResourceIterable responses = client
     *             .listTagsForResourcePaginator(request);
     *     for (software.amazon.awssdk.services.fsx.model.ListTagsForResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.fsx.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.fsx.model.ListTagsForResourceRequest)} operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     *        The request object for <code>ListTagsForResource</code> operation.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws ResourceNotFoundException
     *         The resource specified by the Amazon Resource Name (ARN) can't be found.
     * @throws NotServiceResourceErrorException
     *         The resource specified for the tagging operation is not a resource type owned by Amazon FSx. Use the API
     *         of the relevant service to perform the operation.
     * @throws ResourceDoesNotSupportTaggingException
     *         The resource specified does not support tagging.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceIterable listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, InternalServerErrorException, ResourceNotFoundException,
            NotServiceResourceErrorException, ResourceDoesNotSupportTaggingException, AwsServiceException, SdkClientException,
            FSxException {
        return new ListTagsForResourceIterable(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * Releases the file system lock from an Amazon FSx for OpenZFS file system.
     * </p>
     *
     * @param releaseFileSystemNfsV3LocksRequest
     * @return Result of the ReleaseFileSystemNfsV3Locks operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.ReleaseFileSystemNfsV3Locks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/ReleaseFileSystemNfsV3Locks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ReleaseFileSystemNfsV3LocksResponse releaseFileSystemNfsV3Locks(
            ReleaseFileSystemNfsV3LocksRequest releaseFileSystemNfsV3LocksRequest) throws BadRequestException,
            IncompatibleParameterErrorException, FileSystemNotFoundException, ServiceLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReleaseFileSystemNfsV3LocksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ReleaseFileSystemNfsV3LocksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, releaseFileSystemNfsV3LocksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReleaseFileSystemNfsV3Locks");

            return clientHandler
                    .execute(new ClientExecutionParams<ReleaseFileSystemNfsV3LocksRequest, ReleaseFileSystemNfsV3LocksResponse>()
                            .withOperationName("ReleaseFileSystemNfsV3Locks").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(releaseFileSystemNfsV3LocksRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ReleaseFileSystemNfsV3LocksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an Amazon FSx for OpenZFS volume to the state saved by the specified snapshot.
     * </p>
     *
     * @param restoreVolumeFromSnapshotRequest
     * @return Result of the RestoreVolumeFromSnapshot operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws VolumeNotFoundException
     *         No Amazon FSx for NetApp ONTAP volumes were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.RestoreVolumeFromSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/RestoreVolumeFromSnapshot" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RestoreVolumeFromSnapshotResponse restoreVolumeFromSnapshot(
            RestoreVolumeFromSnapshotRequest restoreVolumeFromSnapshotRequest) throws BadRequestException,
            InternalServerErrorException, VolumeNotFoundException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestoreVolumeFromSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RestoreVolumeFromSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreVolumeFromSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreVolumeFromSnapshot");

            return clientHandler
                    .execute(new ClientExecutionParams<RestoreVolumeFromSnapshotRequest, RestoreVolumeFromSnapshotResponse>()
                            .withOperationName("RestoreVolumeFromSnapshot").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(restoreVolumeFromSnapshotRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RestoreVolumeFromSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags an Amazon FSx resource.
     * </p>
     *
     * @param tagResourceRequest
     *        The request object for the <code>TagResource</code> operation.
     * @return Result of the TagResource operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws ResourceNotFoundException
     *         The resource specified by the Amazon Resource Name (ARN) can't be found.
     * @throws NotServiceResourceErrorException
     *         The resource specified for the tagging operation is not a resource type owned by Amazon FSx. Use the API
     *         of the relevant service to perform the operation.
     * @throws ResourceDoesNotSupportTaggingException
     *         The resource specified does not support tagging.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws BadRequestException,
            InternalServerErrorException, ResourceNotFoundException, NotServiceResourceErrorException,
            ResourceDoesNotSupportTaggingException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This action removes a tag from an Amazon FSx resource.
     * </p>
     *
     * @param untagResourceRequest
     *        The request object for <code>UntagResource</code> action.
     * @return Result of the UntagResource operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws ResourceNotFoundException
     *         The resource specified by the Amazon Resource Name (ARN) can't be found.
     * @throws NotServiceResourceErrorException
     *         The resource specified for the tagging operation is not a resource type owned by Amazon FSx. Use the API
     *         of the relevant service to perform the operation.
     * @throws ResourceDoesNotSupportTaggingException
     *         The resource specified does not support tagging.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws BadRequestException,
            InternalServerErrorException, ResourceNotFoundException, NotServiceResourceErrorException,
            ResourceDoesNotSupportTaggingException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of an existing data repository association on an Amazon FSx for Lustre file system.
     * Data repository associations are supported only for file systems with the <code>Persistent_2</code> deployment
     * type.
     * </p>
     *
     * @param updateDataRepositoryAssociationRequest
     * @return Result of the UpdateDataRepositoryAssociation operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws DataRepositoryAssociationNotFoundException
     *         No data repository associations were found based upon the supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateDataRepositoryAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateDataRepositoryAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDataRepositoryAssociationResponse updateDataRepositoryAssociation(
            UpdateDataRepositoryAssociationRequest updateDataRepositoryAssociationRequest) throws BadRequestException,
            IncompatibleParameterErrorException, DataRepositoryAssociationNotFoundException, ServiceLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDataRepositoryAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDataRepositoryAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateDataRepositoryAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataRepositoryAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateDataRepositoryAssociationRequest, UpdateDataRepositoryAssociationResponse>()
                            .withOperationName("UpdateDataRepositoryAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateDataRepositoryAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateDataRepositoryAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation to update the configuration of an existing Amazon FSx file system. You can update multiple
     * properties in a single request.
     * </p>
     * <p>
     * For Amazon FSx for Windows File Server file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AuditLogConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SelfManagedActiveDirectoryConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ThroughputCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For FSx for Lustre file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AutoImportPolicy</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DataCompressionType</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For FSx for ONTAP file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FsxAdminPassword</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For the Amazon FSx for OpenZFS file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CopyTagsToBackups</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CopyTagsToVolumes</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DiskIopsConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ThroughputCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateFileSystemRequest
     *        The request object for the <code>UpdateFileSystem</code> operation.
     * @return Result of the UpdateFileSystem operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws MissingFileSystemConfigurationException
     *         A file system configuration is required for this operation.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateFileSystemResponse updateFileSystem(UpdateFileSystemRequest updateFileSystemRequest) throws BadRequestException,
            UnsupportedOperationException, IncompatibleParameterErrorException, InternalServerErrorException,
            FileSystemNotFoundException, MissingFileSystemConfigurationException, ServiceLimitExceededException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFileSystemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFileSystem");

            return clientHandler.execute(new ClientExecutionParams<UpdateFileSystemRequest, UpdateFileSystemResponse>()
                    .withOperationName("UpdateFileSystem").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateFileSystemRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFileSystemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the name of a snapshot.
     * </p>
     *
     * @param updateSnapshotRequest
     * @return Result of the UpdateSnapshot operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws SnapshotNotFoundException
     *         No Amazon FSx snapshots were found based on the supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateSnapshotResponse updateSnapshot(UpdateSnapshotRequest updateSnapshotRequest) throws BadRequestException,
            SnapshotNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateSnapshotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSnapshot");

            return clientHandler.execute(new ClientExecutionParams<UpdateSnapshotRequest, UpdateSnapshotResponse>()
                    .withOperationName("UpdateSnapshot").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an Amazon FSx for ONTAP storage virtual machine (SVM).
     * </p>
     *
     * @param updateStorageVirtualMachineRequest
     * @return Result of the UpdateStorageVirtualMachine operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws StorageVirtualMachineNotFoundException
     *         No Amazon FSx for NetApp ONTAP SVMs were found based upon the supplied parameters.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateStorageVirtualMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateStorageVirtualMachine"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateStorageVirtualMachineResponse updateStorageVirtualMachine(
            UpdateStorageVirtualMachineRequest updateStorageVirtualMachineRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, StorageVirtualMachineNotFoundException,
            UnsupportedOperationException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateStorageVirtualMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateStorageVirtualMachineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStorageVirtualMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStorageVirtualMachine");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateStorageVirtualMachineRequest, UpdateStorageVirtualMachineResponse>()
                            .withOperationName("UpdateStorageVirtualMachine").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateStorageVirtualMachineRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateStorageVirtualMachineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of an Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volume.
     * </p>
     *
     * @param updateVolumeRequest
     * @return Result of the UpdateVolume operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws MissingVolumeConfigurationException
     *         A volume configuration is required for this operation.
     * @throws VolumeNotFoundException
     *         No Amazon FSx for NetApp ONTAP volumes were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateVolume" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateVolumeResponse updateVolume(UpdateVolumeRequest updateVolumeRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, MissingVolumeConfigurationException,
            VolumeNotFoundException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateVolumeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVolume");

            return clientHandler.execute(new ClientExecutionParams<UpdateVolumeRequest, UpdateVolumeResponse>()
                    .withOperationName("UpdateVolume").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateVolumeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateVolumeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FSxException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidImportPath")
                                .exceptionBuilderSupplier(InvalidImportPathException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNetworkSettings")
                                .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileSystemNotFound")
                                .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound")
                                .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging")
                                .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound")
                                .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SourceBackupUnavailable")
                                .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperation")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound")
                                .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupRestoring")
                                .exceptionBuilderSupplier(BackupRestoringException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupInProgress")
                                .exceptionBuilderSupplier(BackupInProgressException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidExportPath")
                                .exceptionBuilderSupplier(InvalidExportPathException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRegion")
                                .exceptionBuilderSupplier(InvalidRegionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration")
                                .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey")
                                .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting")
                                .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ")
                                .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupNotFound")
                                .exceptionBuilderSupplier(BackupNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey")
                                .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("VolumeNotFound")
                                .exceptionBuilderSupplier(VolumeNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLimitExceeded")
                                .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotServiceResourceError")
                                .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration")
                                .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupBeingCopied")
                                .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncompatibleParameterError")
                                .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded")
                                .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType")
                                .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SnapshotNotFound")
                                .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput")
                                .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActiveDirectoryError")
                                .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequest")
                                .exceptionBuilderSupplier(BadRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends FSxRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
