/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration of a data repository association that links an Amazon FSx for Lustre file system to an Amazon S3
 * bucket. The data repository association configuration object is returned in the response of the following operations:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>CreateDataRepositoryAssociation</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>UpdateDataRepositoryAssociation</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>DescribeDataRepositoryAssociations</code>
 * </p>
 * </li>
 * </ul>
 * <p>
 * Data repository associations are supported only for file systems with the <code>Persistent_2</code> deployment type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataRepositoryAssociation implements SdkPojo, Serializable,
        ToCopyableBuilder<DataRepositoryAssociation.Builder, DataRepositoryAssociation> {
    private static final SdkField<String> ASSOCIATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociationId").getter(getter(DataRepositoryAssociation::associationId))
            .setter(setter(Builder::associationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationId").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceARN").getter(getter(DataRepositoryAssociation::resourceARN))
            .setter(setter(Builder::resourceARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceARN").build()).build();

    private static final SdkField<String> FILE_SYSTEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemId").getter(getter(DataRepositoryAssociation::fileSystemId))
            .setter(setter(Builder::fileSystemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemId").build()).build();

    private static final SdkField<String> LIFECYCLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Lifecycle").getter(getter(DataRepositoryAssociation::lifecycleAsString))
            .setter(setter(Builder::lifecycle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<DataRepositoryFailureDetails> FAILURE_DETAILS_FIELD = SdkField
            .<DataRepositoryFailureDetails> builder(MarshallingType.SDK_POJO).memberName("FailureDetails")
            .getter(getter(DataRepositoryAssociation::failureDetails)).setter(setter(Builder::failureDetails))
            .constructor(DataRepositoryFailureDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureDetails").build()).build();

    private static final SdkField<String> FILE_SYSTEM_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemPath").getter(getter(DataRepositoryAssociation::fileSystemPath))
            .setter(setter(Builder::fileSystemPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemPath").build()).build();

    private static final SdkField<String> DATA_REPOSITORY_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataRepositoryPath").getter(getter(DataRepositoryAssociation::dataRepositoryPath))
            .setter(setter(Builder::dataRepositoryPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataRepositoryPath").build())
            .build();

    private static final SdkField<Boolean> BATCH_IMPORT_META_DATA_ON_CREATE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("BatchImportMetaDataOnCreate")
            .getter(getter(DataRepositoryAssociation::batchImportMetaDataOnCreate))
            .setter(setter(Builder::batchImportMetaDataOnCreate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BatchImportMetaDataOnCreate")
                    .build()).build();

    private static final SdkField<Integer> IMPORTED_FILE_CHUNK_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ImportedFileChunkSize").getter(getter(DataRepositoryAssociation::importedFileChunkSize))
            .setter(setter(Builder::importedFileChunkSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImportedFileChunkSize").build())
            .build();

    private static final SdkField<S3DataRepositoryConfiguration> S3_FIELD = SdkField
            .<S3DataRepositoryConfiguration> builder(MarshallingType.SDK_POJO).memberName("S3")
            .getter(getter(DataRepositoryAssociation::s3)).setter(setter(Builder::s3))
            .constructor(S3DataRepositoryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(DataRepositoryAssociation::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(DataRepositoryAssociation::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASSOCIATION_ID_FIELD,
            RESOURCE_ARN_FIELD, FILE_SYSTEM_ID_FIELD, LIFECYCLE_FIELD, FAILURE_DETAILS_FIELD, FILE_SYSTEM_PATH_FIELD,
            DATA_REPOSITORY_PATH_FIELD, BATCH_IMPORT_META_DATA_ON_CREATE_FIELD, IMPORTED_FILE_CHUNK_SIZE_FIELD, S3_FIELD,
            TAGS_FIELD, CREATION_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String associationId;

    private final String resourceARN;

    private final String fileSystemId;

    private final String lifecycle;

    private final DataRepositoryFailureDetails failureDetails;

    private final String fileSystemPath;

    private final String dataRepositoryPath;

    private final Boolean batchImportMetaDataOnCreate;

    private final Integer importedFileChunkSize;

    private final S3DataRepositoryConfiguration s3;

    private final List<Tag> tags;

    private final Instant creationTime;

    private DataRepositoryAssociation(BuilderImpl builder) {
        this.associationId = builder.associationId;
        this.resourceARN = builder.resourceARN;
        this.fileSystemId = builder.fileSystemId;
        this.lifecycle = builder.lifecycle;
        this.failureDetails = builder.failureDetails;
        this.fileSystemPath = builder.fileSystemPath;
        this.dataRepositoryPath = builder.dataRepositoryPath;
        this.batchImportMetaDataOnCreate = builder.batchImportMetaDataOnCreate;
        this.importedFileChunkSize = builder.importedFileChunkSize;
        this.s3 = builder.s3;
        this.tags = builder.tags;
        this.creationTime = builder.creationTime;
    }

    /**
     * <p>
     * The system-generated, unique ID of the data repository association.
     * </p>
     * 
     * @return The system-generated, unique ID of the data repository association.
     */
    public final String associationId() {
        return associationId;
    }

    /**
     * Returns the value of the ResourceARN property for this object.
     * 
     * @return The value of the ResourceARN property for this object.
     */
    public final String resourceARN() {
        return resourceARN;
    }

    /**
     * Returns the value of the FileSystemId property for this object.
     * 
     * @return The value of the FileSystemId property for this object.
     */
    public final String fileSystemId() {
        return fileSystemId;
    }

    /**
     * <p>
     * Describes the state of a data repository association. The lifecycle can have the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The data repository association between the FSx file system and the S3 data repository is
     * being created. The data repository is unavailable.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The data repository association is available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket or automatically
     * export updates to the S3 bucket until the data repository association configuration is corrected.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that might
     * affect its availability.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link DataRepositoryLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return Describes the state of a data repository association. The lifecycle can have the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The data repository association between the FSx file system and the S3 data
     *         repository is being created. The data repository is unavailable.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The data repository association is available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket or
     *         automatically export updates to the S3 bucket until the data repository association configuration is
     *         corrected.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that
     *         might affect its availability.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
     *         </p>
     *         </li>
     * @see DataRepositoryLifecycle
     */
    public final DataRepositoryLifecycle lifecycle() {
        return DataRepositoryLifecycle.fromValue(lifecycle);
    }

    /**
     * <p>
     * Describes the state of a data repository association. The lifecycle can have the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The data repository association between the FSx file system and the S3 data repository is
     * being created. The data repository is unavailable.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The data repository association is available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket or automatically
     * export updates to the S3 bucket until the data repository association configuration is corrected.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that might
     * affect its availability.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link DataRepositoryLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return Describes the state of a data repository association. The lifecycle can have the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The data repository association between the FSx file system and the S3 data
     *         repository is being created. The data repository is unavailable.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The data repository association is available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket or
     *         automatically export updates to the S3 bucket until the data repository association configuration is
     *         corrected.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that
     *         might affect its availability.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
     *         </p>
     *         </li>
     * @see DataRepositoryLifecycle
     */
    public final String lifecycleAsString() {
        return lifecycle;
    }

    /**
     * Returns the value of the FailureDetails property for this object.
     * 
     * @return The value of the FailureDetails property for this object.
     */
    public final DataRepositoryFailureDetails failureDetails() {
        return failureDetails;
    }

    /**
     * <p>
     * A path on the file system that points to a high-level directory (such as <code>/ns1/</code>) or subdirectory
     * (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with <code>DataRepositoryPath</code>. The leading
     * forward slash in the name is required. Two data repository associations cannot have overlapping file system
     * paths. For example, if a data repository is associated with file system path <code>/ns1/</code>, then you cannot
     * link another data repository with file system path <code>/ns1/ns2</code>.
     * </p>
     * <p>
     * This path specifies where in your file system files will be exported from or imported to. This file system
     * directory can be linked to only one Amazon S3 bucket, and no other S3 bucket can be linked to the directory.
     * </p>
     * 
     * @return A path on the file system that points to a high-level directory (such as <code>/ns1/</code>) or
     *         subdirectory (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with
     *         <code>DataRepositoryPath</code>. The leading forward slash in the name is required. Two data repository
     *         associations cannot have overlapping file system paths. For example, if a data repository is associated
     *         with file system path <code>/ns1/</code>, then you cannot link another data repository with file system
     *         path <code>/ns1/ns2</code>.</p>
     *         <p>
     *         This path specifies where in your file system files will be exported from or imported to. This file
     *         system directory can be linked to only one Amazon S3 bucket, and no other S3 bucket can be linked to the
     *         directory.
     */
    public final String fileSystemPath() {
        return fileSystemPath;
    }

    /**
     * <p>
     * The path to the Amazon S3 data repository that will be linked to the file system. The path can be an S3 bucket or
     * prefix in the format <code>s3://myBucket/myPrefix/</code>. This path specifies where in the S3 data repository
     * files will be imported from or exported to.
     * </p>
     * 
     * @return The path to the Amazon S3 data repository that will be linked to the file system. The path can be an S3
     *         bucket or prefix in the format <code>s3://myBucket/myPrefix/</code>. This path specifies where in the S3
     *         data repository files will be imported from or exported to.
     */
    public final String dataRepositoryPath() {
        return dataRepositoryPath;
    }

    /**
     * <p>
     * A boolean flag indicating whether an import data repository task to import metadata should run after the data
     * repository association is created. The task runs if this flag is set to <code>true</code>.
     * </p>
     * 
     * @return A boolean flag indicating whether an import data repository task to import metadata should run after the
     *         data repository association is created. The task runs if this flag is set to <code>true</code>.
     */
    public final Boolean batchImportMetaDataOnCreate() {
        return batchImportMetaDataOnCreate;
    }

    /**
     * <p>
     * For files imported from a data repository, this value determines the stripe count and maximum amount of data per
     * file (in MiB) stored on a single physical disk. The maximum number of disks that a single file can be striped
     * across is limited by the total number of disks that make up the file system.
     * </p>
     * <p>
     * The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3 objects have a
     * maximum size of 5 TB.
     * </p>
     * 
     * @return For files imported from a data repository, this value determines the stripe count and maximum amount of
     *         data per file (in MiB) stored on a single physical disk. The maximum number of disks that a single file
     *         can be striped across is limited by the total number of disks that make up the file system.</p>
     *         <p>
     *         The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3
     *         objects have a maximum size of 5 TB.
     */
    public final Integer importedFileChunkSize() {
        return importedFileChunkSize;
    }

    /**
     * <p>
     * The configuration for an Amazon S3 data repository linked to an Amazon FSx Lustre file system with a data
     * repository association. The configuration defines which file events (new, changed, or deleted files or
     * directories) are automatically imported from the linked data repository to the file system or automatically
     * exported from the file system to the data repository.
     * </p>
     * 
     * @return The configuration for an Amazon S3 data repository linked to an Amazon FSx Lustre file system with a data
     *         repository association. The configuration defines which file events (new, changed, or deleted files or
     *         directories) are automatically imported from the linked data repository to the file system or
     *         automatically exported from the file system to the data repository.
     */
    public final S3DataRepositoryConfiguration s3() {
        return s3;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the Tags property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The value of the Tags property for this object.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * Returns the value of the CreationTime property for this object.
     * 
     * @return The value of the CreationTime property for this object.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceARN());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureDetails());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemPath());
        hashCode = 31 * hashCode + Objects.hashCode(dataRepositoryPath());
        hashCode = 31 * hashCode + Objects.hashCode(batchImportMetaDataOnCreate());
        hashCode = 31 * hashCode + Objects.hashCode(importedFileChunkSize());
        hashCode = 31 * hashCode + Objects.hashCode(s3());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataRepositoryAssociation)) {
            return false;
        }
        DataRepositoryAssociation other = (DataRepositoryAssociation) obj;
        return Objects.equals(associationId(), other.associationId()) && Objects.equals(resourceARN(), other.resourceARN())
                && Objects.equals(fileSystemId(), other.fileSystemId())
                && Objects.equals(lifecycleAsString(), other.lifecycleAsString())
                && Objects.equals(failureDetails(), other.failureDetails())
                && Objects.equals(fileSystemPath(), other.fileSystemPath())
                && Objects.equals(dataRepositoryPath(), other.dataRepositoryPath())
                && Objects.equals(batchImportMetaDataOnCreate(), other.batchImportMetaDataOnCreate())
                && Objects.equals(importedFileChunkSize(), other.importedFileChunkSize()) && Objects.equals(s3(), other.s3())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(creationTime(), other.creationTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataRepositoryAssociation").add("AssociationId", associationId())
                .add("ResourceARN", resourceARN()).add("FileSystemId", fileSystemId()).add("Lifecycle", lifecycleAsString())
                .add("FailureDetails", failureDetails()).add("FileSystemPath", fileSystemPath())
                .add("DataRepositoryPath", dataRepositoryPath())
                .add("BatchImportMetaDataOnCreate", batchImportMetaDataOnCreate())
                .add("ImportedFileChunkSize", importedFileChunkSize()).add("S3", s3()).add("Tags", hasTags() ? tags() : null)
                .add("CreationTime", creationTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssociationId":
            return Optional.ofNullable(clazz.cast(associationId()));
        case "ResourceARN":
            return Optional.ofNullable(clazz.cast(resourceARN()));
        case "FileSystemId":
            return Optional.ofNullable(clazz.cast(fileSystemId()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycleAsString()));
        case "FailureDetails":
            return Optional.ofNullable(clazz.cast(failureDetails()));
        case "FileSystemPath":
            return Optional.ofNullable(clazz.cast(fileSystemPath()));
        case "DataRepositoryPath":
            return Optional.ofNullable(clazz.cast(dataRepositoryPath()));
        case "BatchImportMetaDataOnCreate":
            return Optional.ofNullable(clazz.cast(batchImportMetaDataOnCreate()));
        case "ImportedFileChunkSize":
            return Optional.ofNullable(clazz.cast(importedFileChunkSize()));
        case "S3":
            return Optional.ofNullable(clazz.cast(s3()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DataRepositoryAssociation, T> g) {
        return obj -> g.apply((DataRepositoryAssociation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataRepositoryAssociation> {
        /**
         * <p>
         * The system-generated, unique ID of the data repository association.
         * </p>
         * 
         * @param associationId
         *        The system-generated, unique ID of the data repository association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * Sets the value of the ResourceARN property for this object.
         *
         * @param resourceARN
         *        The new value for the ResourceARN property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceARN(String resourceARN);

        /**
         * Sets the value of the FileSystemId property for this object.
         *
         * @param fileSystemId
         *        The new value for the FileSystemId property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemId(String fileSystemId);

        /**
         * <p>
         * Describes the state of a data repository association. The lifecycle can have the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The data repository association between the FSx file system and the S3 data
         * repository is being created. The data repository is unavailable.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The data repository association is available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket or
         * automatically export updates to the S3 bucket until the data repository association configuration is
         * corrected.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that might
         * affect its availability.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        Describes the state of a data repository association. The lifecycle can have the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The data repository association between the FSx file system and the S3 data
         *        repository is being created. The data repository is unavailable.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The data repository association is available for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket or
         *        automatically export updates to the S3 bucket until the data repository association configuration is
         *        corrected.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that
         *        might affect its availability.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
         *        </p>
         *        </li>
         * @see DataRepositoryLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryLifecycle
         */
        Builder lifecycle(String lifecycle);

        /**
         * <p>
         * Describes the state of a data repository association. The lifecycle can have the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The data repository association between the FSx file system and the S3 data
         * repository is being created. The data repository is unavailable.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The data repository association is available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket or
         * automatically export updates to the S3 bucket until the data repository association configuration is
         * corrected.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that might
         * affect its availability.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        Describes the state of a data repository association. The lifecycle can have the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The data repository association between the FSx file system and the S3 data
         *        repository is being created. The data repository is unavailable.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The data repository association is available for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> - Amazon FSx cannot automatically import updates from the S3 bucket or
         *        automatically export updates to the S3 bucket until the data repository association configuration is
         *        corrected.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that
         *        might affect its availability.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
         *        </p>
         *        </li>
         * @see DataRepositoryLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryLifecycle
         */
        Builder lifecycle(DataRepositoryLifecycle lifecycle);

        /**
         * Sets the value of the FailureDetails property for this object.
         *
         * @param failureDetails
         *        The new value for the FailureDetails property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureDetails(DataRepositoryFailureDetails failureDetails);

        /**
         * Sets the value of the FailureDetails property for this object.
         *
         * This is a convenience that creates an instance of the {@link DataRepositoryFailureDetails.Builder} avoiding
         * the need to create one manually via {@link DataRepositoryFailureDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataRepositoryFailureDetails.Builder#build()} is called
         * immediately and its result is passed to {@link #failureDetails(DataRepositoryFailureDetails)}.
         * 
         * @param failureDetails
         *        a consumer that will call methods on {@link DataRepositoryFailureDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureDetails(DataRepositoryFailureDetails)
         */
        default Builder failureDetails(Consumer<DataRepositoryFailureDetails.Builder> failureDetails) {
            return failureDetails(DataRepositoryFailureDetails.builder().applyMutation(failureDetails).build());
        }

        /**
         * <p>
         * A path on the file system that points to a high-level directory (such as <code>/ns1/</code>) or subdirectory
         * (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with <code>DataRepositoryPath</code>. The leading
         * forward slash in the name is required. Two data repository associations cannot have overlapping file system
         * paths. For example, if a data repository is associated with file system path <code>/ns1/</code>, then you
         * cannot link another data repository with file system path <code>/ns1/ns2</code>.
         * </p>
         * <p>
         * This path specifies where in your file system files will be exported from or imported to. This file system
         * directory can be linked to only one Amazon S3 bucket, and no other S3 bucket can be linked to the directory.
         * </p>
         * 
         * @param fileSystemPath
         *        A path on the file system that points to a high-level directory (such as <code>/ns1/</code>) or
         *        subdirectory (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with
         *        <code>DataRepositoryPath</code>. The leading forward slash in the name is required. Two data
         *        repository associations cannot have overlapping file system paths. For example, if a data repository
         *        is associated with file system path <code>/ns1/</code>, then you cannot link another data repository
         *        with file system path <code>/ns1/ns2</code>.</p>
         *        <p>
         *        This path specifies where in your file system files will be exported from or imported to. This file
         *        system directory can be linked to only one Amazon S3 bucket, and no other S3 bucket can be linked to
         *        the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemPath(String fileSystemPath);

        /**
         * <p>
         * The path to the Amazon S3 data repository that will be linked to the file system. The path can be an S3
         * bucket or prefix in the format <code>s3://myBucket/myPrefix/</code>. This path specifies where in the S3 data
         * repository files will be imported from or exported to.
         * </p>
         * 
         * @param dataRepositoryPath
         *        The path to the Amazon S3 data repository that will be linked to the file system. The path can be an
         *        S3 bucket or prefix in the format <code>s3://myBucket/myPrefix/</code>. This path specifies where in
         *        the S3 data repository files will be imported from or exported to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRepositoryPath(String dataRepositoryPath);

        /**
         * <p>
         * A boolean flag indicating whether an import data repository task to import metadata should run after the data
         * repository association is created. The task runs if this flag is set to <code>true</code>.
         * </p>
         * 
         * @param batchImportMetaDataOnCreate
         *        A boolean flag indicating whether an import data repository task to import metadata should run after
         *        the data repository association is created. The task runs if this flag is set to <code>true</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder batchImportMetaDataOnCreate(Boolean batchImportMetaDataOnCreate);

        /**
         * <p>
         * For files imported from a data repository, this value determines the stripe count and maximum amount of data
         * per file (in MiB) stored on a single physical disk. The maximum number of disks that a single file can be
         * striped across is limited by the total number of disks that make up the file system.
         * </p>
         * <p>
         * The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3 objects
         * have a maximum size of 5 TB.
         * </p>
         * 
         * @param importedFileChunkSize
         *        For files imported from a data repository, this value determines the stripe count and maximum amount
         *        of data per file (in MiB) stored on a single physical disk. The maximum number of disks that a single
         *        file can be striped across is limited by the total number of disks that make up the file system.</p>
         *        <p>
         *        The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3
         *        objects have a maximum size of 5 TB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importedFileChunkSize(Integer importedFileChunkSize);

        /**
         * <p>
         * The configuration for an Amazon S3 data repository linked to an Amazon FSx Lustre file system with a data
         * repository association. The configuration defines which file events (new, changed, or deleted files or
         * directories) are automatically imported from the linked data repository to the file system or automatically
         * exported from the file system to the data repository.
         * </p>
         * 
         * @param s3
         *        The configuration for an Amazon S3 data repository linked to an Amazon FSx Lustre file system with a
         *        data repository association. The configuration defines which file events (new, changed, or deleted
         *        files or directories) are automatically imported from the linked data repository to the file system or
         *        automatically exported from the file system to the data repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3(S3DataRepositoryConfiguration s3);

        /**
         * <p>
         * The configuration for an Amazon S3 data repository linked to an Amazon FSx Lustre file system with a data
         * repository association. The configuration defines which file events (new, changed, or deleted files or
         * directories) are automatically imported from the linked data repository to the file system or automatically
         * exported from the file system to the data repository.
         * </p>
         * This is a convenience that creates an instance of the {@link S3DataRepositoryConfiguration.Builder} avoiding
         * the need to create one manually via {@link S3DataRepositoryConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3DataRepositoryConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #s3(S3DataRepositoryConfiguration)}.
         * 
         * @param s3
         *        a consumer that will call methods on {@link S3DataRepositoryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3(S3DataRepositoryConfiguration)
         */
        default Builder s3(Consumer<S3DataRepositoryConfiguration.Builder> s3) {
            return s3(S3DataRepositoryConfiguration.builder().applyMutation(s3).build());
        }

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * Sets the value of the CreationTime property for this object.
         *
         * @param creationTime
         *        The new value for the CreationTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);
    }

    static final class BuilderImpl implements Builder {
        private String associationId;

        private String resourceARN;

        private String fileSystemId;

        private String lifecycle;

        private DataRepositoryFailureDetails failureDetails;

        private String fileSystemPath;

        private String dataRepositoryPath;

        private Boolean batchImportMetaDataOnCreate;

        private Integer importedFileChunkSize;

        private S3DataRepositoryConfiguration s3;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Instant creationTime;

        private BuilderImpl() {
        }

        private BuilderImpl(DataRepositoryAssociation model) {
            associationId(model.associationId);
            resourceARN(model.resourceARN);
            fileSystemId(model.fileSystemId);
            lifecycle(model.lifecycle);
            failureDetails(model.failureDetails);
            fileSystemPath(model.fileSystemPath);
            dataRepositoryPath(model.dataRepositoryPath);
            batchImportMetaDataOnCreate(model.batchImportMetaDataOnCreate);
            importedFileChunkSize(model.importedFileChunkSize);
            s3(model.s3);
            tags(model.tags);
            creationTime(model.creationTime);
        }

        public final String getAssociationId() {
            return associationId;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        @Override
        @Transient
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final String getResourceARN() {
            return resourceARN;
        }

        public final void setResourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
        }

        @Override
        @Transient
        public final Builder resourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
            return this;
        }

        public final String getFileSystemId() {
            return fileSystemId;
        }

        public final void setFileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
        }

        @Override
        @Transient
        public final Builder fileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
            return this;
        }

        public final String getLifecycle() {
            return lifecycle;
        }

        public final void setLifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
        }

        @Override
        @Transient
        public final Builder lifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        @Override
        @Transient
        public final Builder lifecycle(DataRepositoryLifecycle lifecycle) {
            this.lifecycle(lifecycle == null ? null : lifecycle.toString());
            return this;
        }

        public final DataRepositoryFailureDetails.Builder getFailureDetails() {
            return failureDetails != null ? failureDetails.toBuilder() : null;
        }

        public final void setFailureDetails(DataRepositoryFailureDetails.BuilderImpl failureDetails) {
            this.failureDetails = failureDetails != null ? failureDetails.build() : null;
        }

        @Override
        @Transient
        public final Builder failureDetails(DataRepositoryFailureDetails failureDetails) {
            this.failureDetails = failureDetails;
            return this;
        }

        public final String getFileSystemPath() {
            return fileSystemPath;
        }

        public final void setFileSystemPath(String fileSystemPath) {
            this.fileSystemPath = fileSystemPath;
        }

        @Override
        @Transient
        public final Builder fileSystemPath(String fileSystemPath) {
            this.fileSystemPath = fileSystemPath;
            return this;
        }

        public final String getDataRepositoryPath() {
            return dataRepositoryPath;
        }

        public final void setDataRepositoryPath(String dataRepositoryPath) {
            this.dataRepositoryPath = dataRepositoryPath;
        }

        @Override
        @Transient
        public final Builder dataRepositoryPath(String dataRepositoryPath) {
            this.dataRepositoryPath = dataRepositoryPath;
            return this;
        }

        public final Boolean getBatchImportMetaDataOnCreate() {
            return batchImportMetaDataOnCreate;
        }

        public final void setBatchImportMetaDataOnCreate(Boolean batchImportMetaDataOnCreate) {
            this.batchImportMetaDataOnCreate = batchImportMetaDataOnCreate;
        }

        @Override
        @Transient
        public final Builder batchImportMetaDataOnCreate(Boolean batchImportMetaDataOnCreate) {
            this.batchImportMetaDataOnCreate = batchImportMetaDataOnCreate;
            return this;
        }

        public final Integer getImportedFileChunkSize() {
            return importedFileChunkSize;
        }

        public final void setImportedFileChunkSize(Integer importedFileChunkSize) {
            this.importedFileChunkSize = importedFileChunkSize;
        }

        @Override
        @Transient
        public final Builder importedFileChunkSize(Integer importedFileChunkSize) {
            this.importedFileChunkSize = importedFileChunkSize;
            return this;
        }

        public final S3DataRepositoryConfiguration.Builder getS3() {
            return s3 != null ? s3.toBuilder() : null;
        }

        public final void setS3(S3DataRepositoryConfiguration.BuilderImpl s3) {
            this.s3 = s3 != null ? s3.build() : null;
        }

        @Override
        @Transient
        public final Builder s3(S3DataRepositoryConfiguration s3) {
            this.s3 = s3;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        @Transient
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        @Override
        public DataRepositoryAssociation build() {
            return new DataRepositoryAssociation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
