/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for the Amazon FSx for Lustre file system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LustreFileSystemConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LustreFileSystemConfiguration.Builder, LustreFileSystemConfiguration> {
    private static final SdkField<String> WEEKLY_MAINTENANCE_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("WeeklyMaintenanceStartTime")
            .getter(getter(LustreFileSystemConfiguration::weeklyMaintenanceStartTime))
            .setter(setter(Builder::weeklyMaintenanceStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeeklyMaintenanceStartTime").build())
            .build();

    private static final SdkField<DataRepositoryConfiguration> DATA_REPOSITORY_CONFIGURATION_FIELD = SdkField
            .<DataRepositoryConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("DataRepositoryConfiguration")
            .getter(getter(LustreFileSystemConfiguration::dataRepositoryConfiguration))
            .setter(setter(Builder::dataRepositoryConfiguration))
            .constructor(DataRepositoryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataRepositoryConfiguration")
                    .build()).build();

    private static final SdkField<String> DEPLOYMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeploymentType").getter(getter(LustreFileSystemConfiguration::deploymentTypeAsString))
            .setter(setter(Builder::deploymentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeploymentType").build()).build();

    private static final SdkField<Integer> PER_UNIT_STORAGE_THROUGHPUT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("PerUnitStorageThroughput")
            .getter(getter(LustreFileSystemConfiguration::perUnitStorageThroughput))
            .setter(setter(Builder::perUnitStorageThroughput))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PerUnitStorageThroughput").build())
            .build();

    private static final SdkField<String> MOUNT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MountName").getter(getter(LustreFileSystemConfiguration::mountName)).setter(setter(Builder::mountName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MountName").build()).build();

    private static final SdkField<String> DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DailyAutomaticBackupStartTime")
            .getter(getter(LustreFileSystemConfiguration::dailyAutomaticBackupStartTime))
            .setter(setter(Builder::dailyAutomaticBackupStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DailyAutomaticBackupStartTime")
                    .build()).build();

    private static final SdkField<Integer> AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AutomaticBackupRetentionDays")
            .getter(getter(LustreFileSystemConfiguration::automaticBackupRetentionDays))
            .setter(setter(Builder::automaticBackupRetentionDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomaticBackupRetentionDays")
                    .build()).build();

    private static final SdkField<Boolean> COPY_TAGS_TO_BACKUPS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyTagsToBackups").getter(getter(LustreFileSystemConfiguration::copyTagsToBackups))
            .setter(setter(Builder::copyTagsToBackups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTagsToBackups").build()).build();

    private static final SdkField<String> DRIVE_CACHE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DriveCacheType").getter(getter(LustreFileSystemConfiguration::driveCacheTypeAsString))
            .setter(setter(Builder::driveCacheType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DriveCacheType").build()).build();

    private static final SdkField<String> DATA_COMPRESSION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataCompressionType").getter(getter(LustreFileSystemConfiguration::dataCompressionTypeAsString))
            .setter(setter(Builder::dataCompressionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataCompressionType").build())
            .build();

    private static final SdkField<LustreLogConfiguration> LOG_CONFIGURATION_FIELD = SdkField
            .<LustreLogConfiguration> builder(MarshallingType.SDK_POJO).memberName("LogConfiguration")
            .getter(getter(LustreFileSystemConfiguration::logConfiguration)).setter(setter(Builder::logConfiguration))
            .constructor(LustreLogConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            WEEKLY_MAINTENANCE_START_TIME_FIELD, DATA_REPOSITORY_CONFIGURATION_FIELD, DEPLOYMENT_TYPE_FIELD,
            PER_UNIT_STORAGE_THROUGHPUT_FIELD, MOUNT_NAME_FIELD, DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD,
            AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD, COPY_TAGS_TO_BACKUPS_FIELD, DRIVE_CACHE_TYPE_FIELD,
            DATA_COMPRESSION_TYPE_FIELD, LOG_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String weeklyMaintenanceStartTime;

    private final DataRepositoryConfiguration dataRepositoryConfiguration;

    private final String deploymentType;

    private final Integer perUnitStorageThroughput;

    private final String mountName;

    private final String dailyAutomaticBackupStartTime;

    private final Integer automaticBackupRetentionDays;

    private final Boolean copyTagsToBackups;

    private final String driveCacheType;

    private final String dataCompressionType;

    private final LustreLogConfiguration logConfiguration;

    private LustreFileSystemConfiguration(BuilderImpl builder) {
        this.weeklyMaintenanceStartTime = builder.weeklyMaintenanceStartTime;
        this.dataRepositoryConfiguration = builder.dataRepositoryConfiguration;
        this.deploymentType = builder.deploymentType;
        this.perUnitStorageThroughput = builder.perUnitStorageThroughput;
        this.mountName = builder.mountName;
        this.dailyAutomaticBackupStartTime = builder.dailyAutomaticBackupStartTime;
        this.automaticBackupRetentionDays = builder.automaticBackupRetentionDays;
        this.copyTagsToBackups = builder.copyTagsToBackups;
        this.driveCacheType = builder.driveCacheType;
        this.dataCompressionType = builder.dataCompressionType;
        this.logConfiguration = builder.logConfiguration;
    }

    /**
     * <p>
     * The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. Here, d is the
     * weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
     * </p>
     * 
     * @return The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. Here, d
     *         is the weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
     */
    public final String weeklyMaintenanceStartTime() {
        return weeklyMaintenanceStartTime;
    }

    /**
     * Returns the value of the DataRepositoryConfiguration property for this object.
     * 
     * @return The value of the DataRepositoryConfiguration property for this object.
     */
    public final DataRepositoryConfiguration dataRepositoryConfiguration() {
        return dataRepositoryConfiguration;
    }

    /**
     * <p>
     * The deployment type of the FSx for Lustre file system. <i>Scratch deployment type</i> is designed for temporary
     * storage and shorter-term processing of data.
     * </p>
     * <p>
     * <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need temporary
     * storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type provides in-transit
     * encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
     * </p>
     * <p>
     * The <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment type is used for longer-term storage and
     * workloads and encryption of data in transit. <code>PERSISTENT_2</code> is built on Lustre v2.12 and offers higher
     * <code>PerUnitStorageThroughput</code> (up to 1000 MB/s/TiB) along with a lower minimum storage capacity
     * requirement (600 GiB). To learn more about FSx for Lustre deployment types, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre deployment
     * options</a>.
     * </p>
     * <p>
     * The default is <code>SCRATCH_1</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link LustreDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return The deployment type of the FSx for Lustre file system. <i>Scratch deployment type</i> is designed for
     *         temporary storage and shorter-term processing of data.</p>
     *         <p>
     *         <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
     *         temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type
     *         provides in-transit encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
     *         </p>
     *         <p>
     *         The <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment type is used for longer-term
     *         storage and workloads and encryption of data in transit. <code>PERSISTENT_2</code> is built on Lustre
     *         v2.12 and offers higher <code>PerUnitStorageThroughput</code> (up to 1000 MB/s/TiB) along with a lower
     *         minimum storage capacity requirement (600 GiB). To learn more about FSx for Lustre deployment types, see
     *         <a href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre
     *         deployment options</a>.
     *         </p>
     *         <p>
     *         The default is <code>SCRATCH_1</code>.
     * @see LustreDeploymentType
     */
    public final LustreDeploymentType deploymentType() {
        return LustreDeploymentType.fromValue(deploymentType);
    }

    /**
     * <p>
     * The deployment type of the FSx for Lustre file system. <i>Scratch deployment type</i> is designed for temporary
     * storage and shorter-term processing of data.
     * </p>
     * <p>
     * <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need temporary
     * storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type provides in-transit
     * encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
     * </p>
     * <p>
     * The <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment type is used for longer-term storage and
     * workloads and encryption of data in transit. <code>PERSISTENT_2</code> is built on Lustre v2.12 and offers higher
     * <code>PerUnitStorageThroughput</code> (up to 1000 MB/s/TiB) along with a lower minimum storage capacity
     * requirement (600 GiB). To learn more about FSx for Lustre deployment types, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre deployment
     * options</a>.
     * </p>
     * <p>
     * The default is <code>SCRATCH_1</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link LustreDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return The deployment type of the FSx for Lustre file system. <i>Scratch deployment type</i> is designed for
     *         temporary storage and shorter-term processing of data.</p>
     *         <p>
     *         <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
     *         temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type
     *         provides in-transit encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
     *         </p>
     *         <p>
     *         The <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment type is used for longer-term
     *         storage and workloads and encryption of data in transit. <code>PERSISTENT_2</code> is built on Lustre
     *         v2.12 and offers higher <code>PerUnitStorageThroughput</code> (up to 1000 MB/s/TiB) along with a lower
     *         minimum storage capacity requirement (600 GiB). To learn more about FSx for Lustre deployment types, see
     *         <a href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre
     *         deployment options</a>.
     *         </p>
     *         <p>
     *         The default is <code>SCRATCH_1</code>.
     * @see LustreDeploymentType
     */
    public final String deploymentTypeAsString() {
        return deploymentType;
    }

    /**
     * <p>
     * Per unit storage throughput represents the megabytes per second of read or write throughput per 1 tebibyte of
     * storage provisioned. File system throughput capacity is equal to Storage capacity (TiB) *
     * PerUnitStorageThroughput (MB/s/TiB). This option is only valid for <code>PERSISTENT_1</code> and
     * <code>PERSISTENT_2</code> deployment types.
     * </p>
     * <p>
     * Valid values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>PERSISTENT_1</code> SSD storage: 50, 100, 200.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>PERSISTENT_1</code> HDD storage: 12, 40.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>PERSISTENT_2</code> SSD storage: 125, 250, 500, 1000.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Per unit storage throughput represents the megabytes per second of read or write throughput per 1
     *         tebibyte of storage provisioned. File system throughput capacity is equal to Storage capacity (TiB) *
     *         PerUnitStorageThroughput (MB/s/TiB). This option is only valid for <code>PERSISTENT_1</code> and
     *         <code>PERSISTENT_2</code> deployment types. </p>
     *         <p>
     *         Valid values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For <code>PERSISTENT_1</code> SSD storage: 50, 100, 200.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>PERSISTENT_1</code> HDD storage: 12, 40.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>PERSISTENT_2</code> SSD storage: 125, 250, 500, 1000.
     *         </p>
     *         </li>
     */
    public final Integer perUnitStorageThroughput() {
        return perUnitStorageThroughput;
    }

    /**
     * <p>
     * You use the <code>MountName</code> value when mounting the file system.
     * </p>
     * <p>
     * For the <code>SCRATCH_1</code> deployment type, this value is always "<code>fsx</code>". For
     * <code>SCRATCH_2</code>, <code>PERSISTENT_1</code>, and <code>PERSISTENT_2</code> deployment types, this value is
     * a string that is unique within an Amazon Web Services Region.
     * </p>
     * 
     * @return You use the <code>MountName</code> value when mounting the file system.</p>
     *         <p>
     *         For the <code>SCRATCH_1</code> deployment type, this value is always "<code>fsx</code>". For
     *         <code>SCRATCH_2</code>, <code>PERSISTENT_1</code>, and <code>PERSISTENT_2</code> deployment types, this
     *         value is a string that is unique within an Amazon Web Services Region.
     */
    public final String mountName() {
        return mountName;
    }

    /**
     * Returns the value of the DailyAutomaticBackupStartTime property for this object.
     * 
     * @return The value of the DailyAutomaticBackupStartTime property for this object.
     */
    public final String dailyAutomaticBackupStartTime() {
        return dailyAutomaticBackupStartTime;
    }

    /**
     * Returns the value of the AutomaticBackupRetentionDays property for this object.
     * 
     * @return The value of the AutomaticBackupRetentionDays property for this object.
     */
    public final Integer automaticBackupRetentionDays() {
        return automaticBackupRetentionDays;
    }

    /**
     * <p>
     * A boolean flag indicating whether tags on the file system are copied to backups. If it's set to true, all tags on
     * the file system are copied to all automatic backups and any user-initiated backups where the user doesn't specify
     * any tags. If this value is true, and you specify one or more tags, only the specified tags are copied to backups.
     * If you specify one or more tags when creating a user-initiated backup, no tags are copied from the file system,
     * regardless of this value. (Default = false)
     * </p>
     * 
     * @return A boolean flag indicating whether tags on the file system are copied to backups. If it's set to true, all
     *         tags on the file system are copied to all automatic backups and any user-initiated backups where the user
     *         doesn't specify any tags. If this value is true, and you specify one or more tags, only the specified
     *         tags are copied to backups. If you specify one or more tags when creating a user-initiated backup, no
     *         tags are copied from the file system, regardless of this value. (Default = false)
     */
    public final Boolean copyTagsToBackups() {
        return copyTagsToBackups;
    }

    /**
     * <p>
     * The type of drive cache used by <code>PERSISTENT_1</code> file systems that are provisioned with HDD storage
     * devices. This parameter is required when <code>StorageType</code> is HDD. When set to <code>READ</code> the file
     * system has an SSD storage cache that is sized to 20% of the file system's storage capacity. This improves the
     * performance for frequently accessed files by caching up to 20% of the total storage capacity.
     * </p>
     * <p>
     * This parameter is required when <code>StorageType</code> is set to HDD.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #driveCacheType}
     * will return {@link DriveCacheType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #driveCacheTypeAsString}.
     * </p>
     * 
     * @return The type of drive cache used by <code>PERSISTENT_1</code> file systems that are provisioned with HDD
     *         storage devices. This parameter is required when <code>StorageType</code> is HDD. When set to
     *         <code>READ</code> the file system has an SSD storage cache that is sized to 20% of the file system's
     *         storage capacity. This improves the performance for frequently accessed files by caching up to 20% of the
     *         total storage capacity.</p>
     *         <p>
     *         This parameter is required when <code>StorageType</code> is set to HDD.
     * @see DriveCacheType
     */
    public final DriveCacheType driveCacheType() {
        return DriveCacheType.fromValue(driveCacheType);
    }

    /**
     * <p>
     * The type of drive cache used by <code>PERSISTENT_1</code> file systems that are provisioned with HDD storage
     * devices. This parameter is required when <code>StorageType</code> is HDD. When set to <code>READ</code> the file
     * system has an SSD storage cache that is sized to 20% of the file system's storage capacity. This improves the
     * performance for frequently accessed files by caching up to 20% of the total storage capacity.
     * </p>
     * <p>
     * This parameter is required when <code>StorageType</code> is set to HDD.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #driveCacheType}
     * will return {@link DriveCacheType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #driveCacheTypeAsString}.
     * </p>
     * 
     * @return The type of drive cache used by <code>PERSISTENT_1</code> file systems that are provisioned with HDD
     *         storage devices. This parameter is required when <code>StorageType</code> is HDD. When set to
     *         <code>READ</code> the file system has an SSD storage cache that is sized to 20% of the file system's
     *         storage capacity. This improves the performance for frequently accessed files by caching up to 20% of the
     *         total storage capacity.</p>
     *         <p>
     *         This parameter is required when <code>StorageType</code> is set to HDD.
     * @see DriveCacheType
     */
    public final String driveCacheTypeAsString() {
        return driveCacheType;
    }

    /**
     * <p>
     * The data compression configuration for the file system. <code>DataCompressionType</code> can have the following
     * values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code> - Data compression is turned off for the file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LZ4</code> - Data compression is turned on with the LZ4 algorithm.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-compression.html">Lustre data compression</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dataCompressionType} will return {@link DataCompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dataCompressionTypeAsString}.
     * </p>
     * 
     * @return The data compression configuration for the file system. <code>DataCompressionType</code> can have the
     *         following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code> - Data compression is turned off for the file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LZ4</code> - Data compression is turned on with the LZ4 algorithm.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-compression.html">Lustre data
     *         compression</a>.
     * @see DataCompressionType
     */
    public final DataCompressionType dataCompressionType() {
        return DataCompressionType.fromValue(dataCompressionType);
    }

    /**
     * <p>
     * The data compression configuration for the file system. <code>DataCompressionType</code> can have the following
     * values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code> - Data compression is turned off for the file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LZ4</code> - Data compression is turned on with the LZ4 algorithm.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-compression.html">Lustre data compression</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dataCompressionType} will return {@link DataCompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dataCompressionTypeAsString}.
     * </p>
     * 
     * @return The data compression configuration for the file system. <code>DataCompressionType</code> can have the
     *         following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code> - Data compression is turned off for the file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LZ4</code> - Data compression is turned on with the LZ4 algorithm.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-compression.html">Lustre data
     *         compression</a>.
     * @see DataCompressionType
     */
    public final String dataCompressionTypeAsString() {
        return dataCompressionType;
    }

    /**
     * <p>
     * The Lustre logging configuration. Lustre logging writes the enabled log events for your file system to Amazon
     * CloudWatch Logs.
     * </p>
     * 
     * @return The Lustre logging configuration. Lustre logging writes the enabled log events for your file system to
     *         Amazon CloudWatch Logs.
     */
    public final LustreLogConfiguration logConfiguration() {
        return logConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(weeklyMaintenanceStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(dataRepositoryConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(perUnitStorageThroughput());
        hashCode = 31 * hashCode + Objects.hashCode(mountName());
        hashCode = 31 * hashCode + Objects.hashCode(dailyAutomaticBackupStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(automaticBackupRetentionDays());
        hashCode = 31 * hashCode + Objects.hashCode(copyTagsToBackups());
        hashCode = 31 * hashCode + Objects.hashCode(driveCacheTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dataCompressionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(logConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LustreFileSystemConfiguration)) {
            return false;
        }
        LustreFileSystemConfiguration other = (LustreFileSystemConfiguration) obj;
        return Objects.equals(weeklyMaintenanceStartTime(), other.weeklyMaintenanceStartTime())
                && Objects.equals(dataRepositoryConfiguration(), other.dataRepositoryConfiguration())
                && Objects.equals(deploymentTypeAsString(), other.deploymentTypeAsString())
                && Objects.equals(perUnitStorageThroughput(), other.perUnitStorageThroughput())
                && Objects.equals(mountName(), other.mountName())
                && Objects.equals(dailyAutomaticBackupStartTime(), other.dailyAutomaticBackupStartTime())
                && Objects.equals(automaticBackupRetentionDays(), other.automaticBackupRetentionDays())
                && Objects.equals(copyTagsToBackups(), other.copyTagsToBackups())
                && Objects.equals(driveCacheTypeAsString(), other.driveCacheTypeAsString())
                && Objects.equals(dataCompressionTypeAsString(), other.dataCompressionTypeAsString())
                && Objects.equals(logConfiguration(), other.logConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LustreFileSystemConfiguration").add("WeeklyMaintenanceStartTime", weeklyMaintenanceStartTime())
                .add("DataRepositoryConfiguration", dataRepositoryConfiguration())
                .add("DeploymentType", deploymentTypeAsString()).add("PerUnitStorageThroughput", perUnitStorageThroughput())
                .add("MountName", mountName()).add("DailyAutomaticBackupStartTime", dailyAutomaticBackupStartTime())
                .add("AutomaticBackupRetentionDays", automaticBackupRetentionDays())
                .add("CopyTagsToBackups", copyTagsToBackups()).add("DriveCacheType", driveCacheTypeAsString())
                .add("DataCompressionType", dataCompressionTypeAsString()).add("LogConfiguration", logConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WeeklyMaintenanceStartTime":
            return Optional.ofNullable(clazz.cast(weeklyMaintenanceStartTime()));
        case "DataRepositoryConfiguration":
            return Optional.ofNullable(clazz.cast(dataRepositoryConfiguration()));
        case "DeploymentType":
            return Optional.ofNullable(clazz.cast(deploymentTypeAsString()));
        case "PerUnitStorageThroughput":
            return Optional.ofNullable(clazz.cast(perUnitStorageThroughput()));
        case "MountName":
            return Optional.ofNullable(clazz.cast(mountName()));
        case "DailyAutomaticBackupStartTime":
            return Optional.ofNullable(clazz.cast(dailyAutomaticBackupStartTime()));
        case "AutomaticBackupRetentionDays":
            return Optional.ofNullable(clazz.cast(automaticBackupRetentionDays()));
        case "CopyTagsToBackups":
            return Optional.ofNullable(clazz.cast(copyTagsToBackups()));
        case "DriveCacheType":
            return Optional.ofNullable(clazz.cast(driveCacheTypeAsString()));
        case "DataCompressionType":
            return Optional.ofNullable(clazz.cast(dataCompressionTypeAsString()));
        case "LogConfiguration":
            return Optional.ofNullable(clazz.cast(logConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LustreFileSystemConfiguration, T> g) {
        return obj -> g.apply((LustreFileSystemConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LustreFileSystemConfiguration> {
        /**
         * <p>
         * The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. Here, d is
         * the weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
         * </p>
         * 
         * @param weeklyMaintenanceStartTime
         *        The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. Here,
         *        d is the weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime);

        /**
         * Sets the value of the DataRepositoryConfiguration property for this object.
         *
         * @param dataRepositoryConfiguration
         *        The new value for the DataRepositoryConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRepositoryConfiguration(DataRepositoryConfiguration dataRepositoryConfiguration);

        /**
         * Sets the value of the DataRepositoryConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link DataRepositoryConfiguration.Builder}
         * avoiding the need to create one manually via {@link DataRepositoryConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataRepositoryConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #dataRepositoryConfiguration(DataRepositoryConfiguration)}.
         * 
         * @param dataRepositoryConfiguration
         *        a consumer that will call methods on {@link DataRepositoryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataRepositoryConfiguration(DataRepositoryConfiguration)
         */
        default Builder dataRepositoryConfiguration(Consumer<DataRepositoryConfiguration.Builder> dataRepositoryConfiguration) {
            return dataRepositoryConfiguration(DataRepositoryConfiguration.builder().applyMutation(dataRepositoryConfiguration)
                    .build());
        }

        /**
         * <p>
         * The deployment type of the FSx for Lustre file system. <i>Scratch deployment type</i> is designed for
         * temporary storage and shorter-term processing of data.
         * </p>
         * <p>
         * <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
         * temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type provides
         * in-transit encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
         * </p>
         * <p>
         * The <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment type is used for longer-term storage
         * and workloads and encryption of data in transit. <code>PERSISTENT_2</code> is built on Lustre v2.12 and
         * offers higher <code>PerUnitStorageThroughput</code> (up to 1000 MB/s/TiB) along with a lower minimum storage
         * capacity requirement (600 GiB). To learn more about FSx for Lustre deployment types, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre
         * deployment options</a>.
         * </p>
         * <p>
         * The default is <code>SCRATCH_1</code>.
         * </p>
         * 
         * @param deploymentType
         *        The deployment type of the FSx for Lustre file system. <i>Scratch deployment type</i> is designed for
         *        temporary storage and shorter-term processing of data.</p>
         *        <p>
         *        <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
         *        temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type
         *        provides in-transit encryption of data and higher burst throughput capacity than
         *        <code>SCRATCH_1</code>.
         *        </p>
         *        <p>
         *        The <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment type is used for longer-term
         *        storage and workloads and encryption of data in transit. <code>PERSISTENT_2</code> is built on Lustre
         *        v2.12 and offers higher <code>PerUnitStorageThroughput</code> (up to 1000 MB/s/TiB) along with a lower
         *        minimum storage capacity requirement (600 GiB). To learn more about FSx for Lustre deployment types,
         *        see <a href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for
         *        Lustre deployment options</a>.
         *        </p>
         *        <p>
         *        The default is <code>SCRATCH_1</code>.
         * @see LustreDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LustreDeploymentType
         */
        Builder deploymentType(String deploymentType);

        /**
         * <p>
         * The deployment type of the FSx for Lustre file system. <i>Scratch deployment type</i> is designed for
         * temporary storage and shorter-term processing of data.
         * </p>
         * <p>
         * <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
         * temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type provides
         * in-transit encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
         * </p>
         * <p>
         * The <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment type is used for longer-term storage
         * and workloads and encryption of data in transit. <code>PERSISTENT_2</code> is built on Lustre v2.12 and
         * offers higher <code>PerUnitStorageThroughput</code> (up to 1000 MB/s/TiB) along with a lower minimum storage
         * capacity requirement (600 GiB). To learn more about FSx for Lustre deployment types, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre
         * deployment options</a>.
         * </p>
         * <p>
         * The default is <code>SCRATCH_1</code>.
         * </p>
         * 
         * @param deploymentType
         *        The deployment type of the FSx for Lustre file system. <i>Scratch deployment type</i> is designed for
         *        temporary storage and shorter-term processing of data.</p>
         *        <p>
         *        <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
         *        temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type
         *        provides in-transit encryption of data and higher burst throughput capacity than
         *        <code>SCRATCH_1</code>.
         *        </p>
         *        <p>
         *        The <code>PERSISTENT_1</code> and <code>PERSISTENT_2</code> deployment type is used for longer-term
         *        storage and workloads and encryption of data in transit. <code>PERSISTENT_2</code> is built on Lustre
         *        v2.12 and offers higher <code>PerUnitStorageThroughput</code> (up to 1000 MB/s/TiB) along with a lower
         *        minimum storage capacity requirement (600 GiB). To learn more about FSx for Lustre deployment types,
         *        see <a href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for
         *        Lustre deployment options</a>.
         *        </p>
         *        <p>
         *        The default is <code>SCRATCH_1</code>.
         * @see LustreDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LustreDeploymentType
         */
        Builder deploymentType(LustreDeploymentType deploymentType);

        /**
         * <p>
         * Per unit storage throughput represents the megabytes per second of read or write throughput per 1 tebibyte of
         * storage provisioned. File system throughput capacity is equal to Storage capacity (TiB) *
         * PerUnitStorageThroughput (MB/s/TiB). This option is only valid for <code>PERSISTENT_1</code> and
         * <code>PERSISTENT_2</code> deployment types.
         * </p>
         * <p>
         * Valid values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For <code>PERSISTENT_1</code> SSD storage: 50, 100, 200.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>PERSISTENT_1</code> HDD storage: 12, 40.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>PERSISTENT_2</code> SSD storage: 125, 250, 500, 1000.
         * </p>
         * </li>
         * </ul>
         * 
         * @param perUnitStorageThroughput
         *        Per unit storage throughput represents the megabytes per second of read or write throughput per 1
         *        tebibyte of storage provisioned. File system throughput capacity is equal to Storage capacity (TiB) *
         *        PerUnitStorageThroughput (MB/s/TiB). This option is only valid for <code>PERSISTENT_1</code> and
         *        <code>PERSISTENT_2</code> deployment types. </p>
         *        <p>
         *        Valid values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For <code>PERSISTENT_1</code> SSD storage: 50, 100, 200.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>PERSISTENT_1</code> HDD storage: 12, 40.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>PERSISTENT_2</code> SSD storage: 125, 250, 500, 1000.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder perUnitStorageThroughput(Integer perUnitStorageThroughput);

        /**
         * <p>
         * You use the <code>MountName</code> value when mounting the file system.
         * </p>
         * <p>
         * For the <code>SCRATCH_1</code> deployment type, this value is always "<code>fsx</code>". For
         * <code>SCRATCH_2</code>, <code>PERSISTENT_1</code>, and <code>PERSISTENT_2</code> deployment types, this value
         * is a string that is unique within an Amazon Web Services Region.
         * </p>
         * 
         * @param mountName
         *        You use the <code>MountName</code> value when mounting the file system.</p>
         *        <p>
         *        For the <code>SCRATCH_1</code> deployment type, this value is always "<code>fsx</code>". For
         *        <code>SCRATCH_2</code>, <code>PERSISTENT_1</code>, and <code>PERSISTENT_2</code> deployment types,
         *        this value is a string that is unique within an Amazon Web Services Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountName(String mountName);

        /**
         * Sets the value of the DailyAutomaticBackupStartTime property for this object.
         *
         * @param dailyAutomaticBackupStartTime
         *        The new value for the DailyAutomaticBackupStartTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime);

        /**
         * Sets the value of the AutomaticBackupRetentionDays property for this object.
         *
         * @param automaticBackupRetentionDays
         *        The new value for the AutomaticBackupRetentionDays property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays);

        /**
         * <p>
         * A boolean flag indicating whether tags on the file system are copied to backups. If it's set to true, all
         * tags on the file system are copied to all automatic backups and any user-initiated backups where the user
         * doesn't specify any tags. If this value is true, and you specify one or more tags, only the specified tags
         * are copied to backups. If you specify one or more tags when creating a user-initiated backup, no tags are
         * copied from the file system, regardless of this value. (Default = false)
         * </p>
         * 
         * @param copyTagsToBackups
         *        A boolean flag indicating whether tags on the file system are copied to backups. If it's set to true,
         *        all tags on the file system are copied to all automatic backups and any user-initiated backups where
         *        the user doesn't specify any tags. If this value is true, and you specify one or more tags, only the
         *        specified tags are copied to backups. If you specify one or more tags when creating a user-initiated
         *        backup, no tags are copied from the file system, regardless of this value. (Default = false)
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTagsToBackups(Boolean copyTagsToBackups);

        /**
         * <p>
         * The type of drive cache used by <code>PERSISTENT_1</code> file systems that are provisioned with HDD storage
         * devices. This parameter is required when <code>StorageType</code> is HDD. When set to <code>READ</code> the
         * file system has an SSD storage cache that is sized to 20% of the file system's storage capacity. This
         * improves the performance for frequently accessed files by caching up to 20% of the total storage capacity.
         * </p>
         * <p>
         * This parameter is required when <code>StorageType</code> is set to HDD.
         * </p>
         * 
         * @param driveCacheType
         *        The type of drive cache used by <code>PERSISTENT_1</code> file systems that are provisioned with HDD
         *        storage devices. This parameter is required when <code>StorageType</code> is HDD. When set to
         *        <code>READ</code> the file system has an SSD storage cache that is sized to 20% of the file system's
         *        storage capacity. This improves the performance for frequently accessed files by caching up to 20% of
         *        the total storage capacity.</p>
         *        <p>
         *        This parameter is required when <code>StorageType</code> is set to HDD.
         * @see DriveCacheType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DriveCacheType
         */
        Builder driveCacheType(String driveCacheType);

        /**
         * <p>
         * The type of drive cache used by <code>PERSISTENT_1</code> file systems that are provisioned with HDD storage
         * devices. This parameter is required when <code>StorageType</code> is HDD. When set to <code>READ</code> the
         * file system has an SSD storage cache that is sized to 20% of the file system's storage capacity. This
         * improves the performance for frequently accessed files by caching up to 20% of the total storage capacity.
         * </p>
         * <p>
         * This parameter is required when <code>StorageType</code> is set to HDD.
         * </p>
         * 
         * @param driveCacheType
         *        The type of drive cache used by <code>PERSISTENT_1</code> file systems that are provisioned with HDD
         *        storage devices. This parameter is required when <code>StorageType</code> is HDD. When set to
         *        <code>READ</code> the file system has an SSD storage cache that is sized to 20% of the file system's
         *        storage capacity. This improves the performance for frequently accessed files by caching up to 20% of
         *        the total storage capacity.</p>
         *        <p>
         *        This parameter is required when <code>StorageType</code> is set to HDD.
         * @see DriveCacheType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DriveCacheType
         */
        Builder driveCacheType(DriveCacheType driveCacheType);

        /**
         * <p>
         * The data compression configuration for the file system. <code>DataCompressionType</code> can have the
         * following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code> - Data compression is turned off for the file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LZ4</code> - Data compression is turned on with the LZ4 algorithm.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-compression.html">Lustre data compression</a>.
         * </p>
         * 
         * @param dataCompressionType
         *        The data compression configuration for the file system. <code>DataCompressionType</code> can have the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code> - Data compression is turned off for the file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LZ4</code> - Data compression is turned on with the LZ4 algorithm.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-compression.html">Lustre data
         *        compression</a>.
         * @see DataCompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataCompressionType
         */
        Builder dataCompressionType(String dataCompressionType);

        /**
         * <p>
         * The data compression configuration for the file system. <code>DataCompressionType</code> can have the
         * following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code> - Data compression is turned off for the file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LZ4</code> - Data compression is turned on with the LZ4 algorithm.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-compression.html">Lustre data compression</a>.
         * </p>
         * 
         * @param dataCompressionType
         *        The data compression configuration for the file system. <code>DataCompressionType</code> can have the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code> - Data compression is turned off for the file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LZ4</code> - Data compression is turned on with the LZ4 algorithm.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-compression.html">Lustre data
         *        compression</a>.
         * @see DataCompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataCompressionType
         */
        Builder dataCompressionType(DataCompressionType dataCompressionType);

        /**
         * <p>
         * The Lustre logging configuration. Lustre logging writes the enabled log events for your file system to Amazon
         * CloudWatch Logs.
         * </p>
         * 
         * @param logConfiguration
         *        The Lustre logging configuration. Lustre logging writes the enabled log events for your file system to
         *        Amazon CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logConfiguration(LustreLogConfiguration logConfiguration);

        /**
         * <p>
         * The Lustre logging configuration. Lustre logging writes the enabled log events for your file system to Amazon
         * CloudWatch Logs.
         * </p>
         * This is a convenience method that creates an instance of the {@link LustreLogConfiguration.Builder} avoiding
         * the need to create one manually via {@link LustreLogConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link LustreLogConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #logConfiguration(LustreLogConfiguration)}.
         * 
         * @param logConfiguration
         *        a consumer that will call methods on {@link LustreLogConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logConfiguration(LustreLogConfiguration)
         */
        default Builder logConfiguration(Consumer<LustreLogConfiguration.Builder> logConfiguration) {
            return logConfiguration(LustreLogConfiguration.builder().applyMutation(logConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String weeklyMaintenanceStartTime;

        private DataRepositoryConfiguration dataRepositoryConfiguration;

        private String deploymentType;

        private Integer perUnitStorageThroughput;

        private String mountName;

        private String dailyAutomaticBackupStartTime;

        private Integer automaticBackupRetentionDays;

        private Boolean copyTagsToBackups;

        private String driveCacheType;

        private String dataCompressionType;

        private LustreLogConfiguration logConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(LustreFileSystemConfiguration model) {
            weeklyMaintenanceStartTime(model.weeklyMaintenanceStartTime);
            dataRepositoryConfiguration(model.dataRepositoryConfiguration);
            deploymentType(model.deploymentType);
            perUnitStorageThroughput(model.perUnitStorageThroughput);
            mountName(model.mountName);
            dailyAutomaticBackupStartTime(model.dailyAutomaticBackupStartTime);
            automaticBackupRetentionDays(model.automaticBackupRetentionDays);
            copyTagsToBackups(model.copyTagsToBackups);
            driveCacheType(model.driveCacheType);
            dataCompressionType(model.dataCompressionType);
            logConfiguration(model.logConfiguration);
        }

        public final String getWeeklyMaintenanceStartTime() {
            return weeklyMaintenanceStartTime;
        }

        public final void setWeeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
        }

        @Override
        public final Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
            return this;
        }

        public final DataRepositoryConfiguration.Builder getDataRepositoryConfiguration() {
            return dataRepositoryConfiguration != null ? dataRepositoryConfiguration.toBuilder() : null;
        }

        public final void setDataRepositoryConfiguration(DataRepositoryConfiguration.BuilderImpl dataRepositoryConfiguration) {
            this.dataRepositoryConfiguration = dataRepositoryConfiguration != null ? dataRepositoryConfiguration.build() : null;
        }

        @Override
        public final Builder dataRepositoryConfiguration(DataRepositoryConfiguration dataRepositoryConfiguration) {
            this.dataRepositoryConfiguration = dataRepositoryConfiguration;
            return this;
        }

        public final String getDeploymentType() {
            return deploymentType;
        }

        public final void setDeploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
        }

        @Override
        public final Builder deploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
            return this;
        }

        @Override
        public final Builder deploymentType(LustreDeploymentType deploymentType) {
            this.deploymentType(deploymentType == null ? null : deploymentType.toString());
            return this;
        }

        public final Integer getPerUnitStorageThroughput() {
            return perUnitStorageThroughput;
        }

        public final void setPerUnitStorageThroughput(Integer perUnitStorageThroughput) {
            this.perUnitStorageThroughput = perUnitStorageThroughput;
        }

        @Override
        public final Builder perUnitStorageThroughput(Integer perUnitStorageThroughput) {
            this.perUnitStorageThroughput = perUnitStorageThroughput;
            return this;
        }

        public final String getMountName() {
            return mountName;
        }

        public final void setMountName(String mountName) {
            this.mountName = mountName;
        }

        @Override
        public final Builder mountName(String mountName) {
            this.mountName = mountName;
            return this;
        }

        public final String getDailyAutomaticBackupStartTime() {
            return dailyAutomaticBackupStartTime;
        }

        public final void setDailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
        }

        @Override
        public final Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
            return this;
        }

        public final Integer getAutomaticBackupRetentionDays() {
            return automaticBackupRetentionDays;
        }

        public final void setAutomaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
        }

        @Override
        public final Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
            return this;
        }

        public final Boolean getCopyTagsToBackups() {
            return copyTagsToBackups;
        }

        public final void setCopyTagsToBackups(Boolean copyTagsToBackups) {
            this.copyTagsToBackups = copyTagsToBackups;
        }

        @Override
        public final Builder copyTagsToBackups(Boolean copyTagsToBackups) {
            this.copyTagsToBackups = copyTagsToBackups;
            return this;
        }

        public final String getDriveCacheType() {
            return driveCacheType;
        }

        public final void setDriveCacheType(String driveCacheType) {
            this.driveCacheType = driveCacheType;
        }

        @Override
        public final Builder driveCacheType(String driveCacheType) {
            this.driveCacheType = driveCacheType;
            return this;
        }

        @Override
        public final Builder driveCacheType(DriveCacheType driveCacheType) {
            this.driveCacheType(driveCacheType == null ? null : driveCacheType.toString());
            return this;
        }

        public final String getDataCompressionType() {
            return dataCompressionType;
        }

        public final void setDataCompressionType(String dataCompressionType) {
            this.dataCompressionType = dataCompressionType;
        }

        @Override
        public final Builder dataCompressionType(String dataCompressionType) {
            this.dataCompressionType = dataCompressionType;
            return this;
        }

        @Override
        public final Builder dataCompressionType(DataCompressionType dataCompressionType) {
            this.dataCompressionType(dataCompressionType == null ? null : dataCompressionType.toString());
            return this;
        }

        public final LustreLogConfiguration.Builder getLogConfiguration() {
            return logConfiguration != null ? logConfiguration.toBuilder() : null;
        }

        public final void setLogConfiguration(LustreLogConfiguration.BuilderImpl logConfiguration) {
            this.logConfiguration = logConfiguration != null ? logConfiguration.build() : null;
        }

        @Override
        public final Builder logConfiguration(LustreLogConfiguration logConfiguration) {
            this.logConfiguration = logConfiguration;
            return this;
        }

        @Override
        public LustreFileSystemConfiguration build() {
            return new LustreFileSystemConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
