/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The ONTAP configuration properties of the FSx for ONTAP file system that you are creating.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateFileSystemOntapConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CreateFileSystemOntapConfiguration.Builder, CreateFileSystemOntapConfiguration> {
    private static final SdkField<Integer> AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AutomaticBackupRetentionDays")
            .getter(getter(CreateFileSystemOntapConfiguration::automaticBackupRetentionDays))
            .setter(setter(Builder::automaticBackupRetentionDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomaticBackupRetentionDays")
                    .build()).build();

    private static final SdkField<String> DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DailyAutomaticBackupStartTime")
            .getter(getter(CreateFileSystemOntapConfiguration::dailyAutomaticBackupStartTime))
            .setter(setter(Builder::dailyAutomaticBackupStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DailyAutomaticBackupStartTime")
                    .build()).build();

    private static final SdkField<String> DEPLOYMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeploymentType").getter(getter(CreateFileSystemOntapConfiguration::deploymentTypeAsString))
            .setter(setter(Builder::deploymentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeploymentType").build()).build();

    private static final SdkField<String> ENDPOINT_IP_ADDRESS_RANGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointIpAddressRange").getter(getter(CreateFileSystemOntapConfiguration::endpointIpAddressRange))
            .setter(setter(Builder::endpointIpAddressRange))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointIpAddressRange").build())
            .build();

    private static final SdkField<String> FSX_ADMIN_PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FsxAdminPassword").getter(getter(CreateFileSystemOntapConfiguration::fsxAdminPassword))
            .setter(setter(Builder::fsxAdminPassword))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FsxAdminPassword").build()).build();

    private static final SdkField<DiskIopsConfiguration> DISK_IOPS_CONFIGURATION_FIELD = SdkField
            .<DiskIopsConfiguration> builder(MarshallingType.SDK_POJO).memberName("DiskIopsConfiguration")
            .getter(getter(CreateFileSystemOntapConfiguration::diskIopsConfiguration))
            .setter(setter(Builder::diskIopsConfiguration)).constructor(DiskIopsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DiskIopsConfiguration").build())
            .build();

    private static final SdkField<String> PREFERRED_SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredSubnetId").getter(getter(CreateFileSystemOntapConfiguration::preferredSubnetId))
            .setter(setter(Builder::preferredSubnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredSubnetId").build()).build();

    private static final SdkField<List<String>> ROUTE_TABLE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RouteTableIds")
            .getter(getter(CreateFileSystemOntapConfiguration::routeTableIds))
            .setter(setter(Builder::routeTableIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteTableIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> THROUGHPUT_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ThroughputCapacity").getter(getter(CreateFileSystemOntapConfiguration::throughputCapacity))
            .setter(setter(Builder::throughputCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ThroughputCapacity").build())
            .build();

    private static final SdkField<String> WEEKLY_MAINTENANCE_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("WeeklyMaintenanceStartTime")
            .getter(getter(CreateFileSystemOntapConfiguration::weeklyMaintenanceStartTime))
            .setter(setter(Builder::weeklyMaintenanceStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeeklyMaintenanceStartTime").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD, DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD, DEPLOYMENT_TYPE_FIELD,
            ENDPOINT_IP_ADDRESS_RANGE_FIELD, FSX_ADMIN_PASSWORD_FIELD, DISK_IOPS_CONFIGURATION_FIELD, PREFERRED_SUBNET_ID_FIELD,
            ROUTE_TABLE_IDS_FIELD, THROUGHPUT_CAPACITY_FIELD, WEEKLY_MAINTENANCE_START_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer automaticBackupRetentionDays;

    private final String dailyAutomaticBackupStartTime;

    private final String deploymentType;

    private final String endpointIpAddressRange;

    private final String fsxAdminPassword;

    private final DiskIopsConfiguration diskIopsConfiguration;

    private final String preferredSubnetId;

    private final List<String> routeTableIds;

    private final Integer throughputCapacity;

    private final String weeklyMaintenanceStartTime;

    private CreateFileSystemOntapConfiguration(BuilderImpl builder) {
        this.automaticBackupRetentionDays = builder.automaticBackupRetentionDays;
        this.dailyAutomaticBackupStartTime = builder.dailyAutomaticBackupStartTime;
        this.deploymentType = builder.deploymentType;
        this.endpointIpAddressRange = builder.endpointIpAddressRange;
        this.fsxAdminPassword = builder.fsxAdminPassword;
        this.diskIopsConfiguration = builder.diskIopsConfiguration;
        this.preferredSubnetId = builder.preferredSubnetId;
        this.routeTableIds = builder.routeTableIds;
        this.throughputCapacity = builder.throughputCapacity;
        this.weeklyMaintenanceStartTime = builder.weeklyMaintenanceStartTime;
    }

    /**
     * Returns the value of the AutomaticBackupRetentionDays property for this object.
     * 
     * @return The value of the AutomaticBackupRetentionDays property for this object.
     */
    public final Integer automaticBackupRetentionDays() {
        return automaticBackupRetentionDays;
    }

    /**
     * Returns the value of the DailyAutomaticBackupStartTime property for this object.
     * 
     * @return The value of the DailyAutomaticBackupStartTime property for this object.
     */
    public final String dailyAutomaticBackupStartTime() {
        return dailyAutomaticBackupStartTime;
    }

    /**
     * <p>
     * Specifies the FSx for ONTAP file system deployment type to use in creating the file system.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MULTI_AZ_1</code> - (Default) A high availability file system configured for Multi-AZ redundancy to
     * tolerate temporary Availability Zone (AZ) unavailability.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-AZ.html">Choosing a file system
     * deployment type</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link OntapDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return Specifies the FSx for ONTAP file system deployment type to use in creating the file system. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MULTI_AZ_1</code> - (Default) A high availability file system configured for Multi-AZ redundancy to
     *         tolerate temporary Availability Zone (AZ) unavailability.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
     *         href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-AZ.html">Choosing a file system
     *         deployment type</a>.
     * @see OntapDeploymentType
     */
    public final OntapDeploymentType deploymentType() {
        return OntapDeploymentType.fromValue(deploymentType);
    }

    /**
     * <p>
     * Specifies the FSx for ONTAP file system deployment type to use in creating the file system.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>MULTI_AZ_1</code> - (Default) A high availability file system configured for Multi-AZ redundancy to
     * tolerate temporary Availability Zone (AZ) unavailability.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-AZ.html">Choosing a file system
     * deployment type</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link OntapDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return Specifies the FSx for ONTAP file system deployment type to use in creating the file system. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>MULTI_AZ_1</code> - (Default) A high availability file system configured for Multi-AZ redundancy to
     *         tolerate temporary Availability Zone (AZ) unavailability.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
     *         href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-AZ.html">Choosing a file system
     *         deployment type</a>.
     * @see OntapDeploymentType
     */
    public final String deploymentTypeAsString() {
        return deploymentType;
    }

    /**
     * <p>
     * (Multi-AZ only) Specifies the IP address range in which the endpoints to access your file system will be created.
     * By default in the Amazon FSx API, Amazon FSx selects an unused IP address range for you from the 198.19.* range.
     * By default in the Amazon FSx console, Amazon FSx chooses the last 64 IP addresses from the VPC’s primary CIDR
     * range to use as the endpoint IP address range for the file system. You can have overlapping endpoint IP addresses
     * for file systems deployed in the same VPC/route tables.
     * </p>
     * 
     * @return (Multi-AZ only) Specifies the IP address range in which the endpoints to access your file system will be
     *         created. By default in the Amazon FSx API, Amazon FSx selects an unused IP address range for you from the
     *         198.19.* range. By default in the Amazon FSx console, Amazon FSx chooses the last 64 IP addresses from
     *         the VPC’s primary CIDR range to use as the endpoint IP address range for the file system. You can have
     *         overlapping endpoint IP addresses for file systems deployed in the same VPC/route tables.
     */
    public final String endpointIpAddressRange() {
        return endpointIpAddressRange;
    }

    /**
     * <p>
     * The ONTAP administrative password for the <code>fsxadmin</code> user with which you administer your file system
     * using the NetApp ONTAP CLI and REST API.
     * </p>
     * 
     * @return The ONTAP administrative password for the <code>fsxadmin</code> user with which you administer your file
     *         system using the NetApp ONTAP CLI and REST API.
     */
    public final String fsxAdminPassword() {
        return fsxAdminPassword;
    }

    /**
     * <p>
     * The SSD IOPS configuration for the FSx for ONTAP file system.
     * </p>
     * 
     * @return The SSD IOPS configuration for the FSx for ONTAP file system.
     */
    public final DiskIopsConfiguration diskIopsConfiguration() {
        return diskIopsConfiguration;
    }

    /**
     * <p>
     * Required when <code>DeploymentType</code> is set to <code>MULTI_AZ_1</code>. This specifies the subnet in which
     * you want the preferred file server to be located.
     * </p>
     * 
     * @return Required when <code>DeploymentType</code> is set to <code>MULTI_AZ_1</code>. This specifies the subnet in
     *         which you want the preferred file server to be located.
     */
    public final String preferredSubnetId() {
        return preferredSubnetId;
    }

    /**
     * For responses, this returns true if the service returned a value for the RouteTableIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRouteTableIds() {
        return routeTableIds != null && !(routeTableIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Multi-AZ only) Specifies the virtual private cloud (VPC) route tables in which your file system's endpoints will
     * be created. You should specify all VPC route tables associated with the subnets in which your clients are
     * located. By default, Amazon FSx selects your VPC's default route table.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRouteTableIds} method.
     * </p>
     * 
     * @return (Multi-AZ only) Specifies the virtual private cloud (VPC) route tables in which your file system's
     *         endpoints will be created. You should specify all VPC route tables associated with the subnets in which
     *         your clients are located. By default, Amazon FSx selects your VPC's default route table.
     */
    public final List<String> routeTableIds() {
        return routeTableIds;
    }

    /**
     * <p>
     * Sets the throughput capacity for the file system that you're creating. Valid values are 128, 256, 512, 1024,
     * 2048, and 4096 MBps.
     * </p>
     * 
     * @return Sets the throughput capacity for the file system that you're creating. Valid values are 128, 256, 512,
     *         1024, 2048, and 4096 MBps.
     */
    public final Integer throughputCapacity() {
        return throughputCapacity;
    }

    /**
     * Returns the value of the WeeklyMaintenanceStartTime property for this object.
     * 
     * @return The value of the WeeklyMaintenanceStartTime property for this object.
     */
    public final String weeklyMaintenanceStartTime() {
        return weeklyMaintenanceStartTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(automaticBackupRetentionDays());
        hashCode = 31 * hashCode + Objects.hashCode(dailyAutomaticBackupStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(endpointIpAddressRange());
        hashCode = 31 * hashCode + Objects.hashCode(fsxAdminPassword());
        hashCode = 31 * hashCode + Objects.hashCode(diskIopsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(preferredSubnetId());
        hashCode = 31 * hashCode + Objects.hashCode(hasRouteTableIds() ? routeTableIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(throughputCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(weeklyMaintenanceStartTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateFileSystemOntapConfiguration)) {
            return false;
        }
        CreateFileSystemOntapConfiguration other = (CreateFileSystemOntapConfiguration) obj;
        return Objects.equals(automaticBackupRetentionDays(), other.automaticBackupRetentionDays())
                && Objects.equals(dailyAutomaticBackupStartTime(), other.dailyAutomaticBackupStartTime())
                && Objects.equals(deploymentTypeAsString(), other.deploymentTypeAsString())
                && Objects.equals(endpointIpAddressRange(), other.endpointIpAddressRange())
                && Objects.equals(fsxAdminPassword(), other.fsxAdminPassword())
                && Objects.equals(diskIopsConfiguration(), other.diskIopsConfiguration())
                && Objects.equals(preferredSubnetId(), other.preferredSubnetId())
                && hasRouteTableIds() == other.hasRouteTableIds() && Objects.equals(routeTableIds(), other.routeTableIds())
                && Objects.equals(throughputCapacity(), other.throughputCapacity())
                && Objects.equals(weeklyMaintenanceStartTime(), other.weeklyMaintenanceStartTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateFileSystemOntapConfiguration")
                .add("AutomaticBackupRetentionDays", automaticBackupRetentionDays())
                .add("DailyAutomaticBackupStartTime", dailyAutomaticBackupStartTime())
                .add("DeploymentType", deploymentTypeAsString()).add("EndpointIpAddressRange", endpointIpAddressRange())
                .add("FsxAdminPassword", fsxAdminPassword() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DiskIopsConfiguration", diskIopsConfiguration()).add("PreferredSubnetId", preferredSubnetId())
                .add("RouteTableIds", hasRouteTableIds() ? routeTableIds() : null)
                .add("ThroughputCapacity", throughputCapacity()).add("WeeklyMaintenanceStartTime", weeklyMaintenanceStartTime())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AutomaticBackupRetentionDays":
            return Optional.ofNullable(clazz.cast(automaticBackupRetentionDays()));
        case "DailyAutomaticBackupStartTime":
            return Optional.ofNullable(clazz.cast(dailyAutomaticBackupStartTime()));
        case "DeploymentType":
            return Optional.ofNullable(clazz.cast(deploymentTypeAsString()));
        case "EndpointIpAddressRange":
            return Optional.ofNullable(clazz.cast(endpointIpAddressRange()));
        case "FsxAdminPassword":
            return Optional.ofNullable(clazz.cast(fsxAdminPassword()));
        case "DiskIopsConfiguration":
            return Optional.ofNullable(clazz.cast(diskIopsConfiguration()));
        case "PreferredSubnetId":
            return Optional.ofNullable(clazz.cast(preferredSubnetId()));
        case "RouteTableIds":
            return Optional.ofNullable(clazz.cast(routeTableIds()));
        case "ThroughputCapacity":
            return Optional.ofNullable(clazz.cast(throughputCapacity()));
        case "WeeklyMaintenanceStartTime":
            return Optional.ofNullable(clazz.cast(weeklyMaintenanceStartTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateFileSystemOntapConfiguration, T> g) {
        return obj -> g.apply((CreateFileSystemOntapConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CreateFileSystemOntapConfiguration> {
        /**
         * Sets the value of the AutomaticBackupRetentionDays property for this object.
         *
         * @param automaticBackupRetentionDays
         *        The new value for the AutomaticBackupRetentionDays property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays);

        /**
         * Sets the value of the DailyAutomaticBackupStartTime property for this object.
         *
         * @param dailyAutomaticBackupStartTime
         *        The new value for the DailyAutomaticBackupStartTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime);

        /**
         * <p>
         * Specifies the FSx for ONTAP file system deployment type to use in creating the file system.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MULTI_AZ_1</code> - (Default) A high availability file system configured for Multi-AZ redundancy to
         * tolerate temporary Availability Zone (AZ) unavailability.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
         * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-AZ.html">Choosing a file system
         * deployment type</a>.
         * </p>
         * 
         * @param deploymentType
         *        Specifies the FSx for ONTAP file system deployment type to use in creating the file system. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MULTI_AZ_1</code> - (Default) A high availability file system configured for Multi-AZ redundancy
         *        to tolerate temporary Availability Zone (AZ) unavailability.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
         *        href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-AZ.html">Choosing a file
         *        system deployment type</a>.
         * @see OntapDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OntapDeploymentType
         */
        Builder deploymentType(String deploymentType);

        /**
         * <p>
         * Specifies the FSx for ONTAP file system deployment type to use in creating the file system.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>MULTI_AZ_1</code> - (Default) A high availability file system configured for Multi-AZ redundancy to
         * tolerate temporary Availability Zone (AZ) unavailability.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
         * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-AZ.html">Choosing a file system
         * deployment type</a>.
         * </p>
         * 
         * @param deploymentType
         *        Specifies the FSx for ONTAP file system deployment type to use in creating the file system. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>MULTI_AZ_1</code> - (Default) A high availability file system configured for Multi-AZ redundancy
         *        to tolerate temporary Availability Zone (AZ) unavailability.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SINGLE_AZ_1</code> - A file system configured for Single-AZ redundancy.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For information about the use cases for Multi-AZ and Single-AZ deployments, refer to <a
         *        href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-AZ.html">Choosing a file
         *        system deployment type</a>.
         * @see OntapDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OntapDeploymentType
         */
        Builder deploymentType(OntapDeploymentType deploymentType);

        /**
         * <p>
         * (Multi-AZ only) Specifies the IP address range in which the endpoints to access your file system will be
         * created. By default in the Amazon FSx API, Amazon FSx selects an unused IP address range for you from the
         * 198.19.* range. By default in the Amazon FSx console, Amazon FSx chooses the last 64 IP addresses from the
         * VPC’s primary CIDR range to use as the endpoint IP address range for the file system. You can have
         * overlapping endpoint IP addresses for file systems deployed in the same VPC/route tables.
         * </p>
         * 
         * @param endpointIpAddressRange
         *        (Multi-AZ only) Specifies the IP address range in which the endpoints to access your file system will
         *        be created. By default in the Amazon FSx API, Amazon FSx selects an unused IP address range for you
         *        from the 198.19.* range. By default in the Amazon FSx console, Amazon FSx chooses the last 64 IP
         *        addresses from the VPC’s primary CIDR range to use as the endpoint IP address range for the file
         *        system. You can have overlapping endpoint IP addresses for file systems deployed in the same VPC/route
         *        tables.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointIpAddressRange(String endpointIpAddressRange);

        /**
         * <p>
         * The ONTAP administrative password for the <code>fsxadmin</code> user with which you administer your file
         * system using the NetApp ONTAP CLI and REST API.
         * </p>
         * 
         * @param fsxAdminPassword
         *        The ONTAP administrative password for the <code>fsxadmin</code> user with which you administer your
         *        file system using the NetApp ONTAP CLI and REST API.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fsxAdminPassword(String fsxAdminPassword);

        /**
         * <p>
         * The SSD IOPS configuration for the FSx for ONTAP file system.
         * </p>
         * 
         * @param diskIopsConfiguration
         *        The SSD IOPS configuration for the FSx for ONTAP file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder diskIopsConfiguration(DiskIopsConfiguration diskIopsConfiguration);

        /**
         * <p>
         * The SSD IOPS configuration for the FSx for ONTAP file system.
         * </p>
         * This is a convenience method that creates an instance of the {@link DiskIopsConfiguration.Builder} avoiding
         * the need to create one manually via {@link DiskIopsConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DiskIopsConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #diskIopsConfiguration(DiskIopsConfiguration)}.
         * 
         * @param diskIopsConfiguration
         *        a consumer that will call methods on {@link DiskIopsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #diskIopsConfiguration(DiskIopsConfiguration)
         */
        default Builder diskIopsConfiguration(Consumer<DiskIopsConfiguration.Builder> diskIopsConfiguration) {
            return diskIopsConfiguration(DiskIopsConfiguration.builder().applyMutation(diskIopsConfiguration).build());
        }

        /**
         * <p>
         * Required when <code>DeploymentType</code> is set to <code>MULTI_AZ_1</code>. This specifies the subnet in
         * which you want the preferred file server to be located.
         * </p>
         * 
         * @param preferredSubnetId
         *        Required when <code>DeploymentType</code> is set to <code>MULTI_AZ_1</code>. This specifies the subnet
         *        in which you want the preferred file server to be located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredSubnetId(String preferredSubnetId);

        /**
         * <p>
         * (Multi-AZ only) Specifies the virtual private cloud (VPC) route tables in which your file system's endpoints
         * will be created. You should specify all VPC route tables associated with the subnets in which your clients
         * are located. By default, Amazon FSx selects your VPC's default route table.
         * </p>
         * 
         * @param routeTableIds
         *        (Multi-AZ only) Specifies the virtual private cloud (VPC) route tables in which your file system's
         *        endpoints will be created. You should specify all VPC route tables associated with the subnets in
         *        which your clients are located. By default, Amazon FSx selects your VPC's default route table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeTableIds(Collection<String> routeTableIds);

        /**
         * <p>
         * (Multi-AZ only) Specifies the virtual private cloud (VPC) route tables in which your file system's endpoints
         * will be created. You should specify all VPC route tables associated with the subnets in which your clients
         * are located. By default, Amazon FSx selects your VPC's default route table.
         * </p>
         * 
         * @param routeTableIds
         *        (Multi-AZ only) Specifies the virtual private cloud (VPC) route tables in which your file system's
         *        endpoints will be created. You should specify all VPC route tables associated with the subnets in
         *        which your clients are located. By default, Amazon FSx selects your VPC's default route table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeTableIds(String... routeTableIds);

        /**
         * <p>
         * Sets the throughput capacity for the file system that you're creating. Valid values are 128, 256, 512, 1024,
         * 2048, and 4096 MBps.
         * </p>
         * 
         * @param throughputCapacity
         *        Sets the throughput capacity for the file system that you're creating. Valid values are 128, 256, 512,
         *        1024, 2048, and 4096 MBps.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder throughputCapacity(Integer throughputCapacity);

        /**
         * Sets the value of the WeeklyMaintenanceStartTime property for this object.
         *
         * @param weeklyMaintenanceStartTime
         *        The new value for the WeeklyMaintenanceStartTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime);
    }

    static final class BuilderImpl implements Builder {
        private Integer automaticBackupRetentionDays;

        private String dailyAutomaticBackupStartTime;

        private String deploymentType;

        private String endpointIpAddressRange;

        private String fsxAdminPassword;

        private DiskIopsConfiguration diskIopsConfiguration;

        private String preferredSubnetId;

        private List<String> routeTableIds = DefaultSdkAutoConstructList.getInstance();

        private Integer throughputCapacity;

        private String weeklyMaintenanceStartTime;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateFileSystemOntapConfiguration model) {
            automaticBackupRetentionDays(model.automaticBackupRetentionDays);
            dailyAutomaticBackupStartTime(model.dailyAutomaticBackupStartTime);
            deploymentType(model.deploymentType);
            endpointIpAddressRange(model.endpointIpAddressRange);
            fsxAdminPassword(model.fsxAdminPassword);
            diskIopsConfiguration(model.diskIopsConfiguration);
            preferredSubnetId(model.preferredSubnetId);
            routeTableIds(model.routeTableIds);
            throughputCapacity(model.throughputCapacity);
            weeklyMaintenanceStartTime(model.weeklyMaintenanceStartTime);
        }

        public final Integer getAutomaticBackupRetentionDays() {
            return automaticBackupRetentionDays;
        }

        public final void setAutomaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
        }

        @Override
        public final Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
            return this;
        }

        public final String getDailyAutomaticBackupStartTime() {
            return dailyAutomaticBackupStartTime;
        }

        public final void setDailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
        }

        @Override
        public final Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
            return this;
        }

        public final String getDeploymentType() {
            return deploymentType;
        }

        public final void setDeploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
        }

        @Override
        public final Builder deploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
            return this;
        }

        @Override
        public final Builder deploymentType(OntapDeploymentType deploymentType) {
            this.deploymentType(deploymentType == null ? null : deploymentType.toString());
            return this;
        }

        public final String getEndpointIpAddressRange() {
            return endpointIpAddressRange;
        }

        public final void setEndpointIpAddressRange(String endpointIpAddressRange) {
            this.endpointIpAddressRange = endpointIpAddressRange;
        }

        @Override
        public final Builder endpointIpAddressRange(String endpointIpAddressRange) {
            this.endpointIpAddressRange = endpointIpAddressRange;
            return this;
        }

        public final String getFsxAdminPassword() {
            return fsxAdminPassword;
        }

        public final void setFsxAdminPassword(String fsxAdminPassword) {
            this.fsxAdminPassword = fsxAdminPassword;
        }

        @Override
        public final Builder fsxAdminPassword(String fsxAdminPassword) {
            this.fsxAdminPassword = fsxAdminPassword;
            return this;
        }

        public final DiskIopsConfiguration.Builder getDiskIopsConfiguration() {
            return diskIopsConfiguration != null ? diskIopsConfiguration.toBuilder() : null;
        }

        public final void setDiskIopsConfiguration(DiskIopsConfiguration.BuilderImpl diskIopsConfiguration) {
            this.diskIopsConfiguration = diskIopsConfiguration != null ? diskIopsConfiguration.build() : null;
        }

        @Override
        public final Builder diskIopsConfiguration(DiskIopsConfiguration diskIopsConfiguration) {
            this.diskIopsConfiguration = diskIopsConfiguration;
            return this;
        }

        public final String getPreferredSubnetId() {
            return preferredSubnetId;
        }

        public final void setPreferredSubnetId(String preferredSubnetId) {
            this.preferredSubnetId = preferredSubnetId;
        }

        @Override
        public final Builder preferredSubnetId(String preferredSubnetId) {
            this.preferredSubnetId = preferredSubnetId;
            return this;
        }

        public final Collection<String> getRouteTableIds() {
            if (routeTableIds instanceof SdkAutoConstructList) {
                return null;
            }
            return routeTableIds;
        }

        public final void setRouteTableIds(Collection<String> routeTableIds) {
            this.routeTableIds = RouteTableIdsCopier.copy(routeTableIds);
        }

        @Override
        public final Builder routeTableIds(Collection<String> routeTableIds) {
            this.routeTableIds = RouteTableIdsCopier.copy(routeTableIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder routeTableIds(String... routeTableIds) {
            routeTableIds(Arrays.asList(routeTableIds));
            return this;
        }

        public final Integer getThroughputCapacity() {
            return throughputCapacity;
        }

        public final void setThroughputCapacity(Integer throughputCapacity) {
            this.throughputCapacity = throughputCapacity;
        }

        @Override
        public final Builder throughputCapacity(Integer throughputCapacity) {
            this.throughputCapacity = throughputCapacity;
            return this;
        }

        public final String getWeeklyMaintenanceStartTime() {
            return weeklyMaintenanceStartTime;
        }

        public final void setWeeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
        }

        @Override
        public final Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
            return this;
        }

        @Override
        public CreateFileSystemOntapConfiguration build() {
            return new CreateFileSystemOntapConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
