/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration that Amazon FSx uses to join the Windows File Server instance to a self-managed Microsoft Active
 * Directory (AD) directory.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SelfManagedActiveDirectoryConfigurationUpdates implements SdkPojo, Serializable,
        ToCopyableBuilder<SelfManagedActiveDirectoryConfigurationUpdates.Builder, SelfManagedActiveDirectoryConfigurationUpdates> {
    private static final SdkField<String> USER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserName").getter(getter(SelfManagedActiveDirectoryConfigurationUpdates::userName))
            .setter(setter(Builder::userName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserName").build()).build();

    private static final SdkField<String> PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Password").getter(getter(SelfManagedActiveDirectoryConfigurationUpdates::password))
            .setter(setter(Builder::password))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Password").build()).build();

    private static final SdkField<List<String>> DNS_IPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsIps")
            .getter(getter(SelfManagedActiveDirectoryConfigurationUpdates::dnsIps))
            .setter(setter(Builder::dnsIps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_NAME_FIELD,
            PASSWORD_FIELD, DNS_IPS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String userName;

    private final String password;

    private final List<String> dnsIps;

    private SelfManagedActiveDirectoryConfigurationUpdates(BuilderImpl builder) {
        this.userName = builder.userName;
        this.password = builder.password;
        this.dnsIps = builder.dnsIps;
    }

    /**
     * <p>
     * The user name for the service account on your self-managed AD domain that Amazon FSx will use to join to your AD
     * domain. This account must have the permission to join computers to the domain in the organizational unit provided
     * in <code>OrganizationalUnitDistinguishedName</code>.
     * </p>
     * 
     * @return The user name for the service account on your self-managed AD domain that Amazon FSx will use to join to
     *         your AD domain. This account must have the permission to join computers to the domain in the
     *         organizational unit provided in <code>OrganizationalUnitDistinguishedName</code>.
     */
    public final String userName() {
        return userName;
    }

    /**
     * <p>
     * The password for the service account on your self-managed AD domain that Amazon FSx will use to join to your AD
     * domain.
     * </p>
     * 
     * @return The password for the service account on your self-managed AD domain that Amazon FSx will use to join to
     *         your AD domain.
     */
    public final String password() {
        return password;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsIps property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDnsIps() {
        return dnsIps != null && !(dnsIps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD directory.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsIps} method.
     * </p>
     * 
     * @return A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD directory.
     */
    public final List<String> dnsIps() {
        return dnsIps;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userName());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsIps() ? dnsIps() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SelfManagedActiveDirectoryConfigurationUpdates)) {
            return false;
        }
        SelfManagedActiveDirectoryConfigurationUpdates other = (SelfManagedActiveDirectoryConfigurationUpdates) obj;
        return Objects.equals(userName(), other.userName()) && Objects.equals(password(), other.password())
                && hasDnsIps() == other.hasDnsIps() && Objects.equals(dnsIps(), other.dnsIps());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SelfManagedActiveDirectoryConfigurationUpdates").add("UserName", userName())
                .add("Password", password() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DnsIps", hasDnsIps() ? dnsIps() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserName":
            return Optional.ofNullable(clazz.cast(userName()));
        case "Password":
            return Optional.ofNullable(clazz.cast(password()));
        case "DnsIps":
            return Optional.ofNullable(clazz.cast(dnsIps()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SelfManagedActiveDirectoryConfigurationUpdates, T> g) {
        return obj -> g.apply((SelfManagedActiveDirectoryConfigurationUpdates) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SelfManagedActiveDirectoryConfigurationUpdates> {
        /**
         * <p>
         * The user name for the service account on your self-managed AD domain that Amazon FSx will use to join to your
         * AD domain. This account must have the permission to join computers to the domain in the organizational unit
         * provided in <code>OrganizationalUnitDistinguishedName</code>.
         * </p>
         * 
         * @param userName
         *        The user name for the service account on your self-managed AD domain that Amazon FSx will use to join
         *        to your AD domain. This account must have the permission to join computers to the domain in the
         *        organizational unit provided in <code>OrganizationalUnitDistinguishedName</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userName(String userName);

        /**
         * <p>
         * The password for the service account on your self-managed AD domain that Amazon FSx will use to join to your
         * AD domain.
         * </p>
         * 
         * @param password
         *        The password for the service account on your self-managed AD domain that Amazon FSx will use to join
         *        to your AD domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD directory.
         * </p>
         * 
         * @param dnsIps
         *        A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD
         *        directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIps(Collection<String> dnsIps);

        /**
         * <p>
         * A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD directory.
         * </p>
         * 
         * @param dnsIps
         *        A list of up to three IP addresses of DNS servers or domain controllers in the self-managed AD
         *        directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIps(String... dnsIps);
    }

    static final class BuilderImpl implements Builder {
        private String userName;

        private String password;

        private List<String> dnsIps = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SelfManagedActiveDirectoryConfigurationUpdates model) {
            userName(model.userName);
            password(model.password);
            dnsIps(model.dnsIps);
        }

        public final String getUserName() {
            return userName;
        }

        public final void setUserName(String userName) {
            this.userName = userName;
        }

        @Override
        public final Builder userName(String userName) {
            this.userName = userName;
            return this;
        }

        public final String getPassword() {
            return password;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        @Override
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final Collection<String> getDnsIps() {
            if (dnsIps instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsIps;
        }

        public final void setDnsIps(Collection<String> dnsIps) {
            this.dnsIps = DnsIpsCopier.copy(dnsIps);
        }

        @Override
        public final Builder dnsIps(Collection<String> dnsIps) {
            this.dnsIps = DnsIpsCopier.copy(dnsIps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsIps(String... dnsIps) {
            dnsIps(Arrays.asList(dnsIps));
            return this;
        }

        @Override
        public SelfManagedActiveDirectoryConfigurationUpdates build() {
            return new SelfManagedActiveDirectoryConfigurationUpdates(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
