/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The request object used to create a new Amazon FSx file system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateFileSystemRequest extends FSxRequest implements
        ToCopyableBuilder<CreateFileSystemRequest.Builder, CreateFileSystemRequest> {
    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientRequestToken")
            .getter(getter(CreateFileSystemRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<String> FILE_SYSTEM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemType").getter(getter(CreateFileSystemRequest::fileSystemTypeAsString))
            .setter(setter(Builder::fileSystemType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemType").build()).build();

    private static final SdkField<Integer> STORAGE_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("StorageCapacity").getter(getter(CreateFileSystemRequest::storageCapacity))
            .setter(setter(Builder::storageCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageCapacity").build()).build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StorageType").getter(getter(CreateFileSystemRequest::storageTypeAsString))
            .setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SubnetIds")
            .getter(getter(CreateFileSystemRequest::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(CreateFileSystemRequest::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateFileSystemRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(CreateFileSystemRequest::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<CreateFileSystemWindowsConfiguration> WINDOWS_CONFIGURATION_FIELD = SdkField
            .<CreateFileSystemWindowsConfiguration> builder(MarshallingType.SDK_POJO).memberName("WindowsConfiguration")
            .getter(getter(CreateFileSystemRequest::windowsConfiguration)).setter(setter(Builder::windowsConfiguration))
            .constructor(CreateFileSystemWindowsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowsConfiguration").build())
            .build();

    private static final SdkField<CreateFileSystemLustreConfiguration> LUSTRE_CONFIGURATION_FIELD = SdkField
            .<CreateFileSystemLustreConfiguration> builder(MarshallingType.SDK_POJO).memberName("LustreConfiguration")
            .getter(getter(CreateFileSystemRequest::lustreConfiguration)).setter(setter(Builder::lustreConfiguration))
            .constructor(CreateFileSystemLustreConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LustreConfiguration").build())
            .build();

    private static final SdkField<CreateFileSystemOntapConfiguration> ONTAP_CONFIGURATION_FIELD = SdkField
            .<CreateFileSystemOntapConfiguration> builder(MarshallingType.SDK_POJO).memberName("OntapConfiguration")
            .getter(getter(CreateFileSystemRequest::ontapConfiguration)).setter(setter(Builder::ontapConfiguration))
            .constructor(CreateFileSystemOntapConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OntapConfiguration").build())
            .build();

    private static final SdkField<String> FILE_SYSTEM_TYPE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemTypeVersion").getter(getter(CreateFileSystemRequest::fileSystemTypeVersion))
            .setter(setter(Builder::fileSystemTypeVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemTypeVersion").build())
            .build();

    private static final SdkField<CreateFileSystemOpenZFSConfiguration> OPEN_ZFS_CONFIGURATION_FIELD = SdkField
            .<CreateFileSystemOpenZFSConfiguration> builder(MarshallingType.SDK_POJO).memberName("OpenZFSConfiguration")
            .getter(getter(CreateFileSystemRequest::openZFSConfiguration)).setter(setter(Builder::openZFSConfiguration))
            .constructor(CreateFileSystemOpenZFSConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OpenZFSConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_REQUEST_TOKEN_FIELD,
            FILE_SYSTEM_TYPE_FIELD, STORAGE_CAPACITY_FIELD, STORAGE_TYPE_FIELD, SUBNET_IDS_FIELD, SECURITY_GROUP_IDS_FIELD,
            TAGS_FIELD, KMS_KEY_ID_FIELD, WINDOWS_CONFIGURATION_FIELD, LUSTRE_CONFIGURATION_FIELD, ONTAP_CONFIGURATION_FIELD,
            FILE_SYSTEM_TYPE_VERSION_FIELD, OPEN_ZFS_CONFIGURATION_FIELD));

    private final String clientRequestToken;

    private final String fileSystemType;

    private final Integer storageCapacity;

    private final String storageType;

    private final List<String> subnetIds;

    private final List<String> securityGroupIds;

    private final List<Tag> tags;

    private final String kmsKeyId;

    private final CreateFileSystemWindowsConfiguration windowsConfiguration;

    private final CreateFileSystemLustreConfiguration lustreConfiguration;

    private final CreateFileSystemOntapConfiguration ontapConfiguration;

    private final String fileSystemTypeVersion;

    private final CreateFileSystemOpenZFSConfiguration openZFSConfiguration;

    private CreateFileSystemRequest(BuilderImpl builder) {
        super(builder);
        this.clientRequestToken = builder.clientRequestToken;
        this.fileSystemType = builder.fileSystemType;
        this.storageCapacity = builder.storageCapacity;
        this.storageType = builder.storageType;
        this.subnetIds = builder.subnetIds;
        this.securityGroupIds = builder.securityGroupIds;
        this.tags = builder.tags;
        this.kmsKeyId = builder.kmsKeyId;
        this.windowsConfiguration = builder.windowsConfiguration;
        this.lustreConfiguration = builder.lustreConfiguration;
        this.ontapConfiguration = builder.ontapConfiguration;
        this.fileSystemTypeVersion = builder.fileSystemTypeVersion;
        this.openZFSConfiguration = builder.openZFSConfiguration;
    }

    /**
     * <p>
     * A string of up to 63 ASCII characters that Amazon FSx uses to ensure idempotent creation. This string is
     * automatically filled on your behalf when you use the Command Line Interface (CLI) or an Amazon Web Services SDK.
     * </p>
     * 
     * @return A string of up to 63 ASCII characters that Amazon FSx uses to ensure idempotent creation. This string is
     *         automatically filled on your behalf when you use the Command Line Interface (CLI) or an Amazon Web
     *         Services SDK.
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * The type of Amazon FSx file system to create. Valid values are <code>WINDOWS</code>, <code>LUSTRE</code>,
     * <code>ONTAP</code>, and <code>OPENZFS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileSystemType}
     * will return {@link FileSystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fileSystemTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon FSx file system to create. Valid values are <code>WINDOWS</code>, <code>LUSTRE</code>,
     *         <code>ONTAP</code>, and <code>OPENZFS</code>.
     * @see FileSystemType
     */
    public final FileSystemType fileSystemType() {
        return FileSystemType.fromValue(fileSystemType);
    }

    /**
     * <p>
     * The type of Amazon FSx file system to create. Valid values are <code>WINDOWS</code>, <code>LUSTRE</code>,
     * <code>ONTAP</code>, and <code>OPENZFS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileSystemType}
     * will return {@link FileSystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fileSystemTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon FSx file system to create. Valid values are <code>WINDOWS</code>, <code>LUSTRE</code>,
     *         <code>ONTAP</code>, and <code>OPENZFS</code>.
     * @see FileSystemType
     */
    public final String fileSystemTypeAsString() {
        return fileSystemType;
    }

    /**
     * <p>
     * Sets the storage capacity of the file system that you're creating, in gibibytes (GiB).
     * </p>
     * <p>
     * <b>FSx for Lustre file systems</b> - The amount of storage capacity that you can configure depends on the value
     * that you set for <code>StorageType</code> and the Lustre <code>DeploymentType</code>, as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>SCRATCH_2</code>, <code>PERSISTENT_2</code> and <code>PERSISTENT_1</code> deployment types using SSD
     * storage type, the valid values are 1200 GiB, 2400 GiB, and increments of 2400 GiB.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>PERSISTENT_1</code> HDD file systems, valid values are increments of 6000 GiB for 12 MB/s/TiB file
     * systems and increments of 1800 GiB for 40 MB/s/TiB file systems.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>SCRATCH_1</code> deployment type, valid values are 1200 GiB, 2400 GiB, and increments of 3600 GiB.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>FSx for ONTAP file systems</b> - The amount of storage capacity that you can configure depends on the value of
     * the <code>HAPairs</code> property. The minimum value is calculated as 1,024 * <code>HAPairs</code> and the
     * maximum is calculated as 524,288 * <code>HAPairs</code>.
     * </p>
     * <p>
     * <b>FSx for OpenZFS file systems</b> - The amount of storage capacity that you can configure is from 64 GiB up to
     * 524,288 GiB (512 TiB).
     * </p>
     * <p>
     * <b>FSx for Windows File Server file systems</b> - The amount of storage capacity that you can configure depends
     * on the value that you set for <code>StorageType</code> as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For SSD storage, valid values are 32 GiB-65,536 GiB (64 TiB).
     * </p>
     * </li>
     * <li>
     * <p>
     * For HDD storage, valid values are 2000 GiB-65,536 GiB (64 TiB).
     * </p>
     * </li>
     * </ul>
     * 
     * @return Sets the storage capacity of the file system that you're creating, in gibibytes (GiB).</p>
     *         <p>
     *         <b>FSx for Lustre file systems</b> - The amount of storage capacity that you can configure depends on the
     *         value that you set for <code>StorageType</code> and the Lustre <code>DeploymentType</code>, as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For <code>SCRATCH_2</code>, <code>PERSISTENT_2</code> and <code>PERSISTENT_1</code> deployment types
     *         using SSD storage type, the valid values are 1200 GiB, 2400 GiB, and increments of 2400 GiB.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>PERSISTENT_1</code> HDD file systems, valid values are increments of 6000 GiB for 12 MB/s/TiB
     *         file systems and increments of 1800 GiB for 40 MB/s/TiB file systems.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>SCRATCH_1</code> deployment type, valid values are 1200 GiB, 2400 GiB, and increments of 3600
     *         GiB.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>FSx for ONTAP file systems</b> - The amount of storage capacity that you can configure depends on the
     *         value of the <code>HAPairs</code> property. The minimum value is calculated as 1,024 *
     *         <code>HAPairs</code> and the maximum is calculated as 524,288 * <code>HAPairs</code>.
     *         </p>
     *         <p>
     *         <b>FSx for OpenZFS file systems</b> - The amount of storage capacity that you can configure is from 64
     *         GiB up to 524,288 GiB (512 TiB).
     *         </p>
     *         <p>
     *         <b>FSx for Windows File Server file systems</b> - The amount of storage capacity that you can configure
     *         depends on the value that you set for <code>StorageType</code> as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For SSD storage, valid values are 32 GiB-65,536 GiB (64 TiB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For HDD storage, valid values are 2000 GiB-65,536 GiB (64 TiB).
     *         </p>
     *         </li>
     */
    public final Integer storageCapacity() {
        return storageCapacity;
    }

    /**
     * <p>
     * Sets the storage type for the file system that you're creating. Valid values are <code>SSD</code> and
     * <code>HDD</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows, Lustre, ONTAP, and
     * OpenZFS deployment types.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
     * <code>MULTI_AZ_1</code> Windows file system deployment types, and on <code>PERSISTENT_1</code> Lustre file system
     * deployment types.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default value is <code>SSD</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options"> Storage
     * type options</a> in the <i>FSx for Windows File Server User Guide</i> and <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/what-is.html#storage-options">Multiple storage
     * options</a> in the <i>FSx for Lustre User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageType} will
     * return {@link StorageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTypeAsString}.
     * </p>
     * 
     * @return Sets the storage type for the file system that you're creating. Valid values are <code>SSD</code> and
     *         <code>HDD</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows, Lustre, ONTAP,
     *         and OpenZFS deployment types.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
     *         <code>MULTI_AZ_1</code> Windows file system deployment types, and on <code>PERSISTENT_1</code> Lustre
     *         file system deployment types.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default value is <code>SSD</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options">
     *         Storage type options</a> in the <i>FSx for Windows File Server User Guide</i> and <a
     *         href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/what-is.html#storage-options">Multiple storage
     *         options</a> in the <i>FSx for Lustre User Guide</i>.
     * @see StorageType
     */
    public final StorageType storageType() {
        return StorageType.fromValue(storageType);
    }

    /**
     * <p>
     * Sets the storage type for the file system that you're creating. Valid values are <code>SSD</code> and
     * <code>HDD</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows, Lustre, ONTAP, and
     * OpenZFS deployment types.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
     * <code>MULTI_AZ_1</code> Windows file system deployment types, and on <code>PERSISTENT_1</code> Lustre file system
     * deployment types.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default value is <code>SSD</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options"> Storage
     * type options</a> in the <i>FSx for Windows File Server User Guide</i> and <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/what-is.html#storage-options">Multiple storage
     * options</a> in the <i>FSx for Lustre User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageType} will
     * return {@link StorageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTypeAsString}.
     * </p>
     * 
     * @return Sets the storage type for the file system that you're creating. Valid values are <code>SSD</code> and
     *         <code>HDD</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows, Lustre, ONTAP,
     *         and OpenZFS deployment types.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
     *         <code>MULTI_AZ_1</code> Windows file system deployment types, and on <code>PERSISTENT_1</code> Lustre
     *         file system deployment types.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default value is <code>SSD</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options">
     *         Storage type options</a> in the <i>FSx for Windows File Server User Guide</i> and <a
     *         href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/what-is.html#storage-options">Multiple storage
     *         options</a> in the <i>FSx for Lustre User Guide</i>.
     * @see StorageType
     */
    public final String storageTypeAsString() {
        return storageType;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubnetIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the IDs of the subnets that the file system will be accessible from. For Windows and ONTAP
     * <code>MULTI_AZ_1</code> deployment types,provide exactly two subnet IDs, one for the preferred file server and
     * one for the standby file server. You specify one of these subnets as the preferred subnet using the
     * <code>WindowsConfiguration &gt; PreferredSubnetID</code> or
     * <code>OntapConfiguration &gt; PreferredSubnetID</code> properties. For more information about Multi-AZ file
     * system configuration, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html"> Availability and
     * durability: Single-AZ and Multi-AZ file systems</a> in the <i>Amazon FSx for Windows User Guide</i> and <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html"> Availability and
     * durability</a> in the <i>Amazon FSx for ONTAP User Guide</i>.
     * </p>
     * <p>
     * For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> and all Lustre deployment types, provide
     * exactly one subnet ID. The file server is launched in that subnet's Availability Zone.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnetIds} method.
     * </p>
     * 
     * @return Specifies the IDs of the subnets that the file system will be accessible from. For Windows and ONTAP
     *         <code>MULTI_AZ_1</code> deployment types,provide exactly two subnet IDs, one for the preferred file
     *         server and one for the standby file server. You specify one of these subnets as the preferred subnet
     *         using the <code>WindowsConfiguration &gt; PreferredSubnetID</code> or
     *         <code>OntapConfiguration &gt; PreferredSubnetID</code> properties. For more information about Multi-AZ
     *         file system configuration, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html"> Availability
     *         and durability: Single-AZ and Multi-AZ file systems</a> in the <i>Amazon FSx for Windows User Guide</i>
     *         and <a href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html">
     *         Availability and durability</a> in the <i>Amazon FSx for ONTAP User Guide</i>.</p>
     *         <p>
     *         For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> and all Lustre deployment types,
     *         provide exactly one subnet ID. The file server is launched in that subnet's Availability Zone.
     */
    public final List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of IDs specifying the security groups to apply to all network interfaces created for file system access.
     * This list isn't returned in later requests to describe the file system.
     * </p>
     * <important>
     * <p>
     * You must specify a security group if you are creating a Multi-AZ FSx for ONTAP file system in a VPC subnet that
     * has been shared with you.
     * </p>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return A list of IDs specifying the security groups to apply to all network interfaces created for file system
     *         access. This list isn't returned in later requests to describe the file system.</p> <important>
     *         <p>
     *         You must specify a security group if you are creating a Multi-AZ FSx for ONTAP file system in a VPC
     *         subnet that has been shared with you.
     *         </p>
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to apply to the file system that's being created. The key value of the <code>Name</code> tag appears in
     * the console as the file system name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags to apply to the file system that's being created. The key value of the <code>Name</code> tag
     *         appears in the console as the file system name.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * Returns the value of the KmsKeyId property for this object.
     * 
     * @return The value of the KmsKeyId property for this object.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The Microsoft Windows configuration for the file system that's being created.
     * </p>
     * 
     * @return The Microsoft Windows configuration for the file system that's being created.
     */
    public final CreateFileSystemWindowsConfiguration windowsConfiguration() {
        return windowsConfiguration;
    }

    /**
     * Returns the value of the LustreConfiguration property for this object.
     * 
     * @return The value of the LustreConfiguration property for this object.
     */
    public final CreateFileSystemLustreConfiguration lustreConfiguration() {
        return lustreConfiguration;
    }

    /**
     * Returns the value of the OntapConfiguration property for this object.
     * 
     * @return The value of the OntapConfiguration property for this object.
     */
    public final CreateFileSystemOntapConfiguration ontapConfiguration() {
        return ontapConfiguration;
    }

    /**
     * <p>
     * (Optional) For FSx for Lustre file systems, sets the Lustre version for the file system that you're creating.
     * Valid values are <code>2.10</code>, <code>2.12</code>, and <code>2.15</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * 2.10 is supported by the Scratch and Persistent_1 Lustre deployment types.
     * </p>
     * </li>
     * <li>
     * <p>
     * 2.12 and 2.15 are supported by all Lustre deployment types. <code>2.12</code> or <code>2.15</code> is required
     * when setting FSx for Lustre <code>DeploymentType</code> to <code>PERSISTENT_2</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default value = <code>2.10</code>, except when <code>DeploymentType</code> is set to <code>PERSISTENT_2</code>,
     * then the default is <code>2.12</code>.
     * </p>
     * <note>
     * <p>
     * If you set <code>FileSystemTypeVersion</code> to <code>2.10</code> for a <code>PERSISTENT_2</code> Lustre
     * deployment type, the <code>CreateFileSystem</code> operation fails.
     * </p>
     * </note>
     * 
     * @return (Optional) For FSx for Lustre file systems, sets the Lustre version for the file system that you're
     *         creating. Valid values are <code>2.10</code>, <code>2.12</code>, and <code>2.15</code>:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         2.10 is supported by the Scratch and Persistent_1 Lustre deployment types.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         2.12 and 2.15 are supported by all Lustre deployment types. <code>2.12</code> or <code>2.15</code> is
     *         required when setting FSx for Lustre <code>DeploymentType</code> to <code>PERSISTENT_2</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default value = <code>2.10</code>, except when <code>DeploymentType</code> is set to
     *         <code>PERSISTENT_2</code>, then the default is <code>2.12</code>.
     *         </p>
     *         <note>
     *         <p>
     *         If you set <code>FileSystemTypeVersion</code> to <code>2.10</code> for a <code>PERSISTENT_2</code> Lustre
     *         deployment type, the <code>CreateFileSystem</code> operation fails.
     *         </p>
     */
    public final String fileSystemTypeVersion() {
        return fileSystemTypeVersion;
    }

    /**
     * <p>
     * The OpenZFS configuration for the file system that's being created.
     * </p>
     * 
     * @return The OpenZFS configuration for the file system that's being created.
     */
    public final CreateFileSystemOpenZFSConfiguration openZFSConfiguration() {
        return openZFSConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(storageCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(storageTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetIds() ? subnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(windowsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(lustreConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(ontapConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemTypeVersion());
        hashCode = 31 * hashCode + Objects.hashCode(openZFSConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateFileSystemRequest)) {
            return false;
        }
        CreateFileSystemRequest other = (CreateFileSystemRequest) obj;
        return Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(fileSystemTypeAsString(), other.fileSystemTypeAsString())
                && Objects.equals(storageCapacity(), other.storageCapacity())
                && Objects.equals(storageTypeAsString(), other.storageTypeAsString()) && hasSubnetIds() == other.hasSubnetIds()
                && Objects.equals(subnetIds(), other.subnetIds()) && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(windowsConfiguration(), other.windowsConfiguration())
                && Objects.equals(lustreConfiguration(), other.lustreConfiguration())
                && Objects.equals(ontapConfiguration(), other.ontapConfiguration())
                && Objects.equals(fileSystemTypeVersion(), other.fileSystemTypeVersion())
                && Objects.equals(openZFSConfiguration(), other.openZFSConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateFileSystemRequest").add("ClientRequestToken", clientRequestToken())
                .add("FileSystemType", fileSystemTypeAsString()).add("StorageCapacity", storageCapacity())
                .add("StorageType", storageTypeAsString()).add("SubnetIds", hasSubnetIds() ? subnetIds() : null)
                .add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null)
                .add("Tags", hasTags() ? tags() : null).add("KmsKeyId", kmsKeyId())
                .add("WindowsConfiguration", windowsConfiguration()).add("LustreConfiguration", lustreConfiguration())
                .add("OntapConfiguration", ontapConfiguration()).add("FileSystemTypeVersion", fileSystemTypeVersion())
                .add("OpenZFSConfiguration", openZFSConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "FileSystemType":
            return Optional.ofNullable(clazz.cast(fileSystemTypeAsString()));
        case "StorageCapacity":
            return Optional.ofNullable(clazz.cast(storageCapacity()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageTypeAsString()));
        case "SubnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "WindowsConfiguration":
            return Optional.ofNullable(clazz.cast(windowsConfiguration()));
        case "LustreConfiguration":
            return Optional.ofNullable(clazz.cast(lustreConfiguration()));
        case "OntapConfiguration":
            return Optional.ofNullable(clazz.cast(ontapConfiguration()));
        case "FileSystemTypeVersion":
            return Optional.ofNullable(clazz.cast(fileSystemTypeVersion()));
        case "OpenZFSConfiguration":
            return Optional.ofNullable(clazz.cast(openZFSConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateFileSystemRequest, T> g) {
        return obj -> g.apply((CreateFileSystemRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends FSxRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateFileSystemRequest> {
        /**
         * <p>
         * A string of up to 63 ASCII characters that Amazon FSx uses to ensure idempotent creation. This string is
         * automatically filled on your behalf when you use the Command Line Interface (CLI) or an Amazon Web Services
         * SDK.
         * </p>
         * 
         * @param clientRequestToken
         *        A string of up to 63 ASCII characters that Amazon FSx uses to ensure idempotent creation. This string
         *        is automatically filled on your behalf when you use the Command Line Interface (CLI) or an Amazon Web
         *        Services SDK.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * The type of Amazon FSx file system to create. Valid values are <code>WINDOWS</code>, <code>LUSTRE</code>,
         * <code>ONTAP</code>, and <code>OPENZFS</code>.
         * </p>
         * 
         * @param fileSystemType
         *        The type of Amazon FSx file system to create. Valid values are <code>WINDOWS</code>,
         *        <code>LUSTRE</code>, <code>ONTAP</code>, and <code>OPENZFS</code>.
         * @see FileSystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemType
         */
        Builder fileSystemType(String fileSystemType);

        /**
         * <p>
         * The type of Amazon FSx file system to create. Valid values are <code>WINDOWS</code>, <code>LUSTRE</code>,
         * <code>ONTAP</code>, and <code>OPENZFS</code>.
         * </p>
         * 
         * @param fileSystemType
         *        The type of Amazon FSx file system to create. Valid values are <code>WINDOWS</code>,
         *        <code>LUSTRE</code>, <code>ONTAP</code>, and <code>OPENZFS</code>.
         * @see FileSystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemType
         */
        Builder fileSystemType(FileSystemType fileSystemType);

        /**
         * <p>
         * Sets the storage capacity of the file system that you're creating, in gibibytes (GiB).
         * </p>
         * <p>
         * <b>FSx for Lustre file systems</b> - The amount of storage capacity that you can configure depends on the
         * value that you set for <code>StorageType</code> and the Lustre <code>DeploymentType</code>, as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For <code>SCRATCH_2</code>, <code>PERSISTENT_2</code> and <code>PERSISTENT_1</code> deployment types using
         * SSD storage type, the valid values are 1200 GiB, 2400 GiB, and increments of 2400 GiB.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>PERSISTENT_1</code> HDD file systems, valid values are increments of 6000 GiB for 12 MB/s/TiB file
         * systems and increments of 1800 GiB for 40 MB/s/TiB file systems.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>SCRATCH_1</code> deployment type, valid values are 1200 GiB, 2400 GiB, and increments of 3600 GiB.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>FSx for ONTAP file systems</b> - The amount of storage capacity that you can configure depends on the
         * value of the <code>HAPairs</code> property. The minimum value is calculated as 1,024 * <code>HAPairs</code>
         * and the maximum is calculated as 524,288 * <code>HAPairs</code>.
         * </p>
         * <p>
         * <b>FSx for OpenZFS file systems</b> - The amount of storage capacity that you can configure is from 64 GiB up
         * to 524,288 GiB (512 TiB).
         * </p>
         * <p>
         * <b>FSx for Windows File Server file systems</b> - The amount of storage capacity that you can configure
         * depends on the value that you set for <code>StorageType</code> as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For SSD storage, valid values are 32 GiB-65,536 GiB (64 TiB).
         * </p>
         * </li>
         * <li>
         * <p>
         * For HDD storage, valid values are 2000 GiB-65,536 GiB (64 TiB).
         * </p>
         * </li>
         * </ul>
         * 
         * @param storageCapacity
         *        Sets the storage capacity of the file system that you're creating, in gibibytes (GiB).</p>
         *        <p>
         *        <b>FSx for Lustre file systems</b> - The amount of storage capacity that you can configure depends on
         *        the value that you set for <code>StorageType</code> and the Lustre <code>DeploymentType</code>, as
         *        follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For <code>SCRATCH_2</code>, <code>PERSISTENT_2</code> and <code>PERSISTENT_1</code> deployment types
         *        using SSD storage type, the valid values are 1200 GiB, 2400 GiB, and increments of 2400 GiB.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>PERSISTENT_1</code> HDD file systems, valid values are increments of 6000 GiB for 12
         *        MB/s/TiB file systems and increments of 1800 GiB for 40 MB/s/TiB file systems.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>SCRATCH_1</code> deployment type, valid values are 1200 GiB, 2400 GiB, and increments of
         *        3600 GiB.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>FSx for ONTAP file systems</b> - The amount of storage capacity that you can configure depends on
         *        the value of the <code>HAPairs</code> property. The minimum value is calculated as 1,024 *
         *        <code>HAPairs</code> and the maximum is calculated as 524,288 * <code>HAPairs</code>.
         *        </p>
         *        <p>
         *        <b>FSx for OpenZFS file systems</b> - The amount of storage capacity that you can configure is from 64
         *        GiB up to 524,288 GiB (512 TiB).
         *        </p>
         *        <p>
         *        <b>FSx for Windows File Server file systems</b> - The amount of storage capacity that you can
         *        configure depends on the value that you set for <code>StorageType</code> as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For SSD storage, valid values are 32 GiB-65,536 GiB (64 TiB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For HDD storage, valid values are 2000 GiB-65,536 GiB (64 TiB).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageCapacity(Integer storageCapacity);

        /**
         * <p>
         * Sets the storage type for the file system that you're creating. Valid values are <code>SSD</code> and
         * <code>HDD</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows, Lustre, ONTAP, and
         * OpenZFS deployment types.
         * </p>
         * </li>
         * <li>
         * <p>
         * Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
         * <code>MULTI_AZ_1</code> Windows file system deployment types, and on <code>PERSISTENT_1</code> Lustre file
         * system deployment types.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default value is <code>SSD</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options">
         * Storage type options</a> in the <i>FSx for Windows File Server User Guide</i> and <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/what-is.html#storage-options">Multiple storage
         * options</a> in the <i>FSx for Lustre User Guide</i>.
         * </p>
         * 
         * @param storageType
         *        Sets the storage type for the file system that you're creating. Valid values are <code>SSD</code> and
         *        <code>HDD</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows, Lustre,
         *        ONTAP, and OpenZFS deployment types.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code>
         *        and <code>MULTI_AZ_1</code> Windows file system deployment types, and on <code>PERSISTENT_1</code>
         *        Lustre file system deployment types.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default value is <code>SSD</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options"
         *        > Storage type options</a> in the <i>FSx for Windows File Server User Guide</i> and <a
         *        href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/what-is.html#storage-options">Multiple
         *        storage options</a> in the <i>FSx for Lustre User Guide</i>.
         * @see StorageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageType
         */
        Builder storageType(String storageType);

        /**
         * <p>
         * Sets the storage type for the file system that you're creating. Valid values are <code>SSD</code> and
         * <code>HDD</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows, Lustre, ONTAP, and
         * OpenZFS deployment types.
         * </p>
         * </li>
         * <li>
         * <p>
         * Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
         * <code>MULTI_AZ_1</code> Windows file system deployment types, and on <code>PERSISTENT_1</code> Lustre file
         * system deployment types.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default value is <code>SSD</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options">
         * Storage type options</a> in the <i>FSx for Windows File Server User Guide</i> and <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/what-is.html#storage-options">Multiple storage
         * options</a> in the <i>FSx for Lustre User Guide</i>.
         * </p>
         * 
         * @param storageType
         *        Sets the storage type for the file system that you're creating. Valid values are <code>SSD</code> and
         *        <code>HDD</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows, Lustre,
         *        ONTAP, and OpenZFS deployment types.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code>
         *        and <code>MULTI_AZ_1</code> Windows file system deployment types, and on <code>PERSISTENT_1</code>
         *        Lustre file system deployment types.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default value is <code>SSD</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options"
         *        > Storage type options</a> in the <i>FSx for Windows File Server User Guide</i> and <a
         *        href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/what-is.html#storage-options">Multiple
         *        storage options</a> in the <i>FSx for Lustre User Guide</i>.
         * @see StorageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageType
         */
        Builder storageType(StorageType storageType);

        /**
         * <p>
         * Specifies the IDs of the subnets that the file system will be accessible from. For Windows and ONTAP
         * <code>MULTI_AZ_1</code> deployment types,provide exactly two subnet IDs, one for the preferred file server
         * and one for the standby file server. You specify one of these subnets as the preferred subnet using the
         * <code>WindowsConfiguration &gt; PreferredSubnetID</code> or
         * <code>OntapConfiguration &gt; PreferredSubnetID</code> properties. For more information about Multi-AZ file
         * system configuration, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html"> Availability and
         * durability: Single-AZ and Multi-AZ file systems</a> in the <i>Amazon FSx for Windows User Guide</i> and <a
         * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html"> Availability and
         * durability</a> in the <i>Amazon FSx for ONTAP User Guide</i>.
         * </p>
         * <p>
         * For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> and all Lustre deployment types, provide
         * exactly one subnet ID. The file server is launched in that subnet's Availability Zone.
         * </p>
         * 
         * @param subnetIds
         *        Specifies the IDs of the subnets that the file system will be accessible from. For Windows and ONTAP
         *        <code>MULTI_AZ_1</code> deployment types,provide exactly two subnet IDs, one for the preferred file
         *        server and one for the standby file server. You specify one of these subnets as the preferred subnet
         *        using the <code>WindowsConfiguration &gt; PreferredSubnetID</code> or
         *        <code>OntapConfiguration &gt; PreferredSubnetID</code> properties. For more information about Multi-AZ
         *        file system configuration, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">
         *        Availability and durability: Single-AZ and Multi-AZ file systems</a> in the <i>Amazon FSx for Windows
         *        User Guide</i> and <a
         *        href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html"> Availability
         *        and durability</a> in the <i>Amazon FSx for ONTAP User Guide</i>.</p>
         *        <p>
         *        For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> and all Lustre deployment types,
         *        provide exactly one subnet ID. The file server is launched in that subnet's Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * Specifies the IDs of the subnets that the file system will be accessible from. For Windows and ONTAP
         * <code>MULTI_AZ_1</code> deployment types,provide exactly two subnet IDs, one for the preferred file server
         * and one for the standby file server. You specify one of these subnets as the preferred subnet using the
         * <code>WindowsConfiguration &gt; PreferredSubnetID</code> or
         * <code>OntapConfiguration &gt; PreferredSubnetID</code> properties. For more information about Multi-AZ file
         * system configuration, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html"> Availability and
         * durability: Single-AZ and Multi-AZ file systems</a> in the <i>Amazon FSx for Windows User Guide</i> and <a
         * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html"> Availability and
         * durability</a> in the <i>Amazon FSx for ONTAP User Guide</i>.
         * </p>
         * <p>
         * For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> and all Lustre deployment types, provide
         * exactly one subnet ID. The file server is launched in that subnet's Availability Zone.
         * </p>
         * 
         * @param subnetIds
         *        Specifies the IDs of the subnets that the file system will be accessible from. For Windows and ONTAP
         *        <code>MULTI_AZ_1</code> deployment types,provide exactly two subnet IDs, one for the preferred file
         *        server and one for the standby file server. You specify one of these subnets as the preferred subnet
         *        using the <code>WindowsConfiguration &gt; PreferredSubnetID</code> or
         *        <code>OntapConfiguration &gt; PreferredSubnetID</code> properties. For more information about Multi-AZ
         *        file system configuration, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/high-availability-multiAZ.html">
         *        Availability and durability: Single-AZ and Multi-AZ file systems</a> in the <i>Amazon FSx for Windows
         *        User Guide</i> and <a
         *        href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/high-availability-multiAZ.html"> Availability
         *        and durability</a> in the <i>Amazon FSx for ONTAP User Guide</i>.</p>
         *        <p>
         *        For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> and all Lustre deployment types,
         *        provide exactly one subnet ID. The file server is launched in that subnet's Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * A list of IDs specifying the security groups to apply to all network interfaces created for file system
         * access. This list isn't returned in later requests to describe the file system.
         * </p>
         * <important>
         * <p>
         * You must specify a security group if you are creating a Multi-AZ FSx for ONTAP file system in a VPC subnet
         * that has been shared with you.
         * </p>
         * </important>
         * 
         * @param securityGroupIds
         *        A list of IDs specifying the security groups to apply to all network interfaces created for file
         *        system access. This list isn't returned in later requests to describe the file system.</p> <important>
         *        <p>
         *        You must specify a security group if you are creating a Multi-AZ FSx for ONTAP file system in a VPC
         *        subnet that has been shared with you.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * A list of IDs specifying the security groups to apply to all network interfaces created for file system
         * access. This list isn't returned in later requests to describe the file system.
         * </p>
         * <important>
         * <p>
         * You must specify a security group if you are creating a Multi-AZ FSx for ONTAP file system in a VPC subnet
         * that has been shared with you.
         * </p>
         * </important>
         * 
         * @param securityGroupIds
         *        A list of IDs specifying the security groups to apply to all network interfaces created for file
         *        system access. This list isn't returned in later requests to describe the file system.</p> <important>
         *        <p>
         *        You must specify a security group if you are creating a Multi-AZ FSx for ONTAP file system in a VPC
         *        subnet that has been shared with you.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * The tags to apply to the file system that's being created. The key value of the <code>Name</code> tag appears
         * in the console as the file system name.
         * </p>
         * 
         * @param tags
         *        The tags to apply to the file system that's being created. The key value of the <code>Name</code> tag
         *        appears in the console as the file system name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags to apply to the file system that's being created. The key value of the <code>Name</code> tag appears
         * in the console as the file system name.
         * </p>
         * 
         * @param tags
         *        The tags to apply to the file system that's being created. The key value of the <code>Name</code> tag
         *        appears in the console as the file system name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags to apply to the file system that's being created. The key value of the <code>Name</code> tag appears
         * in the console as the file system name.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fsx.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.fsx.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.fsx.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.fsx.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * Sets the value of the KmsKeyId property for this object.
         *
         * @param kmsKeyId
         *        The new value for the KmsKeyId property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The Microsoft Windows configuration for the file system that's being created.
         * </p>
         * 
         * @param windowsConfiguration
         *        The Microsoft Windows configuration for the file system that's being created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowsConfiguration(CreateFileSystemWindowsConfiguration windowsConfiguration);

        /**
         * <p>
         * The Microsoft Windows configuration for the file system that's being created.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link CreateFileSystemWindowsConfiguration.Builder} avoiding the need to create one manually via
         * {@link CreateFileSystemWindowsConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CreateFileSystemWindowsConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #windowsConfiguration(CreateFileSystemWindowsConfiguration)}.
         * 
         * @param windowsConfiguration
         *        a consumer that will call methods on {@link CreateFileSystemWindowsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #windowsConfiguration(CreateFileSystemWindowsConfiguration)
         */
        default Builder windowsConfiguration(Consumer<CreateFileSystemWindowsConfiguration.Builder> windowsConfiguration) {
            return windowsConfiguration(CreateFileSystemWindowsConfiguration.builder().applyMutation(windowsConfiguration)
                    .build());
        }

        /**
         * Sets the value of the LustreConfiguration property for this object.
         *
         * @param lustreConfiguration
         *        The new value for the LustreConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lustreConfiguration(CreateFileSystemLustreConfiguration lustreConfiguration);

        /**
         * Sets the value of the LustreConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the
         * {@link CreateFileSystemLustreConfiguration.Builder} avoiding the need to create one manually via
         * {@link CreateFileSystemLustreConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CreateFileSystemLustreConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #lustreConfiguration(CreateFileSystemLustreConfiguration)}.
         * 
         * @param lustreConfiguration
         *        a consumer that will call methods on {@link CreateFileSystemLustreConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lustreConfiguration(CreateFileSystemLustreConfiguration)
         */
        default Builder lustreConfiguration(Consumer<CreateFileSystemLustreConfiguration.Builder> lustreConfiguration) {
            return lustreConfiguration(CreateFileSystemLustreConfiguration.builder().applyMutation(lustreConfiguration).build());
        }

        /**
         * Sets the value of the OntapConfiguration property for this object.
         *
         * @param ontapConfiguration
         *        The new value for the OntapConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ontapConfiguration(CreateFileSystemOntapConfiguration ontapConfiguration);

        /**
         * Sets the value of the OntapConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the
         * {@link CreateFileSystemOntapConfiguration.Builder} avoiding the need to create one manually via
         * {@link CreateFileSystemOntapConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CreateFileSystemOntapConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #ontapConfiguration(CreateFileSystemOntapConfiguration)}.
         * 
         * @param ontapConfiguration
         *        a consumer that will call methods on {@link CreateFileSystemOntapConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ontapConfiguration(CreateFileSystemOntapConfiguration)
         */
        default Builder ontapConfiguration(Consumer<CreateFileSystemOntapConfiguration.Builder> ontapConfiguration) {
            return ontapConfiguration(CreateFileSystemOntapConfiguration.builder().applyMutation(ontapConfiguration).build());
        }

        /**
         * <p>
         * (Optional) For FSx for Lustre file systems, sets the Lustre version for the file system that you're creating.
         * Valid values are <code>2.10</code>, <code>2.12</code>, and <code>2.15</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * 2.10 is supported by the Scratch and Persistent_1 Lustre deployment types.
         * </p>
         * </li>
         * <li>
         * <p>
         * 2.12 and 2.15 are supported by all Lustre deployment types. <code>2.12</code> or <code>2.15</code> is
         * required when setting FSx for Lustre <code>DeploymentType</code> to <code>PERSISTENT_2</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default value = <code>2.10</code>, except when <code>DeploymentType</code> is set to
         * <code>PERSISTENT_2</code>, then the default is <code>2.12</code>.
         * </p>
         * <note>
         * <p>
         * If you set <code>FileSystemTypeVersion</code> to <code>2.10</code> for a <code>PERSISTENT_2</code> Lustre
         * deployment type, the <code>CreateFileSystem</code> operation fails.
         * </p>
         * </note>
         * 
         * @param fileSystemTypeVersion
         *        (Optional) For FSx for Lustre file systems, sets the Lustre version for the file system that you're
         *        creating. Valid values are <code>2.10</code>, <code>2.12</code>, and <code>2.15</code>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        2.10 is supported by the Scratch and Persistent_1 Lustre deployment types.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        2.12 and 2.15 are supported by all Lustre deployment types. <code>2.12</code> or <code>2.15</code> is
         *        required when setting FSx for Lustre <code>DeploymentType</code> to <code>PERSISTENT_2</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default value = <code>2.10</code>, except when <code>DeploymentType</code> is set to
         *        <code>PERSISTENT_2</code>, then the default is <code>2.12</code>.
         *        </p>
         *        <note>
         *        <p>
         *        If you set <code>FileSystemTypeVersion</code> to <code>2.10</code> for a <code>PERSISTENT_2</code>
         *        Lustre deployment type, the <code>CreateFileSystem</code> operation fails.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemTypeVersion(String fileSystemTypeVersion);

        /**
         * <p>
         * The OpenZFS configuration for the file system that's being created.
         * </p>
         * 
         * @param openZFSConfiguration
         *        The OpenZFS configuration for the file system that's being created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder openZFSConfiguration(CreateFileSystemOpenZFSConfiguration openZFSConfiguration);

        /**
         * <p>
         * The OpenZFS configuration for the file system that's being created.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link CreateFileSystemOpenZFSConfiguration.Builder} avoiding the need to create one manually via
         * {@link CreateFileSystemOpenZFSConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CreateFileSystemOpenZFSConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #openZFSConfiguration(CreateFileSystemOpenZFSConfiguration)}.
         * 
         * @param openZFSConfiguration
         *        a consumer that will call methods on {@link CreateFileSystemOpenZFSConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #openZFSConfiguration(CreateFileSystemOpenZFSConfiguration)
         */
        default Builder openZFSConfiguration(Consumer<CreateFileSystemOpenZFSConfiguration.Builder> openZFSConfiguration) {
            return openZFSConfiguration(CreateFileSystemOpenZFSConfiguration.builder().applyMutation(openZFSConfiguration)
                    .build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends FSxRequest.BuilderImpl implements Builder {
        private String clientRequestToken;

        private String fileSystemType;

        private Integer storageCapacity;

        private String storageType;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String kmsKeyId;

        private CreateFileSystemWindowsConfiguration windowsConfiguration;

        private CreateFileSystemLustreConfiguration lustreConfiguration;

        private CreateFileSystemOntapConfiguration ontapConfiguration;

        private String fileSystemTypeVersion;

        private CreateFileSystemOpenZFSConfiguration openZFSConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateFileSystemRequest model) {
            super(model);
            clientRequestToken(model.clientRequestToken);
            fileSystemType(model.fileSystemType);
            storageCapacity(model.storageCapacity);
            storageType(model.storageType);
            subnetIds(model.subnetIds);
            securityGroupIds(model.securityGroupIds);
            tags(model.tags);
            kmsKeyId(model.kmsKeyId);
            windowsConfiguration(model.windowsConfiguration);
            lustreConfiguration(model.lustreConfiguration);
            ontapConfiguration(model.ontapConfiguration);
            fileSystemTypeVersion(model.fileSystemTypeVersion);
            openZFSConfiguration(model.openZFSConfiguration);
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final String getFileSystemType() {
            return fileSystemType;
        }

        public final void setFileSystemType(String fileSystemType) {
            this.fileSystemType = fileSystemType;
        }

        @Override
        public final Builder fileSystemType(String fileSystemType) {
            this.fileSystemType = fileSystemType;
            return this;
        }

        @Override
        public final Builder fileSystemType(FileSystemType fileSystemType) {
            this.fileSystemType(fileSystemType == null ? null : fileSystemType.toString());
            return this;
        }

        public final Integer getStorageCapacity() {
            return storageCapacity;
        }

        public final void setStorageCapacity(Integer storageCapacity) {
            this.storageCapacity = storageCapacity;
        }

        @Override
        public final Builder storageCapacity(Integer storageCapacity) {
            this.storageCapacity = storageCapacity;
            return this;
        }

        public final String getStorageType() {
            return storageType;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        @Override
        public final Builder storageType(StorageType storageType) {
            this.storageType(storageType == null ? null : storageType.toString());
            return this;
        }

        public final Collection<String> getSubnetIds() {
            if (subnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetIds;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final CreateFileSystemWindowsConfiguration.Builder getWindowsConfiguration() {
            return windowsConfiguration != null ? windowsConfiguration.toBuilder() : null;
        }

        public final void setWindowsConfiguration(CreateFileSystemWindowsConfiguration.BuilderImpl windowsConfiguration) {
            this.windowsConfiguration = windowsConfiguration != null ? windowsConfiguration.build() : null;
        }

        @Override
        public final Builder windowsConfiguration(CreateFileSystemWindowsConfiguration windowsConfiguration) {
            this.windowsConfiguration = windowsConfiguration;
            return this;
        }

        public final CreateFileSystemLustreConfiguration.Builder getLustreConfiguration() {
            return lustreConfiguration != null ? lustreConfiguration.toBuilder() : null;
        }

        public final void setLustreConfiguration(CreateFileSystemLustreConfiguration.BuilderImpl lustreConfiguration) {
            this.lustreConfiguration = lustreConfiguration != null ? lustreConfiguration.build() : null;
        }

        @Override
        public final Builder lustreConfiguration(CreateFileSystemLustreConfiguration lustreConfiguration) {
            this.lustreConfiguration = lustreConfiguration;
            return this;
        }

        public final CreateFileSystemOntapConfiguration.Builder getOntapConfiguration() {
            return ontapConfiguration != null ? ontapConfiguration.toBuilder() : null;
        }

        public final void setOntapConfiguration(CreateFileSystemOntapConfiguration.BuilderImpl ontapConfiguration) {
            this.ontapConfiguration = ontapConfiguration != null ? ontapConfiguration.build() : null;
        }

        @Override
        public final Builder ontapConfiguration(CreateFileSystemOntapConfiguration ontapConfiguration) {
            this.ontapConfiguration = ontapConfiguration;
            return this;
        }

        public final String getFileSystemTypeVersion() {
            return fileSystemTypeVersion;
        }

        public final void setFileSystemTypeVersion(String fileSystemTypeVersion) {
            this.fileSystemTypeVersion = fileSystemTypeVersion;
        }

        @Override
        public final Builder fileSystemTypeVersion(String fileSystemTypeVersion) {
            this.fileSystemTypeVersion = fileSystemTypeVersion;
            return this;
        }

        public final CreateFileSystemOpenZFSConfiguration.Builder getOpenZFSConfiguration() {
            return openZFSConfiguration != null ? openZFSConfiguration.toBuilder() : null;
        }

        public final void setOpenZFSConfiguration(CreateFileSystemOpenZFSConfiguration.BuilderImpl openZFSConfiguration) {
            this.openZFSConfiguration = openZFSConfiguration != null ? openZFSConfiguration.build() : null;
        }

        @Override
        public final Builder openZFSConfiguration(CreateFileSystemOpenZFSConfiguration openZFSConfiguration) {
            this.openZFSConfiguration = openZFSConfiguration;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateFileSystemRequest build() {
            return new CreateFileSystemRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
