/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for a data repository association (DRA) to be created during the Amazon File Cache resource
 * creation. The DRA links the cache to either an Amazon S3 bucket or prefix, or a Network File System (NFS) data
 * repository that supports the NFSv3 protocol.
 * </p>
 * <p>
 * The DRA does not support automatic import or automatic export.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FileCacheDataRepositoryAssociation implements SdkPojo, Serializable,
        ToCopyableBuilder<FileCacheDataRepositoryAssociation.Builder, FileCacheDataRepositoryAssociation> {
    private static final SdkField<String> FILE_CACHE_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileCachePath").getter(getter(FileCacheDataRepositoryAssociation::fileCachePath))
            .setter(setter(Builder::fileCachePath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileCachePath").build()).build();

    private static final SdkField<String> DATA_REPOSITORY_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataRepositoryPath").getter(getter(FileCacheDataRepositoryAssociation::dataRepositoryPath))
            .setter(setter(Builder::dataRepositoryPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataRepositoryPath").build())
            .build();

    private static final SdkField<List<String>> DATA_REPOSITORY_SUBDIRECTORIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DataRepositorySubdirectories")
            .getter(getter(FileCacheDataRepositoryAssociation::dataRepositorySubdirectories))
            .setter(setter(Builder::dataRepositorySubdirectories))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataRepositorySubdirectories")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<FileCacheNFSConfiguration> NFS_FIELD = SdkField
            .<FileCacheNFSConfiguration> builder(MarshallingType.SDK_POJO).memberName("NFS")
            .getter(getter(FileCacheDataRepositoryAssociation::nfs)).setter(setter(Builder::nfs))
            .constructor(FileCacheNFSConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NFS").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FILE_CACHE_PATH_FIELD,
            DATA_REPOSITORY_PATH_FIELD, DATA_REPOSITORY_SUBDIRECTORIES_FIELD, NFS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String fileCachePath;

    private final String dataRepositoryPath;

    private final List<String> dataRepositorySubdirectories;

    private final FileCacheNFSConfiguration nfs;

    private FileCacheDataRepositoryAssociation(BuilderImpl builder) {
        this.fileCachePath = builder.fileCachePath;
        this.dataRepositoryPath = builder.dataRepositoryPath;
        this.dataRepositorySubdirectories = builder.dataRepositorySubdirectories;
        this.nfs = builder.nfs;
    }

    /**
     * <p>
     * A path on the cache that points to a high-level directory (such as <code>/ns1/</code>) or subdirectory (such as
     * <code>/ns1/subdir/</code>) that will be mapped 1-1 with <code>DataRepositoryPath</code>. The leading forward
     * slash in the name is required. Two data repository associations cannot have overlapping cache paths. For example,
     * if a data repository is associated with cache path <code>/ns1/</code>, then you cannot link another data
     * repository with cache path <code>/ns1/ns2</code>.
     * </p>
     * <p>
     * This path specifies where in your cache files will be exported from. This cache directory can be linked to only
     * one data repository, and no data repository other can be linked to the directory.
     * </p>
     * <note>
     * <p>
     * The cache path can only be set to root (/) on an NFS DRA when <code>DataRepositorySubdirectories</code> is
     * specified. If you specify root (/) as the cache path, you can create only one DRA on the cache.
     * </p>
     * <p>
     * The cache path cannot be set to root (/) for an S3 DRA.
     * </p>
     * </note>
     * 
     * @return A path on the cache that points to a high-level directory (such as <code>/ns1/</code>) or subdirectory
     *         (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with <code>DataRepositoryPath</code>. The
     *         leading forward slash in the name is required. Two data repository associations cannot have overlapping
     *         cache paths. For example, if a data repository is associated with cache path <code>/ns1/</code>, then you
     *         cannot link another data repository with cache path <code>/ns1/ns2</code>.</p>
     *         <p>
     *         This path specifies where in your cache files will be exported from. This cache directory can be linked
     *         to only one data repository, and no data repository other can be linked to the directory.
     *         </p>
     *         <note>
     *         <p>
     *         The cache path can only be set to root (/) on an NFS DRA when <code>DataRepositorySubdirectories</code>
     *         is specified. If you specify root (/) as the cache path, you can create only one DRA on the cache.
     *         </p>
     *         <p>
     *         The cache path cannot be set to root (/) for an S3 DRA.
     *         </p>
     */
    public final String fileCachePath() {
        return fileCachePath;
    }

    /**
     * <p>
     * The path to the S3 or NFS data repository that links to the cache. You must provide one of the following paths:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The path can be an NFS data repository that links to the cache. The path can be in one of two formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you are not using the <code>DataRepositorySubdirectories</code> parameter, the path is to an NFS Export
     * directory (or one of its subdirectories) in the format <code>nfs://nfs-domain-name/exportpath</code>. You can
     * therefore link a single NFS Export to a single data repository association.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you are using the <code>DataRepositorySubdirectories</code> parameter, the path is the domain name of the NFS
     * file system in the format <code>nfs://filer-domain-name</code>, which indicates the root of the subdirectories
     * specified with the <code>DataRepositorySubdirectories</code> parameter.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * The path can be an S3 bucket or prefix in the format <code>s3://bucket-name/prefix/</code> (where
     * <code>prefix</code> is optional).
     * </p>
     * </li>
     * </ul>
     * 
     * @return The path to the S3 or NFS data repository that links to the cache. You must provide one of the following
     *         paths:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The path can be an NFS data repository that links to the cache. The path can be in one of two formats:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you are not using the <code>DataRepositorySubdirectories</code> parameter, the path is to an NFS
     *         Export directory (or one of its subdirectories) in the format
     *         <code>nfs://nfs-domain-name/exportpath</code>. You can therefore link a single NFS Export to a single
     *         data repository association.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you are using the <code>DataRepositorySubdirectories</code> parameter, the path is the domain name of
     *         the NFS file system in the format <code>nfs://filer-domain-name</code>, which indicates the root of the
     *         subdirectories specified with the <code>DataRepositorySubdirectories</code> parameter.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         The path can be an S3 bucket or prefix in the format <code>s3://bucket-name/prefix/</code> (where
     *         <code>prefix</code> is optional).
     *         </p>
     *         </li>
     */
    public final String dataRepositoryPath() {
        return dataRepositoryPath;
    }

    /**
     * For responses, this returns true if the service returned a value for the DataRepositorySubdirectories property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDataRepositorySubdirectories() {
        return dataRepositorySubdirectories != null && !(dataRepositorySubdirectories instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of NFS Exports that will be linked with this data repository association. The Export paths are in the
     * format <code>/exportpath1</code>. To use this parameter, you must configure <code>DataRepositoryPath</code> as
     * the domain name of the NFS file system. The NFS file system domain name in effect is the root of the
     * subdirectories. Note that <code>DataRepositorySubdirectories</code> is not supported for S3 data repositories.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDataRepositorySubdirectories} method.
     * </p>
     * 
     * @return A list of NFS Exports that will be linked with this data repository association. The Export paths are in
     *         the format <code>/exportpath1</code>. To use this parameter, you must configure
     *         <code>DataRepositoryPath</code> as the domain name of the NFS file system. The NFS file system domain
     *         name in effect is the root of the subdirectories. Note that <code>DataRepositorySubdirectories</code> is
     *         not supported for S3 data repositories.
     */
    public final List<String> dataRepositorySubdirectories() {
        return dataRepositorySubdirectories;
    }

    /**
     * <p>
     * The configuration for a data repository association that links an Amazon File Cache resource to an NFS data
     * repository.
     * </p>
     * 
     * @return The configuration for a data repository association that links an Amazon File Cache resource to an NFS
     *         data repository.
     */
    public final FileCacheNFSConfiguration nfs() {
        return nfs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fileCachePath());
        hashCode = 31 * hashCode + Objects.hashCode(dataRepositoryPath());
        hashCode = 31 * hashCode + Objects.hashCode(hasDataRepositorySubdirectories() ? dataRepositorySubdirectories() : null);
        hashCode = 31 * hashCode + Objects.hashCode(nfs());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FileCacheDataRepositoryAssociation)) {
            return false;
        }
        FileCacheDataRepositoryAssociation other = (FileCacheDataRepositoryAssociation) obj;
        return Objects.equals(fileCachePath(), other.fileCachePath())
                && Objects.equals(dataRepositoryPath(), other.dataRepositoryPath())
                && hasDataRepositorySubdirectories() == other.hasDataRepositorySubdirectories()
                && Objects.equals(dataRepositorySubdirectories(), other.dataRepositorySubdirectories())
                && Objects.equals(nfs(), other.nfs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FileCacheDataRepositoryAssociation").add("FileCachePath", fileCachePath())
                .add("DataRepositoryPath", dataRepositoryPath())
                .add("DataRepositorySubdirectories", hasDataRepositorySubdirectories() ? dataRepositorySubdirectories() : null)
                .add("NFS", nfs()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FileCachePath":
            return Optional.ofNullable(clazz.cast(fileCachePath()));
        case "DataRepositoryPath":
            return Optional.ofNullable(clazz.cast(dataRepositoryPath()));
        case "DataRepositorySubdirectories":
            return Optional.ofNullable(clazz.cast(dataRepositorySubdirectories()));
        case "NFS":
            return Optional.ofNullable(clazz.cast(nfs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FileCacheDataRepositoryAssociation, T> g) {
        return obj -> g.apply((FileCacheDataRepositoryAssociation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FileCacheDataRepositoryAssociation> {
        /**
         * <p>
         * A path on the cache that points to a high-level directory (such as <code>/ns1/</code>) or subdirectory (such
         * as <code>/ns1/subdir/</code>) that will be mapped 1-1 with <code>DataRepositoryPath</code>. The leading
         * forward slash in the name is required. Two data repository associations cannot have overlapping cache paths.
         * For example, if a data repository is associated with cache path <code>/ns1/</code>, then you cannot link
         * another data repository with cache path <code>/ns1/ns2</code>.
         * </p>
         * <p>
         * This path specifies where in your cache files will be exported from. This cache directory can be linked to
         * only one data repository, and no data repository other can be linked to the directory.
         * </p>
         * <note>
         * <p>
         * The cache path can only be set to root (/) on an NFS DRA when <code>DataRepositorySubdirectories</code> is
         * specified. If you specify root (/) as the cache path, you can create only one DRA on the cache.
         * </p>
         * <p>
         * The cache path cannot be set to root (/) for an S3 DRA.
         * </p>
         * </note>
         * 
         * @param fileCachePath
         *        A path on the cache that points to a high-level directory (such as <code>/ns1/</code>) or subdirectory
         *        (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with <code>DataRepositoryPath</code>. The
         *        leading forward slash in the name is required. Two data repository associations cannot have
         *        overlapping cache paths. For example, if a data repository is associated with cache path
         *        <code>/ns1/</code>, then you cannot link another data repository with cache path <code>/ns1/ns2</code>
         *        .</p>
         *        <p>
         *        This path specifies where in your cache files will be exported from. This cache directory can be
         *        linked to only one data repository, and no data repository other can be linked to the directory.
         *        </p>
         *        <note>
         *        <p>
         *        The cache path can only be set to root (/) on an NFS DRA when
         *        <code>DataRepositorySubdirectories</code> is specified. If you specify root (/) as the cache path, you
         *        can create only one DRA on the cache.
         *        </p>
         *        <p>
         *        The cache path cannot be set to root (/) for an S3 DRA.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileCachePath(String fileCachePath);

        /**
         * <p>
         * The path to the S3 or NFS data repository that links to the cache. You must provide one of the following
         * paths:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The path can be an NFS data repository that links to the cache. The path can be in one of two formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you are not using the <code>DataRepositorySubdirectories</code> parameter, the path is to an NFS Export
         * directory (or one of its subdirectories) in the format <code>nfs://nfs-domain-name/exportpath</code>. You can
         * therefore link a single NFS Export to a single data repository association.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you are using the <code>DataRepositorySubdirectories</code> parameter, the path is the domain name of the
         * NFS file system in the format <code>nfs://filer-domain-name</code>, which indicates the root of the
         * subdirectories specified with the <code>DataRepositorySubdirectories</code> parameter.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * The path can be an S3 bucket or prefix in the format <code>s3://bucket-name/prefix/</code> (where
         * <code>prefix</code> is optional).
         * </p>
         * </li>
         * </ul>
         * 
         * @param dataRepositoryPath
         *        The path to the S3 or NFS data repository that links to the cache. You must provide one of the
         *        following paths:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The path can be an NFS data repository that links to the cache. The path can be in one of two formats:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you are not using the <code>DataRepositorySubdirectories</code> parameter, the path is to an NFS
         *        Export directory (or one of its subdirectories) in the format
         *        <code>nfs://nfs-domain-name/exportpath</code>. You can therefore link a single NFS Export to a single
         *        data repository association.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you are using the <code>DataRepositorySubdirectories</code> parameter, the path is the domain name
         *        of the NFS file system in the format <code>nfs://filer-domain-name</code>, which indicates the root of
         *        the subdirectories specified with the <code>DataRepositorySubdirectories</code> parameter.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        The path can be an S3 bucket or prefix in the format <code>s3://bucket-name/prefix/</code> (where
         *        <code>prefix</code> is optional).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRepositoryPath(String dataRepositoryPath);

        /**
         * <p>
         * A list of NFS Exports that will be linked with this data repository association. The Export paths are in the
         * format <code>/exportpath1</code>. To use this parameter, you must configure <code>DataRepositoryPath</code>
         * as the domain name of the NFS file system. The NFS file system domain name in effect is the root of the
         * subdirectories. Note that <code>DataRepositorySubdirectories</code> is not supported for S3 data
         * repositories.
         * </p>
         * 
         * @param dataRepositorySubdirectories
         *        A list of NFS Exports that will be linked with this data repository association. The Export paths are
         *        in the format <code>/exportpath1</code>. To use this parameter, you must configure
         *        <code>DataRepositoryPath</code> as the domain name of the NFS file system. The NFS file system domain
         *        name in effect is the root of the subdirectories. Note that <code>DataRepositorySubdirectories</code>
         *        is not supported for S3 data repositories.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRepositorySubdirectories(Collection<String> dataRepositorySubdirectories);

        /**
         * <p>
         * A list of NFS Exports that will be linked with this data repository association. The Export paths are in the
         * format <code>/exportpath1</code>. To use this parameter, you must configure <code>DataRepositoryPath</code>
         * as the domain name of the NFS file system. The NFS file system domain name in effect is the root of the
         * subdirectories. Note that <code>DataRepositorySubdirectories</code> is not supported for S3 data
         * repositories.
         * </p>
         * 
         * @param dataRepositorySubdirectories
         *        A list of NFS Exports that will be linked with this data repository association. The Export paths are
         *        in the format <code>/exportpath1</code>. To use this parameter, you must configure
         *        <code>DataRepositoryPath</code> as the domain name of the NFS file system. The NFS file system domain
         *        name in effect is the root of the subdirectories. Note that <code>DataRepositorySubdirectories</code>
         *        is not supported for S3 data repositories.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRepositorySubdirectories(String... dataRepositorySubdirectories);

        /**
         * <p>
         * The configuration for a data repository association that links an Amazon File Cache resource to an NFS data
         * repository.
         * </p>
         * 
         * @param nfs
         *        The configuration for a data repository association that links an Amazon File Cache resource to an NFS
         *        data repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nfs(FileCacheNFSConfiguration nfs);

        /**
         * <p>
         * The configuration for a data repository association that links an Amazon File Cache resource to an NFS data
         * repository.
         * </p>
         * This is a convenience method that creates an instance of the {@link FileCacheNFSConfiguration.Builder}
         * avoiding the need to create one manually via {@link FileCacheNFSConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FileCacheNFSConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #nfs(FileCacheNFSConfiguration)}.
         * 
         * @param nfs
         *        a consumer that will call methods on {@link FileCacheNFSConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nfs(FileCacheNFSConfiguration)
         */
        default Builder nfs(Consumer<FileCacheNFSConfiguration.Builder> nfs) {
            return nfs(FileCacheNFSConfiguration.builder().applyMutation(nfs).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String fileCachePath;

        private String dataRepositoryPath;

        private List<String> dataRepositorySubdirectories = DefaultSdkAutoConstructList.getInstance();

        private FileCacheNFSConfiguration nfs;

        private BuilderImpl() {
        }

        private BuilderImpl(FileCacheDataRepositoryAssociation model) {
            fileCachePath(model.fileCachePath);
            dataRepositoryPath(model.dataRepositoryPath);
            dataRepositorySubdirectories(model.dataRepositorySubdirectories);
            nfs(model.nfs);
        }

        public final String getFileCachePath() {
            return fileCachePath;
        }

        public final void setFileCachePath(String fileCachePath) {
            this.fileCachePath = fileCachePath;
        }

        @Override
        public final Builder fileCachePath(String fileCachePath) {
            this.fileCachePath = fileCachePath;
            return this;
        }

        public final String getDataRepositoryPath() {
            return dataRepositoryPath;
        }

        public final void setDataRepositoryPath(String dataRepositoryPath) {
            this.dataRepositoryPath = dataRepositoryPath;
        }

        @Override
        public final Builder dataRepositoryPath(String dataRepositoryPath) {
            this.dataRepositoryPath = dataRepositoryPath;
            return this;
        }

        public final Collection<String> getDataRepositorySubdirectories() {
            if (dataRepositorySubdirectories instanceof SdkAutoConstructList) {
                return null;
            }
            return dataRepositorySubdirectories;
        }

        public final void setDataRepositorySubdirectories(Collection<String> dataRepositorySubdirectories) {
            this.dataRepositorySubdirectories = SubDirectoriesPathsCopier.copy(dataRepositorySubdirectories);
        }

        @Override
        public final Builder dataRepositorySubdirectories(Collection<String> dataRepositorySubdirectories) {
            this.dataRepositorySubdirectories = SubDirectoriesPathsCopier.copy(dataRepositorySubdirectories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataRepositorySubdirectories(String... dataRepositorySubdirectories) {
            dataRepositorySubdirectories(Arrays.asList(dataRepositorySubdirectories));
            return this;
        }

        public final FileCacheNFSConfiguration.Builder getNfs() {
            return nfs != null ? nfs.toBuilder() : null;
        }

        public final void setNfs(FileCacheNFSConfiguration.BuilderImpl nfs) {
            this.nfs = nfs != null ? nfs.build() : null;
        }

        @Override
        public final Builder nfs(FileCacheNFSConfiguration nfs) {
            this.nfs = nfs;
            return this;
        }

        @Override
        public FileCacheDataRepositoryAssociation build() {
            return new FileCacheDataRepositoryAssociation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
