/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.fsx.internal.FSxServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.fsx.model.ActiveDirectoryErrorException;
import software.amazon.awssdk.services.fsx.model.AssociateFileSystemAliasesRequest;
import software.amazon.awssdk.services.fsx.model.AssociateFileSystemAliasesResponse;
import software.amazon.awssdk.services.fsx.model.BackupBeingCopiedException;
import software.amazon.awssdk.services.fsx.model.BackupInProgressException;
import software.amazon.awssdk.services.fsx.model.BackupNotFoundException;
import software.amazon.awssdk.services.fsx.model.BackupRestoringException;
import software.amazon.awssdk.services.fsx.model.BadRequestException;
import software.amazon.awssdk.services.fsx.model.CancelDataRepositoryTaskRequest;
import software.amazon.awssdk.services.fsx.model.CancelDataRepositoryTaskResponse;
import software.amazon.awssdk.services.fsx.model.CopyBackupRequest;
import software.amazon.awssdk.services.fsx.model.CopyBackupResponse;
import software.amazon.awssdk.services.fsx.model.CopySnapshotAndUpdateVolumeRequest;
import software.amazon.awssdk.services.fsx.model.CopySnapshotAndUpdateVolumeResponse;
import software.amazon.awssdk.services.fsx.model.CreateBackupRequest;
import software.amazon.awssdk.services.fsx.model.CreateBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryAssociationRequest;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryAssociationResponse;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryTaskRequest;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryTaskResponse;
import software.amazon.awssdk.services.fsx.model.CreateFileCacheRequest;
import software.amazon.awssdk.services.fsx.model.CreateFileCacheResponse;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemFromBackupRequest;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemFromBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemResponse;
import software.amazon.awssdk.services.fsx.model.CreateSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.CreateSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.CreateStorageVirtualMachineRequest;
import software.amazon.awssdk.services.fsx.model.CreateStorageVirtualMachineResponse;
import software.amazon.awssdk.services.fsx.model.CreateVolumeFromBackupRequest;
import software.amazon.awssdk.services.fsx.model.CreateVolumeFromBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateVolumeRequest;
import software.amazon.awssdk.services.fsx.model.CreateVolumeResponse;
import software.amazon.awssdk.services.fsx.model.DataRepositoryAssociationNotFoundException;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskEndedException;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskExecutingException;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskNotFoundException;
import software.amazon.awssdk.services.fsx.model.DeleteBackupRequest;
import software.amazon.awssdk.services.fsx.model.DeleteBackupResponse;
import software.amazon.awssdk.services.fsx.model.DeleteDataRepositoryAssociationRequest;
import software.amazon.awssdk.services.fsx.model.DeleteDataRepositoryAssociationResponse;
import software.amazon.awssdk.services.fsx.model.DeleteFileCacheRequest;
import software.amazon.awssdk.services.fsx.model.DeleteFileCacheResponse;
import software.amazon.awssdk.services.fsx.model.DeleteFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.DeleteFileSystemResponse;
import software.amazon.awssdk.services.fsx.model.DeleteSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.DeleteSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.DeleteStorageVirtualMachineRequest;
import software.amazon.awssdk.services.fsx.model.DeleteStorageVirtualMachineResponse;
import software.amazon.awssdk.services.fsx.model.DeleteVolumeRequest;
import software.amazon.awssdk.services.fsx.model.DeleteVolumeResponse;
import software.amazon.awssdk.services.fsx.model.DescribeBackupsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeBackupsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryAssociationsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryAssociationsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksRequest;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksResponse;
import software.amazon.awssdk.services.fsx.model.DescribeFileCachesRequest;
import software.amazon.awssdk.services.fsx.model.DescribeFileCachesResponse;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemAliasesRequest;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemAliasesResponse;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeSharedVpcConfigurationRequest;
import software.amazon.awssdk.services.fsx.model.DescribeSharedVpcConfigurationResponse;
import software.amazon.awssdk.services.fsx.model.DescribeSnapshotsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeSnapshotsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeStorageVirtualMachinesRequest;
import software.amazon.awssdk.services.fsx.model.DescribeStorageVirtualMachinesResponse;
import software.amazon.awssdk.services.fsx.model.DescribeVolumesRequest;
import software.amazon.awssdk.services.fsx.model.DescribeVolumesResponse;
import software.amazon.awssdk.services.fsx.model.DisassociateFileSystemAliasesRequest;
import software.amazon.awssdk.services.fsx.model.DisassociateFileSystemAliasesResponse;
import software.amazon.awssdk.services.fsx.model.FSxException;
import software.amazon.awssdk.services.fsx.model.FileCacheNotFoundException;
import software.amazon.awssdk.services.fsx.model.FileSystemNotFoundException;
import software.amazon.awssdk.services.fsx.model.IncompatibleParameterErrorException;
import software.amazon.awssdk.services.fsx.model.IncompatibleRegionForMultiAzException;
import software.amazon.awssdk.services.fsx.model.InternalServerErrorException;
import software.amazon.awssdk.services.fsx.model.InvalidDataRepositoryTypeException;
import software.amazon.awssdk.services.fsx.model.InvalidDestinationKmsKeyException;
import software.amazon.awssdk.services.fsx.model.InvalidExportPathException;
import software.amazon.awssdk.services.fsx.model.InvalidImportPathException;
import software.amazon.awssdk.services.fsx.model.InvalidNetworkSettingsException;
import software.amazon.awssdk.services.fsx.model.InvalidPerUnitStorageThroughputException;
import software.amazon.awssdk.services.fsx.model.InvalidRegionException;
import software.amazon.awssdk.services.fsx.model.InvalidSourceKmsKeyException;
import software.amazon.awssdk.services.fsx.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.fsx.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.fsx.model.MissingFileCacheConfigurationException;
import software.amazon.awssdk.services.fsx.model.MissingFileSystemConfigurationException;
import software.amazon.awssdk.services.fsx.model.MissingVolumeConfigurationException;
import software.amazon.awssdk.services.fsx.model.NotServiceResourceErrorException;
import software.amazon.awssdk.services.fsx.model.ReleaseFileSystemNfsV3LocksRequest;
import software.amazon.awssdk.services.fsx.model.ReleaseFileSystemNfsV3LocksResponse;
import software.amazon.awssdk.services.fsx.model.ResourceDoesNotSupportTaggingException;
import software.amazon.awssdk.services.fsx.model.ResourceNotFoundException;
import software.amazon.awssdk.services.fsx.model.RestoreVolumeFromSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.RestoreVolumeFromSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.ServiceLimitExceededException;
import software.amazon.awssdk.services.fsx.model.SnapshotNotFoundException;
import software.amazon.awssdk.services.fsx.model.SourceBackupUnavailableException;
import software.amazon.awssdk.services.fsx.model.StartMisconfiguredStateRecoveryRequest;
import software.amazon.awssdk.services.fsx.model.StartMisconfiguredStateRecoveryResponse;
import software.amazon.awssdk.services.fsx.model.StorageVirtualMachineNotFoundException;
import software.amazon.awssdk.services.fsx.model.TagResourceRequest;
import software.amazon.awssdk.services.fsx.model.TagResourceResponse;
import software.amazon.awssdk.services.fsx.model.UnsupportedOperationException;
import software.amazon.awssdk.services.fsx.model.UntagResourceRequest;
import software.amazon.awssdk.services.fsx.model.UntagResourceResponse;
import software.amazon.awssdk.services.fsx.model.UpdateDataRepositoryAssociationRequest;
import software.amazon.awssdk.services.fsx.model.UpdateDataRepositoryAssociationResponse;
import software.amazon.awssdk.services.fsx.model.UpdateFileCacheRequest;
import software.amazon.awssdk.services.fsx.model.UpdateFileCacheResponse;
import software.amazon.awssdk.services.fsx.model.UpdateFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.UpdateFileSystemResponse;
import software.amazon.awssdk.services.fsx.model.UpdateSharedVpcConfigurationRequest;
import software.amazon.awssdk.services.fsx.model.UpdateSharedVpcConfigurationResponse;
import software.amazon.awssdk.services.fsx.model.UpdateSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.UpdateSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.UpdateStorageVirtualMachineRequest;
import software.amazon.awssdk.services.fsx.model.UpdateStorageVirtualMachineResponse;
import software.amazon.awssdk.services.fsx.model.UpdateVolumeRequest;
import software.amazon.awssdk.services.fsx.model.UpdateVolumeResponse;
import software.amazon.awssdk.services.fsx.model.VolumeNotFoundException;
import software.amazon.awssdk.services.fsx.transform.AssociateFileSystemAliasesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CancelDataRepositoryTaskRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CopyBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CopySnapshotAndUpdateVolumeRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateDataRepositoryAssociationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateDataRepositoryTaskRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateFileCacheRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateFileSystemFromBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateFileSystemRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateStorageVirtualMachineRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateVolumeFromBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateVolumeRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteDataRepositoryAssociationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteFileCacheRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteFileSystemRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteStorageVirtualMachineRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteVolumeRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeBackupsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeDataRepositoryAssociationsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeDataRepositoryTasksRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeFileCachesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeFileSystemAliasesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeFileSystemsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeSharedVpcConfigurationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeSnapshotsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeStorageVirtualMachinesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeVolumesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DisassociateFileSystemAliasesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.ReleaseFileSystemNfsV3LocksRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.RestoreVolumeFromSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.StartMisconfiguredStateRecoveryRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateDataRepositoryAssociationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateFileCacheRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateFileSystemRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateSharedVpcConfigurationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateStorageVirtualMachineRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateVolumeRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link FSxAsyncClient}.
 *
 * @see FSxAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFSxAsyncClient implements FSxAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultFSxAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultFSxAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Use this action to associate one or more Domain Name Server (DNS) aliases with an existing Amazon FSx for Windows
     * File Server file system. A file system can have a maximum of 50 DNS aliases associated with it at any one time.
     * If you try to associate a DNS alias that is already associated with the file system, FSx takes no action on that
     * alias in the request. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS Aliases</a>
     * and <a href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/walkthrough05-file-system-custom-CNAME.html">
     * Walkthrough 5: Using DNS aliases to access your file system</a>, including additional steps you must take to be
     * able to access your file system using a DNS alias.
     * </p>
     * <p>
     * The system response shows the DNS aliases that Amazon FSx is attempting to associate with the file system. Use
     * the API operation to monitor the status of the aliases Amazon FSx is associating with the file system.
     * </p>
     *
     * @param associateFileSystemAliasesRequest
     *        The request object specifying one or more DNS alias names to associate with an Amazon FSx for Windows File
     *        Server file system.
     * @return A Java Future containing the result of the AssociateFileSystemAliases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.AssociateFileSystemAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/AssociateFileSystemAliases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateFileSystemAliasesResponse> associateFileSystemAliases(
            AssociateFileSystemAliasesRequest associateFileSystemAliasesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateFileSystemAliasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateFileSystemAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateFileSystemAliases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateFileSystemAliasesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateFileSystemAliasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateFileSystemAliasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateFileSystemAliasesRequest, AssociateFileSystemAliasesResponse>()
                            .withOperationName("AssociateFileSystemAliases").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateFileSystemAliasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateFileSystemAliasesRequest));
            CompletableFuture<AssociateFileSystemAliasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels an existing Amazon FSx for Lustre data repository task if that task is in either the <code>PENDING</code>
     * or <code>EXECUTING</code> state. When you cancel am export task, Amazon FSx does the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Any files that FSx has already exported are not reverted.
     * </p>
     * </li>
     * <li>
     * <p>
     * FSx continues to export any files that are in-flight when the cancel operation is received.
     * </p>
     * </li>
     * <li>
     * <p>
     * FSx does not export any files that have not yet been exported.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For a release task, Amazon FSx will stop releasing files upon cancellation. Any files that have already been
     * released will remain in the released state.
     * </p>
     *
     * @param cancelDataRepositoryTaskRequest
     *        Cancels a data repository task.
     * @return A Java Future containing the result of the CancelDataRepositoryTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>DataRepositoryTaskNotFoundException The data repository task or tasks you specified could not be
     *         found.</li>
     *         <li>DataRepositoryTaskEndedException The data repository task could not be canceled because the task has
     *         already ended.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CancelDataRepositoryTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CancelDataRepositoryTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelDataRepositoryTaskResponse> cancelDataRepositoryTask(
            CancelDataRepositoryTaskRequest cancelDataRepositoryTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelDataRepositoryTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelDataRepositoryTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelDataRepositoryTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelDataRepositoryTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelDataRepositoryTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelDataRepositoryTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelDataRepositoryTaskRequest, CancelDataRepositoryTaskResponse>()
                            .withOperationName("CancelDataRepositoryTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelDataRepositoryTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelDataRepositoryTaskRequest));
            CompletableFuture<CancelDataRepositoryTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Copies an existing backup within the same Amazon Web Services account to another Amazon Web Services Region
     * (cross-Region copy) or within the same Amazon Web Services Region (in-Region copy). You can have up to five
     * backup copy requests in progress to a single destination Region per account.
     * </p>
     * <p>
     * You can use cross-Region backup copies for cross-Region disaster recovery. You can periodically take backups and
     * copy them to another Region so that in the event of a disaster in the primary Region, you can restore from backup
     * and recover availability quickly in the other Region. You can make cross-Region copies only within your Amazon
     * Web Services partition. A partition is a grouping of Regions. Amazon Web Services currently has three partitions:
     * <code>aws</code> (Standard Regions), <code>aws-cn</code> (China Regions), and <code>aws-us-gov</code> (Amazon Web
     * Services GovCloud [US] Regions).
     * </p>
     * <p>
     * You can also use backup copies to clone your file dataset to another Region or within the same Region.
     * </p>
     * <p>
     * You can use the <code>SourceRegion</code> parameter to specify the Amazon Web Services Region from which the
     * backup will be copied. For example, if you make the call from the <code>us-west-1</code> Region and want to copy
     * a backup from the <code>us-east-2</code> Region, you specify <code>us-east-2</code> in the
     * <code>SourceRegion</code> parameter to make a cross-Region copy. If you don't specify a Region, the backup copy
     * is created in the same Region where the request is sent from (in-Region copy).
     * </p>
     * <p>
     * For more information about creating backup copies, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/using-backups.html#copy-backups"> Copying backups</a>
     * in the <i>Amazon FSx for Windows User Guide</i>, <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/using-backups-fsx.html#copy-backups">Copying backups</a>
     * in the <i>Amazon FSx for Lustre User Guide</i>, and <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/using-backups.html#copy-backups">Copying backups</a> in
     * the <i>Amazon FSx for OpenZFS User Guide</i>.
     * </p>
     *
     * @param copyBackupRequest
     * @return A Java Future containing the result of the CopyBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>BackupNotFoundException No Amazon FSx backups were found based upon the supplied parameters.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>InvalidSourceKmsKeyException The Key Management Service (KMS) key of the source backup is not valid.</li>
     *         <li>InvalidDestinationKmsKeyException The Key Management Service (KMS) key of the destination backup is
     *         not valid.</li>
     *         <li>InvalidRegionException The Region provided for <code>SourceRegion</code> is not valid or is in a
     *         different Amazon Web Services partition.</li>
     *         <li>SourceBackupUnavailableException The request was rejected because the lifecycle status of the source
     *         backup isn't <code>AVAILABLE</code>.</li>
     *         <li>IncompatibleRegionForMultiAzException Amazon FSx doesn't support Multi-AZ Windows File Server copy
     *         backup in the destination Region, so the copied backup can't be restored.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CopyBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CopyBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CopyBackupResponse> copyBackup(CopyBackupRequest copyBackupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(copyBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copyBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopyBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CopyBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CopyBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CopyBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CopyBackupRequest, CopyBackupResponse>().withOperationName("CopyBackup")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CopyBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(copyBackupRequest));
            CompletableFuture<CopyBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing volume by using a snapshot from another Amazon FSx for OpenZFS file system. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/on-demand-replication.html">on-demand data
     * replication</a> in the Amazon FSx for OpenZFS User Guide.
     * </p>
     *
     * @param copySnapshotAndUpdateVolumeRequest
     * @return A Java Future containing the result of the CopySnapshotAndUpdateVolume operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CopySnapshotAndUpdateVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CopySnapshotAndUpdateVolume"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CopySnapshotAndUpdateVolumeResponse> copySnapshotAndUpdateVolume(
            CopySnapshotAndUpdateVolumeRequest copySnapshotAndUpdateVolumeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(copySnapshotAndUpdateVolumeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copySnapshotAndUpdateVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopySnapshotAndUpdateVolume");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CopySnapshotAndUpdateVolumeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CopySnapshotAndUpdateVolumeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CopySnapshotAndUpdateVolumeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CopySnapshotAndUpdateVolumeRequest, CopySnapshotAndUpdateVolumeResponse>()
                            .withOperationName("CopySnapshotAndUpdateVolume").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CopySnapshotAndUpdateVolumeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(copySnapshotAndUpdateVolumeRequest));
            CompletableFuture<CopySnapshotAndUpdateVolumeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a backup of an existing Amazon FSx for Windows File Server file system, Amazon FSx for Lustre file
     * system, Amazon FSx for NetApp ONTAP volume, or Amazon FSx for OpenZFS file system. We recommend creating regular
     * backups so that you can restore a file system or volume from a backup if an issue arises with the original file
     * system or volume.
     * </p>
     * <p>
     * For Amazon FSx for Lustre file systems, you can create a backup only for file systems that have the following
     * configuration:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A Persistent deployment type
     * </p>
     * </li>
     * <li>
     * <p>
     * Are <i>not</i> linked to a data repository
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about backups, see the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For Amazon FSx for Lustre, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/using-backups-fsx.html">Working with FSx for Lustre
     * backups</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon FSx for Windows, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/using-backups.html">Working with FSx for Windows
     * backups</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon FSx for NetApp ONTAP, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/using-backups.html">Working with FSx for NetApp ONTAP
     * backups</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon FSx for OpenZFS, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/using-backups.html">Working with FSx for OpenZFS
     * backups</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If a backup with the specified client request token exists and the parameters match, this operation returns the
     * description of the existing backup. If a backup with the specified client request token exists and the parameters
     * don't match, this operation returns <code>IncompatibleParameterError</code>. If a backup with the specified
     * client request token doesn't exist, <code>CreateBackup</code> does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new Amazon FSx backup with an assigned ID, and an initial lifecycle state of <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the backup.
     * </p>
     * </li>
     * </ul>
     * <p>
     * By using the idempotent operation, you can retry a <code>CreateBackup</code> operation without the risk of
     * creating an extra backup. This approach can be useful when an initial call fails in a way that makes it unclear
     * whether a backup was created. If you use the same client request token and the initial call created a backup, the
     * operation returns a successful result because all the parameters are the same.
     * </p>
     * <p>
     * The <code>CreateBackup</code> operation returns while the backup's lifecycle state is still <code>CREATING</code>
     * . You can check the backup creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeBackups.html">DescribeBackups</a>
     * operation, which returns the backup state along with other information.
     * </p>
     *
     * @param createBackupRequest
     *        The request object for the <code>CreateBackup</code> operation.
     * @return A Java Future containing the result of the CreateBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>VolumeNotFoundException No Amazon FSx volumes were found based upon the supplied parameters.</li>
     *         <li>BackupInProgressException Another backup is already under way. Wait for completion before initiating
     *         additional backups of this file system.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBackupResponse> createBackup(CreateBackupRequest createBackupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBackupRequest, CreateBackupResponse>()
                            .withOperationName("CreateBackup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createBackupRequest));
            CompletableFuture<CreateBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon FSx for Lustre data repository association (DRA). A data repository association is a link
     * between a directory on the file system and an Amazon S3 bucket or prefix. You can have a maximum of 8 data
     * repository associations on a file system. Data repository associations are supported on all FSx for Lustre 2.12
     * and 2.15 file systems, excluding <code>scratch_1</code> deployment type.
     * </p>
     * <p>
     * Each data repository association must have a unique Amazon FSx file system directory and a unique S3 bucket or
     * prefix associated with it. You can configure a data repository association for automatic import only, for
     * automatic export only, or for both. To learn more about linking a data repository to your file system, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/create-dra-linked-data-repo.html">Linking your file
     * system to an S3 bucket</a>.
     * </p>
     * <note>
     * <p>
     * <code>CreateDataRepositoryAssociation</code> isn't supported on Amazon File Cache resources. To create a DRA on
     * Amazon File Cache, use the <code>CreateFileCache</code> operation.
     * </p>
     * </note>
     *
     * @param createDataRepositoryAssociationRequest
     * @return A Java Future containing the result of the CreateDataRepositoryAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateDataRepositoryAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateDataRepositoryAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataRepositoryAssociationResponse> createDataRepositoryAssociation(
            CreateDataRepositoryAssociationRequest createDataRepositoryAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataRepositoryAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createDataRepositoryAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataRepositoryAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataRepositoryAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataRepositoryAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataRepositoryAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataRepositoryAssociationRequest, CreateDataRepositoryAssociationResponse>()
                            .withOperationName("CreateDataRepositoryAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataRepositoryAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataRepositoryAssociationRequest));
            CompletableFuture<CreateDataRepositoryAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon FSx for Lustre data repository task. A <code>CreateDataRepositoryTask</code> operation will
     * fail if a data repository is not linked to the FSx file system.
     * </p>
     * <p>
     * You use import and export data repository tasks to perform bulk operations between your FSx for Lustre file
     * system and its linked data repositories. An example of a data repository task is exporting any data and metadata
     * changes, including POSIX metadata, to files, directories, and symbolic links (symlinks) from your FSx file system
     * to a linked data repository.
     * </p>
     * <p>
     * You use release data repository tasks to release data from your file system for files that are exported to S3.
     * The metadata of released files remains on the file system so users or applications can still access released
     * files by reading the files again, which will restore data from Amazon S3 to the FSx for Lustre file system.
     * </p>
     * <p>
     * To learn more about data repository tasks, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-repository-tasks.html">Data Repository Tasks</a>.
     * To learn more about linking a data repository to your file system, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/create-dra-linked-data-repo.html">Linking your file
     * system to an S3 bucket</a>.
     * </p>
     *
     * @param createDataRepositoryTaskRequest
     * @return A Java Future containing the result of the CreateDataRepositoryTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>DataRepositoryTaskExecutingException An existing data repository task is currently executing on the
     *         file system. Wait until the existing task has completed, then create the new task.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateDataRepositoryTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateDataRepositoryTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataRepositoryTaskResponse> createDataRepositoryTask(
            CreateDataRepositoryTaskRequest createDataRepositoryTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataRepositoryTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataRepositoryTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataRepositoryTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataRepositoryTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataRepositoryTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataRepositoryTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataRepositoryTaskRequest, CreateDataRepositoryTaskResponse>()
                            .withOperationName("CreateDataRepositoryTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataRepositoryTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataRepositoryTaskRequest));
            CompletableFuture<CreateDataRepositoryTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Amazon File Cache resource.
     * </p>
     * <p>
     * You can use this operation with a client request token in the request that Amazon File Cache uses to ensure
     * idempotent creation. If a cache with the specified client request token exists and the parameters match,
     * <code>CreateFileCache</code> returns the description of the existing cache. If a cache with the specified client
     * request token exists and the parameters don't match, this call returns <code>IncompatibleParameterError</code>.
     * If a file cache with the specified client request token doesn't exist, <code>CreateFileCache</code> does the
     * following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new, empty Amazon File Cache resourcewith an assigned ID, and an initial lifecycle state of
     * <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the cache in JSON format.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>CreateFileCache</code> call returns while the cache's lifecycle state is still <code>CREATING</code>.
     * You can check the cache creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileCaches.html">DescribeFileCaches</a>
     * operation, which returns the cache state along with other information.
     * </p>
     * </note>
     *
     * @param createFileCacheRequest
     * @return A Java Future containing the result of the CreateFileCache operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InvalidNetworkSettingsException One or more network settings specified in the request are invalid.</li>
     *         <li>InvalidPerUnitStorageThroughputException An invalid value for <code>PerUnitStorageThroughput</code>
     *         was provided. Please create your file system again, using a valid value.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>MissingFileCacheConfigurationException A cache configuration is required for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateFileCache
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateFileCache" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFileCacheResponse> createFileCache(CreateFileCacheRequest createFileCacheRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFileCacheRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFileCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFileCache");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFileCacheResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateFileCacheResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFileCacheResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFileCacheRequest, CreateFileCacheResponse>()
                            .withOperationName("CreateFileCache").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateFileCacheRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createFileCacheRequest));
            CompletableFuture<CreateFileCacheResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new, empty Amazon FSx file system. You can create the following supported Amazon FSx file systems using
     * the <code>CreateFileSystem</code> API operation:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon FSx for Lustre
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx for NetApp ONTAP
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx for OpenZFS
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx for Windows File Server
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation requires a client request token in the request that Amazon FSx uses to ensure idempotent creation.
     * This means that calling the operation multiple times with the same client request token has no effect. By using
     * the idempotent operation, you can retry a <code>CreateFileSystem</code> operation without the risk of creating an
     * extra file system. This approach can be useful when an initial call fails in a way that makes it unclear whether
     * a file system was created. Examples are if a transport level timeout occurred, or your connection was reset. If
     * you use the same client request token and the initial call created a file system, the client receives success as
     * long as the parameters are the same.
     * </p>
     * <p>
     * If a file system with the specified client request token exists and the parameters match,
     * <code>CreateFileSystem</code> returns the description of the existing file system. If a file system with the
     * specified client request token exists and the parameters don't match, this call returns
     * <code>IncompatibleParameterError</code>. If a file system with the specified client request token doesn't exist,
     * <code>CreateFileSystem</code> does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new, empty Amazon FSx file system with an assigned ID, and an initial lifecycle state of
     * <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the file system in JSON format.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>CreateFileSystem</code> call returns while the file system's lifecycle state is still
     * <code>CREATING</code>. You can check the file-system creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystems.html">DescribeFileSystems</a>
     * operation, which returns the file system state along with other information.
     * </p>
     * </note>
     *
     * @param createFileSystemRequest
     *        The request object used to create a new Amazon FSx file system.
     * @return A Java Future containing the result of the CreateFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>ActiveDirectoryErrorException An Active Directory error.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InvalidImportPathException The path provided for data repository import isn't valid.</li>
     *         <li>InvalidExportPathException The path provided for data repository export isn't valid.</li>
     *         <li>InvalidNetworkSettingsException One or more network settings specified in the request are invalid.</li>
     *         <li>InvalidPerUnitStorageThroughputException An invalid value for <code>PerUnitStorageThroughput</code>
     *         was provided. Please create your file system again, using a valid value.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>MissingFileSystemConfigurationException A file system configuration is required for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFileSystemResponse> createFileSystem(CreateFileSystemRequest createFileSystemRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFileSystemRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFileSystem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateFileSystemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFileSystemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFileSystemRequest, CreateFileSystemResponse>()
                            .withOperationName("CreateFileSystem").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateFileSystemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createFileSystemRequest));
            CompletableFuture<CreateFileSystemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Amazon FSx for Lustre, Amazon FSx for Windows File Server, or Amazon FSx for OpenZFS file system
     * from an existing Amazon FSx backup.
     * </p>
     * <p>
     * If a file system with the specified client request token exists and the parameters match, this operation returns
     * the description of the file system. If a file system with the specified client request token exists but the
     * parameters don't match, this call returns <code>IncompatibleParameterError</code>. If a file system with the
     * specified client request token doesn't exist, this operation does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new Amazon FSx file system from backup with an assigned ID, and an initial lifecycle state of
     * <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Parameters like the Active Directory, default share name, automatic backup, and backup settings default to the
     * parameters of the file system that was backed up, unless overridden. You can explicitly supply other settings.
     * </p>
     * <p>
     * By using the idempotent operation, you can retry a <code>CreateFileSystemFromBackup</code> call without the risk
     * of creating an extra file system. This approach can be useful when an initial call fails in a way that makes it
     * unclear whether a file system was created. Examples are if a transport level timeout occurred, or your connection
     * was reset. If you use the same client request token and the initial call created a file system, the client
     * receives a success message as long as the parameters are the same.
     * </p>
     * <note>
     * <p>
     * The <code>CreateFileSystemFromBackup</code> call returns while the file system's lifecycle state is still
     * <code>CREATING</code>. You can check the file-system creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystems.html"> DescribeFileSystems</a>
     * operation, which returns the file system state along with other information.
     * </p>
     * </note>
     *
     * @param createFileSystemFromBackupRequest
     *        The request object for the <code>CreateFileSystemFromBackup</code> operation.
     * @return A Java Future containing the result of the CreateFileSystemFromBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>ActiveDirectoryErrorException An Active Directory error.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InvalidNetworkSettingsException One or more network settings specified in the request are invalid.</li>
     *         <li>InvalidPerUnitStorageThroughputException An invalid value for <code>PerUnitStorageThroughput</code>
     *         was provided. Please create your file system again, using a valid value.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>BackupNotFoundException No Amazon FSx backups were found based upon the supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>MissingFileSystemConfigurationException A file system configuration is required for this operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateFileSystemFromBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateFileSystemFromBackup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFileSystemFromBackupResponse> createFileSystemFromBackup(
            CreateFileSystemFromBackupRequest createFileSystemFromBackupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFileSystemFromBackupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFileSystemFromBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFileSystemFromBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFileSystemFromBackupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateFileSystemFromBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFileSystemFromBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFileSystemFromBackupRequest, CreateFileSystemFromBackupResponse>()
                            .withOperationName("CreateFileSystemFromBackup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateFileSystemFromBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createFileSystemFromBackupRequest));
            CompletableFuture<CreateFileSystemFromBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a snapshot of an existing Amazon FSx for OpenZFS volume. With snapshots, you can easily undo file changes
     * and compare file versions by restoring the volume to a previous version.
     * </p>
     * <p>
     * If a snapshot with the specified client request token exists, and the parameters match, this operation returns
     * the description of the existing snapshot. If a snapshot with the specified client request token exists, and the
     * parameters don't match, this operation returns <code>IncompatibleParameterError</code>. If a snapshot with the
     * specified client request token doesn't exist, <code>CreateSnapshot</code> does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new OpenZFS snapshot with an assigned ID, and an initial lifecycle state of <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the snapshot.
     * </p>
     * </li>
     * </ul>
     * <p>
     * By using the idempotent operation, you can retry a <code>CreateSnapshot</code> operation without the risk of
     * creating an extra snapshot. This approach can be useful when an initial call fails in a way that makes it unclear
     * whether a snapshot was created. If you use the same client request token and the initial call created a snapshot,
     * the operation returns a successful result because all the parameters are the same.
     * </p>
     * <p>
     * The <code>CreateSnapshot</code> operation returns while the snapshot's lifecycle state is still
     * <code>CREATING</code>. You can check the snapshot creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeSnapshots.html">DescribeSnapshots</a>
     * operation, which returns the snapshot state along with other information.
     * </p>
     *
     * @param createSnapshotRequest
     * @return A Java Future containing the result of the CreateSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>VolumeNotFoundException No Amazon FSx volumes were found based upon the supplied parameters.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSnapshotResponse> createSnapshot(CreateSnapshotRequest createSnapshotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSnapshotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSnapshotRequest, CreateSnapshotResponse>()
                            .withOperationName("CreateSnapshot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSnapshotRequest));
            CompletableFuture<CreateSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a storage virtual machine (SVM) for an Amazon FSx for ONTAP file system.
     * </p>
     *
     * @param createStorageVirtualMachineRequest
     * @return A Java Future containing the result of the CreateStorageVirtualMachine operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ActiveDirectoryErrorException An Active Directory error.</li>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateStorageVirtualMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateStorageVirtualMachine"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStorageVirtualMachineResponse> createStorageVirtualMachine(
            CreateStorageVirtualMachineRequest createStorageVirtualMachineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStorageVirtualMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStorageVirtualMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStorageVirtualMachine");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateStorageVirtualMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateStorageVirtualMachineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateStorageVirtualMachineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStorageVirtualMachineRequest, CreateStorageVirtualMachineResponse>()
                            .withOperationName("CreateStorageVirtualMachine").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateStorageVirtualMachineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createStorageVirtualMachineRequest));
            CompletableFuture<CreateStorageVirtualMachineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an FSx for ONTAP or Amazon FSx for OpenZFS storage volume.
     * </p>
     *
     * @param createVolumeRequest
     * @return A Java Future containing the result of the CreateVolume operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>MissingVolumeConfigurationException A volume configuration is required for this operation.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>StorageVirtualMachineNotFoundException No FSx for ONTAP SVMs were found based upon the supplied
     *         parameters.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateVolume" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVolumeResponse> createVolume(CreateVolumeRequest createVolumeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVolumeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVolume");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateVolumeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVolumeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVolumeRequest, CreateVolumeResponse>()
                            .withOperationName("CreateVolume").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateVolumeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createVolumeRequest));
            CompletableFuture<CreateVolumeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Amazon FSx for NetApp ONTAP volume from an existing Amazon FSx volume backup.
     * </p>
     *
     * @param createVolumeFromBackupRequest
     * @return A Java Future containing the result of the CreateVolumeFromBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BackupNotFoundException No Amazon FSx backups were found based upon the supplied parameters.</li>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>MissingVolumeConfigurationException A volume configuration is required for this operation.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>StorageVirtualMachineNotFoundException No FSx for ONTAP SVMs were found based upon the supplied
     *         parameters.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.CreateVolumeFromBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateVolumeFromBackup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVolumeFromBackupResponse> createVolumeFromBackup(
            CreateVolumeFromBackupRequest createVolumeFromBackupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVolumeFromBackupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVolumeFromBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVolumeFromBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVolumeFromBackupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVolumeFromBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVolumeFromBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVolumeFromBackupRequest, CreateVolumeFromBackupResponse>()
                            .withOperationName("CreateVolumeFromBackup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateVolumeFromBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createVolumeFromBackupRequest));
            CompletableFuture<CreateVolumeFromBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon FSx backup. After deletion, the backup no longer exists, and its data is gone.
     * </p>
     * <p>
     * The <code>DeleteBackup</code> call returns instantly. The backup won't show up in later
     * <code>DescribeBackups</code> calls.
     * </p>
     * <important>
     * <p>
     * The data in a deleted backup is also deleted and can't be recovered by any means.
     * </p>
     * </important>
     *
     * @param deleteBackupRequest
     *        The request object for the <code>DeleteBackup</code> operation.
     * @return A Java Future containing the result of the DeleteBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>BackupInProgressException Another backup is already under way. Wait for completion before initiating
     *         additional backups of this file system.</li>
     *         <li>BackupNotFoundException No Amazon FSx backups were found based upon the supplied parameters.</li>
     *         <li>BackupRestoringException You can't delete a backup while it's being used to restore a file system.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>BackupBeingCopiedException You can't delete a backup while it's being copied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DeleteBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBackupResponse> deleteBackup(DeleteBackupRequest deleteBackupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBackupRequest, DeleteBackupResponse>()
                            .withOperationName("DeleteBackup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteBackupRequest));
            CompletableFuture<DeleteBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a data repository association on an Amazon FSx for Lustre file system. Deleting the data repository
     * association unlinks the file system from the Amazon S3 bucket. When deleting a data repository association, you
     * have the option of deleting the data in the file system that corresponds to the data repository association. Data
     * repository associations are supported on all FSx for Lustre 2.12 and 2.15 file systems, excluding
     * <code>scratch_1</code> deployment type.
     * </p>
     *
     * @param deleteDataRepositoryAssociationRequest
     * @return A Java Future containing the result of the DeleteDataRepositoryAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>DataRepositoryAssociationNotFoundException No data repository associations were found based upon the
     *         supplied parameters.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DeleteDataRepositoryAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteDataRepositoryAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataRepositoryAssociationResponse> deleteDataRepositoryAssociation(
            DeleteDataRepositoryAssociationRequest deleteDataRepositoryAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataRepositoryAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteDataRepositoryAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataRepositoryAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataRepositoryAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDataRepositoryAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataRepositoryAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataRepositoryAssociationRequest, DeleteDataRepositoryAssociationResponse>()
                            .withOperationName("DeleteDataRepositoryAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDataRepositoryAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDataRepositoryAssociationRequest));
            CompletableFuture<DeleteDataRepositoryAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon File Cache resource. After deletion, the cache no longer exists, and its data is gone.
     * </p>
     * <p>
     * The <code>DeleteFileCache</code> operation returns while the cache has the <code>DELETING</code> status. You can
     * check the cache deletion status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileCaches.html">DescribeFileCaches</a>
     * operation, which returns a list of caches in your account. If you pass the cache ID for a deleted cache, the
     * <code>DescribeFileCaches</code> operation returns a <code>FileCacheNotFound</code> error.
     * </p>
     * <important>
     * <p>
     * The data in a deleted cache is also deleted and can't be recovered by any means.
     * </p>
     * </important>
     *
     * @param deleteFileCacheRequest
     * @return A Java Future containing the result of the DeleteFileCache operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>FileCacheNotFoundException No caches were found based upon supplied parameters.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DeleteFileCache
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteFileCache" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFileCacheResponse> deleteFileCache(DeleteFileCacheRequest deleteFileCacheRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFileCacheRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFileCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFileCache");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFileCacheResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFileCacheResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFileCacheResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFileCacheRequest, DeleteFileCacheResponse>()
                            .withOperationName("DeleteFileCache").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteFileCacheRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteFileCacheRequest));
            CompletableFuture<DeleteFileCacheResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a file system. After deletion, the file system no longer exists, and its data is gone. Any existing
     * automatic backups and snapshots are also deleted.
     * </p>
     * <p>
     * To delete an Amazon FSx for NetApp ONTAP file system, first delete all the volumes and storage virtual machines
     * (SVMs) on the file system. Then provide a <code>FileSystemId</code> value to the <code>DeleteFileSystem</code>
     * operation.
     * </p>
     * <p>
     * By default, when you delete an Amazon FSx for Windows File Server file system, a final backup is created upon
     * deletion. This final backup isn't subject to the file system's retention policy, and must be manually deleted.
     * </p>
     * <p>
     * To delete an Amazon FSx for Lustre file system, first <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/unmounting-fs.html">unmount</a> it from every connected
     * Amazon EC2 instance, then provide a <code>FileSystemId</code> value to the <code>DeleteFileSystem</code>
     * operation. By default, Amazon FSx will not take a final backup when the <code>DeleteFileSystem</code> operation
     * is invoked. On file systems not linked to an Amazon S3 bucket, set <code>SkipFinalBackup</code> to
     * <code>false</code> to take a final backup of the file system you are deleting. Backups cannot be enabled on
     * S3-linked file systems. To ensure all of your data is written back to S3 before deleting your file system, you
     * can either monitor for the <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/monitoring-cloudwatch.html#auto-import-export-metrics"
     * >AgeOfOldestQueuedMessage</a> metric to be zero (if using automatic export) or you can run an <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/export-data-repo-task-dra.html">export data repository
     * task</a>. If you have automatic export enabled and want to use an export data repository task, you have to
     * disable automatic export before executing the export data repository task.
     * </p>
     * <p>
     * The <code>DeleteFileSystem</code> operation returns while the file system has the <code>DELETING</code> status.
     * You can check the file system deletion status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystems.html">DescribeFileSystems</a>
     * operation, which returns a list of file systems in your account. If you pass the file system ID for a deleted
     * file system, the <code>DescribeFileSystems</code> operation returns a <code>FileSystemNotFound</code> error.
     * </p>
     * <note>
     * <p>
     * If a data repository task is in a <code>PENDING</code> or <code>EXECUTING</code> state, deleting an Amazon FSx
     * for Lustre file system will fail with an HTTP status code 400 (Bad Request).
     * </p>
     * </note> <important>
     * <p>
     * The data in a deleted file system is also deleted and can't be recovered by any means.
     * </p>
     * </important>
     *
     * @param deleteFileSystemRequest
     *        The request object for <code>DeleteFileSystem</code> operation.
     * @return A Java Future containing the result of the DeleteFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DeleteFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFileSystemResponse> deleteFileSystem(DeleteFileSystemRequest deleteFileSystemRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFileSystemRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFileSystem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFileSystemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFileSystemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFileSystemRequest, DeleteFileSystemResponse>()
                            .withOperationName("DeleteFileSystem").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteFileSystemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteFileSystemRequest));
            CompletableFuture<DeleteFileSystemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon FSx for OpenZFS snapshot. After deletion, the snapshot no longer exists, and its data is gone.
     * Deleting a snapshot doesn't affect snapshots stored in a file system backup.
     * </p>
     * <p>
     * The <code>DeleteSnapshot</code> operation returns instantly. The snapshot appears with the lifecycle status of
     * <code>DELETING</code> until the deletion is complete.
     * </p>
     *
     * @param deleteSnapshotRequest
     * @return A Java Future containing the result of the DeleteSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SnapshotNotFoundException No Amazon FSx snapshots were found based on the supplied parameters.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DeleteSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSnapshotResponse> deleteSnapshot(DeleteSnapshotRequest deleteSnapshotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSnapshotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSnapshotRequest, DeleteSnapshotResponse>()
                            .withOperationName("DeleteSnapshot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSnapshotRequest));
            CompletableFuture<DeleteSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing Amazon FSx for ONTAP storage virtual machine (SVM). Prior to deleting an SVM, you must delete
     * all non-root volumes in the SVM, otherwise the operation will fail.
     * </p>
     *
     * @param deleteStorageVirtualMachineRequest
     * @return A Java Future containing the result of the DeleteStorageVirtualMachine operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>StorageVirtualMachineNotFoundException No FSx for ONTAP SVMs were found based upon the supplied
     *         parameters.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DeleteStorageVirtualMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteStorageVirtualMachine"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStorageVirtualMachineResponse> deleteStorageVirtualMachine(
            DeleteStorageVirtualMachineRequest deleteStorageVirtualMachineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStorageVirtualMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStorageVirtualMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStorageVirtualMachine");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteStorageVirtualMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteStorageVirtualMachineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteStorageVirtualMachineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStorageVirtualMachineRequest, DeleteStorageVirtualMachineResponse>()
                            .withOperationName("DeleteStorageVirtualMachine").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteStorageVirtualMachineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteStorageVirtualMachineRequest));
            CompletableFuture<DeleteStorageVirtualMachineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volume.
     * </p>
     *
     * @param deleteVolumeRequest
     * @return A Java Future containing the result of the DeleteVolume operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>VolumeNotFoundException No Amazon FSx volumes were found based upon the supplied parameters.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DeleteVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteVolume" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVolumeResponse> deleteVolume(DeleteVolumeRequest deleteVolumeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVolumeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVolume");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteVolumeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVolumeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVolumeRequest, DeleteVolumeResponse>()
                            .withOperationName("DeleteVolume").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVolumeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVolumeRequest));
            CompletableFuture<DeleteVolumeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the description of a specific Amazon FSx backup, if a <code>BackupIds</code> value is provided for that
     * backup. Otherwise, it returns all backups owned by your Amazon Web Services account in the Amazon Web Services
     * Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all backups, you can optionally specify the <code>MaxResults</code> parameter to limit the number
     * of backups in a response. If more backups remain, Amazon FSx returns a <code>NextToken</code> value in the
     * response. In this case, send a later request with the <code>NextToken</code> request parameter set to the value
     * of the <code>NextToken</code> value from the last response.
     * </p>
     * <p>
     * This operation is used in an iterative process to retrieve a list of your backups. <code>DescribeBackups</code>
     * is called first without a <code>NextToken</code> value. Then the operation continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code> value.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The operation might return fewer than the <code>MaxResults</code> value of backup descriptions while still
     * including a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of the backups returned in the response of one <code>DescribeBackups</code> call and the order of the
     * backups returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeBackupsRequest
     *        The request object for the <code>DescribeBackups</code> operation.
     * @return A Java Future containing the result of the DescribeBackups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>VolumeNotFoundException No Amazon FSx volumes were found based upon the supplied parameters.</li>
     *         <li>BackupNotFoundException No Amazon FSx backups were found based upon the supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeBackups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeBackupsResponse> describeBackups(DescribeBackupsRequest describeBackupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBackupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBackupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBackupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBackupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBackupsRequest, DescribeBackupsResponse>()
                            .withOperationName("DescribeBackups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeBackupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeBackupsRequest));
            CompletableFuture<DescribeBackupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Lustre or Amazon File Cache data repository associations, if
     * one or more <code>AssociationIds</code> values are provided in the request, or if filters are used in the
     * request. Data repository associations are supported on Amazon File Cache resources and all FSx for Lustre 2.12
     * and 2,15 file systems, excluding <code>scratch_1</code> deployment type.
     * </p>
     * <p>
     * You can use filters to narrow the response to include just data repository associations for specific file systems
     * (use the <code>file-system-id</code> filter with the ID of the file system) or caches (use the
     * <code>file-cache-id</code> filter with the ID of the cache), or data repository associations for a specific
     * repository type (use the <code>data-repository-type</code> filter with a value of <code>S3</code> or
     * <code>NFS</code>). If you don't use filters, the response returns all data repository associations owned by your
     * Amazon Web Services account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all data repository associations, you can paginate the response by using the optional
     * <code>MaxResults</code> parameter to limit the number of data repository associations returned in a response. If
     * more data repository associations remain, a <code>NextToken</code> value is returned in the response. In this
     * case, send a later request with the <code>NextToken</code> request parameter set to the value of
     * <code>NextToken</code> from the last response.
     * </p>
     *
     * @param describeDataRepositoryAssociationsRequest
     * @return A Java Future containing the result of the DescribeDataRepositoryAssociations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>DataRepositoryAssociationNotFoundException No data repository associations were found based upon the
     *         supplied parameters.</li>
     *         <li>InvalidDataRepositoryTypeException You have filtered the response to a data repository type that is
     *         not supported.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeDataRepositoryAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeDataRepositoryAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDataRepositoryAssociationsResponse> describeDataRepositoryAssociations(
            DescribeDataRepositoryAssociationsRequest describeDataRepositoryAssociationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDataRepositoryAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeDataRepositoryAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataRepositoryAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDataRepositoryAssociationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeDataRepositoryAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDataRepositoryAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDataRepositoryAssociationsRequest, DescribeDataRepositoryAssociationsResponse>()
                            .withOperationName("DescribeDataRepositoryAssociations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDataRepositoryAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDataRepositoryAssociationsRequest));
            CompletableFuture<DescribeDataRepositoryAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Lustre or Amazon File Cache data repository tasks, if one or
     * more <code>TaskIds</code> values are provided in the request, or if filters are used in the request. You can use
     * filters to narrow the response to include just tasks for specific file systems or caches, or tasks in a specific
     * lifecycle state. Otherwise, it returns all data repository tasks owned by your Amazon Web Services account in the
     * Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all tasks, you can paginate the response by using the optional <code>MaxResults</code> parameter
     * to limit the number of tasks returned in a response. If more tasks remain, a <code>NextToken</code> value is
     * returned in the response. In this case, send a later request with the <code>NextToken</code> request parameter
     * set to the value of <code>NextToken</code> from the last response.
     * </p>
     *
     * @param describeDataRepositoryTasksRequest
     * @return A Java Future containing the result of the DescribeDataRepositoryTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>DataRepositoryTaskNotFoundException The data repository task or tasks you specified could not be
     *         found.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeDataRepositoryTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeDataRepositoryTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDataRepositoryTasksResponse> describeDataRepositoryTasks(
            DescribeDataRepositoryTasksRequest describeDataRepositoryTasksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDataRepositoryTasksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDataRepositoryTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataRepositoryTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDataRepositoryTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDataRepositoryTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDataRepositoryTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDataRepositoryTasksRequest, DescribeDataRepositoryTasksResponse>()
                            .withOperationName("DescribeDataRepositoryTasks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDataRepositoryTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDataRepositoryTasksRequest));
            CompletableFuture<DescribeDataRepositoryTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the description of a specific Amazon File Cache resource, if a <code>FileCacheIds</code> value is
     * provided for that cache. Otherwise, it returns descriptions of all caches owned by your Amazon Web Services
     * account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all cache descriptions, you can optionally specify the <code>MaxResults</code> parameter to limit
     * the number of descriptions in a response. If more cache descriptions remain, the operation returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This operation is used in an iterative process to retrieve a list of your cache descriptions.
     * <code>DescribeFileCaches</code> is called first without a <code>NextToken</code>value. Then the operation
     * continues to be called with the <code>NextToken</code> parameter set to the value of the last
     * <code>NextToken</code> value until a response has no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> cache descriptions while still including a
     * <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of caches returned in the response of one <code>DescribeFileCaches</code> call and the order of caches
     * returned across the responses of a multicall iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeFileCachesRequest
     * @return A Java Future containing the result of the DescribeFileCaches operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileCacheNotFoundException No caches were found based upon supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeFileCaches
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileCaches" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFileCachesResponse> describeFileCaches(DescribeFileCachesRequest describeFileCachesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFileCachesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFileCachesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFileCaches");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFileCachesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFileCachesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFileCachesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFileCachesRequest, DescribeFileCachesResponse>()
                            .withOperationName("DescribeFileCaches").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeFileCachesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeFileCachesRequest));
            CompletableFuture<DescribeFileCachesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the DNS aliases that are associated with the specified Amazon FSx for Windows File Server file system. A
     * history of all DNS aliases that have been associated with and disassociated from the file system is available in
     * the list of <a>AdministrativeAction</a> provided in the <a>DescribeFileSystems</a> operation response.
     * </p>
     *
     * @param describeFileSystemAliasesRequest
     *        The request object for <code>DescribeFileSystemAliases</code> operation.
     * @return A Java Future containing the result of the DescribeFileSystemAliases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeFileSystemAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileSystemAliases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFileSystemAliasesResponse> describeFileSystemAliases(
            DescribeFileSystemAliasesRequest describeFileSystemAliasesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFileSystemAliasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFileSystemAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFileSystemAliases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFileSystemAliasesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFileSystemAliasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFileSystemAliasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFileSystemAliasesRequest, DescribeFileSystemAliasesResponse>()
                            .withOperationName("DescribeFileSystemAliases").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeFileSystemAliasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeFileSystemAliasesRequest));
            CompletableFuture<DescribeFileSystemAliasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx file systems, if a <code>FileSystemIds</code> value is provided
     * for that file system. Otherwise, it returns descriptions of all file systems owned by your Amazon Web Services
     * account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxResults</code> parameter to
     * limit the number of descriptions in a response. If more file system descriptions remain, Amazon FSx returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This operation is used in an iterative process to retrieve a list of your file system descriptions.
     * <code>DescribeFileSystems</code> is called first without a <code>NextToken</code>value. Then the operation
     * continues to be called with the <code>NextToken</code> parameter set to the value of the last
     * <code>NextToken</code> value until a response has no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multicall iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeFileSystemsRequest
     *        The request object for <code>DescribeFileSystems</code> operation.
     * @return A Java Future containing the result of the DescribeFileSystems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileSystems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFileSystemsResponse> describeFileSystems(
            DescribeFileSystemsRequest describeFileSystemsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFileSystemsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFileSystemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFileSystems");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFileSystemsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFileSystemsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFileSystemsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFileSystemsRequest, DescribeFileSystemsResponse>()
                            .withOperationName("DescribeFileSystems").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeFileSystemsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeFileSystemsRequest));
            CompletableFuture<DescribeFileSystemsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Indicates whether participant accounts in your organization can create Amazon FSx for NetApp ONTAP Multi-AZ file
     * systems in subnets that are shared by a virtual private cloud (VPC) owner. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/creating-file-systems.html#fsxn-vpc-shared-subnets"
     * >Creating FSx for ONTAP file systems in shared subnets</a>.
     * </p>
     *
     * @param describeSharedVpcConfigurationRequest
     * @return A Java Future containing the result of the DescribeSharedVpcConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeSharedVpcConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeSharedVpcConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSharedVpcConfigurationResponse> describeSharedVpcConfiguration(
            DescribeSharedVpcConfigurationRequest describeSharedVpcConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSharedVpcConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeSharedVpcConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSharedVpcConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSharedVpcConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSharedVpcConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSharedVpcConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSharedVpcConfigurationRequest, DescribeSharedVpcConfigurationResponse>()
                            .withOperationName("DescribeSharedVpcConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeSharedVpcConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeSharedVpcConfigurationRequest));
            CompletableFuture<DescribeSharedVpcConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for OpenZFS snapshots, if a <code>SnapshotIds</code> value is
     * provided. Otherwise, this operation returns all snapshots owned by your Amazon Web Services account in the Amazon
     * Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all snapshots, you can optionally specify the <code>MaxResults</code> parameter to limit the
     * number of snapshots in a response. If more backups remain, Amazon FSx returns a <code>NextToken</code> value in
     * the response. In this case, send a later request with the <code>NextToken</code> request parameter set to the
     * value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * Use this operation in an iterative process to retrieve a list of your snapshots. <code>DescribeSnapshots</code>
     * is called first without a <code>NextToken</code> value. Then the operation continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code> value.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The operation might return fewer than the <code>MaxResults</code> value of snapshot descriptions while still
     * including a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of snapshots returned in the response of one <code>DescribeSnapshots</code> call and the order of
     * backups returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeSnapshotsRequest
     * @return A Java Future containing the result of the DescribeSnapshots operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SnapshotNotFoundException No Amazon FSx snapshots were found based on the supplied parameters.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeSnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeSnapshots" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSnapshotsResponse> describeSnapshots(DescribeSnapshotsRequest describeSnapshotsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSnapshotsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSnapshotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSnapshots");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSnapshotsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSnapshotsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSnapshotsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSnapshotsRequest, DescribeSnapshotsResponse>()
                            .withOperationName("DescribeSnapshots").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeSnapshotsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeSnapshotsRequest));
            CompletableFuture<DescribeSnapshotsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes one or more Amazon FSx for NetApp ONTAP storage virtual machines (SVMs).
     * </p>
     *
     * @param describeStorageVirtualMachinesRequest
     * @return A Java Future containing the result of the DescribeStorageVirtualMachines operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>StorageVirtualMachineNotFoundException No FSx for ONTAP SVMs were found based upon the supplied
     *         parameters.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeStorageVirtualMachines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeStorageVirtualMachines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStorageVirtualMachinesResponse> describeStorageVirtualMachines(
            DescribeStorageVirtualMachinesRequest describeStorageVirtualMachinesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStorageVirtualMachinesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeStorageVirtualMachinesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStorageVirtualMachines");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeStorageVirtualMachinesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeStorageVirtualMachinesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeStorageVirtualMachinesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeStorageVirtualMachinesRequest, DescribeStorageVirtualMachinesResponse>()
                            .withOperationName("DescribeStorageVirtualMachines").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeStorageVirtualMachinesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeStorageVirtualMachinesRequest));
            CompletableFuture<DescribeStorageVirtualMachinesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes one or more Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volumes.
     * </p>
     *
     * @param describeVolumesRequest
     * @return A Java Future containing the result of the DescribeVolumes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>VolumeNotFoundException No Amazon FSx volumes were found based upon the supplied parameters.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DescribeVolumes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeVolumes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVolumesResponse> describeVolumes(DescribeVolumesRequest describeVolumesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeVolumesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVolumesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVolumes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVolumesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVolumesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVolumesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVolumesRequest, DescribeVolumesResponse>()
                            .withOperationName("DescribeVolumes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeVolumesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeVolumesRequest));
            CompletableFuture<DescribeVolumesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this action to disassociate, or remove, one or more Domain Name Service (DNS) aliases from an Amazon FSx for
     * Windows File Server file system. If you attempt to disassociate a DNS alias that is not associated with the file
     * system, Amazon FSx responds with an HTTP status code 400 (Bad Request). For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS
     * Aliases</a>.
     * </p>
     * <p>
     * The system generated response showing the DNS aliases that Amazon FSx is attempting to disassociate from the file
     * system. Use the API operation to monitor the status of the aliases Amazon FSx is disassociating with the file
     * system.
     * </p>
     *
     * @param disassociateFileSystemAliasesRequest
     *        The request object of DNS aliases to disassociate from an Amazon FSx for Windows File Server file system.
     * @return A Java Future containing the result of the DisassociateFileSystemAliases operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.DisassociateFileSystemAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DisassociateFileSystemAliases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateFileSystemAliasesResponse> disassociateFileSystemAliases(
            DisassociateFileSystemAliasesRequest disassociateFileSystemAliasesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateFileSystemAliasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateFileSystemAliasesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateFileSystemAliases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateFileSystemAliasesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateFileSystemAliasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateFileSystemAliasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateFileSystemAliasesRequest, DisassociateFileSystemAliasesResponse>()
                            .withOperationName("DisassociateFileSystemAliases").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateFileSystemAliasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateFileSystemAliasesRequest));
            CompletableFuture<DisassociateFileSystemAliasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists tags for Amazon FSx resources.
     * </p>
     * <p>
     * When retrieving all tags, you can optionally specify the <code>MaxResults</code> parameter to limit the number of
     * tags in a response. If more tags remain, Amazon FSx returns a <code>NextToken</code> value in the response. In
     * this case, send a later request with the <code>NextToken</code> request parameter set to the value of
     * <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This action is used in an iterative process to retrieve a list of your tags. <code>ListTagsForResource</code> is
     * called first without a <code>NextToken</code>value. Then the action continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this action, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of tags returned in the response of one <code>ListTagsForResource</code> call and the order of tags
     * returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param listTagsForResourceRequest
     *        The request object for <code>ListTagsForResource</code> operation.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>ResourceNotFoundException The resource specified by the Amazon Resource Name (ARN) can't be found.</li>
     *         <li>NotServiceResourceErrorException The resource specified for the tagging operation is not a resource
     *         type owned by Amazon FSx. Use the API of the relevant service to perform the operation.</li>
     *         <li>ResourceDoesNotSupportTaggingException The resource specified does not support tagging.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Releases the file system lock from an Amazon FSx for OpenZFS file system.
     * </p>
     *
     * @param releaseFileSystemNfsV3LocksRequest
     * @return A Java Future containing the result of the ReleaseFileSystemNfsV3Locks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.ReleaseFileSystemNfsV3Locks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/ReleaseFileSystemNfsV3Locks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ReleaseFileSystemNfsV3LocksResponse> releaseFileSystemNfsV3Locks(
            ReleaseFileSystemNfsV3LocksRequest releaseFileSystemNfsV3LocksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(releaseFileSystemNfsV3LocksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, releaseFileSystemNfsV3LocksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReleaseFileSystemNfsV3Locks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReleaseFileSystemNfsV3LocksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ReleaseFileSystemNfsV3LocksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReleaseFileSystemNfsV3LocksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReleaseFileSystemNfsV3LocksRequest, ReleaseFileSystemNfsV3LocksResponse>()
                            .withOperationName("ReleaseFileSystemNfsV3Locks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ReleaseFileSystemNfsV3LocksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(releaseFileSystemNfsV3LocksRequest));
            CompletableFuture<ReleaseFileSystemNfsV3LocksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an Amazon FSx for OpenZFS volume to the state saved by the specified snapshot.
     * </p>
     *
     * @param restoreVolumeFromSnapshotRequest
     * @return A Java Future containing the result of the RestoreVolumeFromSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>VolumeNotFoundException No Amazon FSx volumes were found based upon the supplied parameters.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.RestoreVolumeFromSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/RestoreVolumeFromSnapshot" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreVolumeFromSnapshotResponse> restoreVolumeFromSnapshot(
            RestoreVolumeFromSnapshotRequest restoreVolumeFromSnapshotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreVolumeFromSnapshotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreVolumeFromSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreVolumeFromSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreVolumeFromSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RestoreVolumeFromSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreVolumeFromSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreVolumeFromSnapshotRequest, RestoreVolumeFromSnapshotResponse>()
                            .withOperationName("RestoreVolumeFromSnapshot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RestoreVolumeFromSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(restoreVolumeFromSnapshotRequest));
            CompletableFuture<RestoreVolumeFromSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * After performing steps to repair the Active Directory configuration of an FSx for Windows File Server file
     * system, use this action to initiate the process of Amazon FSx attempting to reconnect to the file system.
     * </p>
     *
     * @param startMisconfiguredStateRecoveryRequest
     * @return A Java Future containing the result of the StartMisconfiguredStateRecovery operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.StartMisconfiguredStateRecovery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/StartMisconfiguredStateRecovery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartMisconfiguredStateRecoveryResponse> startMisconfiguredStateRecovery(
            StartMisconfiguredStateRecoveryRequest startMisconfiguredStateRecoveryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMisconfiguredStateRecoveryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startMisconfiguredStateRecoveryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMisconfiguredStateRecovery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMisconfiguredStateRecoveryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMisconfiguredStateRecoveryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMisconfiguredStateRecoveryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMisconfiguredStateRecoveryRequest, StartMisconfiguredStateRecoveryResponse>()
                            .withOperationName("StartMisconfiguredStateRecovery").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartMisconfiguredStateRecoveryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startMisconfiguredStateRecoveryRequest));
            CompletableFuture<StartMisconfiguredStateRecoveryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags an Amazon FSx resource.
     * </p>
     *
     * @param tagResourceRequest
     *        The request object for the <code>TagResource</code> operation.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>ResourceNotFoundException The resource specified by the Amazon Resource Name (ARN) can't be found.</li>
     *         <li>NotServiceResourceErrorException The resource specified for the tagging operation is not a resource
     *         type owned by Amazon FSx. Use the API of the relevant service to perform the operation.</li>
     *         <li>ResourceDoesNotSupportTaggingException The resource specified does not support tagging.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This action removes a tag from an Amazon FSx resource.
     * </p>
     *
     * @param untagResourceRequest
     *        The request object for <code>UntagResource</code> action.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>ResourceNotFoundException The resource specified by the Amazon Resource Name (ARN) can't be found.</li>
     *         <li>NotServiceResourceErrorException The resource specified for the tagging operation is not a resource
     *         type owned by Amazon FSx. Use the API of the relevant service to perform the operation.</li>
     *         <li>ResourceDoesNotSupportTaggingException The resource specified does not support tagging.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration of an existing data repository association on an Amazon FSx for Lustre file system.
     * Data repository associations are supported on all FSx for Lustre 2.12 and 2.15 file systems, excluding
     * <code>scratch_1</code> deployment type.
     * </p>
     *
     * @param updateDataRepositoryAssociationRequest
     * @return A Java Future containing the result of the UpdateDataRepositoryAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>DataRepositoryAssociationNotFoundException No data repository associations were found based upon the
     *         supplied parameters.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.UpdateDataRepositoryAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateDataRepositoryAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataRepositoryAssociationResponse> updateDataRepositoryAssociation(
            UpdateDataRepositoryAssociationRequest updateDataRepositoryAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDataRepositoryAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateDataRepositoryAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataRepositoryAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataRepositoryAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDataRepositoryAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataRepositoryAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataRepositoryAssociationRequest, UpdateDataRepositoryAssociationResponse>()
                            .withOperationName("UpdateDataRepositoryAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDataRepositoryAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDataRepositoryAssociationRequest));
            CompletableFuture<UpdateDataRepositoryAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration of an existing Amazon File Cache resource. You can update multiple properties in a
     * single request.
     * </p>
     *
     * @param updateFileCacheRequest
     * @return A Java Future containing the result of the UpdateFileCache operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>FileCacheNotFoundException No caches were found based upon supplied parameters.</li>
     *         <li>MissingFileCacheConfigurationException A cache configuration is required for this operation.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.UpdateFileCache
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateFileCache" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFileCacheResponse> updateFileCache(UpdateFileCacheRequest updateFileCacheRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateFileCacheRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFileCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFileCache");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFileCacheResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFileCacheResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFileCacheResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFileCacheRequest, UpdateFileCacheResponse>()
                            .withOperationName("UpdateFileCache").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateFileCacheRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateFileCacheRequest));
            CompletableFuture<UpdateFileCacheResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to update the configuration of an existing Amazon FSx file system. You can update multiple
     * properties in a single request.
     * </p>
     * <p>
     * For FSx for Windows File Server file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AuditLogConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SelfManagedActiveDirectoryConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageType</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ThroughputCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DiskIopsConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For FSx for Lustre file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AutoImportPolicy</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DataCompressionType</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LogConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LustreRootSquashConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MetadataConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PerUnitStorageThroughput</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For FSx for ONTAP file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AddRouteTableIds</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DiskIopsConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FsxAdminPassword</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HAPairs</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RemoveRouteTableIds</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ThroughputCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ThroughputCapacityPerHAPair</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For FSx for OpenZFS file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AddRouteTableIds</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CopyTagsToBackups</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CopyTagsToVolumes</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DiskIopsConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadCacheConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RemoveRouteTableIds</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ThroughputCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateFileSystemRequest
     *        The request object for the <code>UpdateFileSystem</code> operation.
     * @return A Java Future containing the result of the UpdateFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InvalidNetworkSettingsException One or more network settings specified in the request are invalid.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>FileSystemNotFoundException No Amazon FSx file systems were found based upon supplied parameters.</li>
     *         <li>MissingFileSystemConfigurationException A file system configuration is required for this operation.</li>
     *         <li>ServiceLimitExceededException An error indicating that a particular service limit was exceeded. You
     *         can increase some service limits by contacting Amazon Web Services Support.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.UpdateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFileSystemResponse> updateFileSystem(UpdateFileSystemRequest updateFileSystemRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateFileSystemRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFileSystem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFileSystemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFileSystemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFileSystemRequest, UpdateFileSystemResponse>()
                            .withOperationName("UpdateFileSystem").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateFileSystemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateFileSystemRequest));
            CompletableFuture<UpdateFileSystemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures whether participant accounts in your organization can create Amazon FSx for NetApp ONTAP Multi-AZ file
     * systems in subnets that are shared by a virtual private cloud (VPC) owner. For more information, see the <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/maz-shared-vpc.html">Amazon FSx for NetApp ONTAP User
     * Guide</a>.
     * </p>
     * <note>
     * <p>
     * We strongly recommend that participant-created Multi-AZ file systems in the shared VPC are deleted before you
     * disable this feature. Once the feature is disabled, these file systems will enter a <code>MISCONFIGURED</code>
     * state and behave like Single-AZ file systems. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/maz-shared-vpc.html#disabling-maz-vpc-sharing">Important
     * considerations before disabling shared VPC support for Multi-AZ file systems</a>.
     * </p>
     * </note>
     *
     * @param updateSharedVpcConfigurationRequest
     * @return A Java Future containing the result of the UpdateSharedVpcConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.UpdateSharedVpcConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateSharedVpcConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSharedVpcConfigurationResponse> updateSharedVpcConfiguration(
            UpdateSharedVpcConfigurationRequest updateSharedVpcConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSharedVpcConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSharedVpcConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSharedVpcConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSharedVpcConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSharedVpcConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSharedVpcConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSharedVpcConfigurationRequest, UpdateSharedVpcConfigurationResponse>()
                            .withOperationName("UpdateSharedVpcConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSharedVpcConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSharedVpcConfigurationRequest));
            CompletableFuture<UpdateSharedVpcConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the name of an Amazon FSx for OpenZFS snapshot.
     * </p>
     *
     * @param updateSnapshotRequest
     * @return A Java Future containing the result of the UpdateSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>SnapshotNotFoundException No Amazon FSx snapshots were found based on the supplied parameters.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.UpdateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSnapshotResponse> updateSnapshot(UpdateSnapshotRequest updateSnapshotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSnapshotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSnapshotRequest, UpdateSnapshotResponse>()
                            .withOperationName("UpdateSnapshot").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSnapshotRequest));
            CompletableFuture<UpdateSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an FSx for ONTAP storage virtual machine (SVM).
     * </p>
     *
     * @param updateStorageVirtualMachineRequest
     * @return A Java Future containing the result of the UpdateStorageVirtualMachine operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>StorageVirtualMachineNotFoundException No FSx for ONTAP SVMs were found based upon the supplied
     *         parameters.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for this resource or API.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.UpdateStorageVirtualMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateStorageVirtualMachine"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateStorageVirtualMachineResponse> updateStorageVirtualMachine(
            UpdateStorageVirtualMachineRequest updateStorageVirtualMachineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateStorageVirtualMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStorageVirtualMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStorageVirtualMachine");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateStorageVirtualMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateStorageVirtualMachineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateStorageVirtualMachineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateStorageVirtualMachineRequest, UpdateStorageVirtualMachineResponse>()
                            .withOperationName("UpdateStorageVirtualMachine").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateStorageVirtualMachineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateStorageVirtualMachineRequest));
            CompletableFuture<UpdateStorageVirtualMachineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration of an Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volume.
     * </p>
     *
     * @param updateVolumeRequest
     * @return A Java Future containing the result of the UpdateVolume operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A generic error indicating a failure with a client request.</li>
     *         <li>IncompatibleParameterErrorException The error returned when a second request is received with the
     *         same client request token but different parameters settings. A client request token should always
     *         uniquely identify a single request.</li>
     *         <li>InternalServerErrorException A generic error indicating a server-side failure.</li>
     *         <li>MissingVolumeConfigurationException A volume configuration is required for this operation.</li>
     *         <li>VolumeNotFoundException No Amazon FSx volumes were found based upon the supplied parameters.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FSxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample FSxAsyncClient.UpdateVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateVolume" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVolumeResponse> updateVolume(UpdateVolumeRequest updateVolumeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateVolumeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVolume");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateVolumeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVolumeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVolumeRequest, UpdateVolumeResponse>()
                            .withOperationName("UpdateVolume").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateVolumeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateVolumeRequest));
            CompletableFuture<UpdateVolumeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final FSxServiceClientConfiguration serviceClientConfiguration() {
        return new FSxServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FSxException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidImportPath")
                                .exceptionBuilderSupplier(InvalidImportPathException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNetworkSettings")
                                .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileSystemNotFound")
                                .exceptionBuilderSupplier(FileSystemNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound")
                                .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging")
                                .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound")
                                .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SourceBackupUnavailable")
                                .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperation")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound")
                                .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupRestoring")
                                .exceptionBuilderSupplier(BackupRestoringException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupInProgress")
                                .exceptionBuilderSupplier(BackupInProgressException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidExportPath")
                                .exceptionBuilderSupplier(InvalidExportPathException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRegion")
                                .exceptionBuilderSupplier(InvalidRegionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration")
                                .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey")
                                .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting")
                                .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ")
                                .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration")
                                .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupNotFound")
                                .exceptionBuilderSupplier(BackupNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileCacheNotFound")
                                .exceptionBuilderSupplier(FileCacheNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey")
                                .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("VolumeNotFound")
                                .exceptionBuilderSupplier(VolumeNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLimitExceeded")
                                .exceptionBuilderSupplier(ServiceLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotServiceResourceError")
                                .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration")
                                .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupBeingCopied")
                                .exceptionBuilderSupplier(BackupBeingCopiedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncompatibleParameterError")
                                .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded")
                                .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType")
                                .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SnapshotNotFound")
                                .exceptionBuilderSupplier(SnapshotNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput")
                                .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActiveDirectoryError")
                                .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequest")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        FSxServiceClientConfigurationBuilder serviceConfigBuilder = new FSxServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
