/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration of a data repository association that links an Amazon FSx for Lustre file system to an Amazon S3
 * bucket or an Amazon File Cache resource to an Amazon S3 bucket or an NFS file system. The data repository association
 * configuration object is returned in the response of the following operations:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>CreateDataRepositoryAssociation</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>UpdateDataRepositoryAssociation</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>DescribeDataRepositoryAssociations</code>
 * </p>
 * </li>
 * </ul>
 * <p>
 * Data repository associations are supported on Amazon File Cache resources and all FSx for Lustre 2.12 and 2.15 file
 * systems, excluding <code>scratch_1</code> deployment type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataRepositoryAssociation implements SdkPojo, Serializable,
        ToCopyableBuilder<DataRepositoryAssociation.Builder, DataRepositoryAssociation> {
    private static final SdkField<String> ASSOCIATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociationId").getter(getter(DataRepositoryAssociation::associationId))
            .setter(setter(Builder::associationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationId").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceARN").getter(getter(DataRepositoryAssociation::resourceARN))
            .setter(setter(Builder::resourceARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceARN").build()).build();

    private static final SdkField<String> FILE_SYSTEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemId").getter(getter(DataRepositoryAssociation::fileSystemId))
            .setter(setter(Builder::fileSystemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemId").build()).build();

    private static final SdkField<String> LIFECYCLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Lifecycle").getter(getter(DataRepositoryAssociation::lifecycleAsString))
            .setter(setter(Builder::lifecycle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<DataRepositoryFailureDetails> FAILURE_DETAILS_FIELD = SdkField
            .<DataRepositoryFailureDetails> builder(MarshallingType.SDK_POJO).memberName("FailureDetails")
            .getter(getter(DataRepositoryAssociation::failureDetails)).setter(setter(Builder::failureDetails))
            .constructor(DataRepositoryFailureDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureDetails").build()).build();

    private static final SdkField<String> FILE_SYSTEM_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemPath").getter(getter(DataRepositoryAssociation::fileSystemPath))
            .setter(setter(Builder::fileSystemPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemPath").build()).build();

    private static final SdkField<String> DATA_REPOSITORY_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataRepositoryPath").getter(getter(DataRepositoryAssociation::dataRepositoryPath))
            .setter(setter(Builder::dataRepositoryPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataRepositoryPath").build())
            .build();

    private static final SdkField<Boolean> BATCH_IMPORT_META_DATA_ON_CREATE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("BatchImportMetaDataOnCreate")
            .getter(getter(DataRepositoryAssociation::batchImportMetaDataOnCreate))
            .setter(setter(Builder::batchImportMetaDataOnCreate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BatchImportMetaDataOnCreate")
                    .build()).build();

    private static final SdkField<Integer> IMPORTED_FILE_CHUNK_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ImportedFileChunkSize").getter(getter(DataRepositoryAssociation::importedFileChunkSize))
            .setter(setter(Builder::importedFileChunkSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImportedFileChunkSize").build())
            .build();

    private static final SdkField<S3DataRepositoryConfiguration> S3_FIELD = SdkField
            .<S3DataRepositoryConfiguration> builder(MarshallingType.SDK_POJO).memberName("S3")
            .getter(getter(DataRepositoryAssociation::s3)).setter(setter(Builder::s3))
            .constructor(S3DataRepositoryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(DataRepositoryAssociation::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(DataRepositoryAssociation::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> FILE_CACHE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileCacheId").getter(getter(DataRepositoryAssociation::fileCacheId))
            .setter(setter(Builder::fileCacheId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileCacheId").build()).build();

    private static final SdkField<String> FILE_CACHE_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileCachePath").getter(getter(DataRepositoryAssociation::fileCachePath))
            .setter(setter(Builder::fileCachePath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileCachePath").build()).build();

    private static final SdkField<List<String>> DATA_REPOSITORY_SUBDIRECTORIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DataRepositorySubdirectories")
            .getter(getter(DataRepositoryAssociation::dataRepositorySubdirectories))
            .setter(setter(Builder::dataRepositorySubdirectories))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataRepositorySubdirectories")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<NFSDataRepositoryConfiguration> NFS_FIELD = SdkField
            .<NFSDataRepositoryConfiguration> builder(MarshallingType.SDK_POJO).memberName("NFS")
            .getter(getter(DataRepositoryAssociation::nfs)).setter(setter(Builder::nfs))
            .constructor(NFSDataRepositoryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NFS").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASSOCIATION_ID_FIELD,
            RESOURCE_ARN_FIELD, FILE_SYSTEM_ID_FIELD, LIFECYCLE_FIELD, FAILURE_DETAILS_FIELD, FILE_SYSTEM_PATH_FIELD,
            DATA_REPOSITORY_PATH_FIELD, BATCH_IMPORT_META_DATA_ON_CREATE_FIELD, IMPORTED_FILE_CHUNK_SIZE_FIELD, S3_FIELD,
            TAGS_FIELD, CREATION_TIME_FIELD, FILE_CACHE_ID_FIELD, FILE_CACHE_PATH_FIELD, DATA_REPOSITORY_SUBDIRECTORIES_FIELD,
            NFS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String associationId;

    private final String resourceARN;

    private final String fileSystemId;

    private final String lifecycle;

    private final DataRepositoryFailureDetails failureDetails;

    private final String fileSystemPath;

    private final String dataRepositoryPath;

    private final Boolean batchImportMetaDataOnCreate;

    private final Integer importedFileChunkSize;

    private final S3DataRepositoryConfiguration s3;

    private final List<Tag> tags;

    private final Instant creationTime;

    private final String fileCacheId;

    private final String fileCachePath;

    private final List<String> dataRepositorySubdirectories;

    private final NFSDataRepositoryConfiguration nfs;

    private DataRepositoryAssociation(BuilderImpl builder) {
        this.associationId = builder.associationId;
        this.resourceARN = builder.resourceARN;
        this.fileSystemId = builder.fileSystemId;
        this.lifecycle = builder.lifecycle;
        this.failureDetails = builder.failureDetails;
        this.fileSystemPath = builder.fileSystemPath;
        this.dataRepositoryPath = builder.dataRepositoryPath;
        this.batchImportMetaDataOnCreate = builder.batchImportMetaDataOnCreate;
        this.importedFileChunkSize = builder.importedFileChunkSize;
        this.s3 = builder.s3;
        this.tags = builder.tags;
        this.creationTime = builder.creationTime;
        this.fileCacheId = builder.fileCacheId;
        this.fileCachePath = builder.fileCachePath;
        this.dataRepositorySubdirectories = builder.dataRepositorySubdirectories;
        this.nfs = builder.nfs;
    }

    /**
     * <p>
     * The system-generated, unique ID of the data repository association.
     * </p>
     * 
     * @return The system-generated, unique ID of the data repository association.
     */
    public final String associationId() {
        return associationId;
    }

    /**
     * Returns the value of the ResourceARN property for this object.
     * 
     * @return The value of the ResourceARN property for this object.
     */
    public final String resourceARN() {
        return resourceARN;
    }

    /**
     * Returns the value of the FileSystemId property for this object.
     * 
     * @return The value of the FileSystemId property for this object.
     */
    public final String fileSystemId() {
        return fileSystemId;
    }

    /**
     * <p>
     * Describes the state of a data repository association. The lifecycle can have the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The data repository association between the file system or cache and the data repository
     * is being created. The data repository is unavailable.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The data repository association is available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> - The data repository association is misconfigured. Until the configuration is
     * corrected, automatic import and automatic export will not work (only for Amazon FSx for Lustre).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that might
     * affect its availability.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link DataRepositoryLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return Describes the state of a data repository association. The lifecycle can have the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The data repository association between the file system or cache and the data
     *         repository is being created. The data repository is unavailable.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The data repository association is available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> - The data repository association is misconfigured. Until the configuration is
     *         corrected, automatic import and automatic export will not work (only for Amazon FSx for Lustre).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that
     *         might affect its availability.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
     *         </p>
     *         </li>
     * @see DataRepositoryLifecycle
     */
    public final DataRepositoryLifecycle lifecycle() {
        return DataRepositoryLifecycle.fromValue(lifecycle);
    }

    /**
     * <p>
     * Describes the state of a data repository association. The lifecycle can have the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The data repository association between the file system or cache and the data repository
     * is being created. The data repository is unavailable.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The data repository association is available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> - The data repository association is misconfigured. Until the configuration is
     * corrected, automatic import and automatic export will not work (only for Amazon FSx for Lustre).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that might
     * affect its availability.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link DataRepositoryLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return Describes the state of a data repository association. The lifecycle can have the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The data repository association between the file system or cache and the data
     *         repository is being created. The data repository is unavailable.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The data repository association is available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> - The data repository association is misconfigured. Until the configuration is
     *         corrected, automatic import and automatic export will not work (only for Amazon FSx for Lustre).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that
     *         might affect its availability.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
     *         </p>
     *         </li>
     * @see DataRepositoryLifecycle
     */
    public final String lifecycleAsString() {
        return lifecycle;
    }

    /**
     * Returns the value of the FailureDetails property for this object.
     * 
     * @return The value of the FailureDetails property for this object.
     */
    public final DataRepositoryFailureDetails failureDetails() {
        return failureDetails;
    }

    /**
     * <p>
     * A path on the Amazon FSx for Lustre file system that points to a high-level directory (such as <code>/ns1/</code>
     * ) or subdirectory (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with
     * <code>DataRepositoryPath</code>. The leading forward slash in the name is required. Two data repository
     * associations cannot have overlapping file system paths. For example, if a data repository is associated with file
     * system path <code>/ns1/</code>, then you cannot link another data repository with file system path
     * <code>/ns1/ns2</code>.
     * </p>
     * <p>
     * This path specifies where in your file system files will be exported from or imported to. This file system
     * directory can be linked to only one Amazon S3 bucket, and no other S3 bucket can be linked to the directory.
     * </p>
     * <note>
     * <p>
     * If you specify only a forward slash (<code>/</code>) as the file system path, you can link only one data
     * repository to the file system. You can only specify "/" as the file system path for the first data repository
     * associated with a file system.
     * </p>
     * </note>
     * 
     * @return A path on the Amazon FSx for Lustre file system that points to a high-level directory (such as
     *         <code>/ns1/</code>) or subdirectory (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with
     *         <code>DataRepositoryPath</code>. The leading forward slash in the name is required. Two data repository
     *         associations cannot have overlapping file system paths. For example, if a data repository is associated
     *         with file system path <code>/ns1/</code>, then you cannot link another data repository with file system
     *         path <code>/ns1/ns2</code>.</p>
     *         <p>
     *         This path specifies where in your file system files will be exported from or imported to. This file
     *         system directory can be linked to only one Amazon S3 bucket, and no other S3 bucket can be linked to the
     *         directory.
     *         </p>
     *         <note>
     *         <p>
     *         If you specify only a forward slash (<code>/</code>) as the file system path, you can link only one data
     *         repository to the file system. You can only specify "/" as the file system path for the first data
     *         repository associated with a file system.
     *         </p>
     */
    public final String fileSystemPath() {
        return fileSystemPath;
    }

    /**
     * <p>
     * The path to the data repository that will be linked to the cache or file system.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For Amazon File Cache, the path can be an NFS data repository that will be linked to the cache. The path can be
     * in one of two formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you are not using the <code>DataRepositorySubdirectories</code> parameter, the path is to an NFS Export
     * directory (or one of its subdirectories) in the format <code>nsf://nfs-domain-name/exportpath</code>. You can
     * therefore link a single NFS Export to a single data repository association.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you are using the <code>DataRepositorySubdirectories</code> parameter, the path is the domain name of the NFS
     * file system in the format <code>nfs://filer-domain-name</code>, which indicates the root of the subdirectories
     * specified with the <code>DataRepositorySubdirectories</code> parameter.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For Amazon File Cache, the path can be an S3 bucket or prefix in the format <code>s3://bucket-name/prefix/</code>
     * (where <code>prefix</code> is optional).
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon FSx for Lustre, the path can be an S3 bucket or prefix in the format
     * <code>s3://bucket-name/prefix/</code> (where <code>prefix</code> is optional).
     * </p>
     * </li>
     * </ul>
     * 
     * @return The path to the data repository that will be linked to the cache or file system.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For Amazon File Cache, the path can be an NFS data repository that will be linked to the cache. The path
     *         can be in one of two formats:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you are not using the <code>DataRepositorySubdirectories</code> parameter, the path is to an NFS
     *         Export directory (or one of its subdirectories) in the format
     *         <code>nsf://nfs-domain-name/exportpath</code>. You can therefore link a single NFS Export to a single
     *         data repository association.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you are using the <code>DataRepositorySubdirectories</code> parameter, the path is the domain name of
     *         the NFS file system in the format <code>nfs://filer-domain-name</code>, which indicates the root of the
     *         subdirectories specified with the <code>DataRepositorySubdirectories</code> parameter.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For Amazon File Cache, the path can be an S3 bucket or prefix in the format
     *         <code>s3://bucket-name/prefix/</code> (where <code>prefix</code> is optional).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Amazon FSx for Lustre, the path can be an S3 bucket or prefix in the format
     *         <code>s3://bucket-name/prefix/</code> (where <code>prefix</code> is optional).
     *         </p>
     *         </li>
     */
    public final String dataRepositoryPath() {
        return dataRepositoryPath;
    }

    /**
     * <p>
     * A boolean flag indicating whether an import data repository task to import metadata should run after the data
     * repository association is created. The task runs if this flag is set to <code>true</code>.
     * </p>
     * <note>
     * <p>
     * <code>BatchImportMetaDataOnCreate</code> is not supported for data repositories linked to an Amazon File Cache
     * resource.
     * </p>
     * </note>
     * 
     * @return A boolean flag indicating whether an import data repository task to import metadata should run after the
     *         data repository association is created. The task runs if this flag is set to <code>true</code>.</p>
     *         <note>
     *         <p>
     *         <code>BatchImportMetaDataOnCreate</code> is not supported for data repositories linked to an Amazon File
     *         Cache resource.
     *         </p>
     */
    public final Boolean batchImportMetaDataOnCreate() {
        return batchImportMetaDataOnCreate;
    }

    /**
     * <p>
     * For files imported from a data repository, this value determines the stripe count and maximum amount of data per
     * file (in MiB) stored on a single physical disk. The maximum number of disks that a single file can be striped
     * across is limited by the total number of disks that make up the file system or cache.
     * </p>
     * <p>
     * The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3 objects have a
     * maximum size of 5 TB.
     * </p>
     * 
     * @return For files imported from a data repository, this value determines the stripe count and maximum amount of
     *         data per file (in MiB) stored on a single physical disk. The maximum number of disks that a single file
     *         can be striped across is limited by the total number of disks that make up the file system or cache.</p>
     *         <p>
     *         The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3
     *         objects have a maximum size of 5 TB.
     */
    public final Integer importedFileChunkSize() {
        return importedFileChunkSize;
    }

    /**
     * <p>
     * The configuration for an Amazon S3 data repository linked to an Amazon FSx for Lustre file system with a data
     * repository association.
     * </p>
     * 
     * @return The configuration for an Amazon S3 data repository linked to an Amazon FSx for Lustre file system with a
     *         data repository association.
     */
    public final S3DataRepositoryConfiguration s3() {
        return s3;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the Tags property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The value of the Tags property for this object.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * Returns the value of the CreationTime property for this object.
     * 
     * @return The value of the CreationTime property for this object.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The globally unique ID of the Amazon File Cache resource.
     * </p>
     * 
     * @return The globally unique ID of the Amazon File Cache resource.
     */
    public final String fileCacheId() {
        return fileCacheId;
    }

    /**
     * <p>
     * A path on the Amazon File Cache that points to a high-level directory (such as <code>/ns1/</code>) or
     * subdirectory (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with <code>DataRepositoryPath</code>.
     * The leading forward slash in the path is required. Two data repository associations cannot have overlapping cache
     * paths. For example, if a data repository is associated with cache path <code>/ns1/</code>, then you cannot link
     * another data repository with cache path <code>/ns1/ns2</code>.
     * </p>
     * <p>
     * This path specifies the directory in your cache where files will be exported from. This cache directory can be
     * linked to only one data repository (S3 or NFS) and no other data repository can be linked to the directory.
     * </p>
     * <note>
     * <p>
     * The cache path can only be set to root (/) on an NFS DRA when <code>DataRepositorySubdirectories</code> is
     * specified. If you specify root (/) as the cache path, you can create only one DRA on the cache.
     * </p>
     * <p>
     * The cache path cannot be set to root (/) for an S3 DRA.
     * </p>
     * </note>
     * 
     * @return A path on the Amazon File Cache that points to a high-level directory (such as <code>/ns1/</code>) or
     *         subdirectory (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with
     *         <code>DataRepositoryPath</code>. The leading forward slash in the path is required. Two data repository
     *         associations cannot have overlapping cache paths. For example, if a data repository is associated with
     *         cache path <code>/ns1/</code>, then you cannot link another data repository with cache path
     *         <code>/ns1/ns2</code>.</p>
     *         <p>
     *         This path specifies the directory in your cache where files will be exported from. This cache directory
     *         can be linked to only one data repository (S3 or NFS) and no other data repository can be linked to the
     *         directory.
     *         </p>
     *         <note>
     *         <p>
     *         The cache path can only be set to root (/) on an NFS DRA when <code>DataRepositorySubdirectories</code>
     *         is specified. If you specify root (/) as the cache path, you can create only one DRA on the cache.
     *         </p>
     *         <p>
     *         The cache path cannot be set to root (/) for an S3 DRA.
     *         </p>
     */
    public final String fileCachePath() {
        return fileCachePath;
    }

    /**
     * For responses, this returns true if the service returned a value for the DataRepositorySubdirectories property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDataRepositorySubdirectories() {
        return dataRepositorySubdirectories != null && !(dataRepositorySubdirectories instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For Amazon File Cache, a list of NFS Exports that will be linked with an NFS data repository association. All the
     * subdirectories must be on a single NFS file system. The Export paths are in the format <code>/exportpath1</code>.
     * To use this parameter, you must configure <code>DataRepositoryPath</code> as the domain name of the NFS file
     * system. The NFS file system domain name in effect is the root of the subdirectories. Note that
     * <code>DataRepositorySubdirectories</code> is not supported for S3 data repositories.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDataRepositorySubdirectories} method.
     * </p>
     * 
     * @return For Amazon File Cache, a list of NFS Exports that will be linked with an NFS data repository association.
     *         All the subdirectories must be on a single NFS file system. The Export paths are in the format
     *         <code>/exportpath1</code>. To use this parameter, you must configure <code>DataRepositoryPath</code> as
     *         the domain name of the NFS file system. The NFS file system domain name in effect is the root of the
     *         subdirectories. Note that <code>DataRepositorySubdirectories</code> is not supported for S3 data
     *         repositories.
     */
    public final List<String> dataRepositorySubdirectories() {
        return dataRepositorySubdirectories;
    }

    /**
     * <p>
     * The configuration for an NFS data repository linked to an Amazon File Cache resource with a data repository
     * association.
     * </p>
     * 
     * @return The configuration for an NFS data repository linked to an Amazon File Cache resource with a data
     *         repository association.
     */
    public final NFSDataRepositoryConfiguration nfs() {
        return nfs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceARN());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureDetails());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemPath());
        hashCode = 31 * hashCode + Objects.hashCode(dataRepositoryPath());
        hashCode = 31 * hashCode + Objects.hashCode(batchImportMetaDataOnCreate());
        hashCode = 31 * hashCode + Objects.hashCode(importedFileChunkSize());
        hashCode = 31 * hashCode + Objects.hashCode(s3());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(fileCacheId());
        hashCode = 31 * hashCode + Objects.hashCode(fileCachePath());
        hashCode = 31 * hashCode + Objects.hashCode(hasDataRepositorySubdirectories() ? dataRepositorySubdirectories() : null);
        hashCode = 31 * hashCode + Objects.hashCode(nfs());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataRepositoryAssociation)) {
            return false;
        }
        DataRepositoryAssociation other = (DataRepositoryAssociation) obj;
        return Objects.equals(associationId(), other.associationId()) && Objects.equals(resourceARN(), other.resourceARN())
                && Objects.equals(fileSystemId(), other.fileSystemId())
                && Objects.equals(lifecycleAsString(), other.lifecycleAsString())
                && Objects.equals(failureDetails(), other.failureDetails())
                && Objects.equals(fileSystemPath(), other.fileSystemPath())
                && Objects.equals(dataRepositoryPath(), other.dataRepositoryPath())
                && Objects.equals(batchImportMetaDataOnCreate(), other.batchImportMetaDataOnCreate())
                && Objects.equals(importedFileChunkSize(), other.importedFileChunkSize()) && Objects.equals(s3(), other.s3())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(creationTime(), other.creationTime()) && Objects.equals(fileCacheId(), other.fileCacheId())
                && Objects.equals(fileCachePath(), other.fileCachePath())
                && hasDataRepositorySubdirectories() == other.hasDataRepositorySubdirectories()
                && Objects.equals(dataRepositorySubdirectories(), other.dataRepositorySubdirectories())
                && Objects.equals(nfs(), other.nfs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataRepositoryAssociation").add("AssociationId", associationId())
                .add("ResourceARN", resourceARN()).add("FileSystemId", fileSystemId()).add("Lifecycle", lifecycleAsString())
                .add("FailureDetails", failureDetails()).add("FileSystemPath", fileSystemPath())
                .add("DataRepositoryPath", dataRepositoryPath())
                .add("BatchImportMetaDataOnCreate", batchImportMetaDataOnCreate())
                .add("ImportedFileChunkSize", importedFileChunkSize()).add("S3", s3()).add("Tags", hasTags() ? tags() : null)
                .add("CreationTime", creationTime()).add("FileCacheId", fileCacheId()).add("FileCachePath", fileCachePath())
                .add("DataRepositorySubdirectories", hasDataRepositorySubdirectories() ? dataRepositorySubdirectories() : null)
                .add("NFS", nfs()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssociationId":
            return Optional.ofNullable(clazz.cast(associationId()));
        case "ResourceARN":
            return Optional.ofNullable(clazz.cast(resourceARN()));
        case "FileSystemId":
            return Optional.ofNullable(clazz.cast(fileSystemId()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycleAsString()));
        case "FailureDetails":
            return Optional.ofNullable(clazz.cast(failureDetails()));
        case "FileSystemPath":
            return Optional.ofNullable(clazz.cast(fileSystemPath()));
        case "DataRepositoryPath":
            return Optional.ofNullable(clazz.cast(dataRepositoryPath()));
        case "BatchImportMetaDataOnCreate":
            return Optional.ofNullable(clazz.cast(batchImportMetaDataOnCreate()));
        case "ImportedFileChunkSize":
            return Optional.ofNullable(clazz.cast(importedFileChunkSize()));
        case "S3":
            return Optional.ofNullable(clazz.cast(s3()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "FileCacheId":
            return Optional.ofNullable(clazz.cast(fileCacheId()));
        case "FileCachePath":
            return Optional.ofNullable(clazz.cast(fileCachePath()));
        case "DataRepositorySubdirectories":
            return Optional.ofNullable(clazz.cast(dataRepositorySubdirectories()));
        case "NFS":
            return Optional.ofNullable(clazz.cast(nfs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AssociationId", ASSOCIATION_ID_FIELD);
        map.put("ResourceARN", RESOURCE_ARN_FIELD);
        map.put("FileSystemId", FILE_SYSTEM_ID_FIELD);
        map.put("Lifecycle", LIFECYCLE_FIELD);
        map.put("FailureDetails", FAILURE_DETAILS_FIELD);
        map.put("FileSystemPath", FILE_SYSTEM_PATH_FIELD);
        map.put("DataRepositoryPath", DATA_REPOSITORY_PATH_FIELD);
        map.put("BatchImportMetaDataOnCreate", BATCH_IMPORT_META_DATA_ON_CREATE_FIELD);
        map.put("ImportedFileChunkSize", IMPORTED_FILE_CHUNK_SIZE_FIELD);
        map.put("S3", S3_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("FileCacheId", FILE_CACHE_ID_FIELD);
        map.put("FileCachePath", FILE_CACHE_PATH_FIELD);
        map.put("DataRepositorySubdirectories", DATA_REPOSITORY_SUBDIRECTORIES_FIELD);
        map.put("NFS", NFS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DataRepositoryAssociation, T> g) {
        return obj -> g.apply((DataRepositoryAssociation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataRepositoryAssociation> {
        /**
         * <p>
         * The system-generated, unique ID of the data repository association.
         * </p>
         * 
         * @param associationId
         *        The system-generated, unique ID of the data repository association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * Sets the value of the ResourceARN property for this object.
         *
         * @param resourceARN
         *        The new value for the ResourceARN property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceARN(String resourceARN);

        /**
         * Sets the value of the FileSystemId property for this object.
         *
         * @param fileSystemId
         *        The new value for the FileSystemId property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemId(String fileSystemId);

        /**
         * <p>
         * Describes the state of a data repository association. The lifecycle can have the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The data repository association between the file system or cache and the data
         * repository is being created. The data repository is unavailable.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The data repository association is available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> - The data repository association is misconfigured. Until the configuration is
         * corrected, automatic import and automatic export will not work (only for Amazon FSx for Lustre).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that might
         * affect its availability.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        Describes the state of a data repository association. The lifecycle can have the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The data repository association between the file system or cache and the data
         *        repository is being created. The data repository is unavailable.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The data repository association is available for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> - The data repository association is misconfigured. Until the configuration
         *        is corrected, automatic import and automatic export will not work (only for Amazon FSx for Lustre).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that
         *        might affect its availability.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
         *        </p>
         *        </li>
         * @see DataRepositoryLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryLifecycle
         */
        Builder lifecycle(String lifecycle);

        /**
         * <p>
         * Describes the state of a data repository association. The lifecycle can have the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The data repository association between the file system or cache and the data
         * repository is being created. The data repository is unavailable.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The data repository association is available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> - The data repository association is misconfigured. Until the configuration is
         * corrected, automatic import and automatic export will not work (only for Amazon FSx for Lustre).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that might
         * affect its availability.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        Describes the state of a data repository association. The lifecycle can have the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The data repository association between the file system or cache and the data
         *        repository is being created. The data repository is unavailable.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The data repository association is available for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> - The data repository association is misconfigured. Until the configuration
         *        is corrected, automatic import and automatic export will not work (only for Amazon FSx for Lustre).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The data repository association is undergoing a customer initiated update that
         *        might affect its availability.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The data repository association is undergoing a customer initiated deletion.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The data repository association is in a terminal state that cannot be recovered.
         *        </p>
         *        </li>
         * @see DataRepositoryLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryLifecycle
         */
        Builder lifecycle(DataRepositoryLifecycle lifecycle);

        /**
         * Sets the value of the FailureDetails property for this object.
         *
         * @param failureDetails
         *        The new value for the FailureDetails property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureDetails(DataRepositoryFailureDetails failureDetails);

        /**
         * Sets the value of the FailureDetails property for this object.
         *
         * This is a convenience method that creates an instance of the {@link DataRepositoryFailureDetails.Builder}
         * avoiding the need to create one manually via {@link DataRepositoryFailureDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataRepositoryFailureDetails.Builder#build()} is called
         * immediately and its result is passed to {@link #failureDetails(DataRepositoryFailureDetails)}.
         * 
         * @param failureDetails
         *        a consumer that will call methods on {@link DataRepositoryFailureDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureDetails(DataRepositoryFailureDetails)
         */
        default Builder failureDetails(Consumer<DataRepositoryFailureDetails.Builder> failureDetails) {
            return failureDetails(DataRepositoryFailureDetails.builder().applyMutation(failureDetails).build());
        }

        /**
         * <p>
         * A path on the Amazon FSx for Lustre file system that points to a high-level directory (such as
         * <code>/ns1/</code>) or subdirectory (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with
         * <code>DataRepositoryPath</code>. The leading forward slash in the name is required. Two data repository
         * associations cannot have overlapping file system paths. For example, if a data repository is associated with
         * file system path <code>/ns1/</code>, then you cannot link another data repository with file system path
         * <code>/ns1/ns2</code>.
         * </p>
         * <p>
         * This path specifies where in your file system files will be exported from or imported to. This file system
         * directory can be linked to only one Amazon S3 bucket, and no other S3 bucket can be linked to the directory.
         * </p>
         * <note>
         * <p>
         * If you specify only a forward slash (<code>/</code>) as the file system path, you can link only one data
         * repository to the file system. You can only specify "/" as the file system path for the first data repository
         * associated with a file system.
         * </p>
         * </note>
         * 
         * @param fileSystemPath
         *        A path on the Amazon FSx for Lustre file system that points to a high-level directory (such as
         *        <code>/ns1/</code>) or subdirectory (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with
         *        <code>DataRepositoryPath</code>. The leading forward slash in the name is required. Two data
         *        repository associations cannot have overlapping file system paths. For example, if a data repository
         *        is associated with file system path <code>/ns1/</code>, then you cannot link another data repository
         *        with file system path <code>/ns1/ns2</code>.</p>
         *        <p>
         *        This path specifies where in your file system files will be exported from or imported to. This file
         *        system directory can be linked to only one Amazon S3 bucket, and no other S3 bucket can be linked to
         *        the directory.
         *        </p>
         *        <note>
         *        <p>
         *        If you specify only a forward slash (<code>/</code>) as the file system path, you can link only one
         *        data repository to the file system. You can only specify "/" as the file system path for the first
         *        data repository associated with a file system.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemPath(String fileSystemPath);

        /**
         * <p>
         * The path to the data repository that will be linked to the cache or file system.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For Amazon File Cache, the path can be an NFS data repository that will be linked to the cache. The path can
         * be in one of two formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you are not using the <code>DataRepositorySubdirectories</code> parameter, the path is to an NFS Export
         * directory (or one of its subdirectories) in the format <code>nsf://nfs-domain-name/exportpath</code>. You can
         * therefore link a single NFS Export to a single data repository association.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you are using the <code>DataRepositorySubdirectories</code> parameter, the path is the domain name of the
         * NFS file system in the format <code>nfs://filer-domain-name</code>, which indicates the root of the
         * subdirectories specified with the <code>DataRepositorySubdirectories</code> parameter.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For Amazon File Cache, the path can be an S3 bucket or prefix in the format
         * <code>s3://bucket-name/prefix/</code> (where <code>prefix</code> is optional).
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon FSx for Lustre, the path can be an S3 bucket or prefix in the format
         * <code>s3://bucket-name/prefix/</code> (where <code>prefix</code> is optional).
         * </p>
         * </li>
         * </ul>
         * 
         * @param dataRepositoryPath
         *        The path to the data repository that will be linked to the cache or file system.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For Amazon File Cache, the path can be an NFS data repository that will be linked to the cache. The
         *        path can be in one of two formats:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you are not using the <code>DataRepositorySubdirectories</code> parameter, the path is to an NFS
         *        Export directory (or one of its subdirectories) in the format
         *        <code>nsf://nfs-domain-name/exportpath</code>. You can therefore link a single NFS Export to a single
         *        data repository association.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you are using the <code>DataRepositorySubdirectories</code> parameter, the path is the domain name
         *        of the NFS file system in the format <code>nfs://filer-domain-name</code>, which indicates the root of
         *        the subdirectories specified with the <code>DataRepositorySubdirectories</code> parameter.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For Amazon File Cache, the path can be an S3 bucket or prefix in the format
         *        <code>s3://bucket-name/prefix/</code> (where <code>prefix</code> is optional).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amazon FSx for Lustre, the path can be an S3 bucket or prefix in the format
         *        <code>s3://bucket-name/prefix/</code> (where <code>prefix</code> is optional).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRepositoryPath(String dataRepositoryPath);

        /**
         * <p>
         * A boolean flag indicating whether an import data repository task to import metadata should run after the data
         * repository association is created. The task runs if this flag is set to <code>true</code>.
         * </p>
         * <note>
         * <p>
         * <code>BatchImportMetaDataOnCreate</code> is not supported for data repositories linked to an Amazon File
         * Cache resource.
         * </p>
         * </note>
         * 
         * @param batchImportMetaDataOnCreate
         *        A boolean flag indicating whether an import data repository task to import metadata should run after
         *        the data repository association is created. The task runs if this flag is set to <code>true</code>
         *        .</p> <note>
         *        <p>
         *        <code>BatchImportMetaDataOnCreate</code> is not supported for data repositories linked to an Amazon
         *        File Cache resource.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder batchImportMetaDataOnCreate(Boolean batchImportMetaDataOnCreate);

        /**
         * <p>
         * For files imported from a data repository, this value determines the stripe count and maximum amount of data
         * per file (in MiB) stored on a single physical disk. The maximum number of disks that a single file can be
         * striped across is limited by the total number of disks that make up the file system or cache.
         * </p>
         * <p>
         * The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3 objects
         * have a maximum size of 5 TB.
         * </p>
         * 
         * @param importedFileChunkSize
         *        For files imported from a data repository, this value determines the stripe count and maximum amount
         *        of data per file (in MiB) stored on a single physical disk. The maximum number of disks that a single
         *        file can be striped across is limited by the total number of disks that make up the file system or
         *        cache.</p>
         *        <p>
         *        The default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500 GiB). Amazon S3
         *        objects have a maximum size of 5 TB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importedFileChunkSize(Integer importedFileChunkSize);

        /**
         * <p>
         * The configuration for an Amazon S3 data repository linked to an Amazon FSx for Lustre file system with a data
         * repository association.
         * </p>
         * 
         * @param s3
         *        The configuration for an Amazon S3 data repository linked to an Amazon FSx for Lustre file system with
         *        a data repository association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3(S3DataRepositoryConfiguration s3);

        /**
         * <p>
         * The configuration for an Amazon S3 data repository linked to an Amazon FSx for Lustre file system with a data
         * repository association.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3DataRepositoryConfiguration.Builder}
         * avoiding the need to create one manually via {@link S3DataRepositoryConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3DataRepositoryConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #s3(S3DataRepositoryConfiguration)}.
         * 
         * @param s3
         *        a consumer that will call methods on {@link S3DataRepositoryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3(S3DataRepositoryConfiguration)
         */
        default Builder s3(Consumer<S3DataRepositoryConfiguration.Builder> s3) {
            return s3(S3DataRepositoryConfiguration.builder().applyMutation(s3).build());
        }

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fsx.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.fsx.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.fsx.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.fsx.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * Sets the value of the CreationTime property for this object.
         *
         * @param creationTime
         *        The new value for the CreationTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The globally unique ID of the Amazon File Cache resource.
         * </p>
         * 
         * @param fileCacheId
         *        The globally unique ID of the Amazon File Cache resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileCacheId(String fileCacheId);

        /**
         * <p>
         * A path on the Amazon File Cache that points to a high-level directory (such as <code>/ns1/</code>) or
         * subdirectory (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with <code>DataRepositoryPath</code>
         * . The leading forward slash in the path is required. Two data repository associations cannot have overlapping
         * cache paths. For example, if a data repository is associated with cache path <code>/ns1/</code>, then you
         * cannot link another data repository with cache path <code>/ns1/ns2</code>.
         * </p>
         * <p>
         * This path specifies the directory in your cache where files will be exported from. This cache directory can
         * be linked to only one data repository (S3 or NFS) and no other data repository can be linked to the
         * directory.
         * </p>
         * <note>
         * <p>
         * The cache path can only be set to root (/) on an NFS DRA when <code>DataRepositorySubdirectories</code> is
         * specified. If you specify root (/) as the cache path, you can create only one DRA on the cache.
         * </p>
         * <p>
         * The cache path cannot be set to root (/) for an S3 DRA.
         * </p>
         * </note>
         * 
         * @param fileCachePath
         *        A path on the Amazon File Cache that points to a high-level directory (such as <code>/ns1/</code>) or
         *        subdirectory (such as <code>/ns1/subdir/</code>) that will be mapped 1-1 with
         *        <code>DataRepositoryPath</code>. The leading forward slash in the path is required. Two data
         *        repository associations cannot have overlapping cache paths. For example, if a data repository is
         *        associated with cache path <code>/ns1/</code>, then you cannot link another data repository with cache
         *        path <code>/ns1/ns2</code>.</p>
         *        <p>
         *        This path specifies the directory in your cache where files will be exported from. This cache
         *        directory can be linked to only one data repository (S3 or NFS) and no other data repository can be
         *        linked to the directory.
         *        </p>
         *        <note>
         *        <p>
         *        The cache path can only be set to root (/) on an NFS DRA when
         *        <code>DataRepositorySubdirectories</code> is specified. If you specify root (/) as the cache path, you
         *        can create only one DRA on the cache.
         *        </p>
         *        <p>
         *        The cache path cannot be set to root (/) for an S3 DRA.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileCachePath(String fileCachePath);

        /**
         * <p>
         * For Amazon File Cache, a list of NFS Exports that will be linked with an NFS data repository association. All
         * the subdirectories must be on a single NFS file system. The Export paths are in the format
         * <code>/exportpath1</code>. To use this parameter, you must configure <code>DataRepositoryPath</code> as the
         * domain name of the NFS file system. The NFS file system domain name in effect is the root of the
         * subdirectories. Note that <code>DataRepositorySubdirectories</code> is not supported for S3 data
         * repositories.
         * </p>
         * 
         * @param dataRepositorySubdirectories
         *        For Amazon File Cache, a list of NFS Exports that will be linked with an NFS data repository
         *        association. All the subdirectories must be on a single NFS file system. The Export paths are in the
         *        format <code>/exportpath1</code>. To use this parameter, you must configure
         *        <code>DataRepositoryPath</code> as the domain name of the NFS file system. The NFS file system domain
         *        name in effect is the root of the subdirectories. Note that <code>DataRepositorySubdirectories</code>
         *        is not supported for S3 data repositories.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRepositorySubdirectories(Collection<String> dataRepositorySubdirectories);

        /**
         * <p>
         * For Amazon File Cache, a list of NFS Exports that will be linked with an NFS data repository association. All
         * the subdirectories must be on a single NFS file system. The Export paths are in the format
         * <code>/exportpath1</code>. To use this parameter, you must configure <code>DataRepositoryPath</code> as the
         * domain name of the NFS file system. The NFS file system domain name in effect is the root of the
         * subdirectories. Note that <code>DataRepositorySubdirectories</code> is not supported for S3 data
         * repositories.
         * </p>
         * 
         * @param dataRepositorySubdirectories
         *        For Amazon File Cache, a list of NFS Exports that will be linked with an NFS data repository
         *        association. All the subdirectories must be on a single NFS file system. The Export paths are in the
         *        format <code>/exportpath1</code>. To use this parameter, you must configure
         *        <code>DataRepositoryPath</code> as the domain name of the NFS file system. The NFS file system domain
         *        name in effect is the root of the subdirectories. Note that <code>DataRepositorySubdirectories</code>
         *        is not supported for S3 data repositories.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRepositorySubdirectories(String... dataRepositorySubdirectories);

        /**
         * <p>
         * The configuration for an NFS data repository linked to an Amazon File Cache resource with a data repository
         * association.
         * </p>
         * 
         * @param nfs
         *        The configuration for an NFS data repository linked to an Amazon File Cache resource with a data
         *        repository association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nfs(NFSDataRepositoryConfiguration nfs);

        /**
         * <p>
         * The configuration for an NFS data repository linked to an Amazon File Cache resource with a data repository
         * association.
         * </p>
         * This is a convenience method that creates an instance of the {@link NFSDataRepositoryConfiguration.Builder}
         * avoiding the need to create one manually via {@link NFSDataRepositoryConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NFSDataRepositoryConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #nfs(NFSDataRepositoryConfiguration)}.
         * 
         * @param nfs
         *        a consumer that will call methods on {@link NFSDataRepositoryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nfs(NFSDataRepositoryConfiguration)
         */
        default Builder nfs(Consumer<NFSDataRepositoryConfiguration.Builder> nfs) {
            return nfs(NFSDataRepositoryConfiguration.builder().applyMutation(nfs).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String associationId;

        private String resourceARN;

        private String fileSystemId;

        private String lifecycle;

        private DataRepositoryFailureDetails failureDetails;

        private String fileSystemPath;

        private String dataRepositoryPath;

        private Boolean batchImportMetaDataOnCreate;

        private Integer importedFileChunkSize;

        private S3DataRepositoryConfiguration s3;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Instant creationTime;

        private String fileCacheId;

        private String fileCachePath;

        private List<String> dataRepositorySubdirectories = DefaultSdkAutoConstructList.getInstance();

        private NFSDataRepositoryConfiguration nfs;

        private BuilderImpl() {
        }

        private BuilderImpl(DataRepositoryAssociation model) {
            associationId(model.associationId);
            resourceARN(model.resourceARN);
            fileSystemId(model.fileSystemId);
            lifecycle(model.lifecycle);
            failureDetails(model.failureDetails);
            fileSystemPath(model.fileSystemPath);
            dataRepositoryPath(model.dataRepositoryPath);
            batchImportMetaDataOnCreate(model.batchImportMetaDataOnCreate);
            importedFileChunkSize(model.importedFileChunkSize);
            s3(model.s3);
            tags(model.tags);
            creationTime(model.creationTime);
            fileCacheId(model.fileCacheId);
            fileCachePath(model.fileCachePath);
            dataRepositorySubdirectories(model.dataRepositorySubdirectories);
            nfs(model.nfs);
        }

        public final String getAssociationId() {
            return associationId;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final String getResourceARN() {
            return resourceARN;
        }

        public final void setResourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
        }

        @Override
        public final Builder resourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
            return this;
        }

        public final String getFileSystemId() {
            return fileSystemId;
        }

        public final void setFileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
        }

        @Override
        public final Builder fileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
            return this;
        }

        public final String getLifecycle() {
            return lifecycle;
        }

        public final void setLifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
        }

        @Override
        public final Builder lifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        @Override
        public final Builder lifecycle(DataRepositoryLifecycle lifecycle) {
            this.lifecycle(lifecycle == null ? null : lifecycle.toString());
            return this;
        }

        public final DataRepositoryFailureDetails.Builder getFailureDetails() {
            return failureDetails != null ? failureDetails.toBuilder() : null;
        }

        public final void setFailureDetails(DataRepositoryFailureDetails.BuilderImpl failureDetails) {
            this.failureDetails = failureDetails != null ? failureDetails.build() : null;
        }

        @Override
        public final Builder failureDetails(DataRepositoryFailureDetails failureDetails) {
            this.failureDetails = failureDetails;
            return this;
        }

        public final String getFileSystemPath() {
            return fileSystemPath;
        }

        public final void setFileSystemPath(String fileSystemPath) {
            this.fileSystemPath = fileSystemPath;
        }

        @Override
        public final Builder fileSystemPath(String fileSystemPath) {
            this.fileSystemPath = fileSystemPath;
            return this;
        }

        public final String getDataRepositoryPath() {
            return dataRepositoryPath;
        }

        public final void setDataRepositoryPath(String dataRepositoryPath) {
            this.dataRepositoryPath = dataRepositoryPath;
        }

        @Override
        public final Builder dataRepositoryPath(String dataRepositoryPath) {
            this.dataRepositoryPath = dataRepositoryPath;
            return this;
        }

        public final Boolean getBatchImportMetaDataOnCreate() {
            return batchImportMetaDataOnCreate;
        }

        public final void setBatchImportMetaDataOnCreate(Boolean batchImportMetaDataOnCreate) {
            this.batchImportMetaDataOnCreate = batchImportMetaDataOnCreate;
        }

        @Override
        public final Builder batchImportMetaDataOnCreate(Boolean batchImportMetaDataOnCreate) {
            this.batchImportMetaDataOnCreate = batchImportMetaDataOnCreate;
            return this;
        }

        public final Integer getImportedFileChunkSize() {
            return importedFileChunkSize;
        }

        public final void setImportedFileChunkSize(Integer importedFileChunkSize) {
            this.importedFileChunkSize = importedFileChunkSize;
        }

        @Override
        public final Builder importedFileChunkSize(Integer importedFileChunkSize) {
            this.importedFileChunkSize = importedFileChunkSize;
            return this;
        }

        public final S3DataRepositoryConfiguration.Builder getS3() {
            return s3 != null ? s3.toBuilder() : null;
        }

        public final void setS3(S3DataRepositoryConfiguration.BuilderImpl s3) {
            this.s3 = s3 != null ? s3.build() : null;
        }

        @Override
        public final Builder s3(S3DataRepositoryConfiguration s3) {
            this.s3 = s3;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getFileCacheId() {
            return fileCacheId;
        }

        public final void setFileCacheId(String fileCacheId) {
            this.fileCacheId = fileCacheId;
        }

        @Override
        public final Builder fileCacheId(String fileCacheId) {
            this.fileCacheId = fileCacheId;
            return this;
        }

        public final String getFileCachePath() {
            return fileCachePath;
        }

        public final void setFileCachePath(String fileCachePath) {
            this.fileCachePath = fileCachePath;
        }

        @Override
        public final Builder fileCachePath(String fileCachePath) {
            this.fileCachePath = fileCachePath;
            return this;
        }

        public final Collection<String> getDataRepositorySubdirectories() {
            if (dataRepositorySubdirectories instanceof SdkAutoConstructList) {
                return null;
            }
            return dataRepositorySubdirectories;
        }

        public final void setDataRepositorySubdirectories(Collection<String> dataRepositorySubdirectories) {
            this.dataRepositorySubdirectories = SubDirectoriesPathsCopier.copy(dataRepositorySubdirectories);
        }

        @Override
        public final Builder dataRepositorySubdirectories(Collection<String> dataRepositorySubdirectories) {
            this.dataRepositorySubdirectories = SubDirectoriesPathsCopier.copy(dataRepositorySubdirectories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataRepositorySubdirectories(String... dataRepositorySubdirectories) {
            dataRepositorySubdirectories(Arrays.asList(dataRepositorySubdirectories));
            return this;
        }

        public final NFSDataRepositoryConfiguration.Builder getNfs() {
            return nfs != null ? nfs.toBuilder() : null;
        }

        public final void setNfs(NFSDataRepositoryConfiguration.BuilderImpl nfs) {
            this.nfs = nfs != null ? nfs.build() : null;
        }

        @Override
        public final Builder nfs(NFSDataRepositoryConfiguration nfs) {
            this.nfs = nfs;
            return this;
        }

        @Override
        public DataRepositoryAssociation build() {
            return new DataRepositoryAssociation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
