/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for an Amazon S3 data repository linked to an Amazon FSx for Lustre file system with a data
 * repository association. The configuration consists of an <code>AutoImportPolicy</code> that defines which file events
 * on the data repository are automatically imported to the file system and an <code>AutoExportPolicy</code> that
 * defines which file events on the file system are automatically exported to the data repository. File events are when
 * files or directories are added, changed, or deleted on the file system or the data repository.
 * </p>
 * <note>
 * <p>
 * Data repository associations on Amazon File Cache don't use <code>S3DataRepositoryConfiguration</code> because they
 * don't support automatic import or automatic export.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3DataRepositoryConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<S3DataRepositoryConfiguration.Builder, S3DataRepositoryConfiguration> {
    private static final SdkField<AutoImportPolicy> AUTO_IMPORT_POLICY_FIELD = SdkField
            .<AutoImportPolicy> builder(MarshallingType.SDK_POJO).memberName("AutoImportPolicy")
            .getter(getter(S3DataRepositoryConfiguration::autoImportPolicy)).setter(setter(Builder::autoImportPolicy))
            .constructor(AutoImportPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoImportPolicy").build()).build();

    private static final SdkField<AutoExportPolicy> AUTO_EXPORT_POLICY_FIELD = SdkField
            .<AutoExportPolicy> builder(MarshallingType.SDK_POJO).memberName("AutoExportPolicy")
            .getter(getter(S3DataRepositoryConfiguration::autoExportPolicy)).setter(setter(Builder::autoExportPolicy))
            .constructor(AutoExportPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoExportPolicy").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTO_IMPORT_POLICY_FIELD,
            AUTO_EXPORT_POLICY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final AutoImportPolicy autoImportPolicy;

    private final AutoExportPolicy autoExportPolicy;

    private S3DataRepositoryConfiguration(BuilderImpl builder) {
        this.autoImportPolicy = builder.autoImportPolicy;
        this.autoExportPolicy = builder.autoExportPolicy;
    }

    /**
     * <p>
     * Specifies the type of updated objects (new, changed, deleted) that will be automatically imported from the linked
     * S3 bucket to your file system.
     * </p>
     * 
     * @return Specifies the type of updated objects (new, changed, deleted) that will be automatically imported from
     *         the linked S3 bucket to your file system.
     */
    public final AutoImportPolicy autoImportPolicy() {
        return autoImportPolicy;
    }

    /**
     * <p>
     * Specifies the type of updated objects (new, changed, deleted) that will be automatically exported from your file
     * system to the linked S3 bucket.
     * </p>
     * 
     * @return Specifies the type of updated objects (new, changed, deleted) that will be automatically exported from
     *         your file system to the linked S3 bucket.
     */
    public final AutoExportPolicy autoExportPolicy() {
        return autoExportPolicy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(autoImportPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(autoExportPolicy());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3DataRepositoryConfiguration)) {
            return false;
        }
        S3DataRepositoryConfiguration other = (S3DataRepositoryConfiguration) obj;
        return Objects.equals(autoImportPolicy(), other.autoImportPolicy())
                && Objects.equals(autoExportPolicy(), other.autoExportPolicy());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3DataRepositoryConfiguration").add("AutoImportPolicy", autoImportPolicy())
                .add("AutoExportPolicy", autoExportPolicy()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AutoImportPolicy":
            return Optional.ofNullable(clazz.cast(autoImportPolicy()));
        case "AutoExportPolicy":
            return Optional.ofNullable(clazz.cast(autoExportPolicy()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AutoImportPolicy", AUTO_IMPORT_POLICY_FIELD);
        map.put("AutoExportPolicy", AUTO_EXPORT_POLICY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<S3DataRepositoryConfiguration, T> g) {
        return obj -> g.apply((S3DataRepositoryConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3DataRepositoryConfiguration> {
        /**
         * <p>
         * Specifies the type of updated objects (new, changed, deleted) that will be automatically imported from the
         * linked S3 bucket to your file system.
         * </p>
         * 
         * @param autoImportPolicy
         *        Specifies the type of updated objects (new, changed, deleted) that will be automatically imported from
         *        the linked S3 bucket to your file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoImportPolicy(AutoImportPolicy autoImportPolicy);

        /**
         * <p>
         * Specifies the type of updated objects (new, changed, deleted) that will be automatically imported from the
         * linked S3 bucket to your file system.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoImportPolicy.Builder} avoiding the
         * need to create one manually via {@link AutoImportPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoImportPolicy.Builder#build()} is called immediately and its
         * result is passed to {@link #autoImportPolicy(AutoImportPolicy)}.
         * 
         * @param autoImportPolicy
         *        a consumer that will call methods on {@link AutoImportPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoImportPolicy(AutoImportPolicy)
         */
        default Builder autoImportPolicy(Consumer<AutoImportPolicy.Builder> autoImportPolicy) {
            return autoImportPolicy(AutoImportPolicy.builder().applyMutation(autoImportPolicy).build());
        }

        /**
         * <p>
         * Specifies the type of updated objects (new, changed, deleted) that will be automatically exported from your
         * file system to the linked S3 bucket.
         * </p>
         * 
         * @param autoExportPolicy
         *        Specifies the type of updated objects (new, changed, deleted) that will be automatically exported from
         *        your file system to the linked S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoExportPolicy(AutoExportPolicy autoExportPolicy);

        /**
         * <p>
         * Specifies the type of updated objects (new, changed, deleted) that will be automatically exported from your
         * file system to the linked S3 bucket.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoExportPolicy.Builder} avoiding the
         * need to create one manually via {@link AutoExportPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoExportPolicy.Builder#build()} is called immediately and its
         * result is passed to {@link #autoExportPolicy(AutoExportPolicy)}.
         * 
         * @param autoExportPolicy
         *        a consumer that will call methods on {@link AutoExportPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoExportPolicy(AutoExportPolicy)
         */
        default Builder autoExportPolicy(Consumer<AutoExportPolicy.Builder> autoExportPolicy) {
            return autoExportPolicy(AutoExportPolicy.builder().applyMutation(autoExportPolicy).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private AutoImportPolicy autoImportPolicy;

        private AutoExportPolicy autoExportPolicy;

        private BuilderImpl() {
        }

        private BuilderImpl(S3DataRepositoryConfiguration model) {
            autoImportPolicy(model.autoImportPolicy);
            autoExportPolicy(model.autoExportPolicy);
        }

        public final AutoImportPolicy.Builder getAutoImportPolicy() {
            return autoImportPolicy != null ? autoImportPolicy.toBuilder() : null;
        }

        public final void setAutoImportPolicy(AutoImportPolicy.BuilderImpl autoImportPolicy) {
            this.autoImportPolicy = autoImportPolicy != null ? autoImportPolicy.build() : null;
        }

        @Override
        public final Builder autoImportPolicy(AutoImportPolicy autoImportPolicy) {
            this.autoImportPolicy = autoImportPolicy;
            return this;
        }

        public final AutoExportPolicy.Builder getAutoExportPolicy() {
            return autoExportPolicy != null ? autoExportPolicy.toBuilder() : null;
        }

        public final void setAutoExportPolicy(AutoExportPolicy.BuilderImpl autoExportPolicy) {
            this.autoExportPolicy = autoExportPolicy != null ? autoExportPolicy.build() : null;
        }

        @Override
        public final Builder autoExportPolicy(AutoExportPolicy autoExportPolicy) {
            this.autoExportPolicy = autoExportPolicy;
            return this;
        }

        @Override
        public S3DataRepositoryConfiguration build() {
            return new S3DataRepositoryConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
