/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>This data type is part of Amazon GameLift FleetIQ with game server groups, which is in preview release and is
 * subject to change.</b>
 * </p>
 * <p>
 * Properties describing a game server resource.
 * </p>
 * <p>
 * A game server resource is created by a successful call to <a>RegisterGameServer</a> and deleted by calling
 * <a>DeregisterGameServer</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GameServer implements SdkPojo, Serializable, ToCopyableBuilder<GameServer.Builder, GameServer> {
    private static final SdkField<String> GAME_SERVER_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServer::gameServerGroupName)).setter(setter(Builder::gameServerGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerGroupName").build())
            .build();

    private static final SdkField<String> GAME_SERVER_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServer::gameServerGroupArn)).setter(setter(Builder::gameServerGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerGroupArn").build())
            .build();

    private static final SdkField<String> GAME_SERVER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServer::gameServerId)).setter(setter(Builder::gameServerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerId").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServer::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> CONNECTION_INFO_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServer::connectionInfo)).setter(setter(Builder::connectionInfo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionInfo").build()).build();

    private static final SdkField<String> GAME_SERVER_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServer::gameServerData)).setter(setter(Builder::gameServerData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerData").build()).build();

    private static final SdkField<String> CUSTOM_SORT_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServer::customSortKey)).setter(setter(Builder::customSortKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomSortKey").build()).build();

    private static final SdkField<String> CLAIM_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServer::claimStatusAsString)).setter(setter(Builder::claimStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClaimStatus").build()).build();

    private static final SdkField<String> UTILIZATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServer::utilizationStatusAsString)).setter(setter(Builder::utilizationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UtilizationStatus").build()).build();

    private static final SdkField<Instant> REGISTRATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(GameServer::registrationTime)).setter(setter(Builder::registrationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RegistrationTime").build()).build();

    private static final SdkField<Instant> LAST_CLAIM_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(GameServer::lastClaimTime)).setter(setter(Builder::lastClaimTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastClaimTime").build()).build();

    private static final SdkField<Instant> LAST_HEALTH_CHECK_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(GameServer::lastHealthCheckTime)).setter(setter(Builder::lastHealthCheckTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastHealthCheckTime").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GAME_SERVER_GROUP_NAME_FIELD,
            GAME_SERVER_GROUP_ARN_FIELD, GAME_SERVER_ID_FIELD, INSTANCE_ID_FIELD, CONNECTION_INFO_FIELD, GAME_SERVER_DATA_FIELD,
            CUSTOM_SORT_KEY_FIELD, CLAIM_STATUS_FIELD, UTILIZATION_STATUS_FIELD, REGISTRATION_TIME_FIELD, LAST_CLAIM_TIME_FIELD,
            LAST_HEALTH_CHECK_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String gameServerGroupName;

    private final String gameServerGroupArn;

    private final String gameServerId;

    private final String instanceId;

    private final String connectionInfo;

    private final String gameServerData;

    private final String customSortKey;

    private final String claimStatus;

    private final String utilizationStatus;

    private final Instant registrationTime;

    private final Instant lastClaimTime;

    private final Instant lastHealthCheckTime;

    private GameServer(BuilderImpl builder) {
        this.gameServerGroupName = builder.gameServerGroupName;
        this.gameServerGroupArn = builder.gameServerGroupArn;
        this.gameServerId = builder.gameServerId;
        this.instanceId = builder.instanceId;
        this.connectionInfo = builder.connectionInfo;
        this.gameServerData = builder.gameServerData;
        this.customSortKey = builder.customSortKey;
        this.claimStatus = builder.claimStatus;
        this.utilizationStatus = builder.utilizationStatus;
        this.registrationTime = builder.registrationTime;
        this.lastClaimTime = builder.lastClaimTime;
        this.lastHealthCheckTime = builder.lastHealthCheckTime;
    }

    /**
     * <p>
     * The name identifier for the game server group where the game server is located.
     * </p>
     * 
     * @return The name identifier for the game server group where the game server is located.
     */
    public String gameServerGroupName() {
        return gameServerGroupName;
    }

    /**
     * <p>
     * The ARN identifier for the game server group where the game server is located.
     * </p>
     * 
     * @return The ARN identifier for the game server group where the game server is located.
     */
    public String gameServerGroupArn() {
        return gameServerGroupArn;
    }

    /**
     * <p>
     * A custom string that uniquely identifies the game server. Game server IDs are developer-defined and are unique
     * across all game server groups in an AWS account.
     * </p>
     * 
     * @return A custom string that uniquely identifies the game server. Game server IDs are developer-defined and are
     *         unique across all game server groups in an AWS account.
     */
    public String gameServerId() {
        return gameServerId;
    }

    /**
     * <p>
     * The unique identifier for the instance where the game server is located.
     * </p>
     * 
     * @return The unique identifier for the instance where the game server is located.
     */
    public String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The port and IP address that must be used to establish a client connection to the game server.
     * </p>
     * 
     * @return The port and IP address that must be used to establish a client connection to the game server.
     */
    public String connectionInfo() {
        return connectionInfo;
    }

    /**
     * <p>
     * A set of custom game server properties, formatted as a single string value. This data is passed to a game client
     * or service in response to requests <a>ListGameServers</a> or <a>ClaimGameServer</a>. This property can be updated
     * using <a>UpdateGameServer</a>.
     * </p>
     * 
     * @return A set of custom game server properties, formatted as a single string value. This data is passed to a game
     *         client or service in response to requests <a>ListGameServers</a> or <a>ClaimGameServer</a>. This property
     *         can be updated using <a>UpdateGameServer</a>.
     */
    public String gameServerData() {
        return gameServerData;
    }

    /**
     * <p>
     * A game server tag that can be used to request sorted lists of game servers when calling <a>ListGameServers</a>.
     * Custom sort keys are developer-defined. This property can be updated using <a>UpdateGameServer</a>.
     * </p>
     * 
     * @return A game server tag that can be used to request sorted lists of game servers when calling
     *         <a>ListGameServers</a>. Custom sort keys are developer-defined. This property can be updated using
     *         <a>UpdateGameServer</a>.
     */
    public String customSortKey() {
        return customSortKey;
    }

    /**
     * <p>
     * Indicates when an available game server has been reserved but has not yet started hosting a game. Once it is
     * claimed, game server remains in CLAIMED status for a maximum of one minute. During this time, game clients must
     * connect to the game server and start the game, which triggers the game server to update its utilization status.
     * After one minute, the game server claim status reverts to null.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #claimStatus} will
     * return {@link GameServerClaimStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #claimStatusAsString}.
     * </p>
     * 
     * @return Indicates when an available game server has been reserved but has not yet started hosting a game. Once it
     *         is claimed, game server remains in CLAIMED status for a maximum of one minute. During this time, game
     *         clients must connect to the game server and start the game, which triggers the game server to update its
     *         utilization status. After one minute, the game server claim status reverts to null.
     * @see GameServerClaimStatus
     */
    public GameServerClaimStatus claimStatus() {
        return GameServerClaimStatus.fromValue(claimStatus);
    }

    /**
     * <p>
     * Indicates when an available game server has been reserved but has not yet started hosting a game. Once it is
     * claimed, game server remains in CLAIMED status for a maximum of one minute. During this time, game clients must
     * connect to the game server and start the game, which triggers the game server to update its utilization status.
     * After one minute, the game server claim status reverts to null.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #claimStatus} will
     * return {@link GameServerClaimStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #claimStatusAsString}.
     * </p>
     * 
     * @return Indicates when an available game server has been reserved but has not yet started hosting a game. Once it
     *         is claimed, game server remains in CLAIMED status for a maximum of one minute. During this time, game
     *         clients must connect to the game server and start the game, which triggers the game server to update its
     *         utilization status. After one minute, the game server claim status reverts to null.
     * @see GameServerClaimStatus
     */
    public String claimStatusAsString() {
        return claimStatus;
    }

    /**
     * <p>
     * Indicates whether the game server is currently available for new games or is busy. Possible statuses include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * AVAILABLE - The game server is available to be claimed. A game server that has been claimed remains in this
     * status until it reports game hosting activity.
     * </p>
     * </li>
     * <li>
     * <p>
     * IN_USE - The game server is currently hosting a game session with players.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #utilizationStatus}
     * will return {@link GameServerUtilizationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #utilizationStatusAsString}.
     * </p>
     * 
     * @return Indicates whether the game server is currently available for new games or is busy. Possible statuses
     *         include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         AVAILABLE - The game server is available to be claimed. A game server that has been claimed remains in
     *         this status until it reports game hosting activity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IN_USE - The game server is currently hosting a game session with players.
     *         </p>
     *         </li>
     * @see GameServerUtilizationStatus
     */
    public GameServerUtilizationStatus utilizationStatus() {
        return GameServerUtilizationStatus.fromValue(utilizationStatus);
    }

    /**
     * <p>
     * Indicates whether the game server is currently available for new games or is busy. Possible statuses include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * AVAILABLE - The game server is available to be claimed. A game server that has been claimed remains in this
     * status until it reports game hosting activity.
     * </p>
     * </li>
     * <li>
     * <p>
     * IN_USE - The game server is currently hosting a game session with players.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #utilizationStatus}
     * will return {@link GameServerUtilizationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #utilizationStatusAsString}.
     * </p>
     * 
     * @return Indicates whether the game server is currently available for new games or is busy. Possible statuses
     *         include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         AVAILABLE - The game server is available to be claimed. A game server that has been claimed remains in
     *         this status until it reports game hosting activity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IN_USE - The game server is currently hosting a game session with players.
     *         </p>
     *         </li>
     * @see GameServerUtilizationStatus
     */
    public String utilizationStatusAsString() {
        return utilizationStatus;
    }

    /**
     * <p>
     * Time stamp indicating when the game server resource was created with a <a>RegisterGameServer</a> request. Format
     * is a number expressed in Unix time as milliseconds (for example "1469498468.057").
     * </p>
     * 
     * @return Time stamp indicating when the game server resource was created with a <a>RegisterGameServer</a> request.
     *         Format is a number expressed in Unix time as milliseconds (for example "1469498468.057").
     */
    public Instant registrationTime() {
        return registrationTime;
    }

    /**
     * <p>
     * Time stamp indicating the last time the game server was claimed with a <a>ClaimGameServer</a> request. Format is
     * a number expressed in Unix time as milliseconds (for example "1469498468.057"). This value is used to calculate
     * when the game server's claim status.
     * </p>
     * 
     * @return Time stamp indicating the last time the game server was claimed with a <a>ClaimGameServer</a> request.
     *         Format is a number expressed in Unix time as milliseconds (for example "1469498468.057"). This value is
     *         used to calculate when the game server's claim status.
     */
    public Instant lastClaimTime() {
        return lastClaimTime;
    }

    /**
     * <p>
     * Time stamp indicating the last time the game server was updated with health status using an
     * <a>UpdateGameServer</a> request. Format is a number expressed in Unix time as milliseconds (for example
     * "1469498468.057"). After game server registration, this property is only changed when a game server update
     * specifies a health check value.
     * </p>
     * 
     * @return Time stamp indicating the last time the game server was updated with health status using an
     *         <a>UpdateGameServer</a> request. Format is a number expressed in Unix time as milliseconds (for example
     *         "1469498468.057"). After game server registration, this property is only changed when a game server
     *         update specifies a health check value.
     */
    public Instant lastHealthCheckTime() {
        return lastHealthCheckTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(gameServerGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(connectionInfo());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerData());
        hashCode = 31 * hashCode + Objects.hashCode(customSortKey());
        hashCode = 31 * hashCode + Objects.hashCode(claimStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(utilizationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(registrationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastClaimTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastHealthCheckTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GameServer)) {
            return false;
        }
        GameServer other = (GameServer) obj;
        return Objects.equals(gameServerGroupName(), other.gameServerGroupName())
                && Objects.equals(gameServerGroupArn(), other.gameServerGroupArn())
                && Objects.equals(gameServerId(), other.gameServerId()) && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(connectionInfo(), other.connectionInfo())
                && Objects.equals(gameServerData(), other.gameServerData())
                && Objects.equals(customSortKey(), other.customSortKey())
                && Objects.equals(claimStatusAsString(), other.claimStatusAsString())
                && Objects.equals(utilizationStatusAsString(), other.utilizationStatusAsString())
                && Objects.equals(registrationTime(), other.registrationTime())
                && Objects.equals(lastClaimTime(), other.lastClaimTime())
                && Objects.equals(lastHealthCheckTime(), other.lastHealthCheckTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GameServer").add("GameServerGroupName", gameServerGroupName())
                .add("GameServerGroupArn", gameServerGroupArn()).add("GameServerId", gameServerId())
                .add("InstanceId", instanceId()).add("ConnectionInfo", connectionInfo()).add("GameServerData", gameServerData())
                .add("CustomSortKey", customSortKey()).add("ClaimStatus", claimStatusAsString())
                .add("UtilizationStatus", utilizationStatusAsString()).add("RegistrationTime", registrationTime())
                .add("LastClaimTime", lastClaimTime()).add("LastHealthCheckTime", lastHealthCheckTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "GameServerGroupName":
            return Optional.ofNullable(clazz.cast(gameServerGroupName()));
        case "GameServerGroupArn":
            return Optional.ofNullable(clazz.cast(gameServerGroupArn()));
        case "GameServerId":
            return Optional.ofNullable(clazz.cast(gameServerId()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "ConnectionInfo":
            return Optional.ofNullable(clazz.cast(connectionInfo()));
        case "GameServerData":
            return Optional.ofNullable(clazz.cast(gameServerData()));
        case "CustomSortKey":
            return Optional.ofNullable(clazz.cast(customSortKey()));
        case "ClaimStatus":
            return Optional.ofNullable(clazz.cast(claimStatusAsString()));
        case "UtilizationStatus":
            return Optional.ofNullable(clazz.cast(utilizationStatusAsString()));
        case "RegistrationTime":
            return Optional.ofNullable(clazz.cast(registrationTime()));
        case "LastClaimTime":
            return Optional.ofNullable(clazz.cast(lastClaimTime()));
        case "LastHealthCheckTime":
            return Optional.ofNullable(clazz.cast(lastHealthCheckTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GameServer, T> g) {
        return obj -> g.apply((GameServer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GameServer> {
        /**
         * <p>
         * The name identifier for the game server group where the game server is located.
         * </p>
         * 
         * @param gameServerGroupName
         *        The name identifier for the game server group where the game server is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerGroupName(String gameServerGroupName);

        /**
         * <p>
         * The ARN identifier for the game server group where the game server is located.
         * </p>
         * 
         * @param gameServerGroupArn
         *        The ARN identifier for the game server group where the game server is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerGroupArn(String gameServerGroupArn);

        /**
         * <p>
         * A custom string that uniquely identifies the game server. Game server IDs are developer-defined and are
         * unique across all game server groups in an AWS account.
         * </p>
         * 
         * @param gameServerId
         *        A custom string that uniquely identifies the game server. Game server IDs are developer-defined and
         *        are unique across all game server groups in an AWS account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerId(String gameServerId);

        /**
         * <p>
         * The unique identifier for the instance where the game server is located.
         * </p>
         * 
         * @param instanceId
         *        The unique identifier for the instance where the game server is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The port and IP address that must be used to establish a client connection to the game server.
         * </p>
         * 
         * @param connectionInfo
         *        The port and IP address that must be used to establish a client connection to the game server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionInfo(String connectionInfo);

        /**
         * <p>
         * A set of custom game server properties, formatted as a single string value. This data is passed to a game
         * client or service in response to requests <a>ListGameServers</a> or <a>ClaimGameServer</a>. This property can
         * be updated using <a>UpdateGameServer</a>.
         * </p>
         * 
         * @param gameServerData
         *        A set of custom game server properties, formatted as a single string value. This data is passed to a
         *        game client or service in response to requests <a>ListGameServers</a> or <a>ClaimGameServer</a>. This
         *        property can be updated using <a>UpdateGameServer</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerData(String gameServerData);

        /**
         * <p>
         * A game server tag that can be used to request sorted lists of game servers when calling
         * <a>ListGameServers</a>. Custom sort keys are developer-defined. This property can be updated using
         * <a>UpdateGameServer</a>.
         * </p>
         * 
         * @param customSortKey
         *        A game server tag that can be used to request sorted lists of game servers when calling
         *        <a>ListGameServers</a>. Custom sort keys are developer-defined. This property can be updated using
         *        <a>UpdateGameServer</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customSortKey(String customSortKey);

        /**
         * <p>
         * Indicates when an available game server has been reserved but has not yet started hosting a game. Once it is
         * claimed, game server remains in CLAIMED status for a maximum of one minute. During this time, game clients
         * must connect to the game server and start the game, which triggers the game server to update its utilization
         * status. After one minute, the game server claim status reverts to null.
         * </p>
         * 
         * @param claimStatus
         *        Indicates when an available game server has been reserved but has not yet started hosting a game. Once
         *        it is claimed, game server remains in CLAIMED status for a maximum of one minute. During this time,
         *        game clients must connect to the game server and start the game, which triggers the game server to
         *        update its utilization status. After one minute, the game server claim status reverts to null.
         * @see GameServerClaimStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerClaimStatus
         */
        Builder claimStatus(String claimStatus);

        /**
         * <p>
         * Indicates when an available game server has been reserved but has not yet started hosting a game. Once it is
         * claimed, game server remains in CLAIMED status for a maximum of one minute. During this time, game clients
         * must connect to the game server and start the game, which triggers the game server to update its utilization
         * status. After one minute, the game server claim status reverts to null.
         * </p>
         * 
         * @param claimStatus
         *        Indicates when an available game server has been reserved but has not yet started hosting a game. Once
         *        it is claimed, game server remains in CLAIMED status for a maximum of one minute. During this time,
         *        game clients must connect to the game server and start the game, which triggers the game server to
         *        update its utilization status. After one minute, the game server claim status reverts to null.
         * @see GameServerClaimStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerClaimStatus
         */
        Builder claimStatus(GameServerClaimStatus claimStatus);

        /**
         * <p>
         * Indicates whether the game server is currently available for new games or is busy. Possible statuses include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * AVAILABLE - The game server is available to be claimed. A game server that has been claimed remains in this
         * status until it reports game hosting activity.
         * </p>
         * </li>
         * <li>
         * <p>
         * IN_USE - The game server is currently hosting a game session with players.
         * </p>
         * </li>
         * </ul>
         * 
         * @param utilizationStatus
         *        Indicates whether the game server is currently available for new games or is busy. Possible statuses
         *        include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        AVAILABLE - The game server is available to be claimed. A game server that has been claimed remains in
         *        this status until it reports game hosting activity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IN_USE - The game server is currently hosting a game session with players.
         *        </p>
         *        </li>
         * @see GameServerUtilizationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerUtilizationStatus
         */
        Builder utilizationStatus(String utilizationStatus);

        /**
         * <p>
         * Indicates whether the game server is currently available for new games or is busy. Possible statuses include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * AVAILABLE - The game server is available to be claimed. A game server that has been claimed remains in this
         * status until it reports game hosting activity.
         * </p>
         * </li>
         * <li>
         * <p>
         * IN_USE - The game server is currently hosting a game session with players.
         * </p>
         * </li>
         * </ul>
         * 
         * @param utilizationStatus
         *        Indicates whether the game server is currently available for new games or is busy. Possible statuses
         *        include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        AVAILABLE - The game server is available to be claimed. A game server that has been claimed remains in
         *        this status until it reports game hosting activity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IN_USE - The game server is currently hosting a game session with players.
         *        </p>
         *        </li>
         * @see GameServerUtilizationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerUtilizationStatus
         */
        Builder utilizationStatus(GameServerUtilizationStatus utilizationStatus);

        /**
         * <p>
         * Time stamp indicating when the game server resource was created with a <a>RegisterGameServer</a> request.
         * Format is a number expressed in Unix time as milliseconds (for example "1469498468.057").
         * </p>
         * 
         * @param registrationTime
         *        Time stamp indicating when the game server resource was created with a <a>RegisterGameServer</a>
         *        request. Format is a number expressed in Unix time as milliseconds (for example "1469498468.057").
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registrationTime(Instant registrationTime);

        /**
         * <p>
         * Time stamp indicating the last time the game server was claimed with a <a>ClaimGameServer</a> request. Format
         * is a number expressed in Unix time as milliseconds (for example "1469498468.057"). This value is used to
         * calculate when the game server's claim status.
         * </p>
         * 
         * @param lastClaimTime
         *        Time stamp indicating the last time the game server was claimed with a <a>ClaimGameServer</a> request.
         *        Format is a number expressed in Unix time as milliseconds (for example "1469498468.057"). This value
         *        is used to calculate when the game server's claim status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastClaimTime(Instant lastClaimTime);

        /**
         * <p>
         * Time stamp indicating the last time the game server was updated with health status using an
         * <a>UpdateGameServer</a> request. Format is a number expressed in Unix time as milliseconds (for example
         * "1469498468.057"). After game server registration, this property is only changed when a game server update
         * specifies a health check value.
         * </p>
         * 
         * @param lastHealthCheckTime
         *        Time stamp indicating the last time the game server was updated with health status using an
         *        <a>UpdateGameServer</a> request. Format is a number expressed in Unix time as milliseconds (for
         *        example "1469498468.057"). After game server registration, this property is only changed when a game
         *        server update specifies a health check value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastHealthCheckTime(Instant lastHealthCheckTime);
    }

    static final class BuilderImpl implements Builder {
        private String gameServerGroupName;

        private String gameServerGroupArn;

        private String gameServerId;

        private String instanceId;

        private String connectionInfo;

        private String gameServerData;

        private String customSortKey;

        private String claimStatus;

        private String utilizationStatus;

        private Instant registrationTime;

        private Instant lastClaimTime;

        private Instant lastHealthCheckTime;

        private BuilderImpl() {
        }

        private BuilderImpl(GameServer model) {
            gameServerGroupName(model.gameServerGroupName);
            gameServerGroupArn(model.gameServerGroupArn);
            gameServerId(model.gameServerId);
            instanceId(model.instanceId);
            connectionInfo(model.connectionInfo);
            gameServerData(model.gameServerData);
            customSortKey(model.customSortKey);
            claimStatus(model.claimStatus);
            utilizationStatus(model.utilizationStatus);
            registrationTime(model.registrationTime);
            lastClaimTime(model.lastClaimTime);
            lastHealthCheckTime(model.lastHealthCheckTime);
        }

        public final String getGameServerGroupName() {
            return gameServerGroupName;
        }

        @Override
        public final Builder gameServerGroupName(String gameServerGroupName) {
            this.gameServerGroupName = gameServerGroupName;
            return this;
        }

        public final void setGameServerGroupName(String gameServerGroupName) {
            this.gameServerGroupName = gameServerGroupName;
        }

        public final String getGameServerGroupArn() {
            return gameServerGroupArn;
        }

        @Override
        public final Builder gameServerGroupArn(String gameServerGroupArn) {
            this.gameServerGroupArn = gameServerGroupArn;
            return this;
        }

        public final void setGameServerGroupArn(String gameServerGroupArn) {
            this.gameServerGroupArn = gameServerGroupArn;
        }

        public final String getGameServerId() {
            return gameServerId;
        }

        @Override
        public final Builder gameServerId(String gameServerId) {
            this.gameServerId = gameServerId;
            return this;
        }

        public final void setGameServerId(String gameServerId) {
            this.gameServerId = gameServerId;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        public final String getConnectionInfo() {
            return connectionInfo;
        }

        @Override
        public final Builder connectionInfo(String connectionInfo) {
            this.connectionInfo = connectionInfo;
            return this;
        }

        public final void setConnectionInfo(String connectionInfo) {
            this.connectionInfo = connectionInfo;
        }

        public final String getGameServerData() {
            return gameServerData;
        }

        @Override
        public final Builder gameServerData(String gameServerData) {
            this.gameServerData = gameServerData;
            return this;
        }

        public final void setGameServerData(String gameServerData) {
            this.gameServerData = gameServerData;
        }

        public final String getCustomSortKey() {
            return customSortKey;
        }

        @Override
        public final Builder customSortKey(String customSortKey) {
            this.customSortKey = customSortKey;
            return this;
        }

        public final void setCustomSortKey(String customSortKey) {
            this.customSortKey = customSortKey;
        }

        public final String getClaimStatus() {
            return claimStatus;
        }

        @Override
        public final Builder claimStatus(String claimStatus) {
            this.claimStatus = claimStatus;
            return this;
        }

        @Override
        public final Builder claimStatus(GameServerClaimStatus claimStatus) {
            this.claimStatus(claimStatus == null ? null : claimStatus.toString());
            return this;
        }

        public final void setClaimStatus(String claimStatus) {
            this.claimStatus = claimStatus;
        }

        public final String getUtilizationStatus() {
            return utilizationStatus;
        }

        @Override
        public final Builder utilizationStatus(String utilizationStatus) {
            this.utilizationStatus = utilizationStatus;
            return this;
        }

        @Override
        public final Builder utilizationStatus(GameServerUtilizationStatus utilizationStatus) {
            this.utilizationStatus(utilizationStatus == null ? null : utilizationStatus.toString());
            return this;
        }

        public final void setUtilizationStatus(String utilizationStatus) {
            this.utilizationStatus = utilizationStatus;
        }

        public final Instant getRegistrationTime() {
            return registrationTime;
        }

        @Override
        public final Builder registrationTime(Instant registrationTime) {
            this.registrationTime = registrationTime;
            return this;
        }

        public final void setRegistrationTime(Instant registrationTime) {
            this.registrationTime = registrationTime;
        }

        public final Instant getLastClaimTime() {
            return lastClaimTime;
        }

        @Override
        public final Builder lastClaimTime(Instant lastClaimTime) {
            this.lastClaimTime = lastClaimTime;
            return this;
        }

        public final void setLastClaimTime(Instant lastClaimTime) {
            this.lastClaimTime = lastClaimTime;
        }

        public final Instant getLastHealthCheckTime() {
            return lastHealthCheckTime;
        }

        @Override
        public final Builder lastHealthCheckTime(Instant lastHealthCheckTime) {
            this.lastHealthCheckTime = lastHealthCheckTime;
            return this;
        }

        public final void setLastHealthCheckTime(Instant lastHealthCheckTime) {
            this.lastHealthCheckTime = lastHealthCheckTime;
        }

        @Override
        public GameServer build() {
            return new GameServer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
