/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A range of IP addresses and port settings that allow inbound traffic to connect to server processes on an Amazon
 * GameLift hosting resource. New game sessions that are started on the fleet are assigned an IP address/port number
 * combination, which must fall into the fleet's allowed ranges. For fleets created with a custom game server, the
 * ranges reflect the server's game session assignments. For Realtime Servers fleets, Amazon GameLift automatically
 * opens two port ranges, one for TCP messaging and one for UDP for use by the Realtime servers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpPermission implements SdkPojo, Serializable, ToCopyableBuilder<IpPermission.Builder, IpPermission> {
    private static final SdkField<Integer> FROM_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(IpPermission::fromPort)).setter(setter(Builder::fromPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FromPort").build()).build();

    private static final SdkField<Integer> TO_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(IpPermission::toPort)).setter(setter(Builder::toPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ToPort").build()).build();

    private static final SdkField<String> IP_RANGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(IpPermission::ipRange)).setter(setter(Builder::ipRange))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpRange").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(IpPermission::protocolAsString)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocol").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FROM_PORT_FIELD,
            TO_PORT_FIELD, IP_RANGE_FIELD, PROTOCOL_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer fromPort;

    private final Integer toPort;

    private final String ipRange;

    private final String protocol;

    private IpPermission(BuilderImpl builder) {
        this.fromPort = builder.fromPort;
        this.toPort = builder.toPort;
        this.ipRange = builder.ipRange;
        this.protocol = builder.protocol;
    }

    /**
     * <p>
     * A starting value for a range of allowed port numbers.
     * </p>
     * 
     * @return A starting value for a range of allowed port numbers.
     */
    public Integer fromPort() {
        return fromPort;
    }

    /**
     * <p>
     * An ending value for a range of allowed port numbers. Port numbers are end-inclusive. This value must be higher
     * than <code>FromPort</code>.
     * </p>
     * 
     * @return An ending value for a range of allowed port numbers. Port numbers are end-inclusive. This value must be
     *         higher than <code>FromPort</code>.
     */
    public Integer toPort() {
        return toPort;
    }

    /**
     * <p>
     * A range of allowed IP addresses. This value must be expressed in CIDR notation. Example: "
     * <code>000.000.000.000/[subnet mask]</code>" or optionally the shortened version "
     * <code>0.0.0.0/[subnet mask]</code>".
     * </p>
     * 
     * @return A range of allowed IP addresses. This value must be expressed in CIDR notation. Example: "
     *         <code>000.000.000.000/[subnet mask]</code>" or optionally the shortened version "
     *         <code>0.0.0.0/[subnet mask]</code>".
     */
    public String ipRange() {
        return ipRange;
    }

    /**
     * <p>
     * The network communication protocol used by the fleet.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link IpProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The network communication protocol used by the fleet.
     * @see IpProtocol
     */
    public IpProtocol protocol() {
        return IpProtocol.fromValue(protocol);
    }

    /**
     * <p>
     * The network communication protocol used by the fleet.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link IpProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The network communication protocol used by the fleet.
     * @see IpProtocol
     */
    public String protocolAsString() {
        return protocol;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fromPort());
        hashCode = 31 * hashCode + Objects.hashCode(toPort());
        hashCode = 31 * hashCode + Objects.hashCode(ipRange());
        hashCode = 31 * hashCode + Objects.hashCode(protocolAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpPermission)) {
            return false;
        }
        IpPermission other = (IpPermission) obj;
        return Objects.equals(fromPort(), other.fromPort()) && Objects.equals(toPort(), other.toPort())
                && Objects.equals(ipRange(), other.ipRange()) && Objects.equals(protocolAsString(), other.protocolAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("IpPermission").add("FromPort", fromPort()).add("ToPort", toPort()).add("IpRange", ipRange())
                .add("Protocol", protocolAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FromPort":
            return Optional.ofNullable(clazz.cast(fromPort()));
        case "ToPort":
            return Optional.ofNullable(clazz.cast(toPort()));
        case "IpRange":
            return Optional.ofNullable(clazz.cast(ipRange()));
        case "Protocol":
            return Optional.ofNullable(clazz.cast(protocolAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IpPermission, T> g) {
        return obj -> g.apply((IpPermission) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpPermission> {
        /**
         * <p>
         * A starting value for a range of allowed port numbers.
         * </p>
         * 
         * @param fromPort
         *        A starting value for a range of allowed port numbers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromPort(Integer fromPort);

        /**
         * <p>
         * An ending value for a range of allowed port numbers. Port numbers are end-inclusive. This value must be
         * higher than <code>FromPort</code>.
         * </p>
         * 
         * @param toPort
         *        An ending value for a range of allowed port numbers. Port numbers are end-inclusive. This value must
         *        be higher than <code>FromPort</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder toPort(Integer toPort);

        /**
         * <p>
         * A range of allowed IP addresses. This value must be expressed in CIDR notation. Example: "
         * <code>000.000.000.000/[subnet mask]</code>" or optionally the shortened version "
         * <code>0.0.0.0/[subnet mask]</code>".
         * </p>
         * 
         * @param ipRange
         *        A range of allowed IP addresses. This value must be expressed in CIDR notation. Example: "
         *        <code>000.000.000.000/[subnet mask]</code>" or optionally the shortened version "
         *        <code>0.0.0.0/[subnet mask]</code>".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipRange(String ipRange);

        /**
         * <p>
         * The network communication protocol used by the fleet.
         * </p>
         * 
         * @param protocol
         *        The network communication protocol used by the fleet.
         * @see IpProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpProtocol
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The network communication protocol used by the fleet.
         * </p>
         * 
         * @param protocol
         *        The network communication protocol used by the fleet.
         * @see IpProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpProtocol
         */
        Builder protocol(IpProtocol protocol);
    }

    static final class BuilderImpl implements Builder {
        private Integer fromPort;

        private Integer toPort;

        private String ipRange;

        private String protocol;

        private BuilderImpl() {
        }

        private BuilderImpl(IpPermission model) {
            fromPort(model.fromPort);
            toPort(model.toPort);
            ipRange(model.ipRange);
            protocol(model.protocol);
        }

        public final Integer getFromPort() {
            return fromPort;
        }

        @Override
        public final Builder fromPort(Integer fromPort) {
            this.fromPort = fromPort;
            return this;
        }

        public final void setFromPort(Integer fromPort) {
            this.fromPort = fromPort;
        }

        public final Integer getToPort() {
            return toPort;
        }

        @Override
        public final Builder toPort(Integer toPort) {
            this.toPort = toPort;
            return this;
        }

        public final void setToPort(Integer toPort) {
            this.toPort = toPort;
        }

        public final String getIpRange() {
            return ipRange;
        }

        @Override
        public final Builder ipRange(String ipRange) {
            this.ipRange = ipRange;
            return this;
        }

        public final void setIpRange(String ipRange) {
            this.ipRange = ipRange;
        }

        public final String getProtocol() {
            return protocol;
        }

        @Override
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        @Override
        public final Builder protocol(IpProtocol protocol) {
            this.protocol(protocol == null ? null : protocol.toString());
            return this;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        @Override
        public IpPermission build() {
            return new IpPermission(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
