/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.gamelift.model.AcceptMatchRequest;
import software.amazon.awssdk.services.gamelift.model.AcceptMatchResponse;
import software.amazon.awssdk.services.gamelift.model.ClaimGameServerRequest;
import software.amazon.awssdk.services.gamelift.model.ClaimGameServerResponse;
import software.amazon.awssdk.services.gamelift.model.ConflictException;
import software.amazon.awssdk.services.gamelift.model.CreateAliasRequest;
import software.amazon.awssdk.services.gamelift.model.CreateAliasResponse;
import software.amazon.awssdk.services.gamelift.model.CreateBuildRequest;
import software.amazon.awssdk.services.gamelift.model.CreateBuildResponse;
import software.amazon.awssdk.services.gamelift.model.CreateFleetRequest;
import software.amazon.awssdk.services.gamelift.model.CreateFleetResponse;
import software.amazon.awssdk.services.gamelift.model.CreateGameServerGroupRequest;
import software.amazon.awssdk.services.gamelift.model.CreateGameServerGroupResponse;
import software.amazon.awssdk.services.gamelift.model.CreateGameSessionQueueRequest;
import software.amazon.awssdk.services.gamelift.model.CreateGameSessionQueueResponse;
import software.amazon.awssdk.services.gamelift.model.CreateGameSessionRequest;
import software.amazon.awssdk.services.gamelift.model.CreateGameSessionResponse;
import software.amazon.awssdk.services.gamelift.model.CreateMatchmakingConfigurationRequest;
import software.amazon.awssdk.services.gamelift.model.CreateMatchmakingConfigurationResponse;
import software.amazon.awssdk.services.gamelift.model.CreateMatchmakingRuleSetRequest;
import software.amazon.awssdk.services.gamelift.model.CreateMatchmakingRuleSetResponse;
import software.amazon.awssdk.services.gamelift.model.CreatePlayerSessionRequest;
import software.amazon.awssdk.services.gamelift.model.CreatePlayerSessionResponse;
import software.amazon.awssdk.services.gamelift.model.CreatePlayerSessionsRequest;
import software.amazon.awssdk.services.gamelift.model.CreatePlayerSessionsResponse;
import software.amazon.awssdk.services.gamelift.model.CreateScriptRequest;
import software.amazon.awssdk.services.gamelift.model.CreateScriptResponse;
import software.amazon.awssdk.services.gamelift.model.CreateVpcPeeringAuthorizationRequest;
import software.amazon.awssdk.services.gamelift.model.CreateVpcPeeringAuthorizationResponse;
import software.amazon.awssdk.services.gamelift.model.CreateVpcPeeringConnectionRequest;
import software.amazon.awssdk.services.gamelift.model.CreateVpcPeeringConnectionResponse;
import software.amazon.awssdk.services.gamelift.model.DeleteAliasRequest;
import software.amazon.awssdk.services.gamelift.model.DeleteAliasResponse;
import software.amazon.awssdk.services.gamelift.model.DeleteBuildRequest;
import software.amazon.awssdk.services.gamelift.model.DeleteBuildResponse;
import software.amazon.awssdk.services.gamelift.model.DeleteFleetRequest;
import software.amazon.awssdk.services.gamelift.model.DeleteFleetResponse;
import software.amazon.awssdk.services.gamelift.model.DeleteGameServerGroupRequest;
import software.amazon.awssdk.services.gamelift.model.DeleteGameServerGroupResponse;
import software.amazon.awssdk.services.gamelift.model.DeleteGameSessionQueueRequest;
import software.amazon.awssdk.services.gamelift.model.DeleteGameSessionQueueResponse;
import software.amazon.awssdk.services.gamelift.model.DeleteMatchmakingConfigurationRequest;
import software.amazon.awssdk.services.gamelift.model.DeleteMatchmakingConfigurationResponse;
import software.amazon.awssdk.services.gamelift.model.DeleteMatchmakingRuleSetRequest;
import software.amazon.awssdk.services.gamelift.model.DeleteMatchmakingRuleSetResponse;
import software.amazon.awssdk.services.gamelift.model.DeleteScalingPolicyRequest;
import software.amazon.awssdk.services.gamelift.model.DeleteScalingPolicyResponse;
import software.amazon.awssdk.services.gamelift.model.DeleteScriptRequest;
import software.amazon.awssdk.services.gamelift.model.DeleteScriptResponse;
import software.amazon.awssdk.services.gamelift.model.DeleteVpcPeeringAuthorizationRequest;
import software.amazon.awssdk.services.gamelift.model.DeleteVpcPeeringAuthorizationResponse;
import software.amazon.awssdk.services.gamelift.model.DeleteVpcPeeringConnectionRequest;
import software.amazon.awssdk.services.gamelift.model.DeleteVpcPeeringConnectionResponse;
import software.amazon.awssdk.services.gamelift.model.DeregisterGameServerRequest;
import software.amazon.awssdk.services.gamelift.model.DeregisterGameServerResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeAliasRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeAliasResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeBuildRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeBuildResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeEc2InstanceLimitsRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeEc2InstanceLimitsResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeFleetAttributesRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeFleetAttributesResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeFleetCapacityRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeFleetCapacityResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeFleetEventsRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeFleetEventsResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeFleetPortSettingsRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeFleetPortSettingsResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeFleetUtilizationRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeFleetUtilizationResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeGameServerGroupRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeGameServerGroupResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeGameServerInstancesRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeGameServerInstancesResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeGameServerRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeGameServerResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeGameSessionDetailsRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeGameSessionDetailsResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeGameSessionPlacementRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeGameSessionPlacementResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeGameSessionQueuesRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeGameSessionQueuesResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeGameSessionsRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeGameSessionsResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeInstancesRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeInstancesResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingConfigurationsRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingConfigurationsResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingRuleSetsRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingRuleSetsResponse;
import software.amazon.awssdk.services.gamelift.model.DescribePlayerSessionsRequest;
import software.amazon.awssdk.services.gamelift.model.DescribePlayerSessionsResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeRuntimeConfigurationRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeRuntimeConfigurationResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeScalingPoliciesRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeScalingPoliciesResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeScriptRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeScriptResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeVpcPeeringAuthorizationsRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeVpcPeeringAuthorizationsResponse;
import software.amazon.awssdk.services.gamelift.model.DescribeVpcPeeringConnectionsRequest;
import software.amazon.awssdk.services.gamelift.model.DescribeVpcPeeringConnectionsResponse;
import software.amazon.awssdk.services.gamelift.model.FleetCapacityExceededException;
import software.amazon.awssdk.services.gamelift.model.GameLiftException;
import software.amazon.awssdk.services.gamelift.model.GameLiftRequest;
import software.amazon.awssdk.services.gamelift.model.GameSessionFullException;
import software.amazon.awssdk.services.gamelift.model.GetGameSessionLogUrlRequest;
import software.amazon.awssdk.services.gamelift.model.GetGameSessionLogUrlResponse;
import software.amazon.awssdk.services.gamelift.model.GetInstanceAccessRequest;
import software.amazon.awssdk.services.gamelift.model.GetInstanceAccessResponse;
import software.amazon.awssdk.services.gamelift.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.gamelift.model.InternalServiceException;
import software.amazon.awssdk.services.gamelift.model.InvalidFleetStatusException;
import software.amazon.awssdk.services.gamelift.model.InvalidGameSessionStatusException;
import software.amazon.awssdk.services.gamelift.model.InvalidRequestException;
import software.amazon.awssdk.services.gamelift.model.LimitExceededException;
import software.amazon.awssdk.services.gamelift.model.ListAliasesRequest;
import software.amazon.awssdk.services.gamelift.model.ListAliasesResponse;
import software.amazon.awssdk.services.gamelift.model.ListBuildsRequest;
import software.amazon.awssdk.services.gamelift.model.ListBuildsResponse;
import software.amazon.awssdk.services.gamelift.model.ListFleetsRequest;
import software.amazon.awssdk.services.gamelift.model.ListFleetsResponse;
import software.amazon.awssdk.services.gamelift.model.ListGameServerGroupsRequest;
import software.amazon.awssdk.services.gamelift.model.ListGameServerGroupsResponse;
import software.amazon.awssdk.services.gamelift.model.ListGameServersRequest;
import software.amazon.awssdk.services.gamelift.model.ListGameServersResponse;
import software.amazon.awssdk.services.gamelift.model.ListScriptsRequest;
import software.amazon.awssdk.services.gamelift.model.ListScriptsResponse;
import software.amazon.awssdk.services.gamelift.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.gamelift.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.gamelift.model.NotFoundException;
import software.amazon.awssdk.services.gamelift.model.OutOfCapacityException;
import software.amazon.awssdk.services.gamelift.model.PutScalingPolicyRequest;
import software.amazon.awssdk.services.gamelift.model.PutScalingPolicyResponse;
import software.amazon.awssdk.services.gamelift.model.RegisterGameServerRequest;
import software.amazon.awssdk.services.gamelift.model.RegisterGameServerResponse;
import software.amazon.awssdk.services.gamelift.model.RequestUploadCredentialsRequest;
import software.amazon.awssdk.services.gamelift.model.RequestUploadCredentialsResponse;
import software.amazon.awssdk.services.gamelift.model.ResolveAliasRequest;
import software.amazon.awssdk.services.gamelift.model.ResolveAliasResponse;
import software.amazon.awssdk.services.gamelift.model.ResumeGameServerGroupRequest;
import software.amazon.awssdk.services.gamelift.model.ResumeGameServerGroupResponse;
import software.amazon.awssdk.services.gamelift.model.SearchGameSessionsRequest;
import software.amazon.awssdk.services.gamelift.model.SearchGameSessionsResponse;
import software.amazon.awssdk.services.gamelift.model.StartFleetActionsRequest;
import software.amazon.awssdk.services.gamelift.model.StartFleetActionsResponse;
import software.amazon.awssdk.services.gamelift.model.StartGameSessionPlacementRequest;
import software.amazon.awssdk.services.gamelift.model.StartGameSessionPlacementResponse;
import software.amazon.awssdk.services.gamelift.model.StartMatchBackfillRequest;
import software.amazon.awssdk.services.gamelift.model.StartMatchBackfillResponse;
import software.amazon.awssdk.services.gamelift.model.StartMatchmakingRequest;
import software.amazon.awssdk.services.gamelift.model.StartMatchmakingResponse;
import software.amazon.awssdk.services.gamelift.model.StopFleetActionsRequest;
import software.amazon.awssdk.services.gamelift.model.StopFleetActionsResponse;
import software.amazon.awssdk.services.gamelift.model.StopGameSessionPlacementRequest;
import software.amazon.awssdk.services.gamelift.model.StopGameSessionPlacementResponse;
import software.amazon.awssdk.services.gamelift.model.StopMatchmakingRequest;
import software.amazon.awssdk.services.gamelift.model.StopMatchmakingResponse;
import software.amazon.awssdk.services.gamelift.model.SuspendGameServerGroupRequest;
import software.amazon.awssdk.services.gamelift.model.SuspendGameServerGroupResponse;
import software.amazon.awssdk.services.gamelift.model.TagResourceRequest;
import software.amazon.awssdk.services.gamelift.model.TagResourceResponse;
import software.amazon.awssdk.services.gamelift.model.TaggingFailedException;
import software.amazon.awssdk.services.gamelift.model.TerminalRoutingStrategyException;
import software.amazon.awssdk.services.gamelift.model.UnauthorizedException;
import software.amazon.awssdk.services.gamelift.model.UnsupportedRegionException;
import software.amazon.awssdk.services.gamelift.model.UntagResourceRequest;
import software.amazon.awssdk.services.gamelift.model.UntagResourceResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateAliasRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateAliasResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateBuildRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateBuildResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateFleetAttributesRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateFleetAttributesResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateFleetCapacityRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateFleetCapacityResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateFleetPortSettingsRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateFleetPortSettingsResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateGameServerGroupRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateGameServerGroupResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateGameServerRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateGameServerResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateGameSessionQueueRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateGameSessionQueueResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateGameSessionRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateGameSessionResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateMatchmakingConfigurationRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateMatchmakingConfigurationResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateRuntimeConfigurationRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateRuntimeConfigurationResponse;
import software.amazon.awssdk.services.gamelift.model.UpdateScriptRequest;
import software.amazon.awssdk.services.gamelift.model.UpdateScriptResponse;
import software.amazon.awssdk.services.gamelift.model.ValidateMatchmakingRuleSetRequest;
import software.amazon.awssdk.services.gamelift.model.ValidateMatchmakingRuleSetResponse;
import software.amazon.awssdk.services.gamelift.paginators.DescribeFleetAttributesPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribeFleetCapacityPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribeFleetEventsPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribeFleetUtilizationPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribeGameServerInstancesPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribeGameSessionDetailsPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribeGameSessionQueuesPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribeGameSessionsPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribeInstancesPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribeMatchmakingConfigurationsPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribeMatchmakingRuleSetsPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribePlayerSessionsPublisher;
import software.amazon.awssdk.services.gamelift.paginators.DescribeScalingPoliciesPublisher;
import software.amazon.awssdk.services.gamelift.paginators.ListAliasesPublisher;
import software.amazon.awssdk.services.gamelift.paginators.ListBuildsPublisher;
import software.amazon.awssdk.services.gamelift.paginators.ListFleetsPublisher;
import software.amazon.awssdk.services.gamelift.paginators.ListGameServerGroupsPublisher;
import software.amazon.awssdk.services.gamelift.paginators.ListGameServersPublisher;
import software.amazon.awssdk.services.gamelift.paginators.ListScriptsPublisher;
import software.amazon.awssdk.services.gamelift.paginators.SearchGameSessionsPublisher;
import software.amazon.awssdk.services.gamelift.transform.AcceptMatchRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.ClaimGameServerRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreateAliasRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreateBuildRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreateFleetRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreateGameServerGroupRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreateGameSessionQueueRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreateGameSessionRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreateMatchmakingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreateMatchmakingRuleSetRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreatePlayerSessionRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreatePlayerSessionsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreateScriptRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreateVpcPeeringAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.CreateVpcPeeringConnectionRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeleteAliasRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeleteBuildRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeleteFleetRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeleteGameServerGroupRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeleteGameSessionQueueRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeleteMatchmakingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeleteMatchmakingRuleSetRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeleteScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeleteScriptRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeleteVpcPeeringAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeleteVpcPeeringConnectionRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DeregisterGameServerRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeAliasRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeBuildRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeEc2InstanceLimitsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeFleetAttributesRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeFleetCapacityRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeFleetEventsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeFleetPortSettingsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeFleetUtilizationRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeGameServerGroupRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeGameServerInstancesRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeGameServerRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeGameSessionDetailsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeGameSessionPlacementRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeGameSessionQueuesRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeGameSessionsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeInstancesRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeMatchmakingConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeMatchmakingRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeMatchmakingRuleSetsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribePlayerSessionsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeRuntimeConfigurationRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeScalingPoliciesRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeScriptRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeVpcPeeringAuthorizationsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.DescribeVpcPeeringConnectionsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.GetGameSessionLogUrlRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.GetInstanceAccessRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.ListAliasesRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.ListBuildsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.ListFleetsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.ListGameServerGroupsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.ListGameServersRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.ListScriptsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.PutScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.RegisterGameServerRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.RequestUploadCredentialsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.ResolveAliasRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.ResumeGameServerGroupRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.SearchGameSessionsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.StartFleetActionsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.StartGameSessionPlacementRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.StartMatchBackfillRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.StartMatchmakingRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.StopFleetActionsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.StopGameSessionPlacementRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.StopMatchmakingRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.SuspendGameServerGroupRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateAliasRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateBuildRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateFleetAttributesRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateFleetCapacityRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateFleetPortSettingsRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateGameServerGroupRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateGameServerRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateGameSessionQueueRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateGameSessionRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateMatchmakingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateRuntimeConfigurationRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.UpdateScriptRequestMarshaller;
import software.amazon.awssdk.services.gamelift.transform.ValidateMatchmakingRuleSetRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link GameLiftAsyncClient}.
 *
 * @see GameLiftAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGameLiftAsyncClient implements GameLiftAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultGameLiftAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGameLiftAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Registers a player's acceptance or rejection of a proposed FlexMatch match. A matchmaking configuration may
     * require player acceptance; if so, then matches built with that configuration cannot be completed unless all
     * players accept the proposed match within a specified time limit.
     * </p>
     * <p>
     * When FlexMatch builds a match, all the matchmaking tickets involved in the proposed match are placed into status
     * <code>REQUIRES_ACCEPTANCE</code>. This is a trigger for your game to get acceptance from all players in the
     * ticket. Acceptances are only valid for tickets when they are in this status; all other acceptances result in an
     * error.
     * </p>
     * <p>
     * To register acceptance, specify the ticket ID, a response, and one or more players. Once all players have
     * registered acceptance, the matchmaking tickets advance to status <code>PLACING</code>, where a new game session
     * is created for the match.
     * </p>
     * <p>
     * If any player rejects the match, or if acceptances are not received before a specified timeout, the proposed
     * match is dropped. The matchmaking tickets are then handled in one of two ways: For tickets where one or more
     * players rejected the match, the ticket status is returned to <code>SEARCHING</code> to find a new match. For
     * tickets where one or more players failed to respond, the ticket status is set to <code>CANCELLED</code>, and
     * processing is terminated. A new matchmaking request for these players can be submitted as needed.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-client.html"> Add FlexMatch to a Game
     * Client</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-events.html"> FlexMatch Events
     * Reference</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>AcceptMatch</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartMatchBackfill</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param acceptMatchRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the AcceptMatch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.AcceptMatch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/AcceptMatch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptMatchResponse> acceptMatch(AcceptMatchRequest acceptMatchRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptMatchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptMatch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptMatchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AcceptMatchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptMatchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptMatchRequest, AcceptMatchResponse>()
                            .withOperationName("AcceptMatch").withMarshaller(new AcceptMatchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acceptMatchRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = acceptMatchRequest.overrideConfiguration().orElse(null);
            CompletableFuture<AcceptMatchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Locates an available game server and temporarily reserves it to host gameplay and players. This operation is
     * called from a game client or client service (such as a matchmaker) to request hosting resources for a new game
     * session. In response, GameLift FleetIQ locates an available game server, places it in <code>CLAIMED</code> status
     * for 60 seconds, and returns connection information that players can use to connect to the game server.
     * </p>
     * <p>
     * To claim a game server, identify a game server group. You can also specify a game server ID, although this
     * approach bypasses GameLift FleetIQ placement optimization. Optionally, include game data to pass to the game
     * server at the start of a game session, such as a game map or player information.
     * </p>
     * <p>
     * When a game server is successfully claimed, connection information is returned. A claimed game server's
     * utilization status remains <code>AVAILABLE</code> while the claim status is set to <code>CLAIMED</code> for up to
     * 60 seconds. This time period gives the game server time to update its status to <code>UTILIZED</code> (using
     * <a>UpdateGameServer</a>) once players join. If the game server's status is not updated within 60 seconds, the
     * game server reverts to unclaimed status and is available to be claimed by another request. The claim time period
     * is a fixed value and is not configurable.
     * </p>
     * <p>
     * If you try to claim a specific game server, this request will fail in the following cases:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the game server utilization status is <code>UTILIZED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the game server claim status is <code>CLAIMED</code>.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * When claiming a specific game server, this request will succeed even if the game server is running on an instance
     * in <code>DRAINING</code> status. To avoid this, first check the instance status by calling
     * <a>DescribeGameServerInstances</a>.
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>RegisterGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServers</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ClaimGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeregisterGameServer</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param claimGameServerRequest
     * @return A Java Future containing the result of the ClaimGameServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>OutOfCapacityException The specified game server group has no available game servers to fulfill a
     *         <code>ClaimGameServer</code> request. Clients can retry such requests immediately or after a waiting
     *         period.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ClaimGameServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ClaimGameServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ClaimGameServerResponse> claimGameServer(ClaimGameServerRequest claimGameServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, claimGameServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ClaimGameServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ClaimGameServerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ClaimGameServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ClaimGameServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ClaimGameServerRequest, ClaimGameServerResponse>()
                            .withOperationName("ClaimGameServer")
                            .withMarshaller(new ClaimGameServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(claimGameServerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = claimGameServerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ClaimGameServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an alias for a fleet. In most situations, you can use an alias ID in place of a fleet ID. An alias
     * provides a level of abstraction for a fleet that is useful when redirecting player traffic from one fleet to
     * another, such as when updating your game build.
     * </p>
     * <p>
     * Amazon GameLift supports two types of routing strategies for aliases: simple and terminal. A simple alias points
     * to an active fleet. A terminal alias is used to display messaging or link to a URL instead of routing players to
     * an active fleet. For example, you might use a terminal alias when a game version is no longer supported and you
     * want to direct players to an upgrade site.
     * </p>
     * <p>
     * To create a fleet alias, specify an alias name, routing strategy, and optional description. Each simple alias can
     * point to only one fleet, but a fleet can have multiple aliases. If successful, a new alias record is returned,
     * including an alias ID and an ARN. You can reassign an alias to another fleet by calling <code>UpdateAlias</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResolveAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createAliasRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the CreateAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>LimitExceededException The requested operation would cause the resource to exceed the allowed service
     *         limit. Resolve the issue before retrying.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreateAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreateAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAliasResponse> createAlias(CreateAliasRequest createAliasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAliasRequest, CreateAliasResponse>()
                            .withOperationName("CreateAlias").withMarshaller(new CreateAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAliasRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createAliasRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Amazon GameLift build resource for your game server binary files. Game server binaries must be
     * combined into a zip file for use with Amazon GameLift.
     * </p>
     * <important>
     * <p>
     * When setting up a new game build for GameLift, we recommend using the AWS CLI command <b> <a
     * href="https://docs.aws.amazon.com/cli/latest/reference/gamelift/upload-build.html">upload-build</a> </b>. This
     * helper command combines two tasks: (1) it uploads your build files from a file directory to a GameLift Amazon S3
     * location, and (2) it creates a new build resource.
     * </p>
     * </important>
     * <p>
     * The <code>CreateBuild</code> operation can used in the following scenarios:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To create a new game build with build files that are in an S3 location under an AWS account that you control. To
     * use this option, you must first give Amazon GameLift access to the S3 bucket. With permissions in place, call
     * <code>CreateBuild</code> and specify a build name, operating system, and the S3 storage location of your game
     * build.
     * </p>
     * </li>
     * <li>
     * <p>
     * To directly upload your build files to a GameLift S3 location. To use this option, first call
     * <code>CreateBuild</code> and specify a build name and operating system. This operation creates a new build
     * resource and also returns an S3 location with temporary access credentials. Use the credentials to manually
     * upload your build files to the specified S3 location. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/UploadingObjects.html">Uploading Objects</a> in the
     * <i>Amazon S3 Developer Guide</i>. Build files can be uploaded to the GameLift S3 location once only; that can't
     * be updated.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If successful, this operation creates a new build resource with a unique build ID and places it in
     * <code>INITIALIZED</code> status. A build must be in <code>READY</code> status before you can create fleets with
     * it.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-build-intro.html">Uploading Your
     * Game</a>
     * </p>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-build-cli-uploading.html#gamelift-build-cli-uploading-create-build"
     * > Create a Build with Files in Amazon S3</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListBuilds</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteBuild</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createBuildRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the CreateBuild operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreateBuild
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreateBuild" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBuildResponse> createBuild(CreateBuildRequest createBuildRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBuildRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBuild");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBuildResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateBuildResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBuildResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBuildRequest, CreateBuildResponse>()
                            .withOperationName("CreateBuild").withMarshaller(new CreateBuildRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createBuildRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createBuildRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateBuildResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new fleet to run your game servers. whether they are custom game builds or Realtime Servers with
     * game-specific script. A fleet is a set of Amazon Elastic Compute Cloud (Amazon EC2) instances, each of which can
     * host multiple game sessions. When creating a fleet, you choose the hardware specifications, set some
     * configuration options, and specify the game server to deploy on the new fleet.
     * </p>
     * <p>
     * To create a new fleet, provide the following: (1) a fleet name, (2) an EC2 instance type and fleet type (spot or
     * on-demand), (3) the build ID for your game build or script ID if using Realtime Servers, and (4) a runtime
     * configuration, which determines how game servers will run on each instance in the fleet.
     * </p>
     * <p>
     * If the <code>CreateFleet</code> call is successful, Amazon GameLift performs the following tasks. You can track
     * the process of a fleet by checking the fleet status or by monitoring fleet creation events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a fleet resource. Status: <code>NEW</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Begins writing events to the fleet event log, which can be accessed in the Amazon GameLift console.
     * </p>
     * </li>
     * <li>
     * <p>
     * Sets the fleet's target capacity to 1 (desired instances), which triggers Amazon GameLift to start one new EC2
     * instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * Downloads the game build or Realtime script to the new instance and installs it. Statuses:
     * <code>DOWNLOADING</code>, <code>VALIDATING</code>, <code>BUILDING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Starts launching server processes on the instance. If the fleet is configured to run multiple server processes
     * per instance, Amazon GameLift staggers each process launch by a few seconds. Status: <code>ACTIVATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Sets the fleet's status to <code>ACTIVE</code> as soon as one server process is ready to host a game session.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting Up Fleets</a>
     * </p>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html#fleets-creating-debug-creation"
     * >Debug Fleet Creation Issues</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createFleetRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the CreateFleet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>LimitExceededException The requested operation would cause the resource to exceed the allowed service
     *         limit. Resolve the issue before retrying.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreateFleet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreateFleet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFleetResponse> createFleet(CreateFleetRequest createFleetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFleet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFleetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFleetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFleetRequest, CreateFleetResponse>()
                            .withOperationName("CreateFleet").withMarshaller(new CreateFleetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createFleetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createFleetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateFleetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Creates a GameLift FleetIQ game server group for managing game hosting on a collection of Amazon EC2 instances
     * for game hosting. This operation creates the game server group, creates an Auto Scaling group in your AWS
     * account, and establishes a link between the two groups. You can view the status of your game server groups in the
     * GameLift console. Game server group metrics and events are emitted to Amazon CloudWatch.
     * </p>
     * <p>
     * Before creating a new game server group, you must have the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An Amazon EC2 launch template that specifies how to launch Amazon EC2 instances with your game server build. For
     * more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">
     * Launching an Instance from a Launch Template</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * An IAM role that extends limited access to your AWS account to allow GameLift FleetIQ to create and interact with
     * the Auto Scaling group. For more information, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gsg-iam-permissions-roles.html">Create IAM roles
     * for cross-service interaction</a> in the <i>GameLift FleetIQ Developer Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To create a new game server group, specify a unique group name, IAM role and Amazon EC2 launch template, and
     * provide a list of instance types that can be used in the group. You must also set initial maximum and minimum
     * limits on the group's instance count. You can optionally set an Auto Scaling policy with target tracking based on
     * a GameLift FleetIQ metric.
     * </p>
     * <p>
     * Once the game server group and corresponding Auto Scaling group are created, you have full access to change the
     * Auto Scaling group's configuration as needed. Several properties that are set when creating a game server group,
     * including maximum/minimum size and auto-scaling policy settings, must be updated directly in the Auto Scaling
     * group. Keep in mind that some Auto Scaling group properties are periodically updated by GameLift FleetIQ as part
     * of its balancing activities to optimize for availability and cost.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServerGroups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResumeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SuspendGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerInstances</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createGameServerGroupRequest
     * @return A Java Future containing the result of the CreateGameServerGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>LimitExceededException The requested operation would cause the resource to exceed the allowed service
     *         limit. Resolve the issue before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreateGameServerGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreateGameServerGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGameServerGroupResponse> createGameServerGroup(
            CreateGameServerGroupRequest createGameServerGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGameServerGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGameServerGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGameServerGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGameServerGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGameServerGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGameServerGroupRequest, CreateGameServerGroupResponse>()
                            .withOperationName("CreateGameServerGroup")
                            .withMarshaller(new CreateGameServerGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createGameServerGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createGameServerGroupRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateGameServerGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a multiplayer game session for players. This operation creates a game session record and assigns an
     * available server process in the specified fleet to host the game session. A fleet must have an
     * <code>ACTIVE</code> status before a game session can be created in it.
     * </p>
     * <p>
     * To create a game session, specify either fleet ID or alias ID and indicate a maximum number of players to allow
     * in the game session. You can also provide a name and game-specific properties for this game session. If
     * successful, a <a>GameSession</a> object is returned containing the game session properties and other settings you
     * specified.
     * </p>
     * <p>
     * <b>Idempotency tokens.</b> You can add a token that uniquely identifies game session requests. This is useful for
     * ensuring that game session requests are idempotent. Multiple requests with the same idempotency token are
     * processed only once; subsequent requests return the original result. All response values are the same with the
     * exception of game session status, which may change.
     * </p>
     * <p>
     * <b>Resource creation limits.</b> If you are creating a game session on a fleet with a resource creation limit
     * policy in force, then you must specify a creator ID. Without this ID, Amazon GameLift has no way to evaluate the
     * policy for this new game session request.
     * </p>
     * <p>
     * <b>Player acceptance policy.</b> By default, newly created game sessions are open to new players. You can
     * restrict new player access by using <a>UpdateGameSession</a> to change the game session's player session creation
     * policy.
     * </p>
     * <p>
     * <b>Game session logs.</b> Logs are retained for all active game sessions for 14 days. To access the logs, call
     * <a>GetGameSessionLogUrl</a> to download the log files.
     * </p>
     * <p>
     * <i>Available in Amazon GameLift Local.</i>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param createGameSessionRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the CreateGameSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidFleetStatusException The requested operation would cause a conflict with the current state of
     *         a resource associated with the request and/or the fleet. Resolve the conflict before retrying.</li>
     *         <li>TerminalRoutingStrategyException The service is unable to resolve the routing for a particular alias
     *         because it has a terminal <a>RoutingStrategy</a> associated with it. The message returned in this
     *         exception is the message defined in the routing strategy itself. Such requests should only be retried if
     *         the routing strategy for the specified alias is modified.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>FleetCapacityExceededException The specified fleet has no available instances to fulfill a
     *         <code>CreateGameSession</code> request. Clients can retry such requests immediately or after a waiting
     *         period.</li>
     *         <li>LimitExceededException The requested operation would cause the resource to exceed the allowed service
     *         limit. Resolve the issue before retrying.</li>
     *         <li>IdempotentParameterMismatchException A game session with this custom ID string already exists in this
     *         fleet. Resolve this conflict before retrying this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreateGameSession
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreateGameSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGameSessionResponse> createGameSession(CreateGameSessionRequest createGameSessionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGameSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGameSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGameSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGameSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGameSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGameSessionRequest, CreateGameSessionResponse>()
                            .withOperationName("CreateGameSession")
                            .withMarshaller(new CreateGameSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createGameSessionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createGameSessionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateGameSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Establishes a new queue for processing requests to place new game sessions. A queue identifies where new game
     * sessions can be hosted -- by specifying a list of destinations (fleets or aliases) -- and how long requests can
     * wait in the queue before timing out. You can set up a queue to try to place game sessions on fleets in multiple
     * Regions. To add placement requests to a queue, call <a>StartGameSessionPlacement</a> and reference the queue
     * name.
     * </p>
     * <p>
     * <b>Destination order.</b> When processing a request for a game session, Amazon GameLift tries each destination in
     * order until it finds one with available resources to host the new game session. A queue's default order is
     * determined by how destinations are listed. The default order is overridden when a game session placement request
     * provides player latency information. Player latency information enables Amazon GameLift to prioritize
     * destinations where players report the lowest average latency, as a result placing the new game session where the
     * majority of players will have the best possible gameplay experience.
     * </p>
     * <p>
     * <b>Player latency policies.</b> For placement requests containing player latency information, use player latency
     * policies to protect individual players from very high latencies. With a latency cap, even when a destination can
     * deliver a low latency for most players, the game is not placed where any individual player is reporting latency
     * higher than a policy's maximum. A queue can have multiple latency policies, which are enforced consecutively
     * starting with the policy with the lowest latency cap. Use multiple policies to gradually relax latency controls;
     * for example, you might set a policy with a low latency cap for the first 60 seconds, a second policy with a
     * higher cap for the next 60 seconds, etc.
     * </p>
     * <p>
     * To create a new queue, provide a name, timeout value, a list of destinations and, if desired, a set of latency
     * policies. If successful, a new queue object is returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/queues-design.html"> Design a Game Session
     * Queue</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/queues-creating.html"> Create a Game Session
     * Queue</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSessionQueue</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionQueues</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSessionQueue</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameSessionQueue</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createGameSessionQueueRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the CreateGameSessionQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>LimitExceededException The requested operation would cause the resource to exceed the allowed service
     *         limit. Resolve the issue before retrying.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreateGameSessionQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreateGameSessionQueue"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGameSessionQueueResponse> createGameSessionQueue(
            CreateGameSessionQueueRequest createGameSessionQueueRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGameSessionQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGameSessionQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGameSessionQueueResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGameSessionQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGameSessionQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGameSessionQueueRequest, CreateGameSessionQueueResponse>()
                            .withOperationName("CreateGameSessionQueue")
                            .withMarshaller(new CreateGameSessionQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createGameSessionQueueRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createGameSessionQueueRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateGameSessionQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Defines a new matchmaking configuration for use with FlexMatch. A matchmaking configuration sets out guidelines
     * for matching players and getting the matches into games. You can set up multiple matchmaking configurations to
     * handle the scenarios needed for your game. Each matchmaking ticket (<a>StartMatchmaking</a> or
     * <a>StartMatchBackfill</a>) specifies a configuration for the match and provides player attributes to support the
     * configuration being used.
     * </p>
     * <p>
     * To create a matchmaking configuration, at a minimum you must specify the following: configuration name; a rule
     * set that governs how to evaluate players and find acceptable matches; a game session queue to use when placing a
     * new game session for the match; and the maximum time allowed for a matchmaking attempt.
     * </p>
     * <p>
     * To track the progress of matchmaking tickets, set up an Amazon Simple Notification Service (SNS) to receive
     * notifications, and provide the topic ARN in the matchmaking configuration. An alternative method, continuously
     * poling ticket status with <a>DescribeMatchmaking</a>, should only be used for games in development with low
     * matchmaking usage.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-configuration.html"> Design a FlexMatch
     * Matchmaker</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-notification.html"> Set Up FlexMatch
     * Event Notification</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingConfigurations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingRuleSets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ValidateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingRuleSet</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createMatchmakingConfigurationRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the CreateMatchmakingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>LimitExceededException The requested operation would cause the resource to exceed the allowed service
     *         limit. Resolve the issue before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreateMatchmakingConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreateMatchmakingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMatchmakingConfigurationResponse> createMatchmakingConfiguration(
            CreateMatchmakingConfigurationRequest createMatchmakingConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createMatchmakingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMatchmakingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMatchmakingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMatchmakingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMatchmakingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMatchmakingConfigurationRequest, CreateMatchmakingConfigurationResponse>()
                            .withOperationName("CreateMatchmakingConfiguration")
                            .withMarshaller(new CreateMatchmakingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMatchmakingConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createMatchmakingConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateMatchmakingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new rule set for FlexMatch matchmaking. A rule set describes the type of match to create, such as the
     * number and size of teams. It also sets the parameters for acceptable player matches, such as minimum skill level
     * or character type. A rule set is used by a <a>MatchmakingConfiguration</a>.
     * </p>
     * <p>
     * To create a matchmaking rule set, provide unique rule set name and the rule set body in JSON format. Rule sets
     * must be defined in the same Region as the matchmaking configuration they are used with.
     * </p>
     * <p>
     * Since matchmaking rule sets cannot be edited, it is a good idea to check the rule set syntax using
     * <a>ValidateMatchmakingRuleSet</a> before creating a new rule set.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-rulesets.html">Build a Rule Set</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-configuration.html">Design a
     * Matchmaker</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-intro.html">Matchmaking with
     * FlexMatch</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingConfigurations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingRuleSets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ValidateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingRuleSet</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createMatchmakingRuleSetRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the CreateMatchmakingRuleSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreateMatchmakingRuleSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreateMatchmakingRuleSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMatchmakingRuleSetResponse> createMatchmakingRuleSet(
            CreateMatchmakingRuleSetRequest createMatchmakingRuleSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMatchmakingRuleSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMatchmakingRuleSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMatchmakingRuleSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMatchmakingRuleSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMatchmakingRuleSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMatchmakingRuleSetRequest, CreateMatchmakingRuleSetResponse>()
                            .withOperationName("CreateMatchmakingRuleSet")
                            .withMarshaller(new CreateMatchmakingRuleSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMatchmakingRuleSetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createMatchmakingRuleSetRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateMatchmakingRuleSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Reserves an open player slot in an active game session. Before a player can be added, a game session must have an
     * <code>ACTIVE</code> status, have a creation policy of <code>ALLOW_ALL</code>, and have an open player slot. To
     * add a group of players to a game session, use <a>CreatePlayerSessions</a>. When the player connects to the game
     * server and references a player session ID, the game server contacts the Amazon GameLift service to validate the
     * player reservation and accept the player.
     * </p>
     * <p>
     * To create a player session, specify a game session ID, player ID, and optionally a string of player data. If
     * successful, a slot is reserved in the game session for the player and a new <a>PlayerSession</a> object is
     * returned. Player sessions cannot be updated.
     * </p>
     * <p>
     * <i>Available in Amazon GameLift Local.</i>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreatePlayerSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreatePlayerSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribePlayerSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param createPlayerSessionRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the CreatePlayerSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidGameSessionStatusException The requested operation would cause a conflict with the current
     *         state of a resource associated with the request and/or the game instance. Resolve the conflict before
     *         retrying.</li>
     *         <li>GameSessionFullException The game instance is currently full and cannot allow the requested player(s)
     *         to join. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>TerminalRoutingStrategyException The service is unable to resolve the routing for a particular alias
     *         because it has a terminal <a>RoutingStrategy</a> associated with it. The message returned in this
     *         exception is the message defined in the routing strategy itself. Such requests should only be retried if
     *         the routing strategy for the specified alias is modified.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreatePlayerSession
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreatePlayerSession" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePlayerSessionResponse> createPlayerSession(
            CreatePlayerSessionRequest createPlayerSessionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPlayerSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePlayerSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePlayerSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePlayerSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePlayerSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePlayerSessionRequest, CreatePlayerSessionResponse>()
                            .withOperationName("CreatePlayerSession")
                            .withMarshaller(new CreatePlayerSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPlayerSessionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createPlayerSessionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreatePlayerSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Reserves open slots in a game session for a group of players. Before players can be added, a game session must
     * have an <code>ACTIVE</code> status, have a creation policy of <code>ALLOW_ALL</code>, and have an open player
     * slot. To add a single player to a game session, use <a>CreatePlayerSession</a>. When a player connects to the
     * game server and references a player session ID, the game server contacts the Amazon GameLift service to validate
     * the player reservation and accept the player.
     * </p>
     * <p>
     * To create player sessions, specify a game session ID, a list of player IDs, and optionally a set of player data
     * strings. If successful, a slot is reserved in the game session for each player and a set of new
     * <a>PlayerSession</a> objects is returned. Player sessions cannot be updated.
     * </p>
     * <p>
     * <i>Available in Amazon GameLift Local.</i>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreatePlayerSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreatePlayerSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribePlayerSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param createPlayerSessionsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the CreatePlayerSessions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidGameSessionStatusException The requested operation would cause a conflict with the current
     *         state of a resource associated with the request and/or the game instance. Resolve the conflict before
     *         retrying.</li>
     *         <li>GameSessionFullException The game instance is currently full and cannot allow the requested player(s)
     *         to join. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>TerminalRoutingStrategyException The service is unable to resolve the routing for a particular alias
     *         because it has a terminal <a>RoutingStrategy</a> associated with it. The message returned in this
     *         exception is the message defined in the routing strategy itself. Such requests should only be retried if
     *         the routing strategy for the specified alias is modified.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreatePlayerSessions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreatePlayerSessions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePlayerSessionsResponse> createPlayerSessions(
            CreatePlayerSessionsRequest createPlayerSessionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPlayerSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePlayerSessions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePlayerSessionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePlayerSessionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePlayerSessionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePlayerSessionsRequest, CreatePlayerSessionsResponse>()
                            .withOperationName("CreatePlayerSessions")
                            .withMarshaller(new CreatePlayerSessionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPlayerSessionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createPlayerSessionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreatePlayerSessionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new script record for your Realtime Servers script. Realtime scripts are JavaScript that provide
     * configuration settings and optional custom game logic for your game. The script is deployed when you create a
     * Realtime Servers fleet to host your game sessions. Script logic is executed during an active game session.
     * </p>
     * <p>
     * To create a new script record, specify a script name and provide the script file(s). The script files and all
     * dependencies must be zipped into a single file. You can pull the zip file from either of these locations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A locally available directory. Use the <i>ZipFile</i> parameter for this option.
     * </p>
     * </li>
     * <li>
     * <p>
     * An Amazon Simple Storage Service (Amazon S3) bucket under your AWS account. Use the <i>StorageLocation</i>
     * parameter for this option. You'll need to have an Identity Access Management (IAM) role that allows the Amazon
     * GameLift service to access your S3 bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the call is successful, a new script record is created with a unique script ID. If the script file is provided
     * as a local file, the file is uploaded to an Amazon GameLift-owned S3 bucket and the script record's storage
     * location reflects this location. If the script file is provided as an S3 bucket, Amazon GameLift accesses the
     * file at this storage location as needed for deployment.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/realtime-intro.html">Amazon GameLift Realtime
     * Servers</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/setting-up-role.html">Set Up a Role for
     * Amazon GameLift Access</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListScripts</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteScript</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createScriptRequest
     * @return A Java Future containing the result of the CreateScript operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreateScript
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreateScript" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateScriptResponse> createScript(CreateScriptRequest createScriptRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createScriptRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateScript");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateScriptResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateScriptResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateScriptResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateScriptRequest, CreateScriptResponse>()
                            .withOperationName("CreateScript").withMarshaller(new CreateScriptRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createScriptRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createScriptRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateScriptResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Requests authorization to create or delete a peer connection between the VPC for your Amazon GameLift fleet and a
     * virtual private cloud (VPC) in your AWS account. VPC peering enables the game servers on your fleet to
     * communicate directly with other AWS resources. Once you've received authorization, call
     * <a>CreateVpcPeeringConnection</a> to establish the peering connection. For more information, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/vpc-peering.html">VPC Peering with Amazon
     * GameLift Fleets</a>.
     * </p>
     * <p>
     * You can peer with VPCs that are owned by any AWS account you have access to, including the account that you use
     * to manage your Amazon GameLift fleets. You cannot peer with VPCs that are in different Regions.
     * </p>
     * <p>
     * To request authorization to create a connection, call this operation from the AWS account with the VPC that you
     * want to peer to your Amazon GameLift fleet. For example, to enable your game servers to retrieve data from a
     * DynamoDB table, use the account that manages that DynamoDB resource. Identify the following values: (1) The ID of
     * the VPC that you want to peer with, and (2) the ID of the AWS account that you use to manage Amazon GameLift. If
     * successful, VPC peering is authorized for the specified VPC.
     * </p>
     * <p>
     * To request authorization to delete a connection, call this operation from the AWS account with the VPC that is
     * peered with your Amazon GameLift fleet. Identify the following values: (1) VPC ID that you want to delete the
     * peering connection for, and (2) ID of the AWS account that you use to manage Amazon GameLift.
     * </p>
     * <p>
     * The authorization remains valid for 24 hours unless it is canceled by a call to
     * <a>DeleteVpcPeeringAuthorization</a>. You must create or delete the peering connection while the authorization is
     * valid.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringAuthorizations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateVpcPeeringConnection</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringConnections</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringConnection</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createVpcPeeringAuthorizationRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the CreateVpcPeeringAuthorization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreateVpcPeeringAuthorization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreateVpcPeeringAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVpcPeeringAuthorizationResponse> createVpcPeeringAuthorization(
            CreateVpcPeeringAuthorizationRequest createVpcPeeringAuthorizationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createVpcPeeringAuthorizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVpcPeeringAuthorization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVpcPeeringAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVpcPeeringAuthorizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVpcPeeringAuthorizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVpcPeeringAuthorizationRequest, CreateVpcPeeringAuthorizationResponse>()
                            .withOperationName("CreateVpcPeeringAuthorization")
                            .withMarshaller(new CreateVpcPeeringAuthorizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createVpcPeeringAuthorizationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createVpcPeeringAuthorizationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateVpcPeeringAuthorizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Establishes a VPC peering connection between a virtual private cloud (VPC) in an AWS account with the VPC for
     * your Amazon GameLift fleet. VPC peering enables the game servers on your fleet to communicate directly with other
     * AWS resources. You can peer with VPCs in any AWS account that you have access to, including the account that you
     * use to manage your Amazon GameLift fleets. You cannot peer with VPCs that are in different Regions. For more
     * information, see <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/vpc-peering.html">VPC
     * Peering with Amazon GameLift Fleets</a>.
     * </p>
     * <p>
     * Before calling this operation to establish the peering connection, you first need to call
     * <a>CreateVpcPeeringAuthorization</a> and identify the VPC you want to peer with. Once the authorization for the
     * specified VPC is issued, you have 24 hours to establish the connection. These two operations handle all tasks
     * necessary to peer the two VPCs, including acceptance, updating routing tables, etc.
     * </p>
     * <p>
     * To establish the connection, call this operation from the AWS account that is used to manage the Amazon GameLift
     * fleets. Identify the following values: (1) The ID of the fleet you want to be enable a VPC peering connection
     * for; (2) The AWS account with the VPC that you want to peer with; and (3) The ID of the VPC you want to peer
     * with. This operation is asynchronous. If successful, a <a>VpcPeeringConnection</a> request is created. You can
     * use continuous polling to track the request's status using <a>DescribeVpcPeeringConnections</a>, or by monitoring
     * fleet events for success or failure using <a>DescribeFleetEvents</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringAuthorizations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateVpcPeeringConnection</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringConnections</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringConnection</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createVpcPeeringConnectionRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the CreateVpcPeeringConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.CreateVpcPeeringConnection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/CreateVpcPeeringConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVpcPeeringConnectionResponse> createVpcPeeringConnection(
            CreateVpcPeeringConnectionRequest createVpcPeeringConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVpcPeeringConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVpcPeeringConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVpcPeeringConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVpcPeeringConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVpcPeeringConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVpcPeeringConnectionRequest, CreateVpcPeeringConnectionResponse>()
                            .withOperationName("CreateVpcPeeringConnection")
                            .withMarshaller(new CreateVpcPeeringConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createVpcPeeringConnectionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createVpcPeeringConnectionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateVpcPeeringConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an alias. This operation removes all record of the alias. Game clients attempting to access a server
     * process using the deleted alias receive an error. To delete an alias, specify the alias ID to be deleted.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResolveAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteAliasRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DeleteAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeleteAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeleteAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAliasResponse> deleteAlias(DeleteAliasRequest deleteAliasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAliasRequest, DeleteAliasResponse>()
                            .withOperationName("DeleteAlias").withMarshaller(new DeleteAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAliasRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteAliasRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a build. This operation permanently deletes the build resource and any uploaded build files. Deleting a
     * build does not affect the status of any active fleets using the build, but you can no longer create new fleets
     * with the deleted build.
     * </p>
     * <p>
     * To delete a build, specify the build ID.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-build-intro.html"> Upload a Custom
     * Server Build</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListBuilds</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteBuild</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteBuildRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DeleteBuild operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeleteBuild
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeleteBuild" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBuildResponse> deleteBuild(DeleteBuildRequest deleteBuildRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBuildRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBuild");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBuildResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBuildResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBuildResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBuildRequest, DeleteBuildResponse>()
                            .withOperationName("DeleteBuild").withMarshaller(new DeleteBuildRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteBuildRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteBuildRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteBuildResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes everything related to a fleet. Before deleting a fleet, you must set the fleet's desired capacity to
     * zero. See <a>UpdateFleetCapacity</a>.
     * </p>
     * <p>
     * If the fleet being deleted has a VPC peering connection, you first need to get a valid authorization (good for 24
     * hours) by calling <a>CreateVpcPeeringAuthorization</a>. You do not need to explicitly delete the VPC peering
     * connection--this is done as part of the delete fleet process.
     * </p>
     * <p>
     * This operation removes the fleet and its resources. Once a fleet is deleted, you can no longer use any of the
     * resource in that fleet.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteFleetRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DeleteFleet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidFleetStatusException The requested operation would cause a conflict with the current state of
     *         a resource associated with the request and/or the fleet. Resolve the conflict before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeleteFleet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeleteFleet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFleetResponse> deleteFleet(DeleteFleetRequest deleteFleetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFleet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFleetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFleetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFleetRequest, DeleteFleetResponse>()
                            .withOperationName("DeleteFleet").withMarshaller(new DeleteFleetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFleetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteFleetRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteFleetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Terminates a game server group and permanently deletes the game server group record. You have several options for
     * how these resources are impacted when deleting the game server group. Depending on the type of delete operation
     * selected, this operation might affect these resources:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The game server group
     * </p>
     * </li>
     * <li>
     * <p>
     * The corresponding Auto Scaling group
     * </p>
     * </li>
     * <li>
     * <p>
     * All game servers that are currently running in the group
     * </p>
     * </li>
     * </ul>
     * <p>
     * To delete a game server group, identify the game server group to delete and specify the type of delete operation
     * to initiate. Game server groups can only be deleted if they are in <code>ACTIVE</code> or <code>ERROR</code>
     * status.
     * </p>
     * <p>
     * If the delete request is successful, a series of operations are kicked off. The game server group status is
     * changed to <code>DELETE_SCHEDULED</code>, which prevents new game servers from being registered and stops
     * automatic scaling activity. Once all game servers in the game server group are deregistered, GameLift FleetIQ can
     * begin deleting resources. If any of the delete operations fail, the game server group is placed in
     * <code>ERROR</code> status.
     * </p>
     * <p>
     * GameLift FleetIQ emits delete events to Amazon CloudWatch.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServerGroups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResumeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SuspendGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerInstances</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteGameServerGroupRequest
     * @return A Java Future containing the result of the DeleteGameServerGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeleteGameServerGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeleteGameServerGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGameServerGroupResponse> deleteGameServerGroup(
            DeleteGameServerGroupRequest deleteGameServerGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGameServerGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGameServerGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGameServerGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteGameServerGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGameServerGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGameServerGroupRequest, DeleteGameServerGroupResponse>()
                            .withOperationName("DeleteGameServerGroup")
                            .withMarshaller(new DeleteGameServerGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteGameServerGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteGameServerGroupRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteGameServerGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a game session queue. Once a queue is successfully deleted, unfulfilled <a>StartGameSessionPlacement</a>
     * requests that reference the queue will fail. To delete a queue, specify the queue name.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/queues-intro.html"> Using Multi-Region
     * Queues</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSessionQueue</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionQueues</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSessionQueue</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameSessionQueue</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteGameSessionQueueRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DeleteGameSessionQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeleteGameSessionQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeleteGameSessionQueue"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGameSessionQueueResponse> deleteGameSessionQueue(
            DeleteGameSessionQueueRequest deleteGameSessionQueueRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGameSessionQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGameSessionQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGameSessionQueueResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteGameSessionQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGameSessionQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGameSessionQueueRequest, DeleteGameSessionQueueResponse>()
                            .withOperationName("DeleteGameSessionQueue")
                            .withMarshaller(new DeleteGameSessionQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteGameSessionQueueRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteGameSessionQueueRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteGameSessionQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently removes a FlexMatch matchmaking configuration. To delete, specify the configuration name. A
     * matchmaking configuration cannot be deleted if it is being used in any active matchmaking tickets.
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingConfigurations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingRuleSets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ValidateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingRuleSet</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteMatchmakingConfigurationRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DeleteMatchmakingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeleteMatchmakingConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeleteMatchmakingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMatchmakingConfigurationResponse> deleteMatchmakingConfiguration(
            DeleteMatchmakingConfigurationRequest deleteMatchmakingConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteMatchmakingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMatchmakingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMatchmakingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMatchmakingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMatchmakingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMatchmakingConfigurationRequest, DeleteMatchmakingConfigurationResponse>()
                            .withOperationName("DeleteMatchmakingConfiguration")
                            .withMarshaller(new DeleteMatchmakingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMatchmakingConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteMatchmakingConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteMatchmakingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing matchmaking rule set. To delete the rule set, provide the rule set name. Rule sets cannot be
     * deleted if they are currently being used by a matchmaking configuration.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-rulesets.html">Build a Rule Set</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingConfigurations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingRuleSets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ValidateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingRuleSet</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteMatchmakingRuleSetRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DeleteMatchmakingRuleSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeleteMatchmakingRuleSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeleteMatchmakingRuleSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMatchmakingRuleSetResponse> deleteMatchmakingRuleSet(
            DeleteMatchmakingRuleSetRequest deleteMatchmakingRuleSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMatchmakingRuleSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMatchmakingRuleSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMatchmakingRuleSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMatchmakingRuleSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMatchmakingRuleSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMatchmakingRuleSetRequest, DeleteMatchmakingRuleSetResponse>()
                            .withOperationName("DeleteMatchmakingRuleSet")
                            .withMarshaller(new DeleteMatchmakingRuleSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMatchmakingRuleSetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteMatchmakingRuleSetRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteMatchmakingRuleSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a fleet scaling policy. Once deleted, the policy is no longer in force and GameLift removes all record of
     * it. To delete a scaling policy, specify both the scaling policy name and the fleet ID it is associated with.
     * </p>
     * <p>
     * To temporarily suspend scaling policies, call <a>StopFleetActions</a>. This operation suspends all policies for
     * the fleet.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Manage scaling policies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>PutScalingPolicy</a> (auto-scaling)
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeScalingPolicies</a> (auto-scaling)
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteScalingPolicy</a> (auto-scaling)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Manage fleet actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartFleetActions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param deleteScalingPolicyRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DeleteScalingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeleteScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeleteScalingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteScalingPolicyResponse> deleteScalingPolicy(
            DeleteScalingPolicyRequest deleteScalingPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScalingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteScalingPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteScalingPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteScalingPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteScalingPolicyRequest, DeleteScalingPolicyResponse>()
                            .withOperationName("DeleteScalingPolicy")
                            .withMarshaller(new DeleteScalingPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteScalingPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteScalingPolicyRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteScalingPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Realtime script. This operation permanently deletes the script record. If script files were uploaded,
     * they are also deleted (files stored in an S3 bucket are not deleted).
     * </p>
     * <p>
     * To delete a script, specify the script ID. Before deleting a script, be sure to terminate all fleets that are
     * deployed with the script being deleted. Fleet instances periodically check for script updates, and if the script
     * record no longer exists, the instance will go into an error state and be unable to host game sessions.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/realtime-intro.html">Amazon GameLift Realtime
     * Servers</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListScripts</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteScript</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteScriptRequest
     * @return A Java Future containing the result of the DeleteScript operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeleteScript
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeleteScript" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteScriptResponse> deleteScript(DeleteScriptRequest deleteScriptRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScriptRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteScript");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteScriptResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteScriptResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteScriptResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteScriptRequest, DeleteScriptResponse>()
                            .withOperationName("DeleteScript").withMarshaller(new DeleteScriptRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteScriptRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteScriptRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteScriptResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels a pending VPC peering authorization for the specified VPC. If you need to delete an existing VPC peering
     * connection, call <a>DeleteVpcPeeringConnection</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringAuthorizations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateVpcPeeringConnection</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringConnections</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringConnection</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteVpcPeeringAuthorizationRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DeleteVpcPeeringAuthorization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeleteVpcPeeringAuthorization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeleteVpcPeeringAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVpcPeeringAuthorizationResponse> deleteVpcPeeringAuthorization(
            DeleteVpcPeeringAuthorizationRequest deleteVpcPeeringAuthorizationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteVpcPeeringAuthorizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVpcPeeringAuthorization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVpcPeeringAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVpcPeeringAuthorizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVpcPeeringAuthorizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVpcPeeringAuthorizationRequest, DeleteVpcPeeringAuthorizationResponse>()
                            .withOperationName("DeleteVpcPeeringAuthorization")
                            .withMarshaller(new DeleteVpcPeeringAuthorizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteVpcPeeringAuthorizationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteVpcPeeringAuthorizationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteVpcPeeringAuthorizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a VPC peering connection. To delete the connection, you must have a valid authorization for the VPC
     * peering connection that you want to delete. You can check for an authorization by calling
     * <a>DescribeVpcPeeringAuthorizations</a> or request a new one using <a>CreateVpcPeeringAuthorization</a>.
     * </p>
     * <p>
     * Once a valid authorization exists, call this operation from the AWS account that is used to manage the Amazon
     * GameLift fleets. Identify the connection to delete by the connection ID and fleet ID. If successful, the
     * connection is removed.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringAuthorizations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateVpcPeeringConnection</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringConnections</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringConnection</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteVpcPeeringConnectionRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DeleteVpcPeeringConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeleteVpcPeeringConnection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeleteVpcPeeringConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVpcPeeringConnectionResponse> deleteVpcPeeringConnection(
            DeleteVpcPeeringConnectionRequest deleteVpcPeeringConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVpcPeeringConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVpcPeeringConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVpcPeeringConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVpcPeeringConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVpcPeeringConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVpcPeeringConnectionRequest, DeleteVpcPeeringConnectionResponse>()
                            .withOperationName("DeleteVpcPeeringConnection")
                            .withMarshaller(new DeleteVpcPeeringConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteVpcPeeringConnectionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteVpcPeeringConnectionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteVpcPeeringConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Removes the game server from a game server group. As a result of this operation, the deregistered game server can
     * no longer be claimed and will not be returned in a list of active game servers.
     * </p>
     * <p>
     * To deregister a game server, specify the game server group and game server ID. If successful, this operation
     * emits a CloudWatch event with termination timestamp and reason.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>RegisterGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServers</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ClaimGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeregisterGameServer</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deregisterGameServerRequest
     * @return A Java Future containing the result of the DeregisterGameServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DeregisterGameServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DeregisterGameServer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterGameServerResponse> deregisterGameServer(
            DeregisterGameServerRequest deregisterGameServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterGameServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterGameServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterGameServerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterGameServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterGameServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterGameServerRequest, DeregisterGameServerResponse>()
                            .withOperationName("DeregisterGameServer")
                            .withMarshaller(new DeregisterGameServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deregisterGameServerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deregisterGameServerRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeregisterGameServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves properties for an alias. This operation returns all alias metadata and settings. To get an alias's
     * target fleet ID only, use <code>ResolveAlias</code>.
     * </p>
     * <p>
     * To get alias properties, specify the alias ID. If successful, the requested alias record is returned.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResolveAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeAliasRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAliasResponse> describeAlias(DescribeAliasRequest describeAliasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAliasRequest, DescribeAliasResponse>()
                            .withOperationName("DescribeAlias")
                            .withMarshaller(new DescribeAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAliasRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeAliasRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves properties for a custom game build. To request a build resource, specify a build ID. If successful, an
     * object containing the build properties is returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-build-intro.html"> Upload a Custom
     * Server Build</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListBuilds</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteBuild</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeBuildRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeBuild operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeBuild
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeBuild" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeBuildResponse> describeBuild(DescribeBuildRequest describeBuildRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBuildRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBuild");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBuildResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeBuildResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBuildResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBuildRequest, DescribeBuildResponse>()
                            .withOperationName("DescribeBuild")
                            .withMarshaller(new DescribeBuildRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeBuildRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeBuildRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeBuildResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the following information for the specified EC2 instance type:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Maximum number of instances allowed per AWS account (service limit).
     * </p>
     * </li>
     * <li>
     * <p>
     * Current usage for the AWS account.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To learn more about the capabilities of each instance type, see <a
     * href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instance Types</a>. Note that the instance types
     * offered may vary depending on the region.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeEc2InstanceLimitsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeEC2InstanceLimits operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeEC2InstanceLimits
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeEC2InstanceLimits"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEc2InstanceLimitsResponse> describeEC2InstanceLimits(
            DescribeEc2InstanceLimitsRequest describeEc2InstanceLimitsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEc2InstanceLimitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEC2InstanceLimits");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEc2InstanceLimitsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEc2InstanceLimitsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEc2InstanceLimitsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEc2InstanceLimitsRequest, DescribeEc2InstanceLimitsResponse>()
                            .withOperationName("DescribeEC2InstanceLimits")
                            .withMarshaller(new DescribeEc2InstanceLimitsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEc2InstanceLimitsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeEc2InstanceLimitsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeEc2InstanceLimitsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves core properties, including configuration, status, and metadata, for a fleet.
     * </p>
     * <p>
     * To get attributes for one or more fleets, provide a list of fleet IDs or fleet ARNs. To get attributes for all
     * fleets, do not specify a fleet identifier. When requesting attributes for multiple fleets, use the pagination
     * parameters to retrieve results as a set of sequential pages. If successful, a <a>FleetAttributes</a> object is
     * returned for each fleet requested, unless the fleet identifier is not found.
     * </p>
     * <note>
     * <p>
     * Some API operations may limit the number of fleet IDs allowed in one request. If a request exceeds this limit,
     * the request fails and the error message includes the maximum allowed number.
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Describe fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetUtilization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeRuntimeConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetEvents</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeFleetAttributesRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeFleetAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeFleetAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeFleetAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFleetAttributesResponse> describeFleetAttributes(
            DescribeFleetAttributesRequest describeFleetAttributesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFleetAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFleetAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFleetAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFleetAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFleetAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFleetAttributesRequest, DescribeFleetAttributesResponse>()
                            .withOperationName("DescribeFleetAttributes")
                            .withMarshaller(new DescribeFleetAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFleetAttributesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeFleetAttributesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeFleetAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves core properties, including configuration, status, and metadata, for a fleet.
     * </p>
     * <p>
     * To get attributes for one or more fleets, provide a list of fleet IDs or fleet ARNs. To get attributes for all
     * fleets, do not specify a fleet identifier. When requesting attributes for multiple fleets, use the pagination
     * parameters to retrieve results as a set of sequential pages. If successful, a <a>FleetAttributes</a> object is
     * returned for each fleet requested, unless the fleet identifier is not found.
     * </p>
     * <note>
     * <p>
     * Some API operations may limit the number of fleet IDs allowed in one request. If a request exceeds this limit,
     * the request fails and the error message includes the maximum allowed number.
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Describe fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetUtilization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeRuntimeConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetEvents</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeFleetAttributes(software.amazon.awssdk.services.gamelift.model.DescribeFleetAttributesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeFleetAttributesPublisher publisher = client.describeFleetAttributesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeFleetAttributesPublisher publisher = client.describeFleetAttributesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeFleetAttributesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeFleetAttributesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFleetAttributes(software.amazon.awssdk.services.gamelift.model.DescribeFleetAttributesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeFleetAttributesRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeFleetAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeFleetAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeFleetAttributesPublisher describeFleetAttributesPaginator(
            DescribeFleetAttributesRequest describeFleetAttributesRequest) {
        return new DescribeFleetAttributesPublisher(this, applyPaginatorUserAgent(describeFleetAttributesRequest));
    }

    /**
     * <p>
     * Retrieves the current capacity statistics for one or more fleets. These statistics present a snapshot of the
     * fleet's instances and provide insight on current or imminent scaling activity. To get statistics on game hosting
     * activity in the fleet, see <a>DescribeFleetUtilization</a>.
     * </p>
     * <p>
     * You can request capacity for all fleets or specify a list of one or more fleet identifiers. When requesting
     * multiple fleets, use the pagination parameters to retrieve results as a set of sequential pages. If successful, a
     * <a>FleetCapacity</a> object is returned for each requested fleet ID. When a list of fleet IDs is provided,
     * attribute objects are returned only for fleets that currently exist.
     * </p>
     * <note>
     * <p>
     * Some API operations may limit the number of fleet IDs allowed in one request. If a request exceeds this limit,
     * the request fails and the error message includes the maximum allowed.
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html#gamelift-metrics-fleet"
     * >GameLift Metrics for Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Describe fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetUtilization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeRuntimeConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetEvents</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeFleetCapacityRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeFleetCapacity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeFleetCapacity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeFleetCapacity" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFleetCapacityResponse> describeFleetCapacity(
            DescribeFleetCapacityRequest describeFleetCapacityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFleetCapacityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFleetCapacity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFleetCapacityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFleetCapacityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFleetCapacityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFleetCapacityRequest, DescribeFleetCapacityResponse>()
                            .withOperationName("DescribeFleetCapacity")
                            .withMarshaller(new DescribeFleetCapacityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFleetCapacityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeFleetCapacityRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeFleetCapacityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current capacity statistics for one or more fleets. These statistics present a snapshot of the
     * fleet's instances and provide insight on current or imminent scaling activity. To get statistics on game hosting
     * activity in the fleet, see <a>DescribeFleetUtilization</a>.
     * </p>
     * <p>
     * You can request capacity for all fleets or specify a list of one or more fleet identifiers. When requesting
     * multiple fleets, use the pagination parameters to retrieve results as a set of sequential pages. If successful, a
     * <a>FleetCapacity</a> object is returned for each requested fleet ID. When a list of fleet IDs is provided,
     * attribute objects are returned only for fleets that currently exist.
     * </p>
     * <note>
     * <p>
     * Some API operations may limit the number of fleet IDs allowed in one request. If a request exceeds this limit,
     * the request fails and the error message includes the maximum allowed.
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html#gamelift-metrics-fleet"
     * >GameLift Metrics for Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Describe fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetUtilization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeRuntimeConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetEvents</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeFleetCapacity(software.amazon.awssdk.services.gamelift.model.DescribeFleetCapacityRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeFleetCapacityPublisher publisher = client.describeFleetCapacityPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeFleetCapacityPublisher publisher = client.describeFleetCapacityPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeFleetCapacityResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeFleetCapacityResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFleetCapacity(software.amazon.awssdk.services.gamelift.model.DescribeFleetCapacityRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeFleetCapacityRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeFleetCapacity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeFleetCapacity" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeFleetCapacityPublisher describeFleetCapacityPaginator(DescribeFleetCapacityRequest describeFleetCapacityRequest) {
        return new DescribeFleetCapacityPublisher(this, applyPaginatorUserAgent(describeFleetCapacityRequest));
    }

    /**
     * <p>
     * Retrieves entries from the specified fleet's event log. You can specify a time range to limit the result set. Use
     * the pagination parameters to retrieve results as a set of sequential pages. If successful, a collection of event
     * log entries matching the request are returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Describe fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetUtilization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeRuntimeConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetEvents</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeFleetEventsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeFleetEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeFleetEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeFleetEvents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFleetEventsResponse> describeFleetEvents(
            DescribeFleetEventsRequest describeFleetEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFleetEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFleetEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFleetEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFleetEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFleetEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFleetEventsRequest, DescribeFleetEventsResponse>()
                            .withOperationName("DescribeFleetEvents")
                            .withMarshaller(new DescribeFleetEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFleetEventsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeFleetEventsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeFleetEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves entries from the specified fleet's event log. You can specify a time range to limit the result set. Use
     * the pagination parameters to retrieve results as a set of sequential pages. If successful, a collection of event
     * log entries matching the request are returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Describe fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetUtilization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeRuntimeConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetEvents</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeFleetEvents(software.amazon.awssdk.services.gamelift.model.DescribeFleetEventsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeFleetEventsPublisher publisher = client.describeFleetEventsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeFleetEventsPublisher publisher = client.describeFleetEventsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeFleetEventsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeFleetEventsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFleetEvents(software.amazon.awssdk.services.gamelift.model.DescribeFleetEventsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeFleetEventsRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeFleetEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeFleetEvents" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeFleetEventsPublisher describeFleetEventsPaginator(DescribeFleetEventsRequest describeFleetEventsRequest) {
        return new DescribeFleetEventsPublisher(this, applyPaginatorUserAgent(describeFleetEventsRequest));
    }

    /**
     * <p>
     * Retrieves a fleet's inbound connection permissions. Connection permissions specify the range of IP addresses and
     * port settings that incoming traffic can use to access server processes in the fleet. Game sessions that are
     * running on instances in the fleet use connections that fall in this range.
     * </p>
     * <p>
     * To get a fleet's inbound connection permissions, specify the fleet's unique identifier. If successful, a
     * collection of <a>IpPermission</a> objects is returned for the requested fleet ID. If the requested fleet has been
     * deleted, the result set is empty.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Describe fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetUtilization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeRuntimeConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetEvents</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeFleetPortSettingsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeFleetPortSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeFleetPortSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeFleetPortSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFleetPortSettingsResponse> describeFleetPortSettings(
            DescribeFleetPortSettingsRequest describeFleetPortSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFleetPortSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFleetPortSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFleetPortSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFleetPortSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFleetPortSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFleetPortSettingsRequest, DescribeFleetPortSettingsResponse>()
                            .withOperationName("DescribeFleetPortSettings")
                            .withMarshaller(new DescribeFleetPortSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFleetPortSettingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeFleetPortSettingsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeFleetPortSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves utilization statistics for one or more fleets. These statistics provide insight into how available
     * hosting resources are currently being used. To get statistics on available hosting resources, see
     * <a>DescribeFleetCapacity</a>.
     * </p>
     * <p>
     * You can request utilization data for all fleets, or specify a list of one or more fleet IDs. When requesting
     * multiple fleets, use the pagination parameters to retrieve results as a set of sequential pages. If successful, a
     * <a>FleetUtilization</a> object is returned for each requested fleet ID, unless the fleet identifier is not found.
     * </p>
     * <note>
     * <p>
     * Some API operations may limit the number of fleet IDs allowed in one request. If a request exceeds this limit,
     * the request fails and the error message includes the maximum allowed.
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html#gamelift-metrics-fleet"
     * >GameLift Metrics for Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Describe fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetUtilization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeRuntimeConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetEvents</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeFleetUtilizationRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeFleetUtilization operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeFleetUtilization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeFleetUtilization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFleetUtilizationResponse> describeFleetUtilization(
            DescribeFleetUtilizationRequest describeFleetUtilizationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFleetUtilizationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFleetUtilization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFleetUtilizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFleetUtilizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFleetUtilizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFleetUtilizationRequest, DescribeFleetUtilizationResponse>()
                            .withOperationName("DescribeFleetUtilization")
                            .withMarshaller(new DescribeFleetUtilizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFleetUtilizationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeFleetUtilizationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeFleetUtilizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves utilization statistics for one or more fleets. These statistics provide insight into how available
     * hosting resources are currently being used. To get statistics on available hosting resources, see
     * <a>DescribeFleetCapacity</a>.
     * </p>
     * <p>
     * You can request utilization data for all fleets, or specify a list of one or more fleet IDs. When requesting
     * multiple fleets, use the pagination parameters to retrieve results as a set of sequential pages. If successful, a
     * <a>FleetUtilization</a> object is returned for each requested fleet ID, unless the fleet identifier is not found.
     * </p>
     * <note>
     * <p>
     * Some API operations may limit the number of fleet IDs allowed in one request. If a request exceeds this limit,
     * the request fails and the error message includes the maximum allowed.
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html#gamelift-metrics-fleet"
     * >GameLift Metrics for Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Describe fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetUtilization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeRuntimeConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetEvents</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeFleetUtilization(software.amazon.awssdk.services.gamelift.model.DescribeFleetUtilizationRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeFleetUtilizationPublisher publisher = client.describeFleetUtilizationPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeFleetUtilizationPublisher publisher = client.describeFleetUtilizationPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeFleetUtilizationResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeFleetUtilizationResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFleetUtilization(software.amazon.awssdk.services.gamelift.model.DescribeFleetUtilizationRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeFleetUtilizationRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeFleetUtilization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeFleetUtilization"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeFleetUtilizationPublisher describeFleetUtilizationPaginator(
            DescribeFleetUtilizationRequest describeFleetUtilizationRequest) {
        return new DescribeFleetUtilizationPublisher(this, applyPaginatorUserAgent(describeFleetUtilizationRequest));
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Retrieves information for a registered game server. Information includes game server status, health check info,
     * and the instance that the game server is running on.
     * </p>
     * <p>
     * To retrieve game server information, specify the game server ID. If successful, the requested game server object
     * is returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>RegisterGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServers</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ClaimGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeregisterGameServer</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeGameServerRequest
     * @return A Java Future containing the result of the DescribeGameServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeGameServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeGameServer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGameServerResponse> describeGameServer(DescribeGameServerRequest describeGameServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGameServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGameServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGameServerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGameServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGameServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGameServerRequest, DescribeGameServerResponse>()
                            .withOperationName("DescribeGameServer")
                            .withMarshaller(new DescribeGameServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeGameServerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeGameServerRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeGameServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Retrieves information on a game server group. This operation returns only properties related to GameLift FleetIQ.
     * To view or update properties for the corresponding Auto Scaling group, such as launch template, auto scaling
     * policies, and maximum/minimum group size, access the Auto Scaling group directly.
     * </p>
     * <p>
     * To get attributes for a game server group, provide a group name or ARN value. If successful, a
     * <a>GameServerGroup</a> object is returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServerGroups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResumeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SuspendGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerInstances</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeGameServerGroupRequest
     * @return A Java Future containing the result of the DescribeGameServerGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeGameServerGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeGameServerGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGameServerGroupResponse> describeGameServerGroup(
            DescribeGameServerGroupRequest describeGameServerGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGameServerGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGameServerGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGameServerGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGameServerGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGameServerGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGameServerGroupRequest, DescribeGameServerGroupResponse>()
                            .withOperationName("DescribeGameServerGroup")
                            .withMarshaller(new DescribeGameServerGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeGameServerGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeGameServerGroupRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeGameServerGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Retrieves status information about the Amazon EC2 instances associated with a GameLift FleetIQ game server group.
     * Use this operation to detect when instances are active or not available to host new game servers. If you are
     * looking for instance configuration information, call <a>DescribeGameServerGroup</a> or access the corresponding
     * Auto Scaling group properties.
     * </p>
     * <p>
     * To request status for all instances in the game server group, provide a game server group ID only. To request
     * status for specific instances, provide the game server group ID and one or more instance IDs. Use the pagination
     * parameters to retrieve results in sequential segments. If successful, a collection of
     * <code>GameServerInstance</code> objects is returned.
     * </p>
     * <p>
     * This operation is not designed to be called with every game server claim request; this practice can cause you to
     * exceed your API limit, which results in errors. Instead, as a best practice, cache the results and refresh your
     * cache no more than once every 10 seconds.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServerGroups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResumeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SuspendGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerInstances</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeGameServerInstancesRequest
     * @return A Java Future containing the result of the DescribeGameServerInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeGameServerInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeGameServerInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGameServerInstancesResponse> describeGameServerInstances(
            DescribeGameServerInstancesRequest describeGameServerInstancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGameServerInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGameServerInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGameServerInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGameServerInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGameServerInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGameServerInstancesRequest, DescribeGameServerInstancesResponse>()
                            .withOperationName("DescribeGameServerInstances")
                            .withMarshaller(new DescribeGameServerInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeGameServerInstancesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeGameServerInstancesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeGameServerInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Retrieves status information about the Amazon EC2 instances associated with a GameLift FleetIQ game server group.
     * Use this operation to detect when instances are active or not available to host new game servers. If you are
     * looking for instance configuration information, call <a>DescribeGameServerGroup</a> or access the corresponding
     * Auto Scaling group properties.
     * </p>
     * <p>
     * To request status for all instances in the game server group, provide a game server group ID only. To request
     * status for specific instances, provide the game server group ID and one or more instance IDs. Use the pagination
     * parameters to retrieve results in sequential segments. If successful, a collection of
     * <code>GameServerInstance</code> objects is returned.
     * </p>
     * <p>
     * This operation is not designed to be called with every game server claim request; this practice can cause you to
     * exceed your API limit, which results in errors. Instead, as a best practice, cache the results and refresh your
     * cache no more than once every 10 seconds.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServerGroups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResumeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SuspendGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerInstances</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeGameServerInstances(software.amazon.awssdk.services.gamelift.model.DescribeGameServerInstancesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeGameServerInstancesPublisher publisher = client.describeGameServerInstancesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeGameServerInstancesPublisher publisher = client.describeGameServerInstancesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeGameServerInstancesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeGameServerInstancesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeGameServerInstances(software.amazon.awssdk.services.gamelift.model.DescribeGameServerInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeGameServerInstancesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeGameServerInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeGameServerInstances"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeGameServerInstancesPublisher describeGameServerInstancesPaginator(
            DescribeGameServerInstancesRequest describeGameServerInstancesRequest) {
        return new DescribeGameServerInstancesPublisher(this, applyPaginatorUserAgent(describeGameServerInstancesRequest));
    }

    /**
     * <p>
     * Retrieves properties, including the protection policy in force, for one or more game sessions. This operation can
     * be used in several ways: (1) provide a <code>GameSessionId</code> or <code>GameSessionArn</code> to request
     * details for a specific game session; (2) provide either a <code>FleetId</code> or an <code>AliasId</code> to
     * request properties for all game sessions running on a fleet.
     * </p>
     * <p>
     * To get game session record(s), specify just one of the following: game session ID, fleet ID, or alias ID. You can
     * filter this request by game session status. Use the pagination parameters to retrieve results as a set of
     * sequential pages. If successful, a <a>GameSessionDetail</a> object is returned for each session matching the
     * request.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param describeGameSessionDetailsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeGameSessionDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>TerminalRoutingStrategyException The service is unable to resolve the routing for a particular alias
     *         because it has a terminal <a>RoutingStrategy</a> associated with it. The message returned in this
     *         exception is the message defined in the routing strategy itself. Such requests should only be retried if
     *         the routing strategy for the specified alias is modified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeGameSessionDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeGameSessionDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGameSessionDetailsResponse> describeGameSessionDetails(
            DescribeGameSessionDetailsRequest describeGameSessionDetailsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGameSessionDetailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGameSessionDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGameSessionDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGameSessionDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGameSessionDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGameSessionDetailsRequest, DescribeGameSessionDetailsResponse>()
                            .withOperationName("DescribeGameSessionDetails")
                            .withMarshaller(new DescribeGameSessionDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeGameSessionDetailsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeGameSessionDetailsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeGameSessionDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves properties, including the protection policy in force, for one or more game sessions. This operation can
     * be used in several ways: (1) provide a <code>GameSessionId</code> or <code>GameSessionArn</code> to request
     * details for a specific game session; (2) provide either a <code>FleetId</code> or an <code>AliasId</code> to
     * request properties for all game sessions running on a fleet.
     * </p>
     * <p>
     * To get game session record(s), specify just one of the following: game session ID, fleet ID, or alias ID. You can
     * filter this request by game session status. Use the pagination parameters to retrieve results as a set of
     * sequential pages. If successful, a <a>GameSessionDetail</a> object is returned for each session matching the
     * request.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeGameSessionDetails(software.amazon.awssdk.services.gamelift.model.DescribeGameSessionDetailsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeGameSessionDetailsPublisher publisher = client.describeGameSessionDetailsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeGameSessionDetailsPublisher publisher = client.describeGameSessionDetailsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeGameSessionDetailsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeGameSessionDetailsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeGameSessionDetails(software.amazon.awssdk.services.gamelift.model.DescribeGameSessionDetailsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeGameSessionDetailsRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>TerminalRoutingStrategyException The service is unable to resolve the routing for a particular alias
     *         because it has a terminal <a>RoutingStrategy</a> associated with it. The message returned in this
     *         exception is the message defined in the routing strategy itself. Such requests should only be retried if
     *         the routing strategy for the specified alias is modified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeGameSessionDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeGameSessionDetails"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeGameSessionDetailsPublisher describeGameSessionDetailsPaginator(
            DescribeGameSessionDetailsRequest describeGameSessionDetailsRequest) {
        return new DescribeGameSessionDetailsPublisher(this, applyPaginatorUserAgent(describeGameSessionDetailsRequest));
    }

    /**
     * <p>
     * Retrieves properties and current status of a game session placement request. To get game session placement
     * details, specify the placement ID. If successful, a <a>GameSessionPlacement</a> object is returned.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param describeGameSessionPlacementRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeGameSessionPlacement operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeGameSessionPlacement
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeGameSessionPlacement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGameSessionPlacementResponse> describeGameSessionPlacement(
            DescribeGameSessionPlacementRequest describeGameSessionPlacementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGameSessionPlacementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGameSessionPlacement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGameSessionPlacementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGameSessionPlacementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGameSessionPlacementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGameSessionPlacementRequest, DescribeGameSessionPlacementResponse>()
                            .withOperationName("DescribeGameSessionPlacement")
                            .withMarshaller(new DescribeGameSessionPlacementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeGameSessionPlacementRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeGameSessionPlacementRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeGameSessionPlacementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the properties for one or more game session queues. When requesting multiple queues, use the pagination
     * parameters to retrieve results as a set of sequential pages. If successful, a <a>GameSessionQueue</a> object is
     * returned for each requested queue. When specifying a list of queues, objects are returned only for queues that
     * currently exist in the Region.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/queues-console.html"> View Your Queues</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSessionQueue</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionQueues</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSessionQueue</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameSessionQueue</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeGameSessionQueuesRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeGameSessionQueues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeGameSessionQueues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeGameSessionQueues"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGameSessionQueuesResponse> describeGameSessionQueues(
            DescribeGameSessionQueuesRequest describeGameSessionQueuesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGameSessionQueuesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGameSessionQueues");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGameSessionQueuesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGameSessionQueuesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGameSessionQueuesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGameSessionQueuesRequest, DescribeGameSessionQueuesResponse>()
                            .withOperationName("DescribeGameSessionQueues")
                            .withMarshaller(new DescribeGameSessionQueuesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeGameSessionQueuesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeGameSessionQueuesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeGameSessionQueuesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the properties for one or more game session queues. When requesting multiple queues, use the pagination
     * parameters to retrieve results as a set of sequential pages. If successful, a <a>GameSessionQueue</a> object is
     * returned for each requested queue. When specifying a list of queues, objects are returned only for queues that
     * currently exist in the Region.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/queues-console.html"> View Your Queues</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSessionQueue</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionQueues</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSessionQueue</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameSessionQueue</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeGameSessionQueues(software.amazon.awssdk.services.gamelift.model.DescribeGameSessionQueuesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeGameSessionQueuesPublisher publisher = client.describeGameSessionQueuesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeGameSessionQueuesPublisher publisher = client.describeGameSessionQueuesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeGameSessionQueuesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeGameSessionQueuesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeGameSessionQueues(software.amazon.awssdk.services.gamelift.model.DescribeGameSessionQueuesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeGameSessionQueuesRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeGameSessionQueues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeGameSessionQueues"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeGameSessionQueuesPublisher describeGameSessionQueuesPaginator(
            DescribeGameSessionQueuesRequest describeGameSessionQueuesRequest) {
        return new DescribeGameSessionQueuesPublisher(this, applyPaginatorUserAgent(describeGameSessionQueuesRequest));
    }

    /**
     * <p>
     * Retrieves a set of one or more game sessions. Request a specific game session or request all game sessions on a
     * fleet. Alternatively, use <a>SearchGameSessions</a> to request a set of active game sessions that are filtered by
     * certain criteria. To retrieve protection policy settings for game sessions, use
     * <a>DescribeGameSessionDetails</a>.
     * </p>
     * <p>
     * To get game sessions, specify one of the following: game session ID, fleet ID, or alias ID. You can filter this
     * request by game session status. Use the pagination parameters to retrieve results as a set of sequential pages.
     * If successful, a <a>GameSession</a> object is returned for each game session matching the request.
     * </p>
     * <p>
     * <i>Available in Amazon GameLift Local.</i>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param describeGameSessionsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeGameSessions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>TerminalRoutingStrategyException The service is unable to resolve the routing for a particular alias
     *         because it has a terminal <a>RoutingStrategy</a> associated with it. The message returned in this
     *         exception is the message defined in the routing strategy itself. Such requests should only be retried if
     *         the routing strategy for the specified alias is modified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeGameSessions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeGameSessions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGameSessionsResponse> describeGameSessions(
            DescribeGameSessionsRequest describeGameSessionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGameSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGameSessions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGameSessionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGameSessionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGameSessionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGameSessionsRequest, DescribeGameSessionsResponse>()
                            .withOperationName("DescribeGameSessions")
                            .withMarshaller(new DescribeGameSessionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeGameSessionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeGameSessionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeGameSessionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a set of one or more game sessions. Request a specific game session or request all game sessions on a
     * fleet. Alternatively, use <a>SearchGameSessions</a> to request a set of active game sessions that are filtered by
     * certain criteria. To retrieve protection policy settings for game sessions, use
     * <a>DescribeGameSessionDetails</a>.
     * </p>
     * <p>
     * To get game sessions, specify one of the following: game session ID, fleet ID, or alias ID. You can filter this
     * request by game session status. Use the pagination parameters to retrieve results as a set of sequential pages.
     * If successful, a <a>GameSession</a> object is returned for each game session matching the request.
     * </p>
     * <p>
     * <i>Available in Amazon GameLift Local.</i>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeGameSessions(software.amazon.awssdk.services.gamelift.model.DescribeGameSessionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeGameSessionsPublisher publisher = client.describeGameSessionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeGameSessionsPublisher publisher = client.describeGameSessionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeGameSessionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeGameSessionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeGameSessions(software.amazon.awssdk.services.gamelift.model.DescribeGameSessionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeGameSessionsRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>TerminalRoutingStrategyException The service is unable to resolve the routing for a particular alias
     *         because it has a terminal <a>RoutingStrategy</a> associated with it. The message returned in this
     *         exception is the message defined in the routing strategy itself. Such requests should only be retried if
     *         the routing strategy for the specified alias is modified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeGameSessions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeGameSessions" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeGameSessionsPublisher describeGameSessionsPaginator(DescribeGameSessionsRequest describeGameSessionsRequest) {
        return new DescribeGameSessionsPublisher(this, applyPaginatorUserAgent(describeGameSessionsRequest));
    }

    /**
     * <p>
     * Retrieves information about a fleet's instances, including instance IDs. Use this operation to get details on all
     * instances in the fleet or get details on one specific instance.
     * </p>
     * <p>
     * To get a specific instance, specify fleet ID and instance ID. To get all instances in a fleet, specify a fleet ID
     * only. Use the pagination parameters to retrieve results as a set of sequential pages. If successful, an
     * <a>Instance</a> object is returned for each result.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-remote-access.html">Remotely Access
     * Fleet Instances</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html">Debug Fleet
     * Issues</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeInstances</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetInstanceAccess</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeInstancesRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeInstances" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeInstancesResponse> describeInstances(DescribeInstancesRequest describeInstancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInstancesRequest, DescribeInstancesResponse>()
                            .withOperationName("DescribeInstances")
                            .withMarshaller(new DescribeInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeInstancesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeInstancesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a fleet's instances, including instance IDs. Use this operation to get details on all
     * instances in the fleet or get details on one specific instance.
     * </p>
     * <p>
     * To get a specific instance, specify fleet ID and instance ID. To get all instances in a fleet, specify a fleet ID
     * only. Use the pagination parameters to retrieve results as a set of sequential pages. If successful, an
     * <a>Instance</a> object is returned for each result.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-remote-access.html">Remotely Access
     * Fleet Instances</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html">Debug Fleet
     * Issues</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeInstances</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetInstanceAccess</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeInstances(software.amazon.awssdk.services.gamelift.model.DescribeInstancesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeInstancesPublisher publisher = client.describeInstancesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeInstancesPublisher publisher = client.describeInstancesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeInstancesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeInstancesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeInstances(software.amazon.awssdk.services.gamelift.model.DescribeInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeInstancesRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeInstances" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeInstancesPublisher describeInstancesPaginator(DescribeInstancesRequest describeInstancesRequest) {
        return new DescribeInstancesPublisher(this, applyPaginatorUserAgent(describeInstancesRequest));
    }

    /**
     * <p>
     * Retrieves one or more matchmaking tickets. Use this operation to retrieve ticket information, including--after a
     * successful match is made--connection information for the resulting new game session.
     * </p>
     * <p>
     * To request matchmaking tickets, provide a list of up to 10 ticket IDs. If the request is successful, a ticket
     * object is returned for each requested ID that currently exists.
     * </p>
     * <p>
     * This operation is not designed to be continually called to track matchmaking ticket status. This practice can
     * cause you to exceed your API limit, which results in errors. Instead, as a best practice, set up an Amazon Simple
     * Notification Service (SNS) to receive notifications, and provide the topic ARN in the matchmaking configuration.
     * Continuously poling ticket status with <a>DescribeMatchmaking</a> should only be used for games in development
     * with low matchmaking usage.
     * </p>
     * <p/>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-client.html"> Add FlexMatch to a Game
     * Client</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-notification.html"> Set Up FlexMatch
     * Event Notification</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>AcceptMatch</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartMatchBackfill</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeMatchmakingRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeMatchmaking operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeMatchmaking
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeMatchmaking" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMatchmakingResponse> describeMatchmaking(
            DescribeMatchmakingRequest describeMatchmakingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMatchmakingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMatchmaking");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMatchmakingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMatchmakingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMatchmakingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMatchmakingRequest, DescribeMatchmakingResponse>()
                            .withOperationName("DescribeMatchmaking")
                            .withMarshaller(new DescribeMatchmakingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeMatchmakingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeMatchmakingRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeMatchmakingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details of FlexMatch matchmaking configurations.
     * </p>
     * <p>
     * This operation offers the following options: (1) retrieve all matchmaking configurations, (2) retrieve
     * configurations for a specified list, or (3) retrieve all configurations that use a specified rule set name. When
     * requesting multiple items, use the pagination parameters to retrieve results as a set of sequential pages.
     * </p>
     * <p>
     * If successful, a configuration is returned for each requested name. When specifying a list of names, only
     * configurations that currently exist are returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/matchmaker-build.html"> Setting Up FlexMatch
     * Matchmakers</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingConfigurations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingRuleSets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ValidateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingRuleSet</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeMatchmakingConfigurationsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeMatchmakingConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeMatchmakingConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeMatchmakingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMatchmakingConfigurationsResponse> describeMatchmakingConfigurations(
            DescribeMatchmakingConfigurationsRequest describeMatchmakingConfigurationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeMatchmakingConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMatchmakingConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMatchmakingConfigurationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeMatchmakingConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMatchmakingConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMatchmakingConfigurationsRequest, DescribeMatchmakingConfigurationsResponse>()
                            .withOperationName("DescribeMatchmakingConfigurations")
                            .withMarshaller(new DescribeMatchmakingConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeMatchmakingConfigurationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeMatchmakingConfigurationsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DescribeMatchmakingConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details of FlexMatch matchmaking configurations.
     * </p>
     * <p>
     * This operation offers the following options: (1) retrieve all matchmaking configurations, (2) retrieve
     * configurations for a specified list, or (3) retrieve all configurations that use a specified rule set name. When
     * requesting multiple items, use the pagination parameters to retrieve results as a set of sequential pages.
     * </p>
     * <p>
     * If successful, a configuration is returned for each requested name. When specifying a list of names, only
     * configurations that currently exist are returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/matchmaker-build.html"> Setting Up FlexMatch
     * Matchmakers</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingConfigurations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingRuleSets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ValidateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingRuleSet</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeMatchmakingConfigurations(software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingConfigurationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeMatchmakingConfigurationsPublisher publisher = client.describeMatchmakingConfigurationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeMatchmakingConfigurationsPublisher publisher = client.describeMatchmakingConfigurationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingConfigurationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingConfigurationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeMatchmakingConfigurations(software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeMatchmakingConfigurationsRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeMatchmakingConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeMatchmakingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeMatchmakingConfigurationsPublisher describeMatchmakingConfigurationsPaginator(
            DescribeMatchmakingConfigurationsRequest describeMatchmakingConfigurationsRequest) {
        return new DescribeMatchmakingConfigurationsPublisher(this,
                applyPaginatorUserAgent(describeMatchmakingConfigurationsRequest));
    }

    /**
     * <p>
     * Retrieves the details for FlexMatch matchmaking rule sets. You can request all existing rule sets for the Region,
     * or provide a list of one or more rule set names. When requesting multiple items, use the pagination parameters to
     * retrieve results as a set of sequential pages. If successful, a rule set is returned for each requested name.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-rulesets.html">Build a Rule Set</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingConfigurations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingRuleSets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ValidateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingRuleSet</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeMatchmakingRuleSetsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeMatchmakingRuleSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeMatchmakingRuleSets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeMatchmakingRuleSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMatchmakingRuleSetsResponse> describeMatchmakingRuleSets(
            DescribeMatchmakingRuleSetsRequest describeMatchmakingRuleSetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMatchmakingRuleSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMatchmakingRuleSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMatchmakingRuleSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMatchmakingRuleSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMatchmakingRuleSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMatchmakingRuleSetsRequest, DescribeMatchmakingRuleSetsResponse>()
                            .withOperationName("DescribeMatchmakingRuleSets")
                            .withMarshaller(new DescribeMatchmakingRuleSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeMatchmakingRuleSetsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeMatchmakingRuleSetsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeMatchmakingRuleSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details for FlexMatch matchmaking rule sets. You can request all existing rule sets for the Region,
     * or provide a list of one or more rule set names. When requesting multiple items, use the pagination parameters to
     * retrieve results as a set of sequential pages. If successful, a rule set is returned for each requested name.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-rulesets.html">Build a Rule Set</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingConfigurations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingRuleSets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ValidateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingRuleSet</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeMatchmakingRuleSets(software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingRuleSetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeMatchmakingRuleSetsPublisher publisher = client.describeMatchmakingRuleSetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeMatchmakingRuleSetsPublisher publisher = client.describeMatchmakingRuleSetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingRuleSetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingRuleSetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeMatchmakingRuleSets(software.amazon.awssdk.services.gamelift.model.DescribeMatchmakingRuleSetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeMatchmakingRuleSetsRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeMatchmakingRuleSets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeMatchmakingRuleSets"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeMatchmakingRuleSetsPublisher describeMatchmakingRuleSetsPaginator(
            DescribeMatchmakingRuleSetsRequest describeMatchmakingRuleSetsRequest) {
        return new DescribeMatchmakingRuleSetsPublisher(this, applyPaginatorUserAgent(describeMatchmakingRuleSetsRequest));
    }

    /**
     * <p>
     * Retrieves properties for one or more player sessions. This operation can be used in several ways: (1) provide a
     * <code>PlayerSessionId</code> to request properties for a specific player session; (2) provide a
     * <code>GameSessionId</code> to request properties for all player sessions in the specified game session; (3)
     * provide a <code>PlayerId</code> to request properties for all player sessions of a specified player.
     * </p>
     * <p>
     * To get game session record(s), specify only one of the following: a player session ID, a game session ID, or a
     * player ID. You can filter this request by player session status. Use the pagination parameters to retrieve
     * results as a set of sequential pages. If successful, a <a>PlayerSession</a> object is returned for each session
     * matching the request.
     * </p>
     * <p>
     * <i>Available in Amazon GameLift Local.</i>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreatePlayerSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreatePlayerSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribePlayerSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param describePlayerSessionsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribePlayerSessions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribePlayerSessions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribePlayerSessions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePlayerSessionsResponse> describePlayerSessions(
            DescribePlayerSessionsRequest describePlayerSessionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePlayerSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePlayerSessions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePlayerSessionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePlayerSessionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePlayerSessionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePlayerSessionsRequest, DescribePlayerSessionsResponse>()
                            .withOperationName("DescribePlayerSessions")
                            .withMarshaller(new DescribePlayerSessionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePlayerSessionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describePlayerSessionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribePlayerSessionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves properties for one or more player sessions. This operation can be used in several ways: (1) provide a
     * <code>PlayerSessionId</code> to request properties for a specific player session; (2) provide a
     * <code>GameSessionId</code> to request properties for all player sessions in the specified game session; (3)
     * provide a <code>PlayerId</code> to request properties for all player sessions of a specified player.
     * </p>
     * <p>
     * To get game session record(s), specify only one of the following: a player session ID, a game session ID, or a
     * player ID. You can filter this request by player session status. Use the pagination parameters to retrieve
     * results as a set of sequential pages. If successful, a <a>PlayerSession</a> object is returned for each session
     * matching the request.
     * </p>
     * <p>
     * <i>Available in Amazon GameLift Local.</i>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreatePlayerSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreatePlayerSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribePlayerSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describePlayerSessions(software.amazon.awssdk.services.gamelift.model.DescribePlayerSessionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribePlayerSessionsPublisher publisher = client.describePlayerSessionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribePlayerSessionsPublisher publisher = client.describePlayerSessionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribePlayerSessionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribePlayerSessionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describePlayerSessions(software.amazon.awssdk.services.gamelift.model.DescribePlayerSessionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describePlayerSessionsRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribePlayerSessions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribePlayerSessions"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribePlayerSessionsPublisher describePlayerSessionsPaginator(
            DescribePlayerSessionsRequest describePlayerSessionsRequest) {
        return new DescribePlayerSessionsPublisher(this, applyPaginatorUserAgent(describePlayerSessionsRequest));
    }

    /**
     * <p>
     * Retrieves a fleet's runtime configuration settings. The runtime configuration tells Amazon GameLift which server
     * processes to run (and how) on each instance in the fleet.
     * </p>
     * <p>
     * To get a runtime configuration, specify the fleet's unique identifier. If successful, a
     * <a>RuntimeConfiguration</a> object is returned for the requested fleet. If the requested fleet has been deleted,
     * the result set is empty.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-multiprocess.html">Running Multiple
     * Processes on a Fleet</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Describe fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetUtilization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeRuntimeConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetEvents</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeRuntimeConfigurationRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeRuntimeConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeRuntimeConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeRuntimeConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRuntimeConfigurationResponse> describeRuntimeConfiguration(
            DescribeRuntimeConfigurationRequest describeRuntimeConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRuntimeConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRuntimeConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRuntimeConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRuntimeConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRuntimeConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRuntimeConfigurationRequest, DescribeRuntimeConfigurationResponse>()
                            .withOperationName("DescribeRuntimeConfiguration")
                            .withMarshaller(new DescribeRuntimeConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeRuntimeConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeRuntimeConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeRuntimeConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all scaling policies applied to a fleet.
     * </p>
     * <p>
     * To get a fleet's scaling policies, specify the fleet ID. You can filter this request by policy status, such as to
     * retrieve only active scaling policies. Use the pagination parameters to retrieve results as a set of sequential
     * pages. If successful, set of <a>ScalingPolicy</a> objects is returned for the fleet.
     * </p>
     * <p>
     * A fleet may have all of its scaling policies suspended (<a>StopFleetActions</a>). This operation does not affect
     * the status of the scaling policies, which remains ACTIVE. To see whether a fleet's scaling policies are in force
     * or suspended, call <a>DescribeFleetAttributes</a> and check the stopped actions.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Manage scaling policies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>PutScalingPolicy</a> (auto-scaling)
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeScalingPolicies</a> (auto-scaling)
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteScalingPolicy</a> (auto-scaling)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Manage fleet actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartFleetActions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param describeScalingPoliciesRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeScalingPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeScalingPolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeScalingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScalingPoliciesResponse> describeScalingPolicies(
            DescribeScalingPoliciesRequest describeScalingPoliciesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScalingPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScalingPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeScalingPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeScalingPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeScalingPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingPoliciesRequest, DescribeScalingPoliciesResponse>()
                            .withOperationName("DescribeScalingPolicies")
                            .withMarshaller(new DescribeScalingPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeScalingPoliciesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeScalingPoliciesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeScalingPoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all scaling policies applied to a fleet.
     * </p>
     * <p>
     * To get a fleet's scaling policies, specify the fleet ID. You can filter this request by policy status, such as to
     * retrieve only active scaling policies. Use the pagination parameters to retrieve results as a set of sequential
     * pages. If successful, set of <a>ScalingPolicy</a> objects is returned for the fleet.
     * </p>
     * <p>
     * A fleet may have all of its scaling policies suspended (<a>StopFleetActions</a>). This operation does not affect
     * the status of the scaling policies, which remains ACTIVE. To see whether a fleet's scaling policies are in force
     * or suspended, call <a>DescribeFleetAttributes</a> and check the stopped actions.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Manage scaling policies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>PutScalingPolicy</a> (auto-scaling)
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeScalingPolicies</a> (auto-scaling)
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteScalingPolicy</a> (auto-scaling)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Manage fleet actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartFleetActions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeScalingPolicies(software.amazon.awssdk.services.gamelift.model.DescribeScalingPoliciesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeScalingPoliciesPublisher publisher = client.describeScalingPoliciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.DescribeScalingPoliciesPublisher publisher = client.describeScalingPoliciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.DescribeScalingPoliciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.DescribeScalingPoliciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalingPolicies(software.amazon.awssdk.services.gamelift.model.DescribeScalingPoliciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScalingPoliciesRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeScalingPolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeScalingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeScalingPoliciesPublisher describeScalingPoliciesPaginator(
            DescribeScalingPoliciesRequest describeScalingPoliciesRequest) {
        return new DescribeScalingPoliciesPublisher(this, applyPaginatorUserAgent(describeScalingPoliciesRequest));
    }

    /**
     * <p>
     * Retrieves properties for a Realtime script.
     * </p>
     * <p>
     * To request a script record, specify the script ID. If successful, an object containing the script properties is
     * returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/realtime-intro.html">Amazon GameLift Realtime
     * Servers</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListScripts</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteScript</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeScriptRequest
     * @return A Java Future containing the result of the DescribeScript operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeScript
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeScript" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScriptResponse> describeScript(DescribeScriptRequest describeScriptRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScriptRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScript");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeScriptResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeScriptResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeScriptResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScriptRequest, DescribeScriptResponse>()
                            .withOperationName("DescribeScript")
                            .withMarshaller(new DescribeScriptRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeScriptRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeScriptRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeScriptResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves valid VPC peering authorizations that are pending for the AWS account. This operation returns all VPC
     * peering authorizations and requests for peering. This includes those initiated and received by this account.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringAuthorizations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateVpcPeeringConnection</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringConnections</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringConnection</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeVpcPeeringAuthorizationsRequest
     * @return A Java Future containing the result of the DescribeVpcPeeringAuthorizations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeVpcPeeringAuthorizations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeVpcPeeringAuthorizations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVpcPeeringAuthorizationsResponse> describeVpcPeeringAuthorizations(
            DescribeVpcPeeringAuthorizationsRequest describeVpcPeeringAuthorizationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeVpcPeeringAuthorizationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVpcPeeringAuthorizations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVpcPeeringAuthorizationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeVpcPeeringAuthorizationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVpcPeeringAuthorizationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVpcPeeringAuthorizationsRequest, DescribeVpcPeeringAuthorizationsResponse>()
                            .withOperationName("DescribeVpcPeeringAuthorizations")
                            .withMarshaller(new DescribeVpcPeeringAuthorizationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeVpcPeeringAuthorizationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeVpcPeeringAuthorizationsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DescribeVpcPeeringAuthorizationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information on VPC peering connections. Use this operation to get peering information for all fleets or
     * for one specific fleet ID.
     * </p>
     * <p>
     * To retrieve connection information, call this operation from the AWS account that is used to manage the Amazon
     * GameLift fleets. Specify a fleet ID or leave the parameter empty to retrieve all connection records. If
     * successful, the retrieved information includes both active and pending connections. Active connections identify
     * the IpV4 CIDR block that the VPC uses to connect.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringAuthorizations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringAuthorization</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateVpcPeeringConnection</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeVpcPeeringConnections</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteVpcPeeringConnection</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeVpcPeeringConnectionsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the DescribeVpcPeeringConnections operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.DescribeVpcPeeringConnections
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/DescribeVpcPeeringConnections"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVpcPeeringConnectionsResponse> describeVpcPeeringConnections(
            DescribeVpcPeeringConnectionsRequest describeVpcPeeringConnectionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeVpcPeeringConnectionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVpcPeeringConnections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVpcPeeringConnectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVpcPeeringConnectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVpcPeeringConnectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVpcPeeringConnectionsRequest, DescribeVpcPeeringConnectionsResponse>()
                            .withOperationName("DescribeVpcPeeringConnections")
                            .withMarshaller(new DescribeVpcPeeringConnectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeVpcPeeringConnectionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeVpcPeeringConnectionsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeVpcPeeringConnectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the location of stored game session logs for a specified game session. When a game session is
     * terminated, Amazon GameLift automatically stores the logs in Amazon S3 and retains them for 14 days. Use this URL
     * to download the logs.
     * </p>
     * <note>
     * <p>
     * See the <a href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html#limits_gamelift">AWS
     * Service Limits</a> page for maximum log file sizes. Log files that exceed this limit are not saved.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param getGameSessionLogUrlRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the GetGameSessionLogUrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.GetGameSessionLogUrl
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/GetGameSessionLogUrl" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetGameSessionLogUrlResponse> getGameSessionLogUrl(
            GetGameSessionLogUrlRequest getGameSessionLogUrlRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGameSessionLogUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGameSessionLogUrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGameSessionLogUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetGameSessionLogUrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGameSessionLogUrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGameSessionLogUrlRequest, GetGameSessionLogUrlResponse>()
                            .withOperationName("GetGameSessionLogUrl")
                            .withMarshaller(new GetGameSessionLogUrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getGameSessionLogUrlRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getGameSessionLogUrlRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetGameSessionLogUrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Requests remote access to a fleet instance. Remote access is useful for debugging, gathering benchmarking data,
     * or observing activity in real time.
     * </p>
     * <p>
     * To remotely access an instance, you need credentials that match the operating system of the instance. For a
     * Windows instance, Amazon GameLift returns a user name and password as strings for use with a Windows Remote
     * Desktop client. For a Linux instance, Amazon GameLift returns a user name and RSA private key, also as strings,
     * for use with an SSH client. The private key must be saved in the proper format to a <code>.pem</code> file before
     * using. If you're making this request using the AWS CLI, saving the secret can be handled as part of the
     * GetInstanceAccess request, as shown in one of the examples for this operation.
     * </p>
     * <p>
     * To request access to a specific instance, specify the IDs of both the instance and the fleet it belongs to. You
     * can retrieve a fleet's instance IDs by calling <a>DescribeInstances</a>. If successful, an <a>InstanceAccess</a>
     * object is returned that contains the instance's IP address and a set of credentials.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-remote-access.html">Remotely Access
     * Fleet Instances</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-creating-debug.html">Debug Fleet
     * Issues</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeInstances</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetInstanceAccess</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getInstanceAccessRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the GetInstanceAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.GetInstanceAccess
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/GetInstanceAccess" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetInstanceAccessResponse> getInstanceAccess(GetInstanceAccessRequest getInstanceAccessRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInstanceAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInstanceAccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInstanceAccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInstanceAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInstanceAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInstanceAccessRequest, GetInstanceAccessResponse>()
                            .withOperationName("GetInstanceAccess")
                            .withMarshaller(new GetInstanceAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getInstanceAccessRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getInstanceAccessRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetInstanceAccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all aliases for this AWS account. You can filter the result set by alias name and/or routing strategy
     * type. Use the pagination parameters to retrieve results in sequential pages.
     * </p>
     * <note>
     * <p>
     * Returned aliases are not listed in any particular order.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResolveAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listAliasesRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the ListAliases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListAliases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListAliases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAliasesResponse> listAliases(ListAliasesRequest listAliasesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAliases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAliasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAliasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAliasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAliasesRequest, ListAliasesResponse>()
                            .withOperationName("ListAliases").withMarshaller(new ListAliasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAliasesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAliasesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListAliasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all aliases for this AWS account. You can filter the result set by alias name and/or routing strategy
     * type. Use the pagination parameters to retrieve results in sequential pages.
     * </p>
     * <note>
     * <p>
     * Returned aliases are not listed in any particular order.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResolveAlias</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #listAliases(software.amazon.awssdk.services.gamelift.model.ListAliasesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListAliasesPublisher publisher = client.listAliasesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListAliasesPublisher publisher = client.listAliasesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.ListAliasesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.ListAliasesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAliases(software.amazon.awssdk.services.gamelift.model.ListAliasesRequest)} operation.</b>
     * </p>
     *
     * @param listAliasesRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListAliases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListAliases" target="_top">AWS API
     *      Documentation</a>
     */
    public ListAliasesPublisher listAliasesPaginator(ListAliasesRequest listAliasesRequest) {
        return new ListAliasesPublisher(this, applyPaginatorUserAgent(listAliasesRequest));
    }

    /**
     * <p>
     * Retrieves build resources for all builds associated with the AWS account in use. You can limit results to builds
     * that are in a specific status by using the <code>Status</code> parameter. Use the pagination parameters to
     * retrieve results in a set of sequential pages.
     * </p>
     * <note>
     * <p>
     * Build resources are not listed in any particular order.
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-build-intro.html"> Upload a Custom
     * Server Build</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListBuilds</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteBuild</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listBuildsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the ListBuilds operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListBuilds
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListBuilds" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListBuildsResponse> listBuilds(ListBuildsRequest listBuildsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBuildsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBuilds");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBuildsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListBuildsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBuildsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBuildsRequest, ListBuildsResponse>().withOperationName("ListBuilds")
                            .withMarshaller(new ListBuildsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listBuildsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listBuildsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListBuildsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves build resources for all builds associated with the AWS account in use. You can limit results to builds
     * that are in a specific status by using the <code>Status</code> parameter. Use the pagination parameters to
     * retrieve results in a set of sequential pages.
     * </p>
     * <note>
     * <p>
     * Build resources are not listed in any particular order.
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-build-intro.html"> Upload a Custom
     * Server Build</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListBuilds</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteBuild</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #listBuilds(software.amazon.awssdk.services.gamelift.model.ListBuildsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListBuildsPublisher publisher = client.listBuildsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListBuildsPublisher publisher = client.listBuildsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.ListBuildsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.ListBuildsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBuilds(software.amazon.awssdk.services.gamelift.model.ListBuildsRequest)} operation.</b>
     * </p>
     *
     * @param listBuildsRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListBuilds
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListBuilds" target="_top">AWS API
     *      Documentation</a>
     */
    public ListBuildsPublisher listBuildsPaginator(ListBuildsRequest listBuildsRequest) {
        return new ListBuildsPublisher(this, applyPaginatorUserAgent(listBuildsRequest));
    }

    /**
     * <p>
     * Retrieves a collection of fleet resources for this AWS account. You can filter the result set to find only those
     * fleets that are deployed with a specific build or script. Use the pagination parameters to retrieve results in
     * sequential pages.
     * </p>
     * <note>
     * <p>
     * Fleet resources are not listed in a particular order.
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listFleetsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the ListFleets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListFleets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListFleets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFleetsResponse> listFleets(ListFleetsRequest listFleetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFleetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFleets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFleetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFleetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFleetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFleetsRequest, ListFleetsResponse>().withOperationName("ListFleets")
                            .withMarshaller(new ListFleetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFleetsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listFleetsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListFleetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a collection of fleet resources for this AWS account. You can filter the result set to find only those
     * fleets that are deployed with a specific build or script. Use the pagination parameters to retrieve results in
     * sequential pages.
     * </p>
     * <note>
     * <p>
     * Fleet resources are not listed in a particular order.
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #listFleets(software.amazon.awssdk.services.gamelift.model.ListFleetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListFleetsPublisher publisher = client.listFleetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListFleetsPublisher publisher = client.listFleetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.ListFleetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.ListFleetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFleets(software.amazon.awssdk.services.gamelift.model.ListFleetsRequest)} operation.</b>
     * </p>
     *
     * @param listFleetsRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListFleets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListFleets" target="_top">AWS API
     *      Documentation</a>
     */
    public ListFleetsPublisher listFleetsPaginator(ListFleetsRequest listFleetsRequest) {
        return new ListFleetsPublisher(this, applyPaginatorUserAgent(listFleetsRequest));
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Retrieves information on all game servers groups that exist in the current AWS account for the selected Region.
     * Use the pagination parameters to retrieve results in a set of sequential segments.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServerGroups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResumeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SuspendGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerInstances</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listGameServerGroupsRequest
     * @return A Java Future containing the result of the ListGameServerGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListGameServerGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListGameServerGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGameServerGroupsResponse> listGameServerGroups(
            ListGameServerGroupsRequest listGameServerGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGameServerGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGameServerGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGameServerGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGameServerGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGameServerGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGameServerGroupsRequest, ListGameServerGroupsResponse>()
                            .withOperationName("ListGameServerGroups")
                            .withMarshaller(new ListGameServerGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listGameServerGroupsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listGameServerGroupsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListGameServerGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Retrieves information on all game servers groups that exist in the current AWS account for the selected Region.
     * Use the pagination parameters to retrieve results in a set of sequential segments.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServerGroups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResumeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SuspendGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerInstances</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGameServerGroups(software.amazon.awssdk.services.gamelift.model.ListGameServerGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListGameServerGroupsPublisher publisher = client.listGameServerGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListGameServerGroupsPublisher publisher = client.listGameServerGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.ListGameServerGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.ListGameServerGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGameServerGroups(software.amazon.awssdk.services.gamelift.model.ListGameServerGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listGameServerGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListGameServerGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListGameServerGroups" target="_top">AWS
     *      API Documentation</a>
     */
    public ListGameServerGroupsPublisher listGameServerGroupsPaginator(ListGameServerGroupsRequest listGameServerGroupsRequest) {
        return new ListGameServerGroupsPublisher(this, applyPaginatorUserAgent(listGameServerGroupsRequest));
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Retrieves information on all game servers that are currently active in a specified game server group. You can opt
     * to sort the list by game server age. Use the pagination parameters to retrieve results in a set of sequential
     * segments.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>RegisterGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServers</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ClaimGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeregisterGameServer</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listGameServersRequest
     * @return A Java Future containing the result of the ListGameServers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListGameServers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListGameServers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGameServersResponse> listGameServers(ListGameServersRequest listGameServersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGameServersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGameServers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGameServersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGameServersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGameServersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGameServersRequest, ListGameServersResponse>()
                            .withOperationName("ListGameServers")
                            .withMarshaller(new ListGameServersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listGameServersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listGameServersRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListGameServersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Retrieves information on all game servers that are currently active in a specified game server group. You can opt
     * to sort the list by game server age. Use the pagination parameters to retrieve results in a set of sequential
     * segments.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>RegisterGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServers</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ClaimGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeregisterGameServer</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGameServers(software.amazon.awssdk.services.gamelift.model.ListGameServersRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListGameServersPublisher publisher = client.listGameServersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListGameServersPublisher publisher = client.listGameServersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.ListGameServersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.ListGameServersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGameServers(software.amazon.awssdk.services.gamelift.model.ListGameServersRequest)} operation.</b>
     * </p>
     *
     * @param listGameServersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListGameServers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListGameServers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListGameServersPublisher listGameServersPaginator(ListGameServersRequest listGameServersRequest) {
        return new ListGameServersPublisher(this, applyPaginatorUserAgent(listGameServersRequest));
    }

    /**
     * <p>
     * Retrieves script records for all Realtime scripts that are associated with the AWS account in use.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/realtime-intro.html">Amazon GameLift Realtime
     * Servers</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListScripts</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteScript</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listScriptsRequest
     * @return A Java Future containing the result of the ListScripts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListScripts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListScripts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListScriptsResponse> listScripts(ListScriptsRequest listScriptsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listScriptsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListScripts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListScriptsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListScriptsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListScriptsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListScriptsRequest, ListScriptsResponse>()
                            .withOperationName("ListScripts").withMarshaller(new ListScriptsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listScriptsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listScriptsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListScriptsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves script records for all Realtime scripts that are associated with the AWS account in use.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/realtime-intro.html">Amazon GameLift Realtime
     * Servers</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListScripts</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteScript</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #listScripts(software.amazon.awssdk.services.gamelift.model.ListScriptsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListScriptsPublisher publisher = client.listScriptsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.ListScriptsPublisher publisher = client.listScriptsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.ListScriptsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.ListScriptsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listScripts(software.amazon.awssdk.services.gamelift.model.ListScriptsRequest)} operation.</b>
     * </p>
     *
     * @param listScriptsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListScripts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListScripts" target="_top">AWS API
     *      Documentation</a>
     */
    public ListScriptsPublisher listScriptsPaginator(ListScriptsRequest listScriptsRequest) {
        return new ListScriptsPublisher(this, applyPaginatorUserAgent(listScriptsRequest));
    }

    /**
     * <p>
     * Retrieves all tags that are assigned to a GameLift resource. Resource tags are used to organize AWS resources for
     * a range of purposes. This operation handles the permissions necessary to manage tags for the following GameLift
     * resource types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Build
     * </p>
     * </li>
     * <li>
     * <p>
     * Script
     * </p>
     * </li>
     * <li>
     * <p>
     * Fleet
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias
     * </p>
     * </li>
     * <li>
     * <p>
     * GameSessionQueue
     * </p>
     * </li>
     * <li>
     * <p>
     * MatchmakingConfiguration
     * </p>
     * </li>
     * <li>
     * <p>
     * MatchmakingRuleSet
     * </p>
     * </li>
     * </ul>
     * <p>
     * To list tags for a resource, specify the unique ARN value for the resource.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS Resources</a> in the <i>AWS
     * General Reference</i>
     * </p>
     * <p>
     * <a href="http://aws.amazon.com/answers/account-management/aws-tagging-strategies/"> AWS Tagging Strategies</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>TagResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UntagResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListTagsForResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a scaling policy for a fleet. Scaling policies are used to automatically scale a fleet's
     * hosting capacity to meet player demand. An active scaling policy instructs Amazon GameLift to track a fleet
     * metric and automatically change the fleet's capacity when a certain threshold is reached. There are two types of
     * scaling policies: target-based and rule-based. Use a target-based policy to quickly and efficiently manage fleet
     * scaling; this option is the most commonly used. Use rule-based policies when you need to exert fine-grained
     * control over auto-scaling.
     * </p>
     * <p>
     * Fleets can have multiple scaling policies of each type in force at the same time; you can have one target-based
     * policy, one or multiple rule-based scaling policies, or both. We recommend caution, however, because multiple
     * auto-scaling policies can have unintended consequences.
     * </p>
     * <p>
     * You can temporarily suspend all scaling policies for a fleet by calling <a>StopFleetActions</a> with the fleet
     * action AUTO_SCALING. To resume scaling policies, call <a>StartFleetActions</a> with the same fleet action. To
     * stop just one scaling policy--or to permanently remove it, you must delete the policy with
     * <a>DeleteScalingPolicy</a>.
     * </p>
     * <p>
     * Learn more about how to work with auto-scaling in <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-autoscaling.html">Set Up Fleet Automatic
     * Scaling</a>.
     * </p>
     * <p>
     * <b>Target-based policy</b>
     * </p>
     * <p>
     * A target-based policy tracks a single metric: PercentAvailableGameSessions. This metric tells us how much of a
     * fleet's hosting capacity is ready to host game sessions but is not currently in use. This is the fleet's buffer;
     * it measures the additional player demand that the fleet could handle at current capacity. With a target-based
     * policy, you set your ideal buffer size and leave it to Amazon GameLift to take whatever action is needed to
     * maintain that target.
     * </p>
     * <p>
     * For example, you might choose to maintain a 10% buffer for a fleet that has the capacity to host 100 simultaneous
     * game sessions. This policy tells Amazon GameLift to take action whenever the fleet's available capacity falls
     * below or rises above 10 game sessions. Amazon GameLift will start new instances or stop unused instances in order
     * to return to the 10% buffer.
     * </p>
     * <p>
     * To create or update a target-based policy, specify a fleet ID and name, and set the policy type to "TargetBased".
     * Specify the metric to track (PercentAvailableGameSessions) and reference a <a>TargetConfiguration</a> object with
     * your desired buffer value. Exclude all other parameters. On a successful request, the policy name is returned.
     * The scaling policy is automatically in force as soon as it's successfully created. If the fleet's auto-scaling
     * actions are temporarily suspended, the new policy will be in force once the fleet actions are restarted.
     * </p>
     * <p>
     * <b>Rule-based policy</b>
     * </p>
     * <p>
     * A rule-based policy tracks specified fleet metric, sets a threshold value, and specifies the type of action to
     * initiate when triggered. With a rule-based policy, you can select from several available fleet metrics. Each
     * policy specifies whether to scale up or scale down (and by how much), so you need one policy for each type of
     * action.
     * </p>
     * <p>
     * For example, a policy may make the following statement:
     * "If the percentage of idle instances is greater than 20% for more than 15 minutes, then reduce the fleet capacity by 10%."
     * </p>
     * <p>
     * A policy's rule statement has the following structure:
     * </p>
     * <p>
     * If <code>[MetricName]</code> is <code>[ComparisonOperator]</code> <code>[Threshold]</code> for
     * <code>[EvaluationPeriods]</code> minutes, then <code>[ScalingAdjustmentType]</code> to/by
     * <code>[ScalingAdjustment]</code>.
     * </p>
     * <p>
     * To implement the example, the rule statement would look like this:
     * </p>
     * <p>
     * If <code>[PercentIdleInstances]</code> is <code>[GreaterThanThreshold]</code> <code>[20]</code> for
     * <code>[15]</code> minutes, then <code>[PercentChangeInCapacity]</code> to/by <code>[10]</code>.
     * </p>
     * <p>
     * To create or update a scaling policy, specify a unique combination of name and fleet ID, and set the policy type
     * to "RuleBased". Specify the parameter values for a policy rule statement. On a successful request, the policy
     * name is returned. Scaling policies are automatically in force as soon as they're successfully created. If the
     * fleet's auto-scaling actions are temporarily suspended, the new policy will be in force once the fleet actions
     * are restarted.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeEC2InstanceLimits</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Manage scaling policies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>PutScalingPolicy</a> (auto-scaling)
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeScalingPolicies</a> (auto-scaling)
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteScalingPolicy</a> (auto-scaling)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Manage fleet actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartFleetActions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param putScalingPolicyRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the PutScalingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.PutScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/PutScalingPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutScalingPolicyResponse> putScalingPolicy(PutScalingPolicyRequest putScalingPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putScalingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutScalingPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutScalingPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutScalingPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutScalingPolicyRequest, PutScalingPolicyResponse>()
                            .withOperationName("PutScalingPolicy")
                            .withMarshaller(new PutScalingPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putScalingPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putScalingPolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutScalingPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Creates a new game server resource and notifies GameLift FleetIQ that the game server is ready to host gameplay
     * and players. This operation is called by a game server process that is running on an instance in a game server
     * group. Registering game servers enables GameLift FleetIQ to track available game servers and enables game clients
     * and services to claim a game server for a new game session.
     * </p>
     * <p>
     * To register a game server, identify the game server group and instance where the game server is running, and
     * provide a unique identifier for the game server. You can also include connection and game server data. When a
     * game client or service requests a game server by calling <a>ClaimGameServer</a>, this information is returned in
     * the response.
     * </p>
     * <p>
     * Once a game server is successfully registered, it is put in status <code>AVAILABLE</code>. A request to register
     * a game server may fail if the instance it is running on is in the process of shutting down as part of instance
     * balancing or scale-down activity.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>RegisterGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServers</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ClaimGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeregisterGameServer</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param registerGameServerRequest
     * @return A Java Future containing the result of the RegisterGameServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>LimitExceededException The requested operation would cause the resource to exceed the allowed service
     *         limit. Resolve the issue before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.RegisterGameServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/RegisterGameServer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterGameServerResponse> registerGameServer(RegisterGameServerRequest registerGameServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerGameServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterGameServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterGameServerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterGameServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterGameServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterGameServerRequest, RegisterGameServerResponse>()
                            .withOperationName("RegisterGameServer")
                            .withMarshaller(new RegisterGameServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerGameServerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = registerGameServerRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<RegisterGameServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a fresh set of credentials for use when uploading a new set of game build files to Amazon GameLift's
     * Amazon S3. This is done as part of the build creation process; see <a>CreateBuild</a>.
     * </p>
     * <p>
     * To request new credentials, specify the build ID as returned with an initial <code>CreateBuild</code> request. If
     * successful, a new set of credentials are returned, along with the S3 storage location associated with the build
     * ID.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-build-cli-uploading.html#gamelift-build-cli-uploading-create-build"
     * > Create a Build with Files in S3</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListBuilds</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteBuild</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param requestUploadCredentialsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the RequestUploadCredentials operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.RequestUploadCredentials
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/RequestUploadCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RequestUploadCredentialsResponse> requestUploadCredentials(
            RequestUploadCredentialsRequest requestUploadCredentialsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestUploadCredentialsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RequestUploadCredentials");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RequestUploadCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RequestUploadCredentialsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RequestUploadCredentialsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RequestUploadCredentialsRequest, RequestUploadCredentialsResponse>()
                            .withOperationName("RequestUploadCredentials")
                            .withMarshaller(new RequestUploadCredentialsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(requestUploadCredentialsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = requestUploadCredentialsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<RequestUploadCredentialsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the fleet ID that an alias is currently pointing to.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResolveAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param resolveAliasRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the ResolveAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>TerminalRoutingStrategyException The service is unable to resolve the routing for a particular alias
     *         because it has a terminal <a>RoutingStrategy</a> associated with it. The message returned in this
     *         exception is the message defined in the routing strategy itself. Such requests should only be retried if
     *         the routing strategy for the specified alias is modified.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ResolveAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ResolveAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ResolveAliasResponse> resolveAlias(ResolveAliasRequest resolveAliasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resolveAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResolveAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResolveAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ResolveAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResolveAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResolveAliasRequest, ResolveAliasResponse>()
                            .withOperationName("ResolveAlias").withMarshaller(new ResolveAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(resolveAliasRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = resolveAliasRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ResolveAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Reinstates activity on a game server group after it has been suspended. A game server group might be suspended by
     * the<a>SuspendGameServerGroup</a> operation, or it might be suspended involuntarily due to a configuration
     * problem. In the second case, you can manually resume activity on the group once the configuration problem has
     * been resolved. Refer to the game server group status and status reason for more information on why group activity
     * is suspended.
     * </p>
     * <p>
     * To resume activity, specify a game server group ARN and the type of activity to be resumed. If successful, a
     * <a>GameServerGroup</a> object is returned showing that the resumed activity is no longer listed in
     * <code>SuspendedActions</code>.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServerGroups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResumeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SuspendGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerInstances</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param resumeGameServerGroupRequest
     * @return A Java Future containing the result of the ResumeGameServerGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ResumeGameServerGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ResumeGameServerGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ResumeGameServerGroupResponse> resumeGameServerGroup(
            ResumeGameServerGroupRequest resumeGameServerGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resumeGameServerGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResumeGameServerGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResumeGameServerGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ResumeGameServerGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResumeGameServerGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResumeGameServerGroupRequest, ResumeGameServerGroupResponse>()
                            .withOperationName("ResumeGameServerGroup")
                            .withMarshaller(new ResumeGameServerGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(resumeGameServerGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = resumeGameServerGroupRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ResumeGameServerGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all active game sessions that match a set of search criteria and sorts them in a specified order. You
     * can search or sort by the following game session attributes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>gameSessionId</b> -- A unique identifier for the game session. You can use either a <code>GameSessionId</code>
     * or <code>GameSessionArn</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>gameSessionName</b> -- Name assigned to a game session. This value is set when requesting a new game session
     * with <a>CreateGameSession</a> or updating with <a>UpdateGameSession</a>. Game session names do not need to be
     * unique to a game session.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>gameSessionProperties</b> -- Custom data defined in a game session's <code>GameProperty</code> parameter.
     * <code>GameProperty</code> values are stored as key:value pairs; the filter expression must indicate the key and a
     * string to search the data values for. For example, to search for game sessions with custom data containing the
     * key:value pair "gameMode:brawl", specify the following: <code>gameSessionProperties.gameMode = "brawl"</code>.
     * All custom data values are searched as strings.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>maximumSessions</b> -- Maximum number of player sessions allowed for a game session. This value is set when
     * requesting a new game session with <a>CreateGameSession</a> or updating with <a>UpdateGameSession</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>creationTimeMillis</b> -- Value indicating when a game session was created. It is expressed in Unix time as
     * milliseconds.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>playerSessionCount</b> -- Number of players currently connected to a game session. This value changes rapidly
     * as players join the session or drop out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>hasAvailablePlayerSessions</b> -- Boolean value indicating whether a game session has reached its maximum
     * number of players. It is highly recommended that all search requests include this filter attribute to optimize
     * search performance and return only sessions that players can join.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Returned values for <code>playerSessionCount</code> and <code>hasAvailablePlayerSessions</code> change quickly as
     * players join sessions and others drop out. Results should be considered a snapshot in time. Be sure to refresh
     * search results often, and handle sessions that fill up before a player can join.
     * </p>
     * </note>
     * <p>
     * To search or sort, specify either a fleet ID or an alias ID, and provide a search filter expression, a sort
     * expression, or both. If successful, a collection of <a>GameSession</a> objects matching the request is returned.
     * Use the pagination parameters to retrieve results as a set of sequential pages.
     * </p>
     * <p>
     * You can search for game sessions one fleet at a time only. To find game sessions across multiple fleets, you must
     * search each fleet separately and combine the results. This search feature finds only game sessions that are in
     * <code>ACTIVE</code> status. To locate games in statuses other than active, use <a>DescribeGameSessionDetails</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param searchGameSessionsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the SearchGameSessions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>TerminalRoutingStrategyException The service is unable to resolve the routing for a particular alias
     *         because it has a terminal <a>RoutingStrategy</a> associated with it. The message returned in this
     *         exception is the message defined in the routing strategy itself. Such requests should only be retried if
     *         the routing strategy for the specified alias is modified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.SearchGameSessions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/SearchGameSessions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchGameSessionsResponse> searchGameSessions(SearchGameSessionsRequest searchGameSessionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchGameSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchGameSessions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchGameSessionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchGameSessionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchGameSessionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchGameSessionsRequest, SearchGameSessionsResponse>()
                            .withOperationName("SearchGameSessions")
                            .withMarshaller(new SearchGameSessionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchGameSessionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = searchGameSessionsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<SearchGameSessionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all active game sessions that match a set of search criteria and sorts them in a specified order. You
     * can search or sort by the following game session attributes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>gameSessionId</b> -- A unique identifier for the game session. You can use either a <code>GameSessionId</code>
     * or <code>GameSessionArn</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>gameSessionName</b> -- Name assigned to a game session. This value is set when requesting a new game session
     * with <a>CreateGameSession</a> or updating with <a>UpdateGameSession</a>. Game session names do not need to be
     * unique to a game session.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>gameSessionProperties</b> -- Custom data defined in a game session's <code>GameProperty</code> parameter.
     * <code>GameProperty</code> values are stored as key:value pairs; the filter expression must indicate the key and a
     * string to search the data values for. For example, to search for game sessions with custom data containing the
     * key:value pair "gameMode:brawl", specify the following: <code>gameSessionProperties.gameMode = "brawl"</code>.
     * All custom data values are searched as strings.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>maximumSessions</b> -- Maximum number of player sessions allowed for a game session. This value is set when
     * requesting a new game session with <a>CreateGameSession</a> or updating with <a>UpdateGameSession</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>creationTimeMillis</b> -- Value indicating when a game session was created. It is expressed in Unix time as
     * milliseconds.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>playerSessionCount</b> -- Number of players currently connected to a game session. This value changes rapidly
     * as players join the session or drop out.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>hasAvailablePlayerSessions</b> -- Boolean value indicating whether a game session has reached its maximum
     * number of players. It is highly recommended that all search requests include this filter attribute to optimize
     * search performance and return only sessions that players can join.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Returned values for <code>playerSessionCount</code> and <code>hasAvailablePlayerSessions</code> change quickly as
     * players join sessions and others drop out. Results should be considered a snapshot in time. Be sure to refresh
     * search results often, and handle sessions that fill up before a player can join.
     * </p>
     * </note>
     * <p>
     * To search or sort, specify either a fleet ID or an alias ID, and provide a search filter expression, a sort
     * expression, or both. If successful, a collection of <a>GameSession</a> objects matching the request is returned.
     * Use the pagination parameters to retrieve results as a set of sequential pages.
     * </p>
     * <p>
     * You can search for game sessions one fleet at a time only. To find game sessions across multiple fleets, you must
     * search each fleet separately and combine the results. This search feature finds only game sessions that are in
     * <code>ACTIVE</code> status. To locate games in statuses other than active, use <a>DescribeGameSessionDetails</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #searchGameSessions(software.amazon.awssdk.services.gamelift.model.SearchGameSessionsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.SearchGameSessionsPublisher publisher = client.searchGameSessionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamelift.paginators.SearchGameSessionsPublisher publisher = client.searchGameSessionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamelift.model.SearchGameSessionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamelift.model.SearchGameSessionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchGameSessions(software.amazon.awssdk.services.gamelift.model.SearchGameSessionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param searchGameSessionsRequest
     *        Represents the input for a request operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>TerminalRoutingStrategyException The service is unable to resolve the routing for a particular alias
     *         because it has a terminal <a>RoutingStrategy</a> associated with it. The message returned in this
     *         exception is the message defined in the routing strategy itself. Such requests should only be retried if
     *         the routing strategy for the specified alias is modified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.SearchGameSessions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/SearchGameSessions" target="_top">AWS
     *      API Documentation</a>
     */
    public SearchGameSessionsPublisher searchGameSessionsPaginator(SearchGameSessionsRequest searchGameSessionsRequest) {
        return new SearchGameSessionsPublisher(this, applyPaginatorUserAgent(searchGameSessionsRequest));
    }

    /**
     * <p>
     * Resumes activity on a fleet that was suspended with <a>StopFleetActions</a>. Currently, this operation is used to
     * restart a fleet's auto-scaling activity.
     * </p>
     * <p>
     * To start fleet actions, specify the fleet ID and the type of actions to restart. When auto-scaling fleet actions
     * are restarted, Amazon GameLift once again initiates scaling events as triggered by the fleet's scaling policies.
     * If actions on the fleet were never stopped, this operation will have no effect. You can view a fleet's stopped
     * actions using <a>DescribeFleetAttributes</a>.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param startFleetActionsRequest
     * @return A Java Future containing the result of the StartFleetActions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.StartFleetActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/StartFleetActions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartFleetActionsResponse> startFleetActions(StartFleetActionsRequest startFleetActionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFleetActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFleetActions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartFleetActionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartFleetActionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartFleetActionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartFleetActionsRequest, StartFleetActionsResponse>()
                            .withOperationName("StartFleetActions")
                            .withMarshaller(new StartFleetActionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startFleetActionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startFleetActionsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StartFleetActionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Places a request for a new game session in a queue (see <a>CreateGameSessionQueue</a>). When processing a
     * placement request, Amazon GameLift searches for available resources on the queue's destinations, scanning each
     * until it finds resources or the placement request times out.
     * </p>
     * <p>
     * A game session placement request can also request player sessions. When a new game session is successfully
     * created, Amazon GameLift creates a player session for each player included in the request.
     * </p>
     * <p>
     * When placing a game session, by default Amazon GameLift tries each fleet in the order they are listed in the
     * queue configuration. Ideally, a queue's destinations are listed in preference order.
     * </p>
     * <p>
     * Alternatively, when requesting a game session with players, you can also provide latency data for each player in
     * relevant Regions. Latency data indicates the performance lag a player experiences when connected to a fleet in
     * the Region. Amazon GameLift uses latency data to reorder the list of destinations to place the game session in a
     * Region with minimal lag. If latency data is provided for multiple players, Amazon GameLift calculates each
     * Region's average lag for all players and reorders to get the best game play across all players.
     * </p>
     * <p>
     * To place a new game session request, specify the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The queue name and a set of game session properties and settings
     * </p>
     * </li>
     * <li>
     * <p>
     * A unique ID (such as a UUID) for the placement. You use this ID to track the status of the placement request
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) A set of player data and a unique player ID for each player that you are joining to the new game
     * session (player data is optional, but if you include it, you must also provide a unique ID for each player)
     * </p>
     * </li>
     * <li>
     * <p>
     * Latency data for all players (if you want to optimize game play for the players)
     * </p>
     * </li>
     * </ul>
     * <p>
     * If successful, a new game session placement is created.
     * </p>
     * <p>
     * To track the status of a placement request, call <a>DescribeGameSessionPlacement</a> and check the request's
     * status. If the status is <code>FULFILLED</code>, a new game session has been created and a game session ARN and
     * Region are referenced. If the placement request times out, you can resubmit the request or retry it with a
     * different queue.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param startGameSessionPlacementRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the StartGameSessionPlacement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.StartGameSessionPlacement
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/StartGameSessionPlacement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartGameSessionPlacementResponse> startGameSessionPlacement(
            StartGameSessionPlacementRequest startGameSessionPlacementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startGameSessionPlacementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartGameSessionPlacement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartGameSessionPlacementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartGameSessionPlacementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartGameSessionPlacementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartGameSessionPlacementRequest, StartGameSessionPlacementResponse>()
                            .withOperationName("StartGameSessionPlacement")
                            .withMarshaller(new StartGameSessionPlacementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startGameSessionPlacementRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startGameSessionPlacementRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<StartGameSessionPlacementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Finds new players to fill open slots in an existing game session. This operation can be used to add players to
     * matched games that start with fewer than the maximum number of players or to replace players when they drop out.
     * By backfilling with the same matchmaker used to create the original match, you ensure that new players meet the
     * match criteria and maintain a consistent experience throughout the game session. You can backfill a match anytime
     * after a game session has been created.
     * </p>
     * <p>
     * To request a match backfill, specify a unique ticket ID, the existing game session's ARN, a matchmaking
     * configuration, and a set of data that describes all current players in the game session. If successful, a match
     * backfill ticket is created and returned with status set to QUEUED. The ticket is placed in the matchmaker's
     * ticket pool and processed. Track the status of the ticket to respond as needed.
     * </p>
     * <p>
     * The process of finding backfill matches is essentially identical to the initial matchmaking process. The
     * matchmaker searches the pool and groups tickets together to form potential matches, allowing only one backfill
     * ticket per potential match. Once the a match is formed, the matchmaker creates player sessions for the new
     * players. All tickets in the match are updated with the game session's connection information, and the
     * <a>GameSession</a> object is updated to include matchmaker data on the new players. For more detail on how match
     * backfill requests are processed, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-match.html"> How Amazon GameLift
     * FlexMatch Works</a>.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-backfill.html"> Backfill Existing Games
     * with FlexMatch</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-match.html"> How GameLift FlexMatch
     * Works</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>AcceptMatch</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartMatchBackfill</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param startMatchBackfillRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the StartMatchBackfill operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.StartMatchBackfill
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/StartMatchBackfill" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartMatchBackfillResponse> startMatchBackfill(StartMatchBackfillRequest startMatchBackfillRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMatchBackfillRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMatchBackfill");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMatchBackfillResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMatchBackfillResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMatchBackfillResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMatchBackfillRequest, StartMatchBackfillResponse>()
                            .withOperationName("StartMatchBackfill")
                            .withMarshaller(new StartMatchBackfillRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startMatchBackfillRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startMatchBackfillRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<StartMatchBackfillResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uses FlexMatch to create a game match for a group of players based on custom matchmaking rules, and starts a new
     * game for the matched players. Each matchmaking request specifies the type of match to build (team configuration,
     * rules for an acceptable match, etc.). The request also specifies the players to find a match for and where to
     * host the new game session for optimal performance. A matchmaking request might start with a single player or a
     * group of players who want to play together. FlexMatch finds additional players as needed to fill the match. Match
     * type, rules, and the queue used to place a new game session are defined in a
     * <code>MatchmakingConfiguration</code>.
     * </p>
     * <p>
     * To start matchmaking, provide a unique ticket ID, specify a matchmaking configuration, and include the players to
     * be matched. You must also include a set of player attributes relevant for the matchmaking configuration. If
     * successful, a matchmaking ticket is returned with status set to <code>QUEUED</code>.
     * </p>
     * <p>
     * Track the status of the ticket to respond as needed and acquire game session connection information for
     * successfully completed matches. Ticket status updates are tracked using event notification through Amazon Simple
     * Notification Service (SNS), which is defined in the matchmaking configuration.
     * </p>
     * <p>
     * <b>Processing a matchmaking request</b> -- FlexMatch handles a matchmaking request as follows:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Your client code submits a <code>StartMatchmaking</code> request for one or more players and tracks the status of
     * the request ticket.
     * </p>
     * </li>
     * <li>
     * <p>
     * FlexMatch uses this ticket and others in process to build an acceptable match. When a potential match is
     * identified, all tickets in the proposed match are advanced to the next status.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the match requires player acceptance (set in the matchmaking configuration), the tickets move into status
     * <code>REQUIRES_ACCEPTANCE</code>. This status triggers your client code to solicit acceptance from all players in
     * every ticket involved in the match, and then call <a>AcceptMatch</a> for each player. If any player rejects or
     * fails to accept the match before a specified timeout, the proposed match is dropped (see <code>AcceptMatch</code>
     * for more details).
     * </p>
     * </li>
     * <li>
     * <p>
     * Once a match is proposed and accepted, the matchmaking tickets move into status <code>PLACING</code>. FlexMatch
     * locates resources for a new game session using the game session queue (set in the matchmaking configuration) and
     * creates the game session based on the match data.
     * </p>
     * </li>
     * <li>
     * <p>
     * When the match is successfully placed, the matchmaking tickets move into <code>COMPLETED</code> status.
     * Connection information (including game session endpoint and player session) is added to the matchmaking tickets.
     * Matched players can use the connection information to join the game.
     * </p>
     * </li>
     * </ol>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-client.html"> Add FlexMatch to a Game
     * Client</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-notification.html"> Set Up FlexMatch
     * Event Notification</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-tasks.html"> FlexMatch Integration
     * Roadmap</a>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-match.html"> How GameLift FlexMatch
     * Works</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>AcceptMatch</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartMatchBackfill</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param startMatchmakingRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the StartMatchmaking operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.StartMatchmaking
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/StartMatchmaking" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartMatchmakingResponse> startMatchmaking(StartMatchmakingRequest startMatchmakingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMatchmakingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMatchmaking");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMatchmakingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMatchmakingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMatchmakingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMatchmakingRequest, StartMatchmakingResponse>()
                            .withOperationName("StartMatchmaking")
                            .withMarshaller(new StartMatchmakingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startMatchmakingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startMatchmakingRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StartMatchmakingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Suspends activity on a fleet. Currently, this operation is used to stop a fleet's auto-scaling activity. It is
     * used to temporarily stop triggering scaling events. The policies can be retained and auto-scaling activity can be
     * restarted using <a>StartFleetActions</a>. You can view a fleet's stopped actions using
     * <a>DescribeFleetAttributes</a>.
     * </p>
     * <p>
     * To stop fleet actions, specify the fleet ID and the type of actions to suspend. When auto-scaling fleet actions
     * are stopped, Amazon GameLift no longer initiates scaling events except in response to manual changes using
     * <a>UpdateFleetCapacity</a>.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param stopFleetActionsRequest
     * @return A Java Future containing the result of the StopFleetActions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.StopFleetActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/StopFleetActions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopFleetActionsResponse> stopFleetActions(StopFleetActionsRequest stopFleetActionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopFleetActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopFleetActions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopFleetActionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopFleetActionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopFleetActionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopFleetActionsRequest, StopFleetActionsResponse>()
                            .withOperationName("StopFleetActions")
                            .withMarshaller(new StopFleetActionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopFleetActionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = stopFleetActionsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StopFleetActionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels a game session placement that is in <code>PENDING</code> status. To stop a placement, provide the
     * placement ID values. If successful, the placement is moved to <code>CANCELLED</code> status.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param stopGameSessionPlacementRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the StopGameSessionPlacement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.StopGameSessionPlacement
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/StopGameSessionPlacement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopGameSessionPlacementResponse> stopGameSessionPlacement(
            StopGameSessionPlacementRequest stopGameSessionPlacementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopGameSessionPlacementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopGameSessionPlacement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopGameSessionPlacementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopGameSessionPlacementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopGameSessionPlacementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopGameSessionPlacementRequest, StopGameSessionPlacementResponse>()
                            .withOperationName("StopGameSessionPlacement")
                            .withMarshaller(new StopGameSessionPlacementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopGameSessionPlacementRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = stopGameSessionPlacementRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<StopGameSessionPlacementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels a matchmaking ticket or match backfill ticket that is currently being processed. To stop the matchmaking
     * operation, specify the ticket ID. If successful, work on the ticket is stopped, and the ticket status is changed
     * to <code>CANCELLED</code>.
     * </p>
     * <p>
     * This call is also used to turn off automatic backfill for an individual game session. This is for game sessions
     * that are created with a matchmaking configuration that has automatic backfill enabled. The ticket ID is included
     * in the <code>MatchmakerData</code> of an updated game session object, which is provided to the game server.
     * </p>
     * <note>
     * <p>
     * If the operation is successful, the service sends back an empty JSON struct with the HTTP 200 response (not an
     * empty HTTP body).
     * </p>
     * </note>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-client.html"> Add FlexMatch to a Game
     * Client</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopMatchmaking</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>AcceptMatch</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StartMatchBackfill</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param stopMatchmakingRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the StopMatchmaking operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.StopMatchmaking
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/StopMatchmaking" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopMatchmakingResponse> stopMatchmaking(StopMatchmakingRequest stopMatchmakingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopMatchmakingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopMatchmaking");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopMatchmakingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopMatchmakingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopMatchmakingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopMatchmakingRequest, StopMatchmakingResponse>()
                            .withOperationName("StopMatchmaking")
                            .withMarshaller(new StopMatchmakingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopMatchmakingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = stopMatchmakingRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StopMatchmakingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Temporarily stops activity on a game server group without terminating instances or the game server group. You can
     * restart activity by calling <a>ResumeGameServerGroup</a>. You can suspend the following activity:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Instance type replacement</b> - This activity evaluates the current game hosting viability of all Spot
     * instance types that are defined for the game server group. It updates the Auto Scaling group to remove nonviable
     * Spot Instance types, which have a higher chance of game server interruptions. It then balances capacity across
     * the remaining viable Spot Instance types. When this activity is suspended, the Auto Scaling group continues with
     * its current balance, regardless of viability. Instance protection, utilization metrics, and capacity scaling
     * activities continue to be active.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To suspend activity, specify a game server group ARN and the type of activity to be suspended. If successful, a
     * <a>GameServerGroup</a> object is returned showing that the activity is listed in <code>SuspendedActions</code>.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServerGroups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResumeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SuspendGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerInstances</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param suspendGameServerGroupRequest
     * @return A Java Future containing the result of the SuspendGameServerGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.SuspendGameServerGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/SuspendGameServerGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SuspendGameServerGroupResponse> suspendGameServerGroup(
            SuspendGameServerGroupRequest suspendGameServerGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, suspendGameServerGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SuspendGameServerGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SuspendGameServerGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SuspendGameServerGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SuspendGameServerGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SuspendGameServerGroupRequest, SuspendGameServerGroupResponse>()
                            .withOperationName("SuspendGameServerGroup")
                            .withMarshaller(new SuspendGameServerGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(suspendGameServerGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = suspendGameServerGroupRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<SuspendGameServerGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns a tag to a GameLift resource. AWS resource tags provide an additional management tool set. You can use
     * tags to organize resources, create IAM permissions policies to manage access to groups of resources, customize
     * AWS cost breakdowns, etc. This operation handles the permissions necessary to manage tags for the following
     * GameLift resource types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Build
     * </p>
     * </li>
     * <li>
     * <p>
     * Script
     * </p>
     * </li>
     * <li>
     * <p>
     * Fleet
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias
     * </p>
     * </li>
     * <li>
     * <p>
     * GameSessionQueue
     * </p>
     * </li>
     * <li>
     * <p>
     * MatchmakingConfiguration
     * </p>
     * </li>
     * <li>
     * <p>
     * MatchmakingRuleSet
     * </p>
     * </li>
     * </ul>
     * <p>
     * To add a tag to a resource, specify the unique ARN value for the resource and provide a tag list containing one
     * or more tags. The operation succeeds even if the list includes tags that are already assigned to the specified
     * resource.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS Resources</a> in the <i>AWS
     * General Reference</i>
     * </p>
     * <p>
     * <a href="http://aws.amazon.com/answers/account-management/aws-tagging-strategies/"> AWS Tagging Strategies</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>TagResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UntagResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListTagsForResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag that is assigned to a GameLift resource. Resource tags are used to organize AWS resources for a
     * range of purposes. This operation handles the permissions necessary to manage tags for the following GameLift
     * resource types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Build
     * </p>
     * </li>
     * <li>
     * <p>
     * Script
     * </p>
     * </li>
     * <li>
     * <p>
     * Fleet
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias
     * </p>
     * </li>
     * <li>
     * <p>
     * GameSessionQueue
     * </p>
     * </li>
     * <li>
     * <p>
     * MatchmakingConfiguration
     * </p>
     * </li>
     * <li>
     * <p>
     * MatchmakingRuleSet
     * </p>
     * </li>
     * </ul>
     * <p>
     * To remove a tag from a resource, specify the unique ARN value for the resource and provide a string list
     * containing one or more tags to be removed. This operation succeeds even if the list includes tags that are not
     * currently assigned to the specified resource.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS Resources</a> in the <i>AWS
     * General Reference</i>
     * </p>
     * <p>
     * <a href="http://aws.amazon.com/answers/account-management/aws-tagging-strategies/"> AWS Tagging Strategies</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>TagResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UntagResource</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListTagsForResource</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>TaggingFailedException The requested tagging operation did not succeed. This may be due to invalid
     *         tag format or the maximum tag limit may have been exceeded. Resolve the issue before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates properties for an alias. To update properties, specify the alias ID to be updated and provide the
     * information to be changed. To reassign an alias to another fleet, provide an updated routing strategy. If
     * successful, the updated alias record is returned.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListAliases</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteAlias</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResolveAlias</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateAliasRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the UpdateAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAliasResponse> updateAlias(UpdateAliasRequest updateAliasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAliasRequest, UpdateAliasResponse>()
                            .withOperationName("UpdateAlias").withMarshaller(new UpdateAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAliasRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateAliasRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates metadata in a build resource, including the build name and version. To update the metadata, specify the
     * build ID to update and provide the new values. If successful, a build object containing the updated metadata is
     * returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-build-intro.html"> Upload a Custom
     * Server Build</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListBuilds</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateBuild</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteBuild</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateBuildRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the UpdateBuild operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateBuild
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateBuild" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBuildResponse> updateBuild(UpdateBuildRequest updateBuildRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBuildRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBuild");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBuildResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateBuildResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBuildResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBuildRequest, UpdateBuildResponse>()
                            .withOperationName("UpdateBuild").withMarshaller(new UpdateBuildRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateBuildRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateBuildRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateBuildResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates fleet properties, including name and description, for a fleet. To update metadata, specify the fleet ID
     * and the property values that you want to change. If successful, the fleet ID for the updated fleet is returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Update fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateRuntimeConfiguration</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateFleetAttributesRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the UpdateFleetAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>InvalidFleetStatusException The requested operation would cause a conflict with the current state of
     *         a resource associated with the request and/or the fleet. Resolve the conflict before retrying.</li>
     *         <li>LimitExceededException The requested operation would cause the resource to exceed the allowed service
     *         limit. Resolve the issue before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateFleetAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateFleetAttributes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFleetAttributesResponse> updateFleetAttributes(
            UpdateFleetAttributesRequest updateFleetAttributesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFleetAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFleetAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFleetAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFleetAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFleetAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFleetAttributesRequest, UpdateFleetAttributesResponse>()
                            .withOperationName("UpdateFleetAttributes")
                            .withMarshaller(new UpdateFleetAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFleetAttributesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateFleetAttributesRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateFleetAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates capacity settings for a fleet. Use this operation to specify the number of EC2 instances (hosts) that you
     * want this fleet to contain. Before calling this operation, you may want to call <a>DescribeEC2InstanceLimits</a>
     * to get the maximum capacity based on the fleet's EC2 instance type.
     * </p>
     * <p>
     * Specify minimum and maximum number of instances. Amazon GameLift will not change fleet capacity to values fall
     * outside of this range. This is particularly important when using auto-scaling (see <a>PutScalingPolicy</a>) to
     * allow capacity to adjust based on player demand while imposing limits on automatic adjustments.
     * </p>
     * <p>
     * To update fleet capacity, specify the fleet ID and the number of instances you want the fleet to host. If
     * successful, Amazon GameLift starts or terminates instances so that the fleet's active instance count matches the
     * desired instance count. You can view a fleet's current capacity information by calling
     * <a>DescribeFleetCapacity</a>. If the desired instance count is higher than the instance type's limit, the
     * "Limit Exceeded" exception occurs.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Update fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateRuntimeConfiguration</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateFleetCapacityRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the UpdateFleetCapacity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>LimitExceededException The requested operation would cause the resource to exceed the allowed service
     *         limit. Resolve the issue before retrying.</li>
     *         <li>InvalidFleetStatusException The requested operation would cause a conflict with the current state of
     *         a resource associated with the request and/or the fleet. Resolve the conflict before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateFleetCapacity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateFleetCapacity" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFleetCapacityResponse> updateFleetCapacity(
            UpdateFleetCapacityRequest updateFleetCapacityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFleetCapacityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFleetCapacity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFleetCapacityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFleetCapacityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFleetCapacityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFleetCapacityRequest, UpdateFleetCapacityResponse>()
                            .withOperationName("UpdateFleetCapacity")
                            .withMarshaller(new UpdateFleetCapacityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFleetCapacityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateFleetCapacityRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateFleetCapacityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates port settings for a fleet. To update settings, specify the fleet ID to be updated and list the
     * permissions you want to update. List the permissions you want to add in
     * <code>InboundPermissionAuthorizations</code>, and permissions you want to remove in
     * <code>InboundPermissionRevocations</code>. Permissions to be removed must match existing fleet permissions. If
     * successful, the fleet ID for the updated fleet is returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Update fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateRuntimeConfiguration</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateFleetPortSettingsRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the UpdateFleetPortSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>InvalidFleetStatusException The requested operation would cause a conflict with the current state of
     *         a resource associated with the request and/or the fleet. Resolve the conflict before retrying.</li>
     *         <li>LimitExceededException The requested operation would cause the resource to exceed the allowed service
     *         limit. Resolve the issue before retrying.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateFleetPortSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateFleetPortSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFleetPortSettingsResponse> updateFleetPortSettings(
            UpdateFleetPortSettingsRequest updateFleetPortSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFleetPortSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFleetPortSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFleetPortSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFleetPortSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFleetPortSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFleetPortSettingsRequest, UpdateFleetPortSettingsResponse>()
                            .withOperationName("UpdateFleetPortSettings")
                            .withMarshaller(new UpdateFleetPortSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFleetPortSettingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateFleetPortSettingsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateFleetPortSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Updates information about a registered game server to help GameLift FleetIQ to track game server availability.
     * This operation is called by a game server process that is running on an instance in a game server group.
     * </p>
     * <p>
     * Use this operation to update the following types of game server information. You can make all three types of
     * updates in the same request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To update the game server's utilization status, identify the game server and game server group and specify the
     * current utilization status. Use this status to identify when game servers are currently hosting games and when
     * they are available to be claimed.
     * </p>
     * </li>
     * <li>
     * <p>
     * To report health status, identify the game server and game server group and set health check to
     * <code>HEALTHY</code>. If a game server does not report health status for a certain length of time, the game
     * server is no longer considered healthy. As a result, it will be eventually deregistered from the game server
     * group to avoid affecting utilization metrics. The best practice is to report health every 60 seconds.
     * </p>
     * </li>
     * <li>
     * <p>
     * To change game server metadata, provide updated game server data.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Once a game server is successfully updated, the relevant statuses and timestamps are updated.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>RegisterGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServers</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ClaimGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServer</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeregisterGameServer</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateGameServerRequest
     * @return A Java Future containing the result of the UpdateGameServer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateGameServer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateGameServer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGameServerResponse> updateGameServer(UpdateGameServerRequest updateGameServerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGameServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGameServer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGameServerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGameServerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGameServerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGameServerRequest, UpdateGameServerResponse>()
                            .withOperationName("UpdateGameServer")
                            .withMarshaller(new UpdateGameServerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateGameServerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateGameServerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateGameServerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <b>This operation is used with the Amazon GameLift FleetIQ solution and game server groups.</b>
     * </p>
     * <p>
     * Updates GameLift FleetIQ-specific properties for a game server group. Many Auto Scaling group properties are
     * updated on the Auto Scaling group directly, including the launch template, Auto Scaling policies, and
     * maximum/minimum/desired instance counts.
     * </p>
     * <p>
     * To update the game server group, specify the game server group ID and provide the updated values. Before applying
     * the updates, the new values are validated to ensure that GameLift FleetIQ can continue to perform instance
     * balancing activity. If successful, a <a>GameServerGroup</a> object is returned.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html">GameLift FleetIQ Guide</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListGameServerGroups</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ResumeGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SuspendGameServerGroup</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameServerInstances</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateGameServerGroupRequest
     * @return A Java Future containing the result of the UpdateGameServerGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateGameServerGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateGameServerGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGameServerGroupResponse> updateGameServerGroup(
            UpdateGameServerGroupRequest updateGameServerGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGameServerGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGameServerGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGameServerGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGameServerGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGameServerGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGameServerGroupRequest, UpdateGameServerGroupResponse>()
                            .withOperationName("UpdateGameServerGroup")
                            .withMarshaller(new UpdateGameServerGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateGameServerGroupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateGameServerGroupRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateGameServerGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates game session properties. This includes the session name, maximum player count, protection policy, which
     * controls whether or not an active game session can be terminated during a scale-down event, and the player
     * session creation policy, which controls whether or not new players can join the session. To update a game
     * session, specify the game session ID and the values you want to change. If successful, an updated
     * <a>GameSession</a> object is returned.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionDetails</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>SearchGameSessions</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>GetGameSessionLogUrl</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Game session placements
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>StartGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionPlacement</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>StopGameSessionPlacement</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     *
     * @param updateGameSessionRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the UpdateGameSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidGameSessionStatusException The requested operation would cause a conflict with the current
     *         state of a resource associated with the request and/or the game instance. Resolve the conflict before
     *         retrying.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateGameSession
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateGameSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGameSessionResponse> updateGameSession(UpdateGameSessionRequest updateGameSessionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGameSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGameSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGameSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGameSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGameSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGameSessionRequest, UpdateGameSessionResponse>()
                            .withOperationName("UpdateGameSession")
                            .withMarshaller(new UpdateGameSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateGameSessionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateGameSessionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateGameSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates settings for a game session queue, which determines how new game session requests in the queue are
     * processed. To update settings, specify the queue name to be updated and provide the new settings. When updating
     * destinations, provide a complete list of destinations.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/queues-intro.html"> Using Multi-Region
     * Queues</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateGameSessionQueue</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeGameSessionQueues</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateGameSessionQueue</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteGameSessionQueue</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateGameSessionQueueRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the UpdateGameSessionQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateGameSessionQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateGameSessionQueue"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGameSessionQueueResponse> updateGameSessionQueue(
            UpdateGameSessionQueueRequest updateGameSessionQueueRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGameSessionQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGameSessionQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGameSessionQueueResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGameSessionQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGameSessionQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGameSessionQueueRequest, UpdateGameSessionQueueResponse>()
                            .withOperationName("UpdateGameSessionQueue")
                            .withMarshaller(new UpdateGameSessionQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateGameSessionQueueRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateGameSessionQueueRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateGameSessionQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates settings for a FlexMatch matchmaking configuration. These changes affect all matches and game sessions
     * that are created after the update. To update settings, specify the configuration name to be updated and provide
     * the new settings.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-configuration.html"> Design a FlexMatch
     * Matchmaker</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingConfigurations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingRuleSets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ValidateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingRuleSet</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateMatchmakingConfigurationRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the UpdateMatchmakingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateMatchmakingConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateMatchmakingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMatchmakingConfigurationResponse> updateMatchmakingConfiguration(
            UpdateMatchmakingConfigurationRequest updateMatchmakingConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateMatchmakingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMatchmakingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMatchmakingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateMatchmakingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMatchmakingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMatchmakingConfigurationRequest, UpdateMatchmakingConfigurationResponse>()
                            .withOperationName("UpdateMatchmakingConfiguration")
                            .withMarshaller(new UpdateMatchmakingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateMatchmakingConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateMatchmakingConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateMatchmakingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the current runtime configuration for the specified fleet, which tells Amazon GameLift how to launch
     * server processes on instances in the fleet. You can update a fleet's runtime configuration at any time after the
     * fleet is created; it does not need to be in an <code>ACTIVE</code> status.
     * </p>
     * <p>
     * To update runtime configuration, specify the fleet ID and provide a <code>RuntimeConfiguration</code> object with
     * an updated set of server process configurations.
     * </p>
     * <p>
     * Each instance in a Amazon GameLift fleet checks regularly for an updated runtime configuration and changes how it
     * launches server processes to comply with the latest version. Existing server processes are not affected by the
     * update; runtime configuration changes are applied gradually as existing processes shut down and new processes are
     * launched during Amazon GameLift's normal process recycling activity.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/fleets-intro.html">Setting up GameLift
     * Fleets</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListFleets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteFleet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Update fleets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>UpdateFleetAttributes</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetCapacity</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateFleetPortSettings</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateRuntimeConfiguration</a>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <a>StartFleetActions</a> or <a>StopFleetActions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateRuntimeConfigurationRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the UpdateRuntimeConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>InvalidFleetStatusException The requested operation would cause a conflict with the current state of
     *         a resource associated with the request and/or the fleet. Resolve the conflict before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateRuntimeConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateRuntimeConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRuntimeConfigurationResponse> updateRuntimeConfiguration(
            UpdateRuntimeConfigurationRequest updateRuntimeConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRuntimeConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRuntimeConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRuntimeConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRuntimeConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRuntimeConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRuntimeConfigurationRequest, UpdateRuntimeConfigurationResponse>()
                            .withOperationName("UpdateRuntimeConfiguration")
                            .withMarshaller(new UpdateRuntimeConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRuntimeConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateRuntimeConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateRuntimeConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates Realtime script metadata and content.
     * </p>
     * <p>
     * To update script metadata, specify the script ID and provide updated name and/or version values.
     * </p>
     * <p>
     * To update script content, provide an updated zip file by pointing to either a local file or an Amazon S3 bucket
     * location. You can use either method regardless of how the original script was uploaded. Use the <i>Version</i>
     * parameter to track updates to the script.
     * </p>
     * <p>
     * If the call is successful, the updated metadata is stored in the script record and a revised script is uploaded
     * to the Amazon GameLift service. Once the script is updated and acquired by a fleet instance, the new version is
     * used for all new game sessions.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/realtime-intro.html">Amazon GameLift Realtime
     * Servers</a>
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ListScripts</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteScript</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateScriptRequest
     * @return A Java Future containing the result of the UpdateScript operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException The client failed authentication. Clients should not retry such requests.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>NotFoundException A service resource associated with the request could not be found. Clients should
     *         not retry such requests.</li>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.UpdateScript
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/UpdateScript" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateScriptResponse> updateScript(UpdateScriptRequest updateScriptRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateScriptRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateScript");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateScriptResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateScriptResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateScriptResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateScriptRequest, UpdateScriptResponse>()
                            .withOperationName("UpdateScript").withMarshaller(new UpdateScriptRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateScriptRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateScriptRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateScriptResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Validates the syntax of a matchmaking rule or rule set. This operation checks that the rule set is using
     * syntactically correct JSON and that it conforms to allowed property expressions. To validate syntax, provide a
     * rule set JSON string.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-rulesets.html">Build a Rule Set</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>CreateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingConfigurations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>UpdateMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>CreateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DescribeMatchmakingRuleSets</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>ValidateMatchmakingRuleSet</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DeleteMatchmakingRuleSet</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param validateMatchmakingRuleSetRequest
     *        Represents the input for a request operation.
     * @return A Java Future containing the result of the ValidateMatchmakingRuleSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException The service encountered an unrecoverable internal failure while processing
     *         the request. Clients can retry such requests immediately or after a waiting period.</li>
     *         <li>UnsupportedRegionException The requested operation is not supported in the Region specified.</li>
     *         <li>InvalidRequestException One or more parameter values in the request are invalid. Correct the invalid
     *         parameter values before retrying.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameLiftAsyncClient.ValidateMatchmakingRuleSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/ValidateMatchmakingRuleSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ValidateMatchmakingRuleSetResponse> validateMatchmakingRuleSet(
            ValidateMatchmakingRuleSetRequest validateMatchmakingRuleSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validateMatchmakingRuleSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidateMatchmakingRuleSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ValidateMatchmakingRuleSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ValidateMatchmakingRuleSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ValidateMatchmakingRuleSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ValidateMatchmakingRuleSetRequest, ValidateMatchmakingRuleSetResponse>()
                            .withOperationName("ValidateMatchmakingRuleSet")
                            .withMarshaller(new ValidateMatchmakingRuleSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(validateMatchmakingRuleSetRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = validateMatchmakingRuleSetRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ValidateMatchmakingRuleSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GameLiftException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FleetCapacityExceededException")
                                .exceptionBuilderSupplier(FleetCapacityExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TerminalRoutingStrategyException")
                                .exceptionBuilderSupplier(TerminalRoutingStrategyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidGameSessionStatusException")
                                .exceptionBuilderSupplier(InvalidGameSessionStatusException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OutOfCapacityException")
                                .exceptionBuilderSupplier(OutOfCapacityException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GameSessionFullException")
                                .exceptionBuilderSupplier(GameSessionFullException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TaggingFailedException")
                                .exceptionBuilderSupplier(TaggingFailedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedRegionException")
                                .exceptionBuilderSupplier(UnsupportedRegionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidFleetStatusException")
                                .exceptionBuilderSupplier(InvalidFleetStatusException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends GameLiftRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
