/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration of a queue that is used to process game session placement requests. The queue configuration identifies
 * several game features:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The destinations where a new game session can potentially be hosted. Amazon GameLift tries these destinations in an
 * order based on either the queue's default order or player latency information, if provided in a placement request.
 * With latency information, Amazon GameLift can place game sessions where the majority of players are reporting the
 * lowest possible latency.
 * </p>
 * </li>
 * <li>
 * <p>
 * The length of time that placement requests can wait in the queue before timing out.
 * </p>
 * </li>
 * <li>
 * <p>
 * A set of optional latency policies that protect individual players from high latencies, preventing game sessions from
 * being placed where any individual player is reporting latency higher than a policy's maximum.
 * </p>
 * </li>
 * </ul>
 * <ul>
 * <li>
 * <p>
 * <a>CreateGameSessionQueue</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DescribeGameSessionQueues</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>UpdateGameSessionQueue</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DeleteGameSessionQueue</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GameSessionQueue implements SdkPojo, Serializable,
        ToCopyableBuilder<GameSessionQueue.Builder, GameSessionQueue> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(GameSessionQueue::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> GAME_SESSION_QUEUE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameSessionQueueArn").getter(getter(GameSessionQueue::gameSessionQueueArn))
            .setter(setter(Builder::gameSessionQueueArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionQueueArn").build())
            .build();

    private static final SdkField<Integer> TIMEOUT_IN_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TimeoutInSeconds").getter(getter(GameSessionQueue::timeoutInSeconds))
            .setter(setter(Builder::timeoutInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeoutInSeconds").build()).build();

    private static final SdkField<List<PlayerLatencyPolicy>> PLAYER_LATENCY_POLICIES_FIELD = SdkField
            .<List<PlayerLatencyPolicy>> builder(MarshallingType.LIST)
            .memberName("PlayerLatencyPolicies")
            .getter(getter(GameSessionQueue::playerLatencyPolicies))
            .setter(setter(Builder::playerLatencyPolicies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlayerLatencyPolicies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PlayerLatencyPolicy> builder(MarshallingType.SDK_POJO)
                                            .constructor(PlayerLatencyPolicy::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<GameSessionQueueDestination>> DESTINATIONS_FIELD = SdkField
            .<List<GameSessionQueueDestination>> builder(MarshallingType.LIST)
            .memberName("Destinations")
            .getter(getter(GameSessionQueue::destinations))
            .setter(setter(Builder::destinations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GameSessionQueueDestination> builder(MarshallingType.SDK_POJO)
                                            .constructor(GameSessionQueueDestination::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            GAME_SESSION_QUEUE_ARN_FIELD, TIMEOUT_IN_SECONDS_FIELD, PLAYER_LATENCY_POLICIES_FIELD, DESTINATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String gameSessionQueueArn;

    private final Integer timeoutInSeconds;

    private final List<PlayerLatencyPolicy> playerLatencyPolicies;

    private final List<GameSessionQueueDestination> destinations;

    private GameSessionQueue(BuilderImpl builder) {
        this.name = builder.name;
        this.gameSessionQueueArn = builder.gameSessionQueueArn;
        this.timeoutInSeconds = builder.timeoutInSeconds;
        this.playerLatencyPolicies = builder.playerLatencyPolicies;
        this.destinations = builder.destinations;
    }

    /**
     * <p>
     * A descriptive label that is associated with game session queue. Queue names must be unique within each Region.
     * </p>
     * 
     * @return A descriptive label that is associated with game session queue. Queue names must be unique within each
     *         Region.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Amazon Resource Name (<a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is assigned to a
     * GameLift game session queue resource and uniquely identifies it. ARNs are unique across all Regions. In a
     * GameLift game session queue ARN, the resource ID matches the <i>Name</i> value.
     * </p>
     * 
     * @return Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is
     *         assigned to a GameLift game session queue resource and uniquely identifies it. ARNs are unique across all
     *         Regions. In a GameLift game session queue ARN, the resource ID matches the <i>Name</i> value.
     */
    public String gameSessionQueueArn() {
        return gameSessionQueueArn;
    }

    /**
     * <p>
     * The maximum time, in seconds, that a new game session placement request remains in the queue. When a request
     * exceeds this time, the game session placement changes to a <code>TIMED_OUT</code> status.
     * </p>
     * 
     * @return The maximum time, in seconds, that a new game session placement request remains in the queue. When a
     *         request exceeds this time, the game session placement changes to a <code>TIMED_OUT</code> status.
     */
    public Integer timeoutInSeconds() {
        return timeoutInSeconds;
    }

    /**
     * Returns true if the PlayerLatencyPolicies property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasPlayerLatencyPolicies() {
        return playerLatencyPolicies != null && !(playerLatencyPolicies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A collection of latency policies to apply when processing game sessions placement requests with player latency
     * information. Multiple policies are evaluated in order of the maximum latency value, starting with the lowest
     * latency values. With just one policy, the policy is enforced at the start of the game session placement for the
     * duration period. With multiple policies, each policy is enforced consecutively for its duration period. For
     * example, a queue might enforce a 60-second policy followed by a 120-second policy, and then no policy for the
     * remainder of the placement.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPlayerLatencyPolicies()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A collection of latency policies to apply when processing game sessions placement requests with player
     *         latency information. Multiple policies are evaluated in order of the maximum latency value, starting with
     *         the lowest latency values. With just one policy, the policy is enforced at the start of the game session
     *         placement for the duration period. With multiple policies, each policy is enforced consecutively for its
     *         duration period. For example, a queue might enforce a 60-second policy followed by a 120-second policy,
     *         and then no policy for the remainder of the placement.
     */
    public List<PlayerLatencyPolicy> playerLatencyPolicies() {
        return playerLatencyPolicies;
    }

    /**
     * Returns true if the Destinations property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDestinations() {
        return destinations != null && !(destinations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of fleets that can be used to fulfill game session placement requests in the queue. Fleets are identified
     * by either a fleet ARN or a fleet alias ARN. Destinations are listed in default preference order.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDestinations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of fleets that can be used to fulfill game session placement requests in the queue. Fleets are
     *         identified by either a fleet ARN or a fleet alias ARN. Destinations are listed in default preference
     *         order.
     */
    public List<GameSessionQueueDestination> destinations() {
        return destinations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionQueueArn());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(playerLatencyPolicies());
        hashCode = 31 * hashCode + Objects.hashCode(destinations());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GameSessionQueue)) {
            return false;
        }
        GameSessionQueue other = (GameSessionQueue) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(gameSessionQueueArn(), other.gameSessionQueueArn())
                && Objects.equals(timeoutInSeconds(), other.timeoutInSeconds())
                && Objects.equals(playerLatencyPolicies(), other.playerLatencyPolicies())
                && Objects.equals(destinations(), other.destinations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GameSessionQueue").add("Name", name()).add("GameSessionQueueArn", gameSessionQueueArn())
                .add("TimeoutInSeconds", timeoutInSeconds()).add("PlayerLatencyPolicies", playerLatencyPolicies())
                .add("Destinations", destinations()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "GameSessionQueueArn":
            return Optional.ofNullable(clazz.cast(gameSessionQueueArn()));
        case "TimeoutInSeconds":
            return Optional.ofNullable(clazz.cast(timeoutInSeconds()));
        case "PlayerLatencyPolicies":
            return Optional.ofNullable(clazz.cast(playerLatencyPolicies()));
        case "Destinations":
            return Optional.ofNullable(clazz.cast(destinations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GameSessionQueue, T> g) {
        return obj -> g.apply((GameSessionQueue) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GameSessionQueue> {
        /**
         * <p>
         * A descriptive label that is associated with game session queue. Queue names must be unique within each
         * Region.
         * </p>
         * 
         * @param name
         *        A descriptive label that is associated with game session queue. Queue names must be unique within each
         *        Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is assigned
         * to a GameLift game session queue resource and uniquely identifies it. ARNs are unique across all Regions. In
         * a GameLift game session queue ARN, the resource ID matches the <i>Name</i> value.
         * </p>
         * 
         * @param gameSessionQueueArn
         *        Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is
         *        assigned to a GameLift game session queue resource and uniquely identifies it. ARNs are unique across
         *        all Regions. In a GameLift game session queue ARN, the resource ID matches the <i>Name</i> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionQueueArn(String gameSessionQueueArn);

        /**
         * <p>
         * The maximum time, in seconds, that a new game session placement request remains in the queue. When a request
         * exceeds this time, the game session placement changes to a <code>TIMED_OUT</code> status.
         * </p>
         * 
         * @param timeoutInSeconds
         *        The maximum time, in seconds, that a new game session placement request remains in the queue. When a
         *        request exceeds this time, the game session placement changes to a <code>TIMED_OUT</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutInSeconds(Integer timeoutInSeconds);

        /**
         * <p>
         * A collection of latency policies to apply when processing game sessions placement requests with player
         * latency information. Multiple policies are evaluated in order of the maximum latency value, starting with the
         * lowest latency values. With just one policy, the policy is enforced at the start of the game session
         * placement for the duration period. With multiple policies, each policy is enforced consecutively for its
         * duration period. For example, a queue might enforce a 60-second policy followed by a 120-second policy, and
         * then no policy for the remainder of the placement.
         * </p>
         * 
         * @param playerLatencyPolicies
         *        A collection of latency policies to apply when processing game sessions placement requests with player
         *        latency information. Multiple policies are evaluated in order of the maximum latency value, starting
         *        with the lowest latency values. With just one policy, the policy is enforced at the start of the game
         *        session placement for the duration period. With multiple policies, each policy is enforced
         *        consecutively for its duration period. For example, a queue might enforce a 60-second policy followed
         *        by a 120-second policy, and then no policy for the remainder of the placement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerLatencyPolicies(Collection<PlayerLatencyPolicy> playerLatencyPolicies);

        /**
         * <p>
         * A collection of latency policies to apply when processing game sessions placement requests with player
         * latency information. Multiple policies are evaluated in order of the maximum latency value, starting with the
         * lowest latency values. With just one policy, the policy is enforced at the start of the game session
         * placement for the duration period. With multiple policies, each policy is enforced consecutively for its
         * duration period. For example, a queue might enforce a 60-second policy followed by a 120-second policy, and
         * then no policy for the remainder of the placement.
         * </p>
         * 
         * @param playerLatencyPolicies
         *        A collection of latency policies to apply when processing game sessions placement requests with player
         *        latency information. Multiple policies are evaluated in order of the maximum latency value, starting
         *        with the lowest latency values. With just one policy, the policy is enforced at the start of the game
         *        session placement for the duration period. With multiple policies, each policy is enforced
         *        consecutively for its duration period. For example, a queue might enforce a 60-second policy followed
         *        by a 120-second policy, and then no policy for the remainder of the placement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerLatencyPolicies(PlayerLatencyPolicy... playerLatencyPolicies);

        /**
         * <p>
         * A collection of latency policies to apply when processing game sessions placement requests with player
         * latency information. Multiple policies are evaluated in order of the maximum latency value, starting with the
         * lowest latency values. With just one policy, the policy is enforced at the start of the game session
         * placement for the duration period. With multiple policies, each policy is enforced consecutively for its
         * duration period. For example, a queue might enforce a 60-second policy followed by a 120-second policy, and
         * then no policy for the remainder of the placement.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PlayerLatencyPolicy>.Builder} avoiding the
         * need to create one manually via {@link List<PlayerLatencyPolicy>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PlayerLatencyPolicy>.Builder#build()} is called immediately
         * and its result is passed to {@link #playerLatencyPolicies(List<PlayerLatencyPolicy>)}.
         * 
         * @param playerLatencyPolicies
         *        a consumer that will call methods on {@link List<PlayerLatencyPolicy>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #playerLatencyPolicies(List<PlayerLatencyPolicy>)
         */
        Builder playerLatencyPolicies(Consumer<PlayerLatencyPolicy.Builder>... playerLatencyPolicies);

        /**
         * <p>
         * A list of fleets that can be used to fulfill game session placement requests in the queue. Fleets are
         * identified by either a fleet ARN or a fleet alias ARN. Destinations are listed in default preference order.
         * </p>
         * 
         * @param destinations
         *        A list of fleets that can be used to fulfill game session placement requests in the queue. Fleets are
         *        identified by either a fleet ARN or a fleet alias ARN. Destinations are listed in default preference
         *        order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<GameSessionQueueDestination> destinations);

        /**
         * <p>
         * A list of fleets that can be used to fulfill game session placement requests in the queue. Fleets are
         * identified by either a fleet ARN or a fleet alias ARN. Destinations are listed in default preference order.
         * </p>
         * 
         * @param destinations
         *        A list of fleets that can be used to fulfill game session placement requests in the queue. Fleets are
         *        identified by either a fleet ARN or a fleet alias ARN. Destinations are listed in default preference
         *        order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(GameSessionQueueDestination... destinations);

        /**
         * <p>
         * A list of fleets that can be used to fulfill game session placement requests in the queue. Fleets are
         * identified by either a fleet ARN or a fleet alias ARN. Destinations are listed in default preference order.
         * </p>
         * This is a convenience that creates an instance of the {@link List<GameSessionQueueDestination>.Builder}
         * avoiding the need to create one manually via {@link List<GameSessionQueueDestination>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<GameSessionQueueDestination>.Builder#build()} is called
         * immediately and its result is passed to {@link #destinations(List<GameSessionQueueDestination>)}.
         * 
         * @param destinations
         *        a consumer that will call methods on {@link List<GameSessionQueueDestination>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinations(List<GameSessionQueueDestination>)
         */
        Builder destinations(Consumer<GameSessionQueueDestination.Builder>... destinations);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String gameSessionQueueArn;

        private Integer timeoutInSeconds;

        private List<PlayerLatencyPolicy> playerLatencyPolicies = DefaultSdkAutoConstructList.getInstance();

        private List<GameSessionQueueDestination> destinations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GameSessionQueue model) {
            name(model.name);
            gameSessionQueueArn(model.gameSessionQueueArn);
            timeoutInSeconds(model.timeoutInSeconds);
            playerLatencyPolicies(model.playerLatencyPolicies);
            destinations(model.destinations);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getGameSessionQueueArn() {
            return gameSessionQueueArn;
        }

        @Override
        public final Builder gameSessionQueueArn(String gameSessionQueueArn) {
            this.gameSessionQueueArn = gameSessionQueueArn;
            return this;
        }

        public final void setGameSessionQueueArn(String gameSessionQueueArn) {
            this.gameSessionQueueArn = gameSessionQueueArn;
        }

        public final Integer getTimeoutInSeconds() {
            return timeoutInSeconds;
        }

        @Override
        public final Builder timeoutInSeconds(Integer timeoutInSeconds) {
            this.timeoutInSeconds = timeoutInSeconds;
            return this;
        }

        public final void setTimeoutInSeconds(Integer timeoutInSeconds) {
            this.timeoutInSeconds = timeoutInSeconds;
        }

        public final Collection<PlayerLatencyPolicy.Builder> getPlayerLatencyPolicies() {
            return playerLatencyPolicies != null ? playerLatencyPolicies.stream().map(PlayerLatencyPolicy::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder playerLatencyPolicies(Collection<PlayerLatencyPolicy> playerLatencyPolicies) {
            this.playerLatencyPolicies = PlayerLatencyPolicyListCopier.copy(playerLatencyPolicies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder playerLatencyPolicies(PlayerLatencyPolicy... playerLatencyPolicies) {
            playerLatencyPolicies(Arrays.asList(playerLatencyPolicies));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder playerLatencyPolicies(Consumer<PlayerLatencyPolicy.Builder>... playerLatencyPolicies) {
            playerLatencyPolicies(Stream.of(playerLatencyPolicies)
                    .map(c -> PlayerLatencyPolicy.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setPlayerLatencyPolicies(Collection<PlayerLatencyPolicy.BuilderImpl> playerLatencyPolicies) {
            this.playerLatencyPolicies = PlayerLatencyPolicyListCopier.copyFromBuilder(playerLatencyPolicies);
        }

        public final Collection<GameSessionQueueDestination.Builder> getDestinations() {
            return destinations != null ? destinations.stream().map(GameSessionQueueDestination::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder destinations(Collection<GameSessionQueueDestination> destinations) {
            this.destinations = GameSessionQueueDestinationListCopier.copy(destinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(GameSessionQueueDestination... destinations) {
            destinations(Arrays.asList(destinations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(Consumer<GameSessionQueueDestination.Builder>... destinations) {
            destinations(Stream.of(destinations).map(c -> GameSessionQueueDestination.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDestinations(Collection<GameSessionQueueDestination.BuilderImpl> destinations) {
            this.destinations = GameSessionQueueDestinationListCopier.copyFromBuilder(destinations);
        }

        @Override
        public GameSessionQueue build() {
            return new GameSessionQueue(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
