/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Set of rule statements, used with FlexMatch, that determine how to build your player matches. Each rule set describes
 * a type of group to be created and defines the parameters for acceptable player matches. Rule sets are used in
 * <a>MatchmakingConfiguration</a> objects.
 * </p>
 * <p>
 * A rule set may define the following elements for a match. For detailed information and examples showing how to
 * construct a rule set, see <a
 * href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-rulesets.html">Build a FlexMatch Rule Set</a>.
 * </p>
 * <ul>
 * <li>
 * <p>
 * Teams -- Required. A rule set must define one or multiple teams for the match and set minimum and maximum team sizes.
 * For example, a rule set might describe a 4x4 match that requires all eight slots to be filled.
 * </p>
 * </li>
 * <li>
 * <p>
 * Player attributes -- Optional. These attributes specify a set of player characteristics to evaluate when looking for
 * a match. Matchmaking requests that use a rule set with player attributes must provide the corresponding attribute
 * values. For example, an attribute might specify a player's skill or level.
 * </p>
 * </li>
 * <li>
 * <p>
 * Rules -- Optional. Rules define how to evaluate potential players for a match based on player attributes. A rule
 * might specify minimum requirements for individual players, teams, or entire matches. For example, a rule might
 * require each player to meet a certain skill level, each team to have at least one player in a certain role, or the
 * match to have a minimum average skill level. or may describe an entire group--such as all teams must be evenly
 * matched or have at least one player in a certain role.
 * </p>
 * </li>
 * <li>
 * <p>
 * Expansions -- Optional. Expansions allow you to relax the rules after a period of time when no acceptable matches are
 * found. This feature lets you balance getting players into games in a reasonable amount of time instead of making them
 * wait indefinitely for the best possible match. For example, you might use an expansion to increase the maximum skill
 * variance between players after 30 seconds.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MatchmakingRuleSet implements SdkPojo, Serializable,
        ToCopyableBuilder<MatchmakingRuleSet.Builder, MatchmakingRuleSet> {
    private static final SdkField<String> RULE_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleSetName").getter(getter(MatchmakingRuleSet::ruleSetName)).setter(setter(Builder::ruleSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleSetName").build()).build();

    private static final SdkField<String> RULE_SET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleSetArn").getter(getter(MatchmakingRuleSet::ruleSetArn)).setter(setter(Builder::ruleSetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleSetArn").build()).build();

    private static final SdkField<String> RULE_SET_BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleSetBody").getter(getter(MatchmakingRuleSet::ruleSetBody)).setter(setter(Builder::ruleSetBody))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleSetBody").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(MatchmakingRuleSet::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_SET_NAME_FIELD,
            RULE_SET_ARN_FIELD, RULE_SET_BODY_FIELD, CREATION_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ruleSetName;

    private final String ruleSetArn;

    private final String ruleSetBody;

    private final Instant creationTime;

    private MatchmakingRuleSet(BuilderImpl builder) {
        this.ruleSetName = builder.ruleSetName;
        this.ruleSetArn = builder.ruleSetArn;
        this.ruleSetBody = builder.ruleSetBody;
        this.creationTime = builder.creationTime;
    }

    /**
     * <p>
     * A unique identifier for a matchmaking rule set
     * </p>
     * 
     * @return A unique identifier for a matchmaking rule set
     */
    public String ruleSetName() {
        return ruleSetName;
    }

    /**
     * <p>
     * Amazon Resource Name (<a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is assigned to a
     * GameLift matchmaking rule set resource and uniquely identifies it. ARNs are unique across all Regions. In a
     * GameLift rule set ARN, the resource ID matches the <i>RuleSetName</i> value.
     * </p>
     * 
     * @return Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is
     *         assigned to a GameLift matchmaking rule set resource and uniquely identifies it. ARNs are unique across
     *         all Regions. In a GameLift rule set ARN, the resource ID matches the <i>RuleSetName</i> value.
     */
    public String ruleSetArn() {
        return ruleSetArn;
    }

    /**
     * <p>
     * A collection of matchmaking rules, formatted as a JSON string. Comments are not allowed in JSON, but most
     * elements support a description field.
     * </p>
     * 
     * @return A collection of matchmaking rules, formatted as a JSON string. Comments are not allowed in JSON, but most
     *         elements support a description field.
     */
    public String ruleSetBody() {
        return ruleSetBody;
    }

    /**
     * <p>
     * The time stamp indicating when this data object was created. The format is a number expressed in Unix time as
     * milliseconds (for example "1469498468.057").
     * </p>
     * 
     * @return The time stamp indicating when this data object was created. The format is a number expressed in Unix
     *         time as milliseconds (for example "1469498468.057").
     */
    public Instant creationTime() {
        return creationTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ruleSetName());
        hashCode = 31 * hashCode + Objects.hashCode(ruleSetArn());
        hashCode = 31 * hashCode + Objects.hashCode(ruleSetBody());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MatchmakingRuleSet)) {
            return false;
        }
        MatchmakingRuleSet other = (MatchmakingRuleSet) obj;
        return Objects.equals(ruleSetName(), other.ruleSetName()) && Objects.equals(ruleSetArn(), other.ruleSetArn())
                && Objects.equals(ruleSetBody(), other.ruleSetBody()) && Objects.equals(creationTime(), other.creationTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("MatchmakingRuleSet").add("RuleSetName", ruleSetName()).add("RuleSetArn", ruleSetArn())
                .add("RuleSetBody", ruleSetBody()).add("CreationTime", creationTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleSetName":
            return Optional.ofNullable(clazz.cast(ruleSetName()));
        case "RuleSetArn":
            return Optional.ofNullable(clazz.cast(ruleSetArn()));
        case "RuleSetBody":
            return Optional.ofNullable(clazz.cast(ruleSetBody()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MatchmakingRuleSet, T> g) {
        return obj -> g.apply((MatchmakingRuleSet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MatchmakingRuleSet> {
        /**
         * <p>
         * A unique identifier for a matchmaking rule set
         * </p>
         * 
         * @param ruleSetName
         *        A unique identifier for a matchmaking rule set
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleSetName(String ruleSetName);

        /**
         * <p>
         * Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is assigned
         * to a GameLift matchmaking rule set resource and uniquely identifies it. ARNs are unique across all Regions.
         * In a GameLift rule set ARN, the resource ID matches the <i>RuleSetName</i> value.
         * </p>
         * 
         * @param ruleSetArn
         *        Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) that is
         *        assigned to a GameLift matchmaking rule set resource and uniquely identifies it. ARNs are unique
         *        across all Regions. In a GameLift rule set ARN, the resource ID matches the <i>RuleSetName</i> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleSetArn(String ruleSetArn);

        /**
         * <p>
         * A collection of matchmaking rules, formatted as a JSON string. Comments are not allowed in JSON, but most
         * elements support a description field.
         * </p>
         * 
         * @param ruleSetBody
         *        A collection of matchmaking rules, formatted as a JSON string. Comments are not allowed in JSON, but
         *        most elements support a description field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleSetBody(String ruleSetBody);

        /**
         * <p>
         * The time stamp indicating when this data object was created. The format is a number expressed in Unix time as
         * milliseconds (for example "1469498468.057").
         * </p>
         * 
         * @param creationTime
         *        The time stamp indicating when this data object was created. The format is a number expressed in Unix
         *        time as milliseconds (for example "1469498468.057").
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);
    }

    static final class BuilderImpl implements Builder {
        private String ruleSetName;

        private String ruleSetArn;

        private String ruleSetBody;

        private Instant creationTime;

        private BuilderImpl() {
        }

        private BuilderImpl(MatchmakingRuleSet model) {
            ruleSetName(model.ruleSetName);
            ruleSetArn(model.ruleSetArn);
            ruleSetBody(model.ruleSetBody);
            creationTime(model.creationTime);
        }

        public final String getRuleSetName() {
            return ruleSetName;
        }

        @Override
        public final Builder ruleSetName(String ruleSetName) {
            this.ruleSetName = ruleSetName;
            return this;
        }

        public final void setRuleSetName(String ruleSetName) {
            this.ruleSetName = ruleSetName;
        }

        public final String getRuleSetArn() {
            return ruleSetArn;
        }

        @Override
        public final Builder ruleSetArn(String ruleSetArn) {
            this.ruleSetArn = ruleSetArn;
            return this;
        }

        public final void setRuleSetArn(String ruleSetArn) {
            this.ruleSetArn = ruleSetArn;
        }

        public final String getRuleSetBody() {
            return ruleSetBody;
        }

        @Override
        public final Builder ruleSetBody(String ruleSetBody) {
            this.ruleSetBody = ruleSetBody;
            return this;
        }

        public final void setRuleSetBody(String ruleSetBody) {
            this.ruleSetBody = ruleSetBody;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public MatchmakingRuleSet build() {
            return new MatchmakingRuleSet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
