/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Custom prioritization settings for use by a game session queue when placing new game sessions with available game
 * servers. When defined, this configuration replaces the default FleetIQ prioritization process, which is as follows:
 * </p>
 * <ul>
 * <li>
 * <p>
 * If player latency data is included in a game session request, destinations and locations are prioritized first based
 * on lowest average latency (1), then on lowest hosting cost (2), then on destination list order (3), and finally on
 * location (alphabetical) (4). This approach ensures that the queue's top priority is to place game sessions where
 * average player latency is lowest, and--if latency is the same--where the hosting cost is less, etc.
 * </p>
 * </li>
 * <li>
 * <p>
 * If player latency data is not included, destinations and locations are prioritized first on destination list order
 * (1), and then on location (alphabetical) (2). This approach ensures that the queue's top priority is to place game
 * sessions on the first destination fleet listed. If that fleet has multiple locations, the game session is placed on
 * the first location (when listed alphabetically).
 * </p>
 * </li>
 * </ul>
 * <p>
 * Changing the priority order will affect how game sessions are placed.
 * </p>
 * <p>
 * Priority configurations are part of a <a>GameSessionQueue</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PriorityConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<PriorityConfiguration.Builder, PriorityConfiguration> {
    private static final SdkField<List<String>> PRIORITY_ORDER_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PriorityOrder")
            .getter(getter(PriorityConfiguration::priorityOrderAsStrings))
            .setter(setter(Builder::priorityOrderWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PriorityOrder").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> LOCATION_ORDER_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LocationOrder")
            .getter(getter(PriorityConfiguration::locationOrder))
            .setter(setter(Builder::locationOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocationOrder").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PRIORITY_ORDER_FIELD,
            LOCATION_ORDER_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> priorityOrder;

    private final List<String> locationOrder;

    private PriorityConfiguration(BuilderImpl builder) {
        this.priorityOrder = builder.priorityOrder;
        this.locationOrder = builder.locationOrder;
    }

    /**
     * <p>
     * The recommended sequence to use when prioritizing where to place new game sessions. Each type can only be listed
     * once.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in each game
     * session request) is lowest.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is evaluated
     * based on the location, instance type, and fleet type (Spot or On-Demand) for each destination in the queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the queue
     * configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
     * <code>LocationOrder</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPriorityOrder()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The recommended sequence to use when prioritizing where to place new game sessions. Each type can only be
     *         listed once.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in each
     *         game session request) is lowest.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is
     *         evaluated based on the location, instance type, and fleet type (Spot or On-Demand) for each destination
     *         in the queue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the
     *         queue configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
     *         <code>LocationOrder</code>.
     *         </p>
     *         </li>
     */
    public final List<PriorityType> priorityOrder() {
        return PriorityTypeListCopier.copyStringToEnum(priorityOrder);
    }

    /**
     * Returns true if the PriorityOrder property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasPriorityOrder() {
        return priorityOrder != null && !(priorityOrder instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The recommended sequence to use when prioritizing where to place new game sessions. Each type can only be listed
     * once.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in each game
     * session request) is lowest.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is evaluated
     * based on the location, instance type, and fleet type (Spot or On-Demand) for each destination in the queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the queue
     * configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
     * <code>LocationOrder</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPriorityOrder()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The recommended sequence to use when prioritizing where to place new game sessions. Each type can only be
     *         listed once.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in each
     *         game session request) is lowest.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is
     *         evaluated based on the location, instance type, and fleet type (Spot or On-Demand) for each destination
     *         in the queue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the
     *         queue configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
     *         <code>LocationOrder</code>.
     *         </p>
     *         </li>
     */
    public final List<String> priorityOrderAsStrings() {
        return priorityOrder;
    }

    /**
     * Returns true if the LocationOrder property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasLocationOrder() {
        return locationOrder != null && !(locationOrder instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The prioritization order to use for fleet locations, when the <code>PriorityOrder</code> property includes
     * <code>LOCATION</code>. Locations are identified by AWS Region codes such as <code>us-west-2</code>. Each location
     * can only be listed once.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLocationOrder()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The prioritization order to use for fleet locations, when the <code>PriorityOrder</code> property
     *         includes <code>LOCATION</code>. Locations are identified by AWS Region codes such as
     *         <code>us-west-2</code>. Each location can only be listed once.
     */
    public final List<String> locationOrder() {
        return locationOrder;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasPriorityOrder() ? priorityOrderAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasLocationOrder() ? locationOrder() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PriorityConfiguration)) {
            return false;
        }
        PriorityConfiguration other = (PriorityConfiguration) obj;
        return hasPriorityOrder() == other.hasPriorityOrder()
                && Objects.equals(priorityOrderAsStrings(), other.priorityOrderAsStrings())
                && hasLocationOrder() == other.hasLocationOrder() && Objects.equals(locationOrder(), other.locationOrder());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PriorityConfiguration")
                .add("PriorityOrder", hasPriorityOrder() ? priorityOrderAsStrings() : null)
                .add("LocationOrder", hasLocationOrder() ? locationOrder() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PriorityOrder":
            return Optional.ofNullable(clazz.cast(priorityOrderAsStrings()));
        case "LocationOrder":
            return Optional.ofNullable(clazz.cast(locationOrder()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PriorityConfiguration, T> g) {
        return obj -> g.apply((PriorityConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PriorityConfiguration> {
        /**
         * <p>
         * The recommended sequence to use when prioritizing where to place new game sessions. Each type can only be
         * listed once.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in each game
         * session request) is lowest.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is
         * evaluated based on the location, instance type, and fleet type (Spot or On-Demand) for each destination in
         * the queue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the queue
         * configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
         * <code>LocationOrder</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param priorityOrder
         *        The recommended sequence to use when prioritizing where to place new game sessions. Each type can only
         *        be listed once.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in
         *        each game session request) is lowest.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is
         *        evaluated based on the location, instance type, and fleet type (Spot or On-Demand) for each
         *        destination in the queue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the
         *        queue configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
         *        <code>LocationOrder</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priorityOrderWithStrings(Collection<String> priorityOrder);

        /**
         * <p>
         * The recommended sequence to use when prioritizing where to place new game sessions. Each type can only be
         * listed once.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in each game
         * session request) is lowest.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is
         * evaluated based on the location, instance type, and fleet type (Spot or On-Demand) for each destination in
         * the queue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the queue
         * configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
         * <code>LocationOrder</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param priorityOrder
         *        The recommended sequence to use when prioritizing where to place new game sessions. Each type can only
         *        be listed once.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in
         *        each game session request) is lowest.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is
         *        evaluated based on the location, instance type, and fleet type (Spot or On-Demand) for each
         *        destination in the queue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the
         *        queue configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
         *        <code>LocationOrder</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priorityOrderWithStrings(String... priorityOrder);

        /**
         * <p>
         * The recommended sequence to use when prioritizing where to place new game sessions. Each type can only be
         * listed once.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in each game
         * session request) is lowest.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is
         * evaluated based on the location, instance type, and fleet type (Spot or On-Demand) for each destination in
         * the queue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the queue
         * configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
         * <code>LocationOrder</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param priorityOrder
         *        The recommended sequence to use when prioritizing where to place new game sessions. Each type can only
         *        be listed once.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in
         *        each game session request) is lowest.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is
         *        evaluated based on the location, instance type, and fleet type (Spot or On-Demand) for each
         *        destination in the queue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the
         *        queue configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
         *        <code>LocationOrder</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priorityOrder(Collection<PriorityType> priorityOrder);

        /**
         * <p>
         * The recommended sequence to use when prioritizing where to place new game sessions. Each type can only be
         * listed once.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in each game
         * session request) is lowest.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is
         * evaluated based on the location, instance type, and fleet type (Spot or On-Demand) for each destination in
         * the queue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the queue
         * configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
         * <code>LocationOrder</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param priorityOrder
         *        The recommended sequence to use when prioritizing where to place new game sessions. Each type can only
         *        be listed once.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>LATENCY</code> -- FleetIQ prioritizes locations where the average player latency (provided in
         *        each game session request) is lowest.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COST</code> -- FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is
         *        evaluated based on the location, instance type, and fleet type (Spot or On-Demand) for each
         *        destination in the queue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DESTINATION</code> -- FleetIQ prioritizes based on the order that destinations are listed in the
         *        queue configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LOCATION</code> -- FleetIQ prioritizes based on the provided order of locations, as defined in
         *        <code>LocationOrder</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priorityOrder(PriorityType... priorityOrder);

        /**
         * <p>
         * The prioritization order to use for fleet locations, when the <code>PriorityOrder</code> property includes
         * <code>LOCATION</code>. Locations are identified by AWS Region codes such as <code>us-west-2</code>. Each
         * location can only be listed once.
         * </p>
         * 
         * @param locationOrder
         *        The prioritization order to use for fleet locations, when the <code>PriorityOrder</code> property
         *        includes <code>LOCATION</code>. Locations are identified by AWS Region codes such as
         *        <code>us-west-2</code>. Each location can only be listed once.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locationOrder(Collection<String> locationOrder);

        /**
         * <p>
         * The prioritization order to use for fleet locations, when the <code>PriorityOrder</code> property includes
         * <code>LOCATION</code>. Locations are identified by AWS Region codes such as <code>us-west-2</code>. Each
         * location can only be listed once.
         * </p>
         * 
         * @param locationOrder
         *        The prioritization order to use for fleet locations, when the <code>PriorityOrder</code> property
         *        includes <code>LOCATION</code>. Locations are identified by AWS Region codes such as
         *        <code>us-west-2</code>. Each location can only be listed once.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locationOrder(String... locationOrder);
    }

    static final class BuilderImpl implements Builder {
        private List<String> priorityOrder = DefaultSdkAutoConstructList.getInstance();

        private List<String> locationOrder = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PriorityConfiguration model) {
            priorityOrderWithStrings(model.priorityOrder);
            locationOrder(model.locationOrder);
        }

        public final Collection<String> getPriorityOrder() {
            if (priorityOrder instanceof SdkAutoConstructList) {
                return null;
            }
            return priorityOrder;
        }

        @Override
        public final Builder priorityOrderWithStrings(Collection<String> priorityOrder) {
            this.priorityOrder = PriorityTypeListCopier.copy(priorityOrder);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder priorityOrderWithStrings(String... priorityOrder) {
            priorityOrderWithStrings(Arrays.asList(priorityOrder));
            return this;
        }

        @Override
        public final Builder priorityOrder(Collection<PriorityType> priorityOrder) {
            this.priorityOrder = PriorityTypeListCopier.copyEnumToString(priorityOrder);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder priorityOrder(PriorityType... priorityOrder) {
            priorityOrder(Arrays.asList(priorityOrder));
            return this;
        }

        public final void setPriorityOrder(Collection<String> priorityOrder) {
            this.priorityOrder = PriorityTypeListCopier.copy(priorityOrder);
        }

        public final Collection<String> getLocationOrder() {
            if (locationOrder instanceof SdkAutoConstructList) {
                return null;
            }
            return locationOrder;
        }

        @Override
        public final Builder locationOrder(Collection<String> locationOrder) {
            this.locationOrder = LocationListCopier.copy(locationOrder);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder locationOrder(String... locationOrder) {
            locationOrder(Arrays.asList(locationOrder));
            return this;
        }

        public final void setLocationOrder(Collection<String> locationOrder) {
            this.locationOrder = LocationListCopier.copy(locationOrder);
        }

        @Override
        public PriorityConfiguration build() {
            return new PriorityConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
