/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Connection information for a new game session that is created in response to a <a>StartMatchmaking</a> request. Once
 * a match is made, the FlexMatch engine creates a new game session for it. This information, including the game session
 * endpoint and player sessions for each player in the original matchmaking request, is added to the
 * <a>MatchmakingTicket</a>, which can be retrieved by calling <a>DescribeMatchmaking</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GameSessionConnectionInfo implements SdkPojo, Serializable,
        ToCopyableBuilder<GameSessionConnectionInfo.Builder, GameSessionConnectionInfo> {
    private static final SdkField<String> GAME_SESSION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameSessionArn").getter(getter(GameSessionConnectionInfo::gameSessionArn))
            .setter(setter(Builder::gameSessionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionArn").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddress").getter(getter(GameSessionConnectionInfo::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddress").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DnsName").getter(getter(GameSessionConnectionInfo::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(GameSessionConnectionInfo::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<List<MatchedPlayerSession>> MATCHED_PLAYER_SESSIONS_FIELD = SdkField
            .<List<MatchedPlayerSession>> builder(MarshallingType.LIST)
            .memberName("MatchedPlayerSessions")
            .getter(getter(GameSessionConnectionInfo::matchedPlayerSessions))
            .setter(setter(Builder::matchedPlayerSessions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchedPlayerSessions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MatchedPlayerSession> builder(MarshallingType.SDK_POJO)
                                            .constructor(MatchedPlayerSession::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GAME_SESSION_ARN_FIELD,
            IP_ADDRESS_FIELD, DNS_NAME_FIELD, PORT_FIELD, MATCHED_PLAYER_SESSIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String gameSessionArn;

    private final String ipAddress;

    private final String dnsName;

    private final Integer port;

    private final List<MatchedPlayerSession> matchedPlayerSessions;

    private GameSessionConnectionInfo(BuilderImpl builder) {
        this.gameSessionArn = builder.gameSessionArn;
        this.ipAddress = builder.ipAddress;
        this.dnsName = builder.dnsName;
        this.port = builder.port;
        this.matchedPlayerSessions = builder.matchedPlayerSessions;
    }

    /**
     * <p>
     * A unique identifier for the game session. Use the game session ID.
     * </p>
     * 
     * @return A unique identifier for the game session. Use the game session ID.
     */
    public final String gameSessionArn() {
        return gameSessionArn;
    }

    /**
     * <p>
     * The IP address of the game session. To connect to a GameLift game server, an app needs both the IP address and
     * port number.
     * </p>
     * 
     * @return The IP address of the game session. To connect to a GameLift game server, an app needs both the IP
     *         address and port number.
     */
    public final String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * The DNS identifier assigned to the instance that is running the game session. Values have the following format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     * >Amazon EC2 Instance IP Addressing</a>.)
     * </p>
     * </li>
     * </ul>
     * <p>
     * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the IP
     * address.
     * </p>
     * 
     * @return The DNS identifier assigned to the instance that is running the game session. Values have the following
     *         format:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     *         >Amazon EC2 Instance IP Addressing</a>.)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not
     *         the IP address.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * The port number for the game session. To connect to a GameLift game server, an app needs both the IP address and
     * port number.
     * </p>
     * 
     * @return The port number for the game session. To connect to a GameLift game server, an app needs both the IP
     *         address and port number.
     */
    public final Integer port() {
        return port;
    }

    /**
     * Returns true if the MatchedPlayerSessions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasMatchedPlayerSessions() {
        return matchedPlayerSessions != null && !(matchedPlayerSessions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A collection of player session IDs, one for each player ID that was included in the original matchmaking request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMatchedPlayerSessions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A collection of player session IDs, one for each player ID that was included in the original matchmaking
     *         request.
     */
    public final List<MatchedPlayerSession> matchedPlayerSessions() {
        return matchedPlayerSessions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(hasMatchedPlayerSessions() ? matchedPlayerSessions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GameSessionConnectionInfo)) {
            return false;
        }
        GameSessionConnectionInfo other = (GameSessionConnectionInfo) obj;
        return Objects.equals(gameSessionArn(), other.gameSessionArn()) && Objects.equals(ipAddress(), other.ipAddress())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(port(), other.port())
                && hasMatchedPlayerSessions() == other.hasMatchedPlayerSessions()
                && Objects.equals(matchedPlayerSessions(), other.matchedPlayerSessions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GameSessionConnectionInfo").add("GameSessionArn", gameSessionArn())
                .add("IpAddress", ipAddress()).add("DnsName", dnsName()).add("Port", port())
                .add("MatchedPlayerSessions", hasMatchedPlayerSessions() ? matchedPlayerSessions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "GameSessionArn":
            return Optional.ofNullable(clazz.cast(gameSessionArn()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "MatchedPlayerSessions":
            return Optional.ofNullable(clazz.cast(matchedPlayerSessions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GameSessionConnectionInfo, T> g) {
        return obj -> g.apply((GameSessionConnectionInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GameSessionConnectionInfo> {
        /**
         * <p>
         * A unique identifier for the game session. Use the game session ID.
         * </p>
         * 
         * @param gameSessionArn
         *        A unique identifier for the game session. Use the game session ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionArn(String gameSessionArn);

        /**
         * <p>
         * The IP address of the game session. To connect to a GameLift game server, an app needs both the IP address
         * and port number.
         * </p>
         * 
         * @param ipAddress
         *        The IP address of the game session. To connect to a GameLift game server, an app needs both the IP
         *        address and port number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * The DNS identifier assigned to the instance that is running the game session. Values have the following
         * format:
         * </p>
         * <ul>
         * <li>
         * <p>
         * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         * >Amazon EC2 Instance IP Addressing</a>.)
         * </p>
         * </li>
         * </ul>
         * <p>
         * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the
         * IP address.
         * </p>
         * 
         * @param dnsName
         *        The DNS identifier assigned to the instance that is running the game session. Values have the
         *        following format:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        TLS-enabled fleets:
         *        <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a
         *        href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         *        >Amazon EC2 Instance IP Addressing</a>.)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name,
         *        not the IP address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * The port number for the game session. To connect to a GameLift game server, an app needs both the IP address
         * and port number.
         * </p>
         * 
         * @param port
         *        The port number for the game session. To connect to a GameLift game server, an app needs both the IP
         *        address and port number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * A collection of player session IDs, one for each player ID that was included in the original matchmaking
         * request.
         * </p>
         * 
         * @param matchedPlayerSessions
         *        A collection of player session IDs, one for each player ID that was included in the original
         *        matchmaking request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchedPlayerSessions(Collection<MatchedPlayerSession> matchedPlayerSessions);

        /**
         * <p>
         * A collection of player session IDs, one for each player ID that was included in the original matchmaking
         * request.
         * </p>
         * 
         * @param matchedPlayerSessions
         *        A collection of player session IDs, one for each player ID that was included in the original
         *        matchmaking request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchedPlayerSessions(MatchedPlayerSession... matchedPlayerSessions);

        /**
         * <p>
         * A collection of player session IDs, one for each player ID that was included in the original matchmaking
         * request.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MatchedPlayerSession>.Builder} avoiding the
         * need to create one manually via {@link List<MatchedPlayerSession>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MatchedPlayerSession>.Builder#build()} is called immediately
         * and its result is passed to {@link #matchedPlayerSessions(List<MatchedPlayerSession>)}.
         * 
         * @param matchedPlayerSessions
         *        a consumer that will call methods on {@link List<MatchedPlayerSession>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #matchedPlayerSessions(List<MatchedPlayerSession>)
         */
        Builder matchedPlayerSessions(Consumer<MatchedPlayerSession.Builder>... matchedPlayerSessions);
    }

    static final class BuilderImpl implements Builder {
        private String gameSessionArn;

        private String ipAddress;

        private String dnsName;

        private Integer port;

        private List<MatchedPlayerSession> matchedPlayerSessions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GameSessionConnectionInfo model) {
            gameSessionArn(model.gameSessionArn);
            ipAddress(model.ipAddress);
            dnsName(model.dnsName);
            port(model.port);
            matchedPlayerSessions(model.matchedPlayerSessions);
        }

        public final String getGameSessionArn() {
            return gameSessionArn;
        }

        @Override
        public final Builder gameSessionArn(String gameSessionArn) {
            this.gameSessionArn = gameSessionArn;
            return this;
        }

        public final void setGameSessionArn(String gameSessionArn) {
            this.gameSessionArn = gameSessionArn;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        public final String getDnsName() {
            return dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        public final Integer getPort() {
            return port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        public final List<MatchedPlayerSession.Builder> getMatchedPlayerSessions() {
            List<MatchedPlayerSession.Builder> result = MatchedPlayerSessionListCopier.copyToBuilder(this.matchedPlayerSessions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder matchedPlayerSessions(Collection<MatchedPlayerSession> matchedPlayerSessions) {
            this.matchedPlayerSessions = MatchedPlayerSessionListCopier.copy(matchedPlayerSessions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchedPlayerSessions(MatchedPlayerSession... matchedPlayerSessions) {
            matchedPlayerSessions(Arrays.asList(matchedPlayerSessions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchedPlayerSessions(Consumer<MatchedPlayerSession.Builder>... matchedPlayerSessions) {
            matchedPlayerSessions(Stream.of(matchedPlayerSessions)
                    .map(c -> MatchedPlayerSession.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setMatchedPlayerSessions(Collection<MatchedPlayerSession.BuilderImpl> matchedPlayerSessions) {
            this.matchedPlayerSessions = MatchedPlayerSessionListCopier.copyFromBuilder(matchedPlayerSessions);
        }

        @Override
        public GameSessionConnectionInfo build() {
            return new GameSessionConnectionInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
