/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateScriptRequest extends GameLiftRequest implements
        ToCopyableBuilder<CreateScriptRequest.Builder, CreateScriptRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateScriptRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(CreateScriptRequest::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<S3Location> STORAGE_LOCATION_FIELD = SdkField.<S3Location> builder(MarshallingType.SDK_POJO)
            .memberName("StorageLocation").getter(getter(CreateScriptRequest::storageLocation))
            .setter(setter(Builder::storageLocation)).constructor(S3Location::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageLocation").build()).build();

    private static final SdkField<SdkBytes> ZIP_FILE_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("ZipFile").getter(getter(CreateScriptRequest::zipFile)).setter(setter(Builder::zipFile))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ZipFile").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateScriptRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VERSION_FIELD,
            STORAGE_LOCATION_FIELD, ZIP_FILE_FIELD, TAGS_FIELD));

    private final String name;

    private final String version;

    private final S3Location storageLocation;

    private final SdkBytes zipFile;

    private final List<Tag> tags;

    private CreateScriptRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.version = builder.version;
        this.storageLocation = builder.storageLocation;
        this.zipFile = builder.zipFile;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * A descriptive label that is associated with a script. Script names do not need to be unique. You can use
     * <a>UpdateScript</a> to change this value later.
     * </p>
     * 
     * @return A descriptive label that is associated with a script. Script names do not need to be unique. You can use
     *         <a>UpdateScript</a> to change this value later.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Version information that is associated with a build or script. Version strings do not need to be unique. You can
     * use <a>UpdateScript</a> to change this value later.
     * </p>
     * 
     * @return Version information that is associated with a build or script. Version strings do not need to be unique.
     *         You can use <a>UpdateScript</a> to change this value later.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The location of the Amazon S3 bucket where a zipped file containing your Realtime scripts is stored. The storage
     * location must specify the Amazon S3 bucket name, the zip file name (the "key"), and a role ARN that allows Amazon
     * GameLift to access the Amazon S3 storage location. The S3 bucket must be in the same Region where you want to
     * create a new script. By default, Amazon GameLift uploads the latest version of the zip file; if you have S3
     * object versioning turned on, you can use the <code>ObjectVersion</code> parameter to specify an earlier version.
     * </p>
     * 
     * @return The location of the Amazon S3 bucket where a zipped file containing your Realtime scripts is stored. The
     *         storage location must specify the Amazon S3 bucket name, the zip file name (the "key"), and a role ARN
     *         that allows Amazon GameLift to access the Amazon S3 storage location. The S3 bucket must be in the same
     *         Region where you want to create a new script. By default, Amazon GameLift uploads the latest version of
     *         the zip file; if you have S3 object versioning turned on, you can use the <code>ObjectVersion</code>
     *         parameter to specify an earlier version.
     */
    public final S3Location storageLocation() {
        return storageLocation;
    }

    /**
     * <p>
     * A data object containing your Realtime scripts and dependencies as a zip file. The zip file can have one or
     * multiple files. Maximum size of a zip file is 5 MB.
     * </p>
     * <p>
     * When using the Amazon Web Services CLI tool to create a script, this parameter is set to the zip file name. It
     * must be prepended with the string "fileb://" to indicate that the file data is a binary object. For example:
     * <code>--zip-file fileb://myRealtimeScript.zip</code>.
     * </p>
     * 
     * @return A data object containing your Realtime scripts and dependencies as a zip file. The zip file can have one
     *         or multiple files. Maximum size of a zip file is 5 MB.</p>
     *         <p>
     *         When using the Amazon Web Services CLI tool to create a script, this parameter is set to the zip file
     *         name. It must be prepended with the string "fileb://" to indicate that the file data is a binary object.
     *         For example: <code>--zip-file fileb://myRealtimeScript.zip</code>.
     */
    public final SdkBytes zipFile() {
        return zipFile;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of labels to assign to the new script resource. Tags are developer-defined key-value pairs. Tagging Amazon
     * Web Services resources are useful for resource management, access management and cost allocation. For more
     * information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging Amazon Web
     * Services Resources</a> in the <i>Amazon Web Services General Reference</i>. Once the resource is created, you can
     * use <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove, and view tags. The
     * maximum tag limit may be lower than stated. See the Amazon Web Services General Reference for actual tagging
     * limits.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of labels to assign to the new script resource. Tags are developer-defined key-value pairs.
     *         Tagging Amazon Web Services resources are useful for resource management, access management and cost
     *         allocation. For more information, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging Amazon Web Services
     *         Resources</a> in the <i>Amazon Web Services General Reference</i>. Once the resource is created, you can
     *         use <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove, and view
     *         tags. The maximum tag limit may be lower than stated. See the Amazon Web Services General Reference for
     *         actual tagging limits.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(storageLocation());
        hashCode = 31 * hashCode + Objects.hashCode(zipFile());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateScriptRequest)) {
            return false;
        }
        CreateScriptRequest other = (CreateScriptRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(version(), other.version())
                && Objects.equals(storageLocation(), other.storageLocation()) && Objects.equals(zipFile(), other.zipFile())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateScriptRequest").add("Name", name()).add("Version", version())
                .add("StorageLocation", storageLocation()).add("ZipFile", zipFile()).add("Tags", hasTags() ? tags() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "StorageLocation":
            return Optional.ofNullable(clazz.cast(storageLocation()));
        case "ZipFile":
            return Optional.ofNullable(clazz.cast(zipFile()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateScriptRequest, T> g) {
        return obj -> g.apply((CreateScriptRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends GameLiftRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateScriptRequest> {
        /**
         * <p>
         * A descriptive label that is associated with a script. Script names do not need to be unique. You can use
         * <a>UpdateScript</a> to change this value later.
         * </p>
         * 
         * @param name
         *        A descriptive label that is associated with a script. Script names do not need to be unique. You can
         *        use <a>UpdateScript</a> to change this value later.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Version information that is associated with a build or script. Version strings do not need to be unique. You
         * can use <a>UpdateScript</a> to change this value later.
         * </p>
         * 
         * @param version
         *        Version information that is associated with a build or script. Version strings do not need to be
         *        unique. You can use <a>UpdateScript</a> to change this value later.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The location of the Amazon S3 bucket where a zipped file containing your Realtime scripts is stored. The
         * storage location must specify the Amazon S3 bucket name, the zip file name (the "key"), and a role ARN that
         * allows Amazon GameLift to access the Amazon S3 storage location. The S3 bucket must be in the same Region
         * where you want to create a new script. By default, Amazon GameLift uploads the latest version of the zip
         * file; if you have S3 object versioning turned on, you can use the <code>ObjectVersion</code> parameter to
         * specify an earlier version.
         * </p>
         * 
         * @param storageLocation
         *        The location of the Amazon S3 bucket where a zipped file containing your Realtime scripts is stored.
         *        The storage location must specify the Amazon S3 bucket name, the zip file name (the "key"), and a role
         *        ARN that allows Amazon GameLift to access the Amazon S3 storage location. The S3 bucket must be in the
         *        same Region where you want to create a new script. By default, Amazon GameLift uploads the latest
         *        version of the zip file; if you have S3 object versioning turned on, you can use the
         *        <code>ObjectVersion</code> parameter to specify an earlier version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageLocation(S3Location storageLocation);

        /**
         * <p>
         * The location of the Amazon S3 bucket where a zipped file containing your Realtime scripts is stored. The
         * storage location must specify the Amazon S3 bucket name, the zip file name (the "key"), and a role ARN that
         * allows Amazon GameLift to access the Amazon S3 storage location. The S3 bucket must be in the same Region
         * where you want to create a new script. By default, Amazon GameLift uploads the latest version of the zip
         * file; if you have S3 object versioning turned on, you can use the <code>ObjectVersion</code> parameter to
         * specify an earlier version.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3Location.Builder} avoiding the need to
         * create one manually via {@link S3Location#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Location.Builder#build()} is called immediately and its result
         * is passed to {@link #storageLocation(S3Location)}.
         * 
         * @param storageLocation
         *        a consumer that will call methods on {@link S3Location.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #storageLocation(S3Location)
         */
        default Builder storageLocation(Consumer<S3Location.Builder> storageLocation) {
            return storageLocation(S3Location.builder().applyMutation(storageLocation).build());
        }

        /**
         * <p>
         * A data object containing your Realtime scripts and dependencies as a zip file. The zip file can have one or
         * multiple files. Maximum size of a zip file is 5 MB.
         * </p>
         * <p>
         * When using the Amazon Web Services CLI tool to create a script, this parameter is set to the zip file name.
         * It must be prepended with the string "fileb://" to indicate that the file data is a binary object. For
         * example: <code>--zip-file fileb://myRealtimeScript.zip</code>.
         * </p>
         * 
         * @param zipFile
         *        A data object containing your Realtime scripts and dependencies as a zip file. The zip file can have
         *        one or multiple files. Maximum size of a zip file is 5 MB.</p>
         *        <p>
         *        When using the Amazon Web Services CLI tool to create a script, this parameter is set to the zip file
         *        name. It must be prepended with the string "fileb://" to indicate that the file data is a binary
         *        object. For example: <code>--zip-file fileb://myRealtimeScript.zip</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder zipFile(SdkBytes zipFile);

        /**
         * <p>
         * A list of labels to assign to the new script resource. Tags are developer-defined key-value pairs. Tagging
         * Amazon Web Services resources are useful for resource management, access management and cost allocation. For
         * more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging
         * Amazon Web Services Resources</a> in the <i>Amazon Web Services General Reference</i>. Once the resource is
         * created, you can use <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove,
         * and view tags. The maximum tag limit may be lower than stated. See the Amazon Web Services General Reference
         * for actual tagging limits.
         * </p>
         * 
         * @param tags
         *        A list of labels to assign to the new script resource. Tags are developer-defined key-value pairs.
         *        Tagging Amazon Web Services resources are useful for resource management, access management and cost
         *        allocation. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging Amazon Web Services
         *        Resources</a> in the <i>Amazon Web Services General Reference</i>. Once the resource is created, you
         *        can use <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove, and
         *        view tags. The maximum tag limit may be lower than stated. See the Amazon Web Services General
         *        Reference for actual tagging limits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of labels to assign to the new script resource. Tags are developer-defined key-value pairs. Tagging
         * Amazon Web Services resources are useful for resource management, access management and cost allocation. For
         * more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging
         * Amazon Web Services Resources</a> in the <i>Amazon Web Services General Reference</i>. Once the resource is
         * created, you can use <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove,
         * and view tags. The maximum tag limit may be lower than stated. See the Amazon Web Services General Reference
         * for actual tagging limits.
         * </p>
         * 
         * @param tags
         *        A list of labels to assign to the new script resource. Tags are developer-defined key-value pairs.
         *        Tagging Amazon Web Services resources are useful for resource management, access management and cost
         *        allocation. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging Amazon Web Services
         *        Resources</a> in the <i>Amazon Web Services General Reference</i>. Once the resource is created, you
         *        can use <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove, and
         *        view tags. The maximum tag limit may be lower than stated. See the Amazon Web Services General
         *        Reference for actual tagging limits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of labels to assign to the new script resource. Tags are developer-defined key-value pairs. Tagging
         * Amazon Web Services resources are useful for resource management, access management and cost allocation. For
         * more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging
         * Amazon Web Services Resources</a> in the <i>Amazon Web Services General Reference</i>. Once the resource is
         * created, you can use <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove,
         * and view tags. The maximum tag limit may be lower than stated. See the Amazon Web Services General Reference
         * for actual tagging limits.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Tag>.Builder} avoiding the need to
         * create one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GameLiftRequest.BuilderImpl implements Builder {
        private String name;

        private String version;

        private S3Location storageLocation;

        private SdkBytes zipFile;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateScriptRequest model) {
            super(model);
            name(model.name);
            version(model.version);
            storageLocation(model.storageLocation);
            zipFile(model.zipFile);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final S3Location.Builder getStorageLocation() {
            return storageLocation != null ? storageLocation.toBuilder() : null;
        }

        public final void setStorageLocation(S3Location.BuilderImpl storageLocation) {
            this.storageLocation = storageLocation != null ? storageLocation.build() : null;
        }

        @Override
        public final Builder storageLocation(S3Location storageLocation) {
            this.storageLocation = storageLocation;
            return this;
        }

        public final ByteBuffer getZipFile() {
            return zipFile == null ? null : zipFile.asByteBuffer();
        }

        public final void setZipFile(ByteBuffer zipFile) {
            zipFile(zipFile == null ? null : SdkBytes.fromByteBuffer(zipFile));
        }

        @Override
        public final Builder zipFile(SdkBytes zipFile) {
            this.zipFile = zipFile;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateScriptRequest build() {
            return new CreateScriptRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
