/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the input for a request operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateGameSessionQueueRequest extends GameLiftRequest implements
        ToCopyableBuilder<CreateGameSessionQueueRequest.Builder, CreateGameSessionQueueRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateGameSessionQueueRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Integer> TIMEOUT_IN_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TimeoutInSeconds").getter(getter(CreateGameSessionQueueRequest::timeoutInSeconds))
            .setter(setter(Builder::timeoutInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeoutInSeconds").build()).build();

    private static final SdkField<List<PlayerLatencyPolicy>> PLAYER_LATENCY_POLICIES_FIELD = SdkField
            .<List<PlayerLatencyPolicy>> builder(MarshallingType.LIST)
            .memberName("PlayerLatencyPolicies")
            .getter(getter(CreateGameSessionQueueRequest::playerLatencyPolicies))
            .setter(setter(Builder::playerLatencyPolicies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlayerLatencyPolicies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PlayerLatencyPolicy> builder(MarshallingType.SDK_POJO)
                                            .constructor(PlayerLatencyPolicy::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<GameSessionQueueDestination>> DESTINATIONS_FIELD = SdkField
            .<List<GameSessionQueueDestination>> builder(MarshallingType.LIST)
            .memberName("Destinations")
            .getter(getter(CreateGameSessionQueueRequest::destinations))
            .setter(setter(Builder::destinations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GameSessionQueueDestination> builder(MarshallingType.SDK_POJO)
                                            .constructor(GameSessionQueueDestination::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<FilterConfiguration> FILTER_CONFIGURATION_FIELD = SdkField
            .<FilterConfiguration> builder(MarshallingType.SDK_POJO).memberName("FilterConfiguration")
            .getter(getter(CreateGameSessionQueueRequest::filterConfiguration)).setter(setter(Builder::filterConfiguration))
            .constructor(FilterConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FilterConfiguration").build())
            .build();

    private static final SdkField<PriorityConfiguration> PRIORITY_CONFIGURATION_FIELD = SdkField
            .<PriorityConfiguration> builder(MarshallingType.SDK_POJO).memberName("PriorityConfiguration")
            .getter(getter(CreateGameSessionQueueRequest::priorityConfiguration)).setter(setter(Builder::priorityConfiguration))
            .constructor(PriorityConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PriorityConfiguration").build())
            .build();

    private static final SdkField<String> CUSTOM_EVENT_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomEventData").getter(getter(CreateGameSessionQueueRequest::customEventData))
            .setter(setter(Builder::customEventData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomEventData").build()).build();

    private static final SdkField<String> NOTIFICATION_TARGET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NotificationTarget").getter(getter(CreateGameSessionQueueRequest::notificationTarget))
            .setter(setter(Builder::notificationTarget))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationTarget").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateGameSessionQueueRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            TIMEOUT_IN_SECONDS_FIELD, PLAYER_LATENCY_POLICIES_FIELD, DESTINATIONS_FIELD, FILTER_CONFIGURATION_FIELD,
            PRIORITY_CONFIGURATION_FIELD, CUSTOM_EVENT_DATA_FIELD, NOTIFICATION_TARGET_FIELD, TAGS_FIELD));

    private final String name;

    private final Integer timeoutInSeconds;

    private final List<PlayerLatencyPolicy> playerLatencyPolicies;

    private final List<GameSessionQueueDestination> destinations;

    private final FilterConfiguration filterConfiguration;

    private final PriorityConfiguration priorityConfiguration;

    private final String customEventData;

    private final String notificationTarget;

    private final List<Tag> tags;

    private CreateGameSessionQueueRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.timeoutInSeconds = builder.timeoutInSeconds;
        this.playerLatencyPolicies = builder.playerLatencyPolicies;
        this.destinations = builder.destinations;
        this.filterConfiguration = builder.filterConfiguration;
        this.priorityConfiguration = builder.priorityConfiguration;
        this.customEventData = builder.customEventData;
        this.notificationTarget = builder.notificationTarget;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * A descriptive label that is associated with game session queue. Queue names must be unique within each Region.
     * </p>
     * 
     * @return A descriptive label that is associated with game session queue. Queue names must be unique within each
     *         Region.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The maximum time, in seconds, that a new game session placement request remains in the queue. When a request
     * exceeds this time, the game session placement changes to a <code>TIMED_OUT</code> status.
     * </p>
     * 
     * @return The maximum time, in seconds, that a new game session placement request remains in the queue. When a
     *         request exceeds this time, the game session placement changes to a <code>TIMED_OUT</code> status.
     */
    public final Integer timeoutInSeconds() {
        return timeoutInSeconds;
    }

    /**
     * For responses, this returns true if the service returned a value for the PlayerLatencyPolicies property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasPlayerLatencyPolicies() {
        return playerLatencyPolicies != null && !(playerLatencyPolicies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of policies that act as a sliding cap on player latency. FleetIQ works to deliver low latency for most
     * players in a game session. These policies ensure that no individual player can be placed into a game with
     * unreasonably high latency. Use multiple policies to gradually relax latency requirements a step at a time.
     * Multiple policies are applied based on their maximum allowed latency, starting with the lowest value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlayerLatencyPolicies} method.
     * </p>
     * 
     * @return A set of policies that act as a sliding cap on player latency. FleetIQ works to deliver low latency for
     *         most players in a game session. These policies ensure that no individual player can be placed into a game
     *         with unreasonably high latency. Use multiple policies to gradually relax latency requirements a step at a
     *         time. Multiple policies are applied based on their maximum allowed latency, starting with the lowest
     *         value.
     */
    public final List<PlayerLatencyPolicy> playerLatencyPolicies() {
        return playerLatencyPolicies;
    }

    /**
     * For responses, this returns true if the service returned a value for the Destinations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDestinations() {
        return destinations != null && !(destinations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of fleets and/or fleet aliases that can be used to fulfill game session placement requests in the queue.
     * Destinations are identified by either a fleet ARN or a fleet alias ARN, and are listed in order of placement
     * preference.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDestinations} method.
     * </p>
     * 
     * @return A list of fleets and/or fleet aliases that can be used to fulfill game session placement requests in the
     *         queue. Destinations are identified by either a fleet ARN or a fleet alias ARN, and are listed in order of
     *         placement preference.
     */
    public final List<GameSessionQueueDestination> destinations() {
        return destinations;
    }

    /**
     * <p>
     * A list of locations where a queue is allowed to place new game sessions. Locations are specified in the form of
     * AWS Region codes, such as <code>us-west-2</code>. If this parameter is not set, game sessions can be placed in
     * any queue location.
     * </p>
     * 
     * @return A list of locations where a queue is allowed to place new game sessions. Locations are specified in the
     *         form of AWS Region codes, such as <code>us-west-2</code>. If this parameter is not set, game sessions can
     *         be placed in any queue location.
     */
    public final FilterConfiguration filterConfiguration() {
        return filterConfiguration;
    }

    /**
     * <p>
     * Custom settings to use when prioritizing destinations and locations for game session placements. This
     * configuration replaces the FleetIQ default prioritization process. Priority types that are not explicitly named
     * will be automatically applied at the end of the prioritization process.
     * </p>
     * 
     * @return Custom settings to use when prioritizing destinations and locations for game session placements. This
     *         configuration replaces the FleetIQ default prioritization process. Priority types that are not explicitly
     *         named will be automatically applied at the end of the prioritization process.
     */
    public final PriorityConfiguration priorityConfiguration() {
        return priorityConfiguration;
    }

    /**
     * <p>
     * Information to be added to all events that are related to this game session queue.
     * </p>
     * 
     * @return Information to be added to all events that are related to this game session queue.
     */
    public final String customEventData() {
        return customEventData;
    }

    /**
     * <p>
     * An SNS topic ARN that is set up to receive game session placement notifications. See <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/queue-notification.html"> Setting up
     * notifications for game session placement</a>.
     * </p>
     * 
     * @return An SNS topic ARN that is set up to receive game session placement notifications. See <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/queue-notification.html"> Setting up
     *         notifications for game session placement</a>.
     */
    public final String notificationTarget() {
        return notificationTarget;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of labels to assign to the new game session queue resource. Tags are developer-defined key-value pairs.
     * Tagging AWS resources are useful for resource management, access management and cost allocation. For more
     * information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging AWS
     * Resources</a> in the <i>AWS General Reference</i>. Once the resource is created, you can use <a>TagResource</a>,
     * <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove, and view tags. The maximum tag limit may be
     * lower than stated. See the AWS General Reference for actual tagging limits.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of labels to assign to the new game session queue resource. Tags are developer-defined key-value
     *         pairs. Tagging AWS resources are useful for resource management, access management and cost allocation.
     *         For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">
     *         Tagging AWS Resources</a> in the <i>AWS General Reference</i>. Once the resource is created, you can use
     *         <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove, and view tags.
     *         The maximum tag limit may be lower than stated. See the AWS General Reference for actual tagging limits.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlayerLatencyPolicies() ? playerLatencyPolicies() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDestinations() ? destinations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(filterConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(priorityConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(customEventData());
        hashCode = 31 * hashCode + Objects.hashCode(notificationTarget());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateGameSessionQueueRequest)) {
            return false;
        }
        CreateGameSessionQueueRequest other = (CreateGameSessionQueueRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(timeoutInSeconds(), other.timeoutInSeconds())
                && hasPlayerLatencyPolicies() == other.hasPlayerLatencyPolicies()
                && Objects.equals(playerLatencyPolicies(), other.playerLatencyPolicies())
                && hasDestinations() == other.hasDestinations() && Objects.equals(destinations(), other.destinations())
                && Objects.equals(filterConfiguration(), other.filterConfiguration())
                && Objects.equals(priorityConfiguration(), other.priorityConfiguration())
                && Objects.equals(customEventData(), other.customEventData())
                && Objects.equals(notificationTarget(), other.notificationTarget()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateGameSessionQueueRequest").add("Name", name()).add("TimeoutInSeconds", timeoutInSeconds())
                .add("PlayerLatencyPolicies", hasPlayerLatencyPolicies() ? playerLatencyPolicies() : null)
                .add("Destinations", hasDestinations() ? destinations() : null).add("FilterConfiguration", filterConfiguration())
                .add("PriorityConfiguration", priorityConfiguration()).add("CustomEventData", customEventData())
                .add("NotificationTarget", notificationTarget()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "TimeoutInSeconds":
            return Optional.ofNullable(clazz.cast(timeoutInSeconds()));
        case "PlayerLatencyPolicies":
            return Optional.ofNullable(clazz.cast(playerLatencyPolicies()));
        case "Destinations":
            return Optional.ofNullable(clazz.cast(destinations()));
        case "FilterConfiguration":
            return Optional.ofNullable(clazz.cast(filterConfiguration()));
        case "PriorityConfiguration":
            return Optional.ofNullable(clazz.cast(priorityConfiguration()));
        case "CustomEventData":
            return Optional.ofNullable(clazz.cast(customEventData()));
        case "NotificationTarget":
            return Optional.ofNullable(clazz.cast(notificationTarget()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateGameSessionQueueRequest, T> g) {
        return obj -> g.apply((CreateGameSessionQueueRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends GameLiftRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateGameSessionQueueRequest> {
        /**
         * <p>
         * A descriptive label that is associated with game session queue. Queue names must be unique within each
         * Region.
         * </p>
         * 
         * @param name
         *        A descriptive label that is associated with game session queue. Queue names must be unique within each
         *        Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The maximum time, in seconds, that a new game session placement request remains in the queue. When a request
         * exceeds this time, the game session placement changes to a <code>TIMED_OUT</code> status.
         * </p>
         * 
         * @param timeoutInSeconds
         *        The maximum time, in seconds, that a new game session placement request remains in the queue. When a
         *        request exceeds this time, the game session placement changes to a <code>TIMED_OUT</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutInSeconds(Integer timeoutInSeconds);

        /**
         * <p>
         * A set of policies that act as a sliding cap on player latency. FleetIQ works to deliver low latency for most
         * players in a game session. These policies ensure that no individual player can be placed into a game with
         * unreasonably high latency. Use multiple policies to gradually relax latency requirements a step at a time.
         * Multiple policies are applied based on their maximum allowed latency, starting with the lowest value.
         * </p>
         * 
         * @param playerLatencyPolicies
         *        A set of policies that act as a sliding cap on player latency. FleetIQ works to deliver low latency
         *        for most players in a game session. These policies ensure that no individual player can be placed into
         *        a game with unreasonably high latency. Use multiple policies to gradually relax latency requirements a
         *        step at a time. Multiple policies are applied based on their maximum allowed latency, starting with
         *        the lowest value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerLatencyPolicies(Collection<PlayerLatencyPolicy> playerLatencyPolicies);

        /**
         * <p>
         * A set of policies that act as a sliding cap on player latency. FleetIQ works to deliver low latency for most
         * players in a game session. These policies ensure that no individual player can be placed into a game with
         * unreasonably high latency. Use multiple policies to gradually relax latency requirements a step at a time.
         * Multiple policies are applied based on their maximum allowed latency, starting with the lowest value.
         * </p>
         * 
         * @param playerLatencyPolicies
         *        A set of policies that act as a sliding cap on player latency. FleetIQ works to deliver low latency
         *        for most players in a game session. These policies ensure that no individual player can be placed into
         *        a game with unreasonably high latency. Use multiple policies to gradually relax latency requirements a
         *        step at a time. Multiple policies are applied based on their maximum allowed latency, starting with
         *        the lowest value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerLatencyPolicies(PlayerLatencyPolicy... playerLatencyPolicies);

        /**
         * <p>
         * A set of policies that act as a sliding cap on player latency. FleetIQ works to deliver low latency for most
         * players in a game session. These policies ensure that no individual player can be placed into a game with
         * unreasonably high latency. Use multiple policies to gradually relax latency requirements a step at a time.
         * Multiple policies are applied based on their maximum allowed latency, starting with the lowest value.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PlayerLatencyPolicy>.Builder} avoiding the
         * need to create one manually via {@link List<PlayerLatencyPolicy>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PlayerLatencyPolicy>.Builder#build()} is called immediately
         * and its result is passed to {@link #playerLatencyPolicies(List<PlayerLatencyPolicy>)}.
         * 
         * @param playerLatencyPolicies
         *        a consumer that will call methods on {@link List<PlayerLatencyPolicy>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #playerLatencyPolicies(List<PlayerLatencyPolicy>)
         */
        Builder playerLatencyPolicies(Consumer<PlayerLatencyPolicy.Builder>... playerLatencyPolicies);

        /**
         * <p>
         * A list of fleets and/or fleet aliases that can be used to fulfill game session placement requests in the
         * queue. Destinations are identified by either a fleet ARN or a fleet alias ARN, and are listed in order of
         * placement preference.
         * </p>
         * 
         * @param destinations
         *        A list of fleets and/or fleet aliases that can be used to fulfill game session placement requests in
         *        the queue. Destinations are identified by either a fleet ARN or a fleet alias ARN, and are listed in
         *        order of placement preference.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<GameSessionQueueDestination> destinations);

        /**
         * <p>
         * A list of fleets and/or fleet aliases that can be used to fulfill game session placement requests in the
         * queue. Destinations are identified by either a fleet ARN or a fleet alias ARN, and are listed in order of
         * placement preference.
         * </p>
         * 
         * @param destinations
         *        A list of fleets and/or fleet aliases that can be used to fulfill game session placement requests in
         *        the queue. Destinations are identified by either a fleet ARN or a fleet alias ARN, and are listed in
         *        order of placement preference.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(GameSessionQueueDestination... destinations);

        /**
         * <p>
         * A list of fleets and/or fleet aliases that can be used to fulfill game session placement requests in the
         * queue. Destinations are identified by either a fleet ARN or a fleet alias ARN, and are listed in order of
         * placement preference.
         * </p>
         * This is a convenience that creates an instance of the {@link List<GameSessionQueueDestination>.Builder}
         * avoiding the need to create one manually via {@link List<GameSessionQueueDestination>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<GameSessionQueueDestination>.Builder#build()} is called
         * immediately and its result is passed to {@link #destinations(List<GameSessionQueueDestination>)}.
         * 
         * @param destinations
         *        a consumer that will call methods on {@link List<GameSessionQueueDestination>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinations(List<GameSessionQueueDestination>)
         */
        Builder destinations(Consumer<GameSessionQueueDestination.Builder>... destinations);

        /**
         * <p>
         * A list of locations where a queue is allowed to place new game sessions. Locations are specified in the form
         * of AWS Region codes, such as <code>us-west-2</code>. If this parameter is not set, game sessions can be
         * placed in any queue location.
         * </p>
         * 
         * @param filterConfiguration
         *        A list of locations where a queue is allowed to place new game sessions. Locations are specified in
         *        the form of AWS Region codes, such as <code>us-west-2</code>. If this parameter is not set, game
         *        sessions can be placed in any queue location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterConfiguration(FilterConfiguration filterConfiguration);

        /**
         * <p>
         * A list of locations where a queue is allowed to place new game sessions. Locations are specified in the form
         * of AWS Region codes, such as <code>us-west-2</code>. If this parameter is not set, game sessions can be
         * placed in any queue location.
         * </p>
         * This is a convenience that creates an instance of the {@link FilterConfiguration.Builder} avoiding the need
         * to create one manually via {@link FilterConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link FilterConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #filterConfiguration(FilterConfiguration)}.
         * 
         * @param filterConfiguration
         *        a consumer that will call methods on {@link FilterConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filterConfiguration(FilterConfiguration)
         */
        default Builder filterConfiguration(Consumer<FilterConfiguration.Builder> filterConfiguration) {
            return filterConfiguration(FilterConfiguration.builder().applyMutation(filterConfiguration).build());
        }

        /**
         * <p>
         * Custom settings to use when prioritizing destinations and locations for game session placements. This
         * configuration replaces the FleetIQ default prioritization process. Priority types that are not explicitly
         * named will be automatically applied at the end of the prioritization process.
         * </p>
         * 
         * @param priorityConfiguration
         *        Custom settings to use when prioritizing destinations and locations for game session placements. This
         *        configuration replaces the FleetIQ default prioritization process. Priority types that are not
         *        explicitly named will be automatically applied at the end of the prioritization process.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priorityConfiguration(PriorityConfiguration priorityConfiguration);

        /**
         * <p>
         * Custom settings to use when prioritizing destinations and locations for game session placements. This
         * configuration replaces the FleetIQ default prioritization process. Priority types that are not explicitly
         * named will be automatically applied at the end of the prioritization process.
         * </p>
         * This is a convenience that creates an instance of the {@link PriorityConfiguration.Builder} avoiding the need
         * to create one manually via {@link PriorityConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link PriorityConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #priorityConfiguration(PriorityConfiguration)}.
         * 
         * @param priorityConfiguration
         *        a consumer that will call methods on {@link PriorityConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #priorityConfiguration(PriorityConfiguration)
         */
        default Builder priorityConfiguration(Consumer<PriorityConfiguration.Builder> priorityConfiguration) {
            return priorityConfiguration(PriorityConfiguration.builder().applyMutation(priorityConfiguration).build());
        }

        /**
         * <p>
         * Information to be added to all events that are related to this game session queue.
         * </p>
         * 
         * @param customEventData
         *        Information to be added to all events that are related to this game session queue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customEventData(String customEventData);

        /**
         * <p>
         * An SNS topic ARN that is set up to receive game session placement notifications. See <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/queue-notification.html"> Setting up
         * notifications for game session placement</a>.
         * </p>
         * 
         * @param notificationTarget
         *        An SNS topic ARN that is set up to receive game session placement notifications. See <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/queue-notification.html"> Setting up
         *        notifications for game session placement</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTarget(String notificationTarget);

        /**
         * <p>
         * A list of labels to assign to the new game session queue resource. Tags are developer-defined key-value
         * pairs. Tagging AWS resources are useful for resource management, access management and cost allocation. For
         * more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging AWS
         * Resources</a> in the <i>AWS General Reference</i>. Once the resource is created, you can use
         * <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove, and view tags. The
         * maximum tag limit may be lower than stated. See the AWS General Reference for actual tagging limits.
         * </p>
         * 
         * @param tags
         *        A list of labels to assign to the new game session queue resource. Tags are developer-defined
         *        key-value pairs. Tagging AWS resources are useful for resource management, access management and cost
         *        allocation. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging AWS Resources</a> in
         *        the <i>AWS General Reference</i>. Once the resource is created, you can use <a>TagResource</a>,
         *        <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove, and view tags. The maximum tag
         *        limit may be lower than stated. See the AWS General Reference for actual tagging limits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of labels to assign to the new game session queue resource. Tags are developer-defined key-value
         * pairs. Tagging AWS resources are useful for resource management, access management and cost allocation. For
         * more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging AWS
         * Resources</a> in the <i>AWS General Reference</i>. Once the resource is created, you can use
         * <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove, and view tags. The
         * maximum tag limit may be lower than stated. See the AWS General Reference for actual tagging limits.
         * </p>
         * 
         * @param tags
         *        A list of labels to assign to the new game session queue resource. Tags are developer-defined
         *        key-value pairs. Tagging AWS resources are useful for resource management, access management and cost
         *        allocation. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging AWS Resources</a> in
         *        the <i>AWS General Reference</i>. Once the resource is created, you can use <a>TagResource</a>,
         *        <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove, and view tags. The maximum tag
         *        limit may be lower than stated. See the AWS General Reference for actual tagging limits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of labels to assign to the new game session queue resource. Tags are developer-defined key-value
         * pairs. Tagging AWS resources are useful for resource management, access management and cost allocation. For
         * more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging AWS
         * Resources</a> in the <i>AWS General Reference</i>. Once the resource is created, you can use
         * <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a> to add, remove, and view tags. The
         * maximum tag limit may be lower than stated. See the AWS General Reference for actual tagging limits.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GameLiftRequest.BuilderImpl implements Builder {
        private String name;

        private Integer timeoutInSeconds;

        private List<PlayerLatencyPolicy> playerLatencyPolicies = DefaultSdkAutoConstructList.getInstance();

        private List<GameSessionQueueDestination> destinations = DefaultSdkAutoConstructList.getInstance();

        private FilterConfiguration filterConfiguration;

        private PriorityConfiguration priorityConfiguration;

        private String customEventData;

        private String notificationTarget;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateGameSessionQueueRequest model) {
            super(model);
            name(model.name);
            timeoutInSeconds(model.timeoutInSeconds);
            playerLatencyPolicies(model.playerLatencyPolicies);
            destinations(model.destinations);
            filterConfiguration(model.filterConfiguration);
            priorityConfiguration(model.priorityConfiguration);
            customEventData(model.customEventData);
            notificationTarget(model.notificationTarget);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Integer getTimeoutInSeconds() {
            return timeoutInSeconds;
        }

        public final void setTimeoutInSeconds(Integer timeoutInSeconds) {
            this.timeoutInSeconds = timeoutInSeconds;
        }

        @Override
        @Transient
        public final Builder timeoutInSeconds(Integer timeoutInSeconds) {
            this.timeoutInSeconds = timeoutInSeconds;
            return this;
        }

        public final List<PlayerLatencyPolicy.Builder> getPlayerLatencyPolicies() {
            List<PlayerLatencyPolicy.Builder> result = PlayerLatencyPolicyListCopier.copyToBuilder(this.playerLatencyPolicies);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPlayerLatencyPolicies(Collection<PlayerLatencyPolicy.BuilderImpl> playerLatencyPolicies) {
            this.playerLatencyPolicies = PlayerLatencyPolicyListCopier.copyFromBuilder(playerLatencyPolicies);
        }

        @Override
        @Transient
        public final Builder playerLatencyPolicies(Collection<PlayerLatencyPolicy> playerLatencyPolicies) {
            this.playerLatencyPolicies = PlayerLatencyPolicyListCopier.copy(playerLatencyPolicies);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder playerLatencyPolicies(PlayerLatencyPolicy... playerLatencyPolicies) {
            playerLatencyPolicies(Arrays.asList(playerLatencyPolicies));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder playerLatencyPolicies(Consumer<PlayerLatencyPolicy.Builder>... playerLatencyPolicies) {
            playerLatencyPolicies(Stream.of(playerLatencyPolicies)
                    .map(c -> PlayerLatencyPolicy.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<GameSessionQueueDestination.Builder> getDestinations() {
            List<GameSessionQueueDestination.Builder> result = GameSessionQueueDestinationListCopier
                    .copyToBuilder(this.destinations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDestinations(Collection<GameSessionQueueDestination.BuilderImpl> destinations) {
            this.destinations = GameSessionQueueDestinationListCopier.copyFromBuilder(destinations);
        }

        @Override
        @Transient
        public final Builder destinations(Collection<GameSessionQueueDestination> destinations) {
            this.destinations = GameSessionQueueDestinationListCopier.copy(destinations);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder destinations(GameSessionQueueDestination... destinations) {
            destinations(Arrays.asList(destinations));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder destinations(Consumer<GameSessionQueueDestination.Builder>... destinations) {
            destinations(Stream.of(destinations).map(c -> GameSessionQueueDestination.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final FilterConfiguration.Builder getFilterConfiguration() {
            return filterConfiguration != null ? filterConfiguration.toBuilder() : null;
        }

        public final void setFilterConfiguration(FilterConfiguration.BuilderImpl filterConfiguration) {
            this.filterConfiguration = filterConfiguration != null ? filterConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder filterConfiguration(FilterConfiguration filterConfiguration) {
            this.filterConfiguration = filterConfiguration;
            return this;
        }

        public final PriorityConfiguration.Builder getPriorityConfiguration() {
            return priorityConfiguration != null ? priorityConfiguration.toBuilder() : null;
        }

        public final void setPriorityConfiguration(PriorityConfiguration.BuilderImpl priorityConfiguration) {
            this.priorityConfiguration = priorityConfiguration != null ? priorityConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder priorityConfiguration(PriorityConfiguration priorityConfiguration) {
            this.priorityConfiguration = priorityConfiguration;
            return this;
        }

        public final String getCustomEventData() {
            return customEventData;
        }

        public final void setCustomEventData(String customEventData) {
            this.customEventData = customEventData;
        }

        @Override
        @Transient
        public final Builder customEventData(String customEventData) {
            this.customEventData = customEventData;
            return this;
        }

        public final String getNotificationTarget() {
            return notificationTarget;
        }

        public final void setNotificationTarget(String notificationTarget) {
            this.notificationTarget = notificationTarget;
        }

        @Override
        @Transient
        public final Builder notificationTarget(String notificationTarget) {
            this.notificationTarget = notificationTarget;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateGameSessionQueueRequest build() {
            return new CreateGameSessionQueueRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
