/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>This data type is used with the GameLift FleetIQ and game server groups.</b>
 * </p>
 * <p>
 * Additional properties, including status, that describe an EC2 instance in a game server group. Instance
 * configurations are set with game server group properties (see <code>DescribeGameServerGroup</code> and with the EC2
 * launch template that was used when creating the game server group.
 * </p>
 * <p>
 * Retrieve game server instances for a game server group by calling <code>DescribeGameServerInstances</code>.
 * </p>
 * <p>
 * <b>Related actions</b>
 * </p>
 * <p>
 * <a>CreateGameServerGroup</a> | <a>ListGameServerGroups</a> | <a>DescribeGameServerGroup</a> |
 * <a>UpdateGameServerGroup</a> | <a>DeleteGameServerGroup</a> | <a>ResumeGameServerGroup</a> |
 * <a>SuspendGameServerGroup</a> | <a>DescribeGameServerInstances</a> | <a
 * href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/reference-awssdk-fleetiq.html">All APIs by task</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GameServerInstance implements SdkPojo, Serializable,
        ToCopyableBuilder<GameServerInstance.Builder, GameServerInstance> {
    private static final SdkField<String> GAME_SERVER_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameServerGroupName").getter(getter(GameServerInstance::gameServerGroupName))
            .setter(setter(Builder::gameServerGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerGroupName").build())
            .build();

    private static final SdkField<String> GAME_SERVER_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameServerGroupArn").getter(getter(GameServerInstance::gameServerGroupArn))
            .setter(setter(Builder::gameServerGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerGroupArn").build())
            .build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceId").getter(getter(GameServerInstance::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> INSTANCE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceStatus").getter(getter(GameServerInstance::instanceStatusAsString))
            .setter(setter(Builder::instanceStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GAME_SERVER_GROUP_NAME_FIELD,
            GAME_SERVER_GROUP_ARN_FIELD, INSTANCE_ID_FIELD, INSTANCE_STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String gameServerGroupName;

    private final String gameServerGroupArn;

    private final String instanceId;

    private final String instanceStatus;

    private GameServerInstance(BuilderImpl builder) {
        this.gameServerGroupName = builder.gameServerGroupName;
        this.gameServerGroupArn = builder.gameServerGroupArn;
        this.instanceId = builder.instanceId;
        this.instanceStatus = builder.instanceStatus;
    }

    /**
     * <p>
     * A developer-defined identifier for the game server group that includes the game server instance. The name is
     * unique for each Region in each AWS account.
     * </p>
     * 
     * @return A developer-defined identifier for the game server group that includes the game server instance. The name
     *         is unique for each Region in each AWS account.
     */
    public final String gameServerGroupName() {
        return gameServerGroupName;
    }

    /**
     * <p>
     * A generated unique identifier for the game server group that includes the game server instance.
     * </p>
     * 
     * @return A generated unique identifier for the game server group that includes the game server instance.
     */
    public final String gameServerGroupArn() {
        return gameServerGroupArn;
    }

    /**
     * <p>
     * The unique identifier for the instance where the game server is running. This ID is available in the instance
     * metadata. EC2 instance IDs use a 17-character format, for example: <code>i-1234567890abcdef0</code>.
     * </p>
     * 
     * @return The unique identifier for the instance where the game server is running. This ID is available in the
     *         instance metadata. EC2 instance IDs use a 17-character format, for example:
     *         <code>i-1234567890abcdef0</code>.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * Current status of the game server instance.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The instance is viable for hosting game servers.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DRAINING</b> -- The instance is not viable for hosting game servers. Existing game servers are in the process
     * of ending, and new game servers are not started on this instance unless no other resources are available. When
     * the instance is put in DRAINING, a new instance is started up to replace it. Once the instance has no UTILIZED
     * game servers, it will be terminated in favor of the new instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>SPOT_TERMINATING</b> -- The instance is in the process of shutting down due to a Spot instance interruption.
     * No new game servers are started on this instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceStatus}
     * will return {@link GameServerInstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceStatusAsString}.
     * </p>
     * 
     * @return Current status of the game server instance. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The instance is viable for hosting game servers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DRAINING</b> -- The instance is not viable for hosting game servers. Existing game servers are in the
     *         process of ending, and new game servers are not started on this instance unless no other resources are
     *         available. When the instance is put in DRAINING, a new instance is started up to replace it. Once the
     *         instance has no UTILIZED game servers, it will be terminated in favor of the new instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>SPOT_TERMINATING</b> -- The instance is in the process of shutting down due to a Spot instance
     *         interruption. No new game servers are started on this instance.
     *         </p>
     *         </li>
     * @see GameServerInstanceStatus
     */
    public final GameServerInstanceStatus instanceStatus() {
        return GameServerInstanceStatus.fromValue(instanceStatus);
    }

    /**
     * <p>
     * Current status of the game server instance.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The instance is viable for hosting game servers.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DRAINING</b> -- The instance is not viable for hosting game servers. Existing game servers are in the process
     * of ending, and new game servers are not started on this instance unless no other resources are available. When
     * the instance is put in DRAINING, a new instance is started up to replace it. Once the instance has no UTILIZED
     * game servers, it will be terminated in favor of the new instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>SPOT_TERMINATING</b> -- The instance is in the process of shutting down due to a Spot instance interruption.
     * No new game servers are started on this instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceStatus}
     * will return {@link GameServerInstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceStatusAsString}.
     * </p>
     * 
     * @return Current status of the game server instance. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The instance is viable for hosting game servers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DRAINING</b> -- The instance is not viable for hosting game servers. Existing game servers are in the
     *         process of ending, and new game servers are not started on this instance unless no other resources are
     *         available. When the instance is put in DRAINING, a new instance is started up to replace it. Once the
     *         instance has no UTILIZED game servers, it will be terminated in favor of the new instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>SPOT_TERMINATING</b> -- The instance is in the process of shutting down due to a Spot instance
     *         interruption. No new game servers are started on this instance.
     *         </p>
     *         </li>
     * @see GameServerInstanceStatus
     */
    public final String instanceStatusAsString() {
        return instanceStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(gameServerGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceStatusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GameServerInstance)) {
            return false;
        }
        GameServerInstance other = (GameServerInstance) obj;
        return Objects.equals(gameServerGroupName(), other.gameServerGroupName())
                && Objects.equals(gameServerGroupArn(), other.gameServerGroupArn())
                && Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(instanceStatusAsString(), other.instanceStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GameServerInstance").add("GameServerGroupName", gameServerGroupName())
                .add("GameServerGroupArn", gameServerGroupArn()).add("InstanceId", instanceId())
                .add("InstanceStatus", instanceStatusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "GameServerGroupName":
            return Optional.ofNullable(clazz.cast(gameServerGroupName()));
        case "GameServerGroupArn":
            return Optional.ofNullable(clazz.cast(gameServerGroupArn()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "InstanceStatus":
            return Optional.ofNullable(clazz.cast(instanceStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GameServerInstance, T> g) {
        return obj -> g.apply((GameServerInstance) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GameServerInstance> {
        /**
         * <p>
         * A developer-defined identifier for the game server group that includes the game server instance. The name is
         * unique for each Region in each AWS account.
         * </p>
         * 
         * @param gameServerGroupName
         *        A developer-defined identifier for the game server group that includes the game server instance. The
         *        name is unique for each Region in each AWS account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerGroupName(String gameServerGroupName);

        /**
         * <p>
         * A generated unique identifier for the game server group that includes the game server instance.
         * </p>
         * 
         * @param gameServerGroupArn
         *        A generated unique identifier for the game server group that includes the game server instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerGroupArn(String gameServerGroupArn);

        /**
         * <p>
         * The unique identifier for the instance where the game server is running. This ID is available in the instance
         * metadata. EC2 instance IDs use a 17-character format, for example: <code>i-1234567890abcdef0</code>.
         * </p>
         * 
         * @param instanceId
         *        The unique identifier for the instance where the game server is running. This ID is available in the
         *        instance metadata. EC2 instance IDs use a 17-character format, for example:
         *        <code>i-1234567890abcdef0</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * Current status of the game server instance.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The instance is viable for hosting game servers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DRAINING</b> -- The instance is not viable for hosting game servers. Existing game servers are in the
         * process of ending, and new game servers are not started on this instance unless no other resources are
         * available. When the instance is put in DRAINING, a new instance is started up to replace it. Once the
         * instance has no UTILIZED game servers, it will be terminated in favor of the new instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>SPOT_TERMINATING</b> -- The instance is in the process of shutting down due to a Spot instance
         * interruption. No new game servers are started on this instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param instanceStatus
         *        Current status of the game server instance. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The instance is viable for hosting game servers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DRAINING</b> -- The instance is not viable for hosting game servers. Existing game servers are in
         *        the process of ending, and new game servers are not started on this instance unless no other resources
         *        are available. When the instance is put in DRAINING, a new instance is started up to replace it. Once
         *        the instance has no UTILIZED game servers, it will be terminated in favor of the new instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>SPOT_TERMINATING</b> -- The instance is in the process of shutting down due to a Spot instance
         *        interruption. No new game servers are started on this instance.
         *        </p>
         *        </li>
         * @see GameServerInstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerInstanceStatus
         */
        Builder instanceStatus(String instanceStatus);

        /**
         * <p>
         * Current status of the game server instance.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The instance is viable for hosting game servers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DRAINING</b> -- The instance is not viable for hosting game servers. Existing game servers are in the
         * process of ending, and new game servers are not started on this instance unless no other resources are
         * available. When the instance is put in DRAINING, a new instance is started up to replace it. Once the
         * instance has no UTILIZED game servers, it will be terminated in favor of the new instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>SPOT_TERMINATING</b> -- The instance is in the process of shutting down due to a Spot instance
         * interruption. No new game servers are started on this instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param instanceStatus
         *        Current status of the game server instance. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The instance is viable for hosting game servers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DRAINING</b> -- The instance is not viable for hosting game servers. Existing game servers are in
         *        the process of ending, and new game servers are not started on this instance unless no other resources
         *        are available. When the instance is put in DRAINING, a new instance is started up to replace it. Once
         *        the instance has no UTILIZED game servers, it will be terminated in favor of the new instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>SPOT_TERMINATING</b> -- The instance is in the process of shutting down due to a Spot instance
         *        interruption. No new game servers are started on this instance.
         *        </p>
         *        </li>
         * @see GameServerInstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerInstanceStatus
         */
        Builder instanceStatus(GameServerInstanceStatus instanceStatus);
    }

    static final class BuilderImpl implements Builder {
        private String gameServerGroupName;

        private String gameServerGroupArn;

        private String instanceId;

        private String instanceStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(GameServerInstance model) {
            gameServerGroupName(model.gameServerGroupName);
            gameServerGroupArn(model.gameServerGroupArn);
            instanceId(model.instanceId);
            instanceStatus(model.instanceStatus);
        }

        public final String getGameServerGroupName() {
            return gameServerGroupName;
        }

        public final void setGameServerGroupName(String gameServerGroupName) {
            this.gameServerGroupName = gameServerGroupName;
        }

        @Override
        @Transient
        public final Builder gameServerGroupName(String gameServerGroupName) {
            this.gameServerGroupName = gameServerGroupName;
            return this;
        }

        public final String getGameServerGroupArn() {
            return gameServerGroupArn;
        }

        public final void setGameServerGroupArn(String gameServerGroupArn) {
            this.gameServerGroupArn = gameServerGroupArn;
        }

        @Override
        @Transient
        public final Builder gameServerGroupArn(String gameServerGroupArn) {
            this.gameServerGroupArn = gameServerGroupArn;
            return this;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        @Transient
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final String getInstanceStatus() {
            return instanceStatus;
        }

        public final void setInstanceStatus(String instanceStatus) {
            this.instanceStatus = instanceStatus;
        }

        @Override
        @Transient
        public final Builder instanceStatus(String instanceStatus) {
            this.instanceStatus = instanceStatus;
            return this;
        }

        @Override
        @Transient
        public final Builder instanceStatus(GameServerInstanceStatus instanceStatus) {
            this.instanceStatus(instanceStatus == null ? null : instanceStatus.toString());
            return this;
        }

        @Override
        public GameServerInstance build() {
            return new GameServerInstance(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
