/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Properties describing a game session.
 * </p>
 * <p>
 * A game session in ACTIVE status can host players. When a game session ends, its status is set to
 * <code>TERMINATED</code>.
 * </p>
 * <p>
 * Once the session ends, the game session object is retained for 30 days. This means you can reuse idempotency token
 * values after this time. Game session logs are retained for 14 days.
 * </p>
 * <p>
 * <b>Related actions</b>
 * </p>
 * <p>
 * <a>CreateGameSession</a> | <a>DescribeGameSessions</a> | <a>DescribeGameSessionDetails</a> |
 * <a>SearchGameSessions</a> | <a>UpdateGameSession</a> | <a>GetGameSessionLogUrl</a> | <a>StartGameSessionPlacement</a>
 * | <a>DescribeGameSessionPlacement</a> | <a>StopGameSessionPlacement</a> | <a href=
 * "https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-awssdk.html#reference-awssdk-resources-fleets"
 * >All APIs by task</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GameSession implements SdkPojo, Serializable, ToCopyableBuilder<GameSession.Builder, GameSession> {
    private static final SdkField<String> GAME_SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameSessionId").getter(getter(GameSession::gameSessionId)).setter(setter(Builder::gameSessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(GameSession::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> FLEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetId").getter(getter(GameSession::fleetId)).setter(setter(Builder::fleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetId").build()).build();

    private static final SdkField<String> FLEET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetArn").getter(getter(GameSession::fleetArn)).setter(setter(Builder::fleetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetArn").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(GameSession::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> TERMINATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("TerminationTime").getter(getter(GameSession::terminationTime)).setter(setter(Builder::terminationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TerminationTime").build()).build();

    private static final SdkField<Integer> CURRENT_PLAYER_SESSION_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("CurrentPlayerSessionCount")
            .getter(getter(GameSession::currentPlayerSessionCount)).setter(setter(Builder::currentPlayerSessionCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrentPlayerSessionCount").build())
            .build();

    private static final SdkField<Integer> MAXIMUM_PLAYER_SESSION_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("MaximumPlayerSessionCount")
            .getter(getter(GameSession::maximumPlayerSessionCount)).setter(setter(Builder::maximumPlayerSessionCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumPlayerSessionCount").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(GameSession::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusReason").getter(getter(GameSession::statusReasonAsString)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final SdkField<List<GameProperty>> GAME_PROPERTIES_FIELD = SdkField
            .<List<GameProperty>> builder(MarshallingType.LIST)
            .memberName("GameProperties")
            .getter(getter(GameSession::gameProperties))
            .setter(setter(Builder::gameProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameProperties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GameProperty> builder(MarshallingType.SDK_POJO)
                                            .constructor(GameProperty::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddress").getter(getter(GameSession::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddress").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DnsName").getter(getter(GameSession::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(GameSession::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> PLAYER_SESSION_CREATION_POLICY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PlayerSessionCreationPolicy")
            .getter(getter(GameSession::playerSessionCreationPolicyAsString))
            .setter(setter(Builder::playerSessionCreationPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlayerSessionCreationPolicy")
                    .build()).build();

    private static final SdkField<String> CREATOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CreatorId").getter(getter(GameSession::creatorId)).setter(setter(Builder::creatorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatorId").build()).build();

    private static final SdkField<String> GAME_SESSION_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameSessionData").getter(getter(GameSession::gameSessionData)).setter(setter(Builder::gameSessionData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionData").build()).build();

    private static final SdkField<String> MATCHMAKER_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MatchmakerData").getter(getter(GameSession::matchmakerData)).setter(setter(Builder::matchmakerData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchmakerData").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Location").getter(getter(GameSession::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Location").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GAME_SESSION_ID_FIELD,
            NAME_FIELD, FLEET_ID_FIELD, FLEET_ARN_FIELD, CREATION_TIME_FIELD, TERMINATION_TIME_FIELD,
            CURRENT_PLAYER_SESSION_COUNT_FIELD, MAXIMUM_PLAYER_SESSION_COUNT_FIELD, STATUS_FIELD, STATUS_REASON_FIELD,
            GAME_PROPERTIES_FIELD, IP_ADDRESS_FIELD, DNS_NAME_FIELD, PORT_FIELD, PLAYER_SESSION_CREATION_POLICY_FIELD,
            CREATOR_ID_FIELD, GAME_SESSION_DATA_FIELD, MATCHMAKER_DATA_FIELD, LOCATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String gameSessionId;

    private final String name;

    private final String fleetId;

    private final String fleetArn;

    private final Instant creationTime;

    private final Instant terminationTime;

    private final Integer currentPlayerSessionCount;

    private final Integer maximumPlayerSessionCount;

    private final String status;

    private final String statusReason;

    private final List<GameProperty> gameProperties;

    private final String ipAddress;

    private final String dnsName;

    private final Integer port;

    private final String playerSessionCreationPolicy;

    private final String creatorId;

    private final String gameSessionData;

    private final String matchmakerData;

    private final String location;

    private GameSession(BuilderImpl builder) {
        this.gameSessionId = builder.gameSessionId;
        this.name = builder.name;
        this.fleetId = builder.fleetId;
        this.fleetArn = builder.fleetArn;
        this.creationTime = builder.creationTime;
        this.terminationTime = builder.terminationTime;
        this.currentPlayerSessionCount = builder.currentPlayerSessionCount;
        this.maximumPlayerSessionCount = builder.maximumPlayerSessionCount;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.gameProperties = builder.gameProperties;
        this.ipAddress = builder.ipAddress;
        this.dnsName = builder.dnsName;
        this.port = builder.port;
        this.playerSessionCreationPolicy = builder.playerSessionCreationPolicy;
        this.creatorId = builder.creatorId;
        this.gameSessionData = builder.gameSessionData;
        this.matchmakerData = builder.matchmakerData;
        this.location = builder.location;
    }

    /**
     * <p>
     * A unique identifier for the game session. A game session ARN has the following format:
     * <code>arn:aws:gamelift:&lt;region&gt;::gamesession/&lt;fleet ID&gt;/&lt;custom ID string or idempotency token&gt;</code>
     * .
     * </p>
     * 
     * @return A unique identifier for the game session. A game session ARN has the following format:
     *         <code>arn:aws:gamelift:&lt;region&gt;::gamesession/&lt;fleet ID&gt;/&lt;custom ID string or idempotency token&gt;</code>
     *         .
     */
    public final String gameSessionId() {
        return gameSessionId;
    }

    /**
     * <p>
     * A descriptive label that is associated with a game session. Session names do not need to be unique.
     * </p>
     * 
     * @return A descriptive label that is associated with a game session. Session names do not need to be unique.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A unique identifier for the fleet that the game session is running on.
     * </p>
     * 
     * @return A unique identifier for the fleet that the game session is running on.
     */
    public final String fleetId() {
        return fleetId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * associated with the GameLift fleet that this game session is running on.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
     *         GameLift fleet that this game session is running on.
     */
    public final String fleetArn() {
        return fleetArn;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was terminated. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was terminated. Format is a number expressed in Unix time
     *         as milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant terminationTime() {
        return terminationTime;
    }

    /**
     * <p>
     * Number of players currently in the game session.
     * </p>
     * 
     * @return Number of players currently in the game session.
     */
    public final Integer currentPlayerSessionCount() {
        return currentPlayerSessionCount;
    }

    /**
     * <p>
     * The maximum number of players that can be connected simultaneously to the game session.
     * </p>
     * 
     * @return The maximum number of players that can be connected simultaneously to the game session.
     */
    public final Integer maximumPlayerSessionCount() {
        return maximumPlayerSessionCount;
    }

    /**
     * <p>
     * Current status of the game session. A game session must have an <code>ACTIVE</code> status to have player
     * sessions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GameSessionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the game session. A game session must have an <code>ACTIVE</code> status to have player
     *         sessions.
     * @see GameSessionStatus
     */
    public final GameSessionStatus status() {
        return GameSessionStatus.fromValue(status);
    }

    /**
     * <p>
     * Current status of the game session. A game session must have an <code>ACTIVE</code> status to have player
     * sessions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GameSessionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the game session. A game session must have an <code>ACTIVE</code> status to have player
     *         sessions.
     * @see GameSessionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Provides additional information about game session status. <code>INTERRUPTED</code> indicates that the game
     * session was hosted on a spot instance that was reclaimed, causing the active game session to be terminated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusReason} will
     * return {@link GameSessionStatusReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusReasonAsString}.
     * </p>
     * 
     * @return Provides additional information about game session status. <code>INTERRUPTED</code> indicates that the
     *         game session was hosted on a spot instance that was reclaimed, causing the active game session to be
     *         terminated.
     * @see GameSessionStatusReason
     */
    public final GameSessionStatusReason statusReason() {
        return GameSessionStatusReason.fromValue(statusReason);
    }

    /**
     * <p>
     * Provides additional information about game session status. <code>INTERRUPTED</code> indicates that the game
     * session was hosted on a spot instance that was reclaimed, causing the active game session to be terminated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusReason} will
     * return {@link GameSessionStatusReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusReasonAsString}.
     * </p>
     * 
     * @return Provides additional information about game session status. <code>INTERRUPTED</code> indicates that the
     *         game session was hosted on a spot instance that was reclaimed, causing the active game session to be
     *         terminated.
     * @see GameSessionStatusReason
     */
    public final String statusReasonAsString() {
        return statusReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the GameProperties property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasGameProperties() {
        return gameProperties != null && !(gameProperties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a
     * game server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     * >Start a Game Session</a>). You can search for active game sessions based on this custom data with
     * <a>SearchGameSessions</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGameProperties} method.
     * </p>
     * 
     * @return A set of custom properties for a game session, formatted as key:value pairs. These properties are passed
     *         to a game server process in the <a>GameSession</a> object with a request to start a new game session (see
     *         <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     *         >Start a Game Session</a>). You can search for active game sessions based on this custom data with
     *         <a>SearchGameSessions</a>.
     */
    public final List<GameProperty> gameProperties() {
        return gameProperties;
    }

    /**
     * <p>
     * The IP address of the game session. To connect to a GameLift game server, an app needs both the IP address and
     * port number.
     * </p>
     * 
     * @return The IP address of the game session. To connect to a GameLift game server, an app needs both the IP
     *         address and port number.
     */
    public final String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * The DNS identifier assigned to the instance that is running the game session. Values have the following format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     * >Amazon EC2 Instance IP Addressing</a>.)
     * </p>
     * </li>
     * </ul>
     * <p>
     * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the IP
     * address.
     * </p>
     * 
     * @return The DNS identifier assigned to the instance that is running the game session. Values have the following
     *         format:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     *         >Amazon EC2 Instance IP Addressing</a>.)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not
     *         the IP address.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * The port number for the game session. To connect to a GameLift game server, an app needs both the IP address and
     * port number.
     * </p>
     * 
     * @return The port number for the game session. To connect to a GameLift game server, an app needs both the IP
     *         address and port number.
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * Indicates whether or not the game session is accepting new players.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #playerSessionCreationPolicy} will return {@link PlayerSessionCreationPolicy#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #playerSessionCreationPolicyAsString}.
     * </p>
     * 
     * @return Indicates whether or not the game session is accepting new players.
     * @see PlayerSessionCreationPolicy
     */
    public final PlayerSessionCreationPolicy playerSessionCreationPolicy() {
        return PlayerSessionCreationPolicy.fromValue(playerSessionCreationPolicy);
    }

    /**
     * <p>
     * Indicates whether or not the game session is accepting new players.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #playerSessionCreationPolicy} will return {@link PlayerSessionCreationPolicy#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #playerSessionCreationPolicyAsString}.
     * </p>
     * 
     * @return Indicates whether or not the game session is accepting new players.
     * @see PlayerSessionCreationPolicy
     */
    public final String playerSessionCreationPolicyAsString() {
        return playerSessionCreationPolicy;
    }

    /**
     * <p>
     * A unique identifier for a player. This ID is used to enforce a resource protection policy (if one exists), that
     * limits the number of game sessions a player can create.
     * </p>
     * 
     * @return A unique identifier for a player. This ID is used to enforce a resource protection policy (if one
     *         exists), that limits the number of game sessions a player can create.
     */
    public final String creatorId() {
        return creatorId;
    }

    /**
     * <p>
     * A set of custom game session properties, formatted as a single string value. This data is passed to a game server
     * process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     * >Start a Game Session</a>).
     * </p>
     * 
     * @return A set of custom game session properties, formatted as a single string value. This data is passed to a
     *         game server process in the <a>GameSession</a> object with a request to start a new game session (see <a
     *         href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     *         >Start a Game Session</a>).
     */
    public final String gameSessionData() {
        return gameSessionData;
    }

    /**
     * <p>
     * Information about the matchmaking process that was used to create the game session. It is in JSON syntax,
     * formatted as a string. In addition the matchmaking configuration used, it contains data on all players assigned
     * to the match, including player attributes and team assignments. For more details on matchmaker data, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-server.html#match-server-data">Match
     * Data</a>. Matchmaker data is useful when requesting match backfills, and is updated whenever new players are
     * added during a successful backfill (see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchBackfill.html"
     * >StartMatchBackfill</a>).
     * </p>
     * 
     * @return Information about the matchmaking process that was used to create the game session. It is in JSON syntax,
     *         formatted as a string. In addition the matchmaking configuration used, it contains data on all players
     *         assigned to the match, including player attributes and team assignments. For more details on matchmaker
     *         data, see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-server.html#match-server-data"
     *         >Match Data</a>. Matchmaker data is useful when requesting match backfills, and is updated whenever new
     *         players are added during a successful backfill (see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchBackfill.html"
     *         >StartMatchBackfill</a>).
     */
    public final String matchmakerData() {
        return matchmakerData;
    }

    /**
     * <p>
     * The fleet location where the game session is running. This value might specify the fleet's home Region or a
     * remote location. Location is expressed as an Amazon Web Services Region code such as <code>us-west-2</code>.
     * </p>
     * 
     * @return The fleet location where the game session is running. This value might specify the fleet's home Region or
     *         a remote location. Location is expressed as an Amazon Web Services Region code such as
     *         <code>us-west-2</code>.
     */
    public final String location() {
        return location;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(fleetId());
        hashCode = 31 * hashCode + Objects.hashCode(fleetArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(terminationTime());
        hashCode = 31 * hashCode + Objects.hashCode(currentPlayerSessionCount());
        hashCode = 31 * hashCode + Objects.hashCode(maximumPlayerSessionCount());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReasonAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasGameProperties() ? gameProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(playerSessionCreationPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creatorId());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionData());
        hashCode = 31 * hashCode + Objects.hashCode(matchmakerData());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GameSession)) {
            return false;
        }
        GameSession other = (GameSession) obj;
        return Objects.equals(gameSessionId(), other.gameSessionId()) && Objects.equals(name(), other.name())
                && Objects.equals(fleetId(), other.fleetId()) && Objects.equals(fleetArn(), other.fleetArn())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(terminationTime(), other.terminationTime())
                && Objects.equals(currentPlayerSessionCount(), other.currentPlayerSessionCount())
                && Objects.equals(maximumPlayerSessionCount(), other.maximumPlayerSessionCount())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReasonAsString(), other.statusReasonAsString())
                && hasGameProperties() == other.hasGameProperties() && Objects.equals(gameProperties(), other.gameProperties())
                && Objects.equals(ipAddress(), other.ipAddress()) && Objects.equals(dnsName(), other.dnsName())
                && Objects.equals(port(), other.port())
                && Objects.equals(playerSessionCreationPolicyAsString(), other.playerSessionCreationPolicyAsString())
                && Objects.equals(creatorId(), other.creatorId()) && Objects.equals(gameSessionData(), other.gameSessionData())
                && Objects.equals(matchmakerData(), other.matchmakerData()) && Objects.equals(location(), other.location());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GameSession").add("GameSessionId", gameSessionId()).add("Name", name())
                .add("FleetId", fleetId()).add("FleetArn", fleetArn()).add("CreationTime", creationTime())
                .add("TerminationTime", terminationTime()).add("CurrentPlayerSessionCount", currentPlayerSessionCount())
                .add("MaximumPlayerSessionCount", maximumPlayerSessionCount()).add("Status", statusAsString())
                .add("StatusReason", statusReasonAsString()).add("GameProperties", hasGameProperties() ? gameProperties() : null)
                .add("IpAddress", ipAddress()).add("DnsName", dnsName()).add("Port", port())
                .add("PlayerSessionCreationPolicy", playerSessionCreationPolicyAsString()).add("CreatorId", creatorId())
                .add("GameSessionData", gameSessionData()).add("MatchmakerData", matchmakerData()).add("Location", location())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "GameSessionId":
            return Optional.ofNullable(clazz.cast(gameSessionId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "FleetId":
            return Optional.ofNullable(clazz.cast(fleetId()));
        case "FleetArn":
            return Optional.ofNullable(clazz.cast(fleetArn()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "TerminationTime":
            return Optional.ofNullable(clazz.cast(terminationTime()));
        case "CurrentPlayerSessionCount":
            return Optional.ofNullable(clazz.cast(currentPlayerSessionCount()));
        case "MaximumPlayerSessionCount":
            return Optional.ofNullable(clazz.cast(maximumPlayerSessionCount()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReasonAsString()));
        case "GameProperties":
            return Optional.ofNullable(clazz.cast(gameProperties()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "PlayerSessionCreationPolicy":
            return Optional.ofNullable(clazz.cast(playerSessionCreationPolicyAsString()));
        case "CreatorId":
            return Optional.ofNullable(clazz.cast(creatorId()));
        case "GameSessionData":
            return Optional.ofNullable(clazz.cast(gameSessionData()));
        case "MatchmakerData":
            return Optional.ofNullable(clazz.cast(matchmakerData()));
        case "Location":
            return Optional.ofNullable(clazz.cast(location()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GameSession, T> g) {
        return obj -> g.apply((GameSession) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GameSession> {
        /**
         * <p>
         * A unique identifier for the game session. A game session ARN has the following format:
         * <code>arn:aws:gamelift:&lt;region&gt;::gamesession/&lt;fleet ID&gt;/&lt;custom ID string or idempotency token&gt;</code>
         * .
         * </p>
         * 
         * @param gameSessionId
         *        A unique identifier for the game session. A game session ARN has the following format:
         *        <code>arn:aws:gamelift:&lt;region&gt;::gamesession/&lt;fleet ID&gt;/&lt;custom ID string or idempotency token&gt;</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionId(String gameSessionId);

        /**
         * <p>
         * A descriptive label that is associated with a game session. Session names do not need to be unique.
         * </p>
         * 
         * @param name
         *        A descriptive label that is associated with a game session. Session names do not need to be unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A unique identifier for the fleet that the game session is running on.
         * </p>
         * 
         * @param fleetId
         *        A unique identifier for the fleet that the game session is running on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetId(String fleetId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         * GameLift fleet that this game session is running on.
         * </p>
         * 
         * @param fleetArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         *        GameLift fleet that this game session is running on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetArn(String fleetArn);

        /**
         * <p>
         * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param creationTime
         *        A time stamp indicating when this data object was created. Format is a number expressed in Unix time
         *        as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A time stamp indicating when this data object was terminated. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param terminationTime
         *        A time stamp indicating when this data object was terminated. Format is a number expressed in Unix
         *        time as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminationTime(Instant terminationTime);

        /**
         * <p>
         * Number of players currently in the game session.
         * </p>
         * 
         * @param currentPlayerSessionCount
         *        Number of players currently in the game session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentPlayerSessionCount(Integer currentPlayerSessionCount);

        /**
         * <p>
         * The maximum number of players that can be connected simultaneously to the game session.
         * </p>
         * 
         * @param maximumPlayerSessionCount
         *        The maximum number of players that can be connected simultaneously to the game session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumPlayerSessionCount(Integer maximumPlayerSessionCount);

        /**
         * <p>
         * Current status of the game session. A game session must have an <code>ACTIVE</code> status to have player
         * sessions.
         * </p>
         * 
         * @param status
         *        Current status of the game session. A game session must have an <code>ACTIVE</code> status to have
         *        player sessions.
         * @see GameSessionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameSessionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the game session. A game session must have an <code>ACTIVE</code> status to have player
         * sessions.
         * </p>
         * 
         * @param status
         *        Current status of the game session. A game session must have an <code>ACTIVE</code> status to have
         *        player sessions.
         * @see GameSessionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameSessionStatus
         */
        Builder status(GameSessionStatus status);

        /**
         * <p>
         * Provides additional information about game session status. <code>INTERRUPTED</code> indicates that the game
         * session was hosted on a spot instance that was reclaimed, causing the active game session to be terminated.
         * </p>
         * 
         * @param statusReason
         *        Provides additional information about game session status. <code>INTERRUPTED</code> indicates that the
         *        game session was hosted on a spot instance that was reclaimed, causing the active game session to be
         *        terminated.
         * @see GameSessionStatusReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameSessionStatusReason
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * Provides additional information about game session status. <code>INTERRUPTED</code> indicates that the game
         * session was hosted on a spot instance that was reclaimed, causing the active game session to be terminated.
         * </p>
         * 
         * @param statusReason
         *        Provides additional information about game session status. <code>INTERRUPTED</code> indicates that the
         *        game session was hosted on a spot instance that was reclaimed, causing the active game session to be
         *        terminated.
         * @see GameSessionStatusReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameSessionStatusReason
         */
        Builder statusReason(GameSessionStatusReason statusReason);

        /**
         * <p>
         * A set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a
         * game server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>). You can search for active game sessions based on this custom data with
         * <a>SearchGameSessions</a>.
         * </p>
         * 
         * @param gameProperties
         *        A set of custom properties for a game session, formatted as key:value pairs. These properties are
         *        passed to a game server process in the <a>GameSession</a> object with a request to start a new game
         *        session (see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>). You can search for active game sessions based on this custom data with
         *        <a>SearchGameSessions</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameProperties(Collection<GameProperty> gameProperties);

        /**
         * <p>
         * A set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a
         * game server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>). You can search for active game sessions based on this custom data with
         * <a>SearchGameSessions</a>.
         * </p>
         * 
         * @param gameProperties
         *        A set of custom properties for a game session, formatted as key:value pairs. These properties are
         *        passed to a game server process in the <a>GameSession</a> object with a request to start a new game
         *        session (see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>). You can search for active game sessions based on this custom data with
         *        <a>SearchGameSessions</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameProperties(GameProperty... gameProperties);

        /**
         * <p>
         * A set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a
         * game server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>). You can search for active game sessions based on this custom data with
         * <a>SearchGameSessions</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.GameProperty.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.gamelift.model.GameProperty#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.GameProperty.Builder#build()} is called immediately and
         * its result is passed to {@link #gameProperties(List<GameProperty>)}.
         * 
         * @param gameProperties
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.GameProperty.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gameProperties(java.util.Collection<GameProperty>)
         */
        Builder gameProperties(Consumer<GameProperty.Builder>... gameProperties);

        /**
         * <p>
         * The IP address of the game session. To connect to a GameLift game server, an app needs both the IP address
         * and port number.
         * </p>
         * 
         * @param ipAddress
         *        The IP address of the game session. To connect to a GameLift game server, an app needs both the IP
         *        address and port number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * The DNS identifier assigned to the instance that is running the game session. Values have the following
         * format:
         * </p>
         * <ul>
         * <li>
         * <p>
         * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         * >Amazon EC2 Instance IP Addressing</a>.)
         * </p>
         * </li>
         * </ul>
         * <p>
         * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the
         * IP address.
         * </p>
         * 
         * @param dnsName
         *        The DNS identifier assigned to the instance that is running the game session. Values have the
         *        following format:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        TLS-enabled fleets:
         *        <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a
         *        href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         *        >Amazon EC2 Instance IP Addressing</a>.)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name,
         *        not the IP address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * The port number for the game session. To connect to a GameLift game server, an app needs both the IP address
         * and port number.
         * </p>
         * 
         * @param port
         *        The port number for the game session. To connect to a GameLift game server, an app needs both the IP
         *        address and port number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * Indicates whether or not the game session is accepting new players.
         * </p>
         * 
         * @param playerSessionCreationPolicy
         *        Indicates whether or not the game session is accepting new players.
         * @see PlayerSessionCreationPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlayerSessionCreationPolicy
         */
        Builder playerSessionCreationPolicy(String playerSessionCreationPolicy);

        /**
         * <p>
         * Indicates whether or not the game session is accepting new players.
         * </p>
         * 
         * @param playerSessionCreationPolicy
         *        Indicates whether or not the game session is accepting new players.
         * @see PlayerSessionCreationPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlayerSessionCreationPolicy
         */
        Builder playerSessionCreationPolicy(PlayerSessionCreationPolicy playerSessionCreationPolicy);

        /**
         * <p>
         * A unique identifier for a player. This ID is used to enforce a resource protection policy (if one exists),
         * that limits the number of game sessions a player can create.
         * </p>
         * 
         * @param creatorId
         *        A unique identifier for a player. This ID is used to enforce a resource protection policy (if one
         *        exists), that limits the number of game sessions a player can create.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creatorId(String creatorId);

        /**
         * <p>
         * A set of custom game session properties, formatted as a single string value. This data is passed to a game
         * server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>).
         * </p>
         * 
         * @param gameSessionData
         *        A set of custom game session properties, formatted as a single string value. This data is passed to a
         *        game server process in the <a>GameSession</a> object with a request to start a new game session (see
         *        <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionData(String gameSessionData);

        /**
         * <p>
         * Information about the matchmaking process that was used to create the game session. It is in JSON syntax,
         * formatted as a string. In addition the matchmaking configuration used, it contains data on all players
         * assigned to the match, including player attributes and team assignments. For more details on matchmaker data,
         * see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-server.html#match-server-data">Match
         * Data</a>. Matchmaker data is useful when requesting match backfills, and is updated whenever new players are
         * added during a successful backfill (see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchBackfill.html"
         * >StartMatchBackfill</a>).
         * </p>
         * 
         * @param matchmakerData
         *        Information about the matchmaking process that was used to create the game session. It is in JSON
         *        syntax, formatted as a string. In addition the matchmaking configuration used, it contains data on all
         *        players assigned to the match, including player attributes and team assignments. For more details on
         *        matchmaker data, see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-server.html#match-server-data">Match
         *        Data</a>. Matchmaker data is useful when requesting match backfills, and is updated whenever new
         *        players are added during a successful backfill (see <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StartMatchBackfill.html"
         *        >StartMatchBackfill</a>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchmakerData(String matchmakerData);

        /**
         * <p>
         * The fleet location where the game session is running. This value might specify the fleet's home Region or a
         * remote location. Location is expressed as an Amazon Web Services Region code such as <code>us-west-2</code>.
         * </p>
         * 
         * @param location
         *        The fleet location where the game session is running. This value might specify the fleet's home Region
         *        or a remote location. Location is expressed as an Amazon Web Services Region code such as
         *        <code>us-west-2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);
    }

    static final class BuilderImpl implements Builder {
        private String gameSessionId;

        private String name;

        private String fleetId;

        private String fleetArn;

        private Instant creationTime;

        private Instant terminationTime;

        private Integer currentPlayerSessionCount;

        private Integer maximumPlayerSessionCount;

        private String status;

        private String statusReason;

        private List<GameProperty> gameProperties = DefaultSdkAutoConstructList.getInstance();

        private String ipAddress;

        private String dnsName;

        private Integer port;

        private String playerSessionCreationPolicy;

        private String creatorId;

        private String gameSessionData;

        private String matchmakerData;

        private String location;

        private BuilderImpl() {
        }

        private BuilderImpl(GameSession model) {
            gameSessionId(model.gameSessionId);
            name(model.name);
            fleetId(model.fleetId);
            fleetArn(model.fleetArn);
            creationTime(model.creationTime);
            terminationTime(model.terminationTime);
            currentPlayerSessionCount(model.currentPlayerSessionCount);
            maximumPlayerSessionCount(model.maximumPlayerSessionCount);
            status(model.status);
            statusReason(model.statusReason);
            gameProperties(model.gameProperties);
            ipAddress(model.ipAddress);
            dnsName(model.dnsName);
            port(model.port);
            playerSessionCreationPolicy(model.playerSessionCreationPolicy);
            creatorId(model.creatorId);
            gameSessionData(model.gameSessionData);
            matchmakerData(model.matchmakerData);
            location(model.location);
        }

        public final String getGameSessionId() {
            return gameSessionId;
        }

        public final void setGameSessionId(String gameSessionId) {
            this.gameSessionId = gameSessionId;
        }

        @Override
        public final Builder gameSessionId(String gameSessionId) {
            this.gameSessionId = gameSessionId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getFleetId() {
            return fleetId;
        }

        public final void setFleetId(String fleetId) {
            this.fleetId = fleetId;
        }

        @Override
        public final Builder fleetId(String fleetId) {
            this.fleetId = fleetId;
            return this;
        }

        public final String getFleetArn() {
            return fleetArn;
        }

        public final void setFleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
        }

        @Override
        public final Builder fleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getTerminationTime() {
            return terminationTime;
        }

        public final void setTerminationTime(Instant terminationTime) {
            this.terminationTime = terminationTime;
        }

        @Override
        public final Builder terminationTime(Instant terminationTime) {
            this.terminationTime = terminationTime;
            return this;
        }

        public final Integer getCurrentPlayerSessionCount() {
            return currentPlayerSessionCount;
        }

        public final void setCurrentPlayerSessionCount(Integer currentPlayerSessionCount) {
            this.currentPlayerSessionCount = currentPlayerSessionCount;
        }

        @Override
        public final Builder currentPlayerSessionCount(Integer currentPlayerSessionCount) {
            this.currentPlayerSessionCount = currentPlayerSessionCount;
            return this;
        }

        public final Integer getMaximumPlayerSessionCount() {
            return maximumPlayerSessionCount;
        }

        public final void setMaximumPlayerSessionCount(Integer maximumPlayerSessionCount) {
            this.maximumPlayerSessionCount = maximumPlayerSessionCount;
        }

        @Override
        public final Builder maximumPlayerSessionCount(Integer maximumPlayerSessionCount) {
            this.maximumPlayerSessionCount = maximumPlayerSessionCount;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(GameSessionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        @Override
        public final Builder statusReason(GameSessionStatusReason statusReason) {
            this.statusReason(statusReason == null ? null : statusReason.toString());
            return this;
        }

        public final List<GameProperty.Builder> getGameProperties() {
            List<GameProperty.Builder> result = GamePropertyListCopier.copyToBuilder(this.gameProperties);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setGameProperties(Collection<GameProperty.BuilderImpl> gameProperties) {
            this.gameProperties = GamePropertyListCopier.copyFromBuilder(gameProperties);
        }

        @Override
        public final Builder gameProperties(Collection<GameProperty> gameProperties) {
            this.gameProperties = GamePropertyListCopier.copy(gameProperties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameProperties(GameProperty... gameProperties) {
            gameProperties(Arrays.asList(gameProperties));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameProperties(Consumer<GameProperty.Builder>... gameProperties) {
            gameProperties(Stream.of(gameProperties).map(c -> GameProperty.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final String getDnsName() {
            return dnsName;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final String getPlayerSessionCreationPolicy() {
            return playerSessionCreationPolicy;
        }

        public final void setPlayerSessionCreationPolicy(String playerSessionCreationPolicy) {
            this.playerSessionCreationPolicy = playerSessionCreationPolicy;
        }

        @Override
        public final Builder playerSessionCreationPolicy(String playerSessionCreationPolicy) {
            this.playerSessionCreationPolicy = playerSessionCreationPolicy;
            return this;
        }

        @Override
        public final Builder playerSessionCreationPolicy(PlayerSessionCreationPolicy playerSessionCreationPolicy) {
            this.playerSessionCreationPolicy(playerSessionCreationPolicy == null ? null : playerSessionCreationPolicy.toString());
            return this;
        }

        public final String getCreatorId() {
            return creatorId;
        }

        public final void setCreatorId(String creatorId) {
            this.creatorId = creatorId;
        }

        @Override
        public final Builder creatorId(String creatorId) {
            this.creatorId = creatorId;
            return this;
        }

        public final String getGameSessionData() {
            return gameSessionData;
        }

        public final void setGameSessionData(String gameSessionData) {
            this.gameSessionData = gameSessionData;
        }

        @Override
        public final Builder gameSessionData(String gameSessionData) {
            this.gameSessionData = gameSessionData;
            return this;
        }

        public final String getMatchmakerData() {
            return matchmakerData;
        }

        public final void setMatchmakerData(String matchmakerData) {
            this.matchmakerData = matchmakerData;
        }

        @Override
        public final Builder matchmakerData(String matchmakerData) {
            this.matchmakerData = matchmakerData;
            return this;
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        @Override
        public GameSession build() {
            return new GameSession(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
