/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Guidelines for use with FlexMatch to match players into games. All matchmaking requests must specify a matchmaking
 * configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MatchmakingConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<MatchmakingConfiguration.Builder, MatchmakingConfiguration> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(MatchmakingConfiguration::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> CONFIGURATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationArn").getter(getter(MatchmakingConfiguration::configurationArn))
            .setter(setter(Builder::configurationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationArn").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(MatchmakingConfiguration::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<String>> GAME_SESSION_QUEUE_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("GameSessionQueueArns")
            .getter(getter(MatchmakingConfiguration::gameSessionQueueArns))
            .setter(setter(Builder::gameSessionQueueArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionQueueArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> REQUEST_TIMEOUT_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RequestTimeoutSeconds").getter(getter(MatchmakingConfiguration::requestTimeoutSeconds))
            .setter(setter(Builder::requestTimeoutSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestTimeoutSeconds").build())
            .build();

    private static final SdkField<Integer> ACCEPTANCE_TIMEOUT_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("AcceptanceTimeoutSeconds").getter(getter(MatchmakingConfiguration::acceptanceTimeoutSeconds))
            .setter(setter(Builder::acceptanceTimeoutSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceptanceTimeoutSeconds").build())
            .build();

    private static final SdkField<Boolean> ACCEPTANCE_REQUIRED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AcceptanceRequired").getter(getter(MatchmakingConfiguration::acceptanceRequired))
            .setter(setter(Builder::acceptanceRequired))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceptanceRequired").build())
            .build();

    private static final SdkField<String> RULE_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleSetName").getter(getter(MatchmakingConfiguration::ruleSetName)).setter(setter(Builder::ruleSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleSetName").build()).build();

    private static final SdkField<String> RULE_SET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleSetArn").getter(getter(MatchmakingConfiguration::ruleSetArn)).setter(setter(Builder::ruleSetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleSetArn").build()).build();

    private static final SdkField<String> NOTIFICATION_TARGET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NotificationTarget").getter(getter(MatchmakingConfiguration::notificationTarget))
            .setter(setter(Builder::notificationTarget))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationTarget").build())
            .build();

    private static final SdkField<Integer> ADDITIONAL_PLAYER_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("AdditionalPlayerCount").getter(getter(MatchmakingConfiguration::additionalPlayerCount))
            .setter(setter(Builder::additionalPlayerCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalPlayerCount").build())
            .build();

    private static final SdkField<String> CUSTOM_EVENT_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomEventData").getter(getter(MatchmakingConfiguration::customEventData))
            .setter(setter(Builder::customEventData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomEventData").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(MatchmakingConfiguration::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<List<GameProperty>> GAME_PROPERTIES_FIELD = SdkField
            .<List<GameProperty>> builder(MarshallingType.LIST)
            .memberName("GameProperties")
            .getter(getter(MatchmakingConfiguration::gameProperties))
            .setter(setter(Builder::gameProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameProperties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GameProperty> builder(MarshallingType.SDK_POJO)
                                            .constructor(GameProperty::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> GAME_SESSION_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameSessionData").getter(getter(MatchmakingConfiguration::gameSessionData))
            .setter(setter(Builder::gameSessionData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionData").build()).build();

    private static final SdkField<String> BACKFILL_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackfillMode").getter(getter(MatchmakingConfiguration::backfillModeAsString))
            .setter(setter(Builder::backfillMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackfillMode").build()).build();

    private static final SdkField<String> FLEX_MATCH_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FlexMatchMode").getter(getter(MatchmakingConfiguration::flexMatchModeAsString))
            .setter(setter(Builder::flexMatchMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FlexMatchMode").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            CONFIGURATION_ARN_FIELD, DESCRIPTION_FIELD, GAME_SESSION_QUEUE_ARNS_FIELD, REQUEST_TIMEOUT_SECONDS_FIELD,
            ACCEPTANCE_TIMEOUT_SECONDS_FIELD, ACCEPTANCE_REQUIRED_FIELD, RULE_SET_NAME_FIELD, RULE_SET_ARN_FIELD,
            NOTIFICATION_TARGET_FIELD, ADDITIONAL_PLAYER_COUNT_FIELD, CUSTOM_EVENT_DATA_FIELD, CREATION_TIME_FIELD,
            GAME_PROPERTIES_FIELD, GAME_SESSION_DATA_FIELD, BACKFILL_MODE_FIELD, FLEX_MATCH_MODE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String configurationArn;

    private final String description;

    private final List<String> gameSessionQueueArns;

    private final Integer requestTimeoutSeconds;

    private final Integer acceptanceTimeoutSeconds;

    private final Boolean acceptanceRequired;

    private final String ruleSetName;

    private final String ruleSetArn;

    private final String notificationTarget;

    private final Integer additionalPlayerCount;

    private final String customEventData;

    private final Instant creationTime;

    private final List<GameProperty> gameProperties;

    private final String gameSessionData;

    private final String backfillMode;

    private final String flexMatchMode;

    private MatchmakingConfiguration(BuilderImpl builder) {
        this.name = builder.name;
        this.configurationArn = builder.configurationArn;
        this.description = builder.description;
        this.gameSessionQueueArns = builder.gameSessionQueueArns;
        this.requestTimeoutSeconds = builder.requestTimeoutSeconds;
        this.acceptanceTimeoutSeconds = builder.acceptanceTimeoutSeconds;
        this.acceptanceRequired = builder.acceptanceRequired;
        this.ruleSetName = builder.ruleSetName;
        this.ruleSetArn = builder.ruleSetArn;
        this.notificationTarget = builder.notificationTarget;
        this.additionalPlayerCount = builder.additionalPlayerCount;
        this.customEventData = builder.customEventData;
        this.creationTime = builder.creationTime;
        this.gameProperties = builder.gameProperties;
        this.gameSessionData = builder.gameSessionData;
        this.backfillMode = builder.backfillMode;
        this.flexMatchMode = builder.flexMatchMode;
    }

    /**
     * <p>
     * A unique identifier for the matchmaking configuration. This name is used to identify the configuration associated
     * with a matchmaking request or ticket.
     * </p>
     * 
     * @return A unique identifier for the matchmaking configuration. This name is used to identify the configuration
     *         associated with a matchmaking request or ticket.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * that is assigned to a Amazon GameLift matchmaking configuration resource and uniquely identifies it. ARNs are
     * unique across all Regions. Format is
     * <code>arn:aws:gamelift:&lt;region&gt;::matchmakingconfiguration/&lt;matchmaking configuration name&gt;</code>. In
     * a Amazon GameLift configuration ARN, the resource ID matches the <i>Name</i> value.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
     *         Amazon GameLift matchmaking configuration resource and uniquely identifies it. ARNs are unique across all
     *         Regions. Format is
     *         <code>arn:aws:gamelift:&lt;region&gt;::matchmakingconfiguration/&lt;matchmaking configuration name&gt;</code>
     *         . In a Amazon GameLift configuration ARN, the resource ID matches the <i>Name</i> value.
     */
    public final String configurationArn() {
        return configurationArn;
    }

    /**
     * <p>
     * A descriptive label that is associated with matchmaking configuration.
     * </p>
     * 
     * @return A descriptive label that is associated with matchmaking configuration.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the GameSessionQueueArns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasGameSessionQueueArns() {
        return gameSessionQueueArns != null && !(gameSessionQueueArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * that is assigned to a Amazon GameLift game session queue resource and uniquely identifies it. ARNs are unique
     * across all Regions. Format is <code>arn:aws:gamelift:&lt;region&gt;::gamesessionqueue/&lt;queue name&gt;</code>.
     * Queues can be located in any Region. Queues are used to start new Amazon GameLift-hosted game sessions for
     * matches that are created with this matchmaking configuration. This property is not set when
     * <code>FlexMatchMode</code> is set to <code>STANDALONE</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGameSessionQueueArns} method.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
     *         Amazon GameLift game session queue resource and uniquely identifies it. ARNs are unique across all
     *         Regions. Format is <code>arn:aws:gamelift:&lt;region&gt;::gamesessionqueue/&lt;queue name&gt;</code>.
     *         Queues can be located in any Region. Queues are used to start new Amazon GameLift-hosted game sessions
     *         for matches that are created with this matchmaking configuration. This property is not set when
     *         <code>FlexMatchMode</code> is set to <code>STANDALONE</code>.
     */
    public final List<String> gameSessionQueueArns() {
        return gameSessionQueueArns;
    }

    /**
     * <p>
     * The maximum duration, in seconds, that a matchmaking ticket can remain in process before timing out. Requests
     * that fail due to timing out can be resubmitted as needed.
     * </p>
     * 
     * @return The maximum duration, in seconds, that a matchmaking ticket can remain in process before timing out.
     *         Requests that fail due to timing out can be resubmitted as needed.
     */
    public final Integer requestTimeoutSeconds() {
        return requestTimeoutSeconds;
    }

    /**
     * <p>
     * The length of time (in seconds) to wait for players to accept a proposed match, if acceptance is required. If any
     * player rejects the match or fails to accept before the timeout, the ticket continues to look for an acceptable
     * match.
     * </p>
     * 
     * @return The length of time (in seconds) to wait for players to accept a proposed match, if acceptance is
     *         required. If any player rejects the match or fails to accept before the timeout, the ticket continues to
     *         look for an acceptable match.
     */
    public final Integer acceptanceTimeoutSeconds() {
        return acceptanceTimeoutSeconds;
    }

    /**
     * <p>
     * A flag that indicates whether a match that was created with this configuration must be accepted by the matched
     * players. To require acceptance, set to TRUE. When this option is enabled, matchmaking tickets use the status
     * <code>REQUIRES_ACCEPTANCE</code> to indicate when a completed potential match is waiting for player acceptance.
     * </p>
     * 
     * @return A flag that indicates whether a match that was created with this configuration must be accepted by the
     *         matched players. To require acceptance, set to TRUE. When this option is enabled, matchmaking tickets use
     *         the status <code>REQUIRES_ACCEPTANCE</code> to indicate when a completed potential match is waiting for
     *         player acceptance.
     */
    public final Boolean acceptanceRequired() {
        return acceptanceRequired;
    }

    /**
     * <p>
     * A unique identifier for the matchmaking rule set to use with this configuration. A matchmaking configuration can
     * only use rule sets that are defined in the same Region.
     * </p>
     * 
     * @return A unique identifier for the matchmaking rule set to use with this configuration. A matchmaking
     *         configuration can only use rule sets that are defined in the same Region.
     */
    public final String ruleSetName() {
        return ruleSetName;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * associated with the GameLift matchmaking rule set resource that this configuration uses.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
     *         GameLift matchmaking rule set resource that this configuration uses.
     */
    public final String ruleSetArn() {
        return ruleSetArn;
    }

    /**
     * <p>
     * An SNS topic ARN that is set up to receive matchmaking notifications.
     * </p>
     * 
     * @return An SNS topic ARN that is set up to receive matchmaking notifications.
     */
    public final String notificationTarget() {
        return notificationTarget;
    }

    /**
     * <p>
     * The number of player slots in a match to keep open for future players. For example, if the configuration's rule
     * set specifies a match for a single 12-person team, and the additional player count is set to 2, only 10 players
     * are selected for the match. This parameter is not used when <code>FlexMatchMode</code> is set to
     * <code>STANDALONE</code>.
     * </p>
     * 
     * @return The number of player slots in a match to keep open for future players. For example, if the
     *         configuration's rule set specifies a match for a single 12-person team, and the additional player count
     *         is set to 2, only 10 players are selected for the match. This parameter is not used when
     *         <code>FlexMatchMode</code> is set to <code>STANDALONE</code>.
     */
    public final Integer additionalPlayerCount() {
        return additionalPlayerCount;
    }

    /**
     * <p>
     * Information to attach to all events related to the matchmaking configuration.
     * </p>
     * 
     * @return Information to attach to all events related to the matchmaking configuration.
     */
    public final String customEventData() {
        return customEventData;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the GameProperties property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasGameProperties() {
        return gameProperties != null && !(gameProperties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a
     * game server process with a request to start a new game session (see <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     * >Start a Game Session</a>). This information is added to the new <code>GameSession</code> object that is created
     * for a successful match. This parameter is not used when <code>FlexMatchMode</code> is set to
     * <code>STANDALONE</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGameProperties} method.
     * </p>
     * 
     * @return A set of custom properties for a game session, formatted as key:value pairs. These properties are passed
     *         to a game server process with a request to start a new game session (see <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     *         >Start a Game Session</a>). This information is added to the new <code>GameSession</code> object that is
     *         created for a successful match. This parameter is not used when <code>FlexMatchMode</code> is set to
     *         <code>STANDALONE</code>.
     */
    public final List<GameProperty> gameProperties() {
        return gameProperties;
    }

    /**
     * <p>
     * A set of custom game session properties, formatted as a single string value. This data is passed to a game server
     * process with a request to start a new game session (see <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     * >Start a Game Session</a>). This information is added to the new <code>GameSession</code> object that is created
     * for a successful match. This parameter is not used when <code>FlexMatchMode</code> is set to
     * <code>STANDALONE</code>.
     * </p>
     * 
     * @return A set of custom game session properties, formatted as a single string value. This data is passed to a
     *         game server process with a request to start a new game session (see <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     *         >Start a Game Session</a>). This information is added to the new <code>GameSession</code> object that is
     *         created for a successful match. This parameter is not used when <code>FlexMatchMode</code> is set to
     *         <code>STANDALONE</code>.
     */
    public final String gameSessionData() {
        return gameSessionData;
    }

    /**
     * <p>
     * The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates that the
     * game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates that GameLift
     * creates backfill requests whenever a game session has one or more open slots. Learn more about manual and
     * automatic backfill in <a
     * href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-backfill.html">Backfill existing games
     * with FlexMatch</a>. Automatic backfill is not available when <code>FlexMatchMode</code> is set to
     * <code>STANDALONE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #backfillMode} will
     * return {@link BackfillMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #backfillModeAsString}.
     * </p>
     * 
     * @return The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates
     *         that the game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates
     *         that GameLift creates backfill requests whenever a game session has one or more open slots. Learn more
     *         about manual and automatic backfill in <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-backfill.html">Backfill existing
     *         games with FlexMatch</a>. Automatic backfill is not available when <code>FlexMatchMode</code> is set to
     *         <code>STANDALONE</code>.
     * @see BackfillMode
     */
    public final BackfillMode backfillMode() {
        return BackfillMode.fromValue(backfillMode);
    }

    /**
     * <p>
     * The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates that the
     * game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates that GameLift
     * creates backfill requests whenever a game session has one or more open slots. Learn more about manual and
     * automatic backfill in <a
     * href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-backfill.html">Backfill existing games
     * with FlexMatch</a>. Automatic backfill is not available when <code>FlexMatchMode</code> is set to
     * <code>STANDALONE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #backfillMode} will
     * return {@link BackfillMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #backfillModeAsString}.
     * </p>
     * 
     * @return The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates
     *         that the game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates
     *         that GameLift creates backfill requests whenever a game session has one or more open slots. Learn more
     *         about manual and automatic backfill in <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-backfill.html">Backfill existing
     *         games with FlexMatch</a>. Automatic backfill is not available when <code>FlexMatchMode</code> is set to
     *         <code>STANDALONE</code>.
     * @see BackfillMode
     */
    public final String backfillModeAsString() {
        return backfillMode;
    }

    /**
     * <p>
     * Indicates whether this matchmaking configuration is being used with Amazon GameLift hosting or as a standalone
     * matchmaking solution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>STANDALONE</b> - FlexMatch forms matches and returns match information, including players and team
     * assignments, in a <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-events.html#match-events-matchmakingsucceeded">
     * MatchmakingSucceeded</a> event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>WITH_QUEUE</b> - FlexMatch forms matches and uses the specified Amazon GameLift queue to start a game session
     * for the match.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #flexMatchMode}
     * will return {@link FlexMatchMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #flexMatchModeAsString}.
     * </p>
     * 
     * @return Indicates whether this matchmaking configuration is being used with Amazon GameLift hosting or as a
     *         standalone matchmaking solution. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>STANDALONE</b> - FlexMatch forms matches and returns match information, including players and team
     *         assignments, in a <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-events.html#match-events-matchmakingsucceeded"
     *         > MatchmakingSucceeded</a> event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WITH_QUEUE</b> - FlexMatch forms matches and uses the specified Amazon GameLift queue to start a game
     *         session for the match.
     *         </p>
     *         </li>
     * @see FlexMatchMode
     */
    public final FlexMatchMode flexMatchMode() {
        return FlexMatchMode.fromValue(flexMatchMode);
    }

    /**
     * <p>
     * Indicates whether this matchmaking configuration is being used with Amazon GameLift hosting or as a standalone
     * matchmaking solution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>STANDALONE</b> - FlexMatch forms matches and returns match information, including players and team
     * assignments, in a <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-events.html#match-events-matchmakingsucceeded">
     * MatchmakingSucceeded</a> event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>WITH_QUEUE</b> - FlexMatch forms matches and uses the specified Amazon GameLift queue to start a game session
     * for the match.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #flexMatchMode}
     * will return {@link FlexMatchMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #flexMatchModeAsString}.
     * </p>
     * 
     * @return Indicates whether this matchmaking configuration is being used with Amazon GameLift hosting or as a
     *         standalone matchmaking solution. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>STANDALONE</b> - FlexMatch forms matches and returns match information, including players and team
     *         assignments, in a <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-events.html#match-events-matchmakingsucceeded"
     *         > MatchmakingSucceeded</a> event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WITH_QUEUE</b> - FlexMatch forms matches and uses the specified Amazon GameLift queue to start a game
     *         session for the match.
     *         </p>
     *         </li>
     * @see FlexMatchMode
     */
    public final String flexMatchModeAsString() {
        return flexMatchMode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(configurationArn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasGameSessionQueueArns() ? gameSessionQueueArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(requestTimeoutSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(acceptanceTimeoutSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(acceptanceRequired());
        hashCode = 31 * hashCode + Objects.hashCode(ruleSetName());
        hashCode = 31 * hashCode + Objects.hashCode(ruleSetArn());
        hashCode = 31 * hashCode + Objects.hashCode(notificationTarget());
        hashCode = 31 * hashCode + Objects.hashCode(additionalPlayerCount());
        hashCode = 31 * hashCode + Objects.hashCode(customEventData());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasGameProperties() ? gameProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionData());
        hashCode = 31 * hashCode + Objects.hashCode(backfillModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(flexMatchModeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MatchmakingConfiguration)) {
            return false;
        }
        MatchmakingConfiguration other = (MatchmakingConfiguration) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(configurationArn(), other.configurationArn())
                && Objects.equals(description(), other.description())
                && hasGameSessionQueueArns() == other.hasGameSessionQueueArns()
                && Objects.equals(gameSessionQueueArns(), other.gameSessionQueueArns())
                && Objects.equals(requestTimeoutSeconds(), other.requestTimeoutSeconds())
                && Objects.equals(acceptanceTimeoutSeconds(), other.acceptanceTimeoutSeconds())
                && Objects.equals(acceptanceRequired(), other.acceptanceRequired())
                && Objects.equals(ruleSetName(), other.ruleSetName()) && Objects.equals(ruleSetArn(), other.ruleSetArn())
                && Objects.equals(notificationTarget(), other.notificationTarget())
                && Objects.equals(additionalPlayerCount(), other.additionalPlayerCount())
                && Objects.equals(customEventData(), other.customEventData())
                && Objects.equals(creationTime(), other.creationTime()) && hasGameProperties() == other.hasGameProperties()
                && Objects.equals(gameProperties(), other.gameProperties())
                && Objects.equals(gameSessionData(), other.gameSessionData())
                && Objects.equals(backfillModeAsString(), other.backfillModeAsString())
                && Objects.equals(flexMatchModeAsString(), other.flexMatchModeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MatchmakingConfiguration").add("Name", name()).add("ConfigurationArn", configurationArn())
                .add("Description", description())
                .add("GameSessionQueueArns", hasGameSessionQueueArns() ? gameSessionQueueArns() : null)
                .add("RequestTimeoutSeconds", requestTimeoutSeconds())
                .add("AcceptanceTimeoutSeconds", acceptanceTimeoutSeconds()).add("AcceptanceRequired", acceptanceRequired())
                .add("RuleSetName", ruleSetName()).add("RuleSetArn", ruleSetArn())
                .add("NotificationTarget", notificationTarget()).add("AdditionalPlayerCount", additionalPlayerCount())
                .add("CustomEventData", customEventData()).add("CreationTime", creationTime())
                .add("GameProperties", hasGameProperties() ? gameProperties() : null).add("GameSessionData", gameSessionData())
                .add("BackfillMode", backfillModeAsString()).add("FlexMatchMode", flexMatchModeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ConfigurationArn":
            return Optional.ofNullable(clazz.cast(configurationArn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "GameSessionQueueArns":
            return Optional.ofNullable(clazz.cast(gameSessionQueueArns()));
        case "RequestTimeoutSeconds":
            return Optional.ofNullable(clazz.cast(requestTimeoutSeconds()));
        case "AcceptanceTimeoutSeconds":
            return Optional.ofNullable(clazz.cast(acceptanceTimeoutSeconds()));
        case "AcceptanceRequired":
            return Optional.ofNullable(clazz.cast(acceptanceRequired()));
        case "RuleSetName":
            return Optional.ofNullable(clazz.cast(ruleSetName()));
        case "RuleSetArn":
            return Optional.ofNullable(clazz.cast(ruleSetArn()));
        case "NotificationTarget":
            return Optional.ofNullable(clazz.cast(notificationTarget()));
        case "AdditionalPlayerCount":
            return Optional.ofNullable(clazz.cast(additionalPlayerCount()));
        case "CustomEventData":
            return Optional.ofNullable(clazz.cast(customEventData()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "GameProperties":
            return Optional.ofNullable(clazz.cast(gameProperties()));
        case "GameSessionData":
            return Optional.ofNullable(clazz.cast(gameSessionData()));
        case "BackfillMode":
            return Optional.ofNullable(clazz.cast(backfillModeAsString()));
        case "FlexMatchMode":
            return Optional.ofNullable(clazz.cast(flexMatchModeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MatchmakingConfiguration, T> g) {
        return obj -> g.apply((MatchmakingConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MatchmakingConfiguration> {
        /**
         * <p>
         * A unique identifier for the matchmaking configuration. This name is used to identify the configuration
         * associated with a matchmaking request or ticket.
         * </p>
         * 
         * @param name
         *        A unique identifier for the matchmaking configuration. This name is used to identify the configuration
         *        associated with a matchmaking request or ticket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
         * Amazon GameLift matchmaking configuration resource and uniquely identifies it. ARNs are unique across all
         * Regions. Format is
         * <code>arn:aws:gamelift:&lt;region&gt;::matchmakingconfiguration/&lt;matchmaking configuration name&gt;</code>
         * . In a Amazon GameLift configuration ARN, the resource ID matches the <i>Name</i> value.
         * </p>
         * 
         * @param configurationArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        a Amazon GameLift matchmaking configuration resource and uniquely identifies it. ARNs are unique
         *        across all Regions. Format is
         *        <code>arn:aws:gamelift:&lt;region&gt;::matchmakingconfiguration/&lt;matchmaking configuration name&gt;</code>
         *        . In a Amazon GameLift configuration ARN, the resource ID matches the <i>Name</i> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationArn(String configurationArn);

        /**
         * <p>
         * A descriptive label that is associated with matchmaking configuration.
         * </p>
         * 
         * @param description
         *        A descriptive label that is associated with matchmaking configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
         * Amazon GameLift game session queue resource and uniquely identifies it. ARNs are unique across all Regions.
         * Format is <code>arn:aws:gamelift:&lt;region&gt;::gamesessionqueue/&lt;queue name&gt;</code>. Queues can be
         * located in any Region. Queues are used to start new Amazon GameLift-hosted game sessions for matches that are
         * created with this matchmaking configuration. This property is not set when <code>FlexMatchMode</code> is set
         * to <code>STANDALONE</code>.
         * </p>
         * 
         * @param gameSessionQueueArns
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        a Amazon GameLift game session queue resource and uniquely identifies it. ARNs are unique across all
         *        Regions. Format is <code>arn:aws:gamelift:&lt;region&gt;::gamesessionqueue/&lt;queue name&gt;</code>.
         *        Queues can be located in any Region. Queues are used to start new Amazon GameLift-hosted game sessions
         *        for matches that are created with this matchmaking configuration. This property is not set when
         *        <code>FlexMatchMode</code> is set to <code>STANDALONE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionQueueArns(Collection<String> gameSessionQueueArns);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
         * Amazon GameLift game session queue resource and uniquely identifies it. ARNs are unique across all Regions.
         * Format is <code>arn:aws:gamelift:&lt;region&gt;::gamesessionqueue/&lt;queue name&gt;</code>. Queues can be
         * located in any Region. Queues are used to start new Amazon GameLift-hosted game sessions for matches that are
         * created with this matchmaking configuration. This property is not set when <code>FlexMatchMode</code> is set
         * to <code>STANDALONE</code>.
         * </p>
         * 
         * @param gameSessionQueueArns
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        a Amazon GameLift game session queue resource and uniquely identifies it. ARNs are unique across all
         *        Regions. Format is <code>arn:aws:gamelift:&lt;region&gt;::gamesessionqueue/&lt;queue name&gt;</code>.
         *        Queues can be located in any Region. Queues are used to start new Amazon GameLift-hosted game sessions
         *        for matches that are created with this matchmaking configuration. This property is not set when
         *        <code>FlexMatchMode</code> is set to <code>STANDALONE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionQueueArns(String... gameSessionQueueArns);

        /**
         * <p>
         * The maximum duration, in seconds, that a matchmaking ticket can remain in process before timing out. Requests
         * that fail due to timing out can be resubmitted as needed.
         * </p>
         * 
         * @param requestTimeoutSeconds
         *        The maximum duration, in seconds, that a matchmaking ticket can remain in process before timing out.
         *        Requests that fail due to timing out can be resubmitted as needed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestTimeoutSeconds(Integer requestTimeoutSeconds);

        /**
         * <p>
         * The length of time (in seconds) to wait for players to accept a proposed match, if acceptance is required. If
         * any player rejects the match or fails to accept before the timeout, the ticket continues to look for an
         * acceptable match.
         * </p>
         * 
         * @param acceptanceTimeoutSeconds
         *        The length of time (in seconds) to wait for players to accept a proposed match, if acceptance is
         *        required. If any player rejects the match or fails to accept before the timeout, the ticket continues
         *        to look for an acceptable match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceptanceTimeoutSeconds(Integer acceptanceTimeoutSeconds);

        /**
         * <p>
         * A flag that indicates whether a match that was created with this configuration must be accepted by the
         * matched players. To require acceptance, set to TRUE. When this option is enabled, matchmaking tickets use the
         * status <code>REQUIRES_ACCEPTANCE</code> to indicate when a completed potential match is waiting for player
         * acceptance.
         * </p>
         * 
         * @param acceptanceRequired
         *        A flag that indicates whether a match that was created with this configuration must be accepted by the
         *        matched players. To require acceptance, set to TRUE. When this option is enabled, matchmaking tickets
         *        use the status <code>REQUIRES_ACCEPTANCE</code> to indicate when a completed potential match is
         *        waiting for player acceptance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceptanceRequired(Boolean acceptanceRequired);

        /**
         * <p>
         * A unique identifier for the matchmaking rule set to use with this configuration. A matchmaking configuration
         * can only use rule sets that are defined in the same Region.
         * </p>
         * 
         * @param ruleSetName
         *        A unique identifier for the matchmaking rule set to use with this configuration. A matchmaking
         *        configuration can only use rule sets that are defined in the same Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleSetName(String ruleSetName);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         * GameLift matchmaking rule set resource that this configuration uses.
         * </p>
         * 
         * @param ruleSetArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         *        GameLift matchmaking rule set resource that this configuration uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleSetArn(String ruleSetArn);

        /**
         * <p>
         * An SNS topic ARN that is set up to receive matchmaking notifications.
         * </p>
         * 
         * @param notificationTarget
         *        An SNS topic ARN that is set up to receive matchmaking notifications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTarget(String notificationTarget);

        /**
         * <p>
         * The number of player slots in a match to keep open for future players. For example, if the configuration's
         * rule set specifies a match for a single 12-person team, and the additional player count is set to 2, only 10
         * players are selected for the match. This parameter is not used when <code>FlexMatchMode</code> is set to
         * <code>STANDALONE</code>.
         * </p>
         * 
         * @param additionalPlayerCount
         *        The number of player slots in a match to keep open for future players. For example, if the
         *        configuration's rule set specifies a match for a single 12-person team, and the additional player
         *        count is set to 2, only 10 players are selected for the match. This parameter is not used when
         *        <code>FlexMatchMode</code> is set to <code>STANDALONE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalPlayerCount(Integer additionalPlayerCount);

        /**
         * <p>
         * Information to attach to all events related to the matchmaking configuration.
         * </p>
         * 
         * @param customEventData
         *        Information to attach to all events related to the matchmaking configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customEventData(String customEventData);

        /**
         * <p>
         * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param creationTime
         *        A time stamp indicating when this data object was created. Format is a number expressed in Unix time
         *        as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a
         * game server process with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>). This information is added to the new <code>GameSession</code> object that is
         * created for a successful match. This parameter is not used when <code>FlexMatchMode</code> is set to
         * <code>STANDALONE</code>.
         * </p>
         * 
         * @param gameProperties
         *        A set of custom properties for a game session, formatted as key:value pairs. These properties are
         *        passed to a game server process with a request to start a new game session (see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>). This information is added to the new <code>GameSession</code> object that
         *        is created for a successful match. This parameter is not used when <code>FlexMatchMode</code> is set
         *        to <code>STANDALONE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameProperties(Collection<GameProperty> gameProperties);

        /**
         * <p>
         * A set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a
         * game server process with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>). This information is added to the new <code>GameSession</code> object that is
         * created for a successful match. This parameter is not used when <code>FlexMatchMode</code> is set to
         * <code>STANDALONE</code>.
         * </p>
         * 
         * @param gameProperties
         *        A set of custom properties for a game session, formatted as key:value pairs. These properties are
         *        passed to a game server process with a request to start a new game session (see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>). This information is added to the new <code>GameSession</code> object that
         *        is created for a successful match. This parameter is not used when <code>FlexMatchMode</code> is set
         *        to <code>STANDALONE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameProperties(GameProperty... gameProperties);

        /**
         * <p>
         * A set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a
         * game server process with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>). This information is added to the new <code>GameSession</code> object that is
         * created for a successful match. This parameter is not used when <code>FlexMatchMode</code> is set to
         * <code>STANDALONE</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.GameProperty.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.gamelift.model.GameProperty#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.GameProperty.Builder#build()} is called immediately and
         * its result is passed to {@link #gameProperties(List<GameProperty>)}.
         * 
         * @param gameProperties
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.GameProperty.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gameProperties(java.util.Collection<GameProperty>)
         */
        Builder gameProperties(Consumer<GameProperty.Builder>... gameProperties);

        /**
         * <p>
         * A set of custom game session properties, formatted as a single string value. This data is passed to a game
         * server process with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>). This information is added to the new <code>GameSession</code> object that is
         * created for a successful match. This parameter is not used when <code>FlexMatchMode</code> is set to
         * <code>STANDALONE</code>.
         * </p>
         * 
         * @param gameSessionData
         *        A set of custom game session properties, formatted as a single string value. This data is passed to a
         *        game server process with a request to start a new game session (see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>). This information is added to the new <code>GameSession</code> object that
         *        is created for a successful match. This parameter is not used when <code>FlexMatchMode</code> is set
         *        to <code>STANDALONE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionData(String gameSessionData);

        /**
         * <p>
         * The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates that
         * the game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates that
         * GameLift creates backfill requests whenever a game session has one or more open slots. Learn more about
         * manual and automatic backfill in <a
         * href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-backfill.html">Backfill existing games
         * with FlexMatch</a>. Automatic backfill is not available when <code>FlexMatchMode</code> is set to
         * <code>STANDALONE</code>.
         * </p>
         * 
         * @param backfillMode
         *        The method used to backfill game sessions created with this matchmaking configuration. MANUAL
         *        indicates that the game makes backfill requests or does not use the match backfill feature. AUTOMATIC
         *        indicates that GameLift creates backfill requests whenever a game session has one or more open slots.
         *        Learn more about manual and automatic backfill in <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-backfill.html">Backfill
         *        existing games with FlexMatch</a>. Automatic backfill is not available when <code>FlexMatchMode</code>
         *        is set to <code>STANDALONE</code>.
         * @see BackfillMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackfillMode
         */
        Builder backfillMode(String backfillMode);

        /**
         * <p>
         * The method used to backfill game sessions created with this matchmaking configuration. MANUAL indicates that
         * the game makes backfill requests or does not use the match backfill feature. AUTOMATIC indicates that
         * GameLift creates backfill requests whenever a game session has one or more open slots. Learn more about
         * manual and automatic backfill in <a
         * href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-backfill.html">Backfill existing games
         * with FlexMatch</a>. Automatic backfill is not available when <code>FlexMatchMode</code> is set to
         * <code>STANDALONE</code>.
         * </p>
         * 
         * @param backfillMode
         *        The method used to backfill game sessions created with this matchmaking configuration. MANUAL
         *        indicates that the game makes backfill requests or does not use the match backfill feature. AUTOMATIC
         *        indicates that GameLift creates backfill requests whenever a game session has one or more open slots.
         *        Learn more about manual and automatic backfill in <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-backfill.html">Backfill
         *        existing games with FlexMatch</a>. Automatic backfill is not available when <code>FlexMatchMode</code>
         *        is set to <code>STANDALONE</code>.
         * @see BackfillMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackfillMode
         */
        Builder backfillMode(BackfillMode backfillMode);

        /**
         * <p>
         * Indicates whether this matchmaking configuration is being used with Amazon GameLift hosting or as a
         * standalone matchmaking solution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>STANDALONE</b> - FlexMatch forms matches and returns match information, including players and team
         * assignments, in a <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-events.html#match-events-matchmakingsucceeded"
         * > MatchmakingSucceeded</a> event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>WITH_QUEUE</b> - FlexMatch forms matches and uses the specified Amazon GameLift queue to start a game
         * session for the match.
         * </p>
         * </li>
         * </ul>
         * 
         * @param flexMatchMode
         *        Indicates whether this matchmaking configuration is being used with Amazon GameLift hosting or as a
         *        standalone matchmaking solution. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>STANDALONE</b> - FlexMatch forms matches and returns match information, including players and team
         *        assignments, in a <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-events.html#match-events-matchmakingsucceeded"
         *        > MatchmakingSucceeded</a> event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>WITH_QUEUE</b> - FlexMatch forms matches and uses the specified Amazon GameLift queue to start a
         *        game session for the match.
         *        </p>
         *        </li>
         * @see FlexMatchMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FlexMatchMode
         */
        Builder flexMatchMode(String flexMatchMode);

        /**
         * <p>
         * Indicates whether this matchmaking configuration is being used with Amazon GameLift hosting or as a
         * standalone matchmaking solution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>STANDALONE</b> - FlexMatch forms matches and returns match information, including players and team
         * assignments, in a <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-events.html#match-events-matchmakingsucceeded"
         * > MatchmakingSucceeded</a> event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>WITH_QUEUE</b> - FlexMatch forms matches and uses the specified Amazon GameLift queue to start a game
         * session for the match.
         * </p>
         * </li>
         * </ul>
         * 
         * @param flexMatchMode
         *        Indicates whether this matchmaking configuration is being used with Amazon GameLift hosting or as a
         *        standalone matchmaking solution. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>STANDALONE</b> - FlexMatch forms matches and returns match information, including players and team
         *        assignments, in a <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-events.html#match-events-matchmakingsucceeded"
         *        > MatchmakingSucceeded</a> event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>WITH_QUEUE</b> - FlexMatch forms matches and uses the specified Amazon GameLift queue to start a
         *        game session for the match.
         *        </p>
         *        </li>
         * @see FlexMatchMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FlexMatchMode
         */
        Builder flexMatchMode(FlexMatchMode flexMatchMode);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String configurationArn;

        private String description;

        private List<String> gameSessionQueueArns = DefaultSdkAutoConstructList.getInstance();

        private Integer requestTimeoutSeconds;

        private Integer acceptanceTimeoutSeconds;

        private Boolean acceptanceRequired;

        private String ruleSetName;

        private String ruleSetArn;

        private String notificationTarget;

        private Integer additionalPlayerCount;

        private String customEventData;

        private Instant creationTime;

        private List<GameProperty> gameProperties = DefaultSdkAutoConstructList.getInstance();

        private String gameSessionData;

        private String backfillMode;

        private String flexMatchMode;

        private BuilderImpl() {
        }

        private BuilderImpl(MatchmakingConfiguration model) {
            name(model.name);
            configurationArn(model.configurationArn);
            description(model.description);
            gameSessionQueueArns(model.gameSessionQueueArns);
            requestTimeoutSeconds(model.requestTimeoutSeconds);
            acceptanceTimeoutSeconds(model.acceptanceTimeoutSeconds);
            acceptanceRequired(model.acceptanceRequired);
            ruleSetName(model.ruleSetName);
            ruleSetArn(model.ruleSetArn);
            notificationTarget(model.notificationTarget);
            additionalPlayerCount(model.additionalPlayerCount);
            customEventData(model.customEventData);
            creationTime(model.creationTime);
            gameProperties(model.gameProperties);
            gameSessionData(model.gameSessionData);
            backfillMode(model.backfillMode);
            flexMatchMode(model.flexMatchMode);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getConfigurationArn() {
            return configurationArn;
        }

        public final void setConfigurationArn(String configurationArn) {
            this.configurationArn = configurationArn;
        }

        @Override
        public final Builder configurationArn(String configurationArn) {
            this.configurationArn = configurationArn;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getGameSessionQueueArns() {
            if (gameSessionQueueArns instanceof SdkAutoConstructList) {
                return null;
            }
            return gameSessionQueueArns;
        }

        public final void setGameSessionQueueArns(Collection<String> gameSessionQueueArns) {
            this.gameSessionQueueArns = QueueArnsListCopier.copy(gameSessionQueueArns);
        }

        @Override
        public final Builder gameSessionQueueArns(Collection<String> gameSessionQueueArns) {
            this.gameSessionQueueArns = QueueArnsListCopier.copy(gameSessionQueueArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameSessionQueueArns(String... gameSessionQueueArns) {
            gameSessionQueueArns(Arrays.asList(gameSessionQueueArns));
            return this;
        }

        public final Integer getRequestTimeoutSeconds() {
            return requestTimeoutSeconds;
        }

        public final void setRequestTimeoutSeconds(Integer requestTimeoutSeconds) {
            this.requestTimeoutSeconds = requestTimeoutSeconds;
        }

        @Override
        public final Builder requestTimeoutSeconds(Integer requestTimeoutSeconds) {
            this.requestTimeoutSeconds = requestTimeoutSeconds;
            return this;
        }

        public final Integer getAcceptanceTimeoutSeconds() {
            return acceptanceTimeoutSeconds;
        }

        public final void setAcceptanceTimeoutSeconds(Integer acceptanceTimeoutSeconds) {
            this.acceptanceTimeoutSeconds = acceptanceTimeoutSeconds;
        }

        @Override
        public final Builder acceptanceTimeoutSeconds(Integer acceptanceTimeoutSeconds) {
            this.acceptanceTimeoutSeconds = acceptanceTimeoutSeconds;
            return this;
        }

        public final Boolean getAcceptanceRequired() {
            return acceptanceRequired;
        }

        public final void setAcceptanceRequired(Boolean acceptanceRequired) {
            this.acceptanceRequired = acceptanceRequired;
        }

        @Override
        public final Builder acceptanceRequired(Boolean acceptanceRequired) {
            this.acceptanceRequired = acceptanceRequired;
            return this;
        }

        public final String getRuleSetName() {
            return ruleSetName;
        }

        public final void setRuleSetName(String ruleSetName) {
            this.ruleSetName = ruleSetName;
        }

        @Override
        public final Builder ruleSetName(String ruleSetName) {
            this.ruleSetName = ruleSetName;
            return this;
        }

        public final String getRuleSetArn() {
            return ruleSetArn;
        }

        public final void setRuleSetArn(String ruleSetArn) {
            this.ruleSetArn = ruleSetArn;
        }

        @Override
        public final Builder ruleSetArn(String ruleSetArn) {
            this.ruleSetArn = ruleSetArn;
            return this;
        }

        public final String getNotificationTarget() {
            return notificationTarget;
        }

        public final void setNotificationTarget(String notificationTarget) {
            this.notificationTarget = notificationTarget;
        }

        @Override
        public final Builder notificationTarget(String notificationTarget) {
            this.notificationTarget = notificationTarget;
            return this;
        }

        public final Integer getAdditionalPlayerCount() {
            return additionalPlayerCount;
        }

        public final void setAdditionalPlayerCount(Integer additionalPlayerCount) {
            this.additionalPlayerCount = additionalPlayerCount;
        }

        @Override
        public final Builder additionalPlayerCount(Integer additionalPlayerCount) {
            this.additionalPlayerCount = additionalPlayerCount;
            return this;
        }

        public final String getCustomEventData() {
            return customEventData;
        }

        public final void setCustomEventData(String customEventData) {
            this.customEventData = customEventData;
        }

        @Override
        public final Builder customEventData(String customEventData) {
            this.customEventData = customEventData;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final List<GameProperty.Builder> getGameProperties() {
            List<GameProperty.Builder> result = GamePropertyListCopier.copyToBuilder(this.gameProperties);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setGameProperties(Collection<GameProperty.BuilderImpl> gameProperties) {
            this.gameProperties = GamePropertyListCopier.copyFromBuilder(gameProperties);
        }

        @Override
        public final Builder gameProperties(Collection<GameProperty> gameProperties) {
            this.gameProperties = GamePropertyListCopier.copy(gameProperties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameProperties(GameProperty... gameProperties) {
            gameProperties(Arrays.asList(gameProperties));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameProperties(Consumer<GameProperty.Builder>... gameProperties) {
            gameProperties(Stream.of(gameProperties).map(c -> GameProperty.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getGameSessionData() {
            return gameSessionData;
        }

        public final void setGameSessionData(String gameSessionData) {
            this.gameSessionData = gameSessionData;
        }

        @Override
        public final Builder gameSessionData(String gameSessionData) {
            this.gameSessionData = gameSessionData;
            return this;
        }

        public final String getBackfillMode() {
            return backfillMode;
        }

        public final void setBackfillMode(String backfillMode) {
            this.backfillMode = backfillMode;
        }

        @Override
        public final Builder backfillMode(String backfillMode) {
            this.backfillMode = backfillMode;
            return this;
        }

        @Override
        public final Builder backfillMode(BackfillMode backfillMode) {
            this.backfillMode(backfillMode == null ? null : backfillMode.toString());
            return this;
        }

        public final String getFlexMatchMode() {
            return flexMatchMode;
        }

        public final void setFlexMatchMode(String flexMatchMode) {
            this.flexMatchMode = flexMatchMode;
        }

        @Override
        public final Builder flexMatchMode(String flexMatchMode) {
            this.flexMatchMode = flexMatchMode;
            return this;
        }

        @Override
        public final Builder flexMatchMode(FlexMatchMode flexMatchMode) {
            this.flexMatchMode(flexMatchMode == null ? null : flexMatchMode.toString());
            return this;
        }

        @Override
        public MatchmakingConfiguration build() {
            return new MatchmakingConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
