/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a player in matchmaking. When starting a matchmaking request, a player has a player ID, attributes, and
 * may have latency data. Team information is added after a match has been successfully completed.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Player implements SdkPojo, Serializable, ToCopyableBuilder<Player.Builder, Player> {
    private static final SdkField<String> PLAYER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PlayerId").getter(getter(Player::playerId)).setter(setter(Builder::playerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlayerId").build()).build();

    private static final SdkField<Map<String, AttributeValue>> PLAYER_ATTRIBUTES_FIELD = SdkField
            .<Map<String, AttributeValue>> builder(MarshallingType.MAP)
            .memberName("PlayerAttributes")
            .getter(getter(Player::playerAttributes))
            .setter(setter(Builder::playerAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlayerAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> TEAM_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Team")
            .getter(getter(Player::team)).setter(setter(Builder::team))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Team").build()).build();

    private static final SdkField<Map<String, Integer>> LATENCY_IN_MS_FIELD = SdkField
            .<Map<String, Integer>> builder(MarshallingType.MAP)
            .memberName("LatencyInMs")
            .getter(getter(Player::latencyInMs))
            .setter(setter(Builder::latencyInMs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LatencyInMs").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PLAYER_ID_FIELD,
            PLAYER_ATTRIBUTES_FIELD, TEAM_FIELD, LATENCY_IN_MS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String playerId;

    private final Map<String, AttributeValue> playerAttributes;

    private final String team;

    private final Map<String, Integer> latencyInMs;

    private Player(BuilderImpl builder) {
        this.playerId = builder.playerId;
        this.playerAttributes = builder.playerAttributes;
        this.team = builder.team;
        this.latencyInMs = builder.latencyInMs;
    }

    /**
     * <p>
     * A unique identifier for a player
     * </p>
     * 
     * @return A unique identifier for a player
     */
    public final String playerId() {
        return playerId;
    }

    /**
     * For responses, this returns true if the service returned a value for the PlayerAttributes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlayerAttributes() {
        return playerAttributes != null && !(playerAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A collection of key:value pairs containing player information for use in matchmaking. Player attribute keys must
     * match the <i>playerAttributes</i> used in a matchmaking rule set. Example:
     * <code>"PlayerAttributes": {"skill": {"N": "23"}, "gameMode": {"S": "deathmatch"}}</code>.
     * </p>
     * <p>
     * You can provide up to 10 <code>PlayerAttributes</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlayerAttributes} method.
     * </p>
     * 
     * @return A collection of key:value pairs containing player information for use in matchmaking. Player attribute
     *         keys must match the <i>playerAttributes</i> used in a matchmaking rule set. Example:
     *         <code>"PlayerAttributes": {"skill": {"N": "23"}, "gameMode": {"S": "deathmatch"}}</code>.</p>
     *         <p>
     *         You can provide up to 10 <code>PlayerAttributes</code>.
     */
    public final Map<String, AttributeValue> playerAttributes() {
        return playerAttributes;
    }

    /**
     * <p>
     * Name of the team that the player is assigned to in a match. Team names are defined in a matchmaking rule set.
     * </p>
     * 
     * @return Name of the team that the player is assigned to in a match. Team names are defined in a matchmaking rule
     *         set.
     */
    public final String team() {
        return team;
    }

    /**
     * For responses, this returns true if the service returned a value for the LatencyInMs property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLatencyInMs() {
        return latencyInMs != null && !(latencyInMs instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A set of values, expressed in milliseconds, that indicates the amount of latency that a player experiences when
     * connected to @aws; Regions. If this property is present, FlexMatch considers placing the match only in Regions
     * for which latency is reported.
     * </p>
     * <p>
     * If a matchmaker has a rule that evaluates player latency, players must report latency in order to be matched. If
     * no latency is reported in this scenario, FlexMatch assumes that no Regions are available to the player and the
     * ticket is not matchable.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLatencyInMs} method.
     * </p>
     * 
     * @return A set of values, expressed in milliseconds, that indicates the amount of latency that a player
     *         experiences when connected to @aws; Regions. If this property is present, FlexMatch considers placing the
     *         match only in Regions for which latency is reported. </p>
     *         <p>
     *         If a matchmaker has a rule that evaluates player latency, players must report latency in order to be
     *         matched. If no latency is reported in this scenario, FlexMatch assumes that no Regions are available to
     *         the player and the ticket is not matchable.
     */
    public final Map<String, Integer> latencyInMs() {
        return latencyInMs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(playerId());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlayerAttributes() ? playerAttributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(team());
        hashCode = 31 * hashCode + Objects.hashCode(hasLatencyInMs() ? latencyInMs() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Player)) {
            return false;
        }
        Player other = (Player) obj;
        return Objects.equals(playerId(), other.playerId()) && hasPlayerAttributes() == other.hasPlayerAttributes()
                && Objects.equals(playerAttributes(), other.playerAttributes()) && Objects.equals(team(), other.team())
                && hasLatencyInMs() == other.hasLatencyInMs() && Objects.equals(latencyInMs(), other.latencyInMs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Player").add("PlayerId", playerId() == null ? null : "*** Sensitive Data Redacted ***")
                .add("PlayerAttributes", hasPlayerAttributes() ? playerAttributes() : null).add("Team", team())
                .add("LatencyInMs", hasLatencyInMs() ? latencyInMs() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PlayerId":
            return Optional.ofNullable(clazz.cast(playerId()));
        case "PlayerAttributes":
            return Optional.ofNullable(clazz.cast(playerAttributes()));
        case "Team":
            return Optional.ofNullable(clazz.cast(team()));
        case "LatencyInMs":
            return Optional.ofNullable(clazz.cast(latencyInMs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Player, T> g) {
        return obj -> g.apply((Player) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Player> {
        /**
         * <p>
         * A unique identifier for a player
         * </p>
         * 
         * @param playerId
         *        A unique identifier for a player
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerId(String playerId);

        /**
         * <p>
         * A collection of key:value pairs containing player information for use in matchmaking. Player attribute keys
         * must match the <i>playerAttributes</i> used in a matchmaking rule set. Example:
         * <code>"PlayerAttributes": {"skill": {"N": "23"}, "gameMode": {"S": "deathmatch"}}</code>.
         * </p>
         * <p>
         * You can provide up to 10 <code>PlayerAttributes</code>.
         * </p>
         * 
         * @param playerAttributes
         *        A collection of key:value pairs containing player information for use in matchmaking. Player attribute
         *        keys must match the <i>playerAttributes</i> used in a matchmaking rule set. Example:
         *        <code>"PlayerAttributes": {"skill": {"N": "23"}, "gameMode": {"S": "deathmatch"}}</code>.</p>
         *        <p>
         *        You can provide up to 10 <code>PlayerAttributes</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerAttributes(Map<String, AttributeValue> playerAttributes);

        /**
         * <p>
         * Name of the team that the player is assigned to in a match. Team names are defined in a matchmaking rule set.
         * </p>
         * 
         * @param team
         *        Name of the team that the player is assigned to in a match. Team names are defined in a matchmaking
         *        rule set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder team(String team);

        /**
         * <p>
         * A set of values, expressed in milliseconds, that indicates the amount of latency that a player experiences
         * when connected to @aws; Regions. If this property is present, FlexMatch considers placing the match only in
         * Regions for which latency is reported.
         * </p>
         * <p>
         * If a matchmaker has a rule that evaluates player latency, players must report latency in order to be matched.
         * If no latency is reported in this scenario, FlexMatch assumes that no Regions are available to the player and
         * the ticket is not matchable.
         * </p>
         * 
         * @param latencyInMs
         *        A set of values, expressed in milliseconds, that indicates the amount of latency that a player
         *        experiences when connected to @aws; Regions. If this property is present, FlexMatch considers placing
         *        the match only in Regions for which latency is reported. </p>
         *        <p>
         *        If a matchmaker has a rule that evaluates player latency, players must report latency in order to be
         *        matched. If no latency is reported in this scenario, FlexMatch assumes that no Regions are available
         *        to the player and the ticket is not matchable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latencyInMs(Map<String, Integer> latencyInMs);
    }

    static final class BuilderImpl implements Builder {
        private String playerId;

        private Map<String, AttributeValue> playerAttributes = DefaultSdkAutoConstructMap.getInstance();

        private String team;

        private Map<String, Integer> latencyInMs = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Player model) {
            playerId(model.playerId);
            playerAttributes(model.playerAttributes);
            team(model.team);
            latencyInMs(model.latencyInMs);
        }

        public final String getPlayerId() {
            return playerId;
        }

        public final void setPlayerId(String playerId) {
            this.playerId = playerId;
        }

        @Override
        public final Builder playerId(String playerId) {
            this.playerId = playerId;
            return this;
        }

        public final Map<String, AttributeValue.Builder> getPlayerAttributes() {
            Map<String, AttributeValue.Builder> result = PlayerAttributeMapCopier.copyToBuilder(this.playerAttributes);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setPlayerAttributes(Map<String, AttributeValue.BuilderImpl> playerAttributes) {
            this.playerAttributes = PlayerAttributeMapCopier.copyFromBuilder(playerAttributes);
        }

        @Override
        public final Builder playerAttributes(Map<String, AttributeValue> playerAttributes) {
            this.playerAttributes = PlayerAttributeMapCopier.copy(playerAttributes);
            return this;
        }

        public final String getTeam() {
            return team;
        }

        public final void setTeam(String team) {
            this.team = team;
        }

        @Override
        public final Builder team(String team) {
            this.team = team;
            return this;
        }

        public final Map<String, Integer> getLatencyInMs() {
            if (latencyInMs instanceof SdkAutoConstructMap) {
                return null;
            }
            return latencyInMs;
        }

        public final void setLatencyInMs(Map<String, Integer> latencyInMs) {
            this.latencyInMs = LatencyMapCopier.copy(latencyInMs);
        }

        @Override
        public final Builder latencyInMs(Map<String, Integer> latencyInMs) {
            this.latencyInMs = LatencyMapCopier.copy(latencyInMs);
            return this;
        }

        @Override
        public Player build() {
            return new Player(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
