/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>This data type is used with the Amazon GameLift containers feature, which is currently in public preview.</b>
 * </p>
 * <p>
 * Defines how an internal-facing container port is mapped to an external-facing connection port on a fleet instance of
 * compute type <code>CONTAINER</code>. Incoming traffic, such as a game client, uses a connection port to connect to a
 * process in the container fleet. Amazon GameLift directs the inbound traffic to the container port that is assigned to
 * the process, such as a game session, running on a container.
 * </p>
 * <p>
 * <b>Part of:</b> <a>ContainerAttributes</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerPortMapping implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerPortMapping.Builder, ContainerPortMapping> {
    private static final SdkField<Integer> CONTAINER_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ContainerPort").getter(getter(ContainerPortMapping::containerPort))
            .setter(setter(Builder::containerPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerPort").build()).build();

    private static final SdkField<Integer> CONNECTION_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ConnectionPort").getter(getter(ContainerPortMapping::connectionPort))
            .setter(setter(Builder::connectionPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionPort").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Protocol").getter(getter(ContainerPortMapping::protocolAsString)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocol").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_PORT_FIELD,
            CONNECTION_PORT_FIELD, PROTOCOL_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer containerPort;

    private final Integer connectionPort;

    private final String protocol;

    private ContainerPortMapping(BuilderImpl builder) {
        this.containerPort = builder.containerPort;
        this.connectionPort = builder.connectionPort;
        this.protocol = builder.protocol;
    }

    /**
     * <p>
     * The port opened on the container.
     * </p>
     * 
     * @return The port opened on the container.
     */
    public final Integer containerPort() {
        return containerPort;
    }

    /**
     * <p>
     * The port opened on the fleet instance. This is also called the "host port".
     * </p>
     * 
     * @return The port opened on the fleet instance. This is also called the "host port".
     */
    public final Integer connectionPort() {
        return connectionPort;
    }

    /**
     * <p>
     * The network protocol that this mapping supports.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link IpProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The network protocol that this mapping supports.
     * @see IpProtocol
     */
    public final IpProtocol protocol() {
        return IpProtocol.fromValue(protocol);
    }

    /**
     * <p>
     * The network protocol that this mapping supports.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link IpProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The network protocol that this mapping supports.
     * @see IpProtocol
     */
    public final String protocolAsString() {
        return protocol;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerPort());
        hashCode = 31 * hashCode + Objects.hashCode(connectionPort());
        hashCode = 31 * hashCode + Objects.hashCode(protocolAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerPortMapping)) {
            return false;
        }
        ContainerPortMapping other = (ContainerPortMapping) obj;
        return Objects.equals(containerPort(), other.containerPort()) && Objects.equals(connectionPort(), other.connectionPort())
                && Objects.equals(protocolAsString(), other.protocolAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerPortMapping")
                .add("ContainerPort", containerPort() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ConnectionPort", connectionPort() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Protocol", protocolAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerPort":
            return Optional.ofNullable(clazz.cast(containerPort()));
        case "ConnectionPort":
            return Optional.ofNullable(clazz.cast(connectionPort()));
        case "Protocol":
            return Optional.ofNullable(clazz.cast(protocolAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerPortMapping, T> g) {
        return obj -> g.apply((ContainerPortMapping) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerPortMapping> {
        /**
         * <p>
         * The port opened on the container.
         * </p>
         * 
         * @param containerPort
         *        The port opened on the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerPort(Integer containerPort);

        /**
         * <p>
         * The port opened on the fleet instance. This is also called the "host port".
         * </p>
         * 
         * @param connectionPort
         *        The port opened on the fleet instance. This is also called the "host port".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionPort(Integer connectionPort);

        /**
         * <p>
         * The network protocol that this mapping supports.
         * </p>
         * 
         * @param protocol
         *        The network protocol that this mapping supports.
         * @see IpProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpProtocol
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The network protocol that this mapping supports.
         * </p>
         * 
         * @param protocol
         *        The network protocol that this mapping supports.
         * @see IpProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpProtocol
         */
        Builder protocol(IpProtocol protocol);
    }

    static final class BuilderImpl implements Builder {
        private Integer containerPort;

        private Integer connectionPort;

        private String protocol;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerPortMapping model) {
            containerPort(model.containerPort);
            connectionPort(model.connectionPort);
            protocol(model.protocol);
        }

        public final Integer getContainerPort() {
            return containerPort;
        }

        public final void setContainerPort(Integer containerPort) {
            this.containerPort = containerPort;
        }

        @Override
        public final Builder containerPort(Integer containerPort) {
            this.containerPort = containerPort;
            return this;
        }

        public final Integer getConnectionPort() {
            return connectionPort;
        }

        public final void setConnectionPort(Integer connectionPort) {
            this.connectionPort = connectionPort;
        }

        @Override
        public final Builder connectionPort(Integer connectionPort) {
            this.connectionPort = connectionPort;
            return this;
        }

        public final String getProtocol() {
            return protocol;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        @Override
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        @Override
        public final Builder protocol(IpProtocol protocol) {
            this.protocol(protocol == null ? null : protocol.toString());
            return this;
        }

        @Override
        public ContainerPortMapping build() {
            return new ContainerPortMapping(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
