/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>This data type is used with the Amazon GameLift containers feature, which is currently in public preview.</b>
 * </p>
 * <p>
 * Describes a container in a container fleet, the resources available to the container, and the commands that are run
 * when the container starts. Container properties can't be updated. To change a property, create a new container group
 * definition. See also <a>ContainerDefinitionInput</a>.
 * </p>
 * <p>
 * <b>Part of:</b> <a>ContainerGroupDefinition</a>
 * </p>
 * <p>
 * <b>Returned by:</b> <a>DescribeContainerGroupDefinition</a>, <a>ListContainerGroupDefinitions</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerDefinition.Builder, ContainerDefinition> {
    private static final SdkField<String> CONTAINER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainerName").getter(getter(ContainerDefinition::containerName))
            .setter(setter(Builder::containerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerName").build()).build();

    private static final SdkField<String> IMAGE_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageUri").getter(getter(ContainerDefinition::imageUri)).setter(setter(Builder::imageUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageUri").build()).build();

    private static final SdkField<String> RESOLVED_IMAGE_DIGEST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResolvedImageDigest").getter(getter(ContainerDefinition::resolvedImageDigest))
            .setter(setter(Builder::resolvedImageDigest))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResolvedImageDigest").build())
            .build();

    private static final SdkField<ContainerMemoryLimits> MEMORY_LIMITS_FIELD = SdkField
            .<ContainerMemoryLimits> builder(MarshallingType.SDK_POJO).memberName("MemoryLimits")
            .getter(getter(ContainerDefinition::memoryLimits)).setter(setter(Builder::memoryLimits))
            .constructor(ContainerMemoryLimits::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MemoryLimits").build()).build();

    private static final SdkField<ContainerPortConfiguration> PORT_CONFIGURATION_FIELD = SdkField
            .<ContainerPortConfiguration> builder(MarshallingType.SDK_POJO).memberName("PortConfiguration")
            .getter(getter(ContainerDefinition::portConfiguration)).setter(setter(Builder::portConfiguration))
            .constructor(ContainerPortConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PortConfiguration").build()).build();

    private static final SdkField<Integer> CPU_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Cpu")
            .getter(getter(ContainerDefinition::cpu)).setter(setter(Builder::cpu))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Cpu").build()).build();

    private static final SdkField<ContainerHealthCheck> HEALTH_CHECK_FIELD = SdkField
            .<ContainerHealthCheck> builder(MarshallingType.SDK_POJO).memberName("HealthCheck")
            .getter(getter(ContainerDefinition::healthCheck)).setter(setter(Builder::healthCheck))
            .constructor(ContainerHealthCheck::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthCheck").build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Command")
            .getter(getter(ContainerDefinition::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ESSENTIAL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Essential").getter(getter(ContainerDefinition::essential)).setter(setter(Builder::essential))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Essential").build()).build();

    private static final SdkField<List<String>> ENTRY_POINT_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EntryPoint")
            .getter(getter(ContainerDefinition::entryPoint))
            .setter(setter(Builder::entryPoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntryPoint").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> WORKING_DIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkingDirectory").getter(getter(ContainerDefinition::workingDirectory))
            .setter(setter(Builder::workingDirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkingDirectory").build()).build();

    private static final SdkField<List<ContainerEnvironment>> ENVIRONMENT_FIELD = SdkField
            .<List<ContainerEnvironment>> builder(MarshallingType.LIST)
            .memberName("Environment")
            .getter(getter(ContainerDefinition::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Environment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerEnvironment> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerEnvironment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ContainerDependency>> DEPENDS_ON_FIELD = SdkField
            .<List<ContainerDependency>> builder(MarshallingType.LIST)
            .memberName("DependsOn")
            .getter(getter(ContainerDefinition::dependsOn))
            .setter(setter(Builder::dependsOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DependsOn").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerDependency> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerDependency::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_NAME_FIELD,
            IMAGE_URI_FIELD, RESOLVED_IMAGE_DIGEST_FIELD, MEMORY_LIMITS_FIELD, PORT_CONFIGURATION_FIELD, CPU_FIELD,
            HEALTH_CHECK_FIELD, COMMAND_FIELD, ESSENTIAL_FIELD, ENTRY_POINT_FIELD, WORKING_DIRECTORY_FIELD, ENVIRONMENT_FIELD,
            DEPENDS_ON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String containerName;

    private final String imageUri;

    private final String resolvedImageDigest;

    private final ContainerMemoryLimits memoryLimits;

    private final ContainerPortConfiguration portConfiguration;

    private final Integer cpu;

    private final ContainerHealthCheck healthCheck;

    private final List<String> command;

    private final Boolean essential;

    private final List<String> entryPoint;

    private final String workingDirectory;

    private final List<ContainerEnvironment> environment;

    private final List<ContainerDependency> dependsOn;

    private ContainerDefinition(BuilderImpl builder) {
        this.containerName = builder.containerName;
        this.imageUri = builder.imageUri;
        this.resolvedImageDigest = builder.resolvedImageDigest;
        this.memoryLimits = builder.memoryLimits;
        this.portConfiguration = builder.portConfiguration;
        this.cpu = builder.cpu;
        this.healthCheck = builder.healthCheck;
        this.command = builder.command;
        this.essential = builder.essential;
        this.entryPoint = builder.entryPoint;
        this.workingDirectory = builder.workingDirectory;
        this.environment = builder.environment;
        this.dependsOn = builder.dependsOn;
    }

    /**
     * <p>
     * The container definition identifier. Container names are unique within a container group definition.
     * </p>
     * 
     * @return The container definition identifier. Container names are unique within a container group definition.
     */
    public final String containerName() {
        return containerName;
    }

    /**
     * <p>
     * The URI to the image that $short; copied and deployed to a container fleet. For a more specific identifier, see
     * <code>ResolvedImageDigest</code>.
     * </p>
     * 
     * @return The URI to the image that $short; copied and deployed to a container fleet. For a more specific
     *         identifier, see <code>ResolvedImageDigest</code>.
     */
    public final String imageUri() {
        return imageUri;
    }

    /**
     * <p>
     * A unique and immutable identifier for the container image that is deployed to a container fleet. The digest is a
     * SHA 256 hash of the container image manifest.
     * </p>
     * 
     * @return A unique and immutable identifier for the container image that is deployed to a container fleet. The
     *         digest is a SHA 256 hash of the container image manifest.
     */
    public final String resolvedImageDigest() {
        return resolvedImageDigest;
    }

    /**
     * <p>
     * The amount of memory that Amazon GameLift makes available to the container. If memory limits aren't set for an
     * individual container, the container shares the container group's total memory allocation.
     * </p>
     * <p>
     * <b>Related data type: </b> <a>ContainerGroupDefinition$TotalMemoryLimit</a>
     * </p>
     * 
     * @return The amount of memory that Amazon GameLift makes available to the container. If memory limits aren't set
     *         for an individual container, the container shares the container group's total memory allocation.</p>
     *         <p>
     *         <b>Related data type: </b> <a>ContainerGroupDefinition$TotalMemoryLimit</a>
     */
    public final ContainerMemoryLimits memoryLimits() {
        return memoryLimits;
    }

    /**
     * <p>
     * Defines the ports that are available to assign to processes in the container. For example, a game server process
     * requires a container port to allow game clients to connect to it. Container ports aren't directly accessed by
     * inbound traffic. Amazon GameLift maps these container ports to externally accessible connection ports, which are
     * assigned as needed from the container fleet's <code>ConnectionPortRange</code>.
     * </p>
     * 
     * @return Defines the ports that are available to assign to processes in the container. For example, a game server
     *         process requires a container port to allow game clients to connect to it. Container ports aren't directly
     *         accessed by inbound traffic. Amazon GameLift maps these container ports to externally accessible
     *         connection ports, which are assigned as needed from the container fleet's
     *         <code>ConnectionPortRange</code>.
     */
    public final ContainerPortConfiguration portConfiguration() {
        return portConfiguration;
    }

    /**
     * <p>
     * The number of CPU units that are reserved for the container. Note: 1 vCPU unit equals 1024 CPU units. If no
     * resources are reserved, the container shares the total CPU limit for the container group.
     * </p>
     * <p>
     * <b>Related data type: </b> <a>ContainerGroupDefinition$TotalCpuLimit</a>
     * </p>
     * 
     * @return The number of CPU units that are reserved for the container. Note: 1 vCPU unit equals 1024 CPU units. If
     *         no resources are reserved, the container shares the total CPU limit for the container group.</p>
     *         <p>
     *         <b>Related data type: </b> <a>ContainerGroupDefinition$TotalCpuLimit</a>
     */
    public final Integer cpu() {
        return cpu;
    }

    /**
     * <p>
     * A configuration for a non-terminal health check. A container, which automatically restarts if it stops
     * functioning, also restarts if it fails this health check. If an essential container in the daemon group fails a
     * health check, the entire container group is restarted. The essential container in the replica group doesn't use
     * this health check mechanism, because the Amazon GameLift Agent automatically handles the task.
     * </p>
     * 
     * @return A configuration for a non-terminal health check. A container, which automatically restarts if it stops
     *         functioning, also restarts if it fails this health check. If an essential container in the daemon group
     *         fails a health check, the entire container group is restarted. The essential container in the replica
     *         group doesn't use this health check mechanism, because the Amazon GameLift Agent automatically handles
     *         the task.
     */
    public final ContainerHealthCheck healthCheck() {
        return healthCheck;
    }

    /**
     * For responses, this returns true if the service returned a value for the Command property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A command that's passed to the container on startup. Each argument for the command is an additional string in the
     * array. See the <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
     * >ContainerDefinition::command</a> parameter in the <i>Amazon Elastic Container Service API reference.</i>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommand} method.
     * </p>
     * 
     * @return A command that's passed to the container on startup. Each argument for the command is an additional
     *         string in the array. See the <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
     *         >ContainerDefinition::command</a> parameter in the <i>Amazon Elastic Container Service API reference.</i>
     */
    public final List<String> command() {
        return command;
    }

    /**
     * <p>
     * Indicates whether the container is vital to the container group. If an essential container fails, the entire
     * container group is restarted.
     * </p>
     * 
     * @return Indicates whether the container is vital to the container group. If an essential container fails, the
     *         entire container group is restarted.
     */
    public final Boolean essential() {
        return essential;
    }

    /**
     * For responses, this returns true if the service returned a value for the EntryPoint property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEntryPoint() {
        return entryPoint != null && !(entryPoint instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The entry point that's passed to the container on startup. If there are multiple arguments, each argument is an
     * additional string in the array. See the <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
     * >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntryPoint} method.
     * </p>
     * 
     * @return The entry point that's passed to the container on startup. If there are multiple arguments, each argument
     *         is an additional string in the array. See the <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
     *         >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API
     *         Reference</i>.
     */
    public final List<String> entryPoint() {
        return entryPoint;
    }

    /**
     * <p>
     * The directory in the container where commands are run. See the <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-workingDirectory"
     * >ContainerDefinition::workingDirectory</a> parameter in the <i>Amazon Elastic Container Service API
     * Reference</i>.
     * </p>
     * 
     * @return The directory in the container where commands are run. See the <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-workingDirectory"
     *         >ContainerDefinition::workingDirectory</a> parameter in the <i>Amazon Elastic Container Service API
     *         Reference</i>.
     */
    public final String workingDirectory() {
        return workingDirectory;
    }

    /**
     * For responses, this returns true if the service returned a value for the Environment property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of environment variables that's passed to the container on startup. See the <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
     * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironment} method.
     * </p>
     * 
     * @return A set of environment variables that's passed to the container on startup. See the <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
     *         >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
     *         Reference</i>.
     */
    public final List<ContainerEnvironment> environment() {
        return environment;
    }

    /**
     * For responses, this returns true if the service returned a value for the DependsOn property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDependsOn() {
        return dependsOn != null && !(dependsOn instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates that the container relies on the status of other containers in the same container group during its
     * startup and shutdown sequences. A container might have dependencies on multiple containers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDependsOn} method.
     * </p>
     * 
     * @return Indicates that the container relies on the status of other containers in the same container group during
     *         its startup and shutdown sequences. A container might have dependencies on multiple containers.
     */
    public final List<ContainerDependency> dependsOn() {
        return dependsOn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerName());
        hashCode = 31 * hashCode + Objects.hashCode(imageUri());
        hashCode = 31 * hashCode + Objects.hashCode(resolvedImageDigest());
        hashCode = 31 * hashCode + Objects.hashCode(memoryLimits());
        hashCode = 31 * hashCode + Objects.hashCode(portConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(cpu());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheck());
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(essential());
        hashCode = 31 * hashCode + Objects.hashCode(hasEntryPoint() ? entryPoint() : null);
        hashCode = 31 * hashCode + Objects.hashCode(workingDirectory());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDependsOn() ? dependsOn() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerDefinition)) {
            return false;
        }
        ContainerDefinition other = (ContainerDefinition) obj;
        return Objects.equals(containerName(), other.containerName()) && Objects.equals(imageUri(), other.imageUri())
                && Objects.equals(resolvedImageDigest(), other.resolvedImageDigest())
                && Objects.equals(memoryLimits(), other.memoryLimits())
                && Objects.equals(portConfiguration(), other.portConfiguration()) && Objects.equals(cpu(), other.cpu())
                && Objects.equals(healthCheck(), other.healthCheck()) && hasCommand() == other.hasCommand()
                && Objects.equals(command(), other.command()) && Objects.equals(essential(), other.essential())
                && hasEntryPoint() == other.hasEntryPoint() && Objects.equals(entryPoint(), other.entryPoint())
                && Objects.equals(workingDirectory(), other.workingDirectory()) && hasEnvironment() == other.hasEnvironment()
                && Objects.equals(environment(), other.environment()) && hasDependsOn() == other.hasDependsOn()
                && Objects.equals(dependsOn(), other.dependsOn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerDefinition").add("ContainerName", containerName()).add("ImageUri", imageUri())
                .add("ResolvedImageDigest", resolvedImageDigest()).add("MemoryLimits", memoryLimits())
                .add("PortConfiguration", portConfiguration()).add("Cpu", cpu()).add("HealthCheck", healthCheck())
                .add("Command", hasCommand() ? command() : null).add("Essential", essential())
                .add("EntryPoint", hasEntryPoint() ? entryPoint() : null).add("WorkingDirectory", workingDirectory())
                .add("Environment", hasEnvironment() ? environment() : null)
                .add("DependsOn", hasDependsOn() ? dependsOn() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerName":
            return Optional.ofNullable(clazz.cast(containerName()));
        case "ImageUri":
            return Optional.ofNullable(clazz.cast(imageUri()));
        case "ResolvedImageDigest":
            return Optional.ofNullable(clazz.cast(resolvedImageDigest()));
        case "MemoryLimits":
            return Optional.ofNullable(clazz.cast(memoryLimits()));
        case "PortConfiguration":
            return Optional.ofNullable(clazz.cast(portConfiguration()));
        case "Cpu":
            return Optional.ofNullable(clazz.cast(cpu()));
        case "HealthCheck":
            return Optional.ofNullable(clazz.cast(healthCheck()));
        case "Command":
            return Optional.ofNullable(clazz.cast(command()));
        case "Essential":
            return Optional.ofNullable(clazz.cast(essential()));
        case "EntryPoint":
            return Optional.ofNullable(clazz.cast(entryPoint()));
        case "WorkingDirectory":
            return Optional.ofNullable(clazz.cast(workingDirectory()));
        case "Environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "DependsOn":
            return Optional.ofNullable(clazz.cast(dependsOn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerDefinition, T> g) {
        return obj -> g.apply((ContainerDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerDefinition> {
        /**
         * <p>
         * The container definition identifier. Container names are unique within a container group definition.
         * </p>
         * 
         * @param containerName
         *        The container definition identifier. Container names are unique within a container group definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerName(String containerName);

        /**
         * <p>
         * The URI to the image that $short; copied and deployed to a container fleet. For a more specific identifier,
         * see <code>ResolvedImageDigest</code>.
         * </p>
         * 
         * @param imageUri
         *        The URI to the image that $short; copied and deployed to a container fleet. For a more specific
         *        identifier, see <code>ResolvedImageDigest</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUri(String imageUri);

        /**
         * <p>
         * A unique and immutable identifier for the container image that is deployed to a container fleet. The digest
         * is a SHA 256 hash of the container image manifest.
         * </p>
         * 
         * @param resolvedImageDigest
         *        A unique and immutable identifier for the container image that is deployed to a container fleet. The
         *        digest is a SHA 256 hash of the container image manifest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolvedImageDigest(String resolvedImageDigest);

        /**
         * <p>
         * The amount of memory that Amazon GameLift makes available to the container. If memory limits aren't set for
         * an individual container, the container shares the container group's total memory allocation.
         * </p>
         * <p>
         * <b>Related data type: </b> <a>ContainerGroupDefinition$TotalMemoryLimit</a>
         * </p>
         * 
         * @param memoryLimits
         *        The amount of memory that Amazon GameLift makes available to the container. If memory limits aren't
         *        set for an individual container, the container shares the container group's total memory
         *        allocation.</p>
         *        <p>
         *        <b>Related data type: </b> <a>ContainerGroupDefinition$TotalMemoryLimit</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryLimits(ContainerMemoryLimits memoryLimits);

        /**
         * <p>
         * The amount of memory that Amazon GameLift makes available to the container. If memory limits aren't set for
         * an individual container, the container shares the container group's total memory allocation.
         * </p>
         * <p>
         * <b>Related data type: </b> <a>ContainerGroupDefinition$TotalMemoryLimit</a>
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerMemoryLimits.Builder} avoiding
         * the need to create one manually via {@link ContainerMemoryLimits#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerMemoryLimits.Builder#build()} is called immediately and
         * its result is passed to {@link #memoryLimits(ContainerMemoryLimits)}.
         * 
         * @param memoryLimits
         *        a consumer that will call methods on {@link ContainerMemoryLimits.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #memoryLimits(ContainerMemoryLimits)
         */
        default Builder memoryLimits(Consumer<ContainerMemoryLimits.Builder> memoryLimits) {
            return memoryLimits(ContainerMemoryLimits.builder().applyMutation(memoryLimits).build());
        }

        /**
         * <p>
         * Defines the ports that are available to assign to processes in the container. For example, a game server
         * process requires a container port to allow game clients to connect to it. Container ports aren't directly
         * accessed by inbound traffic. Amazon GameLift maps these container ports to externally accessible connection
         * ports, which are assigned as needed from the container fleet's <code>ConnectionPortRange</code>.
         * </p>
         * 
         * @param portConfiguration
         *        Defines the ports that are available to assign to processes in the container. For example, a game
         *        server process requires a container port to allow game clients to connect to it. Container ports
         *        aren't directly accessed by inbound traffic. Amazon GameLift maps these container ports to externally
         *        accessible connection ports, which are assigned as needed from the container fleet's
         *        <code>ConnectionPortRange</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder portConfiguration(ContainerPortConfiguration portConfiguration);

        /**
         * <p>
         * Defines the ports that are available to assign to processes in the container. For example, a game server
         * process requires a container port to allow game clients to connect to it. Container ports aren't directly
         * accessed by inbound traffic. Amazon GameLift maps these container ports to externally accessible connection
         * ports, which are assigned as needed from the container fleet's <code>ConnectionPortRange</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerPortConfiguration.Builder}
         * avoiding the need to create one manually via {@link ContainerPortConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerPortConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #portConfiguration(ContainerPortConfiguration)}.
         * 
         * @param portConfiguration
         *        a consumer that will call methods on {@link ContainerPortConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #portConfiguration(ContainerPortConfiguration)
         */
        default Builder portConfiguration(Consumer<ContainerPortConfiguration.Builder> portConfiguration) {
            return portConfiguration(ContainerPortConfiguration.builder().applyMutation(portConfiguration).build());
        }

        /**
         * <p>
         * The number of CPU units that are reserved for the container. Note: 1 vCPU unit equals 1024 CPU units. If no
         * resources are reserved, the container shares the total CPU limit for the container group.
         * </p>
         * <p>
         * <b>Related data type: </b> <a>ContainerGroupDefinition$TotalCpuLimit</a>
         * </p>
         * 
         * @param cpu
         *        The number of CPU units that are reserved for the container. Note: 1 vCPU unit equals 1024 CPU units.
         *        If no resources are reserved, the container shares the total CPU limit for the container group.</p>
         *        <p>
         *        <b>Related data type: </b> <a>ContainerGroupDefinition$TotalCpuLimit</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpu(Integer cpu);

        /**
         * <p>
         * A configuration for a non-terminal health check. A container, which automatically restarts if it stops
         * functioning, also restarts if it fails this health check. If an essential container in the daemon group fails
         * a health check, the entire container group is restarted. The essential container in the replica group doesn't
         * use this health check mechanism, because the Amazon GameLift Agent automatically handles the task.
         * </p>
         * 
         * @param healthCheck
         *        A configuration for a non-terminal health check. A container, which automatically restarts if it stops
         *        functioning, also restarts if it fails this health check. If an essential container in the daemon
         *        group fails a health check, the entire container group is restarted. The essential container in the
         *        replica group doesn't use this health check mechanism, because the Amazon GameLift Agent automatically
         *        handles the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheck(ContainerHealthCheck healthCheck);

        /**
         * <p>
         * A configuration for a non-terminal health check. A container, which automatically restarts if it stops
         * functioning, also restarts if it fails this health check. If an essential container in the daemon group fails
         * a health check, the entire container group is restarted. The essential container in the replica group doesn't
         * use this health check mechanism, because the Amazon GameLift Agent automatically handles the task.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerHealthCheck.Builder} avoiding
         * the need to create one manually via {@link ContainerHealthCheck#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerHealthCheck.Builder#build()} is called immediately and
         * its result is passed to {@link #healthCheck(ContainerHealthCheck)}.
         * 
         * @param healthCheck
         *        a consumer that will call methods on {@link ContainerHealthCheck.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #healthCheck(ContainerHealthCheck)
         */
        default Builder healthCheck(Consumer<ContainerHealthCheck.Builder> healthCheck) {
            return healthCheck(ContainerHealthCheck.builder().applyMutation(healthCheck).build());
        }

        /**
         * <p>
         * A command that's passed to the container on startup. Each argument for the command is an additional string in
         * the array. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
         * >ContainerDefinition::command</a> parameter in the <i>Amazon Elastic Container Service API reference.</i>
         * </p>
         * 
         * @param command
         *        A command that's passed to the container on startup. Each argument for the command is an additional
         *        string in the array. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
         *        >ContainerDefinition::command</a> parameter in the <i>Amazon Elastic Container Service API
         *        reference.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * A command that's passed to the container on startup. Each argument for the command is an additional string in
         * the array. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
         * >ContainerDefinition::command</a> parameter in the <i>Amazon Elastic Container Service API reference.</i>
         * </p>
         * 
         * @param command
         *        A command that's passed to the container on startup. Each argument for the command is an additional
         *        string in the array. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-command"
         *        >ContainerDefinition::command</a> parameter in the <i>Amazon Elastic Container Service API
         *        reference.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * Indicates whether the container is vital to the container group. If an essential container fails, the entire
         * container group is restarted.
         * </p>
         * 
         * @param essential
         *        Indicates whether the container is vital to the container group. If an essential container fails, the
         *        entire container group is restarted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder essential(Boolean essential);

        /**
         * <p>
         * The entry point that's passed to the container on startup. If there are multiple arguments, each argument is
         * an additional string in the array. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
         * >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param entryPoint
         *        The entry point that's passed to the container on startup. If there are multiple arguments, each
         *        argument is an additional string in the array. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
         *        >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryPoint(Collection<String> entryPoint);

        /**
         * <p>
         * The entry point that's passed to the container on startup. If there are multiple arguments, each argument is
         * an additional string in the array. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
         * >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param entryPoint
         *        The entry point that's passed to the container on startup. If there are multiple arguments, each
         *        argument is an additional string in the array. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-entryPoint"
         *        >ContainerDefinition::entryPoint</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryPoint(String... entryPoint);

        /**
         * <p>
         * The directory in the container where commands are run. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-workingDirectory"
         * >ContainerDefinition::workingDirectory</a> parameter in the <i>Amazon Elastic Container Service API
         * Reference</i>.
         * </p>
         * 
         * @param workingDirectory
         *        The directory in the container where commands are run. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-workingDirectory"
         *        >ContainerDefinition::workingDirectory</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workingDirectory(String workingDirectory);

        /**
         * <p>
         * A set of environment variables that's passed to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param environment
         *        A set of environment variables that's passed to the container on startup. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         *        >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<ContainerEnvironment> environment);

        /**
         * <p>
         * A set of environment variables that's passed to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param environment
         *        A set of environment variables that's passed to the container on startup. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         *        >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(ContainerEnvironment... environment);

        /**
         * <p>
         * A set of environment variables that's passed to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder#build()} is called
         * immediately and its result is passed to {@link #environment(List<ContainerEnvironment>)}.
         * 
         * @param environment
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(java.util.Collection<ContainerEnvironment>)
         */
        Builder environment(Consumer<ContainerEnvironment.Builder>... environment);

        /**
         * <p>
         * Indicates that the container relies on the status of other containers in the same container group during its
         * startup and shutdown sequences. A container might have dependencies on multiple containers.
         * </p>
         * 
         * @param dependsOn
         *        Indicates that the container relies on the status of other containers in the same container group
         *        during its startup and shutdown sequences. A container might have dependencies on multiple containers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(Collection<ContainerDependency> dependsOn);

        /**
         * <p>
         * Indicates that the container relies on the status of other containers in the same container group during its
         * startup and shutdown sequences. A container might have dependencies on multiple containers.
         * </p>
         * 
         * @param dependsOn
         *        Indicates that the container relies on the status of other containers in the same container group
         *        during its startup and shutdown sequences. A container might have dependencies on multiple containers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(ContainerDependency... dependsOn);

        /**
         * <p>
         * Indicates that the container relies on the status of other containers in the same container group during its
         * startup and shutdown sequences. A container might have dependencies on multiple containers.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder#build()} is called
         * immediately and its result is passed to {@link #dependsOn(List<ContainerDependency>)}.
         * 
         * @param dependsOn
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dependsOn(java.util.Collection<ContainerDependency>)
         */
        Builder dependsOn(Consumer<ContainerDependency.Builder>... dependsOn);
    }

    static final class BuilderImpl implements Builder {
        private String containerName;

        private String imageUri;

        private String resolvedImageDigest;

        private ContainerMemoryLimits memoryLimits;

        private ContainerPortConfiguration portConfiguration;

        private Integer cpu;

        private ContainerHealthCheck healthCheck;

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private Boolean essential;

        private List<String> entryPoint = DefaultSdkAutoConstructList.getInstance();

        private String workingDirectory;

        private List<ContainerEnvironment> environment = DefaultSdkAutoConstructList.getInstance();

        private List<ContainerDependency> dependsOn = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerDefinition model) {
            containerName(model.containerName);
            imageUri(model.imageUri);
            resolvedImageDigest(model.resolvedImageDigest);
            memoryLimits(model.memoryLimits);
            portConfiguration(model.portConfiguration);
            cpu(model.cpu);
            healthCheck(model.healthCheck);
            command(model.command);
            essential(model.essential);
            entryPoint(model.entryPoint);
            workingDirectory(model.workingDirectory);
            environment(model.environment);
            dependsOn(model.dependsOn);
        }

        public final String getContainerName() {
            return containerName;
        }

        public final void setContainerName(String containerName) {
            this.containerName = containerName;
        }

        @Override
        public final Builder containerName(String containerName) {
            this.containerName = containerName;
            return this;
        }

        public final String getImageUri() {
            return imageUri;
        }

        public final void setImageUri(String imageUri) {
            this.imageUri = imageUri;
        }

        @Override
        public final Builder imageUri(String imageUri) {
            this.imageUri = imageUri;
            return this;
        }

        public final String getResolvedImageDigest() {
            return resolvedImageDigest;
        }

        public final void setResolvedImageDigest(String resolvedImageDigest) {
            this.resolvedImageDigest = resolvedImageDigest;
        }

        @Override
        public final Builder resolvedImageDigest(String resolvedImageDigest) {
            this.resolvedImageDigest = resolvedImageDigest;
            return this;
        }

        public final ContainerMemoryLimits.Builder getMemoryLimits() {
            return memoryLimits != null ? memoryLimits.toBuilder() : null;
        }

        public final void setMemoryLimits(ContainerMemoryLimits.BuilderImpl memoryLimits) {
            this.memoryLimits = memoryLimits != null ? memoryLimits.build() : null;
        }

        @Override
        public final Builder memoryLimits(ContainerMemoryLimits memoryLimits) {
            this.memoryLimits = memoryLimits;
            return this;
        }

        public final ContainerPortConfiguration.Builder getPortConfiguration() {
            return portConfiguration != null ? portConfiguration.toBuilder() : null;
        }

        public final void setPortConfiguration(ContainerPortConfiguration.BuilderImpl portConfiguration) {
            this.portConfiguration = portConfiguration != null ? portConfiguration.build() : null;
        }

        @Override
        public final Builder portConfiguration(ContainerPortConfiguration portConfiguration) {
            this.portConfiguration = portConfiguration;
            return this;
        }

        public final Integer getCpu() {
            return cpu;
        }

        public final void setCpu(Integer cpu) {
            this.cpu = cpu;
        }

        @Override
        public final Builder cpu(Integer cpu) {
            this.cpu = cpu;
            return this;
        }

        public final ContainerHealthCheck.Builder getHealthCheck() {
            return healthCheck != null ? healthCheck.toBuilder() : null;
        }

        public final void setHealthCheck(ContainerHealthCheck.BuilderImpl healthCheck) {
            this.healthCheck = healthCheck != null ? healthCheck.build() : null;
        }

        @Override
        public final Builder healthCheck(ContainerHealthCheck healthCheck) {
            this.healthCheck = healthCheck;
            return this;
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        public final void setCommand(Collection<String> command) {
            this.command = ContainerCommandStringListCopier.copy(command);
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = ContainerCommandStringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final Boolean getEssential() {
            return essential;
        }

        public final void setEssential(Boolean essential) {
            this.essential = essential;
        }

        @Override
        public final Builder essential(Boolean essential) {
            this.essential = essential;
            return this;
        }

        public final Collection<String> getEntryPoint() {
            if (entryPoint instanceof SdkAutoConstructList) {
                return null;
            }
            return entryPoint;
        }

        public final void setEntryPoint(Collection<String> entryPoint) {
            this.entryPoint = ContainerEntryPointListCopier.copy(entryPoint);
        }

        @Override
        public final Builder entryPoint(Collection<String> entryPoint) {
            this.entryPoint = ContainerEntryPointListCopier.copy(entryPoint);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entryPoint(String... entryPoint) {
            entryPoint(Arrays.asList(entryPoint));
            return this;
        }

        public final String getWorkingDirectory() {
            return workingDirectory;
        }

        public final void setWorkingDirectory(String workingDirectory) {
            this.workingDirectory = workingDirectory;
        }

        @Override
        public final Builder workingDirectory(String workingDirectory) {
            this.workingDirectory = workingDirectory;
            return this;
        }

        public final List<ContainerEnvironment.Builder> getEnvironment() {
            List<ContainerEnvironment.Builder> result = ContainerEnvironmentListCopier.copyToBuilder(this.environment);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnvironment(Collection<ContainerEnvironment.BuilderImpl> environment) {
            this.environment = ContainerEnvironmentListCopier.copyFromBuilder(environment);
        }

        @Override
        public final Builder environment(Collection<ContainerEnvironment> environment) {
            this.environment = ContainerEnvironmentListCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(ContainerEnvironment... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(Consumer<ContainerEnvironment.Builder>... environment) {
            environment(Stream.of(environment).map(c -> ContainerEnvironment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ContainerDependency.Builder> getDependsOn() {
            List<ContainerDependency.Builder> result = ContainerDependencyListCopier.copyToBuilder(this.dependsOn);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDependsOn(Collection<ContainerDependency.BuilderImpl> dependsOn) {
            this.dependsOn = ContainerDependencyListCopier.copyFromBuilder(dependsOn);
        }

        @Override
        public final Builder dependsOn(Collection<ContainerDependency> dependsOn) {
            this.dependsOn = ContainerDependencyListCopier.copy(dependsOn);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(ContainerDependency... dependsOn) {
            dependsOn(Arrays.asList(dependsOn));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(Consumer<ContainerDependency.Builder>... dependsOn) {
            dependsOn(Stream.of(dependsOn).map(c -> ContainerDependency.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ContainerDefinition build() {
            return new ContainerDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
