/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>This data type is used with the Amazon GameLift containers feature, which is currently in public preview.</b>
 * </p>
 * <p>
 * The properties that describe a container group resource. Container group definition properties can't be updated. To
 * change a property, create a new container group definition.
 * </p>
 * <p>
 * <b>Used with:</b> <a>CreateContainerGroupDefinition</a>
 * </p>
 * <p>
 * <b>Returned by:</b> <a>DescribeContainerGroupDefinition</a>, <a>ListContainerGroupDefinitions</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerGroupDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerGroupDefinition.Builder, ContainerGroupDefinition> {
    private static final SdkField<String> CONTAINER_GROUP_DEFINITION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ContainerGroupDefinitionArn")
            .getter(getter(ContainerGroupDefinition::containerGroupDefinitionArn))
            .setter(setter(Builder::containerGroupDefinitionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerGroupDefinitionArn")
                    .build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(ContainerGroupDefinition::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> OPERATING_SYSTEM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperatingSystem").getter(getter(ContainerGroupDefinition::operatingSystemAsString))
            .setter(setter(Builder::operatingSystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingSystem").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ContainerGroupDefinition::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> SCHEDULING_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SchedulingStrategy").getter(getter(ContainerGroupDefinition::schedulingStrategyAsString))
            .setter(setter(Builder::schedulingStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchedulingStrategy").build())
            .build();

    private static final SdkField<Integer> TOTAL_MEMORY_LIMIT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TotalMemoryLimit").getter(getter(ContainerGroupDefinition::totalMemoryLimit))
            .setter(setter(Builder::totalMemoryLimit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalMemoryLimit").build()).build();

    private static final SdkField<Integer> TOTAL_CPU_LIMIT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TotalCpuLimit").getter(getter(ContainerGroupDefinition::totalCpuLimit))
            .setter(setter(Builder::totalCpuLimit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalCpuLimit").build()).build();

    private static final SdkField<List<ContainerDefinition>> CONTAINER_DEFINITIONS_FIELD = SdkField
            .<List<ContainerDefinition>> builder(MarshallingType.LIST)
            .memberName("ContainerDefinitions")
            .getter(getter(ContainerGroupDefinition::containerDefinitions))
            .setter(setter(Builder::containerDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ContainerGroupDefinition::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusReason").getter(getter(ContainerGroupDefinition::statusReason))
            .setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CONTAINER_GROUP_DEFINITION_ARN_FIELD, CREATION_TIME_FIELD, OPERATING_SYSTEM_FIELD, NAME_FIELD,
            SCHEDULING_STRATEGY_FIELD, TOTAL_MEMORY_LIMIT_FIELD, TOTAL_CPU_LIMIT_FIELD, CONTAINER_DEFINITIONS_FIELD,
            STATUS_FIELD, STATUS_REASON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String containerGroupDefinitionArn;

    private final Instant creationTime;

    private final String operatingSystem;

    private final String name;

    private final String schedulingStrategy;

    private final Integer totalMemoryLimit;

    private final Integer totalCpuLimit;

    private final List<ContainerDefinition> containerDefinitions;

    private final String status;

    private final String statusReason;

    private ContainerGroupDefinition(BuilderImpl builder) {
        this.containerGroupDefinitionArn = builder.containerGroupDefinitionArn;
        this.creationTime = builder.creationTime;
        this.operatingSystem = builder.operatingSystem;
        this.name = builder.name;
        this.schedulingStrategy = builder.schedulingStrategy;
        this.totalMemoryLimit = builder.totalMemoryLimit;
        this.totalCpuLimit = builder.totalCpuLimit;
        this.containerDefinitions = builder.containerDefinitions;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * that is assigned to an Amazon GameLift <code>ContainerGroupDefinition</code> resource. It uniquely identifies the
     * resource across all Amazon Web Services Regions. Format is
     * <code>arn:aws:gamelift:&lt;region&gt;::containergroupdefinition/[container group definition name]</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to an
     *         Amazon GameLift <code>ContainerGroupDefinition</code> resource. It uniquely identifies the resource
     *         across all Amazon Web Services Regions. Format is
     *         <code>arn:aws:gamelift:&lt;region&gt;::containergroupdefinition/[container group definition name]</code>.
     */
    public final String containerGroupDefinitionArn() {
        return containerGroupDefinitionArn;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The platform required for all containers in the container group definition.
     * </p>
     * <note>
     * <p>
     * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted on
     * AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK 5.x, and then
     * deploy to AL2023 instances. See <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html"> Migrate to
     * Amazon GameLift server SDK version 5.</a>
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link ContainerOperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return The platform required for all containers in the container group definition.</p> <note>
     *         <p>
     *         Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     *         href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
     *         hosted on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK
     *         5.x, and then deploy to AL2023 instances. See <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
     *         Migrate to Amazon GameLift server SDK version 5.</a>
     *         </p>
     * @see ContainerOperatingSystem
     */
    public final ContainerOperatingSystem operatingSystem() {
        return ContainerOperatingSystem.fromValue(operatingSystem);
    }

    /**
     * <p>
     * The platform required for all containers in the container group definition.
     * </p>
     * <note>
     * <p>
     * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted on
     * AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK 5.x, and then
     * deploy to AL2023 instances. See <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html"> Migrate to
     * Amazon GameLift server SDK version 5.</a>
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link ContainerOperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return The platform required for all containers in the container group definition.</p> <note>
     *         <p>
     *         Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     *         href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
     *         hosted on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK
     *         5.x, and then deploy to AL2023 instances. See <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
     *         Migrate to Amazon GameLift server SDK version 5.</a>
     *         </p>
     * @see ContainerOperatingSystem
     */
    public final String operatingSystemAsString() {
        return operatingSystem;
    }

    /**
     * <p>
     * A descriptive identifier for the container group definition. The name value is unique in an Amazon Web Services
     * Region.
     * </p>
     * 
     * @return A descriptive identifier for the container group definition. The name value is unique in an Amazon Web
     *         Services Region.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The method for deploying the container group across fleet instances. A replica container group might have
     * multiple copies on each fleet instance. A daemon container group maintains only one copy per fleet instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #schedulingStrategy} will return {@link ContainerSchedulingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #schedulingStrategyAsString}.
     * </p>
     * 
     * @return The method for deploying the container group across fleet instances. A replica container group might have
     *         multiple copies on each fleet instance. A daemon container group maintains only one copy per fleet
     *         instance.
     * @see ContainerSchedulingStrategy
     */
    public final ContainerSchedulingStrategy schedulingStrategy() {
        return ContainerSchedulingStrategy.fromValue(schedulingStrategy);
    }

    /**
     * <p>
     * The method for deploying the container group across fleet instances. A replica container group might have
     * multiple copies on each fleet instance. A daemon container group maintains only one copy per fleet instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #schedulingStrategy} will return {@link ContainerSchedulingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #schedulingStrategyAsString}.
     * </p>
     * 
     * @return The method for deploying the container group across fleet instances. A replica container group might have
     *         multiple copies on each fleet instance. A daemon container group maintains only one copy per fleet
     *         instance.
     * @see ContainerSchedulingStrategy
     */
    public final String schedulingStrategyAsString() {
        return schedulingStrategy;
    }

    /**
     * <p>
     * The amount of memory (in MiB) on a fleet instance to allocate for the container group. All containers in the
     * group share these resources.
     * </p>
     * <p>
     * You can set additional limits for each <a>ContainerDefinition</a> in the group. If individual containers have
     * limits, this value must meet the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Equal to or greater than the sum of all container-specific soft memory limits in the group.
     * </p>
     * </li>
     * <li>
     * <p>
     * Equal to or greater than any container-specific hard limits in the group.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more details on memory allocation, see the <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-design-fleet">Container fleet design
     * guide</a>.
     * </p>
     * 
     * @return The amount of memory (in MiB) on a fleet instance to allocate for the container group. All containers in
     *         the group share these resources. </p>
     *         <p>
     *         You can set additional limits for each <a>ContainerDefinition</a> in the group. If individual containers
     *         have limits, this value must meet the following requirements:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Equal to or greater than the sum of all container-specific soft memory limits in the group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Equal to or greater than any container-specific hard limits in the group.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more details on memory allocation, see the <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-design-fleet">Container fleet
     *         design guide</a>.
     */
    public final Integer totalMemoryLimit() {
        return totalMemoryLimit;
    }

    /**
     * <p>
     * The amount of CPU units on a fleet instance to allocate for the container group. All containers in the group
     * share these resources. This property is an integer value in CPU units (1 vCPU is equal to 1024 CPU units).
     * </p>
     * <p>
     * You can set additional limits for each <a>ContainerDefinition</a> in the group. If individual containers have
     * limits, this value must be equal to or greater than the sum of all container-specific CPU limits in the group.
     * </p>
     * <p>
     * For more details on memory allocation, see the <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-design-fleet">Container fleet design
     * guide</a>.
     * </p>
     * 
     * @return The amount of CPU units on a fleet instance to allocate for the container group. All containers in the
     *         group share these resources. This property is an integer value in CPU units (1 vCPU is equal to 1024 CPU
     *         units). </p>
     *         <p>
     *         You can set additional limits for each <a>ContainerDefinition</a> in the group. If individual containers
     *         have limits, this value must be equal to or greater than the sum of all container-specific CPU limits in
     *         the group.
     *         </p>
     *         <p>
     *         For more details on memory allocation, see the <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-design-fleet">Container fleet
     *         design guide</a>.
     */
    public final Integer totalCpuLimit() {
        return totalCpuLimit;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContainerDefinitions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContainerDefinitions() {
        return containerDefinitions != null && !(containerDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of container definitions that are included in the container group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContainerDefinitions} method.
     * </p>
     * 
     * @return The set of container definitions that are included in the container group.
     */
    public final List<ContainerDefinition> containerDefinitions() {
        return containerDefinitions;
    }

    /**
     * <p>
     * Current status of the container group definition resource. Values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that are
     * defined in the group. While in this state, the resource can't be used to create a container fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined in the
     * group. You can use a container group definition in this status to create a container fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For more
     * details on the cause of the failure, see <code>StatusReason</code>. A container group definition resource in
     * failed status will be deleted within a few minutes.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ContainerGroupDefinitionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the container group definition resource. Values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that
     *         are defined in the group. While in this state, the resource can't be used to create a container fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined
     *         in the group. You can use a container group definition in this status to create a container fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For
     *         more details on the cause of the failure, see <code>StatusReason</code>. A container group definition
     *         resource in failed status will be deleted within a few minutes.
     *         </p>
     *         </li>
     * @see ContainerGroupDefinitionStatus
     */
    public final ContainerGroupDefinitionStatus status() {
        return ContainerGroupDefinitionStatus.fromValue(status);
    }

    /**
     * <p>
     * Current status of the container group definition resource. Values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that are
     * defined in the group. While in this state, the resource can't be used to create a container fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined in the
     * group. You can use a container group definition in this status to create a container fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For more
     * details on the cause of the failure, see <code>StatusReason</code>. A container group definition resource in
     * failed status will be deleted within a few minutes.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ContainerGroupDefinitionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the container group definition resource. Values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that
     *         are defined in the group. While in this state, the resource can't be used to create a container fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined
     *         in the group. You can use a container group definition in this status to create a container fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For
     *         more details on the cause of the failure, see <code>StatusReason</code>. A container group definition
     *         resource in failed status will be deleted within a few minutes.
     *         </p>
     *         </li>
     * @see ContainerGroupDefinitionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Additional information about a container group definition that's in <code>FAILED</code> status. Possible reasons
     * include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An internal issue prevented Amazon GameLift from creating the container group definition resource. Delete the
     * failed resource and call <a>CreateContainerGroupDefinition</a>again.
     * </p>
     * </li>
     * <li>
     * <p>
     * An access-denied message means that you don't have permissions to access the container image on ECR. See <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-iam-policy-examples.html"> IAM
     * permission examples</a> for help setting up required IAM permissions for Amazon GameLift.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>ImageUri</code> value for at least one of the containers in the container group definition was invalid
     * or not found in the current Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * At least one of the container images referenced in the container group definition exceeds the allowed size. For
     * size limits, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html"> Amazon GameLift endpoints
     * and quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * At least one of the container images referenced in the container group definition uses a different operating
     * system than the one defined for the container group.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Additional information about a container group definition that's in <code>FAILED</code> status. Possible
     *         reasons include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         An internal issue prevented Amazon GameLift from creating the container group definition resource. Delete
     *         the failed resource and call <a>CreateContainerGroupDefinition</a>again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An access-denied message means that you don't have permissions to access the container image on ECR. See
     *         <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-iam-policy-examples.html">
     *         IAM permission examples</a> for help setting up required IAM permissions for Amazon GameLift.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>ImageUri</code> value for at least one of the containers in the container group definition was
     *         invalid or not found in the current Amazon Web Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         At least one of the container images referenced in the container group definition exceeds the allowed
     *         size. For size limits, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html"> Amazon
     *         GameLift endpoints and quotas</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         At least one of the container images referenced in the container group definition uses a different
     *         operating system than the one defined for the container group.
     *         </p>
     *         </li>
     */
    public final String statusReason() {
        return statusReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerGroupDefinitionArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystemAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(schedulingStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(totalMemoryLimit());
        hashCode = 31 * hashCode + Objects.hashCode(totalCpuLimit());
        hashCode = 31 * hashCode + Objects.hashCode(hasContainerDefinitions() ? containerDefinitions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerGroupDefinition)) {
            return false;
        }
        ContainerGroupDefinition other = (ContainerGroupDefinition) obj;
        return Objects.equals(containerGroupDefinitionArn(), other.containerGroupDefinitionArn())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(operatingSystemAsString(), other.operatingSystemAsString())
                && Objects.equals(name(), other.name())
                && Objects.equals(schedulingStrategyAsString(), other.schedulingStrategyAsString())
                && Objects.equals(totalMemoryLimit(), other.totalMemoryLimit())
                && Objects.equals(totalCpuLimit(), other.totalCpuLimit())
                && hasContainerDefinitions() == other.hasContainerDefinitions()
                && Objects.equals(containerDefinitions(), other.containerDefinitions())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerGroupDefinition").add("ContainerGroupDefinitionArn", containerGroupDefinitionArn())
                .add("CreationTime", creationTime()).add("OperatingSystem", operatingSystemAsString()).add("Name", name())
                .add("SchedulingStrategy", schedulingStrategyAsString()).add("TotalMemoryLimit", totalMemoryLimit())
                .add("TotalCpuLimit", totalCpuLimit())
                .add("ContainerDefinitions", hasContainerDefinitions() ? containerDefinitions() : null)
                .add("Status", statusAsString()).add("StatusReason", statusReason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerGroupDefinitionArn":
            return Optional.ofNullable(clazz.cast(containerGroupDefinitionArn()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "OperatingSystem":
            return Optional.ofNullable(clazz.cast(operatingSystemAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "SchedulingStrategy":
            return Optional.ofNullable(clazz.cast(schedulingStrategyAsString()));
        case "TotalMemoryLimit":
            return Optional.ofNullable(clazz.cast(totalMemoryLimit()));
        case "TotalCpuLimit":
            return Optional.ofNullable(clazz.cast(totalCpuLimit()));
        case "ContainerDefinitions":
            return Optional.ofNullable(clazz.cast(containerDefinitions()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerGroupDefinition, T> g) {
        return obj -> g.apply((ContainerGroupDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerGroupDefinition> {
        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to an
         * Amazon GameLift <code>ContainerGroupDefinition</code> resource. It uniquely identifies the resource across
         * all Amazon Web Services Regions. Format is
         * <code>arn:aws:gamelift:&lt;region&gt;::containergroupdefinition/[container group definition name]</code>.
         * </p>
         * 
         * @param containerGroupDefinitionArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        an Amazon GameLift <code>ContainerGroupDefinition</code> resource. It uniquely identifies the resource
         *        across all Amazon Web Services Regions. Format is
         *        <code>arn:aws:gamelift:&lt;region&gt;::containergroupdefinition/[container group definition name]</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerGroupDefinitionArn(String containerGroupDefinitionArn);

        /**
         * <p>
         * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param creationTime
         *        A time stamp indicating when this data object was created. Format is a number expressed in Unix time
         *        as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The platform required for all containers in the container group definition.
         * </p>
         * <note>
         * <p>
         * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted
         * on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK 5.x, and
         * then deploy to AL2023 instances. See <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         * Migrate to Amazon GameLift server SDK version 5.</a>
         * </p>
         * </note>
         * 
         * @param operatingSystem
         *        The platform required for all containers in the container group definition.</p> <note>
         *        <p>
         *        Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         *        href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
         *        hosted on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server
         *        SDK 5.x, and then deploy to AL2023 instances. See <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         *        Migrate to Amazon GameLift server SDK version 5.</a>
         *        </p>
         * @see ContainerOperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerOperatingSystem
         */
        Builder operatingSystem(String operatingSystem);

        /**
         * <p>
         * The platform required for all containers in the container group definition.
         * </p>
         * <note>
         * <p>
         * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted
         * on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK 5.x, and
         * then deploy to AL2023 instances. See <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         * Migrate to Amazon GameLift server SDK version 5.</a>
         * </p>
         * </note>
         * 
         * @param operatingSystem
         *        The platform required for all containers in the container group definition.</p> <note>
         *        <p>
         *        Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         *        href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
         *        hosted on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server
         *        SDK 5.x, and then deploy to AL2023 instances. See <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         *        Migrate to Amazon GameLift server SDK version 5.</a>
         *        </p>
         * @see ContainerOperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerOperatingSystem
         */
        Builder operatingSystem(ContainerOperatingSystem operatingSystem);

        /**
         * <p>
         * A descriptive identifier for the container group definition. The name value is unique in an Amazon Web
         * Services Region.
         * </p>
         * 
         * @param name
         *        A descriptive identifier for the container group definition. The name value is unique in an Amazon Web
         *        Services Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The method for deploying the container group across fleet instances. A replica container group might have
         * multiple copies on each fleet instance. A daemon container group maintains only one copy per fleet instance.
         * </p>
         * 
         * @param schedulingStrategy
         *        The method for deploying the container group across fleet instances. A replica container group might
         *        have multiple copies on each fleet instance. A daemon container group maintains only one copy per
         *        fleet instance.
         * @see ContainerSchedulingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerSchedulingStrategy
         */
        Builder schedulingStrategy(String schedulingStrategy);

        /**
         * <p>
         * The method for deploying the container group across fleet instances. A replica container group might have
         * multiple copies on each fleet instance. A daemon container group maintains only one copy per fleet instance.
         * </p>
         * 
         * @param schedulingStrategy
         *        The method for deploying the container group across fleet instances. A replica container group might
         *        have multiple copies on each fleet instance. A daemon container group maintains only one copy per
         *        fleet instance.
         * @see ContainerSchedulingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerSchedulingStrategy
         */
        Builder schedulingStrategy(ContainerSchedulingStrategy schedulingStrategy);

        /**
         * <p>
         * The amount of memory (in MiB) on a fleet instance to allocate for the container group. All containers in the
         * group share these resources.
         * </p>
         * <p>
         * You can set additional limits for each <a>ContainerDefinition</a> in the group. If individual containers have
         * limits, this value must meet the following requirements:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Equal to or greater than the sum of all container-specific soft memory limits in the group.
         * </p>
         * </li>
         * <li>
         * <p>
         * Equal to or greater than any container-specific hard limits in the group.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more details on memory allocation, see the <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-design-fleet">Container fleet
         * design guide</a>.
         * </p>
         * 
         * @param totalMemoryLimit
         *        The amount of memory (in MiB) on a fleet instance to allocate for the container group. All containers
         *        in the group share these resources. </p>
         *        <p>
         *        You can set additional limits for each <a>ContainerDefinition</a> in the group. If individual
         *        containers have limits, this value must meet the following requirements:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Equal to or greater than the sum of all container-specific soft memory limits in the group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Equal to or greater than any container-specific hard limits in the group.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more details on memory allocation, see the <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-design-fleet">Container
         *        fleet design guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalMemoryLimit(Integer totalMemoryLimit);

        /**
         * <p>
         * The amount of CPU units on a fleet instance to allocate for the container group. All containers in the group
         * share these resources. This property is an integer value in CPU units (1 vCPU is equal to 1024 CPU units).
         * </p>
         * <p>
         * You can set additional limits for each <a>ContainerDefinition</a> in the group. If individual containers have
         * limits, this value must be equal to or greater than the sum of all container-specific CPU limits in the
         * group.
         * </p>
         * <p>
         * For more details on memory allocation, see the <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-design-fleet">Container fleet
         * design guide</a>.
         * </p>
         * 
         * @param totalCpuLimit
         *        The amount of CPU units on a fleet instance to allocate for the container group. All containers in the
         *        group share these resources. This property is an integer value in CPU units (1 vCPU is equal to 1024
         *        CPU units). </p>
         *        <p>
         *        You can set additional limits for each <a>ContainerDefinition</a> in the group. If individual
         *        containers have limits, this value must be equal to or greater than the sum of all container-specific
         *        CPU limits in the group.
         *        </p>
         *        <p>
         *        For more details on memory allocation, see the <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/containers-design-fleet">Container
         *        fleet design guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalCpuLimit(Integer totalCpuLimit);

        /**
         * <p>
         * The set of container definitions that are included in the container group.
         * </p>
         * 
         * @param containerDefinitions
         *        The set of container definitions that are included in the container group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerDefinitions(Collection<ContainerDefinition> containerDefinitions);

        /**
         * <p>
         * The set of container definitions that are included in the container group.
         * </p>
         * 
         * @param containerDefinitions
         *        The set of container definitions that are included in the container group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerDefinitions(ContainerDefinition... containerDefinitions);

        /**
         * <p>
         * The set of container definitions that are included in the container group.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerDefinition.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.ContainerDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerDefinition.Builder#build()} is called
         * immediately and its result is passed to {@link #containerDefinitions(List<ContainerDefinition>)}.
         * 
         * @param containerDefinitions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #containerDefinitions(java.util.Collection<ContainerDefinition>)
         */
        Builder containerDefinitions(Consumer<ContainerDefinition.Builder>... containerDefinitions);

        /**
         * <p>
         * Current status of the container group definition resource. Values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that are
         * defined in the group. While in this state, the resource can't be used to create a container fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined in
         * the group. You can use a container group definition in this status to create a container fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For more
         * details on the cause of the failure, see <code>StatusReason</code>. A container group definition resource in
         * failed status will be deleted within a few minutes.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the container group definition resource. Values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images
         *        that are defined in the group. While in this state, the resource can't be used to create a container
         *        fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are
         *        defined in the group. You can use a container group definition in this status to create a container
         *        fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource.
         *        For more details on the cause of the failure, see <code>StatusReason</code>. A container group
         *        definition resource in failed status will be deleted within a few minutes.
         *        </p>
         *        </li>
         * @see ContainerGroupDefinitionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerGroupDefinitionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the container group definition resource. Values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that are
         * defined in the group. While in this state, the resource can't be used to create a container fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined in
         * the group. You can use a container group definition in this status to create a container fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For more
         * details on the cause of the failure, see <code>StatusReason</code>. A container group definition resource in
         * failed status will be deleted within a few minutes.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the container group definition resource. Values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images
         *        that are defined in the group. While in this state, the resource can't be used to create a container
         *        fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are
         *        defined in the group. You can use a container group definition in this status to create a container
         *        fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource.
         *        For more details on the cause of the failure, see <code>StatusReason</code>. A container group
         *        definition resource in failed status will be deleted within a few minutes.
         *        </p>
         *        </li>
         * @see ContainerGroupDefinitionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerGroupDefinitionStatus
         */
        Builder status(ContainerGroupDefinitionStatus status);

        /**
         * <p>
         * Additional information about a container group definition that's in <code>FAILED</code> status. Possible
         * reasons include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An internal issue prevented Amazon GameLift from creating the container group definition resource. Delete the
         * failed resource and call <a>CreateContainerGroupDefinition</a>again.
         * </p>
         * </li>
         * <li>
         * <p>
         * An access-denied message means that you don't have permissions to access the container image on ECR. See <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-iam-policy-examples.html"> IAM
         * permission examples</a> for help setting up required IAM permissions for Amazon GameLift.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>ImageUri</code> value for at least one of the containers in the container group definition was
         * invalid or not found in the current Amazon Web Services account.
         * </p>
         * </li>
         * <li>
         * <p>
         * At least one of the container images referenced in the container group definition exceeds the allowed size.
         * For size limits, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html"> Amazon GameLift
         * endpoints and quotas</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * At least one of the container images referenced in the container group definition uses a different operating
         * system than the one defined for the container group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statusReason
         *        Additional information about a container group definition that's in <code>FAILED</code> status.
         *        Possible reasons include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        An internal issue prevented Amazon GameLift from creating the container group definition resource.
         *        Delete the failed resource and call <a>CreateContainerGroupDefinition</a>again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An access-denied message means that you don't have permissions to access the container image on ECR.
         *        See <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-iam-policy-examples.html">
         *        IAM permission examples</a> for help setting up required IAM permissions for Amazon GameLift.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>ImageUri</code> value for at least one of the containers in the container group definition
         *        was invalid or not found in the current Amazon Web Services account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        At least one of the container images referenced in the container group definition exceeds the allowed
         *        size. For size limits, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html">
         *        Amazon GameLift endpoints and quotas</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        At least one of the container images referenced in the container group definition uses a different
         *        operating system than the one defined for the container group.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);
    }

    static final class BuilderImpl implements Builder {
        private String containerGroupDefinitionArn;

        private Instant creationTime;

        private String operatingSystem;

        private String name;

        private String schedulingStrategy;

        private Integer totalMemoryLimit;

        private Integer totalCpuLimit;

        private List<ContainerDefinition> containerDefinitions = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String statusReason;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerGroupDefinition model) {
            containerGroupDefinitionArn(model.containerGroupDefinitionArn);
            creationTime(model.creationTime);
            operatingSystem(model.operatingSystem);
            name(model.name);
            schedulingStrategy(model.schedulingStrategy);
            totalMemoryLimit(model.totalMemoryLimit);
            totalCpuLimit(model.totalCpuLimit);
            containerDefinitions(model.containerDefinitions);
            status(model.status);
            statusReason(model.statusReason);
        }

        public final String getContainerGroupDefinitionArn() {
            return containerGroupDefinitionArn;
        }

        public final void setContainerGroupDefinitionArn(String containerGroupDefinitionArn) {
            this.containerGroupDefinitionArn = containerGroupDefinitionArn;
        }

        @Override
        public final Builder containerGroupDefinitionArn(String containerGroupDefinitionArn) {
            this.containerGroupDefinitionArn = containerGroupDefinitionArn;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getOperatingSystem() {
            return operatingSystem;
        }

        public final void setOperatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
        }

        @Override
        public final Builder operatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
            return this;
        }

        @Override
        public final Builder operatingSystem(ContainerOperatingSystem operatingSystem) {
            this.operatingSystem(operatingSystem == null ? null : operatingSystem.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getSchedulingStrategy() {
            return schedulingStrategy;
        }

        public final void setSchedulingStrategy(String schedulingStrategy) {
            this.schedulingStrategy = schedulingStrategy;
        }

        @Override
        public final Builder schedulingStrategy(String schedulingStrategy) {
            this.schedulingStrategy = schedulingStrategy;
            return this;
        }

        @Override
        public final Builder schedulingStrategy(ContainerSchedulingStrategy schedulingStrategy) {
            this.schedulingStrategy(schedulingStrategy == null ? null : schedulingStrategy.toString());
            return this;
        }

        public final Integer getTotalMemoryLimit() {
            return totalMemoryLimit;
        }

        public final void setTotalMemoryLimit(Integer totalMemoryLimit) {
            this.totalMemoryLimit = totalMemoryLimit;
        }

        @Override
        public final Builder totalMemoryLimit(Integer totalMemoryLimit) {
            this.totalMemoryLimit = totalMemoryLimit;
            return this;
        }

        public final Integer getTotalCpuLimit() {
            return totalCpuLimit;
        }

        public final void setTotalCpuLimit(Integer totalCpuLimit) {
            this.totalCpuLimit = totalCpuLimit;
        }

        @Override
        public final Builder totalCpuLimit(Integer totalCpuLimit) {
            this.totalCpuLimit = totalCpuLimit;
            return this;
        }

        public final List<ContainerDefinition.Builder> getContainerDefinitions() {
            List<ContainerDefinition.Builder> result = ContainerDefinitionListCopier.copyToBuilder(this.containerDefinitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setContainerDefinitions(Collection<ContainerDefinition.BuilderImpl> containerDefinitions) {
            this.containerDefinitions = ContainerDefinitionListCopier.copyFromBuilder(containerDefinitions);
        }

        @Override
        public final Builder containerDefinitions(Collection<ContainerDefinition> containerDefinitions) {
            this.containerDefinitions = ContainerDefinitionListCopier.copy(containerDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerDefinitions(ContainerDefinition... containerDefinitions) {
            containerDefinitions(Arrays.asList(containerDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerDefinitions(Consumer<ContainerDefinition.Builder>... containerDefinitions) {
            containerDefinitions(Stream.of(containerDefinitions).map(c -> ContainerDefinition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ContainerGroupDefinitionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        @Override
        public ContainerGroupDefinition build() {
            return new ContainerGroupDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
