/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Ticket generated to track the progress of a matchmaking request. Each ticket is uniquely identified by a ticket ID,
 * supplied by the requester, when creating a matchmaking request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MatchmakingTicket implements SdkPojo, Serializable,
        ToCopyableBuilder<MatchmakingTicket.Builder, MatchmakingTicket> {
    private static final SdkField<String> TICKET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TicketId").getter(getter(MatchmakingTicket::ticketId)).setter(setter(Builder::ticketId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TicketId").build()).build();

    private static final SdkField<String> CONFIGURATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationName").getter(getter(MatchmakingTicket::configurationName))
            .setter(setter(Builder::configurationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationName").build()).build();

    private static final SdkField<String> CONFIGURATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationArn").getter(getter(MatchmakingTicket::configurationArn))
            .setter(setter(Builder::configurationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(MatchmakingTicket::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusReason").getter(getter(MatchmakingTicket::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage").getter(getter(MatchmakingTicket::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(MatchmakingTicket::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(MatchmakingTicket::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<List<Player>> PLAYERS_FIELD = SdkField
            .<List<Player>> builder(MarshallingType.LIST)
            .memberName("Players")
            .getter(getter(MatchmakingTicket::players))
            .setter(setter(Builder::players))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Players").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Player> builder(MarshallingType.SDK_POJO)
                                            .constructor(Player::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<GameSessionConnectionInfo> GAME_SESSION_CONNECTION_INFO_FIELD = SdkField
            .<GameSessionConnectionInfo> builder(MarshallingType.SDK_POJO).memberName("GameSessionConnectionInfo")
            .getter(getter(MatchmakingTicket::gameSessionConnectionInfo)).setter(setter(Builder::gameSessionConnectionInfo))
            .constructor(GameSessionConnectionInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionConnectionInfo").build())
            .build();

    private static final SdkField<Integer> ESTIMATED_WAIT_TIME_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("EstimatedWaitTime").getter(getter(MatchmakingTicket::estimatedWaitTime))
            .setter(setter(Builder::estimatedWaitTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EstimatedWaitTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TICKET_ID_FIELD,
            CONFIGURATION_NAME_FIELD, CONFIGURATION_ARN_FIELD, STATUS_FIELD, STATUS_REASON_FIELD, STATUS_MESSAGE_FIELD,
            START_TIME_FIELD, END_TIME_FIELD, PLAYERS_FIELD, GAME_SESSION_CONNECTION_INFO_FIELD, ESTIMATED_WAIT_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ticketId;

    private final String configurationName;

    private final String configurationArn;

    private final String status;

    private final String statusReason;

    private final String statusMessage;

    private final Instant startTime;

    private final Instant endTime;

    private final List<Player> players;

    private final GameSessionConnectionInfo gameSessionConnectionInfo;

    private final Integer estimatedWaitTime;

    private MatchmakingTicket(BuilderImpl builder) {
        this.ticketId = builder.ticketId;
        this.configurationName = builder.configurationName;
        this.configurationArn = builder.configurationArn;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.statusMessage = builder.statusMessage;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.players = builder.players;
        this.gameSessionConnectionInfo = builder.gameSessionConnectionInfo;
        this.estimatedWaitTime = builder.estimatedWaitTime;
    }

    /**
     * <p>
     * A unique identifier for a matchmaking ticket.
     * </p>
     * 
     * @return A unique identifier for a matchmaking ticket.
     */
    public final String ticketId() {
        return ticketId;
    }

    /**
     * <p>
     * Name of the matchmaking configuration that is used with this ticket. Matchmaking configurations determine how
     * players are grouped into a match and how a new game session is created for the match.
     * </p>
     * 
     * @return Name of the matchmaking configuration that is used with this ticket. Matchmaking configurations determine
     *         how players are grouped into a match and how a new game session is created for the match.
     */
    public final String configurationName() {
        return configurationName;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * associated with the GameLift matchmaking configuration resource that is used with this ticket.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
     *         GameLift matchmaking configuration resource that is used with this ticket.
     */
    public final String configurationArn() {
        return configurationArn;
    }

    /**
     * <p>
     * Current status of the matchmaking request.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>QUEUED</b> -- The matchmaking request has been received and is currently waiting to be processed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>SEARCHING</b> -- The matchmaking request is currently being processed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>REQUIRES_ACCEPTANCE</b> -- A match has been proposed and the players must accept the match. This status is
     * used only with requests that use a matchmaking configuration with a player acceptance requirement.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PLACING</b> -- The FlexMatch engine has matched players and is in the process of placing a new game session
     * for the match.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>COMPLETED</b> -- Players have been matched and a game session is ready to host the players. A ticket in this
     * state contains the necessary connection information for players.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FAILED</b> -- The matchmaking request was not completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELLED</b> -- The matchmaking request was canceled. This may be the result of a
     * <code>StopMatchmaking</code> operation or a proposed match that one or more players failed to accept.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TIMED_OUT</b> -- The matchmaking request was not successful within the duration specified in the matchmaking
     * configuration.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Matchmaking requests that fail to successfully complete (statuses FAILED, CANCELLED, TIMED_OUT) can be
     * resubmitted as new requests with new ticket IDs.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MatchmakingConfigurationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the matchmaking request.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>QUEUED</b> -- The matchmaking request has been received and is currently waiting to be processed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>SEARCHING</b> -- The matchmaking request is currently being processed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>REQUIRES_ACCEPTANCE</b> -- A match has been proposed and the players must accept the match. This
     *         status is used only with requests that use a matchmaking configuration with a player acceptance
     *         requirement.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PLACING</b> -- The FlexMatch engine has matched players and is in the process of placing a new game
     *         session for the match.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>COMPLETED</b> -- Players have been matched and a game session is ready to host the players. A ticket
     *         in this state contains the necessary connection information for players.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FAILED</b> -- The matchmaking request was not completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELLED</b> -- The matchmaking request was canceled. This may be the result of a
     *         <code>StopMatchmaking</code> operation or a proposed match that one or more players failed to accept.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TIMED_OUT</b> -- The matchmaking request was not successful within the duration specified in the
     *         matchmaking configuration.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         Matchmaking requests that fail to successfully complete (statuses FAILED, CANCELLED, TIMED_OUT) can be
     *         resubmitted as new requests with new ticket IDs.
     *         </p>
     * @see MatchmakingConfigurationStatus
     */
    public final MatchmakingConfigurationStatus status() {
        return MatchmakingConfigurationStatus.fromValue(status);
    }

    /**
     * <p>
     * Current status of the matchmaking request.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>QUEUED</b> -- The matchmaking request has been received and is currently waiting to be processed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>SEARCHING</b> -- The matchmaking request is currently being processed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>REQUIRES_ACCEPTANCE</b> -- A match has been proposed and the players must accept the match. This status is
     * used only with requests that use a matchmaking configuration with a player acceptance requirement.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PLACING</b> -- The FlexMatch engine has matched players and is in the process of placing a new game session
     * for the match.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>COMPLETED</b> -- Players have been matched and a game session is ready to host the players. A ticket in this
     * state contains the necessary connection information for players.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FAILED</b> -- The matchmaking request was not completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELLED</b> -- The matchmaking request was canceled. This may be the result of a
     * <code>StopMatchmaking</code> operation or a proposed match that one or more players failed to accept.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TIMED_OUT</b> -- The matchmaking request was not successful within the duration specified in the matchmaking
     * configuration.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Matchmaking requests that fail to successfully complete (statuses FAILED, CANCELLED, TIMED_OUT) can be
     * resubmitted as new requests with new ticket IDs.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MatchmakingConfigurationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the matchmaking request.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>QUEUED</b> -- The matchmaking request has been received and is currently waiting to be processed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>SEARCHING</b> -- The matchmaking request is currently being processed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>REQUIRES_ACCEPTANCE</b> -- A match has been proposed and the players must accept the match. This
     *         status is used only with requests that use a matchmaking configuration with a player acceptance
     *         requirement.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PLACING</b> -- The FlexMatch engine has matched players and is in the process of placing a new game
     *         session for the match.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>COMPLETED</b> -- Players have been matched and a game session is ready to host the players. A ticket
     *         in this state contains the necessary connection information for players.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FAILED</b> -- The matchmaking request was not completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELLED</b> -- The matchmaking request was canceled. This may be the result of a
     *         <code>StopMatchmaking</code> operation or a proposed match that one or more players failed to accept.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TIMED_OUT</b> -- The matchmaking request was not successful within the duration specified in the
     *         matchmaking configuration.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         Matchmaking requests that fail to successfully complete (statuses FAILED, CANCELLED, TIMED_OUT) can be
     *         resubmitted as new requests with new ticket IDs.
     *         </p>
     * @see MatchmakingConfigurationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Code to explain the current status. For example, a status reason may indicate when a ticket has returned to
     * <code>SEARCHING</code> status after a proposed match fails to receive player acceptances.
     * </p>
     * 
     * @return Code to explain the current status. For example, a status reason may indicate when a ticket has returned
     *         to <code>SEARCHING</code> status after a proposed match fails to receive player acceptances.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * Additional information about the current status.
     * </p>
     * 
     * @return Additional information about the current status.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * Time stamp indicating when this matchmaking request was received. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return Time stamp indicating when this matchmaking request was received. Format is a number expressed in Unix
     *         time as milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * Time stamp indicating when the matchmaking request stopped being processed due to successful completion, timeout,
     * or cancellation. Format is a number expressed in Unix time as milliseconds (for example
     * <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return Time stamp indicating when the matchmaking request stopped being processed due to successful completion,
     *         timeout, or cancellation. Format is a number expressed in Unix time as milliseconds (for example
     *         <code>"1469498468.057"</code>).
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the Players property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPlayers() {
        return players != null && !(players instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of <code>Player</code> objects, each representing a player to find matches for. Players are identified by a
     * unique player ID and may include latency data for use during matchmaking. If the ticket is in status
     * <code>COMPLETED</code>, the <code>Player</code> objects include the team the players were assigned to in the
     * resulting match.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlayers} method.
     * </p>
     * 
     * @return A set of <code>Player</code> objects, each representing a player to find matches for. Players are
     *         identified by a unique player ID and may include latency data for use during matchmaking. If the ticket
     *         is in status <code>COMPLETED</code>, the <code>Player</code> objects include the team the players were
     *         assigned to in the resulting match.
     */
    public final List<Player> players() {
        return players;
    }

    /**
     * <p>
     * Connection information for a new game session. Once a match is made, the FlexMatch engine creates a new game
     * session for it. This information is added to the matchmaking ticket, which you can be retrieve by calling <a
     * href=
     * "https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmaking.html">DescribeMatchmaking</a> .
     * </p>
     * 
     * @return Connection information for a new game session. Once a match is made, the FlexMatch engine creates a new
     *         game session for it. This information is added to the matchmaking ticket, which you can be retrieve by
     *         calling <a href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmaking.html">
     *         DescribeMatchmaking</a> .
     */
    public final GameSessionConnectionInfo gameSessionConnectionInfo() {
        return gameSessionConnectionInfo;
    }

    /**
     * <p>
     * Average amount of time (in seconds) that players are currently waiting for a match. If there is not enough recent
     * data, this property may be empty.
     * </p>
     * 
     * @return Average amount of time (in seconds) that players are currently waiting for a match. If there is not
     *         enough recent data, this property may be empty.
     */
    public final Integer estimatedWaitTime() {
        return estimatedWaitTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ticketId());
        hashCode = 31 * hashCode + Objects.hashCode(configurationName());
        hashCode = 31 * hashCode + Objects.hashCode(configurationArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlayers() ? players() : null);
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionConnectionInfo());
        hashCode = 31 * hashCode + Objects.hashCode(estimatedWaitTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MatchmakingTicket)) {
            return false;
        }
        MatchmakingTicket other = (MatchmakingTicket) obj;
        return Objects.equals(ticketId(), other.ticketId()) && Objects.equals(configurationName(), other.configurationName())
                && Objects.equals(configurationArn(), other.configurationArn())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason()) && Objects.equals(statusMessage(), other.statusMessage())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && hasPlayers() == other.hasPlayers() && Objects.equals(players(), other.players())
                && Objects.equals(gameSessionConnectionInfo(), other.gameSessionConnectionInfo())
                && Objects.equals(estimatedWaitTime(), other.estimatedWaitTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MatchmakingTicket").add("TicketId", ticketId()).add("ConfigurationName", configurationName())
                .add("ConfigurationArn", configurationArn()).add("Status", statusAsString()).add("StatusReason", statusReason())
                .add("StatusMessage", statusMessage()).add("StartTime", startTime()).add("EndTime", endTime())
                .add("Players", hasPlayers() ? players() : null).add("GameSessionConnectionInfo", gameSessionConnectionInfo())
                .add("EstimatedWaitTime", estimatedWaitTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TicketId":
            return Optional.ofNullable(clazz.cast(ticketId()));
        case "ConfigurationName":
            return Optional.ofNullable(clazz.cast(configurationName()));
        case "ConfigurationArn":
            return Optional.ofNullable(clazz.cast(configurationArn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "Players":
            return Optional.ofNullable(clazz.cast(players()));
        case "GameSessionConnectionInfo":
            return Optional.ofNullable(clazz.cast(gameSessionConnectionInfo()));
        case "EstimatedWaitTime":
            return Optional.ofNullable(clazz.cast(estimatedWaitTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MatchmakingTicket, T> g) {
        return obj -> g.apply((MatchmakingTicket) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MatchmakingTicket> {
        /**
         * <p>
         * A unique identifier for a matchmaking ticket.
         * </p>
         * 
         * @param ticketId
         *        A unique identifier for a matchmaking ticket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ticketId(String ticketId);

        /**
         * <p>
         * Name of the matchmaking configuration that is used with this ticket. Matchmaking configurations determine how
         * players are grouped into a match and how a new game session is created for the match.
         * </p>
         * 
         * @param configurationName
         *        Name of the matchmaking configuration that is used with this ticket. Matchmaking configurations
         *        determine how players are grouped into a match and how a new game session is created for the match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationName(String configurationName);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         * GameLift matchmaking configuration resource that is used with this ticket.
         * </p>
         * 
         * @param configurationArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         *        GameLift matchmaking configuration resource that is used with this ticket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationArn(String configurationArn);

        /**
         * <p>
         * Current status of the matchmaking request.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>QUEUED</b> -- The matchmaking request has been received and is currently waiting to be processed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>SEARCHING</b> -- The matchmaking request is currently being processed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>REQUIRES_ACCEPTANCE</b> -- A match has been proposed and the players must accept the match. This status is
         * used only with requests that use a matchmaking configuration with a player acceptance requirement.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PLACING</b> -- The FlexMatch engine has matched players and is in the process of placing a new game
         * session for the match.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>COMPLETED</b> -- Players have been matched and a game session is ready to host the players. A ticket in
         * this state contains the necessary connection information for players.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FAILED</b> -- The matchmaking request was not completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELLED</b> -- The matchmaking request was canceled. This may be the result of a
         * <code>StopMatchmaking</code> operation or a proposed match that one or more players failed to accept.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TIMED_OUT</b> -- The matchmaking request was not successful within the duration specified in the
         * matchmaking configuration.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Matchmaking requests that fail to successfully complete (statuses FAILED, CANCELLED, TIMED_OUT) can be
         * resubmitted as new requests with new ticket IDs.
         * </p>
         * </note>
         * 
         * @param status
         *        Current status of the matchmaking request.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>QUEUED</b> -- The matchmaking request has been received and is currently waiting to be processed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>SEARCHING</b> -- The matchmaking request is currently being processed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>REQUIRES_ACCEPTANCE</b> -- A match has been proposed and the players must accept the match. This
         *        status is used only with requests that use a matchmaking configuration with a player acceptance
         *        requirement.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PLACING</b> -- The FlexMatch engine has matched players and is in the process of placing a new game
         *        session for the match.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>COMPLETED</b> -- Players have been matched and a game session is ready to host the players. A
         *        ticket in this state contains the necessary connection information for players.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FAILED</b> -- The matchmaking request was not completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELLED</b> -- The matchmaking request was canceled. This may be the result of a
         *        <code>StopMatchmaking</code> operation or a proposed match that one or more players failed to accept.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TIMED_OUT</b> -- The matchmaking request was not successful within the duration specified in the
         *        matchmaking configuration.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Matchmaking requests that fail to successfully complete (statuses FAILED, CANCELLED, TIMED_OUT) can be
         *        resubmitted as new requests with new ticket IDs.
         *        </p>
         * @see MatchmakingConfigurationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MatchmakingConfigurationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the matchmaking request.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>QUEUED</b> -- The matchmaking request has been received and is currently waiting to be processed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>SEARCHING</b> -- The matchmaking request is currently being processed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>REQUIRES_ACCEPTANCE</b> -- A match has been proposed and the players must accept the match. This status is
         * used only with requests that use a matchmaking configuration with a player acceptance requirement.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PLACING</b> -- The FlexMatch engine has matched players and is in the process of placing a new game
         * session for the match.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>COMPLETED</b> -- Players have been matched and a game session is ready to host the players. A ticket in
         * this state contains the necessary connection information for players.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FAILED</b> -- The matchmaking request was not completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELLED</b> -- The matchmaking request was canceled. This may be the result of a
         * <code>StopMatchmaking</code> operation or a proposed match that one or more players failed to accept.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TIMED_OUT</b> -- The matchmaking request was not successful within the duration specified in the
         * matchmaking configuration.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Matchmaking requests that fail to successfully complete (statuses FAILED, CANCELLED, TIMED_OUT) can be
         * resubmitted as new requests with new ticket IDs.
         * </p>
         * </note>
         * 
         * @param status
         *        Current status of the matchmaking request.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>QUEUED</b> -- The matchmaking request has been received and is currently waiting to be processed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>SEARCHING</b> -- The matchmaking request is currently being processed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>REQUIRES_ACCEPTANCE</b> -- A match has been proposed and the players must accept the match. This
         *        status is used only with requests that use a matchmaking configuration with a player acceptance
         *        requirement.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PLACING</b> -- The FlexMatch engine has matched players and is in the process of placing a new game
         *        session for the match.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>COMPLETED</b> -- Players have been matched and a game session is ready to host the players. A
         *        ticket in this state contains the necessary connection information for players.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FAILED</b> -- The matchmaking request was not completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELLED</b> -- The matchmaking request was canceled. This may be the result of a
         *        <code>StopMatchmaking</code> operation or a proposed match that one or more players failed to accept.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TIMED_OUT</b> -- The matchmaking request was not successful within the duration specified in the
         *        matchmaking configuration.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Matchmaking requests that fail to successfully complete (statuses FAILED, CANCELLED, TIMED_OUT) can be
         *        resubmitted as new requests with new ticket IDs.
         *        </p>
         * @see MatchmakingConfigurationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MatchmakingConfigurationStatus
         */
        Builder status(MatchmakingConfigurationStatus status);

        /**
         * <p>
         * Code to explain the current status. For example, a status reason may indicate when a ticket has returned to
         * <code>SEARCHING</code> status after a proposed match fails to receive player acceptances.
         * </p>
         * 
         * @param statusReason
         *        Code to explain the current status. For example, a status reason may indicate when a ticket has
         *        returned to <code>SEARCHING</code> status after a proposed match fails to receive player acceptances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * Additional information about the current status.
         * </p>
         * 
         * @param statusMessage
         *        Additional information about the current status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * Time stamp indicating when this matchmaking request was received. Format is a number expressed in Unix time
         * as milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param startTime
         *        Time stamp indicating when this matchmaking request was received. Format is a number expressed in Unix
         *        time as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * Time stamp indicating when the matchmaking request stopped being processed due to successful completion,
         * timeout, or cancellation. Format is a number expressed in Unix time as milliseconds (for example
         * <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param endTime
         *        Time stamp indicating when the matchmaking request stopped being processed due to successful
         *        completion, timeout, or cancellation. Format is a number expressed in Unix time as milliseconds (for
         *        example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * A set of <code>Player</code> objects, each representing a player to find matches for. Players are identified
         * by a unique player ID and may include latency data for use during matchmaking. If the ticket is in status
         * <code>COMPLETED</code>, the <code>Player</code> objects include the team the players were assigned to in the
         * resulting match.
         * </p>
         * 
         * @param players
         *        A set of <code>Player</code> objects, each representing a player to find matches for. Players are
         *        identified by a unique player ID and may include latency data for use during matchmaking. If the
         *        ticket is in status <code>COMPLETED</code>, the <code>Player</code> objects include the team the
         *        players were assigned to in the resulting match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder players(Collection<Player> players);

        /**
         * <p>
         * A set of <code>Player</code> objects, each representing a player to find matches for. Players are identified
         * by a unique player ID and may include latency data for use during matchmaking. If the ticket is in status
         * <code>COMPLETED</code>, the <code>Player</code> objects include the team the players were assigned to in the
         * resulting match.
         * </p>
         * 
         * @param players
         *        A set of <code>Player</code> objects, each representing a player to find matches for. Players are
         *        identified by a unique player ID and may include latency data for use during matchmaking. If the
         *        ticket is in status <code>COMPLETED</code>, the <code>Player</code> objects include the team the
         *        players were assigned to in the resulting match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder players(Player... players);

        /**
         * <p>
         * A set of <code>Player</code> objects, each representing a player to find matches for. Players are identified
         * by a unique player ID and may include latency data for use during matchmaking. If the ticket is in status
         * <code>COMPLETED</code>, the <code>Player</code> objects include the team the players were assigned to in the
         * resulting match.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.Player.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.gamelift.model.Player#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.Player.Builder#build()} is called immediately and its
         * result is passed to {@link #players(List<Player>)}.
         * 
         * @param players
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.Player.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #players(java.util.Collection<Player>)
         */
        Builder players(Consumer<Player.Builder>... players);

        /**
         * <p>
         * Connection information for a new game session. Once a match is made, the FlexMatch engine creates a new game
         * session for it. This information is added to the matchmaking ticket, which you can be retrieve by calling <a
         * href
         * ="https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmaking.html">DescribeMatchmaking
         * </a> .
         * </p>
         * 
         * @param gameSessionConnectionInfo
         *        Connection information for a new game session. Once a match is made, the FlexMatch engine creates a
         *        new game session for it. This information is added to the matchmaking ticket, which you can be
         *        retrieve by calling <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmaking.html"
         *        >DescribeMatchmaking</a> .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionConnectionInfo(GameSessionConnectionInfo gameSessionConnectionInfo);

        /**
         * <p>
         * Connection information for a new game session. Once a match is made, the FlexMatch engine creates a new game
         * session for it. This information is added to the matchmaking ticket, which you can be retrieve by calling <a
         * href
         * ="https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeMatchmaking.html">DescribeMatchmaking
         * </a> .
         * </p>
         * This is a convenience method that creates an instance of the {@link GameSessionConnectionInfo.Builder}
         * avoiding the need to create one manually via {@link GameSessionConnectionInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GameSessionConnectionInfo.Builder#build()} is called immediately
         * and its result is passed to {@link #gameSessionConnectionInfo(GameSessionConnectionInfo)}.
         * 
         * @param gameSessionConnectionInfo
         *        a consumer that will call methods on {@link GameSessionConnectionInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gameSessionConnectionInfo(GameSessionConnectionInfo)
         */
        default Builder gameSessionConnectionInfo(Consumer<GameSessionConnectionInfo.Builder> gameSessionConnectionInfo) {
            return gameSessionConnectionInfo(GameSessionConnectionInfo.builder().applyMutation(gameSessionConnectionInfo).build());
        }

        /**
         * <p>
         * Average amount of time (in seconds) that players are currently waiting for a match. If there is not enough
         * recent data, this property may be empty.
         * </p>
         * 
         * @param estimatedWaitTime
         *        Average amount of time (in seconds) that players are currently waiting for a match. If there is not
         *        enough recent data, this property may be empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder estimatedWaitTime(Integer estimatedWaitTime);
    }

    static final class BuilderImpl implements Builder {
        private String ticketId;

        private String configurationName;

        private String configurationArn;

        private String status;

        private String statusReason;

        private String statusMessage;

        private Instant startTime;

        private Instant endTime;

        private List<Player> players = DefaultSdkAutoConstructList.getInstance();

        private GameSessionConnectionInfo gameSessionConnectionInfo;

        private Integer estimatedWaitTime;

        private BuilderImpl() {
        }

        private BuilderImpl(MatchmakingTicket model) {
            ticketId(model.ticketId);
            configurationName(model.configurationName);
            configurationArn(model.configurationArn);
            status(model.status);
            statusReason(model.statusReason);
            statusMessage(model.statusMessage);
            startTime(model.startTime);
            endTime(model.endTime);
            players(model.players);
            gameSessionConnectionInfo(model.gameSessionConnectionInfo);
            estimatedWaitTime(model.estimatedWaitTime);
        }

        public final String getTicketId() {
            return ticketId;
        }

        public final void setTicketId(String ticketId) {
            this.ticketId = ticketId;
        }

        @Override
        public final Builder ticketId(String ticketId) {
            this.ticketId = ticketId;
            return this;
        }

        public final String getConfigurationName() {
            return configurationName;
        }

        public final void setConfigurationName(String configurationName) {
            this.configurationName = configurationName;
        }

        @Override
        public final Builder configurationName(String configurationName) {
            this.configurationName = configurationName;
            return this;
        }

        public final String getConfigurationArn() {
            return configurationArn;
        }

        public final void setConfigurationArn(String configurationArn) {
            this.configurationArn = configurationArn;
        }

        @Override
        public final Builder configurationArn(String configurationArn) {
            this.configurationArn = configurationArn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(MatchmakingConfigurationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final List<Player.Builder> getPlayers() {
            List<Player.Builder> result = PlayerListCopier.copyToBuilder(this.players);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPlayers(Collection<Player.BuilderImpl> players) {
            this.players = PlayerListCopier.copyFromBuilder(players);
        }

        @Override
        public final Builder players(Collection<Player> players) {
            this.players = PlayerListCopier.copy(players);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder players(Player... players) {
            players(Arrays.asList(players));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder players(Consumer<Player.Builder>... players) {
            players(Stream.of(players).map(c -> Player.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final GameSessionConnectionInfo.Builder getGameSessionConnectionInfo() {
            return gameSessionConnectionInfo != null ? gameSessionConnectionInfo.toBuilder() : null;
        }

        public final void setGameSessionConnectionInfo(GameSessionConnectionInfo.BuilderImpl gameSessionConnectionInfo) {
            this.gameSessionConnectionInfo = gameSessionConnectionInfo != null ? gameSessionConnectionInfo.build() : null;
        }

        @Override
        public final Builder gameSessionConnectionInfo(GameSessionConnectionInfo gameSessionConnectionInfo) {
            this.gameSessionConnectionInfo = gameSessionConnectionInfo;
            return this;
        }

        public final Integer getEstimatedWaitTime() {
            return estimatedWaitTime;
        }

        public final void setEstimatedWaitTime(Integer estimatedWaitTime) {
            this.estimatedWaitTime = estimatedWaitTime;
        }

        @Override
        public final Builder estimatedWaitTime(Integer estimatedWaitTime) {
            this.estimatedWaitTime = estimatedWaitTime;
            return this;
        }

        @Override
        public MatchmakingTicket build() {
            return new MatchmakingTicket(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
