/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the input for a request action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateBuildRequest extends GameLiftRequest implements
        ToCopyableBuilder<CreateBuildRequest.Builder, CreateBuildRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateBuildRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateBuildRequest::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<S3Location> STORAGE_LOCATION_FIELD = SdkField.<S3Location> builder(MarshallingType.SDK_POJO)
            .getter(getter(CreateBuildRequest::storageLocation)).setter(setter(Builder::storageLocation))
            .constructor(S3Location::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageLocation").build()).build();

    private static final SdkField<String> OPERATING_SYSTEM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateBuildRequest::operatingSystemAsString)).setter(setter(Builder::operatingSystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingSystem").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VERSION_FIELD,
            STORAGE_LOCATION_FIELD, OPERATING_SYSTEM_FIELD));

    private final String name;

    private final String version;

    private final S3Location storageLocation;

    private final String operatingSystem;

    private CreateBuildRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.version = builder.version;
        this.storageLocation = builder.storageLocation;
        this.operatingSystem = builder.operatingSystem;
    }

    /**
     * <p>
     * Descriptive label that is associated with a build. Build names do not need to be unique. You can use
     * <a>UpdateBuild</a> to change this value later.
     * </p>
     * 
     * @return Descriptive label that is associated with a build. Build names do not need to be unique. You can use
     *         <a>UpdateBuild</a> to change this value later.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Version that is associated with this build. Version strings do not need to be unique. You can use
     * <a>UpdateBuild</a> to change this value later.
     * </p>
     * 
     * @return Version that is associated with this build. Version strings do not need to be unique. You can use
     *         <a>UpdateBuild</a> to change this value later.
     */
    public String version() {
        return version;
    }

    /**
     * <p>
     * Information indicating where your game build files are stored. Use this parameter only when creating a build with
     * files stored in an Amazon S3 bucket that you own. The storage location must specify an Amazon S3 bucket name and
     * key, as well as a role ARN that you set up to allow Amazon GameLift to access your Amazon S3 bucket. The S3
     * bucket must be in the same region that you want to create a new build in.
     * </p>
     * 
     * @return Information indicating where your game build files are stored. Use this parameter only when creating a
     *         build with files stored in an Amazon S3 bucket that you own. The storage location must specify an Amazon
     *         S3 bucket name and key, as well as a role ARN that you set up to allow Amazon GameLift to access your
     *         Amazon S3 bucket. The S3 bucket must be in the same region that you want to create a new build in.
     */
    public S3Location storageLocation() {
        return storageLocation;
    }

    /**
     * <p>
     * Operating system that the game server binaries are built to run on. This value determines the type of fleet
     * resources that you can use for this build. If your game build contains multiple executables, they all must run on
     * the same operating system. If an operating system is not specified when creating a build, Amazon GameLift uses
     * the default value (WINDOWS_2012). This value cannot be changed later.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Operating system that the game server binaries are built to run on. This value determines the type of
     *         fleet resources that you can use for this build. If your game build contains multiple executables, they
     *         all must run on the same operating system. If an operating system is not specified when creating a build,
     *         Amazon GameLift uses the default value (WINDOWS_2012). This value cannot be changed later.
     * @see OperatingSystem
     */
    public OperatingSystem operatingSystem() {
        return OperatingSystem.fromValue(operatingSystem);
    }

    /**
     * <p>
     * Operating system that the game server binaries are built to run on. This value determines the type of fleet
     * resources that you can use for this build. If your game build contains multiple executables, they all must run on
     * the same operating system. If an operating system is not specified when creating a build, Amazon GameLift uses
     * the default value (WINDOWS_2012). This value cannot be changed later.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Operating system that the game server binaries are built to run on. This value determines the type of
     *         fleet resources that you can use for this build. If your game build contains multiple executables, they
     *         all must run on the same operating system. If an operating system is not specified when creating a build,
     *         Amazon GameLift uses the default value (WINDOWS_2012). This value cannot be changed later.
     * @see OperatingSystem
     */
    public String operatingSystemAsString() {
        return operatingSystem;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(storageLocation());
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystemAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateBuildRequest)) {
            return false;
        }
        CreateBuildRequest other = (CreateBuildRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(version(), other.version())
                && Objects.equals(storageLocation(), other.storageLocation())
                && Objects.equals(operatingSystemAsString(), other.operatingSystemAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("CreateBuildRequest").add("Name", name()).add("Version", version())
                .add("StorageLocation", storageLocation()).add("OperatingSystem", operatingSystemAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "StorageLocation":
            return Optional.ofNullable(clazz.cast(storageLocation()));
        case "OperatingSystem":
            return Optional.ofNullable(clazz.cast(operatingSystemAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateBuildRequest, T> g) {
        return obj -> g.apply((CreateBuildRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends GameLiftRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateBuildRequest> {
        /**
         * <p>
         * Descriptive label that is associated with a build. Build names do not need to be unique. You can use
         * <a>UpdateBuild</a> to change this value later.
         * </p>
         * 
         * @param name
         *        Descriptive label that is associated with a build. Build names do not need to be unique. You can use
         *        <a>UpdateBuild</a> to change this value later.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Version that is associated with this build. Version strings do not need to be unique. You can use
         * <a>UpdateBuild</a> to change this value later.
         * </p>
         * 
         * @param version
         *        Version that is associated with this build. Version strings do not need to be unique. You can use
         *        <a>UpdateBuild</a> to change this value later.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * Information indicating where your game build files are stored. Use this parameter only when creating a build
         * with files stored in an Amazon S3 bucket that you own. The storage location must specify an Amazon S3 bucket
         * name and key, as well as a role ARN that you set up to allow Amazon GameLift to access your Amazon S3 bucket.
         * The S3 bucket must be in the same region that you want to create a new build in.
         * </p>
         * 
         * @param storageLocation
         *        Information indicating where your game build files are stored. Use this parameter only when creating a
         *        build with files stored in an Amazon S3 bucket that you own. The storage location must specify an
         *        Amazon S3 bucket name and key, as well as a role ARN that you set up to allow Amazon GameLift to
         *        access your Amazon S3 bucket. The S3 bucket must be in the same region that you want to create a new
         *        build in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageLocation(S3Location storageLocation);

        /**
         * <p>
         * Information indicating where your game build files are stored. Use this parameter only when creating a build
         * with files stored in an Amazon S3 bucket that you own. The storage location must specify an Amazon S3 bucket
         * name and key, as well as a role ARN that you set up to allow Amazon GameLift to access your Amazon S3 bucket.
         * The S3 bucket must be in the same region that you want to create a new build in.
         * </p>
         * This is a convenience that creates an instance of the {@link S3Location.Builder} avoiding the need to create
         * one manually via {@link S3Location#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Location.Builder#build()} is called immediately and its result
         * is passed to {@link #storageLocation(S3Location)}.
         * 
         * @param storageLocation
         *        a consumer that will call methods on {@link S3Location.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #storageLocation(S3Location)
         */
        default Builder storageLocation(Consumer<S3Location.Builder> storageLocation) {
            return storageLocation(S3Location.builder().applyMutation(storageLocation).build());
        }

        /**
         * <p>
         * Operating system that the game server binaries are built to run on. This value determines the type of fleet
         * resources that you can use for this build. If your game build contains multiple executables, they all must
         * run on the same operating system. If an operating system is not specified when creating a build, Amazon
         * GameLift uses the default value (WINDOWS_2012). This value cannot be changed later.
         * </p>
         * 
         * @param operatingSystem
         *        Operating system that the game server binaries are built to run on. This value determines the type of
         *        fleet resources that you can use for this build. If your game build contains multiple executables,
         *        they all must run on the same operating system. If an operating system is not specified when creating
         *        a build, Amazon GameLift uses the default value (WINDOWS_2012). This value cannot be changed later.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(String operatingSystem);

        /**
         * <p>
         * Operating system that the game server binaries are built to run on. This value determines the type of fleet
         * resources that you can use for this build. If your game build contains multiple executables, they all must
         * run on the same operating system. If an operating system is not specified when creating a build, Amazon
         * GameLift uses the default value (WINDOWS_2012). This value cannot be changed later.
         * </p>
         * 
         * @param operatingSystem
         *        Operating system that the game server binaries are built to run on. This value determines the type of
         *        fleet resources that you can use for this build. If your game build contains multiple executables,
         *        they all must run on the same operating system. If an operating system is not specified when creating
         *        a build, Amazon GameLift uses the default value (WINDOWS_2012). This value cannot be changed later.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(OperatingSystem operatingSystem);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GameLiftRequest.BuilderImpl implements Builder {
        private String name;

        private String version;

        private S3Location storageLocation;

        private String operatingSystem;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateBuildRequest model) {
            super(model);
            name(model.name);
            version(model.version);
            storageLocation(model.storageLocation);
            operatingSystem(model.operatingSystem);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getVersion() {
            return version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        public final S3Location.Builder getStorageLocation() {
            return storageLocation != null ? storageLocation.toBuilder() : null;
        }

        @Override
        public final Builder storageLocation(S3Location storageLocation) {
            this.storageLocation = storageLocation;
            return this;
        }

        public final void setStorageLocation(S3Location.BuilderImpl storageLocation) {
            this.storageLocation = storageLocation != null ? storageLocation.build() : null;
        }

        public final String getOperatingSystemAsString() {
            return operatingSystem;
        }

        @Override
        public final Builder operatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
            return this;
        }

        @Override
        public final Builder operatingSystem(OperatingSystem operatingSystem) {
            this.operatingSystem(operatingSystem.toString());
            return this;
        }

        public final void setOperatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateBuildRequest build() {
            return new CreateBuildRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
